/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.appfabric.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.TimestampFormatTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about an app authorization.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AppAuthorization implements SdkPojo, Serializable,
        ToCopyableBuilder<AppAuthorization.Builder, AppAuthorization> {
    private static final SdkField<String> APP_AUTHORIZATION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("appAuthorizationArn").getter(getter(AppAuthorization::appAuthorizationArn))
            .setter(setter(Builder::appAuthorizationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appAuthorizationArn").build())
            .build();

    private static final SdkField<String> APP_BUNDLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("appBundleArn").getter(getter(AppAuthorization::appBundleArn)).setter(setter(Builder::appBundleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("appBundleArn").build()).build();

    private static final SdkField<String> APP_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("app")
            .getter(getter(AppAuthorization::app)).setter(setter(Builder::app))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("app").build()).build();

    private static final SdkField<Tenant> TENANT_FIELD = SdkField.<Tenant> builder(MarshallingType.SDK_POJO).memberName("tenant")
            .getter(getter(AppAuthorization::tenant)).setter(setter(Builder::tenant)).constructor(Tenant::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tenant").build()).build();

    private static final SdkField<String> AUTH_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authType").getter(getter(AppAuthorization::authTypeAsString)).setter(setter(Builder::authType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authType").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(AppAuthorization::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("createdAt")
            .getter(getter(AppAuthorization::createdAt))
            .setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<Instant> UPDATED_AT_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("updatedAt")
            .getter(getter(AppAuthorization::updatedAt))
            .setter(setter(Builder::updatedAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("updatedAt").build(),
                    TimestampFormatTrait.create(TimestampFormatTrait.Format.ISO_8601)).build();

    private static final SdkField<String> PERSONA_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("persona")
            .getter(getter(AppAuthorization::personaAsString)).setter(setter(Builder::persona))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("persona").build()).build();

    private static final SdkField<String> AUTH_URL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("authUrl").getter(getter(AppAuthorization::authUrl)).setter(setter(Builder::authUrl))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authUrl").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(APP_AUTHORIZATION_ARN_FIELD,
            APP_BUNDLE_ARN_FIELD, APP_FIELD, TENANT_FIELD, AUTH_TYPE_FIELD, STATUS_FIELD, CREATED_AT_FIELD, UPDATED_AT_FIELD,
            PERSONA_FIELD, AUTH_URL_FIELD));

    private static final long serialVersionUID = 1L;

    private final String appAuthorizationArn;

    private final String appBundleArn;

    private final String app;

    private final Tenant tenant;

    private final String authType;

    private final String status;

    private final Instant createdAt;

    private final Instant updatedAt;

    private final String persona;

    private final String authUrl;

    private AppAuthorization(BuilderImpl builder) {
        this.appAuthorizationArn = builder.appAuthorizationArn;
        this.appBundleArn = builder.appBundleArn;
        this.app = builder.app;
        this.tenant = builder.tenant;
        this.authType = builder.authType;
        this.status = builder.status;
        this.createdAt = builder.createdAt;
        this.updatedAt = builder.updatedAt;
        this.persona = builder.persona;
        this.authUrl = builder.authUrl;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the app authorization.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the app authorization.
     */
    public final String appAuthorizationArn() {
        return appAuthorizationArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the app bundle for the app authorization.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the app bundle for the app authorization.
     */
    public final String appBundleArn() {
        return appBundleArn;
    }

    /**
     * <p>
     * The name of the application.
     * </p>
     * 
     * @return The name of the application.
     */
    public final String app() {
        return app;
    }

    /**
     * <p>
     * Contains information about an application tenant, such as the application display name and identifier.
     * </p>
     * 
     * @return Contains information about an application tenant, such as the application display name and identifier.
     */
    public final Tenant tenant() {
        return tenant;
    }

    /**
     * <p>
     * The authorization type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authType} will
     * return {@link AuthType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authTypeAsString}.
     * </p>
     * 
     * @return The authorization type.
     * @see AuthType
     */
    public final AuthType authType() {
        return AuthType.fromValue(authType);
    }

    /**
     * <p>
     * The authorization type.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authType} will
     * return {@link AuthType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #authTypeAsString}.
     * </p>
     * 
     * @return The authorization type.
     * @see AuthType
     */
    public final String authTypeAsString() {
        return authType;
    }

    /**
     * <p>
     * The state of the app authorization.
     * </p>
     * <p>
     * The following states are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but not
     * yet connected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
     * connect to the application. If the app authorization is in this state, you should verify the configured
     * credentials and try to connect the app authorization again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization is
     * in this state, you should try to reconnect the app authorization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AppAuthorizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The state of the app authorization.</p>
     *         <p>
     *         The following states are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created
     *         but not yet connected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
     *         trying to connect to the application. If the app authorization is in this state, you should verify the
     *         configured credentials and try to connect the app authorization again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
     *         authorization is in this state, you should try to reconnect the app authorization.
     *         </p>
     *         </li>
     * @see AppAuthorizationStatus
     */
    public final AppAuthorizationStatus status() {
        return AppAuthorizationStatus.fromValue(status);
    }

    /**
     * <p>
     * The state of the app authorization.
     * </p>
     * <p>
     * The following states are possible:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but not
     * yet connected.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
     * connect to the application. If the app authorization is in this state, you should verify the configured
     * credentials and try to connect the app authorization again.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization is
     * in this state, you should try to reconnect the app authorization.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AppAuthorizationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The state of the app authorization.</p>
     *         <p>
     *         The following states are possible:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created
     *         but not yet connected.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
     *         trying to connect to the application. If the app authorization is in this state, you should verify the
     *         configured credentials and try to connect the app authorization again.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
     *         authorization is in this state, you should try to reconnect the app authorization.
     *         </p>
     *         </li>
     * @see AppAuthorizationStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The timestamp of when the app authorization was created.
     * </p>
     * 
     * @return The timestamp of when the app authorization was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The timestamp of when the app authorization was last updated.
     * </p>
     * 
     * @return The timestamp of when the app authorization was last updated.
     */
    public final Instant updatedAt() {
        return updatedAt;
    }

    /**
     * <p>
     * The user persona of the app authorization.
     * </p>
     * <p>
     * This field should always be <code>admin</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #persona} will
     * return {@link Persona#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #personaAsString}.
     * </p>
     * 
     * @return The user persona of the app authorization.</p>
     *         <p>
     *         This field should always be <code>admin</code>.
     * @see Persona
     */
    public final Persona persona() {
        return Persona.fromValue(persona);
    }

    /**
     * <p>
     * The user persona of the app authorization.
     * </p>
     * <p>
     * This field should always be <code>admin</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #persona} will
     * return {@link Persona#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #personaAsString}.
     * </p>
     * 
     * @return The user persona of the app authorization.</p>
     *         <p>
     *         This field should always be <code>admin</code>.
     * @see Persona
     */
    public final String personaAsString() {
        return persona;
    }

    /**
     * <p>
     * The application URL for the OAuth flow.
     * </p>
     * 
     * @return The application URL for the OAuth flow.
     */
    public final String authUrl() {
        return authUrl;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(appAuthorizationArn());
        hashCode = 31 * hashCode + Objects.hashCode(appBundleArn());
        hashCode = 31 * hashCode + Objects.hashCode(app());
        hashCode = 31 * hashCode + Objects.hashCode(tenant());
        hashCode = 31 * hashCode + Objects.hashCode(authTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(updatedAt());
        hashCode = 31 * hashCode + Objects.hashCode(personaAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authUrl());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AppAuthorization)) {
            return false;
        }
        AppAuthorization other = (AppAuthorization) obj;
        return Objects.equals(appAuthorizationArn(), other.appAuthorizationArn())
                && Objects.equals(appBundleArn(), other.appBundleArn()) && Objects.equals(app(), other.app())
                && Objects.equals(tenant(), other.tenant()) && Objects.equals(authTypeAsString(), other.authTypeAsString())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(updatedAt(), other.updatedAt()) && Objects.equals(personaAsString(), other.personaAsString())
                && Objects.equals(authUrl(), other.authUrl());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AppAuthorization").add("AppAuthorizationArn", appAuthorizationArn())
                .add("AppBundleArn", appBundleArn()).add("App", app()).add("Tenant", tenant())
                .add("AuthType", authTypeAsString()).add("Status", statusAsString()).add("CreatedAt", createdAt())
                .add("UpdatedAt", updatedAt()).add("Persona", personaAsString()).add("AuthUrl", authUrl()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "appAuthorizationArn":
            return Optional.ofNullable(clazz.cast(appAuthorizationArn()));
        case "appBundleArn":
            return Optional.ofNullable(clazz.cast(appBundleArn()));
        case "app":
            return Optional.ofNullable(clazz.cast(app()));
        case "tenant":
            return Optional.ofNullable(clazz.cast(tenant()));
        case "authType":
            return Optional.ofNullable(clazz.cast(authTypeAsString()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "createdAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "updatedAt":
            return Optional.ofNullable(clazz.cast(updatedAt()));
        case "persona":
            return Optional.ofNullable(clazz.cast(personaAsString()));
        case "authUrl":
            return Optional.ofNullable(clazz.cast(authUrl()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AppAuthorization, T> g) {
        return obj -> g.apply((AppAuthorization) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AppAuthorization> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the app authorization.
         * </p>
         * 
         * @param appAuthorizationArn
         *        The Amazon Resource Name (ARN) of the app authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appAuthorizationArn(String appAuthorizationArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the app bundle for the app authorization.
         * </p>
         * 
         * @param appBundleArn
         *        The Amazon Resource Name (ARN) of the app bundle for the app authorization.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder appBundleArn(String appBundleArn);

        /**
         * <p>
         * The name of the application.
         * </p>
         * 
         * @param app
         *        The name of the application.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder app(String app);

        /**
         * <p>
         * Contains information about an application tenant, such as the application display name and identifier.
         * </p>
         * 
         * @param tenant
         *        Contains information about an application tenant, such as the application display name and identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tenant(Tenant tenant);

        /**
         * <p>
         * Contains information about an application tenant, such as the application display name and identifier.
         * </p>
         * This is a convenience method that creates an instance of the {@link Tenant.Builder} avoiding the need to
         * create one manually via {@link Tenant#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Tenant.Builder#build()} is called immediately and its result is
         * passed to {@link #tenant(Tenant)}.
         * 
         * @param tenant
         *        a consumer that will call methods on {@link Tenant.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tenant(Tenant)
         */
        default Builder tenant(Consumer<Tenant.Builder> tenant) {
            return tenant(Tenant.builder().applyMutation(tenant).build());
        }

        /**
         * <p>
         * The authorization type.
         * </p>
         * 
         * @param authType
         *        The authorization type.
         * @see AuthType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthType
         */
        Builder authType(String authType);

        /**
         * <p>
         * The authorization type.
         * </p>
         * 
         * @param authType
         *        The authorization type.
         * @see AuthType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthType
         */
        Builder authType(AuthType authType);

        /**
         * <p>
         * The state of the app authorization.
         * </p>
         * <p>
         * The following states are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but
         * not yet connected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
         * connect to the application. If the app authorization is in this state, you should verify the configured
         * credentials and try to connect the app authorization again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization
         * is in this state, you should try to reconnect the app authorization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The state of the app authorization.</p>
         *        <p>
         *        The following states are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is
         *        created but not yet connected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Connected</code>: The app authorization is connected to the application, and is ready to be
         *        used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
         *        trying to connect to the application. If the app authorization is in this state, you should verify the
         *        configured credentials and try to connect the app authorization again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
         *        authorization is in this state, you should try to reconnect the app authorization.
         *        </p>
         *        </li>
         * @see AppAuthorizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppAuthorizationStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The state of the app authorization.
         * </p>
         * <p>
         * The following states are possible:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is created but
         * not yet connected.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Connected</code>: The app authorization is connected to the application, and is ready to be used.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when trying to
         * connect to the application. If the app authorization is in this state, you should verify the configured
         * credentials and try to connect the app authorization again.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app authorization
         * is in this state, you should try to reconnect the app authorization.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The state of the app authorization.</p>
         *        <p>
         *        The following states are possible:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>PendingConnect</code>: The initial state of the app authorization. The app authorization is
         *        created but not yet connected.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Connected</code>: The app authorization is connected to the application, and is ready to be
         *        used.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ConnectionValidationFailed</code>: The app authorization received a validation exception when
         *        trying to connect to the application. If the app authorization is in this state, you should verify the
         *        configured credentials and try to connect the app authorization again.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>TokenAutoRotationFailed</code>: AppFabric failed to refresh the access token. If the app
         *        authorization is in this state, you should try to reconnect the app authorization.
         *        </p>
         *        </li>
         * @see AppAuthorizationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AppAuthorizationStatus
         */
        Builder status(AppAuthorizationStatus status);

        /**
         * <p>
         * The timestamp of when the app authorization was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp of when the app authorization was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The timestamp of when the app authorization was last updated.
         * </p>
         * 
         * @param updatedAt
         *        The timestamp of when the app authorization was last updated.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder updatedAt(Instant updatedAt);

        /**
         * <p>
         * The user persona of the app authorization.
         * </p>
         * <p>
         * This field should always be <code>admin</code>.
         * </p>
         * 
         * @param persona
         *        The user persona of the app authorization.</p>
         *        <p>
         *        This field should always be <code>admin</code>.
         * @see Persona
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Persona
         */
        Builder persona(String persona);

        /**
         * <p>
         * The user persona of the app authorization.
         * </p>
         * <p>
         * This field should always be <code>admin</code>.
         * </p>
         * 
         * @param persona
         *        The user persona of the app authorization.</p>
         *        <p>
         *        This field should always be <code>admin</code>.
         * @see Persona
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Persona
         */
        Builder persona(Persona persona);

        /**
         * <p>
         * The application URL for the OAuth flow.
         * </p>
         * 
         * @param authUrl
         *        The application URL for the OAuth flow.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authUrl(String authUrl);
    }

    static final class BuilderImpl implements Builder {
        private String appAuthorizationArn;

        private String appBundleArn;

        private String app;

        private Tenant tenant;

        private String authType;

        private String status;

        private Instant createdAt;

        private Instant updatedAt;

        private String persona;

        private String authUrl;

        private BuilderImpl() {
        }

        private BuilderImpl(AppAuthorization model) {
            appAuthorizationArn(model.appAuthorizationArn);
            appBundleArn(model.appBundleArn);
            app(model.app);
            tenant(model.tenant);
            authType(model.authType);
            status(model.status);
            createdAt(model.createdAt);
            updatedAt(model.updatedAt);
            persona(model.persona);
            authUrl(model.authUrl);
        }

        public final String getAppAuthorizationArn() {
            return appAuthorizationArn;
        }

        public final void setAppAuthorizationArn(String appAuthorizationArn) {
            this.appAuthorizationArn = appAuthorizationArn;
        }

        @Override
        public final Builder appAuthorizationArn(String appAuthorizationArn) {
            this.appAuthorizationArn = appAuthorizationArn;
            return this;
        }

        public final String getAppBundleArn() {
            return appBundleArn;
        }

        public final void setAppBundleArn(String appBundleArn) {
            this.appBundleArn = appBundleArn;
        }

        @Override
        public final Builder appBundleArn(String appBundleArn) {
            this.appBundleArn = appBundleArn;
            return this;
        }

        public final String getApp() {
            return app;
        }

        public final void setApp(String app) {
            this.app = app;
        }

        @Override
        public final Builder app(String app) {
            this.app = app;
            return this;
        }

        public final Tenant.Builder getTenant() {
            return tenant != null ? tenant.toBuilder() : null;
        }

        public final void setTenant(Tenant.BuilderImpl tenant) {
            this.tenant = tenant != null ? tenant.build() : null;
        }

        @Override
        public final Builder tenant(Tenant tenant) {
            this.tenant = tenant;
            return this;
        }

        public final String getAuthType() {
            return authType;
        }

        public final void setAuthType(String authType) {
            this.authType = authType;
        }

        @Override
        public final Builder authType(String authType) {
            this.authType = authType;
            return this;
        }

        @Override
        public final Builder authType(AuthType authType) {
            this.authType(authType == null ? null : authType.toString());
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AppAuthorizationStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final Instant getUpdatedAt() {
            return updatedAt;
        }

        public final void setUpdatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
        }

        @Override
        public final Builder updatedAt(Instant updatedAt) {
            this.updatedAt = updatedAt;
            return this;
        }

        public final String getPersona() {
            return persona;
        }

        public final void setPersona(String persona) {
            this.persona = persona;
        }

        @Override
        public final Builder persona(String persona) {
            this.persona = persona;
            return this;
        }

        @Override
        public final Builder persona(Persona persona) {
            this.persona(persona == null ? null : persona.toString());
            return this;
        }

        public final String getAuthUrl() {
            return authUrl;
        }

        public final void setAuthUrl(String authUrl) {
            this.authUrl = authUrl;
        }

        @Override
        public final Builder authUrl(String authUrl) {
            this.authUrl = authUrl;
            return this;
        }

        @Override
        public AppAuthorization build() {
            return new AppAuthorization(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
