/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigatewayv2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for HTTP APIs.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class JWTConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<JWTConfiguration.Builder, JWTConfiguration> {
    private static final SdkField<List<String>> AUDIENCE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(JWTConfiguration::audience))
            .setter(setter(Builder::audience))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("audience").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> ISSUER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(JWTConfiguration::issuer)).setter(setter(Builder::issuer))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("issuer").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(AUDIENCE_FIELD, ISSUER_FIELD));

    private static final long serialVersionUID = 1L;

    private final List<String> audience;

    private final String issuer;

    private JWTConfiguration(BuilderImpl builder) {
        this.audience = builder.audience;
        this.issuer = builder.issuer;
    }

    /**
     * Returns true if the Audience property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasAudience() {
        return audience != null && !(audience instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the intended recipients of the JWT. A valid JWT must provide an aud that matches at least one entry in
     * this list. See <a href="https://tools.ietf.org/html/rfc7519#section-4.1.3">RFC 7519</a>. Supported only for HTTP
     * APIs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasAudience()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of the intended recipients of the JWT. A valid JWT must provide an aud that matches at least one
     *         entry in this list. See <a href="https://tools.ietf.org/html/rfc7519#section-4.1.3">RFC 7519</a>.
     *         Supported only for HTTP APIs.
     */
    public List<String> audience() {
        return audience;
    }

    /**
     * <p>
     * The base domain of the identity provider that issues JSON Web Tokens. For example, an Amazon Cognito user pool
     * has the following format:
     * https://cognito-idp.<replaceable>{region}</replaceable>.amazonaws.com/<replaceable>{userPoolId}</replaceable> .
     * Required for the JWT authorizer type. Supported only for HTTP APIs.
     * </p>
     * 
     * @return The base domain of the identity provider that issues JSON Web Tokens. For example, an Amazon Cognito user
     *         pool has the following format:
     *         https://cognito-idp.<replaceable>{region}</replaceable>.amazonaws.com/<replaceable
     *         >{userPoolId}</replaceable> . Required for the JWT authorizer type. Supported only for HTTP APIs.
     */
    public String issuer() {
        return issuer;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(audience());
        hashCode = 31 * hashCode + Objects.hashCode(issuer());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof JWTConfiguration)) {
            return false;
        }
        JWTConfiguration other = (JWTConfiguration) obj;
        return Objects.equals(audience(), other.audience()) && Objects.equals(issuer(), other.issuer());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("JWTConfiguration").add("Audience", audience()).add("Issuer", issuer()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Audience":
            return Optional.ofNullable(clazz.cast(audience()));
        case "Issuer":
            return Optional.ofNullable(clazz.cast(issuer()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<JWTConfiguration, T> g) {
        return obj -> g.apply((JWTConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, JWTConfiguration> {
        /**
         * <p>
         * A list of the intended recipients of the JWT. A valid JWT must provide an aud that matches at least one entry
         * in this list. See <a href="https://tools.ietf.org/html/rfc7519#section-4.1.3">RFC 7519</a>. Supported only
         * for HTTP APIs.
         * </p>
         * 
         * @param audience
         *        A list of the intended recipients of the JWT. A valid JWT must provide an aud that matches at least
         *        one entry in this list. See <a href="https://tools.ietf.org/html/rfc7519#section-4.1.3">RFC 7519</a>.
         *        Supported only for HTTP APIs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audience(Collection<String> audience);

        /**
         * <p>
         * A list of the intended recipients of the JWT. A valid JWT must provide an aud that matches at least one entry
         * in this list. See <a href="https://tools.ietf.org/html/rfc7519#section-4.1.3">RFC 7519</a>. Supported only
         * for HTTP APIs.
         * </p>
         * 
         * @param audience
         *        A list of the intended recipients of the JWT. A valid JWT must provide an aud that matches at least
         *        one entry in this list. See <a href="https://tools.ietf.org/html/rfc7519#section-4.1.3">RFC 7519</a>.
         *        Supported only for HTTP APIs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder audience(String... audience);

        /**
         * <p>
         * The base domain of the identity provider that issues JSON Web Tokens. For example, an Amazon Cognito user
         * pool has the following format:
         * https://cognito-idp.<replaceable>{region}</replaceable>.amazonaws.com/<replaceable>{userPoolId}</replaceable>
         * . Required for the JWT authorizer type. Supported only for HTTP APIs.
         * </p>
         * 
         * @param issuer
         *        The base domain of the identity provider that issues JSON Web Tokens. For example, an Amazon Cognito
         *        user pool has the following format:
         *        https://cognito-idp.<replaceable>{region}</replaceable>.amazonaws.com
         *        /<replaceable>{userPoolId}</replaceable> . Required for the JWT authorizer type. Supported only for
         *        HTTP APIs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder issuer(String issuer);
    }

    static final class BuilderImpl implements Builder {
        private List<String> audience = DefaultSdkAutoConstructList.getInstance();

        private String issuer;

        private BuilderImpl() {
        }

        private BuilderImpl(JWTConfiguration model) {
            audience(model.audience);
            issuer(model.issuer);
        }

        public final Collection<String> getAudience() {
            return audience;
        }

        @Override
        public final Builder audience(Collection<String> audience) {
            this.audience = ___listOf__stringCopier.copy(audience);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder audience(String... audience) {
            audience(Arrays.asList(audience));
            return this;
        }

        public final void setAudience(Collection<String> audience) {
            this.audience = ___listOf__stringCopier.copy(audience);
        }

        public final String getIssuer() {
            return issuer;
        }

        @Override
        public final Builder issuer(String issuer) {
            this.issuer = issuer;
            return this;
        }

        public final void setIssuer(String issuer) {
            this.issuer = issuer;
        }

        @Override
        public JWTConfiguration build() {
            return new JWTConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
