/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.apigatewayv2.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Creates a new Authorizer resource to represent an authorizer.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateAuthorizerRequest extends ApiGatewayV2Request implements
        ToCopyableBuilder<CreateAuthorizerRequest.Builder, CreateAuthorizerRequest> {
    private static final SdkField<String> API_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAuthorizerRequest::apiId)).setter(setter(Builder::apiId))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("apiId").build()).build();

    private static final SdkField<String> AUTHORIZER_CREDENTIALS_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAuthorizerRequest::authorizerCredentialsArn)).setter(setter(Builder::authorizerCredentialsArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerCredentialsArn").build())
            .build();

    private static final SdkField<Integer> AUTHORIZER_RESULT_TTL_IN_SECONDS_FIELD = SdkField
            .<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(CreateAuthorizerRequest::authorizerResultTtlInSeconds))
            .setter(setter(Builder::authorizerResultTtlInSeconds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerResultTtlInSeconds")
                    .build()).build();

    private static final SdkField<String> AUTHORIZER_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAuthorizerRequest::authorizerTypeAsString)).setter(setter(Builder::authorizerType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerType").build()).build();

    private static final SdkField<String> AUTHORIZER_URI_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAuthorizerRequest::authorizerUri)).setter(setter(Builder::authorizerUri))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("authorizerUri").build()).build();

    private static final SdkField<List<String>> IDENTITY_SOURCE_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(CreateAuthorizerRequest::identitySource))
            .setter(setter(Builder::identitySource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identitySource").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> IDENTITY_VALIDATION_EXPRESSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAuthorizerRequest::identityValidationExpression))
            .setter(setter(Builder::identityValidationExpression))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("identityValidationExpression")
                    .build()).build();

    private static final SdkField<JWTConfiguration> JWT_CONFIGURATION_FIELD = SdkField
            .<JWTConfiguration> builder(MarshallingType.SDK_POJO).getter(getter(CreateAuthorizerRequest::jwtConfiguration))
            .setter(setter(Builder::jwtConfiguration)).constructor(JWTConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("jwtConfiguration").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(CreateAuthorizerRequest::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(API_ID_FIELD,
            AUTHORIZER_CREDENTIALS_ARN_FIELD, AUTHORIZER_RESULT_TTL_IN_SECONDS_FIELD, AUTHORIZER_TYPE_FIELD,
            AUTHORIZER_URI_FIELD, IDENTITY_SOURCE_FIELD, IDENTITY_VALIDATION_EXPRESSION_FIELD, JWT_CONFIGURATION_FIELD,
            NAME_FIELD));

    private final String apiId;

    private final String authorizerCredentialsArn;

    private final Integer authorizerResultTtlInSeconds;

    private final String authorizerType;

    private final String authorizerUri;

    private final List<String> identitySource;

    private final String identityValidationExpression;

    private final JWTConfiguration jwtConfiguration;

    private final String name;

    private CreateAuthorizerRequest(BuilderImpl builder) {
        super(builder);
        this.apiId = builder.apiId;
        this.authorizerCredentialsArn = builder.authorizerCredentialsArn;
        this.authorizerResultTtlInSeconds = builder.authorizerResultTtlInSeconds;
        this.authorizerType = builder.authorizerType;
        this.authorizerUri = builder.authorizerUri;
        this.identitySource = builder.identitySource;
        this.identityValidationExpression = builder.identityValidationExpression;
        this.jwtConfiguration = builder.jwtConfiguration;
        this.name = builder.name;
    }

    /**
     * <p>
     * The API identifier.
     * </p>
     * 
     * @return The API identifier.
     */
    public String apiId() {
        return apiId;
    }

    /**
     * <p>
     * Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM
     * role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions on
     * the Lambda function, specify null. Supported only for REQUEST authorizers.
     * </p>
     * 
     * @return Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an
     *         IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based
     *         permissions on the Lambda function, specify null. Supported only for REQUEST authorizers.
     */
    public String authorizerCredentialsArn() {
        return authorizerCredentialsArn;
    }

    /**
     * <p>
     * Authorizer caching is not currently supported. Don't specify this value for authorizers.
     * </p>
     * 
     * @return Authorizer caching is not currently supported. Don't specify this value for authorizers.
     */
    public Integer authorizerResultTtlInSeconds() {
        return authorizerResultTtlInSeconds;
    }

    /**
     * <p>
     * The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request parameters.
     * For HTTP APIs, specify JWT to use JSON Web Tokens.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authorizerType}
     * will return {@link AuthorizerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #authorizerTypeAsString}.
     * </p>
     * 
     * @return The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request
     *         parameters. For HTTP APIs, specify JWT to use JSON Web Tokens.
     * @see AuthorizerType
     */
    public AuthorizerType authorizerType() {
        return AuthorizerType.fromValue(authorizerType);
    }

    /**
     * <p>
     * The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request parameters.
     * For HTTP APIs, specify JWT to use JSON Web Tokens.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #authorizerType}
     * will return {@link AuthorizerType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #authorizerTypeAsString}.
     * </p>
     * 
     * @return The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request
     *         parameters. For HTTP APIs, specify JWT to use JSON Web Tokens.
     * @see AuthorizerType
     */
    public String authorizerTypeAsString() {
        return authorizerType;
    }

    /**
     * <p>
     * The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a well-formed Lambda
     * function URI, for example,
     * arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda:us-west-2:
     * <replaceable>{account_id}</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations.
     * In general, the URI has this form:
     * arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>{service_api}</replaceable> ,
     * where <replaceable></replaceable>{region} is the same as the region hosting the Lambda function, path indicates
     * that the remaining substring in the URI should be treated as the path to the resource, including the initial /.
     * For Lambda functions, this is usually of the form /2015-03-31/functions/[FunctionARN]/invocations. Supported only
     * for REQUEST authorizers.
     * </p>
     * 
     * @return The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a well-formed
     *         Lambda function URI, for example,
     *         arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda
     *         :us-west-2:<replaceable>{account_id
     *         }</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations. In general, the
     *         URI has this form:
     *         arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>{service_api
     *         }</replaceable> , where <replaceable></replaceable>{region} is the same as the region hosting the Lambda
     *         function, path indicates that the remaining substring in the URI should be treated as the path to the
     *         resource, including the initial /. For Lambda functions, this is usually of the form
     *         /2015-03-31/functions/[FunctionARN]/invocations. Supported only for REQUEST authorizers.
     */
    public String authorizerUri() {
        return authorizerUri;
    }

    /**
     * Returns true if the IdentitySource property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasIdentitySource() {
        return identitySource != null && !(identitySource instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The identity source for which authorization is requested.
     * </p>
     * <p>
     * For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
     * specified request parameters. Currently, the identity source can be headers, query string parameters, stage
     * variables, and context parameters. For example, if an Auth header and a Name query string parameter are defined
     * as identity sources, this value is route.request.header.Auth, route.request.querystring.Name. These parameters
     * will be used to perform runtime validation for Lambda-based authorizers by verifying all of the identity-related
     * request parameters are present in the request, not null, and non-empty. Only when this is true does the
     * authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401 Unauthorized response without
     * calling the Lambda function.
     * </p>
     * <p>
     * For JWT, a single entry that specifies where to extract the JSON Web Token (JWT )from inbound requests. Currently
     * only header-based and query parameter-based selections are supported, for example
     * "$request.header.Authorization".
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasIdentitySource()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The identity source for which authorization is requested.</p>
     *         <p>
     *         For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
     *         specified request parameters. Currently, the identity source can be headers, query string parameters,
     *         stage variables, and context parameters. For example, if an Auth header and a Name query string parameter
     *         are defined as identity sources, this value is route.request.header.Auth, route.request.querystring.Name.
     *         These parameters will be used to perform runtime validation for Lambda-based authorizers by verifying all
     *         of the identity-related request parameters are present in the request, not null, and non-empty. Only when
     *         this is true does the authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401
     *         Unauthorized response without calling the Lambda function.
     *         </p>
     *         <p>
     *         For JWT, a single entry that specifies where to extract the JSON Web Token (JWT )from inbound requests.
     *         Currently only header-based and query parameter-based selections are supported, for example
     *         "$request.header.Authorization".
     */
    public List<String> identitySource() {
        return identitySource;
    }

    /**
     * <p>
     * This parameter is not used.
     * </p>
     * 
     * @return This parameter is not used.
     */
    public String identityValidationExpression() {
        return identityValidationExpression;
    }

    /**
     * <p>
     * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for HTTP
     * APIs.
     * </p>
     * 
     * @return Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only
     *         for HTTP APIs.
     */
    public JWTConfiguration jwtConfiguration() {
        return jwtConfiguration;
    }

    /**
     * <p>
     * The name of the authorizer.
     * </p>
     * 
     * @return The name of the authorizer.
     */
    public String name() {
        return name;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(apiId());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerCredentialsArn());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerResultTtlInSeconds());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(authorizerUri());
        hashCode = 31 * hashCode + Objects.hashCode(identitySource());
        hashCode = 31 * hashCode + Objects.hashCode(identityValidationExpression());
        hashCode = 31 * hashCode + Objects.hashCode(jwtConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateAuthorizerRequest)) {
            return false;
        }
        CreateAuthorizerRequest other = (CreateAuthorizerRequest) obj;
        return Objects.equals(apiId(), other.apiId())
                && Objects.equals(authorizerCredentialsArn(), other.authorizerCredentialsArn())
                && Objects.equals(authorizerResultTtlInSeconds(), other.authorizerResultTtlInSeconds())
                && Objects.equals(authorizerTypeAsString(), other.authorizerTypeAsString())
                && Objects.equals(authorizerUri(), other.authorizerUri())
                && Objects.equals(identitySource(), other.identitySource())
                && Objects.equals(identityValidationExpression(), other.identityValidationExpression())
                && Objects.equals(jwtConfiguration(), other.jwtConfiguration()) && Objects.equals(name(), other.name());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("CreateAuthorizerRequest").add("ApiId", apiId())
                .add("AuthorizerCredentialsArn", authorizerCredentialsArn())
                .add("AuthorizerResultTtlInSeconds", authorizerResultTtlInSeconds())
                .add("AuthorizerType", authorizerTypeAsString()).add("AuthorizerUri", authorizerUri())
                .add("IdentitySource", identitySource()).add("IdentityValidationExpression", identityValidationExpression())
                .add("JwtConfiguration", jwtConfiguration()).add("Name", name()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ApiId":
            return Optional.ofNullable(clazz.cast(apiId()));
        case "AuthorizerCredentialsArn":
            return Optional.ofNullable(clazz.cast(authorizerCredentialsArn()));
        case "AuthorizerResultTtlInSeconds":
            return Optional.ofNullable(clazz.cast(authorizerResultTtlInSeconds()));
        case "AuthorizerType":
            return Optional.ofNullable(clazz.cast(authorizerTypeAsString()));
        case "AuthorizerUri":
            return Optional.ofNullable(clazz.cast(authorizerUri()));
        case "IdentitySource":
            return Optional.ofNullable(clazz.cast(identitySource()));
        case "IdentityValidationExpression":
            return Optional.ofNullable(clazz.cast(identityValidationExpression()));
        case "JwtConfiguration":
            return Optional.ofNullable(clazz.cast(jwtConfiguration()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateAuthorizerRequest, T> g) {
        return obj -> g.apply((CreateAuthorizerRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends ApiGatewayV2Request.Builder, SdkPojo, CopyableBuilder<Builder, CreateAuthorizerRequest> {
        /**
         * <p>
         * The API identifier.
         * </p>
         * 
         * @param apiId
         *        The API identifier.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder apiId(String apiId);

        /**
         * <p>
         * Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify an IAM
         * role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use resource-based permissions
         * on the Lambda function, specify null. Supported only for REQUEST authorizers.
         * </p>
         * 
         * @param authorizerCredentialsArn
         *        Specifies the required credentials as an IAM role for API Gateway to invoke the authorizer. To specify
         *        an IAM role for API Gateway to assume, use the role's Amazon Resource Name (ARN). To use
         *        resource-based permissions on the Lambda function, specify null. Supported only for REQUEST
         *        authorizers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerCredentialsArn(String authorizerCredentialsArn);

        /**
         * <p>
         * Authorizer caching is not currently supported. Don't specify this value for authorizers.
         * </p>
         * 
         * @param authorizerResultTtlInSeconds
         *        Authorizer caching is not currently supported. Don't specify this value for authorizers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds);

        /**
         * <p>
         * The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request
         * parameters. For HTTP APIs, specify JWT to use JSON Web Tokens.
         * </p>
         * 
         * @param authorizerType
         *        The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request
         *        parameters. For HTTP APIs, specify JWT to use JSON Web Tokens.
         * @see AuthorizerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerType
         */
        Builder authorizerType(String authorizerType);

        /**
         * <p>
         * The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request
         * parameters. For HTTP APIs, specify JWT to use JSON Web Tokens.
         * </p>
         * 
         * @param authorizerType
         *        The authorizer type. For WebSocket APIs, specify REQUEST for a Lambda function using incoming request
         *        parameters. For HTTP APIs, specify JWT to use JSON Web Tokens.
         * @see AuthorizerType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AuthorizerType
         */
        Builder authorizerType(AuthorizerType authorizerType);

        /**
         * <p>
         * The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a well-formed
         * Lambda function URI, for example,
         * arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions/arn:aws:lambda
         * :us-west-2:<replaceable>{account_id
         * }</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations. In general, the URI
         * has this form:
         * arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>{service_api}</replaceable> ,
         * where <replaceable></replaceable>{region} is the same as the region hosting the Lambda function, path
         * indicates that the remaining substring in the URI should be treated as the path to the resource, including
         * the initial /. For Lambda functions, this is usually of the form
         * /2015-03-31/functions/[FunctionARN]/invocations. Supported only for REQUEST authorizers.
         * </p>
         * 
         * @param authorizerUri
         *        The authorizer's Uniform Resource Identifier (URI). For REQUEST authorizers, this must be a
         *        well-formed Lambda function URI, for example,
         *        arn:aws:apigateway:us-west-2:lambda:path/2015-03-31/functions
         *        /arn:aws:lambda:us-west-2:<replaceable>{account_id
         *        }</replaceable>:function:<replaceable>{lambda_function_name}</replaceable>/invocations. In general,
         *        the URI has this form:
         *        arn:aws:apigateway:<replaceable>{region}</replaceable>:lambda:path/<replaceable>
         *        {service_api}</replaceable> , where <replaceable></replaceable>{region} is the same as the region
         *        hosting the Lambda function, path indicates that the remaining substring in the URI should be treated
         *        as the path to the resource, including the initial /. For Lambda functions, this is usually of the
         *        form /2015-03-31/functions/[FunctionARN]/invocations. Supported only for REQUEST authorizers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder authorizerUri(String authorizerUri);

        /**
         * <p>
         * The identity source for which authorization is requested.
         * </p>
         * <p>
         * For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
         * specified request parameters. Currently, the identity source can be headers, query string parameters, stage
         * variables, and context parameters. For example, if an Auth header and a Name query string parameter are
         * defined as identity sources, this value is route.request.header.Auth, route.request.querystring.Name. These
         * parameters will be used to perform runtime validation for Lambda-based authorizers by verifying all of the
         * identity-related request parameters are present in the request, not null, and non-empty. Only when this is
         * true does the authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401 Unauthorized
         * response without calling the Lambda function.
         * </p>
         * <p>
         * For JWT, a single entry that specifies where to extract the JSON Web Token (JWT )from inbound requests.
         * Currently only header-based and query parameter-based selections are supported, for example
         * "$request.header.Authorization".
         * </p>
         * 
         * @param identitySource
         *        The identity source for which authorization is requested.</p>
         *        <p>
         *        For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of
         *        the specified request parameters. Currently, the identity source can be headers, query string
         *        parameters, stage variables, and context parameters. For example, if an Auth header and a Name query
         *        string parameter are defined as identity sources, this value is route.request.header.Auth,
         *        route.request.querystring.Name. These parameters will be used to perform runtime validation for
         *        Lambda-based authorizers by verifying all of the identity-related request parameters are present in
         *        the request, not null, and non-empty. Only when this is true does the authorizer invoke the authorizer
         *        Lambda function. Otherwise, it returns a 401 Unauthorized response without calling the Lambda
         *        function.
         *        </p>
         *        <p>
         *        For JWT, a single entry that specifies where to extract the JSON Web Token (JWT )from inbound
         *        requests. Currently only header-based and query parameter-based selections are supported, for example
         *        "$request.header.Authorization".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identitySource(Collection<String> identitySource);

        /**
         * <p>
         * The identity source for which authorization is requested.
         * </p>
         * <p>
         * For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of the
         * specified request parameters. Currently, the identity source can be headers, query string parameters, stage
         * variables, and context parameters. For example, if an Auth header and a Name query string parameter are
         * defined as identity sources, this value is route.request.header.Auth, route.request.querystring.Name. These
         * parameters will be used to perform runtime validation for Lambda-based authorizers by verifying all of the
         * identity-related request parameters are present in the request, not null, and non-empty. Only when this is
         * true does the authorizer invoke the authorizer Lambda function. Otherwise, it returns a 401 Unauthorized
         * response without calling the Lambda function.
         * </p>
         * <p>
         * For JWT, a single entry that specifies where to extract the JSON Web Token (JWT )from inbound requests.
         * Currently only header-based and query parameter-based selections are supported, for example
         * "$request.header.Authorization".
         * </p>
         * 
         * @param identitySource
         *        The identity source for which authorization is requested.</p>
         *        <p>
         *        For a REQUEST authorizer, this is optional. The value is a set of one or more mapping expressions of
         *        the specified request parameters. Currently, the identity source can be headers, query string
         *        parameters, stage variables, and context parameters. For example, if an Auth header and a Name query
         *        string parameter are defined as identity sources, this value is route.request.header.Auth,
         *        route.request.querystring.Name. These parameters will be used to perform runtime validation for
         *        Lambda-based authorizers by verifying all of the identity-related request parameters are present in
         *        the request, not null, and non-empty. Only when this is true does the authorizer invoke the authorizer
         *        Lambda function. Otherwise, it returns a 401 Unauthorized response without calling the Lambda
         *        function.
         *        </p>
         *        <p>
         *        For JWT, a single entry that specifies where to extract the JSON Web Token (JWT )from inbound
         *        requests. Currently only header-based and query parameter-based selections are supported, for example
         *        "$request.header.Authorization".
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identitySource(String... identitySource);

        /**
         * <p>
         * This parameter is not used.
         * </p>
         * 
         * @param identityValidationExpression
         *        This parameter is not used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder identityValidationExpression(String identityValidationExpression);

        /**
         * <p>
         * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for
         * HTTP APIs.
         * </p>
         * 
         * @param jwtConfiguration
         *        Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only
         *        for HTTP APIs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder jwtConfiguration(JWTConfiguration jwtConfiguration);

        /**
         * <p>
         * Represents the configuration of a JWT authorizer. Required for the JWT authorizer type. Supported only for
         * HTTP APIs.
         * </p>
         * This is a convenience that creates an instance of the {@link JWTConfiguration.Builder} avoiding the need to
         * create one manually via {@link JWTConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link JWTConfiguration.Builder#build()} is called immediately and its
         * result is passed to {@link #jwtConfiguration(JWTConfiguration)}.
         * 
         * @param jwtConfiguration
         *        a consumer that will call methods on {@link JWTConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #jwtConfiguration(JWTConfiguration)
         */
        default Builder jwtConfiguration(Consumer<JWTConfiguration.Builder> jwtConfiguration) {
            return jwtConfiguration(JWTConfiguration.builder().applyMutation(jwtConfiguration).build());
        }

        /**
         * <p>
         * The name of the authorizer.
         * </p>
         * 
         * @param name
         *        The name of the authorizer.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends ApiGatewayV2Request.BuilderImpl implements Builder {
        private String apiId;

        private String authorizerCredentialsArn;

        private Integer authorizerResultTtlInSeconds;

        private String authorizerType;

        private String authorizerUri;

        private List<String> identitySource = DefaultSdkAutoConstructList.getInstance();

        private String identityValidationExpression;

        private JWTConfiguration jwtConfiguration;

        private String name;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateAuthorizerRequest model) {
            super(model);
            apiId(model.apiId);
            authorizerCredentialsArn(model.authorizerCredentialsArn);
            authorizerResultTtlInSeconds(model.authorizerResultTtlInSeconds);
            authorizerType(model.authorizerType);
            authorizerUri(model.authorizerUri);
            identitySource(model.identitySource);
            identityValidationExpression(model.identityValidationExpression);
            jwtConfiguration(model.jwtConfiguration);
            name(model.name);
        }

        public final String getApiId() {
            return apiId;
        }

        @Override
        public final Builder apiId(String apiId) {
            this.apiId = apiId;
            return this;
        }

        public final void setApiId(String apiId) {
            this.apiId = apiId;
        }

        public final String getAuthorizerCredentialsArn() {
            return authorizerCredentialsArn;
        }

        @Override
        public final Builder authorizerCredentialsArn(String authorizerCredentialsArn) {
            this.authorizerCredentialsArn = authorizerCredentialsArn;
            return this;
        }

        public final void setAuthorizerCredentialsArn(String authorizerCredentialsArn) {
            this.authorizerCredentialsArn = authorizerCredentialsArn;
        }

        public final Integer getAuthorizerResultTtlInSeconds() {
            return authorizerResultTtlInSeconds;
        }

        @Override
        public final Builder authorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds) {
            this.authorizerResultTtlInSeconds = authorizerResultTtlInSeconds;
            return this;
        }

        public final void setAuthorizerResultTtlInSeconds(Integer authorizerResultTtlInSeconds) {
            this.authorizerResultTtlInSeconds = authorizerResultTtlInSeconds;
        }

        public final String getAuthorizerTypeAsString() {
            return authorizerType;
        }

        @Override
        public final Builder authorizerType(String authorizerType) {
            this.authorizerType = authorizerType;
            return this;
        }

        @Override
        public final Builder authorizerType(AuthorizerType authorizerType) {
            this.authorizerType(authorizerType == null ? null : authorizerType.toString());
            return this;
        }

        public final void setAuthorizerType(String authorizerType) {
            this.authorizerType = authorizerType;
        }

        public final String getAuthorizerUri() {
            return authorizerUri;
        }

        @Override
        public final Builder authorizerUri(String authorizerUri) {
            this.authorizerUri = authorizerUri;
            return this;
        }

        public final void setAuthorizerUri(String authorizerUri) {
            this.authorizerUri = authorizerUri;
        }

        public final Collection<String> getIdentitySource() {
            return identitySource;
        }

        @Override
        public final Builder identitySource(Collection<String> identitySource) {
            this.identitySource = IdentitySourceListCopier.copy(identitySource);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder identitySource(String... identitySource) {
            identitySource(Arrays.asList(identitySource));
            return this;
        }

        public final void setIdentitySource(Collection<String> identitySource) {
            this.identitySource = IdentitySourceListCopier.copy(identitySource);
        }

        public final String getIdentityValidationExpression() {
            return identityValidationExpression;
        }

        @Override
        public final Builder identityValidationExpression(String identityValidationExpression) {
            this.identityValidationExpression = identityValidationExpression;
            return this;
        }

        public final void setIdentityValidationExpression(String identityValidationExpression) {
            this.identityValidationExpression = identityValidationExpression;
        }

        public final JWTConfiguration.Builder getJwtConfiguration() {
            return jwtConfiguration != null ? jwtConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder jwtConfiguration(JWTConfiguration jwtConfiguration) {
            this.jwtConfiguration = jwtConfiguration;
            return this;
        }

        public final void setJwtConfiguration(JWTConfiguration.BuilderImpl jwtConfiguration) {
            this.jwtConfiguration = jwtConfiguration != null ? jwtConfiguration.build() : null;
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateAuthorizerRequest build() {
            return new CreateAuthorizerRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
