/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.acmpca;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.acmpca.model.AcmPcaException;
import software.amazon.awssdk.services.acmpca.model.AcmPcaRequest;
import software.amazon.awssdk.services.acmpca.model.CertificateMismatchException;
import software.amazon.awssdk.services.acmpca.model.ConcurrentModificationException;
import software.amazon.awssdk.services.acmpca.model.CreateCertificateAuthorityAuditReportRequest;
import software.amazon.awssdk.services.acmpca.model.CreateCertificateAuthorityAuditReportResponse;
import software.amazon.awssdk.services.acmpca.model.CreateCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.CreateCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.model.CreatePermissionRequest;
import software.amazon.awssdk.services.acmpca.model.CreatePermissionResponse;
import software.amazon.awssdk.services.acmpca.model.DeleteCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.DeleteCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.model.DeletePermissionRequest;
import software.amazon.awssdk.services.acmpca.model.DeletePermissionResponse;
import software.amazon.awssdk.services.acmpca.model.DeletePolicyRequest;
import software.amazon.awssdk.services.acmpca.model.DeletePolicyResponse;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityAuditReportRequest;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityAuditReportResponse;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.DescribeCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCertificateResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCsrRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateAuthorityCsrResponse;
import software.amazon.awssdk.services.acmpca.model.GetCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.GetCertificateResponse;
import software.amazon.awssdk.services.acmpca.model.GetPolicyRequest;
import software.amazon.awssdk.services.acmpca.model.GetPolicyResponse;
import software.amazon.awssdk.services.acmpca.model.ImportCertificateAuthorityCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.ImportCertificateAuthorityCertificateResponse;
import software.amazon.awssdk.services.acmpca.model.InvalidArgsException;
import software.amazon.awssdk.services.acmpca.model.InvalidArnException;
import software.amazon.awssdk.services.acmpca.model.InvalidNextTokenException;
import software.amazon.awssdk.services.acmpca.model.InvalidPolicyException;
import software.amazon.awssdk.services.acmpca.model.InvalidRequestException;
import software.amazon.awssdk.services.acmpca.model.InvalidStateException;
import software.amazon.awssdk.services.acmpca.model.InvalidTagException;
import software.amazon.awssdk.services.acmpca.model.IssueCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.IssueCertificateResponse;
import software.amazon.awssdk.services.acmpca.model.LimitExceededException;
import software.amazon.awssdk.services.acmpca.model.ListCertificateAuthoritiesRequest;
import software.amazon.awssdk.services.acmpca.model.ListCertificateAuthoritiesResponse;
import software.amazon.awssdk.services.acmpca.model.ListPermissionsRequest;
import software.amazon.awssdk.services.acmpca.model.ListPermissionsResponse;
import software.amazon.awssdk.services.acmpca.model.ListTagsRequest;
import software.amazon.awssdk.services.acmpca.model.ListTagsResponse;
import software.amazon.awssdk.services.acmpca.model.LockoutPreventedException;
import software.amazon.awssdk.services.acmpca.model.MalformedCertificateException;
import software.amazon.awssdk.services.acmpca.model.MalformedCsrException;
import software.amazon.awssdk.services.acmpca.model.PermissionAlreadyExistsException;
import software.amazon.awssdk.services.acmpca.model.PutPolicyRequest;
import software.amazon.awssdk.services.acmpca.model.PutPolicyResponse;
import software.amazon.awssdk.services.acmpca.model.RequestAlreadyProcessedException;
import software.amazon.awssdk.services.acmpca.model.RequestFailedException;
import software.amazon.awssdk.services.acmpca.model.RequestInProgressException;
import software.amazon.awssdk.services.acmpca.model.ResourceNotFoundException;
import software.amazon.awssdk.services.acmpca.model.RestoreCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.RestoreCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.model.RevokeCertificateRequest;
import software.amazon.awssdk.services.acmpca.model.RevokeCertificateResponse;
import software.amazon.awssdk.services.acmpca.model.TagCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.TagCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.model.TooManyTagsException;
import software.amazon.awssdk.services.acmpca.model.UntagCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.UntagCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.model.UpdateCertificateAuthorityRequest;
import software.amazon.awssdk.services.acmpca.model.UpdateCertificateAuthorityResponse;
import software.amazon.awssdk.services.acmpca.paginators.ListCertificateAuthoritiesPublisher;
import software.amazon.awssdk.services.acmpca.paginators.ListPermissionsPublisher;
import software.amazon.awssdk.services.acmpca.paginators.ListTagsPublisher;
import software.amazon.awssdk.services.acmpca.transform.CreateCertificateAuthorityAuditReportRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.CreateCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.CreatePermissionRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.DeleteCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.DeletePermissionRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.DeletePolicyRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.DescribeCertificateAuthorityAuditReportRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.DescribeCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.GetCertificateAuthorityCertificateRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.GetCertificateAuthorityCsrRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.GetCertificateRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.GetPolicyRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.ImportCertificateAuthorityCertificateRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.IssueCertificateRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.ListCertificateAuthoritiesRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.ListPermissionsRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.PutPolicyRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.RestoreCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.RevokeCertificateRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.TagCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.UntagCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.services.acmpca.transform.UpdateCertificateAuthorityRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link AcmPcaAsyncClient}.
 *
 * @see AcmPcaAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultAcmPcaAsyncClient implements AcmPcaAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultAcmPcaAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultAcmPcaAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Creates a root or subordinate private certificate authority (CA). You must specify the CA configuration, the
     * certificate revocation list (CRL) configuration, the CA type, and an optional idempotency token to avoid
     * accidental creation of multiple CAs. The CA configuration specifies the name of the algorithm and key size to be
     * used to create the CA private key, the type of signing algorithm that the CA uses, and X.500 subject information.
     * The CRL configuration specifies the CRL expiration period in days (the validity period of the CRL), the Amazon S3
     * bucket that will contain the CRL, and a CNAME alias for the S3 bucket that is included in certificates issued by
     * the CA. If successful, this action returns the Amazon Resource Name (ARN) of the CA.
     * </p>
     * <p>
     * ACM Private CAA assets that are stored in Amazon S3 can be protected with encryption. For more information, see
     * <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCreateCa.html#crl-encryption">Encrypting Your
     * CRLs</a>.
     * </p>
     * <note>
     * <p>
     * Both PCA and the IAM principal must have permission to write to the S3 bucket that you specify. If the IAM
     * principal making the call does not have permission to write to the bucket, then an exception is thrown. For more
     * information, see <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaAuthAccess.html">Configure
     * Access to ACM Private CA</a>.
     * </p>
     * </note>
     *
     * @param createCertificateAuthorityRequest
     * @return A Java Future containing the result of the CreateCertificateAuthority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArgsException One or more of the specified arguments was not valid.</li>
     *         <li>InvalidPolicyException The resource policy is invalid or is missing a required statement. For general
     *         information about IAM policy and statement structure, see <a href=
     *         "https://docs.aws.amazon.com/https:/docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policies-json"
     *         >Overview of JSON Policies</a>.</li>
     *         <li>InvalidTagException The tag associated with the CA is not valid. The invalid argument is contained in
     *         the message field.</li>
     *         <li>LimitExceededException An ACM Private CA quota has been exceeded. See the exception message returned
     *         to determine the quota that was exceeded.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.CreateCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/CreateCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCertificateAuthorityResponse> createCertificateAuthority(
            CreateCertificateAuthorityRequest createCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCertificateAuthorityRequest, CreateCertificateAuthorityResponse>()
                            .withOperationName("CreateCertificateAuthority")
                            .withMarshaller(new CreateCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<CreateCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an audit report that lists every time that your CA private key is used. The report is saved in the Amazon
     * S3 bucket that you specify on input. The <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html">IssueCertificate</a> and
     * <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html">RevokeCertificate</a>
     * actions use the private key.
     * </p>
     * <note>
     * <p>
     * Both PCA and the IAM principal must have permission to write to the S3 bucket that you specify. If the IAM
     * principal making the call does not have permission to write to the bucket, then an exception is thrown. For more
     * information, see <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaAuthAccess.html">Configure
     * Access to ACM Private CA</a>.
     * </p>
     * </note>
     * <p>
     * ACM Private CAA assets that are stored in Amazon S3 can be protected with encryption. For more information, see
     * <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaAuditReport.html#audit-report-encryption">
     * Encrypting Your Audit Reports</a>.
     * </p>
     *
     * @param createCertificateAuthorityAuditReportRequest
     * @return A Java Future containing the result of the CreateCertificateAuthorityAuditReport operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RequestInProgressException Your request is already in progress.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidArgsException One or more of the specified arguments was not valid.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.CreateCertificateAuthorityAuditReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/CreateCertificateAuthorityAuditReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCertificateAuthorityAuditReportResponse> createCertificateAuthorityAuditReport(
            CreateCertificateAuthorityAuditReportRequest createCertificateAuthorityAuditReportRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createCertificateAuthorityAuditReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCertificateAuthorityAuditReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCertificateAuthorityAuditReportResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateCertificateAuthorityAuditReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreateCertificateAuthorityAuditReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCertificateAuthorityAuditReportRequest, CreateCertificateAuthorityAuditReportResponse>()
                            .withOperationName("CreateCertificateAuthorityAuditReport")
                            .withMarshaller(new CreateCertificateAuthorityAuditReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createCertificateAuthorityAuditReportRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createCertificateAuthorityAuditReportRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<CreateCertificateAuthorityAuditReportResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Grants one or more permissions on a private CA to the AWS Certificate Manager (ACM) service principal (
     * <code>acm.amazonaws.com</code>). These permissions allow ACM to issue and renew ACM certificates that reside in
     * the same AWS account as the CA.
     * </p>
     * <p>
     * You can list current permissions with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html">ListPermissions</a>
     * action and revoke them with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html">DeletePermission</a>
     * action.
     * </p>
     * <p class="title">
     * <b>About Permissions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the private CA and the certificates it issues reside in the same account, you can use
     * <code>CreatePermission</code> to grant permissions for ACM to carry out automatic certificate renewals.
     * </p>
     * </li>
     * <li>
     * <p>
     * For automatic certificate renewal to succeed, the ACM service principal needs permissions to create, retrieve,
     * and list certificates.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the private CA and the ACM certificates reside in different accounts, then permissions cannot be used to
     * enable automatic renewals. Instead, the ACM certificate owner must set up a resource-based policy to enable
     * cross-account issuance and renewals. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param createPermissionRequest
     * @return A Java Future containing the result of the CreatePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>PermissionAlreadyExistsException The designated permission has already been given to the user.</li>
     *         <li>LimitExceededException An ACM Private CA quota has been exceeded. See the exception message returned
     *         to determine the quota that was exceeded.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.CreatePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/CreatePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<CreatePermissionResponse> createPermission(CreatePermissionRequest createPermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createPermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreatePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreatePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreatePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<CreatePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreatePermissionRequest, CreatePermissionResponse>()
                            .withOperationName("CreatePermission")
                            .withMarshaller(new CreatePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(createPermissionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = createPermissionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<CreatePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a private certificate authority (CA). You must provide the Amazon Resource Name (ARN) of the private CA
     * that you want to delete. You can find the ARN by calling the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html"
     * >ListCertificateAuthorities</a> action.
     * </p>
     * <note>
     * <p>
     * Deleting a CA will invalidate other CAs and certificates below it in your CA hierarchy.
     * </p>
     * </note>
     * <p>
     * Before you can delete a CA that you have created and activated, you must disable it. To do this, call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html">
     * UpdateCertificateAuthority</a> action and set the <b>CertificateAuthorityStatus</b> parameter to
     * <code>DISABLED</code>.
     * </p>
     * <p>
     * Additionally, you can delete a CA if you are waiting for it to be created (that is, the status of the CA is
     * <code>CREATING</code>). You can also delete it if the CA has been created but you haven't yet imported the signed
     * certificate into ACM Private CA (that is, the status of the CA is <code>PENDING_CERTIFICATE</code>).
     * </p>
     * <p>
     * When you successfully call <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html"
     * >DeleteCertificateAuthority</a>, the CA's status changes to <code>DELETED</code>. However, the CA won't be
     * permanently deleted until the restoration period has passed. By default, if you do not set the
     * <code>PermanentDeletionTimeInDays</code> parameter, the CA remains restorable for 30 days. You can set the
     * parameter from 7 to 30 days. The <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html"
     * >DescribeCertificateAuthority</a> action returns the time remaining in the restoration window of a private CA in
     * the <code>DELETED</code> state. To restore an eligible CA, call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RestoreCertificateAuthority.html"
     * >RestoreCertificateAuthority</a> action.
     * </p>
     *
     * @param deleteCertificateAuthorityRequest
     * @return A Java Future containing the result of the DeleteCertificateAuthority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException A previous update to your private CA is still ongoing.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.DeleteCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/DeleteCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCertificateAuthorityResponse> deleteCertificateAuthority(
            DeleteCertificateAuthorityRequest deleteCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeleteCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCertificateAuthorityRequest, DeleteCertificateAuthorityResponse>()
                            .withOperationName("DeleteCertificateAuthority")
                            .withMarshaller(new DeleteCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deleteCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deleteCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DeleteCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes permissions on a private CA granted to the AWS Certificate Manager (ACM) service principal
     * (acm.amazonaws.com).
     * </p>
     * <p>
     * These permissions allow ACM to issue and renew ACM certificates that reside in the same AWS account as the CA. If
     * you revoke these permissions, ACM will no longer renew the affected certificates automatically.
     * </p>
     * <p>
     * Permissions can be granted with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html">CreatePermission</a>
     * action and listed with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListPermissions.html">ListPermissions</a>
     * action.
     * </p>
     * <p class="title">
     * <b>About Permissions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the private CA and the certificates it issues reside in the same account, you can use
     * <code>CreatePermission</code> to grant permissions for ACM to carry out automatic certificate renewals.
     * </p>
     * </li>
     * <li>
     * <p>
     * For automatic certificate renewal to succeed, the ACM service principal needs permissions to create, retrieve,
     * and list certificates.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the private CA and the ACM certificates reside in different accounts, then permissions cannot be used to
     * enable automatic renewals. Instead, the ACM certificate owner must set up a resource-based policy to enable
     * cross-account issuance and renewals. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param deletePermissionRequest
     * @return A Java Future containing the result of the DeletePermission operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.DeletePermission
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/DeletePermission" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePermissionResponse> deletePermission(DeletePermissionRequest deletePermissionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePermissionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePermission");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePermissionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeletePermissionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePermissionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePermissionRequest, DeletePermissionResponse>()
                            .withOperationName("DeletePermission")
                            .withMarshaller(new DeletePermissionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePermissionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePermissionRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeletePermissionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the resource-based policy attached to a private CA. Deletion will remove any access that the policy has
     * granted. If there is no policy attached to the private CA, this action will return successful.
     * </p>
     * <p>
     * If you delete a policy that was applied through AWS Resource Access Manager (RAM), the CA will be removed from
     * all shares in which it was included.
     * </p>
     * <p>
     * The AWS Certificate Manager Service Linked Role that the policy supports is not affected when you delete the
     * policy.
     * </p>
     * <p>
     * The current policy can be shown with <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html">GetPolicy</a> and updated with
     * <a href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html">PutPolicy</a>.
     * </p>
     * <p class="title">
     * <b>About Policies</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A policy grants access on a private CA to an AWS customer account, to AWS Organizations, or to an AWS
     * Organizations unit. Policies are under the control of a CA administrator. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A policy permits a user of AWS Certificate Manager (ACM) to issue ACM certificates signed by a CA in another
     * account.
     * </p>
     * </li>
     * <li>
     * <p>
     * For ACM to manage automatic renewal of these certificates, the ACM user must configure a Service Linked Role
     * (SLR). The SLR allows the ACM service to assume the identity of the user, subject to confirmation against the ACM
     * Private CA policy. For more information, see <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide/acm-slr.html">Using a Service Linked Role with ACM</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Updates made in AWS Resource Manager (RAM) are reflected in policies. For more information, see <a
     * href="acm-pca/latest/userguide/pca-ram.html">Using AWS Resource Access Manager (RAM) with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param deletePolicyRequest
     * @return A Java Future containing the result of the DeletePolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException A previous update to your private CA is still ongoing.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>LockoutPreventedException The current action was prevented because it would lock the caller out from
     *         performing subsequent actions. Verify that the specified parameters would not result in the caller being
     *         denied access to the resource.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.DeletePolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/DeletePolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<DeletePolicyResponse> deletePolicy(DeletePolicyRequest deletePolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deletePolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeletePolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeletePolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DeletePolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DeletePolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeletePolicyRequest, DeletePolicyResponse>()
                            .withOperationName("DeletePolicy").withMarshaller(new DeletePolicyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(deletePolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = deletePolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DeletePolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about your private certificate authority (CA) or one that has been shared with you. You specify
     * the private CA on input by its ARN (Amazon Resource Name). The output contains the status of your CA. This can be
     * any of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>CREATING</code> - ACM Private CA is creating your private certificate authority.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PENDING_CERTIFICATE</code> - The certificate is pending. You must use your ACM Private CA-hosted or
     * on-premises root or subordinate CA to sign your private CA CSR and then import it into PCA.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ACTIVE</code> - Your private CA is active.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DISABLED</code> - Your private CA has been disabled.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EXPIRED</code> - Your private CA certificate has expired.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FAILED</code> - Your private CA has failed. Your CA can fail because of problems such a network outage or
     * backend AWS failure or other errors. A failed CA can never return to the pending state. You must create a new CA.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DELETED</code> - Your private CA is within the restoration period, after which it is permanently deleted.
     * The length of time remaining in the CA's restoration period is also included in this action's output.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeCertificateAuthorityRequest
     * @return A Java Future containing the result of the DescribeCertificateAuthority operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.DescribeCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/DescribeCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCertificateAuthorityResponse> describeCertificateAuthority(
            DescribeCertificateAuthorityRequest describeCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCertificateAuthorityRequest, DescribeCertificateAuthorityResponse>()
                            .withOperationName("DescribeCertificateAuthority")
                            .withMarshaller(new DescribeCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists information about a specific audit report created by calling the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html"
     * >CreateCertificateAuthorityAuditReport</a> action. Audit information is created every time the certificate
     * authority (CA) private key is used. The private key is used when you call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html">IssueCertificate</a>
     * action or the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_RevokeCertificate.html">RevokeCertificate</a>
     * action.
     * </p>
     *
     * @param describeCertificateAuthorityAuditReportRequest
     * @return A Java Future containing the result of the DescribeCertificateAuthorityAuditReport operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidArgsException One or more of the specified arguments was not valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.DescribeCertificateAuthorityAuditReport
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/DescribeCertificateAuthorityAuditReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeCertificateAuthorityAuditReportResponse> describeCertificateAuthorityAuditReport(
            DescribeCertificateAuthorityAuditReportRequest describeCertificateAuthorityAuditReportRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeCertificateAuthorityAuditReportRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeCertificateAuthorityAuditReport");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeCertificateAuthorityAuditReportResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeCertificateAuthorityAuditReportResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeCertificateAuthorityAuditReportResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeCertificateAuthorityAuditReportRequest, DescribeCertificateAuthorityAuditReportResponse>()
                            .withOperationName("DescribeCertificateAuthorityAuditReport")
                            .withMarshaller(new DescribeCertificateAuthorityAuditReportRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector)
                            .withInput(describeCertificateAuthorityAuditReportRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeCertificateAuthorityAuditReportRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeCertificateAuthorityAuditReportResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves a certificate from your private CA or one that has been shared with you. The ARN of the certificate is
     * returned when you call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_IssueCertificate.html">IssueCertificate</a>
     * action. You must specify both the ARN of your private CA and the ARN of the issued certificate when calling the
     * <b>GetCertificate</b> action. You can retrieve the certificate if it is in the <b>ISSUED</b> state. You can call
     * the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html"
     * >CreateCertificateAuthorityAuditReport</a> action to create a report that contains information about all of the
     * certificates issued and revoked by your private CA.
     * </p>
     *
     * @param getCertificateRequest
     * @return A Java Future containing the result of the GetCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RequestInProgressException Your request is already in progress.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.GetCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/GetCertificate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCertificateResponse> getCertificate(GetCertificateRequest getCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCertificateRequest, GetCertificateResponse>()
                            .withOperationName("GetCertificate")
                            .withMarshaller(new GetCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCertificateRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the certificate and certificate chain for your private certificate authority (CA) or one that has been
     * shared with you. Both the certificate and the chain are base64 PEM-encoded. The chain does not include the CA
     * certificate. Each certificate in the chain signs the one before it.
     * </p>
     *
     * @param getCertificateAuthorityCertificateRequest
     * @return A Java Future containing the result of the GetCertificateAuthorityCertificate operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.GetCertificateAuthorityCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/GetCertificateAuthorityCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCertificateAuthorityCertificateResponse> getCertificateAuthorityCertificate(
            GetCertificateAuthorityCertificateRequest getCertificateAuthorityCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getCertificateAuthorityCertificateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCertificateAuthorityCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCertificateAuthorityCertificateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetCertificateAuthorityCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCertificateAuthorityCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCertificateAuthorityCertificateRequest, GetCertificateAuthorityCertificateResponse>()
                            .withOperationName("GetCertificateAuthorityCertificate")
                            .withMarshaller(new GetCertificateAuthorityCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCertificateAuthorityCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCertificateAuthorityCertificateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetCertificateAuthorityCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the certificate signing request (CSR) for your private certificate authority (CA). The CSR is created
     * when you call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a> action. Sign the CSR with your ACM Private CA-hosted or on-premises root or
     * subordinate CA. Then import the signed certificate back into ACM Private CA by calling the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html"
     * >ImportCertificateAuthorityCertificate</a> action. The CSR is returned as a base64 PEM-encoded string.
     * </p>
     *
     * @param getCertificateAuthorityCsrRequest
     * @return A Java Future containing the result of the GetCertificateAuthorityCsr operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>RequestInProgressException Your request is already in progress.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.GetCertificateAuthorityCsr
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/GetCertificateAuthorityCsr"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetCertificateAuthorityCsrResponse> getCertificateAuthorityCsr(
            GetCertificateAuthorityCsrRequest getCertificateAuthorityCsrRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCertificateAuthorityCsrRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCertificateAuthorityCsr");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCertificateAuthorityCsrResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCertificateAuthorityCsrResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetCertificateAuthorityCsrResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCertificateAuthorityCsrRequest, GetCertificateAuthorityCsrResponse>()
                            .withOperationName("GetCertificateAuthorityCsr")
                            .withMarshaller(new GetCertificateAuthorityCsrRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getCertificateAuthorityCsrRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getCertificateAuthorityCsrRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetCertificateAuthorityCsrResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Retrieves the resource-based policy attached to a private CA. If either the private CA resource or the policy
     * cannot be found, this action returns a <code>ResourceNotFoundException</code>.
     * </p>
     * <p>
     * The policy can be attached or updated with <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_PutPolicy.html">PutPolicy</a> and removed with
     * <a href="acm-pca/latest/APIReference/API_DeletePolicy.html">DeletePolicy</a>.
     * </p>
     * <p class="title">
     * <b>About Policies</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A policy grants access on a private CA to an AWS customer account, to AWS Organizations, or to an AWS
     * Organizations unit. Policies are under the control of a CA administrator. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A policy permits a user of AWS Certificate Manager (ACM) to issue ACM certificates signed by a CA in another
     * account.
     * </p>
     * </li>
     * <li>
     * <p>
     * For ACM to manage automatic renewal of these certificates, the ACM user must configure a Service Linked Role
     * (SLR). The SLR allows the ACM service to assume the identity of the user, subject to confirmation against the ACM
     * Private CA policy. For more information, see <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide/acm-slr.html">Using a Service Linked Role with ACM</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Updates made in AWS Resource Manager (RAM) are reflected in policies. For more information, see <a
     * href="acm-pca/latest/userguide/pca-ram.html">Using AWS Resource Access Manager (RAM) with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param getPolicyRequest
     * @return A Java Future containing the result of the GetPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.GetPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/GetPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetPolicyResponse> getPolicy(GetPolicyRequest getPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    GetPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetPolicyRequest, GetPolicyResponse>().withOperationName("GetPolicy")
                            .withMarshaller(new GetPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(getPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<GetPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Imports a signed private CA certificate into ACM Private CA. This action is used when you are using a chain of
     * trust whose root is located outside ACM Private CA. Before you can call this action, the following preparations
     * must in place:
     * </p>
     * <ol>
     * <li>
     * <p>
     * In ACM Private CA, call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a> action to create the private CA that that you plan to back with the imported
     * certificate.
     * </p>
     * </li>
     * <li>
     * <p>
     * Call the <a href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificateAuthorityCsr.html">
     * GetCertificateAuthorityCsr</a> action to generate a certificate signing request (CSR).
     * </p>
     * </li>
     * <li>
     * <p>
     * Sign the CSR using a root or intermediate CA hosted by either an on-premises PKI hierarchy or by a commercial CA.
     * </p>
     * </li>
     * <li>
     * <p>
     * Create a certificate chain and copy the signed certificate and the certificate chain to your working directory.
     * </p>
     * </li>
     * </ol>
     * <p>
     * The following requirements apply when you import a CA certificate.
     * </p>
     * <ul>
     * <li>
     * <p>
     * You cannot import a non-self-signed certificate for use as a root CA.
     * </p>
     * </li>
     * <li>
     * <p>
     * You cannot import a self-signed certificate for use as a subordinate CA.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your certificate chain must not include the private CA certificate that you are importing.
     * </p>
     * </li>
     * <li>
     * <p>
     * Your ACM Private CA-hosted or on-premises CA certificate must be the last certificate in your chain. The
     * subordinate certificate, if any, that your root CA signed must be next to last. The subordinate certificate
     * signed by the preceding subordinate CA must come next, and so on until your chain is built.
     * </p>
     * </li>
     * <li>
     * <p>
     * The chain must be PEM-encoded.
     * </p>
     * </li>
     * <li>
     * <p>
     * The maximum allowed size of a certificate is 32 KB.
     * </p>
     * </li>
     * <li>
     * <p>
     * The maximum allowed size of a certificate chain is 2 MB.
     * </p>
     * </li>
     * </ul>
     * <p>
     * <i>Enforcement of Critical Constraints</i>
     * </p>
     * <p>
     * ACM Private CA allows the following extensions to be marked critical in the imported CA certificate or chain.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Basic constraints (<i>must</i> be marked critical)
     * </p>
     * </li>
     * <li>
     * <p>
     * Subject alternative names
     * </p>
     * </li>
     * <li>
     * <p>
     * Key usage
     * </p>
     * </li>
     * <li>
     * <p>
     * Extended key usage
     * </p>
     * </li>
     * <li>
     * <p>
     * Authority key identifier
     * </p>
     * </li>
     * <li>
     * <p>
     * Subject key identifier
     * </p>
     * </li>
     * <li>
     * <p>
     * Issuer alternative name
     * </p>
     * </li>
     * <li>
     * <p>
     * Subject directory attributes
     * </p>
     * </li>
     * <li>
     * <p>
     * Subject information access
     * </p>
     * </li>
     * <li>
     * <p>
     * Certificate policies
     * </p>
     * </li>
     * <li>
     * <p>
     * Policy mappings
     * </p>
     * </li>
     * <li>
     * <p>
     * Inhibit anyPolicy
     * </p>
     * </li>
     * </ul>
     * <p>
     * ACM Private CA rejects the following extensions when they are marked critical in an imported CA certificate or
     * chain.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Name constraints
     * </p>
     * </li>
     * <li>
     * <p>
     * Policy constraints
     * </p>
     * </li>
     * <li>
     * <p>
     * CRL distribution points
     * </p>
     * </li>
     * <li>
     * <p>
     * Authority information access
     * </p>
     * </li>
     * <li>
     * <p>
     * Freshest CRL
     * </p>
     * </li>
     * <li>
     * <p>
     * Any other extension
     * </p>
     * </li>
     * </ul>
     *
     * @param importCertificateAuthorityCertificateRequest
     * @return A Java Future containing the result of the ImportCertificateAuthorityCertificate operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException A previous update to your private CA is still ongoing.</li>
     *         <li>RequestInProgressException Your request is already in progress.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidRequestException The request action cannot be performed or is prohibited.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>MalformedCertificateException One or more fields in the certificate are invalid.</li>
     *         <li>CertificateMismatchException The certificate authority certificate you are importing does not comply
     *         with conditions specified in the certificate that signed it.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ImportCertificateAuthorityCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ImportCertificateAuthorityCertificate"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ImportCertificateAuthorityCertificateResponse> importCertificateAuthorityCertificate(
            ImportCertificateAuthorityCertificateRequest importCertificateAuthorityCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                importCertificateAuthorityCertificateRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ImportCertificateAuthorityCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ImportCertificateAuthorityCertificateResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ImportCertificateAuthorityCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ImportCertificateAuthorityCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ImportCertificateAuthorityCertificateRequest, ImportCertificateAuthorityCertificateResponse>()
                            .withOperationName("ImportCertificateAuthorityCertificate")
                            .withMarshaller(new ImportCertificateAuthorityCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(importCertificateAuthorityCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = importCertificateAuthorityCertificateRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ImportCertificateAuthorityCertificateResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Uses your private certificate authority (CA), or one that has been shared with you, to issue a client
     * certificate. This action returns the Amazon Resource Name (ARN) of the certificate. You can retrieve the
     * certificate by calling the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetCertificate.html">GetCertificate</a> action
     * and specifying the ARN.
     * </p>
     * <note>
     * <p>
     * You cannot use the ACM <b>ListCertificateAuthorities</b> action to retrieve the ARNs of the certificates that you
     * issue by using ACM Private CA.
     * </p>
     * </note>
     *
     * @param issueCertificateRequest
     * @return A Java Future containing the result of the IssueCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>LimitExceededException An ACM Private CA quota has been exceeded. See the exception message returned
     *         to determine the quota that was exceeded.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidArgsException One or more of the specified arguments was not valid.</li>
     *         <li>MalformedCsrException The certificate signing request is invalid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.IssueCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/IssueCertificate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<IssueCertificateResponse> issueCertificate(IssueCertificateRequest issueCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, issueCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "IssueCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<IssueCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, IssueCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<IssueCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<IssueCertificateRequest, IssueCertificateResponse>()
                            .withOperationName("IssueCertificate")
                            .withMarshaller(new IssueCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(issueCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = issueCertificateRequest.overrideConfiguration().orElse(null);
            CompletableFuture<IssueCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the private certificate authorities that you created by using the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a> action.
     * </p>
     *
     * @param listCertificateAuthoritiesRequest
     * @return A Java Future containing the result of the ListCertificateAuthorities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token specified in the <code>NextToken</code> argument is not valid.
     *         Use the token returned from your previous call to <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html"
     *         >ListCertificateAuthorities</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ListCertificateAuthorities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ListCertificateAuthorities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListCertificateAuthoritiesResponse> listCertificateAuthorities(
            ListCertificateAuthoritiesRequest listCertificateAuthoritiesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCertificateAuthoritiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCertificateAuthorities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCertificateAuthoritiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCertificateAuthoritiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListCertificateAuthoritiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCertificateAuthoritiesRequest, ListCertificateAuthoritiesResponse>()
                            .withOperationName("ListCertificateAuthorities")
                            .withMarshaller(new ListCertificateAuthoritiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listCertificateAuthoritiesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listCertificateAuthoritiesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListCertificateAuthoritiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the private certificate authorities that you created by using the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthority.html"
     * >CreateCertificateAuthority</a> action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listCertificateAuthorities(software.amazon.awssdk.services.acmpca.model.ListCertificateAuthoritiesRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.acmpca.paginators.ListCertificateAuthoritiesPublisher publisher = client.listCertificateAuthoritiesPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.acmpca.paginators.ListCertificateAuthoritiesPublisher publisher = client.listCertificateAuthoritiesPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.acmpca.model.ListCertificateAuthoritiesResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.acmpca.model.ListCertificateAuthoritiesResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listCertificateAuthorities(software.amazon.awssdk.services.acmpca.model.ListCertificateAuthoritiesRequest)}
     * operation.</b>
     * </p>
     *
     * @param listCertificateAuthoritiesRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InvalidNextTokenException The token specified in the <code>NextToken</code> argument is not valid.
     *         Use the token returned from your previous call to <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html"
     *         >ListCertificateAuthorities</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ListCertificateAuthorities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ListCertificateAuthorities"
     *      target="_top">AWS API Documentation</a>
     */
    public ListCertificateAuthoritiesPublisher listCertificateAuthoritiesPaginator(
            ListCertificateAuthoritiesRequest listCertificateAuthoritiesRequest) {
        return new ListCertificateAuthoritiesPublisher(this, applyPaginatorUserAgent(listCertificateAuthoritiesRequest));
    }

    /**
     * <p>
     * List all permissions on a private CA, if any, granted to the AWS Certificate Manager (ACM) service principal
     * (acm.amazonaws.com).
     * </p>
     * <p>
     * These permissions allow ACM to issue and renew ACM certificates that reside in the same AWS account as the CA.
     * </p>
     * <p>
     * Permissions can be granted with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html">CreatePermission</a>
     * action and revoked with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html">DeletePermission</a>
     * action.
     * </p>
     * <p class="title">
     * <b>About Permissions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the private CA and the certificates it issues reside in the same account, you can use
     * <code>CreatePermission</code> to grant permissions for ACM to carry out automatic certificate renewals.
     * </p>
     * </li>
     * <li>
     * <p>
     * For automatic certificate renewal to succeed, the ACM service principal needs permissions to create, retrieve,
     * and list certificates.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the private CA and the ACM certificates reside in different accounts, then permissions cannot be used to
     * enable automatic renewals. Instead, the ACM certificate owner must set up a resource-based policy to enable
     * cross-account issuance and renewals. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param listPermissionsRequest
     * @return A Java Future containing the result of the ListPermissions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidNextTokenException The token specified in the <code>NextToken</code> argument is not valid.
     *         Use the token returned from your previous call to <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html"
     *         >ListCertificateAuthorities</a>.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ListPermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListPermissionsResponse> listPermissions(ListPermissionsRequest listPermissionsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listPermissionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListPermissions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListPermissionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListPermissionsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListPermissionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListPermissionsRequest, ListPermissionsResponse>()
                            .withOperationName("ListPermissions")
                            .withMarshaller(new ListPermissionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listPermissionsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listPermissionsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListPermissionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all permissions on a private CA, if any, granted to the AWS Certificate Manager (ACM) service principal
     * (acm.amazonaws.com).
     * </p>
     * <p>
     * These permissions allow ACM to issue and renew ACM certificates that reside in the same AWS account as the CA.
     * </p>
     * <p>
     * Permissions can be granted with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreatePermission.html">CreatePermission</a>
     * action and revoked with the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePermission.html">DeletePermission</a>
     * action.
     * </p>
     * <p class="title">
     * <b>About Permissions</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * If the private CA and the certificates it issues reside in the same account, you can use
     * <code>CreatePermission</code> to grant permissions for ACM to carry out automatic certificate renewals.
     * </p>
     * </li>
     * <li>
     * <p>
     * For automatic certificate renewal to succeed, the ACM service principal needs permissions to create, retrieve,
     * and list certificates.
     * </p>
     * </li>
     * <li>
     * <p>
     * If the private CA and the ACM certificates reside in different accounts, then permissions cannot be used to
     * enable automatic renewals. Instead, the ACM certificate owner must set up a resource-based policy to enable
     * cross-account issuance and renewals. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listPermissions(software.amazon.awssdk.services.acmpca.model.ListPermissionsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.acmpca.paginators.ListPermissionsPublisher publisher = client.listPermissionsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.acmpca.paginators.ListPermissionsPublisher publisher = client.listPermissionsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.acmpca.model.ListPermissionsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.acmpca.model.ListPermissionsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listPermissions(software.amazon.awssdk.services.acmpca.model.ListPermissionsRequest)} operation.</b>
     * </p>
     *
     * @param listPermissionsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidNextTokenException The token specified in the <code>NextToken</code> argument is not valid.
     *         Use the token returned from your previous call to <a
     *         href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html"
     *         >ListCertificateAuthorities</a>.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ListPermissions
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ListPermissions" target="_top">AWS API
     *      Documentation</a>
     */
    public ListPermissionsPublisher listPermissionsPaginator(ListPermissionsRequest listPermissionsRequest) {
        return new ListPermissionsPublisher(this, applyPaginatorUserAgent(listPermissionsRequest));
    }

    /**
     * <p>
     * Lists the tags, if any, that are associated with your private CA or one that has been shared with you. Tags are
     * labels that you can use to identify and organize your CAs. Each tag consists of a key and an optional value. Call
     * the <a href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html">
     * TagCertificateAuthority</a> action to add one or more tags to your CA. Call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html"
     * >UntagCertificateAuthority</a> action to remove tags.
     * </p>
     *
     * @param listTagsRequest
     * @return A Java Future containing the result of the ListTags operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ListTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsResponse> listTags(ListTagsRequest listTagsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListTagsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListTagsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>().withOperationName("ListTags")
                            .withMarshaller(new ListTagsRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listTagsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListTagsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists the tags, if any, that are associated with your private CA or one that has been shared with you. Tags are
     * labels that you can use to identify and organize your CAs. Each tag consists of a key and an optional value. Call
     * the <a href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html">
     * TagCertificateAuthority</a> action to add one or more tags to your CA. Call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html"
     * >UntagCertificateAuthority</a> action to remove tags.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTags(software.amazon.awssdk.services.acmpca.model.ListTagsRequest)} operation.
     * The return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.acmpca.paginators.ListTagsPublisher publisher = client.listTagsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.acmpca.paginators.ListTagsPublisher publisher = client.listTagsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.acmpca.model.ListTagsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.acmpca.model.ListTagsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTags(software.amazon.awssdk.services.acmpca.model.ListTagsRequest)} operation.</b>
     * </p>
     *
     * @param listTagsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.ListTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    public ListTagsPublisher listTagsPaginator(ListTagsRequest listTagsRequest) {
        return new ListTagsPublisher(this, applyPaginatorUserAgent(listTagsRequest));
    }

    /**
     * <p>
     * Attaches a resource-based policy to a private CA.
     * </p>
     * <p>
     * A policy can also be applied by <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/pca-ram.html">sharing</a> a private CA through AWS
     * Resource Access Manager (RAM).
     * </p>
     * <p>
     * The policy can be displayed with <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_GetPolicy.html">GetPolicy</a> and removed with
     * <a href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeletePolicy.html">DeletePolicy</a>.
     * </p>
     * <p class="title">
     * <b>About Policies</b>
     * </p>
     * <ul>
     * <li>
     * <p>
     * A policy grants access on a private CA to an AWS customer account, to AWS Organizations, or to an AWS
     * Organizations unit. Policies are under the control of a CA administrator. For more information, see <a
     * href="acm-pca/latest/userguide/pca-rbp.html">Using a Resource Based Policy with ACM Private CA</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * A policy permits a user of AWS Certificate Manager (ACM) to issue ACM certificates signed by a CA in another
     * account.
     * </p>
     * </li>
     * <li>
     * <p>
     * For ACM to manage automatic renewal of these certificates, the ACM user must configure a Service Linked Role
     * (SLR). The SLR allows the ACM service to assume the identity of the user, subject to confirmation against the ACM
     * Private CA policy. For more information, see <a
     * href="https://docs.aws.amazon.com/acm/latest/userguide/acm-slr.html">Using a Service Linked Role with ACM</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Updates made in AWS Resource Manager (RAM) are reflected in policies. For more information, see <a
     * href="acm-pca/latest/userguide/pca-ram.html">Using AWS Resource Access Manager (RAM) with ACM Private CA</a>.
     * </p>
     * </li>
     * </ul>
     *
     * @param putPolicyRequest
     * @return A Java Future containing the result of the PutPolicy operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException A previous update to your private CA is still ongoing.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidPolicyException The resource policy is invalid or is missing a required statement. For general
     *         information about IAM policy and statement structure, see <a href=
     *         "https://docs.aws.amazon.com/https:/docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policies-json"
     *         >Overview of JSON Policies</a>.</li>
     *         <li>LockoutPreventedException The current action was prevented because it would lock the caller out from
     *         performing subsequent actions. Verify that the specified parameters would not result in the caller being
     *         denied access to the resource.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.PutPolicy
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/PutPolicy" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutPolicyResponse> putPolicy(PutPolicyRequest putPolicyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putPolicyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutPolicy");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutPolicyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutPolicyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutPolicyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutPolicyRequest, PutPolicyResponse>().withOperationName("PutPolicy")
                            .withMarshaller(new PutPolicyRequestMarshaller(protocolFactory)).withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withMetricCollector(apiCallMetricCollector)
                            .withInput(putPolicyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putPolicyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutPolicyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Restores a certificate authority (CA) that is in the <code>DELETED</code> state. You can restore a CA during the
     * period that you defined in the <b>PermanentDeletionTimeInDays</b> parameter of the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DeleteCertificateAuthority.html"
     * >DeleteCertificateAuthority</a> action. Currently, you can specify 7 to 30 days. If you did not specify a
     * <b>PermanentDeletionTimeInDays</b> value, by default you can restore the CA at any time in a 30 day period. You
     * can check the time remaining in the restoration period of a private CA in the <code>DELETED</code> state by
     * calling the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_DescribeCertificateAuthority.html"
     * >DescribeCertificateAuthority</a> or <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListCertificateAuthorities.html"
     * >ListCertificateAuthorities</a> actions. The status of a restored CA is set to its pre-deletion status when the
     * <b>RestoreCertificateAuthority</b> action returns. To change its status to <code>ACTIVE</code>, call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UpdateCertificateAuthority.html"
     * >UpdateCertificateAuthority</a> action. If the private CA was in the <code>PENDING_CERTIFICATE</code> state at
     * deletion, you must use the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ImportCertificateAuthorityCertificate.html"
     * >ImportCertificateAuthorityCertificate</a> action to import a certificate authority into the private CA before it
     * can be activated. You cannot restore a CA after the restoration period has ended.
     * </p>
     *
     * @param restoreCertificateAuthorityRequest
     * @return A Java Future containing the result of the RestoreCertificateAuthority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.RestoreCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/RestoreCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RestoreCertificateAuthorityResponse> restoreCertificateAuthority(
            RestoreCertificateAuthorityRequest restoreCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RestoreCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RestoreCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RestoreCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RestoreCertificateAuthorityRequest, RestoreCertificateAuthorityResponse>()
                            .withOperationName("RestoreCertificateAuthority")
                            .withMarshaller(new RestoreCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(restoreCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = restoreCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RestoreCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Revokes a certificate that was issued inside ACM Private CA. If you enable a certificate revocation list (CRL)
     * when you create or update your private CA, information about the revoked certificates will be included in the
     * CRL. ACM Private CA writes the CRL to an S3 bucket that you specify. A CRL is typically updated approximately 30
     * minutes after a certificate is revoked. If for any reason the CRL update fails, ACM Private CA attempts makes
     * further attempts every 15 minutes. With Amazon CloudWatch, you can create alarms for the metrics
     * <code>CRLGenerated</code> and <code>MisconfiguredCRLBucket</code>. For more information, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaCloudWatch.html">Supported CloudWatch Metrics</a>.
     * </p>
     * <note>
     * <p>
     * Both PCA and the IAM principal must have permission to write to the S3 bucket that you specify. If the IAM
     * principal making the call does not have permission to write to the bucket, then an exception is thrown. For more
     * information, see <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaAuthAccess.html">Configure
     * Access to ACM Private CA</a>.
     * </p>
     * </note>
     * <p>
     * ACM Private CA also writes revocation information to the audit report. For more information, see <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_CreateCertificateAuthorityAuditReport.html"
     * >CreateCertificateAuthorityAuditReport</a>.
     * </p>
     * <note>
     * <p>
     * You cannot revoke a root CA self-signed certificate.
     * </p>
     * </note>
     *
     * @param revokeCertificateRequest
     * @return A Java Future containing the result of the RevokeCertificate operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException A previous update to your private CA is still ongoing.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidRequestException The request action cannot be performed or is prohibited.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>LimitExceededException An ACM Private CA quota has been exceeded. See the exception message returned
     *         to determine the quota that was exceeded.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>RequestAlreadyProcessedException Your request has already been completed.</li>
     *         <li>RequestInProgressException Your request is already in progress.</li>
     *         <li>RequestFailedException The request has failed for an unspecified reason.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.RevokeCertificate
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/RevokeCertificate" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<RevokeCertificateResponse> revokeCertificate(RevokeCertificateRequest revokeCertificateRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, revokeCertificateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RevokeCertificate");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RevokeCertificateResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RevokeCertificateResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RevokeCertificateResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RevokeCertificateRequest, RevokeCertificateResponse>()
                            .withOperationName("RevokeCertificate")
                            .withMarshaller(new RevokeCertificateRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(revokeCertificateRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = revokeCertificateRequest.overrideConfiguration().orElse(null);
            CompletableFuture<RevokeCertificateResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Adds one or more tags to your private CA. Tags are labels that you can use to identify and organize your AWS
     * resources. Each tag consists of a key and an optional value. You specify the private CA on input by its Amazon
     * Resource Name (ARN). You specify the tag by using a key-value pair. You can apply a tag to just one private CA if
     * you want to identify a specific characteristic of that CA, or you can apply the same tag to multiple private CAs
     * if you want to filter for a common relationship among those CAs. To remove one or more tags, use the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_UntagCertificateAuthority.html"
     * >UntagCertificateAuthority</a> action. Call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html">ListTags</a> action to see what
     * tags are associated with your CA.
     * </p>
     *
     * @param tagCertificateAuthorityRequest
     * @return A Java Future containing the result of the TagCertificateAuthority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidTagException The tag associated with the CA is not valid. The invalid argument is contained in
     *         the message field.</li>
     *         <li>TooManyTagsException You can associate up to 50 tags with a private CA. Exception information is
     *         contained in the exception message field.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.TagCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/TagCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<TagCertificateAuthorityResponse> tagCertificateAuthority(
            TagCertificateAuthorityRequest tagCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, TagCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<TagCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagCertificateAuthorityRequest, TagCertificateAuthorityResponse>()
                            .withOperationName("TagCertificateAuthority")
                            .withMarshaller(new TagCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(tagCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = tagCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<TagCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Remove one or more tags from your private CA. A tag consists of a key-value pair. If you do not specify the value
     * portion of the tag when calling this action, the tag will be removed regardless of value. If you specify a value,
     * the tag is removed only if it is associated with the specified value. To add tags to a private CA, use the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_TagCertificateAuthority.html">
     * TagCertificateAuthority</a>. Call the <a
     * href="https://docs.aws.amazon.com/acm-pca/latest/APIReference/API_ListTags.html">ListTags</a> action to see what
     * tags are associated with your CA.
     * </p>
     *
     * @param untagCertificateAuthorityRequest
     * @return A Java Future containing the result of the UntagCertificateAuthority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidTagException The tag associated with the CA is not valid. The invalid argument is contained in
     *         the message field.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.UntagCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/UntagCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UntagCertificateAuthorityResponse> untagCertificateAuthority(
            UntagCertificateAuthorityRequest untagCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UntagCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UntagCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagCertificateAuthorityRequest, UntagCertificateAuthorityResponse>()
                            .withOperationName("UntagCertificateAuthority")
                            .withMarshaller(new UntagCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(untagCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = untagCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UntagCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the status or configuration of a private certificate authority (CA). Your private CA must be in the
     * <code>ACTIVE</code> or <code>DISABLED</code> state before you can update it. You can disable a private CA that is
     * in the <code>ACTIVE</code> state or make a CA that is in the <code>DISABLED</code> state active again.
     * </p>
     * <note>
     * <p>
     * Both PCA and the IAM principal must have permission to write to the S3 bucket that you specify. If the IAM
     * principal making the call does not have permission to write to the bucket, then an exception is thrown. For more
     * information, see <a href="https://docs.aws.amazon.com/acm-pca/latest/userguide/PcaAuthAccess.html">Configure
     * Access to ACM Private CA</a>.
     * </p>
     * </note>
     *
     * @param updateCertificateAuthorityRequest
     * @return A Java Future containing the result of the UpdateCertificateAuthority operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>ConcurrentModificationException A previous update to your private CA is still ongoing.</li>
     *         <li>ResourceNotFoundException A resource such as a private CA, S3 bucket, certificate, audit report, or
     *         policy cannot be found.</li>
     *         <li>InvalidArgsException One or more of the specified arguments was not valid.</li>
     *         <li>InvalidArnException The requested Amazon Resource Name (ARN) does not refer to an existing resource.</li>
     *         <li>InvalidStateException The state of the private CA does not allow this action to occur.</li>
     *         <li>InvalidPolicyException The resource policy is invalid or is missing a required statement. For general
     *         information about IAM policy and statement structure, see <a href=
     *         "https://docs.aws.amazon.com/https:/docs.aws.amazon.com/IAM/latest/UserGuide/access_policies.html#access_policies-json"
     *         >Overview of JSON Policies</a>.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>AcmPcaException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample AcmPcaAsyncClient.UpdateCertificateAuthority
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/acm-pca-2017-08-22/UpdateCertificateAuthority"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCertificateAuthorityResponse> updateCertificateAuthority(
            UpdateCertificateAuthorityRequest updateCertificateAuthorityRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCertificateAuthorityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "ACM PCA");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCertificateAuthority");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCertificateAuthorityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCertificateAuthorityResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateCertificateAuthorityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCertificateAuthorityRequest, UpdateCertificateAuthorityResponse>()
                            .withOperationName("UpdateCertificateAuthority")
                            .withMarshaller(new UpdateCertificateAuthorityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateCertificateAuthorityRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateCertificateAuthorityRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateCertificateAuthorityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(AcmPcaException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidTagException")
                                .exceptionBuilderSupplier(InvalidTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CertificateMismatchException")
                                .exceptionBuilderSupplier(CertificateMismatchException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyTagsException")
                                .exceptionBuilderSupplier(TooManyTagsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedCSRException")
                                .exceptionBuilderSupplier(MalformedCsrException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidPolicyException")
                                .exceptionBuilderSupplier(InvalidPolicyException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgsException")
                                .exceptionBuilderSupplier(InvalidArgsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArnException")
                                .exceptionBuilderSupplier(InvalidArnException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestInProgressException")
                                .exceptionBuilderSupplier(RequestInProgressException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LockoutPreventedException")
                                .exceptionBuilderSupplier(LockoutPreventedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidStateException")
                                .exceptionBuilderSupplier(InvalidStateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("PermissionAlreadyExistsException")
                                .exceptionBuilderSupplier(PermissionAlreadyExistsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestAlreadyProcessedException")
                                .exceptionBuilderSupplier(RequestAlreadyProcessedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MalformedCertificateException")
                                .exceptionBuilderSupplier(MalformedCertificateException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("RequestFailedException")
                                .exceptionBuilderSupplier(RequestFailedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextTokenException")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends AcmPcaRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
