/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.account;

import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkPublicApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsClient;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.regions.ServiceMetadata;
import software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateRequest;
import software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateResponse;
import software.amazon.awssdk.services.account.model.AccessDeniedException;
import software.amazon.awssdk.services.account.model.AccountException;
import software.amazon.awssdk.services.account.model.ConflictException;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactRequest;
import software.amazon.awssdk.services.account.model.DeleteAlternateContactResponse;
import software.amazon.awssdk.services.account.model.DisableRegionRequest;
import software.amazon.awssdk.services.account.model.DisableRegionResponse;
import software.amazon.awssdk.services.account.model.EnableRegionRequest;
import software.amazon.awssdk.services.account.model.EnableRegionResponse;
import software.amazon.awssdk.services.account.model.GetAccountInformationRequest;
import software.amazon.awssdk.services.account.model.GetAccountInformationResponse;
import software.amazon.awssdk.services.account.model.GetAlternateContactRequest;
import software.amazon.awssdk.services.account.model.GetAlternateContactResponse;
import software.amazon.awssdk.services.account.model.GetContactInformationRequest;
import software.amazon.awssdk.services.account.model.GetContactInformationResponse;
import software.amazon.awssdk.services.account.model.GetPrimaryEmailRequest;
import software.amazon.awssdk.services.account.model.GetPrimaryEmailResponse;
import software.amazon.awssdk.services.account.model.GetRegionOptStatusRequest;
import software.amazon.awssdk.services.account.model.GetRegionOptStatusResponse;
import software.amazon.awssdk.services.account.model.InternalServerException;
import software.amazon.awssdk.services.account.model.ListRegionsRequest;
import software.amazon.awssdk.services.account.model.ListRegionsResponse;
import software.amazon.awssdk.services.account.model.PutAccountNameRequest;
import software.amazon.awssdk.services.account.model.PutAccountNameResponse;
import software.amazon.awssdk.services.account.model.PutAlternateContactRequest;
import software.amazon.awssdk.services.account.model.PutAlternateContactResponse;
import software.amazon.awssdk.services.account.model.PutContactInformationRequest;
import software.amazon.awssdk.services.account.model.PutContactInformationResponse;
import software.amazon.awssdk.services.account.model.ResourceNotFoundException;
import software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateRequest;
import software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateResponse;
import software.amazon.awssdk.services.account.model.TooManyRequestsException;
import software.amazon.awssdk.services.account.model.ValidationException;
import software.amazon.awssdk.services.account.paginators.ListRegionsIterable;

/**
 * Service client for accessing AWS Account. This can be created using the static {@link #builder()} method.
 *
 * <p>
 * Operations for Amazon Web Services Account Management
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
@SdkPublicApi
@ThreadSafe
public interface AccountClient extends AwsClient {
    String SERVICE_NAME = "account";

    /**
     * Value for looking up the service's metadata from the
     * {@link software.amazon.awssdk.regions.ServiceMetadataProvider}.
     */
    String SERVICE_METADATA_ID = "account";

    /**
     * <p>
     * Accepts the request that originated from <a>StartPrimaryEmailUpdate</a> to update the primary email address (also
     * known as the root user email address) for the specified account.
     * </p>
     *
     * @param acceptPrimaryEmailUpdateRequest
     * @return Result of the AcceptPrimaryEmailUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.AcceptPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/AcceptPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    default AcceptPrimaryEmailUpdateResponse acceptPrimaryEmailUpdate(
            AcceptPrimaryEmailUpdateRequest acceptPrimaryEmailUpdateRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ConflictException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Accepts the request that originated from <a>StartPrimaryEmailUpdate</a> to update the primary email address (also
     * known as the root user email address) for the specified account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link AcceptPrimaryEmailUpdateRequest.Builder} avoiding
     * the need to create one manually via {@link AcceptPrimaryEmailUpdateRequest#builder()}
     * </p>
     *
     * @param acceptPrimaryEmailUpdateRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.AcceptPrimaryEmailUpdateRequest.Builder} to create a
     *        request.
     * @return Result of the AcceptPrimaryEmailUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.AcceptPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/AcceptPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    default AcceptPrimaryEmailUpdateResponse acceptPrimaryEmailUpdate(
            Consumer<AcceptPrimaryEmailUpdateRequest.Builder> acceptPrimaryEmailUpdateRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ConflictException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        return acceptPrimaryEmailUpdate(AcceptPrimaryEmailUpdateRequest.builder().applyMutation(acceptPrimaryEmailUpdateRequest)
                .build());
    }

    /**
     * <p>
     * Deletes the specified alternate contact from an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param deleteAlternateContactRequest
     * @return Result of the DeleteAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DeleteAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DeleteAlternateContact"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteAlternateContactResponse deleteAlternateContact(DeleteAlternateContactRequest deleteAlternateContactRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Deletes the specified alternate contact from an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DeleteAlternateContactRequest.Builder} avoiding the
     * need to create one manually via {@link DeleteAlternateContactRequest#builder()}
     * </p>
     *
     * @param deleteAlternateContactRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.DeleteAlternateContactRequest.Builder} to create a
     *        request.
     * @return Result of the DeleteAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DeleteAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DeleteAlternateContact"
     *      target="_top">AWS API Documentation</a>
     */
    default DeleteAlternateContactResponse deleteAlternateContact(
            Consumer<DeleteAlternateContactRequest.Builder> deleteAlternateContactRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        return deleteAlternateContact(DeleteAlternateContactRequest.builder().applyMutation(deleteAlternateContactRequest)
                .build());
    }

    /**
     * <p>
     * Disables (opts-out) a particular Region for an account.
     * </p>
     * <note>
     * <p>
     * The act of disabling a Region will remove all IAM access to any resources that reside in that Region.
     * </p>
     * </note>
     *
     * @param disableRegionRequest
     * @return Result of the DisableRegion operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DisableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DisableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    default DisableRegionResponse disableRegion(DisableRegionRequest disableRegionRequest) throws AccessDeniedException,
            ValidationException, ConflictException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Disables (opts-out) a particular Region for an account.
     * </p>
     * <note>
     * <p>
     * The act of disabling a Region will remove all IAM access to any resources that reside in that Region.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link DisableRegionRequest.Builder} avoiding the need to
     * create one manually via {@link DisableRegionRequest#builder()}
     * </p>
     *
     * @param disableRegionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.DisableRegionRequest.Builder} to create a request.
     * @return Result of the DisableRegion operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.DisableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/DisableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    default DisableRegionResponse disableRegion(Consumer<DisableRegionRequest.Builder> disableRegionRequest)
            throws AccessDeniedException, ValidationException, ConflictException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        return disableRegion(DisableRegionRequest.builder().applyMutation(disableRegionRequest).build());
    }

    /**
     * <p>
     * Enables (opts-in) a particular Region for an account.
     * </p>
     *
     * @param enableRegionRequest
     * @return Result of the EnableRegion operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.EnableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/EnableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    default EnableRegionResponse enableRegion(EnableRegionRequest enableRegionRequest) throws AccessDeniedException,
            ValidationException, ConflictException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Enables (opts-in) a particular Region for an account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link EnableRegionRequest.Builder} avoiding the need to
     * create one manually via {@link EnableRegionRequest#builder()}
     * </p>
     *
     * @param enableRegionRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.EnableRegionRequest.Builder} to create a request.
     * @return Result of the EnableRegion operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.EnableRegion
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/EnableRegion" target="_top">AWS API
     *      Documentation</a>
     */
    default EnableRegionResponse enableRegion(Consumer<EnableRegionRequest.Builder> enableRegionRequest)
            throws AccessDeniedException, ValidationException, ConflictException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        return enableRegion(EnableRegionRequest.builder().applyMutation(enableRegionRequest).build());
    }

    /**
     * <p>
     * Retrieves information about the specified account including its account name, account ID, and account creation
     * date and time. To use this API, an IAM user or role must have the <code>account:GetAccountInformation</code> IAM
     * permission.
     * </p>
     *
     * @param getAccountInformationRequest
     * @return Result of the GetAccountInformation operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetAccountInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAccountInformation" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountInformationResponse getAccountInformation(GetAccountInformationRequest getAccountInformationRequest)
            throws AccessDeniedException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves information about the specified account including its account name, account ID, and account creation
     * date and time. To use this API, an IAM user or role must have the <code>account:GetAccountInformation</code> IAM
     * permission.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAccountInformationRequest.Builder} avoiding the
     * need to create one manually via {@link GetAccountInformationRequest#builder()}
     * </p>
     *
     * @param getAccountInformationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.GetAccountInformationRequest.Builder} to create a
     *        request.
     * @return Result of the GetAccountInformation operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetAccountInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAccountInformation" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAccountInformationResponse getAccountInformation(
            Consumer<GetAccountInformationRequest.Builder> getAccountInformationRequest) throws AccessDeniedException,
            ValidationException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        return getAccountInformation(GetAccountInformationRequest.builder().applyMutation(getAccountInformationRequest).build());
    }

    /**
     * <p>
     * Retrieves the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param getAlternateContactRequest
     * @return Result of the GetAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAlternateContactResponse getAlternateContact(GetAlternateContactRequest getAlternateContactRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetAlternateContactRequest.Builder} avoiding the
     * need to create one manually via {@link GetAlternateContactRequest#builder()}
     * </p>
     *
     * @param getAlternateContactRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.GetAlternateContactRequest.Builder} to create a
     *        request.
     * @return Result of the GetAlternateContact operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    default GetAlternateContactResponse getAlternateContact(
            Consumer<GetAlternateContactRequest.Builder> getAlternateContactRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        return getAlternateContact(GetAlternateContactRequest.builder().applyMutation(getAlternateContactRequest).build());
    }

    /**
     * <p>
     * Retrieves the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     *
     * @param getContactInformationRequest
     * @return Result of the GetContactInformation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    default GetContactInformationResponse getContactInformation(GetContactInformationRequest getContactInformationRequest)
            throws ResourceNotFoundException, ValidationException, AccessDeniedException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetContactInformationRequest.Builder} avoiding the
     * need to create one manually via {@link GetContactInformationRequest#builder()}
     * </p>
     *
     * @param getContactInformationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.GetContactInformationRequest.Builder} to create a
     *        request.
     * @return Result of the GetContactInformation operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    default GetContactInformationResponse getContactInformation(
            Consumer<GetContactInformationRequest.Builder> getContactInformationRequest) throws ResourceNotFoundException,
            ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException,
            SdkClientException, AccountException {
        return getContactInformation(GetContactInformationRequest.builder().applyMutation(getContactInformationRequest).build());
    }

    /**
     * <p>
     * Retrieves the primary email address for the specified account.
     * </p>
     *
     * @param getPrimaryEmailRequest
     * @return Result of the GetPrimaryEmail operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetPrimaryEmail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetPrimaryEmail" target="_top">AWS API
     *      Documentation</a>
     */
    default GetPrimaryEmailResponse getPrimaryEmail(GetPrimaryEmailRequest getPrimaryEmailRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the primary email address for the specified account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetPrimaryEmailRequest.Builder} avoiding the need
     * to create one manually via {@link GetPrimaryEmailRequest#builder()}
     * </p>
     *
     * @param getPrimaryEmailRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.GetPrimaryEmailRequest.Builder} to create a request.
     * @return Result of the GetPrimaryEmail operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetPrimaryEmail
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetPrimaryEmail" target="_top">AWS API
     *      Documentation</a>
     */
    default GetPrimaryEmailResponse getPrimaryEmail(Consumer<GetPrimaryEmailRequest.Builder> getPrimaryEmailRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, TooManyRequestsException,
            InternalServerException, AwsServiceException, SdkClientException, AccountException {
        return getPrimaryEmail(GetPrimaryEmailRequest.builder().applyMutation(getPrimaryEmailRequest).build());
    }

    /**
     * <p>
     * Retrieves the opt-in status of a particular Region.
     * </p>
     *
     * @param getRegionOptStatusRequest
     * @return Result of the GetRegionOptStatus operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetRegionOptStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetRegionOptStatus" target="_top">AWS
     *      API Documentation</a>
     */
    default GetRegionOptStatusResponse getRegionOptStatus(GetRegionOptStatusRequest getRegionOptStatusRequest)
            throws AccessDeniedException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Retrieves the opt-in status of a particular Region.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link GetRegionOptStatusRequest.Builder} avoiding the
     * need to create one manually via {@link GetRegionOptStatusRequest#builder()}
     * </p>
     *
     * @param getRegionOptStatusRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.GetRegionOptStatusRequest.Builder} to create a
     *        request.
     * @return Result of the GetRegionOptStatus operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.GetRegionOptStatus
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/GetRegionOptStatus" target="_top">AWS
     *      API Documentation</a>
     */
    default GetRegionOptStatusResponse getRegionOptStatus(Consumer<GetRegionOptStatusRequest.Builder> getRegionOptStatusRequest)
            throws AccessDeniedException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        return getRegionOptStatus(GetRegionOptStatusRequest.builder().applyMutation(getRegionOptStatusRequest).build());
    }

    /**
     * <p>
     * Lists all the Regions for a given account and their respective opt-in statuses. Optionally, this list can be
     * filtered by the <code>region-opt-status-contains</code> parameter.
     * </p>
     *
     * @param listRegionsRequest
     * @return Result of the ListRegions operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/ListRegions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRegionsResponse listRegions(ListRegionsRequest listRegionsRequest) throws AccessDeniedException,
            ValidationException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Lists all the Regions for a given account and their respective opt-in statuses. Optionally, this list can be
     * filtered by the <code>region-opt-status-contains</code> parameter.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRegionsRequest.Builder} avoiding the need to
     * create one manually via {@link ListRegionsRequest#builder()}
     * </p>
     *
     * @param listRegionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.ListRegionsRequest.Builder} to create a request.
     * @return Result of the ListRegions operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/ListRegions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRegionsResponse listRegions(Consumer<ListRegionsRequest.Builder> listRegionsRequest)
            throws AccessDeniedException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        return listRegions(ListRegionsRequest.builder().applyMutation(listRegionsRequest).build());
    }

    /**
     * <p>
     * This is a variant of {@link #listRegions(software.amazon.awssdk.services.account.model.ListRegionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.account.paginators.ListRegionsIterable responses = client.listRegionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.account.paginators.ListRegionsIterable responses = client.listRegionsPaginator(request);
     *     for (software.amazon.awssdk.services.account.model.ListRegionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.account.paginators.ListRegionsIterable responses = client.listRegionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRegions(software.amazon.awssdk.services.account.model.ListRegionsRequest)} operation.</b>
     * </p>
     *
     * @param listRegionsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/ListRegions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRegionsIterable listRegionsPaginator(ListRegionsRequest listRegionsRequest) throws AccessDeniedException,
            ValidationException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        return new ListRegionsIterable(this, listRegionsRequest);
    }

    /**
     * <p>
     * This is a variant of {@link #listRegions(software.amazon.awssdk.services.account.model.ListRegionsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.account.paginators.ListRegionsIterable responses = client.listRegionsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.account.paginators.ListRegionsIterable responses = client.listRegionsPaginator(request);
     *     for (software.amazon.awssdk.services.account.model.ListRegionsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.account.paginators.ListRegionsIterable responses = client.listRegionsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRegions(software.amazon.awssdk.services.account.model.ListRegionsRequest)} operation.</b>
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link ListRegionsRequest.Builder} avoiding the need to
     * create one manually via {@link ListRegionsRequest#builder()}
     * </p>
     *
     * @param listRegionsRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.ListRegionsRequest.Builder} to create a request.
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.ListRegions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/ListRegions" target="_top">AWS API
     *      Documentation</a>
     */
    default ListRegionsIterable listRegionsPaginator(Consumer<ListRegionsRequest.Builder> listRegionsRequest)
            throws AccessDeniedException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        return listRegionsPaginator(ListRegionsRequest.builder().applyMutation(listRegionsRequest).build());
    }

    /**
     * <p>
     * Updates the account name of the specified account. To use this API, IAM principals must have the
     * <code>account:PutAccountName</code> IAM permission.
     * </p>
     *
     * @param putAccountNameRequest
     * @return Result of the PutAccountName operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutAccountName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAccountName" target="_top">AWS API
     *      Documentation</a>
     */
    default PutAccountNameResponse putAccountName(PutAccountNameRequest putAccountNameRequest) throws AccessDeniedException,
            ValidationException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the account name of the specified account. To use this API, IAM principals must have the
     * <code>account:PutAccountName</code> IAM permission.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutAccountNameRequest.Builder} avoiding the need to
     * create one manually via {@link PutAccountNameRequest#builder()}
     * </p>
     *
     * @param putAccountNameRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.PutAccountNameRequest.Builder} to create a request.
     * @return Result of the PutAccountName operation returned by the service.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutAccountName
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAccountName" target="_top">AWS API
     *      Documentation</a>
     */
    default PutAccountNameResponse putAccountName(Consumer<PutAccountNameRequest.Builder> putAccountNameRequest)
            throws AccessDeniedException, ValidationException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        return putAccountName(PutAccountNameRequest.builder().applyMutation(putAccountNameRequest).build());
    }

    /**
     * <p>
     * Modifies the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note>
     *
     * @param putAlternateContactRequest
     * @return Result of the PutAlternateContact operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    default PutAlternateContactResponse putAlternateContact(PutAlternateContactRequest putAlternateContactRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Modifies the specified alternate contact attached to an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the alternate contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Access or updating
     * the alternate contacts</a>.
     * </p>
     * <note>
     * <p>
     * Before you can update the alternate contact information for an Amazon Web Services account that is managed by
     * Organizations, you must first enable integration between Amazon Web Services Account Management and
     * Organizations. For more information, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/using-orgs-trusted-access.html">Enabling trusted
     * access for Amazon Web Services Account Management</a>.
     * </p>
     * </note><br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutAlternateContactRequest.Builder} avoiding the
     * need to create one manually via {@link PutAlternateContactRequest#builder()}
     * </p>
     *
     * @param putAlternateContactRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.PutAlternateContactRequest.Builder} to create a
     *        request.
     * @return Result of the PutAlternateContact operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutAlternateContact
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutAlternateContact" target="_top">AWS
     *      API Documentation</a>
     */
    default PutAlternateContactResponse putAlternateContact(
            Consumer<PutAlternateContactRequest.Builder> putAlternateContactRequest) throws ValidationException,
            AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        return putAlternateContact(PutAlternateContactRequest.builder().applyMutation(putAlternateContactRequest).build());
    }

    /**
     * <p>
     * Updates the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     *
     * @param putContactInformationRequest
     * @return Result of the PutContactInformation operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    default PutContactInformationResponse putContactInformation(PutContactInformationRequest putContactInformationRequest)
            throws ValidationException, AccessDeniedException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Updates the primary contact information of an Amazon Web Services account.
     * </p>
     * <p>
     * For complete details about how to use the primary contact operations, see <a
     * href="https://docs.aws.amazon.com/accounts/latest/reference/manage-acct-update-contact.html">Update the primary
     * and alternate contact information</a>.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link PutContactInformationRequest.Builder} avoiding the
     * need to create one manually via {@link PutContactInformationRequest#builder()}
     * </p>
     *
     * @param putContactInformationRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.PutContactInformationRequest.Builder} to create a
     *        request.
     * @return Result of the PutContactInformation operation returned by the service.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.PutContactInformation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/PutContactInformation" target="_top">AWS
     *      API Documentation</a>
     */
    default PutContactInformationResponse putContactInformation(
            Consumer<PutContactInformationRequest.Builder> putContactInformationRequest) throws ValidationException,
            AccessDeniedException, TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException,
            AccountException {
        return putContactInformation(PutContactInformationRequest.builder().applyMutation(putContactInformationRequest).build());
    }

    /**
     * <p>
     * Starts the process to update the primary email address for the specified account.
     * </p>
     *
     * @param startPrimaryEmailUpdateRequest
     * @return Result of the StartPrimaryEmailUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.StartPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/StartPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    default StartPrimaryEmailUpdateResponse startPrimaryEmailUpdate(StartPrimaryEmailUpdateRequest startPrimaryEmailUpdateRequest)
            throws ResourceNotFoundException, AccessDeniedException, ValidationException, ConflictException,
            TooManyRequestsException, InternalServerException, AwsServiceException, SdkClientException, AccountException {
        throw new UnsupportedOperationException();
    }

    /**
     * <p>
     * Starts the process to update the primary email address for the specified account.
     * </p>
     * <br/>
     * <p>
     * This is a convenience which creates an instance of the {@link StartPrimaryEmailUpdateRequest.Builder} avoiding
     * the need to create one manually via {@link StartPrimaryEmailUpdateRequest#builder()}
     * </p>
     *
     * @param startPrimaryEmailUpdateRequest
     *        A {@link Consumer} that will call methods on
     *        {@link software.amazon.awssdk.services.account.model.StartPrimaryEmailUpdateRequest.Builder} to create a
     *        request.
     * @return Result of the StartPrimaryEmailUpdate operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation failed because it specified a resource that can't be found.
     * @throws AccessDeniedException
     *         The operation failed because the calling identity doesn't have the minimum required permissions.
     * @throws ValidationException
     *         The operation failed because one of the input parameters was invalid.
     * @throws ConflictException
     *         The request could not be processed because of a conflict in the current status of the resource. For
     *         example, this happens if you try to enable a Region that is currently being disabled (in a status of
     *         DISABLING) or if you try to change an account’s root user email to an email address which is already in
     *         use.
     * @throws TooManyRequestsException
     *         The operation failed because it was called too frequently and exceeded a throttle limit.
     * @throws InternalServerException
     *         The operation failed because of an error internal to Amazon Web Services. Try your operation again later.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws AccountException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample AccountClient.StartPrimaryEmailUpdate
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/account-2021-02-01/StartPrimaryEmailUpdate"
     *      target="_top">AWS API Documentation</a>
     */
    default StartPrimaryEmailUpdateResponse startPrimaryEmailUpdate(
            Consumer<StartPrimaryEmailUpdateRequest.Builder> startPrimaryEmailUpdateRequest) throws ResourceNotFoundException,
            AccessDeniedException, ValidationException, ConflictException, TooManyRequestsException, InternalServerException,
            AwsServiceException, SdkClientException, AccountException {
        return startPrimaryEmailUpdate(StartPrimaryEmailUpdateRequest.builder().applyMutation(startPrimaryEmailUpdateRequest)
                .build());
    }

    /**
     * Create a {@link AccountClient} with the region loaded from the
     * {@link software.amazon.awssdk.regions.providers.DefaultAwsRegionProviderChain} and credentials loaded from the
     * {@link software.amazon.awssdk.auth.credentials.DefaultCredentialsProvider}.
     */
    static AccountClient create() {
        return builder().build();
    }

    /**
     * Create a builder that can be used to configure and create a {@link AccountClient}.
     */
    static AccountClientBuilder builder() {
        return new DefaultAccountClientBuilder();
    }

    static ServiceMetadata serviceMetadata() {
        return ServiceMetadata.of(SERVICE_METADATA_ID);
    }

    @Override
    default AccountServiceClientConfiguration serviceClientConfiguration() {
        throw new UnsupportedOperationException();
    }
}
