/**
 * <h1>Amazon Simple Email Service Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>Email receiving</h2>
 * <p>
 * Create a receipt rule set with rules and actions (actions can be found in the
 * <code>aws-cdk-lib/aws-ses-actions</code> package):
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.s3.*;
 * import software.amazon.awscdk.services.ses.actions.*;
 * 
 * 
 * Bucket bucket = new Bucket(this, "Bucket");
 * Topic topic = new Topic(this, "Topic");
 * 
 * ReceiptRuleSet.Builder.create(this, "RuleSet")
 *         .rules(List.of(ReceiptRuleOptions.builder()
 *                 .recipients(List.of("hello&#64;aws.com"))
 *                 .actions(List.of(
 *                     AddHeader.Builder.create()
 *                             .name("X-Special-Header")
 *                             .value("aws")
 *                             .build(),
 *                     S3.Builder.create()
 *                             .bucket(bucket)
 *                             .objectKeyPrefix("emails/")
 *                             .topic(topic)
 *                             .build()))
 *                 .build(), ReceiptRuleOptions.builder()
 *                 .recipients(List.of("aws.com"))
 *                 .actions(List.of(
 *                     Sns.Builder.create()
 *                             .topic(topic)
 *                             .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, rules can be added to a rule set:
 * <p>
 * <blockquote><pre>
 * ReceiptRuleSet ruleSet = new ReceiptRuleSet(this, "RuleSet");
 * 
 * ReceiptRule awsRule = ruleSet.addRule("Aws", ReceiptRuleOptions.builder()
 *         .recipients(List.of("aws.com"))
 *         .build());
 * </pre></blockquote>
 * <p>
 * And actions to rules:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ses.actions.*;
 * 
 * ReceiptRule awsRule;
 * Topic topic;
 * 
 * awsRule.addAction(Sns.Builder.create()
 *         .topic(topic)
 *         .build());
 * </pre></blockquote>
 * <p>
 * When using <code>addRule</code>, the new rule is added after the last added rule unless <code>after</code> is specified.
 * <p>
 * <h3>Drop spams</h3>
 * <p>
 * A rule to drop spam can be added by setting <code>dropSpam</code> to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * ReceiptRuleSet.Builder.create(this, "RuleSet")
 *         .dropSpam(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will add a rule at the top of the rule set with a Lambda action that stops processing messages that have at least one spam indicator. See <a href="https://docs.aws.amazon.com/ses/latest/DeveloperGuide/receiving-email-action-lambda-example-functions.html">Lambda Function Examples</a>.
 * <p>
 * <h3>Receipt filter</h3>
 * <p>
 * Create a receipt filter:
 * <p>
 * <blockquote><pre>
 * ReceiptFilter.Builder.create(this, "Filter")
 *         .ip("1.2.3.4/16")
 *         .build();
 * </pre></blockquote>
 * <p>
 * An allow list filter is also available:
 * <p>
 * <blockquote><pre>
 * AllowListReceiptFilter.Builder.create(this, "AllowList")
 *         .ips(List.of("10.0.0.0/16", "1.2.3.4/16"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will first create a block all filter and then create allow filters for the listed ip addresses.
 * <p>
 * <h2>Email sending</h2>
 * <p>
 * <h3>Dedicated IP pools</h3>
 * <p>
 * When you create a new Amazon SES account, your emails are sent from IP addresses that are shared with other
 * Amazon SES users. For <a href="https://aws.amazon.com/ses/pricing/">an additional monthly charge</a>, you can lease
 * dedicated IP addresses that are reserved for your exclusive use.
 * <p>
 * Use the DedicatedIpPool construct to create a pool of dedicated IP addresses. When specifying a name for your dedicated IP pool, ensure that it adheres to the following naming convention:
 * <p>
 * <ul>
 * <li>The name must include only lowercase letters (a-z), numbers (0-9), underscores (_), and hyphens (-).</li>
 * <li>The name must not exceed 64 characters in length.</li>
 * </ul>
 * <p>
 * <blockquote><pre>
 * DedicatedIpPool.Builder.create(this, "Pool")
 *         .dedicatedIpPoolName("mypool")
 *         .scalingMode(ScalingMode.STANDARD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The pool can then be used in a configuration set. If the provided dedicatedIpPoolName does not follow the specified naming convention, an error will be thrown.
 * <p>
 * <h3>Configuration sets</h3>
 * <p>
 * Configuration sets are groups of rules that you can apply to your verified identities. A verified identity is
 * a domain, subdomain, or email address you use to send email through Amazon SES. When you apply a configuration
 * set to an email, all of the rules in that configuration set are applied to the email.
 * <p>
 * Use the <code>ConfigurationSet</code> construct to create a configuration set:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.Duration;
 * 
 * IDedicatedIpPool myPool;
 * 
 * 
 * ConfigurationSet.Builder.create(this, "ConfigurationSet")
 *         .customTrackingRedirectDomain("track.cdk.dev")
 *         .tlsPolicy(ConfigurationSetTlsPolicy.REQUIRE)
 *         .dedicatedIpPool(myPool)
 *         // Specify maximum delivery time
 *         // This configuration can be useful in such cases as time-sensitive emails (like those containing a one-time-password),
 *         // transactional emails, and email that you want to ensure isn't delivered during non-business hours.
 *         .maxDeliveryDuration(Duration.minutes(10))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Use <code>addEventDestination()</code> to publish email sending events to Amazon SNS, Amazon CloudWatch, Amazon Data Firehose or Amazon EventBridge:
 * <p>
 * <blockquote><pre>
 * ConfigurationSet myConfigurationSet;
 * Topic myTopic;
 * 
 * 
 * myConfigurationSet.addEventDestination("ToSns", ConfigurationSetEventDestinationOptions.builder()
 *         .destination(EventDestination.snsTopic(myTopic))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: For EventBridge, you must specify the default EventBus:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * 
 * ConfigurationSet myConfigurationSet;
 * 
 * 
 * IEventBus bus = EventBus.fromEventBusName(this, "EventBus", "default");
 * 
 * myConfigurationSet.addEventDestination("ToEventBus", ConfigurationSetEventDestinationOptions.builder()
 *         .destination(EventDestination.eventBus(bus))
 *         .build());
 * </pre></blockquote>
 * <p>
 * For Firehose, if you don't specify IAM Role ARN for Amazon SES to send events. An IAM Role will be created automatically following https://docs.aws.amazon.com/ses/latest/dg/event-publishing-add-event-destination-firehose.html.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * 
 * ConfigurationSet myConfigurationSet;
 * IDeliveryStream firehoseDeliveryStream;
 * IRole iamRole;
 * 
 * 
 * // Create IAM Role automatically
 * myConfigurationSet.addEventDestination("ToFirehose", ConfigurationSetEventDestinationOptions.builder()
 *         .destination(EventDestination.firehoseDeliveryStream(FirehoseDeliveryStreamDestination.builder()
 *                 .deliveryStream(firehoseDeliveryStream)
 *                 .build()))
 *         .build());
 * 
 * // Specify your IAM Role
 * myConfigurationSet.addEventDestination("ToFirehose", ConfigurationSetEventDestinationOptions.builder()
 *         .destination(EventDestination.firehoseDeliveryStream(FirehoseDeliveryStreamDestination.builder()
 *                 .deliveryStream(firehoseDeliveryStream)
 *                 .role(iamRole)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Override account-level suppression list settings</h3>
 * <p>
 * You can customize account-level suppression list separately for different configuration sets by overriding it
 * with configuration set-level suppression.
 * <p>
 * For details, see <a href="https://docs.aws.amazon.com/ses/latest/dg/sending-email-suppression-list-config-level.html">Using configuration set-level suppression to override your account-level suppression list</a>.
 * <p>
 * By default, the configuration set uses your account-level suppression list settings.
 * <p>
 * You can disable account-level suppression list by specifying <code>disableSuppressionList</code> to true. Email sent with this configuration set will not use any suppression settings at all.
 * <p>
 * <blockquote><pre>
 * ConfigurationSet.Builder.create(this, "ConfigurationSet")
 *         .disableSuppressionList(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also override account level settings with configuration set-level suppression enabled. Email sent with this configuration set will only use the suppression conditions you enabled for it (bounces, complaints, or bounces and complaints) - regardless of what your account-level suppression list settings are, it will override them.
 * <p>
 * <blockquote><pre>
 * // Only bounces will be suppressed.
 * // Only bounces will be suppressed.
 * ConfigurationSet.Builder.create(this, "ConfigurationSet")
 *         .suppressionReasons(SuppressionReasons.BOUNCES_ONLY)
 *         .build();
 * 
 * // Only complaints will be suppressed.
 * // Only complaints will be suppressed.
 * ConfigurationSet.Builder.create(this, "ConfigurationSet")
 *         .suppressionReasons(SuppressionReasons.COMPLAINTS_ONLY)
 *         .build();
 * 
 * // Both bounces and complaints will be suppressed.
 * // Both bounces and complaints will be suppressed.
 * ConfigurationSet.Builder.create(this, "ConfigurationSet")
 *         .suppressionReasons(SuppressionReasons.BOUNCES_AND_COMPLAINTS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Email identity</h3>
 * <p>
 * In Amazon SES, a verified identity is a domain or email address that you use to send or receive email. Before you
 * can send an email using Amazon SES, you must create and verify each identity that you're going to use as a <code>From</code>,
 * <code>Source</code>, <code>Sender</code>, or <code>Return-Path</code> address. Verifying an identity with Amazon SES confirms that you own it and
 * helps prevent unauthorized use.
 * <p>
 * To verify an identity for a hosted zone, you create an <code>EmailIdentity</code>:
 * <p>
 * <blockquote><pre>
 * IPublicHostedZone myHostedZone;
 * 
 * 
 * EmailIdentity identity = EmailIdentity.Builder.create(this, "Identity")
 *         .identity(Identity.publicHostedZone(myHostedZone))
 *         .mailFromDomain("mail.cdk.dev")
 *         .build();
 * </pre></blockquote>
 * <p>
 * By default, <a href="https://docs.aws.amazon.com/ses/latest/dg/send-email-authentication-dkim-easy.html">Easy DKIM</a> with
 * a 2048-bit DKIM key is used.
 * <p>
 * You can instead configure DKIM authentication by using your own public-private key pair. This process is known
 * as <a href="https://docs.aws.amazon.com/ses/latest/dg/send-email-authentication-dkim-bring-your-own.html">Bring Your Own DKIM (BYODKIM)</a>:
 * <p>
 * <blockquote><pre>
 * IPublicHostedZone myHostedZone;
 * 
 * 
 * EmailIdentity.Builder.create(this, "Identity")
 *         .identity(Identity.publicHostedZone(myHostedZone))
 *         .dkimIdentity(DkimIdentity.byoDkim(ByoDkimOptions.builder()
 *                 .privateKey(SecretValue.secretsManager("dkim-private-key"))
 *                 .publicKey("...base64-encoded-public-key...")
 *                 .selector("selector")
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * When using <code>publicHostedZone()</code> for the identity, all necessary Amazon Route 53 records are created automatically:
 * <p>
 * <ul>
 * <li>CNAME records for Easy DKIM</li>
 * <li>TXT record for BYOD DKIM</li>
 * <li>MX and TXT records for the custom MAIL FROM</li>
 * </ul>
 * <p>
 * When working with <code>domain()</code>, records must be created manually:
 * <p>
 * <blockquote><pre>
 * EmailIdentity identity = EmailIdentity.Builder.create(this, "Identity")
 *         .identity(Identity.domain("cdk.dev"))
 *         .build();
 * 
 * for (Object record : identity.getDkimRecords()) {
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Grants</h4>
 * <p>
 * To grant a specific action to a principal use the <code>grant</code> method.
 * For sending emails, <code>grantSendEmail</code> can be used instead:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * User user;
 * 
 * 
 * EmailIdentity identity = EmailIdentity.Builder.create(this, "Identity")
 *         .identity(Identity.domain("cdk.dev"))
 *         .build();
 * 
 * identity.grantSendEmail(user);
 * </pre></blockquote>
 * <p>
 * <h3>Virtual Deliverability Manager (VDM)</h3>
 * <p>
 * Virtual Deliverability Manager is an Amazon SES feature that helps you enhance email deliverability,
 * like increasing inbox deliverability and email conversions, by providing insights into your sending
 * and delivery data, and giving advice on how to fix the issues that are negatively affecting your
 * delivery success rate and reputation.
 * <p>
 * Use the <code>VdmAttributes</code> construct to configure the Virtual Deliverability Manager for your account:
 * <p>
 * <blockquote><pre>
 * // Enables engagement tracking and optimized shared delivery by default
 * // Enables engagement tracking and optimized shared delivery by default
 * new VdmAttributes(this, "Vdm");
 * </pre></blockquote>
 * <p>
 * If you want to override the VDM settings in the specified configuration set, use <code>vdmOptions</code> in the <code>ConfigurationSet</code> construct.
 * <p>
 * <blockquote>
 * <p>
 * <strong>Note:</strong> The configuration set level settings need to be used together with the account level settings. (To set the account level settings using CDK, use the <code>VdmAttributes</code> Construct.)
 * If you enable only the configuration set level settings, VDM will not be enabled until the account level settings are configured.
 * For more information, see <a href="https://docs.aws.amazon.com/ses/latest/dg/vdm-settings.html">Virtual Deliverability Manager settings</a>.
 * <p>
 * </blockquote>
 * <p>
 * <blockquote><pre>
 * ConfigurationSet.Builder.create(this, "ConfigurationSetWithVdmOptions")
 *         .vdmOptions(VdmOptions.builder()
 *                 .engagementMetrics(true)
 *                 .optimizedSharedDelivery(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.ses;
