/**
 * <h1>Amazon EventBridge Scheduler Targets Construct Library</h1>
 * <p>
 * <a href="https://aws.amazon.com/blogs/compute/introducing-amazon-eventbridge-scheduler/">Amazon EventBridge Scheduler</a> is a feature from Amazon EventBridge
 * that allows you to create, run, and manage scheduled tasks at scale. With EventBridge Scheduler, you can schedule millions of one-time or recurring tasks across various AWS services without provisioning or managing underlying infrastructure.
 * <p>
 * This library contains integration classes for Amazon EventBridge Scheduler to call any
 * number of supported AWS Services.
 * <p>
 * The following targets are supported:
 * <p>
 * <ol>
 * <li><code>targets.LambdaInvoke</code>: <a href="#invoke-a-lambda-function">Invoke an AWS Lambda function</a></li>
 * <li><code>targets.StepFunctionsStartExecution</code>: <a href="#start-an-aws-step-function">Start an AWS Step Function</a></li>
 * <li><code>targets.CodeBuildStartBuild</code>: <a href="#start-a-codebuild-job">Start a CodeBuild job</a></li>
 * <li><code>targets.SqsSendMessage</code>: <a href="#send-a-message-to-an-sqs-queue">Send a Message to an Amazon SQS Queue</a></li>
 * <li><code>targets.SnsPublish</code>: <a href="#publish-messages-to-an-amazon-sns-topic">Publish messages to an Amazon SNS topic</a></li>
 * <li><code>targets.EventBridgePutEvents</code>: <a href="#send-events-to-an-eventbridge-event-bus">Put Events on EventBridge</a></li>
 * <li><code>targets.InspectorStartAssessmentRun</code>: <a href="#start-an-amazon-inspector-assessment-run">Start an Amazon Inspector assessment run</a></li>
 * <li><code>targets.KinesisStreamPutRecord</code>: <a href="#put-a-record-to-an-amazon-kinesis-data-stream">Put a record to an Amazon Kinesis Data Stream</a></li>
 * <li><code>targets.FirehosePutRecord</code>: <a href="#put-a-record-to-an-amazon-data-firehose">Put a record to an Amazon Data Firehose</a></li>
 * <li><code>targets.CodePipelineStartPipelineExecution</code>: <a href="#start-a-codepipeline-execution">Start a CodePipeline execution</a></li>
 * <li><code>targets.SageMakerStartPipelineExecution</code>: <a href="#start-a-sagemaker-pipeline-execution">Start a SageMaker pipeline execution</a></li>
 * <li><code>targets.EcsRunTask</code>: <a href="#schedule-an-ecs-task-run">Start a new ECS task</a></li>
 * <li><code>targets.Universal</code>: <a href="#invoke-a-wider-set-of-aws-api">Invoke a wider set of AWS API</a></li>
 * </ol>
 * <p>
 * <h2>Invoke a Lambda function</h2>
 * <p>
 * Use the <code>LambdaInvoke</code> target to invoke a lambda function.
 * <p>
 * The code snippet below creates an event rule with a Lambda function as a target
 * called every hour by EventBridge Scheduler with a custom payload. You can optionally attach a
 * <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/rule-dlq.html">dead letter queue</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.lambda.*;
 * 
 * 
 * Function fn = Function.Builder.create(this, "MyFunc")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromInline("exports.handler = handler.toString()"))
 *         .build();
 * 
 * Queue dlq = Queue.Builder.create(this, "DLQ")
 *         .queueName("MyDLQ")
 *         .build();
 * 
 * LambdaInvoke target = LambdaInvoke.Builder.create(fn)
 *         .deadLetterQueue(dlq)
 *         .maxEventAge(Duration.minutes(1))
 *         .retryAttempts(3)
 *         .input(ScheduleTargetInput.fromObject(Map.of(
 *                 "payload", "useful")))
 *         .build();
 * 
 * Schedule schedule = Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.hours(1)))
 *         .target(target)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Start an AWS Step Function</h2>
 * <p>
 * Use the <code>StepFunctionsStartExecution</code> target to start a new execution on a StepFunction.
 * <p>
 * The code snippet below creates an event rule with a Step Function as a target
 * called every hour by EventBridge Scheduler with a custom payload.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.stepfunctions.*;
 * import software.amazon.awscdk.services.stepfunctions.tasks.*;
 * 
 * 
 * Map&lt;String, String&gt; payload = Map.of(
 *         "Name", "MyParameter",
 *         "Value", "🌥️");
 * 
 * CallAwsService putParameterStep = CallAwsService.Builder.create(this, "PutParameter")
 *         .service("ssm")
 *         .action("putParameter")
 *         .iamResources(List.of("*"))
 *         .parameters(Map.of(
 *                 "Name.$", "$.Name",
 *                 "Value.$", "$.Value",
 *                 "Type", "String",
 *                 "Overwrite", true))
 *         .build();
 * 
 * StateMachine stateMachine = StateMachine.Builder.create(this, "StateMachine")
 *         .definitionBody(DefinitionBody.fromChainable(putParameterStep))
 *         .build();
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.hours(1)))
 *         .target(StepFunctionsStartExecution.Builder.create(stateMachine)
 *                 .input(ScheduleTargetInput.fromObject(payload))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodeBuild job</h2>
 * <p>
 * Use the <code>CodeBuildStartBuild</code> target to start a new build run on a CodeBuild project.
 * <p>
 * The code snippet below creates an event rule with a CodeBuild project as target which is
 * called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codebuild.*;
 * 
 * Project project;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(new CodeBuildStartBuild(project))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Send a Message To an SQS Queue</h2>
 * <p>
 * Use the <code>SqsSendMessage</code> target to send a message to an SQS Queue.
 * <p>
 * The code snippet below creates an event rule with an SQS Queue as a target
 * called every hour by EventBridge Scheduler with a custom payload.
 * <p>
 * Contains the <code>messageGroupId</code> to use when the target is a FIFO queue. If you specify
 * a FIFO queue as a target, the queue must have content-based deduplication enabled.
 * <p>
 * <blockquote><pre>
 * String payload = "test";
 * String messageGroupId = "id";
 * Queue queue = Queue.Builder.create(this, "MyQueue")
 *         .fifo(true)
 *         .contentBasedDeduplication(true)
 *         .build();
 * 
 * SqsSendMessage target = SqsSendMessage.Builder.create(queue)
 *         .input(ScheduleTargetInput.fromText(payload))
 *         .messageGroupId(messageGroupId)
 *         .build();
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(1)))
 *         .target(target)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Publish messages to an Amazon SNS topic</h2>
 * <p>
 * Use the <code>SnsPublish</code> target to publish messages to an Amazon SNS topic.
 * <p>
 * The code snippets below create an event rule with a Amazon SNS topic as a target.
 * It's called every hour by Amazon EventBridge Scheduler with a custom payload.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * 
 * 
 * Topic topic = new Topic(this, "Topic");
 * 
 * Map&lt;String, String&gt; payload = Map.of(
 *         "message", "Hello scheduler!");
 * 
 * SnsPublish target = SnsPublish.Builder.create(topic)
 *         .input(ScheduleTargetInput.fromObject(payload))
 *         .build();
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.hours(1)))
 *         .target(target)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Send events to an EventBridge event bus</h2>
 * <p>
 * Use the <code>EventBridgePutEvents</code> target to send events to an EventBridge event bus.
 * <p>
 * The code snippet below creates an event rule with an EventBridge event bus as a target
 * called every hour by EventBridge Scheduler with a custom event payload.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.events.*;
 * 
 * 
 * EventBus eventBus = EventBus.Builder.create(this, "EventBus")
 *         .eventBusName("DomainEvents")
 *         .build();
 * 
 * EventBridgePutEventsEntry eventEntry = EventBridgePutEventsEntry.builder()
 *         .eventBus(eventBus)
 *         .source("PetService")
 *         .detail(ScheduleTargetInput.fromObject(Map.of("Name", "Fluffy")))
 *         .detailType("🐶")
 *         .build();
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.hours(1)))
 *         .target(new EventBridgePutEvents(eventEntry))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Start an Amazon Inspector assessment run</h2>
 * <p>
 * Use the <code>InspectorStartAssessmentRun</code> target to start an Inspector assessment run.
 * <p>
 * The code snippet below creates an event rule with an assessment template as the target which is
 * called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.inspector.*;
 * 
 * CfnAssessmentTemplate cfnAssessmentTemplate;
 * 
 * 
 * IAssessmentTemplate assessmentTemplate = AssessmentTemplate.fromCfnAssessmentTemplate(this, "MyAssessmentTemplate", cfnAssessmentTemplate);
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(new InspectorStartAssessmentRun(assessmentTemplate))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put a record to an Amazon Kinesis Data Stream</h2>
 * <p>
 * Use the <code>KinesisStreamPutRecord</code> target to put a record to an Amazon Kinesis Data Stream.
 * <p>
 * The code snippet below creates an event rule with a stream as the target which is
 * called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesis.*;
 * 
 * 
 * Stream stream = new Stream(this, "MyStream");
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(KinesisStreamPutRecord.Builder.create(stream)
 *                 .partitionKey("key")
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Put a record to an Amazon Data Firehose</h2>
 * <p>
 * Use the <code>FirehosePutRecord</code> target to put a record to an Amazon Data Firehose delivery stream.
 * <p>
 * The code snippet below creates an event rule with a delivery stream as a target
 * called every hour by EventBridge Scheduler with a custom payload.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kinesisfirehose.*;
 * IDeliveryStream deliveryStream;
 * 
 * 
 * Map&lt;String, String&gt; payload = Map.of(
 *         "Data", "record");
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(FirehosePutRecord.Builder.create(deliveryStream)
 *                 .input(ScheduleTargetInput.fromObject(payload))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Start a CodePipeline execution</h2>
 * <p>
 * Use the <code>CodePipelineStartPipelineExecution</code> target to start a new execution for a CodePipeline pipeline.
 * <p>
 * The code snippet below creates an event rule with a CodePipeline pipeline as the target which is
 * called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codepipeline.*;
 * 
 * Pipeline pipeline;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(new CodePipelineStartPipelineExecution(pipeline))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Start a SageMaker pipeline execution</h2>
 * <p>
 * Use the <code>SageMakerStartPipelineExecution</code> target to start a new execution for a SageMaker pipeline.
 * <p>
 * The code snippet below creates an event rule with a SageMaker pipeline as the target which is
 * called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sagemaker.*;
 * 
 * IPipeline pipeline;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(SageMakerStartPipelineExecution.Builder.create(pipeline)
 *                 .pipelineParameterList(List.of(SageMakerPipelineParameter.builder()
 *                         .name("parameter-name")
 *                         .value("parameter-value")
 *                         .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Schedule an ECS task run</h2>
 * <p>
 * Use the <code>EcsRunTask</code> target to schedule an ECS task run for a cluster.
 * <p>
 * The code snippet below creates an event rule with a Fargate task definition and cluster as the target which is called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * ICluster cluster;
 * FargateTaskDefinition taskDefinition;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(EcsRunFargateTask.Builder.create(cluster)
 *                 .taskDefinition(taskDefinition)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The code snippet below creates an event rule with a EC2 task definition and cluster as the target which is called every hour by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * ICluster cluster;
 * Ec2TaskDefinition taskDefinition;
 * 
 * 
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(EcsRunEc2Task.Builder.create(cluster)
 *                 .taskDefinition(taskDefinition)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Invoke a wider set of AWS API</h2>
 * <p>
 * Use the <code>Universal</code> target to invoke AWS API. See <a href="https://docs.aws.amazon.com/scheduler/latest/UserGuide/managing-targets-universal.html">https://docs.aws.amazon.com/scheduler/latest/UserGuide/managing-targets-universal.html</a>
 * <p>
 * The code snippet below creates an event rule with AWS API as the target which is
 * called at midnight every day by EventBridge Scheduler.
 * <p>
 * <blockquote><pre>
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.cron(CronOptionsWithTimezone.builder()
 *                 .minute("0")
 *                 .hour("0")
 *                 .build()))
 *         .target(Universal.Builder.create()
 *                 .service("rds")
 *                 .action("stopDBCluster")
 *                 .input(ScheduleTargetInput.fromObject(Map.of(
 *                         "DbClusterIdentifier", "my-db")))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * The <code>service</code> must be in lowercase and the <code>action</code> must be in camelCase.
 * <p>
 * By default, an IAM policy for the Scheduler is extracted from the API call. The action in the policy is constructed using the <code>service</code> and <code>action</code> prop.
 * Re-using the example above, the action will be <code>rds:stopDBCluster</code>. Note that not all IAM actions follow the same pattern. In such scenario, please use the
 * <code>policyStatements</code> prop to override the policy:
 * <p>
 * <blockquote><pre>
 * Schedule.Builder.create(this, "Schedule")
 *         .schedule(ScheduleExpression.rate(Duration.minutes(60)))
 *         .target(Universal.Builder.create()
 *                 .service("sqs")
 *                 .action("sendMessage")
 *                 .policyStatements(List.of(
 *                     PolicyStatement.Builder.create()
 *                             .actions(List.of("sqs:SendMessage"))
 *                             .resources(List.of("arn:aws:sqs:us-east-1:123456789012:my_queue"))
 *                             .build(),
 *                     PolicyStatement.Builder.create()
 *                             .actions(List.of("kms:Decrypt", "kms:GenerateDataKey*"))
 *                             .resources(List.of("arn:aws:kms:us-east-1:123456789012:key/0987dcba-09fe-87dc-65ba-ab0987654321"))
 *                             .build()))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <blockquote>
 * <p>
 * Note: The default policy uses <code>*</code> in the resources field as CDK does not have a straight forward way to auto-discover the resources permission required.
 * It is recommended that you scope the field down to specific resources to have a better security posture.
 * <p>
 * </blockquote>
 */
package software.amazon.awscdk.services.scheduler.targets;
