/**
 * <h1>Amazon Lambda Destinations Library</h1>
 * <p>
 * This library provides constructs for adding destinations to a Lambda function.
 * Destinations can be added by specifying the <code>onFailure</code> or <code>onSuccess</code> props when creating a function or alias.
 * <p>
 * <h2>Destinations</h2>
 * <p>
 * The following destinations are supported
 * <p>
 * <ul>
 * <li>Lambda function</li>
 * <li>SQS queue - Only standard SQS queues are supported for failure destinations, FIFO queues are not supported.</li>
 * <li>SNS topic</li>
 * <li>EventBridge event bus</li>
 * <li>S3 bucket</li>
 * </ul>
 * <p>
 * Example with a SNS topic for successful invocations:
 * <p>
 * <blockquote><pre>
 * // An sns topic for successful invocations of a lambda function
 * import software.amazon.awscdk.services.sns.*;
 * 
 * 
 * Topic myTopic = new Topic(this, "Topic");
 * 
 * Function myFn = Function.Builder.create(this, "Fn")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         // sns topic for successful invocations
 *         .onSuccess(new SnsDestination(myTopic))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Example with a SQS queue for unsuccessful invocations:
 * <p>
 * <blockquote><pre>
 * // An sqs queue for unsuccessful invocations of a lambda function
 * import software.amazon.awscdk.services.sqs.*;
 * 
 * 
 * Queue deadLetterQueue = new Queue(this, "DeadLetterQueue");
 * 
 * Function myFn = Function.Builder.create(this, "Fn")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromInline("// your code"))
 *         // sqs queue for unsuccessful invocations
 *         .onFailure(new SqsDestination(deadLetterQueue))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See also <a href="https://docs.aws.amazon.com/lambda/latest/dg/invocation-async.html#invocation-async-destinations">Configuring Destinations for Asynchronous Invocation</a>.
 * <p>
 * <h3>Invocation record</h3>
 * <p>
 * When a lambda function is configured with a destination, an invocation record is created by the Lambda service
 * when the lambda function completes. The invocation record contains the details of the function, its context, and
 * the request and response payloads.
 * <p>
 * The following example shows the format of the invocation record for a successful invocation:
 * <p>
 * <blockquote><pre>
 * {
 * 	"version": "1.0",
 * 	"timestamp": "2019-11-24T23:08:25.651Z",
 * 	"requestContext": {
 * 		"requestId": "c2a6f2ae-7dbb-4d22-8782-d0485c9877e2",
 * 		"functionArn": "arn:aws:lambda:sa-east-1:123456789123:function:event-destinations:$LATEST",
 * 		"condition": "Success",
 * 		"approximateInvokeCount": 1
 * 	},
 * 	"requestPayload": {
 * 		"Success": true
 * 	},
 * 	"responseContext": {
 * 		"statusCode": 200,
 * 		"executedVersion": "$LATEST"
 * 	},
 * 	"responsePayload": "&lt;data returned by the function here&gt;"
 * }
 * </pre></blockquote>
 * <p>
 * In case of failure, the record contains the reason and error object:
 * <p>
 * <blockquote><pre>
 * {
 *   "version": "1.0",
 *   "timestamp": "2019-11-24T21:52:47.333Z",
 *   "requestContext": {
 *     "requestId": "8ea123e4-1db7-4aca-ad10-d9ca1234c1fd",
 *     "functionArn": "arn:aws:lambda:sa-east-1:123456678912:function:event-destinations:$LATEST",
 *     "condition": "RetriesExhausted",
 *     "approximateInvokeCount": 3
 *   },
 *   "requestPayload": {
 *     "Success": false
 *   },
 *   "responseContext": {
 *     "statusCode": 200,
 *     "executedVersion": "$LATEST",
 *     "functionError": "Handled"
 *   },
 *   "responsePayload": {
 *     "errorMessage": "Failure from event, Success = false, I am failing!",
 *     "errorType": "Error",
 *     "stackTrace": [ "exports.handler (/var/task/index.js:18:18)" ]
 *   }
 * }
 * </pre></blockquote>
 * <p>
 * <h4>Destination-specific JSON format</h4>
 * <p>
 * <ul>
 * <li>For SNS/SQS (<code>SnsDestionation</code>/<code>SqsDestination</code>), the invocation record JSON is passed as the <code>Message</code> to the destination.</li>
 * <li>For Lambda (<code>LambdaDestination</code>), the invocation record JSON is passed as the payload to the function.</li>
 * <li>For EventBridge (<code>EventBridgeDestination</code>), the invocation record JSON is passed as the <code>detail</code> in the PutEvents call.
 * The value for the event field <code>source</code> is <code>lambda</code>, and the value for the event field <code>detail-type</code>
 * is either 'Lambda Function Invocation Result - Success' or 'Lambda Function Invocation Result – Failure',
 * depending on whether the lambda function invocation succeeded or failed. The event field <code>resource</code>
 * contains the function and destination ARNs. See <a href="https://docs.aws.amazon.com/eventbridge/latest/userguide/aws-events.html">AWS Events</a>
 * for the different event fields.</li>
 * <li>For S3 (<code>S3Destination</code>), the invocation record json is stored as a <code>File</code> in the destination bucket. The path of a destination
 * payload file in the configured bucket is <code>aws/lambda/async/&lt;function-name&gt;/YYYY/MM/DD/YYYY-MM-DDTHH.MM.SS-&lt;Random UUID&gt;</code>.</li>
 * </ul>
 * <p>
 * <h3>Auto-extract response payload with lambda destination</h3>
 * <p>
 * The <code>responseOnly</code> option of <code>LambdaDestination</code> allows to auto-extract the response payload from the
 * invocation record:
 * <p>
 * <blockquote><pre>
 * // Auto-extract response payload with a lambda destination
 * Function destinationFn;
 * 
 * 
 * Function sourceFn = Function.Builder.create(this, "Source")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .handler("index.handler")
 *         .code(Code.fromAsset(join(__dirname, "lambda-handler")))
 *         // auto-extract on success
 *         .onSuccess(LambdaDestination.Builder.create(destinationFn)
 *                 .responseOnly(true)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * In the above example, <code>destinationFn</code> will be invoked with the payload returned by <code>sourceFn</code>
 * (<code>responsePayload</code> in the invocation record, not the full record).
 * <p>
 * When used with <code>onFailure</code>, the destination function is invoked with the error object returned
 * by the source function.
 * <p>
 * Using the <code>responseOnly</code> option allows to easily chain asynchronous Lambda functions without
 * having to deal with data extraction in the runtime code.
 */
package software.amazon.awscdk.services.lambda.destinations;
