/**
 * <h1>Amazon Elastic File System Construct Library</h1>
 * <p>
 * <a href="https://docs.aws.amazon.com/efs/latest/ug/whatisefs.html">Amazon Elastic File System</a> (Amazon EFS) provides a simple, scalable,
 * fully managed elastic NFS file system for use with AWS Cloud services and on-premises resources.
 * Amazon EFS provides file storage in the AWS Cloud. With Amazon EFS, you can create a file system,
 * mount the file system on an Amazon EC2 instance, and then read and write data to and from your file system.
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * <h2>File Systems</h2>
 * <p>
 * Amazon EFS provides elastic, shared file storage that is POSIX-compliant. The file system you create
 * supports concurrent read and write access from multiple Amazon EC2 instances and is accessible from
 * all of the Availability Zones in the AWS Region where it is created. Learn more about <a href="https://docs.aws.amazon.com/efs/latest/ug/creating-using.html">EFS file systems</a>
 * <p>
 * <h3>Create an Amazon EFS file system</h3>
 * <p>
 * A Virtual Private Cloud (VPC) is required to create an Amazon EFS file system.
 * The following example creates a file system that is encrypted at rest, running in <code>General Purpose</code>
 * performance mode, and <code>Bursting</code> throughput mode and does not transition files to the Infrequent
 * Access (IA) storage class.
 * <p>
 * <blockquote><pre>
 * FileSystem fileSystem = FileSystem.Builder.create(this, "MyEfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .lifecyclePolicy(LifecyclePolicy.AFTER_14_DAYS) // files are not transitioned to infrequent access (IA) storage by default
 *         .performanceMode(PerformanceMode.GENERAL_PURPOSE) // default
 *         .outOfInfrequentAccessPolicy(OutOfInfrequentAccessPolicy.AFTER_1_ACCESS) // files are not transitioned back from (infrequent access) IA to primary storage by default
 *         .transitionToArchivePolicy(LifecyclePolicy.AFTER_14_DAYS) // files are not transitioned to Archive by default
 *         .replicationOverwriteProtection(ReplicationOverwriteProtection.ENABLED)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ An Amazon EFS file system's performance mode can't be MAX_IO when its throughputMode is ELASTIC.
 * <p>
 * ⚠️ An Amazon EFS file system's performance mode can't be changed after the file system has been created.
 * Updating this property will replace the file system.
 * <p>
 * Any file system that has been created outside the stack can be imported into your CDK app.
 * <p>
 * Use the <code>fromFileSystemAttributes()</code> API to import an existing file system.
 * Here is an example of giving a role write permissions on a file system.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * 
 * IFileSystem importedFileSystem = FileSystem.fromFileSystemAttributes(this, "existingFS", FileSystemAttributes.builder()
 *         .fileSystemId("fs-12345678") // You can also use fileSystemArn instead of fileSystemId.
 *         .securityGroup(SecurityGroup.fromSecurityGroupId(this, "SG", "sg-123456789", SecurityGroupImportOptions.builder()
 *                 .allowAllOutbound(false)
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>One Zone file system</h3>
 * <p>
 * To initialize a One Zone file system use the <code>oneZone</code> property:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * FileSystem.Builder.create(this, "OneZoneFileSystem")
 *         .vpc(vpc)
 *         .oneZone(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ One Zone file systems are not compatible with the MAX_IO performance mode.
 * <p>
 * ⚠️ When <code>oneZone</code> is enabled, the file system is automatically placed in the first availability zone of the VPC.
 * To specify a different availability zone:
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * FileSystem.Builder.create(this, "OneZoneFileSystem")
 *         .vpc(vpc)
 *         .oneZone(true)
 *         .vpcSubnets(SubnetSelection.builder()
 *                 .availabilityZones(List.of("us-east-1b"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * ⚠️ When <code>oneZone</code> is enabled, mount targets will be created only in the specified availability zone.
 * This is to prevent deployment failures due to cross-AZ configurations.
 * <p>
 * ⚠️ When <code>oneZone</code> is enabled, <code>vpcSubnets</code> can be specified with
 * <code>availabilityZones</code> that contains exactly one single zone.
 * <p>
 * <h3>Replicating file systems</h3>
 * <p>
 * You can create a replica of your EFS file system in the AWS Region of your preference.
 * <p>
 * <blockquote><pre>
 * Vpc vpc;
 * 
 * 
 * // auto generate a regional replication destination file system
 * // auto generate a regional replication destination file system
 * FileSystem.Builder.create(this, "RegionalReplicationFileSystem")
 *         .vpc(vpc)
 *         .replicationConfiguration(ReplicationConfiguration.regionalFileSystem("us-west-2"))
 *         .build();
 * 
 * // auto generate a one zone replication destination file system
 * // auto generate a one zone replication destination file system
 * FileSystem.Builder.create(this, "OneZoneReplicationFileSystem")
 *         .vpc(vpc)
 *         .replicationConfiguration(ReplicationConfiguration.oneZoneFileSystem("us-east-1", "us-east-1a"))
 *         .build();
 * 
 * FileSystem destinationFileSystem = FileSystem.Builder.create(this, "DestinationFileSystem")
 *         .vpc(vpc)
 *         // set as the read-only file system for use as a replication destination
 *         .replicationOverwriteProtection(ReplicationOverwriteProtection.DISABLED)
 *         .build();
 * // specify the replication destination file system
 * // specify the replication destination file system
 * FileSystem.Builder.create(this, "ReplicationFileSystem")
 *         .vpc(vpc)
 *         .replicationConfiguration(ReplicationConfiguration.existingFileSystem(destinationFileSystem))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: EFS now supports only one replication destination and thus allows specifying just one <code>replicationConfiguration</code> for each file system.
 * <p>
 * <blockquote>
 * <p>
 * Visit <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-replication.html">Replicating file systems</a> for more details.
 * <p>
 * </blockquote>
 * <p>
 * <h3>IAM to control file system data access</h3>
 * <p>
 * You can use both IAM identity policies and resource policies to control client access to Amazon EFS resources in a way that is scalable and optimized for cloud environments. Using IAM, you can permit clients to perform specific actions on a file system, including read-only, write, and root access.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * 
 * PolicyDocument myFileSystemPolicy = PolicyDocument.Builder.create()
 *         .statements(List.of(PolicyStatement.Builder.create()
 *                 .actions(List.of("elasticfilesystem:ClientWrite", "elasticfilesystem:ClientMount"))
 *                 .principals(List.of(new AccountRootPrincipal()))
 *                 .resources(List.of("*"))
 *                 .conditions(Map.of(
 *                         "Bool", Map.of(
 *                                 "elasticfilesystem:AccessedViaMountTarget", "true")))
 *                 .build()))
 *         .build();
 * 
 * FileSystem fileSystem = FileSystem.Builder.create(this, "MyEfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .fileSystemPolicy(myFileSystemPolicy)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Alternatively, a resource policy can be added later using <code>addToResourcePolicy(statement)</code>. Note that this will not work with imported FileSystem.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * PolicyStatement statement;
 * 
 * FileSystem fileSystem = FileSystem.Builder.create(this, "MyEfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .build();
 * 
 * fileSystem.addToResourcePolicy(statement);
 * </pre></blockquote>
 * <p>
 * <h3>Permissions</h3>
 * <p>
 * If you need to grant file system permissions to another resource, you can use the <code>.grant()</code> API.
 * As an example, the following code gives <code>elasticfilesystem:Backup</code> permissions to an IAM role.
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "Role")
 *         .assumedBy(new AnyPrincipal())
 *         .build();
 * 
 * fileSystem.grant(role, "elasticfilesystem:Backup");
 * </pre></blockquote>
 * <p>
 * APIs for clients also include <code>.grantRead()</code>, <code>.grantReadWrite()</code>, and <code>.grantRootAccess()</code>. Using these APIs grants access to clients.
 * Also, by default, the file system policy is updated to only allow access to clients using IAM authentication and deny access to anonymous clients.
 * <p>
 * <blockquote><pre>
 * Role role = Role.Builder.create(this, "ClientRole")
 *         .assumedBy(new AnyPrincipal())
 *         .build();
 * 
 * fileSystem.grantRead(role);
 * </pre></blockquote>
 * <p>
 * You can control this behavior with <code>allowAnonymousAccess</code>. The following example continues to allow anonymous client access.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.iam.*;
 * 
 * 
 * Role role = Role.Builder.create(this, "ClientRole")
 *         .assumedBy(new AnyPrincipal())
 *         .build();
 * FileSystem fileSystem = FileSystem.Builder.create(this, "MyEfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .allowAnonymousAccess(true)
 *         .build();
 * 
 * fileSystem.grantRead(role);
 * </pre></blockquote>
 * <p>
 * <h3>Access Point</h3>
 * <p>
 * An access point is an application-specific view into an EFS file system that applies an operating
 * system user and group, and a file system path, to any file system request made through the access
 * point. The operating system user and group override any identity information provided by the NFS
 * client. The file system path is exposed as the access point's root directory. Applications using
 * the access point can only access data in its own directory and below. To learn more, see <a href="https://docs.aws.amazon.com/efs/latest/ug/efs-access-points.html">Mounting a File System Using EFS Access Points</a>.
 * <p>
 * Use the <code>addAccessPoint</code> API to create an access point from a fileSystem.
 * <p>
 * <blockquote><pre>
 * fileSystem.addAccessPoint("MyAccessPoint", AccessPointOptions.builder()
 *         // create a unique access point via an optional client token
 *         .clientToken("client-token")
 *         .build());
 * </pre></blockquote>
 * <p>
 * By default, when you create an access point, the root(<code>/</code>) directory is exposed to the client
 * connecting to the access point. You can specify a custom path with the <code>path</code> property.
 * <p>
 * If <code>path</code> does not exist, it will be created with the settings defined in the <code>creationInfo</code>.
 * See <a href="https://docs.aws.amazon.com/efs/latest/ug/create-access-point.html">Creating Access Points</a> for more details.
 * <p>
 * Any access point that has been created outside the stack can be imported into your CDK app.
 * <p>
 * Use the <code>fromAccessPointAttributes()</code> API to import an existing access point.
 * <p>
 * <blockquote><pre>
 * AccessPoint.fromAccessPointAttributes(this, "ap", AccessPointAttributes.builder()
 *         .accessPointId("fsap-1293c4d9832fo0912")
 *         .fileSystem(FileSystem.fromFileSystemAttributes(this, "efs", FileSystemAttributes.builder()
 *                 .fileSystemId("fs-099d3e2f")
 *                 .securityGroup(SecurityGroup.fromSecurityGroupId(this, "sg", "sg-51530134"))
 *                 .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * ⚠️ Notice: When importing an Access Point using <code>fromAccessPointAttributes()</code>, you must make sure
 * the mount targets are deployed and their lifecycle state is <code>available</code>. Otherwise, you may encounter
 * the following error when deploying:
 * <p>
 * <blockquote>
 * <p>
 * EFS file system <ARN of efs> referenced by access point <ARN of access point of EFS> has
 * mount targets created in all availability zones the function will execute in, but not all
 * are in the available life cycle state yet. Please wait for them to become available and
 * try the request again.
 * <p>
 * </blockquote>
 * <p>
 * <h3>Connecting</h3>
 * <p>
 * To control who can access the EFS, use the <code>.connections</code> attribute. EFS has
 * a fixed default port, so you don't need to specify the port:
 * <p>
 * <blockquote><pre>
 * fileSystem.connections.allowDefaultPortFrom(instance);
 * </pre></blockquote>
 * <p>
 * Learn more about <a href="https://docs.aws.amazon.com/efs/latest/ug/manage-fs-access.html">managing file system network accessibility</a>
 * <p>
 * <h3>Mounting the file system using User Data</h3>
 * <p>
 * After you create a file system, you can create mount targets. Then you can mount the file system on
 * EC2 instances, containers, and Lambda functions in your virtual private cloud (VPC).
 * <p>
 * The following example automatically mounts a file system during instance launch.
 * <p>
 * <blockquote><pre>
 * fileSystem.connections.allowDefaultPortFrom(instance);
 * 
 * instance.userData.addCommands("yum check-update -y", "yum upgrade -y", "yum install -y amazon-efs-utils", "yum install -y nfs-utils", "file_system_id_1=" + fileSystem.getFileSystemId(), "efs_mount_point_1=/mnt/efs/fs1", "mkdir -p \"${efs_mount_point_1}\"", "test -f \"/sbin/mount.efs\" &amp;&amp; echo \"${file_system_id_1}:/ ${efs_mount_point_1} efs defaults,_netdev\" &gt;&gt; /etc/fstab || " + "echo \"${file_system_id_1}.efs." + Stack.of(this).getRegion() + ".amazonaws.com:/ ${efs_mount_point_1} nfs4 nfsvers=4.1,rsize=1048576,wsize=1048576,hard,timeo=600,retrans=2,noresvport,_netdev 0 0\" &gt;&gt; /etc/fstab", "mount -a -t efs,nfs4 defaults");
 * </pre></blockquote>
 * <p>
 * Learn more about <a href="https://docs.aws.amazon.com/efs/latest/ug/mounting-fs.html">mounting EFS file systems</a>
 * <p>
 * <h3>Deleting</h3>
 * <p>
 * Since file systems are stateful resources, by default the file system will not be deleted when your
 * stack is deleted.
 * <p>
 * You can configure the file system to be destroyed on stack deletion by setting a <code>removalPolicy</code>
 * <p>
 * <blockquote><pre>
 * FileSystem fileSystem = FileSystem.Builder.create(this, "EfsFileSystem")
 *         .vpc(new Vpc(this, "VPC"))
 *         .removalPolicy(RemovalPolicy.DESTROY)
 *         .build();
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.efs;
