/**
 * <h1>AWS CodePipeline Actions</h1>
 * <p>
 * This package contains Actions that can be used in a CodePipeline.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.codepipeline.*;
 * import software.amazon.awscdk.services.codepipeline.actions.*;
 * </pre></blockquote>
 * <p>
 * <h2>Sources</h2>
 * <p>
 * <h3>AWS CodeCommit</h3>
 * <p>
 * To use a CodeCommit Repository in a CodePipeline:
 * <p>
 * <blockquote><pre>
 * Repository repo = Repository.Builder.create(this, "Repo")
 *         .repositoryName("MyRepo")
 *         .build();
 * 
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyPipeline")
 *         .pipelineName("MyPipeline")
 *         .build();
 * Artifact sourceOutput = new Artifact();
 * CodeCommitSourceAction sourceAction = CodeCommitSourceAction.Builder.create()
 *         .actionName("CodeCommit")
 *         .repository(repo)
 *         .output(sourceOutput)
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(sourceAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you want to use existing role which can be used by on commit event rule.
 * You can specify the role object in eventRole property.
 * <p>
 * <blockquote><pre>
 * Repository repo;
 * IRole eventRole = Role.fromRoleArn(this, "Event-role", "roleArn");
 * CodeCommitSourceAction sourceAction = CodeCommitSourceAction.Builder.create()
 *         .actionName("CodeCommit")
 *         .repository(repo)
 *         .output(new Artifact())
 *         .eventRole(eventRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to clone the entire CodeCommit repository (only available for CodeBuild actions),
 * you can set the <code>codeBuildCloneOutput</code> property to <code>true</code>:
 * <p>
 * <blockquote><pre>
 * PipelineProject project;
 * Repository repo;
 * 
 * Artifact sourceOutput = new Artifact();
 * CodeCommitSourceAction sourceAction = CodeCommitSourceAction.Builder.create()
 *         .actionName("CodeCommit")
 *         .repository(repo)
 *         .output(sourceOutput)
 *         .codeBuildCloneOutput(true)
 *         .build();
 * 
 * CodeBuildAction buildAction = CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput) // The build action must use the CodeCommitSourceAction output as input.
 *         .outputs(List.of(new Artifact()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The CodeCommit source action emits variables:
 * <p>
 * <blockquote><pre>
 * PipelineProject project;
 * Repository repo;
 * 
 * Artifact sourceOutput = new Artifact();
 * CodeCommitSourceAction sourceAction = CodeCommitSourceAction.Builder.create()
 *         .actionName("CodeCommit")
 *         .repository(repo)
 *         .output(sourceOutput)
 *         .variablesNamespace("MyNamespace")
 *         .build();
 * 
 * // later:
 * 
 * // later:
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "COMMIT_ID", BuildEnvironmentVariable.builder()
 *                         .value(sourceAction.getVariables().getCommitId())
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to use a custom event for your <code>CodeCommitSourceAction</code>, you can pass in
 * a <code>customEventRule</code> which needs an event pattern (see <a href="https://docs.aws.amazon.com/codecommit/latest/userguide/monitoring-events.html">here</a>) and an <code>IRuleTarget</code> (see <a href="https://docs.aws.amazon.com/cdk/api/v2/docs/aws-cdk-lib.aws_events_targets-readme.html">here</a>)
 * <p>
 * <blockquote><pre>
 * Repository repo;
 * Function lambdaFunction;
 * Map&lt;String, Object&gt; eventPattern = Map.of(
 *         "detail-type", List.of("CodeCommit Repository State Change"),
 *         "resources", List.of("foo"),
 *         "source", List.of("aws.codecommit"),
 *         "detail", Map.of(
 *                 "referenceType", List.of("branch"),
 *                 "event", List.of("referenceCreated", "referenceUpdated"),
 *                 "referenceName", List.of("master")));
 * Artifact sourceOutput = new Artifact();
 * CodeCommitSourceAction sourceAction = CodeCommitSourceAction.Builder.create()
 *         .actionName("CodeCommit")
 *         .repository(repo)
 *         .output(sourceOutput)
 *         .customEventRule(Map.of(
 *                 "eventPattern", eventPattern,
 *                 "target", new LambdaFunction(lambdaFunction)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>GitHub</h3>
 * <p>
 * If you want to use a GitHub repository as the source, you must create:
 * <p>
 * <ul>
 * <li>A <a href="https://help.github.com/en/github/authenticating-to-github/creating-a-personal-access-token-for-the-command-line">GitHub Access Token</a>,
 * with scopes <strong>repo</strong> and <strong>admin:repo_hook</strong>.</li>
 * <li>A <a href="https://docs.aws.amazon.com/secretsmanager/latest/userguide/manage_create-basic-secret.html">Secrets Manager Secret</a>
 * with the value of the <strong>GitHub Access Token</strong>. Pick whatever name you want (for example <code>my-github-token</code>).
 * This token can be stored either as Plaintext or as a Secret key/value.
 * If you stored the token as Plaintext,
 * set <code>SecretValue.secretsManager('my-github-token')</code> as the value of <code>oauthToken</code>.
 * If you stored it as a Secret key/value,
 * you must set <code>SecretValue.secretsManager('my-github-token', { jsonField : 'my-github-token' })</code> as the value of <code>oauthToken</code>.</li>
 * </ul>
 * <p>
 * To use GitHub as the source of a CodePipeline:
 * <p>
 * <blockquote><pre>
 * // Read the secret from Secrets Manager
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Artifact sourceOutput = new Artifact();
 * GitHubSourceAction sourceAction = GitHubSourceAction.Builder.create()
 *         .actionName("GitHub_Source")
 *         .owner("awslabs")
 *         .repo("aws-cdk")
 *         .oauthToken(SecretValue.secretsManager("my-github-token"))
 *         .output(sourceOutput)
 *         .branch("develop")
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(sourceAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The GitHub source action emits variables:
 * <p>
 * <blockquote><pre>
 * Artifact sourceOutput;
 * PipelineProject project;
 * 
 * 
 * GitHubSourceAction sourceAction = GitHubSourceAction.Builder.create()
 *         .actionName("Github_Source")
 *         .output(sourceOutput)
 *         .owner("my-owner")
 *         .repo("my-repo")
 *         .oauthToken(SecretValue.secretsManager("my-github-token"))
 *         .variablesNamespace("MyNamespace")
 *         .build();
 * 
 * // later:
 * 
 * // later:
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "COMMIT_URL", BuildEnvironmentVariable.builder()
 *                         .value(sourceAction.getVariables().getCommitUrl())
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>BitBucket</h3>
 * <p>
 * CodePipeline can use a BitBucket Git repository as a source:
 * <p>
 * <strong>Note</strong>: you have to manually connect CodePipeline through the AWS Console with your BitBucket account.
 * This is a one-time operation for a given AWS account in a given region.
 * The simplest way to do that is to either start creating a new CodePipeline,
 * or edit an existing one, while being logged in to BitBucket.
 * Choose BitBucket as the source,
 * and grant CodePipeline permissions to your BitBucket account.
 * Copy &amp; paste the Connection ARN that you get in the console,
 * or use the <a href="https://docs.aws.amazon.com/cli/latest/reference/codestar-connections/list-connections.html"><code>codestar-connections list-connections</code> AWS CLI operation</a>
 * to find it.
 * After that, you can safely abort creating or editing the pipeline -
 * the connection has already been created.
 * <p>
 * <blockquote><pre>
 * Artifact sourceOutput = new Artifact();
 * CodeStarConnectionsSourceAction sourceAction = CodeStarConnectionsSourceAction.Builder.create()
 *         .actionName("BitBucket_Source")
 *         .owner("aws")
 *         .repo("aws-cdk")
 *         .output(sourceOutput)
 *         .connectionArn("arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh")
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also use the <code>CodeStarConnectionsSourceAction</code> to connect to GitHub, in the same way
 * (you just have to select GitHub as the source when creating the connection in the console).
 * <p>
 * Similarly to <code>GitHubSourceAction</code>, <code>CodeStarConnectionsSourceAction</code> also emits the variables:
 * <p>
 * <blockquote><pre>
 * Project project;
 * 
 * 
 * Artifact sourceOutput = new Artifact();
 * CodeStarConnectionsSourceAction sourceAction = CodeStarConnectionsSourceAction.Builder.create()
 *         .actionName("BitBucket_Source")
 *         .owner("aws")
 *         .repo("aws-cdk")
 *         .output(sourceOutput)
 *         .connectionArn("arn:aws:codestar-connections:us-east-1:123456789012:connection/12345678-abcd-12ab-34cdef5678gh")
 *         .variablesNamespace("SomeSpace")
 *         .build();
 * 
 * // later:
 * 
 * // later:
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "COMMIT_ID", BuildEnvironmentVariable.builder()
 *                         .value(sourceAction.getVariables().getCommitId())
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>AWS S3 Source</h3>
 * <p>
 * To use an S3 Bucket as a source in CodePipeline:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket = Bucket.Builder.create(this, "MyBucket")
 *         .versioned(true)
 *         .build();
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Artifact sourceOutput = new Artifact();
 * S3SourceAction sourceAction = S3SourceAction.Builder.create()
 *         .actionName("S3Source")
 *         .bucket(sourceBucket)
 *         .bucketKey("path/to/file.zip")
 *         .output(sourceOutput)
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(sourceAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The region of the action will be determined by the region the bucket itself is in.
 * When using a newly created bucket,
 * that region will be taken from the stack the bucket belongs to;
 * for an imported bucket,
 * you can specify the region explicitly:
 * <p>
 * <blockquote><pre>
 * IBucket sourceBucket = Bucket.fromBucketAttributes(this, "SourceBucket", BucketAttributes.builder()
 *         .bucketName("amzn-s3-demo-bucket")
 *         .region("ap-southeast-1")
 *         .build());
 * </pre></blockquote>
 * <p>
 * By default, the Pipeline will poll the Bucket to detect changes.
 * You can change that behavior to use CloudWatch Events by setting the <code>trigger</code>
 * property to <code>S3Trigger.EVENTS</code> (it's <code>S3Trigger.POLL</code> by default).
 * If you do that, make sure the source Bucket is part of an AWS CloudTrail Trail -
 * otherwise, the CloudWatch Events will not be emitted,
 * and your Pipeline will not react to changes in the Bucket.
 * You can do it through the CDK:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.cloudtrail.*;
 * 
 * Bucket sourceBucket;
 * 
 * Artifact sourceOutput = new Artifact();
 * String key = "some/key.zip";
 * Trail trail = new Trail(this, "CloudTrail");
 * trail.addS3EventSelector(List.of(S3EventSelector.builder()
 *         .bucket(sourceBucket)
 *         .objectPrefix(key)
 *         .build()), AddEventSelectorOptions.builder()
 *         .readWriteType(ReadWriteType.WRITE_ONLY)
 *         .build());
 * S3SourceAction sourceAction = S3SourceAction.Builder.create()
 *         .actionName("S3Source")
 *         .bucketKey(key)
 *         .bucket(sourceBucket)
 *         .output(sourceOutput)
 *         .trigger(S3Trigger.EVENTS)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The S3 source action emits variables:
 * <p>
 * <blockquote><pre>
 * Bucket sourceBucket;
 * 
 * // later:
 * PipelineProject project;
 * String key = "some/key.zip";
 * Artifact sourceOutput = new Artifact();
 * S3SourceAction sourceAction = S3SourceAction.Builder.create()
 *         .actionName("S3Source")
 *         .bucketKey(key)
 *         .bucket(sourceBucket)
 *         .output(sourceOutput)
 *         .variablesNamespace("MyNamespace")
 *         .build();
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "VERSION_ID", BuildEnvironmentVariable.builder()
 *                         .value(sourceAction.getVariables().getVersionId())
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>AWS ECR</h3>
 * <p>
 * To use an ECR Repository as a source in a Pipeline:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.*;
 * 
 * Repository ecrRepository;
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Artifact sourceOutput = new Artifact();
 * EcrSourceAction sourceAction = EcrSourceAction.Builder.create()
 *         .actionName("ECR")
 *         .repository(ecrRepository)
 *         .imageTag("some-tag") // optional, default: 'latest'
 *         .output(sourceOutput)
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(sourceAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The ECR source action emits variables:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.*;
 * Repository ecrRepository;
 * 
 * // later:
 * PipelineProject project;
 * 
 * 
 * Artifact sourceOutput = new Artifact();
 * EcrSourceAction sourceAction = EcrSourceAction.Builder.create()
 *         .actionName("Source")
 *         .output(sourceOutput)
 *         .repository(ecrRepository)
 *         .variablesNamespace("MyNamespace")
 *         .build();
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "IMAGE_URI", BuildEnvironmentVariable.builder()
 *                         .value(sourceAction.getVariables().getImageUri())
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Build &amp; test</h2>
 * <p>
 * <h3>AWS CodeBuild</h3>
 * <p>
 * Example of a CodeBuild Project used in a Pipeline, alongside CodeCommit:
 * <p>
 * <blockquote><pre>
 * PipelineProject project;
 * 
 * Repository repository = Repository.Builder.create(this, "MyRepository")
 *         .repositoryName("MyRepository")
 *         .build();
 * PipelineProject project = new PipelineProject(this, "MyProject");
 * 
 * Artifact sourceOutput = new Artifact();
 * CodeCommitSourceAction sourceAction = CodeCommitSourceAction.Builder.create()
 *         .actionName("CodeCommit")
 *         .repository(repository)
 *         .output(sourceOutput)
 *         .build();
 * CodeBuildAction buildAction = CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .outputs(List.of(new Artifact())) // optional
 *         .executeBatchBuild(true) // optional, defaults to false
 *         .combineBatchBuildArtifacts(true)
 *         .build();
 * 
 * Pipeline.Builder.create(this, "MyPipeline")
 *         .stages(List.of(StageProps.builder()
 *                 .stageName("Source")
 *                 .actions(List.of(sourceAction))
 *                 .build(), StageProps.builder()
 *                 .stageName("Build")
 *                 .actions(List.of(buildAction))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * The default category of the CodeBuild Action is <code>Build</code>;
 * if you want a <code>Test</code> Action instead,
 * override the <code>type</code> property:
 * <p>
 * <blockquote><pre>
 * PipelineProject project;
 * 
 * Artifact sourceOutput = new Artifact();
 * CodeBuildAction testAction = CodeBuildAction.Builder.create()
 *         .actionName("IntegrationTest")
 *         .project(project)
 *         .input(sourceOutput)
 *         .type(CodeBuildActionType.TEST)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Multiple inputs and outputs</h4>
 * <p>
 * When you want to have multiple inputs and/or outputs for a Project used in a
 * Pipeline, instead of using the <code>secondarySources</code> and <code>secondaryArtifacts</code>
 * properties of the <code>Project</code> class, you need to use the <code>extraInputs</code> and
 * <code>outputs</code> properties of the CodeBuild CodePipeline
 * Actions. Example:
 * <p>
 * <blockquote><pre>
 * Repository repository1;
 * Repository repository2;
 * 
 * PipelineProject project;
 * 
 * Artifact sourceOutput1 = new Artifact();
 * CodeCommitSourceAction sourceAction1 = CodeCommitSourceAction.Builder.create()
 *         .actionName("Source1")
 *         .repository(repository1)
 *         .output(sourceOutput1)
 *         .build();
 * Artifact sourceOutput2 = new Artifact("source2");
 * CodeCommitSourceAction sourceAction2 = CodeCommitSourceAction.Builder.create()
 *         .actionName("Source2")
 *         .repository(repository2)
 *         .output(sourceOutput2)
 *         .build();
 * CodeBuildAction buildAction = CodeBuildAction.Builder.create()
 *         .actionName("Build")
 *         .project(project)
 *         .input(sourceOutput1)
 *         .extraInputs(List.of(sourceOutput2))
 *         .outputs(List.of(
 *             new Artifact("artifact1"),  // for better buildspec readability - see below
 *             new Artifact("artifact2")))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <strong>Note</strong>: when a CodeBuild Action in a Pipeline has more than one output, it
 * only uses the <code>secondary-artifacts</code> field of the buildspec, never the
 * primary output specification directly under <code>artifacts</code>. Because of that, it
 * pays to explicitly name all output artifacts of that Action, like we did
 * above, so that you know what name to use in the buildspec.
 * <p>
 * Example buildspec for the above project:
 * <p>
 * <blockquote><pre>
 * PipelineProject project = PipelineProject.Builder.create(this, "MyProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of())),
 *                 "artifacts", Map.of(
 *                         "secondary-artifacts", Map.of(
 *                                 "artifact1", Map.of(),
 *                                 "artifact2", Map.of())))))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Variables</h4>
 * <p>
 * The CodeBuild action emits variables.
 * Unlike many other actions, the variables are not static,
 * but dynamic, defined in the buildspec,
 * in the 'exported-variables' subsection of the 'env' section.
 * Example:
 * <p>
 * <blockquote><pre>
 * // later:
 * PipelineProject project;
 * Artifact sourceOutput = new Artifact();
 * CodeBuildAction buildAction = CodeBuildAction.Builder.create()
 *         .actionName("Build1")
 *         .input(sourceOutput)
 *         .project(PipelineProject.Builder.create(this, "Project")
 *                 .buildSpec(BuildSpec.fromObject(Map.of(
 *                         "version", "0.2",
 *                         "env", Map.of(
 *                                 "exported-variables", List.of("MY_VAR")),
 *                         "phases", Map.of(
 *                                 "build", Map.of(
 *                                         "commands", "export MY_VAR=\"some value\"")))))
 *                 .build())
 *         .variablesNamespace("MyNamespace")
 *         .build();
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "MyVar", BuildEnvironmentVariable.builder()
 *                         .value(buildAction.variable("MY_VAR"))
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Jenkins</h3>
 * <p>
 * In order to use Jenkins Actions in the Pipeline,
 * you first need to create a <code>JenkinsProvider</code>:
 * <p>
 * <blockquote><pre>
 * JenkinsProvider jenkinsProvider = JenkinsProvider.Builder.create(this, "JenkinsProvider")
 *         .providerName("MyJenkinsProvider")
 *         .serverUrl("http://my-jenkins.com:8080")
 *         .version("2")
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you've registered a Jenkins provider in a different CDK app,
 * or outside the CDK (in the CodePipeline AWS Console, for example),
 * you can import it:
 * <p>
 * <blockquote><pre>
 * IJenkinsProvider jenkinsProvider = JenkinsProvider.fromJenkinsProviderAttributes(this, "JenkinsProvider", JenkinsProviderAttributes.builder()
 *         .providerName("MyJenkinsProvider")
 *         .serverUrl("http://my-jenkins.com:8080")
 *         .version("2")
 *         .build());
 * </pre></blockquote>
 * <p>
 * Note that a Jenkins provider
 * (identified by the provider name-category(build/test)-version tuple)
 * must always be registered in the given account, in the given AWS region,
 * before it can be used in CodePipeline.
 * <p>
 * With a <code>JenkinsProvider</code>,
 * we can create a Jenkins Action:
 * <p>
 * <blockquote><pre>
 * JenkinsProvider jenkinsProvider;
 * 
 * JenkinsAction buildAction = JenkinsAction.Builder.create()
 *         .actionName("JenkinsBuild")
 *         .jenkinsProvider(jenkinsProvider)
 *         .projectName("MyProject")
 *         .type(JenkinsActionType.BUILD)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Build</h2>
 * <p>
 * <h3>ECR Build And Publish</h3>
 * <p>
 * This build action <code>ECRBuildAndPublish</code> allows you to automate building and pushing a new image when a change occurs in your source.
 * <p>
 * This action builds based on a specified Docker file location and pushes the image. This build action is not the
 * same as the Amazon ECR source action in CodePipeline, which triggers pipeline when a change occurs in your
 * Amazon ECR source repository.
 * <p>
 * For information about the <code>ECRBuildAndPublish</code> build action,
 * see <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/action-reference-ECRBuildAndPublish.html">ECRBuildAndPublish build action reference</a>.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.*;
 * 
 * Pipeline pipeline;
 * IRepository repository;
 * 
 * 
 * Artifact sourceOutput = new Artifact();
 * // your source repository
 * CodeStarConnectionsSourceAction sourceAction = CodeStarConnectionsSourceAction.Builder.create()
 *         .actionName("CodeStarConnectionsSourceAction")
 *         .output(sourceOutput)
 *         .connectionArn("your-connection-arn")
 *         .owner("your-owner")
 *         .repo("your-repo")
 *         .build();
 * 
 * EcrBuildAndPublishAction buildAction = EcrBuildAndPublishAction.Builder.create()
 *         .actionName("EcrBuildAndPublishAction")
 *         .repositoryName(repository.getRepositoryName())
 *         .registryType(RegistryType.PRIVATE)
 *         .dockerfileDirectoryPath("./my-dir") // The path indicates ./my-dir/Dockerfile in the source repository
 *         .imageTags(List.of("my-tag-1", "my-tag-2"))
 *         .input(sourceOutput)
 *         .build();
 * 
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(sourceAction))
 *         .build());
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Build")
 *         .actions(List.of(buildAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Deploy</h2>
 * <p>
 * <h3>AWS CloudFormation</h3>
 * <p>
 * This module contains Actions that allows you to deploy to CloudFormation from AWS CodePipeline.
 * <p>
 * For example, the following code fragment defines a pipeline that automatically deploys a CloudFormation template
 * directly from a CodeCommit repository, with a manual approval step in between to confirm the changes:
 * <p>
 * <blockquote><pre>
 * // Source stage: read from repository
 * Repository repo = Repository.Builder.create(stack, "TemplateRepo")
 *         .repositoryName("template-repo")
 *         .build();
 * Artifact sourceOutput = new Artifact("SourceArtifact");
 * CodeCommitSourceAction source = CodeCommitSourceAction.Builder.create()
 *         .actionName("Source")
 *         .repository(repo)
 *         .output(sourceOutput)
 *         .trigger(CodeCommitTrigger.POLL)
 *         .build();
 * Map&lt;String, Object&gt; sourceStage = Map.of(
 *         "stageName", "Source",
 *         "actions", List.of(source));
 * 
 * // Deployment stage: create and deploy changeset with manual approval
 * String stackName = "OurStack";
 * String changeSetName = "StagedChangeSet";
 * 
 * Map&lt;String, Object&gt; prodStage = Map.of(
 *         "stageName", "Deploy",
 *         "actions", List.of(
 *             CloudFormationCreateReplaceChangeSetAction.Builder.create()
 *                     .actionName("PrepareChanges")
 *                     .stackName(stackName)
 *                     .changeSetName(changeSetName)
 *                     .adminPermissions(true)
 *                     .templatePath(sourceOutput.atPath("template.yaml"))
 *                     .runOrder(1)
 *                     .build(),
 *             ManualApprovalAction.Builder.create()
 *                     .actionName("ApproveChanges")
 *                     .runOrder(2)
 *                     .build(),
 *             CloudFormationExecuteChangeSetAction.Builder.create()
 *                     .actionName("ExecuteChanges")
 *                     .stackName(stackName)
 *                     .changeSetName(changeSetName)
 *                     .runOrder(3)
 *                     .build()));
 * 
 * Pipeline.Builder.create(stack, "Pipeline")
 *         .crossAccountKeys(true)
 *         .stages(List.of(sourceStage, prodStage))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/continuous-delivery-codepipeline.html">the AWS documentation</a>
 * for more details about using CloudFormation in CodePipeline.
 * <p>
 * <h4>Actions for updating individual CloudFormation Stacks</h4>
 * <p>
 * This package contains the following CloudFormation actions:
 * <p>
 * <ul>
 * <li><strong>CloudFormationCreateUpdateStackAction</strong> - Deploy a CloudFormation template directly from the pipeline. The indicated stack is created,
 * or updated if it already exists. If the stack is in a failure state, deployment will fail (unless <code>replaceOnFailure</code>
 * is set to <code>true</code>, in which case it will be destroyed and recreated).</li>
 * <li><strong>CloudFormationDeleteStackAction</strong> - Delete the stack with the given name.</li>
 * <li><strong>CloudFormationCreateReplaceChangeSetAction</strong> - Prepare a change set to be applied later. You will typically use change sets if you want
 * to manually verify the changes that are being staged, or if you want to separate the people (or system) preparing the
 * changes from the people (or system) applying the changes.</li>
 * <li><strong>CloudFormationExecuteChangeSetAction</strong> - Execute a change set prepared previously.</li>
 * </ul>
 * <p>
 * <h4>Actions for deploying CloudFormation StackSets to multiple accounts</h4>
 * <p>
 * You can use CloudFormation StackSets to deploy the same CloudFormation template to multiple
 * accounts in a managed way. If you use AWS Organizations, StackSets can be deployed to
 * all accounts in a particular Organizational Unit (OU), and even automatically to new
 * accounts as soon as they are added to a particular OU. For more information, see
 * the <a href="https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/what-is-cfnstacksets.html">Working with StackSets</a>
 * section of the CloudFormation developer guide.
 * <p>
 * The actions available for updating StackSets are:
 * <p>
 * <ul>
 * <li><strong>CloudFormationDeployStackSetAction</strong> - Create or update a CloudFormation StackSet directly from the pipeline, optionally
 * immediately create and update Stack Instances as well.</li>
 * <li><strong>CloudFormationDeployStackInstancesAction</strong> - Update outdated Stack Instances using the current version of the StackSet.</li>
 * </ul>
 * <p>
 * Here's an example of using both of these actions:
 * <p>
 * <blockquote><pre>
 * Pipeline pipeline;
 * Artifact sourceOutput;
 * 
 * 
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("DeployStackSets")
 *         .actions(List.of(
 *             // First, update the StackSet itself with the newest template
 *             CloudFormationDeployStackSetAction.Builder.create()
 *                     .actionName("UpdateStackSet")
 *                     .runOrder(1)
 *                     .stackSetName("MyStackSet")
 *                     .template(StackSetTemplate.fromArtifactPath(sourceOutput.atPath("template.yaml")))
 * 
 *                     // Change this to 'StackSetDeploymentModel.organizations()' if you want to deploy to OUs
 *                     .deploymentModel(StackSetDeploymentModel.selfManaged())
 *                     // This deploys to a set of accounts
 *                     .stackInstances(StackInstances.inAccounts(List.of("111111111111"), List.of("us-east-1", "eu-west-1")))
 *                     .build(),
 * 
 *             // Afterwards, update/create additional instances in other accounts
 *             CloudFormationDeployStackInstancesAction.Builder.create()
 *                     .actionName("AddMoreInstances")
 *                     .runOrder(2)
 *                     .stackSetName("MyStackSet")
 *                     .stackInstances(StackInstances.inAccounts(List.of("222222222222", "333333333333"), List.of("us-east-1", "eu-west-1")))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Lambda deployed through CodePipeline</h4>
 * <p>
 * If you want to deploy your Lambda through CodePipeline,
 * and you don't use assets (for example, because your CDK code and Lambda code are separate),
 * you can use a special Lambda <code>Code</code> class, <code>CfnParametersCode</code>.
 * Note that your Lambda must be in a different Stack than your Pipeline.
 * The Lambda itself will be deployed, alongside the entire Stack it belongs to,
 * using a CloudFormation CodePipeline Action. Example:
 * <p>
 * <blockquote><pre>
 * Stack lambdaStack = new Stack(app, "LambdaStack");
 * CfnParametersCode lambdaCode = Code.fromCfnParameters();
 * Function.Builder.create(lambdaStack, "Lambda")
 *         .code(lambdaCode)
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .build();
 * // other resources that your Lambda needs, added to the lambdaStack...
 * 
 * Stack pipelineStack = new Stack(app, "PipelineStack");
 * Pipeline pipeline = Pipeline.Builder.create(pipelineStack, "Pipeline")
 *         .crossAccountKeys(true)
 *         .build();
 * 
 * // add the source code repository containing this code to your Pipeline,
 * // and the source code of the Lambda Function, if they're separate
 * Artifact cdkSourceOutput = new Artifact();
 * CodeCommitSourceAction cdkSourceAction = CodeCommitSourceAction.Builder.create()
 *         .repository(Repository.Builder.create(pipelineStack, "CdkCodeRepo")
 *                 .repositoryName("CdkCodeRepo")
 *                 .build())
 *         .actionName("CdkCode_Source")
 *         .output(cdkSourceOutput)
 *         .build();
 * Artifact lambdaSourceOutput = new Artifact();
 * CodeCommitSourceAction lambdaSourceAction = CodeCommitSourceAction.Builder.create()
 *         .repository(Repository.Builder.create(pipelineStack, "LambdaCodeRepo")
 *                 .repositoryName("LambdaCodeRepo")
 *                 .build())
 *         .actionName("LambdaCode_Source")
 *         .output(lambdaSourceOutput)
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(cdkSourceAction, lambdaSourceAction))
 *         .build());
 * 
 * // synthesize the Lambda CDK template, using CodeBuild
 * // the below values are just examples, assuming your CDK code is in TypeScript/JavaScript -
 * // adjust the build environment and/or commands accordingly
 * Project cdkBuildProject = Project.Builder.create(pipelineStack, "CdkBuildProject")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(LinuxBuildImage.STANDARD_7_0)
 *                 .build())
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "install", Map.of(
 *                                 "commands", "npm install"),
 *                         "build", Map.of(
 *                                 "commands", List.of("npm run build", "npm run cdk synth LambdaStack -- -o ."))),
 *                 "artifacts", Map.of(
 *                         "files", "LambdaStack.template.yaml"))))
 *         .build();
 * Artifact cdkBuildOutput = new Artifact();
 * CodeBuildAction cdkBuildAction = CodeBuildAction.Builder.create()
 *         .actionName("CDK_Build")
 *         .project(cdkBuildProject)
 *         .input(cdkSourceOutput)
 *         .outputs(List.of(cdkBuildOutput))
 *         .build();
 * 
 * // build your Lambda code, using CodeBuild
 * // again, this example assumes your Lambda is written in TypeScript/JavaScript -
 * // make sure to adjust the build environment and/or commands if they don't match your specific situation
 * Project lambdaBuildProject = Project.Builder.create(pipelineStack, "LambdaBuildProject")
 *         .environment(BuildEnvironment.builder()
 *                 .buildImage(LinuxBuildImage.STANDARD_7_0)
 *                 .build())
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "install", Map.of(
 *                                 "commands", "npm install"),
 *                         "build", Map.of(
 *                                 "commands", "npm run build")),
 *                 "artifacts", Map.of(
 *                         "files", List.of("index.js", "node_modules/**&#47;*")))))
 *         .build();
 * Artifact lambdaBuildOutput = new Artifact();
 * CodeBuildAction lambdaBuildAction = CodeBuildAction.Builder.create()
 *         .actionName("Lambda_Build")
 *         .project(lambdaBuildProject)
 *         .input(lambdaSourceOutput)
 *         .outputs(List.of(lambdaBuildOutput))
 *         .build();
 * 
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Build")
 *         .actions(List.of(cdkBuildAction, lambdaBuildAction))
 *         .build());
 * 
 * // finally, deploy your Lambda Stack
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Deploy")
 *         .actions(List.of(
 *             CloudFormationCreateUpdateStackAction.Builder.create()
 *                     .actionName("Lambda_CFN_Deploy")
 *                     .templatePath(cdkBuildOutput.atPath("LambdaStack.template.yaml"))
 *                     .stackName("LambdaStackDeployedName")
 *                     .adminPermissions(true)
 *                     .parameterOverrides(lambdaCode.assign(lambdaBuildOutput.getS3Location()))
 *                     .extraInputs(List.of(lambdaBuildOutput))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Cross-account actions</h4>
 * <p>
 * If you want to update stacks in a different account,
 * pass the <code>account</code> property when creating the action:
 * <p>
 * <blockquote><pre>
 * Artifact sourceOutput = new Artifact();
 * CloudFormationCreateUpdateStackAction.Builder.create()
 *         .actionName("CloudFormationCreateUpdate")
 *         .stackName("MyStackName")
 *         .adminPermissions(true)
 *         .templatePath(sourceOutput.atPath("template.yaml"))
 *         .account("123456789012")
 *         .build();
 * </pre></blockquote>
 * <p>
 * This will create a new stack, called <code>&lt;PipelineStackName&gt;-support-123456789012</code>, in your <code>App</code>,
 * that will contain the role that the pipeline will assume in account 123456789012 before executing this action.
 * This support stack will automatically be deployed before the stack containing the pipeline.
 * <p>
 * You can also pass a role explicitly when creating the action -
 * in that case, the <code>account</code> property is ignored,
 * and the action will operate in the same account the role belongs to:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.PhysicalName;
 * 
 * // in stack for account 123456789012...
 * Stack otherAccountStack;
 * 
 * Role actionRole = Role.Builder.create(otherAccountStack, "ActionRole")
 *         .assumedBy(new AccountPrincipal("123456789012"))
 *         // the role has to have a physical name set
 *         .roleName(PhysicalName.GENERATE_IF_NEEDED)
 *         .build();
 * 
 * // in the pipeline stack...
 * Artifact sourceOutput = new Artifact();
 * CloudFormationCreateUpdateStackAction.Builder.create()
 *         .actionName("CloudFormationCreateUpdate")
 *         .stackName("MyStackName")
 *         .adminPermissions(true)
 *         .templatePath(sourceOutput.atPath("template.yaml"))
 *         .role(actionRole)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>AWS CodeDeploy</h3>
 * <p>
 * <h4>Server deployments</h4>
 * <p>
 * To use CodeDeploy for EC2/on-premise deployments in a Pipeline:
 * <p>
 * <blockquote><pre>
 * ServerDeploymentGroup deploymentGroup;
 * Pipeline pipeline = Pipeline.Builder.create(this, "MyPipeline")
 *         .pipelineName("MyPipeline")
 *         .build();
 * 
 * // add the source and build Stages to the Pipeline...
 * Artifact buildOutput = new Artifact();
 * CodeDeployServerDeployAction deployAction = CodeDeployServerDeployAction.Builder.create()
 *         .actionName("CodeDeploy")
 *         .input(buildOutput)
 *         .deploymentGroup(deploymentGroup)
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Deploy")
 *         .actions(List.of(deployAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h5>Lambda deployments</h5>
 * <p>
 * To use CodeDeploy for blue-green Lambda deployments in a Pipeline:
 * <p>
 * <blockquote><pre>
 * CfnParametersCode lambdaCode = Code.fromCfnParameters();
 * Function func = Function.Builder.create(this, "Lambda")
 *         .code(lambdaCode)
 *         .handler("index.handler")
 *         .runtime(Runtime.NODEJS_LATEST)
 *         .build();
 * // used to make sure each CDK synthesis produces a different Version
 * Version version = func.getCurrentVersion();
 * Alias alias = Alias.Builder.create(this, "LambdaAlias")
 *         .aliasName("Prod")
 *         .version(version)
 *         .build();
 * 
 * LambdaDeploymentGroup.Builder.create(this, "DeploymentGroup")
 *         .alias(alias)
 *         .deploymentConfig(LambdaDeploymentConfig.LINEAR_10PERCENT_EVERY_1MINUTE)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Then, you need to create your Pipeline Stack,
 * where you will define your Pipeline,
 * and deploy the <code>lambdaStack</code> using a CloudFormation CodePipeline Action
 * (see above for a complete example).
 * <p>
 * <h3>ECS</h3>
 * <p>
 * CodePipeline can deploy an ECS service.
 * The deploy Action receives one input Artifact which contains the <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/pipelines-create.html#pipelines-create-image-definitions">image definition file</a>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * FargateService service;
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Artifact buildOutput = new Artifact();
 * IStage deployStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("Deploy")
 *         .actions(List.of(
 *             EcsDeployAction.Builder.create()
 *                     .actionName("DeployAction")
 *                     .service(service)
 *                     // if your file is called imagedefinitions.json,
 *                     // use the `input` property,
 *                     // and leave out the `imageFile` property
 *                     .input(buildOutput)
 *                     // if your file name is _not_ imagedefinitions.json,
 *                     // use the `imageFile` property,
 *                     // and leave out the `input` property
 *                     .imageFile(buildOutput.atPath("imageDef.json"))
 *                     .deploymentTimeout(Duration.minutes(60))
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Deploying ECS applications to existing services</h4>
 * <p>
 * CodePipeline can deploy to an existing ECS service which uses the
 * <a href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs-account-settings.html#ecs-resource-ids">ECS service ARN format that contains the Cluster name</a>.
 * This also works if the service is in a different account and/or region than the pipeline:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecs.*;
 * 
 * 
 * IBaseService service = BaseService.fromServiceArnWithCluster(this, "EcsService", "arn:aws:ecs:us-east-1:123456789012:service/myClusterName/myServiceName");
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Artifact buildOutput = new Artifact();
 * // add source and build stages to the pipeline as usual...
 * IStage deployStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("Deploy")
 *         .actions(List.of(
 *             EcsDeployAction.Builder.create()
 *                     .actionName("DeployAction")
 *                     .service(service)
 *                     .input(buildOutput)
 *                     .build()))
 *         .build());
 * </pre></blockquote>
 * <p>
 * When deploying across accounts, especially in a CDK Pipelines self-mutating pipeline,
 * it is recommended to provide the <code>role</code> property to the <code>EcsDeployAction</code>.
 * The Role will need to have permissions assigned to it for ECS deployment.
 * See <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/security-iam.html#how-to-custom-role">the CodePipeline documentation</a>
 * for the permissions needed.
 * <p>
 * <h4>Deploying ECS applications stored in a separate source code repository</h4>
 * <p>
 * The idiomatic CDK way of deploying an ECS application is to have your Dockerfiles and your CDK code in the same source code repository,
 * leveraging <a href="https://docs.aws.amazon.com/cdk/latest/guide/assets.html#assets_types_docker">Docker Assets</a>,
 * and use the <a href="https://docs.aws.amazon.com/cdk/api/latest/docs/pipelines-readme.html">CDK Pipelines module</a>.
 * <p>
 * However, if you want to deploy a Docker application whose source code is kept in a separate version control repository than the CDK code,
 * you can use the <code>TagParameterContainerImage</code> class from the ECS module.
 * Here's an example:
 * <p>
 * <blockquote><pre>
 * /**
 *  * These are the construction properties for `EcsAppStack`.
 *  * They extend the standard Stack properties,
 *  * but also require providing the ContainerImage that the service will use.
 *  * That Image will be provided from the Stack containing the CodePipeline.
 *  *&#47;
 * public class EcsAppStackProps extends StackProps {
 *     private ContainerImage image;
 *     public ContainerImage getImage() {
 *         return this.image;
 *     }
 *     public EcsAppStackProps image(ContainerImage image) {
 *         this.image = image;
 *         return this;
 *     }
 * }
 * 
 * /**
 *  * This is the Stack containing a simple ECS Service that uses the provided ContainerImage.
 *  *&#47;
 * public class EcsAppStack extends Stack {
 *     public EcsAppStack(Construct scope, String id, EcsAppStackProps props) {
 *         super(scope, id, props);
 * 
 *         TaskDefinition taskDefinition = TaskDefinition.Builder.create(this, "TaskDefinition")
 *                 .compatibility(Compatibility.FARGATE)
 *                 .cpu("1024")
 *                 .memoryMiB("2048")
 *                 .build();
 *         taskDefinition.addContainer("AppContainer", ContainerDefinitionOptions.builder()
 *                 .image(props.getImage())
 *                 .build());
 *         FargateService.Builder.create(this, "EcsService")
 *                 .taskDefinition(taskDefinition)
 *                 .cluster(Cluster.Builder.create(this, "Cluster")
 *                         .vpc(Vpc.Builder.create(this, "Vpc")
 *                                 .maxAzs(1)
 *                                 .build())
 *                         .build())
 *                 .build();
 *     }
 * }
 * 
 * /**
 *  * This is the Stack containing the CodePipeline definition that deploys an ECS Service.
 *  *&#47;
 * public class PipelineStack extends Stack {
 *     public final TagParameterContainerImage tagParameterContainerImage;
 * 
 *     public PipelineStack(Construct scope, String id) {
 *         this(scope, id, null);
 *     }
 * 
 *     public PipelineStack(Construct scope, String id, StackProps props) {
 *         super(scope, id, props);
 * 
 *         /* ********** ECS part **************** *&#47;
 * 
 *         // this is the ECR repository where the built Docker image will be pushed
 *         Repository appEcrRepo = new Repository(this, "EcsDeployRepository");
 *         // the build that creates the Docker image, and pushes it to the ECR repo
 *         PipelineProject appCodeDockerBuild = PipelineProject.Builder.create(this, "AppCodeDockerImageBuildAndPushProject")
 *                 .environment(BuildEnvironment.builder()
 *                         // we need to run Docker
 *                         .privileged(true)
 *                         .build())
 *                 .buildSpec(BuildSpec.fromObject(Map.of(
 *                         "version", "0.2",
 *                         "phases", Map.of(
 *                                 "build", Map.of(
 *                                         "commands", List.of("$(aws ecr get-login --region $AWS_DEFAULT_REGION --no-include-email)", "docker build -t $REPOSITORY_URI:$CODEBUILD_RESOLVED_SOURCE_VERSION .")),
 *                                 "post_build", Map.of(
 *                                         "commands", List.of("docker push $REPOSITORY_URI:$CODEBUILD_RESOLVED_SOURCE_VERSION", "export imageTag=$CODEBUILD_RESOLVED_SOURCE_VERSION"))),
 *                         "env", Map.of(
 *                                 // save the imageTag environment variable as a CodePipeline Variable
 *                                 "exported-variables", List.of("imageTag")))))
 *                 .environmentVariables(Map.of(
 *                         "REPOSITORY_URI", BuildEnvironmentVariable.builder()
 *                                 .value(appEcrRepo.getRepositoryUri())
 *                                 .build()))
 *                 .build();
 *         // needed for `docker push`
 *         appEcrRepo.grantPullPush(appCodeDockerBuild);
 *         // create the ContainerImage used for the ECS application Stack
 *         this.tagParameterContainerImage = new TagParameterContainerImage(appEcrRepo);
 * 
 *         PipelineProject cdkCodeBuild = PipelineProject.Builder.create(this, "CdkCodeBuildProject")
 *                 .buildSpec(BuildSpec.fromObject(Map.of(
 *                         "version", "0.2",
 *                         "phases", Map.of(
 *                                 "install", Map.of(
 *                                         "commands", List.of("npm install")),
 *                                 "build", Map.of(
 *                                         "commands", List.of("npx cdk synth --verbose"))),
 *                         "artifacts", Map.of(
 *                                 // store the entire Cloud Assembly as the output artifact
 *                                 "base-directory", "cdk.out",
 *                                 "files", "**&#47;*"))))
 *                 .build();
 * 
 *         /* ********** Pipeline part **************** *&#47;
 * 
 *         Artifact appCodeSourceOutput = new Artifact();
 *         Artifact cdkCodeSourceOutput = new Artifact();
 *         Artifact cdkCodeBuildOutput = new Artifact();
 *         CodeBuildAction appCodeBuildAction = CodeBuildAction.Builder.create()
 *                 .actionName("AppCodeDockerImageBuildAndPush")
 *                 .project(appCodeDockerBuild)
 *                 .input(appCodeSourceOutput)
 *                 .build();
 *         Pipeline.Builder.create(this, "CodePipelineDeployingEcsApplication")
 *                 .artifactBucket(Bucket.Builder.create(this, "ArtifactBucket")
 *                         .removalPolicy(RemovalPolicy.DESTROY)
 *                         .build())
 *                 .stages(List.of(StageProps.builder()
 *                         .stageName("Source")
 *                         .actions(List.of(
 *                             // this is the Action that takes the source of your application code
 *                             CodeCommitSourceAction.Builder.create()
 *                                     .actionName("AppCodeSource")
 *                                     .repository(Repository.Builder.create(this, "AppCodeSourceRepository").repositoryName("AppCodeSourceRepository").build())
 *                                     .output(appCodeSourceOutput)
 *                                     .build(),
 *                             // this is the Action that takes the source of your CDK code
 *                             // (which would probably include this Pipeline code as well)
 *                             CodeCommitSourceAction.Builder.create()
 *                                     .actionName("CdkCodeSource")
 *                                     .repository(Repository.Builder.create(this, "CdkCodeSourceRepository").repositoryName("CdkCodeSourceRepository").build())
 *                                     .output(cdkCodeSourceOutput)
 *                                     .build()))
 *                         .build(), StageProps.builder()
 *                         .stageName("Build")
 *                         .actions(List.of(appCodeBuildAction,
 *                             CodeBuildAction.Builder.create()
 *                                     .actionName("CdkCodeBuildAndSynth")
 *                                     .project(cdkCodeBuild)
 *                                     .input(cdkCodeSourceOutput)
 *                                     .outputs(List.of(cdkCodeBuildOutput))
 *                                     .build()))
 *                         .build(), StageProps.builder()
 *                         .stageName("Deploy")
 *                         .actions(List.of(
 *                             CloudFormationCreateUpdateStackAction.Builder.create()
 *                                     .actionName("CFN_Deploy")
 *                                     .stackName("SampleEcsStackDeployedFromCodePipeline")
 *                                     // this name has to be the same name as used below in the CDK code for the application Stack
 *                                     .templatePath(cdkCodeBuildOutput.atPath("EcsStackDeployedInPipeline.template.json"))
 *                                     .adminPermissions(true)
 *                                     .parameterOverrides(Map.of(
 *                                             // read the tag pushed to the ECR repository from the CodePipeline Variable saved by the application build step,
 *                                             // and pass it as the CloudFormation Parameter for the tag
 *                                             this.tagParameterContainerImage.getTagParameterName(), appCodeBuildAction.variable("imageTag")))
 *                                     .build()))
 *                         .build()))
 *                 .build();
 *     }
 * }
 * 
 * App app = new App();
 * 
 * // the CodePipeline Stack needs to be created first
 * PipelineStack pipelineStack = new PipelineStack(app, "aws-cdk-pipeline-ecs-separate-sources");
 * // we supply the image to the ECS application Stack from the CodePipeline Stack
 * // we supply the image to the ECS application Stack from the CodePipeline Stack
 * new EcsAppStack(app, "EcsStackDeployedInPipeline", new EcsAppStackProps()
 *         .image(pipelineStack.getTagParameterContainerImage())
 *         );
 * </pre></blockquote>
 * <p>
 * <h3>AWS S3 Deployment</h3>
 * <p>
 * To use an S3 Bucket as a deployment target in CodePipeline:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.kms.*;
 * 
 * 
 * Artifact sourceOutput = new Artifact();
 * Bucket targetBucket = new Bucket(this, "MyBucket");
 * IKey key = Key.Builder.create(this, "EnvVarEncryptKey")
 *         .description("sample key")
 *         .build();
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * S3DeployAction deployAction = S3DeployAction.Builder.create()
 *         .actionName("S3Deploy")
 *         .bucket(targetBucket)
 *         .input(sourceOutput)
 *         .encryptionKey(key)
 *         .build();
 * IStage deployStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("Deploy")
 *         .actions(List.of(deployAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Invalidating the CloudFront cache when deploying to S3</h4>
 * <p>
 * There is currently no native support in CodePipeline for invalidating a CloudFront cache after deployment.
 * One workaround is to add another build step after the deploy step,
 * and use the AWS CLI to invalidate the cache:
 * <p>
 * <blockquote><pre>
 * // Create a Cloudfront Web Distribution
 * import software.amazon.awscdk.services.cloudfront.*;
 * Distribution distribution;
 * 
 * 
 * // Create the build project that will invalidate the cache
 * PipelineProject invalidateBuildProject = PipelineProject.Builder.create(this, "InvalidateProject")
 *         .buildSpec(BuildSpec.fromObject(Map.of(
 *                 "version", "0.2",
 *                 "phases", Map.of(
 *                         "build", Map.of(
 *                                 "commands", List.of("aws cloudfront create-invalidation --distribution-id ${CLOUDFRONT_ID} --paths \"/*\""))))))
 *         .environmentVariables(Map.of(
 *                 "CLOUDFRONT_ID", BuildEnvironmentVariable.builder().value(distribution.getDistributionId()).build()))
 *         .build();
 * 
 * // Add Cloudfront invalidation permissions to the project
 * String distributionArn = String.format("arn:aws:cloudfront::%s:distribution/%s", this.account, distribution.getDistributionId());
 * invalidateBuildProject.addToRolePolicy(PolicyStatement.Builder.create()
 *         .resources(List.of(distributionArn))
 *         .actions(List.of("cloudfront:CreateInvalidation"))
 *         .build());
 * 
 * // Create the pipeline (here only the S3 deploy and Invalidate cache build)
 * Bucket deployBucket = new Bucket(this, "DeployBucket");
 * Artifact deployInput = new Artifact();
 * Pipeline.Builder.create(this, "Pipeline")
 *         .stages(List.of(StageProps.builder()
 *                 .stageName("Deploy")
 *                 .actions(List.of(
 *                     S3DeployAction.Builder.create()
 *                             .actionName("S3Deploy")
 *                             .bucket(deployBucket)
 *                             .input(deployInput)
 *                             .runOrder(1)
 *                             .build(),
 *                     CodeBuildAction.Builder.create()
 *                             .actionName("InvalidateCache")
 *                             .project(invalidateBuildProject)
 *                             .input(deployInput)
 *                             .runOrder(2)
 *                             .build()))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Elastic Beanstalk Deployment</h3>
 * <p>
 * To deploy an Elastic Beanstalk Application in CodePipeline:
 * <p>
 * <blockquote><pre>
 * Artifact sourceOutput = new Artifact();
 * Bucket targetBucket = new Bucket(this, "MyBucket");
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * ElasticBeanstalkDeployAction deployAction = ElasticBeanstalkDeployAction.Builder.create()
 *         .actionName("ElasticBeanstalkDeploy")
 *         .input(sourceOutput)
 *         .environmentName("envName")
 *         .applicationName("appName")
 *         .build();
 * 
 * IStage deployStage = pipeline.addStage(StageOptions.builder()
 *         .stageName("Deploy")
 *         .actions(List.of(deployAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h3>Alexa Skill</h3>
 * <p>
 * You can deploy to Alexa using CodePipeline with the following Action:
 * <p>
 * <blockquote><pre>
 * // Read the secrets from ParameterStore
 * SecretValue clientId = SecretValue.secretsManager("AlexaClientId");
 * SecretValue clientSecret = SecretValue.secretsManager("AlexaClientSecret");
 * SecretValue refreshToken = SecretValue.secretsManager("AlexaRefreshToken");
 * 
 * // Add deploy action
 * Artifact sourceOutput = new Artifact();
 * AlexaSkillDeployAction.Builder.create()
 *         .actionName("DeploySkill")
 *         .runOrder(1)
 *         .input(sourceOutput)
 *         .clientId(clientId.toString())
 *         .clientSecret(clientSecret)
 *         .refreshToken(refreshToken)
 *         .skillId("amzn1.ask.skill.12345678-1234-1234-1234-123456789012")
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you need manifest overrides you can specify them as <code>parameterOverridesArtifact</code> in the action:
 * <p>
 * <blockquote><pre>
 * // Deploy some CFN change set and store output
 * Artifact executeOutput = new Artifact("CloudFormation");
 * CloudFormationExecuteChangeSetAction executeChangeSetAction = CloudFormationExecuteChangeSetAction.Builder.create()
 *         .actionName("ExecuteChangesTest")
 *         .runOrder(2)
 *         .stackName("MyStack")
 *         .changeSetName("MyChangeSet")
 *         .outputFileName("overrides.json")
 *         .output(executeOutput)
 *         .build();
 * 
 * // Provide CFN output as manifest overrides
 * SecretValue clientId = SecretValue.secretsManager("AlexaClientId");
 * SecretValue clientSecret = SecretValue.secretsManager("AlexaClientSecret");
 * SecretValue refreshToken = SecretValue.secretsManager("AlexaRefreshToken");
 * Artifact sourceOutput = new Artifact();
 * AlexaSkillDeployAction.Builder.create()
 *         .actionName("DeploySkill")
 *         .runOrder(1)
 *         .input(sourceOutput)
 *         .parameterOverridesArtifact(executeOutput)
 *         .clientId(clientId.toString())
 *         .clientSecret(clientSecret)
 *         .refreshToken(refreshToken)
 *         .skillId("amzn1.ask.skill.12345678-1234-1234-1234-123456789012")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>AWS Service Catalog</h3>
 * <p>
 * You can deploy a CloudFormation template to an existing Service Catalog product with the following Action:
 * <p>
 * <blockquote><pre>
 * Artifact cdkBuildOutput = new Artifact();
 * ServiceCatalogDeployActionBeta1 serviceCatalogDeployAction = ServiceCatalogDeployActionBeta1.Builder.create()
 *         .actionName("ServiceCatalogDeploy")
 *         .templatePath(cdkBuildOutput.atPath("Sample.template.json"))
 *         .productVersionName("Version - " + Date.getNow().getToString())
 *         .productVersionDescription("This is a version from the pipeline with a new description.")
 *         .productId("prod-XXXXXXXX")
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h2>Approve &amp; invoke</h2>
 * <p>
 * <h3>Manual approval Action</h3>
 * <p>
 * This package contains an Action that stops the Pipeline until someone manually clicks the approve button:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.sns.*;
 * 
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * IStage approveStage = pipeline.addStage(StageOptions.builder().stageName("Approve").build());
 * ManualApprovalAction manualApprovalAction = ManualApprovalAction.Builder.create()
 *         .actionName("Approve")
 *         .notificationTopic(new Topic(this, "Topic")) // optional
 *         .notifyEmails(List.of("some_email&#64;example.com")) // optional
 *         .additionalInformation("additional info") // optional
 *         .timeout(Duration.minutes(10))
 *         .build();
 * approveStage.addAction(manualApprovalAction);
 * </pre></blockquote>
 * <p>
 * If the <code>notificationTopic</code> has not been provided,
 * but <code>notifyEmails</code> were,
 * a new SNS Topic will be created
 * (and accessible through the <code>notificationTopic</code> property of the Action).
 * <p>
 * If you want to grant a principal permissions to approve the changes,
 * you can invoke the method <code>grantManualApproval</code> passing it a <code>IGrantable</code>:
 * <p>
 * <blockquote><pre>
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * IStage approveStage = pipeline.addStage(StageOptions.builder().stageName("Approve").build());
 * ManualApprovalAction manualApprovalAction = ManualApprovalAction.Builder.create()
 *         .actionName("Approve")
 *         .build();
 * approveStage.addAction(manualApprovalAction);
 * 
 * IRole role = Role.fromRoleArn(this, "Admin", Arn.format(ArnComponents.builder().service("iam").resource("role").resourceName("Admin").build(), this));
 * manualApprovalAction.grantManualApproval(role);
 * </pre></blockquote>
 * <p>
 * <h3>AWS Lambda</h3>
 * <p>
 * This module contains an Action that allows you to invoke a Lambda function in a Pipeline:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * LambdaInvokeAction lambdaAction = LambdaInvokeAction.Builder.create()
 *         .actionName("Lambda")
 *         .lambda(fn)
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Lambda")
 *         .actions(List.of(lambdaAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The Lambda Action can have up to 5 inputs,
 * and up to 5 outputs:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * Artifact sourceOutput = new Artifact();
 * Artifact buildOutput = new Artifact();
 * LambdaInvokeAction lambdaAction = LambdaInvokeAction.Builder.create()
 *         .actionName("Lambda")
 *         .inputs(List.of(sourceOutput, buildOutput))
 *         .outputs(List.of(
 *             new Artifact("Out1"),
 *             new Artifact("Out2")))
 *         .lambda(fn)
 *         .build();
 * </pre></blockquote>
 * <p>
 * The Lambda Action supports custom user parameters that pipeline
 * will pass to the Lambda function:
 * <p>
 * <blockquote><pre>
 * Function fn;
 * 
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * LambdaInvokeAction lambdaAction = LambdaInvokeAction.Builder.create()
 *         .actionName("Lambda")
 *         .lambda(fn)
 *         .userParameters(Map.of(
 *                 "foo", "bar",
 *                 "baz", "qux"))
 *         // OR
 *         .userParametersString("my-parameter-string")
 *         .build();
 * </pre></blockquote>
 * <p>
 * The Lambda invoke action emits variables.
 * Unlike many other actions, the variables are not static,
 * but dynamic, defined by the function calling the <code>PutJobSuccessResult</code>
 * API with the <code>outputVariables</code> property filled with the map of variables
 * Example:
 * <p>
 * <blockquote><pre>
 * // later:
 * PipelineProject project;
 * LambdaInvokeAction lambdaInvokeAction = LambdaInvokeAction.Builder.create()
 *         .actionName("Lambda")
 *         .lambda(Function.Builder.create(this, "Func")
 *                 .runtime(Runtime.NODEJS_LATEST)
 *                 .handler("index.handler")
 *                 .code(Code.fromInline("\n        const { CodePipeline } = require('&#64;aws-sdk/client-codepipeline');\n\n        exports.handler = async function(event, context) {\n            const codepipeline = new AWS.CodePipeline();\n            await codepipeline.putJobSuccessResult({\n                jobId: event['CodePipeline.job'].id,\n                outputVariables: {\n                    MY_VAR: \"some value\",\n                },\n            });\n        }\n    "))
 *                 .build())
 *         .variablesNamespace("MyNamespace")
 *         .build();
 * Artifact sourceOutput = new Artifact();
 * CodeBuildAction.Builder.create()
 *         .actionName("CodeBuild")
 *         .project(project)
 *         .input(sourceOutput)
 *         .environmentVariables(Map.of(
 *                 "MyVar", BuildEnvironmentVariable.builder()
 *                         .value(lambdaInvokeAction.variable("MY_VAR"))
 *                         .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-invoke-lambda-function.html">the AWS documentation</a>
 * on how to write a Lambda function invoked from CodePipeline.
 * <p>
 * <h3>AWS Step Functions</h3>
 * <p>
 * This module contains an Action that allows you to invoke a Step Function in a Pipeline:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Pass startState = new Pass(this, "StartState");
 * StateMachine simpleStateMachine = StateMachine.Builder.create(this, "SimpleStateMachine")
 *         .definition(startState)
 *         .build();
 * StepFunctionInvokeAction stepFunctionAction = StepFunctionInvokeAction.Builder.create()
 *         .actionName("Invoke")
 *         .stateMachine(simpleStateMachine)
 *         .stateMachineInput(StateMachineInput.literal(Map.of("IsHelloWorldExample", true)))
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("StepFunctions")
 *         .actions(List.of(stepFunctionAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * The <code>StateMachineInput</code> can be created with one of 2 static factory methods:
 * <code>literal</code>, which takes an arbitrary map as its only argument, or <code>filePath</code>:
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.stepfunctions.*;
 * 
 * 
 * Pipeline pipeline = new Pipeline(this, "MyPipeline");
 * Artifact inputArtifact = new Artifact();
 * Pass startState = new Pass(this, "StartState");
 * StateMachine simpleStateMachine = StateMachine.Builder.create(this, "SimpleStateMachine")
 *         .definition(startState)
 *         .build();
 * StepFunctionInvokeAction stepFunctionAction = StepFunctionInvokeAction.Builder.create()
 *         .actionName("Invoke")
 *         .stateMachine(simpleStateMachine)
 *         .stateMachineInput(StateMachineInput.filePath(inputArtifact.atPath("assets/input.json")))
 *         .build();
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("StepFunctions")
 *         .actions(List.of(stepFunctionAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/action-reference-StepFunctions.html">the AWS documentation</a>
 * for information on Action structure reference.
 * <p>
 * <h2>Invoke</h2>
 * <p>
 * <h3>Inspector</h3>
 * <p>
 * Amazon Inspector is a vulnerability management service that automatically discovers workloads and continually scans them
 * for software vulnerabilities and unintended network exposure.
 * <p>
 * The actions <code>InspectorSourceCodeScanAction</code> and <code>InspectorEcrImageScanAction</code> automate detecting and fixing security
 * vulnerabilities in your open source code. The actions are managed compute actions with security scanning capabilities.
 * You can use the actions with application source code in your third-party repository, such as GitHub or Bitbucket Cloud,
 * or with images for container applications.
 * <p>
 * Your actions will scan and report on vulnerability levels and alerts that you configure.
 * <p>
 * <h4>Inspector Source Code Scan</h4>
 * <p>
 * The <code>InspectorSourceCodeScanAction</code> allows you to scan the application source code for vulnerabilities in your repository.
 * <p>
 * <blockquote><pre>
 * Pipeline pipeline;
 * 
 * 
 * Artifact sourceOutput = new Artifact();
 * CodeStarConnectionsSourceAction sourceAction = CodeStarConnectionsSourceAction.Builder.create()
 *         .actionName("CodeStarConnectionsSourceAction")
 *         .output(sourceOutput)
 *         .connectionArn("your-connection-arn")
 *         .owner("your-owner")
 *         .repo("your-repo")
 *         .build();
 * 
 * Artifact scanOutput = new Artifact();
 * InspectorSourceCodeScanAction scanAction = InspectorSourceCodeScanAction.Builder.create()
 *         .actionName("InspectorSourceCodeScanAction")
 *         .input(sourceOutput)
 *         .output(scanOutput)
 *         .build();
 * 
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Source")
 *         .actions(List.of(sourceAction))
 *         .build());
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Scan")
 *         .actions(List.of(scanAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h4>Inspector ECR Image Scan</h4>
 * <p>
 * The <code>InspectorEcrImageScanAction</code> allows you to scan the image for vulnerabilities in your container applications.
 * <p>
 * <blockquote><pre>
 * import software.amazon.awscdk.services.ecr.*;
 * 
 * Pipeline pipeline;
 * IRepository repository;
 * 
 * 
 * Artifact scanOutput = new Artifact();
 * InspectorEcrImageScanAction scanAction = InspectorEcrImageScanAction.Builder.create()
 *         .actionName("InspectorEcrImageScanAction")
 *         .output(scanOutput)
 *         .repository(repository)
 *         .build();
 * 
 * pipeline.addStage(StageOptions.builder()
 *         .stageName("Scan")
 *         .actions(List.of(scanAction))
 *         .build());
 * </pre></blockquote>
 * <p>
 * <h2>Compute</h2>
 * <p>
 * <h3>Commands</h3>
 * <p>
 * The Commands action allows you to run shell commands in a virtual compute instance. When you run the action, commands
 * specified in the action configuration are run in a separate container. All artifacts that are specified as input
 * artifacts to a CodeBuild action are available inside of the container running the commands. This action allows you
 * to specify commands without first creating a CodeBuild project.
 * <p>
 * <blockquote><pre>
 * // Source action
 * Bucket bucket = Bucket.Builder.create(this, "SourceBucket")
 *         .versioned(true)
 *         .build();
 * Artifact sourceArtifact = new Artifact("SourceArtifact");
 * S3SourceAction sourceAction = S3SourceAction.Builder.create()
 *         .actionName("Source")
 *         .output(sourceArtifact)
 *         .bucket(bucket)
 *         .bucketKey("my.zip")
 *         .build();
 * 
 * // Commands action
 * Artifact outputArtifact = new Artifact("OutputArtifact");
 * CommandsAction commandsAction = CommandsAction.Builder.create()
 *         .actionName("Commands")
 *         .commands(List.of("echo \"some commands\""))
 *         .input(sourceArtifact)
 *         .output(outputArtifact)
 *         .build();
 * 
 * Pipeline pipeline = Pipeline.Builder.create(this, "Pipeline")
 *         .stages(List.of(StageProps.builder()
 *                 .stageName("Source")
 *                 .actions(List.of(sourceAction))
 *                 .build(), StageProps.builder()
 *                 .stageName("Commands")
 *                 .actions(List.of(commandsAction))
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you want to filter the files to be included in the output artifact, you can specify their paths as the second
 * argument to <code>Artifact</code>.
 * <p>
 * <blockquote><pre>
 * Artifact sourceArtifact;
 * 
 * 
 * // filter files to be included in the output artifact
 * Artifact outputArtifact = new Artifact("OutputArtifact", List.of("my-dir/**&#47;*"));
 * CommandsAction commandsAction = CommandsAction.Builder.create()
 *         .actionName("Commands")
 *         .commands(List.of("mkdir -p my-dir", "echo \"HelloWorld\" &gt; my-dir/file.txt"))
 *         .input(sourceArtifact)
 *         .output(outputArtifact)
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can also specify the <code>outputVariables</code> property in the <code>CommandsAction</code> to emit environment variables that can be used
 * in subsequent actions. The variables are those defined in your shell commands or exported as defaults by the CodeBuild service.
 * For a reference of CodeBuild environment variables, see
 * <a href="https://docs.aws.amazon.com/codebuild/latest/userguide/build-env-ref-env-vars.html">Environment variables in build environments</a>
 * in the CodeBuild User Guide.
 * <p>
 * To use the output variables in a subsequent action, you can use the <code>variable</code> method on the action:
 * <p>
 * <blockquote><pre>
 * Artifact sourceArtifact;
 * Artifact outputArtifact;
 * 
 * 
 * CommandsAction commandsAction = CommandsAction.Builder.create()
 *         .actionName("Commands")
 *         .commands(List.of("export MY_OUTPUT=my-key"))
 *         .input(sourceArtifact)
 *         .output(outputArtifact)
 *         .outputVariables(List.of("MY_OUTPUT", "CODEBUILD_BUILD_ID"))
 *         .build();
 * 
 * // Deploy action
 * S3DeployAction deployAction = S3DeployAction.Builder.create()
 *         .actionName("DeployAction")
 *         .extract(true)
 *         .input(outputArtifact)
 *         .bucket(new Bucket(this, "DeployBucket"))
 *         .objectKey(commandsAction.variable("MY_OUTPUT"))
 *         .build();
 * </pre></blockquote>
 * <p>
 * See <a href="https://docs.aws.amazon.com/codepipeline/latest/userguide/action-reference-Commands.html">the AWS documentation</a>
 * for more details about using Commands action in CodePipeline.
 */
package software.amazon.awscdk.services.codepipeline.actions;
