/**
 * <h1>AWS Batch Construct Library</h1>
 * <p>
 * This module is part of the <a href="https://github.com/aws/aws-cdk">AWS Cloud Development Kit</a> project.
 * <p>
 * AWS Batch is a batch processing tool for efficiently running hundreds of thousands computing jobs in AWS.
 * Batch can dynamically provision <a href="https://aws.amazon.com/ec2/">Amazon EC2</a> Instances to meet the resource requirements of submitted jobs
 * and simplifies the planning, scheduling, and executions of your batch workloads. Batch achieves this through four different resources:
 * <p>
 * <ul>
 * <li>ComputeEnvironments: Contain the resources used to execute Jobs</li>
 * <li>JobDefinitions: Define a type of Job that can be submitted</li>
 * <li>JobQueues: Route waiting Jobs to ComputeEnvironments</li>
 * <li>SchedulingPolicies: Applied to Queues to control how and when Jobs exit the JobQueue and enter the ComputeEnvironment</li>
 * </ul>
 * <p>
 * <code>ComputeEnvironment</code>s can be managed or unmanaged. Batch will automatically provision EC2 Instances in a managed <code>ComputeEnvironment</code> and will
 * not provision any Instances in an unmanaged <code>ComputeEnvironment</code>. Managed <code>ComputeEnvironment</code>s can use ECS, Fargate, or EKS resources to spin up
 * EC2 Instances in (ensure your EKS Cluster has <a href="https://docs.aws.amazon.com/batch/latest/userguide/getting-started-eks.html">been configured</a>
 * to support a Batch ComputeEnvironment before linking it). You can use Launch Templates and Placement Groups to configure exactly how these resources
 * will be provisioned.
 * <p>
 * <code>JobDefinition</code>s can use either ECS resources or EKS resources. ECS <code>JobDefinition</code>s can use multiple containers to execute distributed workloads.
 * EKS <code>JobDefinition</code>s can only execute a single container. Submitted Jobs use <code>JobDefinition</code>s as templates.
 * <p>
 * <code>JobQueue</code>s must link at least one <code>ComputeEnvironment</code>. Jobs exit the Queue in FIFO order unless a <code>SchedulingPolicy</code> is specified.
 * <p>
 * <code>SchedulingPolicy</code>s tell the Scheduler how to choose which Jobs should be executed next by the ComputeEnvironment.
 * <p>
 * <h2>Use Cases &amp; Examples</h2>
 * <p>
 * <h3>Cost Optimization</h3>
 * <p>
 * <h4>Spot Instances</h4>
 * <p>
 * Spot instances are significantly discounted EC2 instances that can be reclaimed at any time by AWS.
 * Workloads that are fault-tolerant or stateless can take advantage of spot pricing.
 * To use spot spot instances, set <code>spot</code> to <code>true</code> on a managed Ec2 or Fargate Compute Environment:
 * <p>
 * <blockquote><pre>
 * Vpc vpc = new Vpc(this, "VPC");
 * FargateComputeEnvironment.Builder.create(this, "myFargateComputeEnv")
 *         .vpc(vpc)
 *         .spot(true)
 *         .build();
 * </pre></blockquote>
 * <p>
 * Batch allows you to specify the percentage of the on-demand instance that the current spot price
 * must be to provision the instance using the <code>spotBidPercentage</code>.
 * This defaults to 100%, which is the recommended value.
 * This value cannot be specified for <code>FargateComputeEnvironment</code>s
 * and only applies to <code>ManagedEc2EcsComputeEnvironment</code>s.
 * The following code configures a Compute Environment to only use spot instances that
 * are at most 20% the price of the on-demand instance price:
 * <p>
 * <em>Note</em>: For <code>FargateComputeEnvironment</code>, while the <code>FargateComputeEnvironmentProps</code> interface includes properties like <code>replaceComputeEnvironment</code>, <code>terminateOnUpdate</code>, <code>updateTimeout</code>, and <code>updateToLatestImageVersion</code>, these specific properties are <strong>not applicable</strong> when configuring AWS Batch Fargate compute environments. They primarily apply to EC2-based compute environments. Please refer to the official <a href="https://docs.aws.amazon.com/batch/latest/APIReference/API_UpdateComputeEnvironment.html">AWS Batch UpdateComputeEnvironment API documentation</a> and <a href="https://docs.aws.amazon.com/batch/latest/userguide/updating-compute-environments.html">User Guide</a> for details on updating Fargate compute environments.
 * <p>
 * <blockquote><pre>
 * Vpc vpc = new Vpc(this, "VPC");
 * ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .spot(true)
 *         .spotBidPercentage(20)
 *         .build();
 * </pre></blockquote>
 * <p>
 * For stateful or otherwise non-interruption-tolerant workflows, omit <code>spot</code> or set it to <code>false</code> to only provision on-demand instances.
 * <p>
 * <h4>Choosing Your Instance Types</h4>
 * <p>
 * Batch allows you to choose the instance types or classes that will run your workload.
 * This example configures your <code>ComputeEnvironment</code> to use only the <code>M5AD.large</code> instance:
 * <p>
 * <blockquote><pre>
 * Vpc vpc = new Vpc(this, "VPC");
 * 
 * ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .instanceTypes(List.of(InstanceType.of(InstanceClass.M5AD, InstanceSize.LARGE)))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Batch allows you to specify only the instance class and to let it choose the size, which you can do like this:
 * <p>
 * <blockquote><pre>
 * IManagedEc2EcsComputeEnvironment computeEnv;
 * Vpc vpc = new Vpc(this, "VPC");
 * computeEnv.addInstanceClass(InstanceClass.M5AD);
 * // Or, specify it on the constructor:
 * // Or, specify it on the constructor:
 * ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .instanceClasses(List.of(InstanceClass.R4))
 *         .build();
 * </pre></blockquote>
 * <p>
 * Unless you explicitly specify <code>useOptimalInstanceClasses: false</code>, this compute environment will use <code>'optimal'</code> instances,
 * which tells Batch to pick an instance from the C4, M4, and R4 instance families.
 * <em>Note</em>: Batch does not allow specifying instance types or classes with different architectures.
 * For example, <code>InstanceClass.A1</code> cannot be specified alongside <code>'optimal'</code>,
 * because <code>A1</code> uses ARM and <code>'optimal'</code> uses x86_64.
 * You can specify both <code>'optimal'</code> alongside several different instance types in the same compute environment:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * ManagedEc2EcsComputeEnvironment computeEnv = ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .instanceTypes(List.of(InstanceType.of(InstanceClass.M5AD, InstanceSize.LARGE)))
 *         .useOptimalInstanceClasses(true) // default
 *         .vpc(vpc)
 *         .build();
 * // Note: this is equivalent to specifying
 * computeEnv.addInstanceType(InstanceType.of(InstanceClass.M5AD, InstanceSize.LARGE));
 * computeEnv.addInstanceClass(InstanceClass.C4);
 * computeEnv.addInstanceClass(InstanceClass.M4);
 * computeEnv.addInstanceClass(InstanceClass.R4);
 * </pre></blockquote>
 * <p>
 * <h4>Configure AMIs</h4>
 * <p>
 * You can configure Amazon Machine Images (AMIs). This example configures your <code>ComputeEnvironment</code> to use Amazon Linux 2023.
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .images(List.of(EcsMachineImage.builder()
 *                 .imageType(EcsMachineImageType.ECS_AL2023)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h4>Allocation Strategies</h4>
 * <p>
 * | Allocation Strategy           | Optimized for              | Downsides                     |
 * | -----------------------       | -------------              | ----------------------------- |
 * | BEST_FIT                      | Cost                       | May limit throughput          |
 * | BEST_FIT_PROGRESSIVE          | Throughput                 | May increase cost             |
 * | SPOT_CAPACITY_OPTIMIZED       | Least interruption         | Only useful on Spot instances |
 * | SPOT_PRICE_CAPACITY_OPTIMIZED | Least interruption + Price | Only useful on Spot instances |
 * <p>
 * Batch provides different Allocation Strategies to help it choose which instances to provision.
 * If your workflow tolerates interruptions, you should enable <code>spot</code> on your <code>ComputeEnvironment</code>
 * and use <code>SPOT_PRICE_CAPACITY_OPTIMIZED</code> (this is the default if <code>spot</code> is enabled).
 * This will tell Batch to choose the instance types from the ones you’ve specified that have
 * the most spot capacity available to minimize the chance of interruption and have the lowest price.
 * To get the most benefit from your spot instances,
 * you should allow Batch to choose from as many different instance types as possible.
 * If you only care about minimal interruptions and not want Batch to optimize for cost, use
 * <code>SPOT_CAPACITY_OPTIMIZED</code>. <code>SPOT_PRICE_CAPACITY_OPTIMIZED</code> is recommended over <code>SPOT_CAPACITY_OPTIMIZED</code>
 * for most use cases.
 * <p>
 * If your workflow does not tolerate interruptions and you want to minimize your costs at the expense
 * of potentially longer waiting times, use <code>AllocationStrategy.BEST_FIT</code>.
 * This will choose the lowest-cost instance type that fits all the jobs in the queue.
 * If instances of that type are not available,
 * the queue will not choose a new type; instead, it will wait for the instance to become available.
 * This can stall your <code>Queue</code>, with your compute environment only using part of its max capacity
 * (or none at all) until the <code>BEST_FIT</code> instance becomes available.
 * <p>
 * If you are running a workflow that does not tolerate interruptions and you want to maximize throughput,
 * you can use <code>AllocationStrategy.BEST_FIT_PROGRESSIVE</code>.
 * This is the default Allocation Strategy if <code>spot</code> is <code>false</code> or unspecified.
 * This strategy will examine the Jobs in the queue and choose whichever instance type meets the requirements
 * of the jobs in the queue and with the lowest cost per vCPU, just as <code>BEST_FIT</code>.
 * However, if not all of the capacity can be filled with this instance type,
 * it will choose a new next-best instance type to run any jobs that couldn’t fit into the <code>BEST_FIT</code> capacity.
 * To make the most use of this allocation strategy,
 * it is recommended to use as many instance classes as is feasible for your workload.
 * This example shows a <code>ComputeEnvironment</code> that uses <code>BEST_FIT_PROGRESSIVE</code>
 * with <code>'optimal'</code> and <code>InstanceClass.M5</code> instance types:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * ManagedEc2EcsComputeEnvironment computeEnv = ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .instanceClasses(List.of(InstanceClass.M5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * This example shows a <code>ComputeEnvironment</code> that uses <code>BEST_FIT</code> with <code>'optimal'</code> instances:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * ManagedEc2EcsComputeEnvironment computeEnv = ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .allocationStrategy(AllocationStrategy.BEST_FIT)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <em>Note</em>: <code>allocationStrategy</code> cannot be specified on Fargate Compute Environments.
 * <p>
 * <h3>Controlling vCPU allocation</h3>
 * <p>
 * You can specify the maximum and minimum vCPUs a managed <code>ComputeEnvironment</code> can have at any given time.
 * Batch will <em>always</em> maintain <code>minvCpus</code> worth of instances in your ComputeEnvironment, even if it is not executing any jobs,
 * and even if it is disabled. Batch will scale the instances up to <code>maxvCpus</code> worth of instances as
 * jobs exit the JobQueue and enter the ComputeEnvironment. If you use <code>AllocationStrategy.BEST_FIT_PROGRESSIVE</code>,
 * <code>AllocationStrategy.SPOT_PRICE_CAPACITY_OPTIMIZED</code>, or <code>AllocationStrategy.SPOT_CAPACITY_OPTIMIZED</code>,
 * batch may exceed <code>maxvCpus</code>; it will never exceed <code>maxvCpus</code> by more than a single instance type. This example configures a
 * <code>minvCpus</code> of 10 and a <code>maxvCpus</code> of 100:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * ManagedEc2EcsComputeEnvironment.Builder.create(this, "myEc2ComputeEnv")
 *         .vpc(vpc)
 *         .instanceClasses(List.of(InstanceClass.R4))
 *         .minvCpus(10)
 *         .maxvCpus(100)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Tagging Instances</h3>
 * <p>
 * You can tag any instances launched by your managed EC2 ComputeEnvironments by using the CDK <code>Tags</code> API:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * ManagedEc2EcsComputeEnvironment tagCE = ManagedEc2EcsComputeEnvironment.Builder.create(this, "CEThatMakesTaggedInstnaces")
 *         .vpc(vpc)
 *         .build();
 * 
 * Tags.of(tagCE).add("super", "salamander");
 * </pre></blockquote>
 * <p>
 * Unmanaged <code>ComputeEnvironment</code>s do not support <code>maxvCpus</code> or <code>minvCpus</code> because you must provision and manage the instances yourself;
 * that is, Batch will not scale them up and down as needed.
 * <p>
 * <h3>Sharing a ComputeEnvironment between multiple JobQueues</h3>
 * <p>
 * Multiple <code>JobQueue</code>s can share the same <code>ComputeEnvironment</code>.
 * If multiple Queues are attempting to submit Jobs to the same <code>ComputeEnvironment</code>,
 * Batch will pick the Job from the Queue with the highest priority.
 * This example creates two <code>JobQueue</code>s that share a <code>ComputeEnvironment</code>:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * FargateComputeEnvironment sharedComputeEnv = FargateComputeEnvironment.Builder.create(this, "spotEnv")
 *         .vpc(vpc)
 *         .spot(true)
 *         .build();
 * JobQueue lowPriorityQueue = JobQueue.Builder.create(this, "JobQueue")
 *         .priority(1)
 *         .build();
 * JobQueue highPriorityQueue = JobQueue.Builder.create(this, "JobQueue")
 *         .priority(10)
 *         .build();
 * lowPriorityQueue.addComputeEnvironment(sharedComputeEnv, 1);
 * highPriorityQueue.addComputeEnvironment(sharedComputeEnv, 1);
 * </pre></blockquote>
 * <p>
 * <h3>React to jobs stuck in RUNNABLE state</h3>
 * <p>
 * You can react to jobs stuck in RUNNABLE state by setting a <code>jobStateTimeLimitActions</code> in <code>JobQueue</code>.
 * Specifies actions that AWS Batch will take after the job has remained at the head of the queue in the
 * specified state for longer than the specified time.
 * <p>
 * <blockquote><pre>
 * JobQueue.Builder.create(this, "JobQueue")
 *         .jobStateTimeLimitActions(List.of(JobStateTimeLimitAction.builder()
 *                 .action(JobStateTimeLimitActionsAction.CANCEL)
 *                 .maxTime(Duration.minutes(10))
 *                 .reason(JobStateTimeLimitActionsReason.INSUFFICIENT_INSTANCE_CAPACITY)
 *                 .state(JobStateTimeLimitActionsState.RUNNABLE)
 *                 .build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Fairshare Scheduling</h3>
 * <p>
 * Batch <code>JobQueue</code>s execute Jobs submitted to them in FIFO order unless you specify a <code>SchedulingPolicy</code>.
 * FIFO queuing can cause short-running jobs to be starved while long-running jobs fill the compute environment.
 * To solve this, Jobs can be associated with a share.
 * <p>
 * Shares consist of a <code>shareIdentifier</code> and a <code>weightFactor</code>, which is inversely correlated with the vCPU allocated to that share identifier.
 * When submitting a Job, you can specify its <code>shareIdentifier</code> to associate that particular job with that share.
 * Let's see how the scheduler uses this information to schedule jobs.
 * <p>
 * For example, if there are two shares defined as follows:
 * <p>
 * | Share Identifier | Weight Factor |
 * | ---------------- | ------------- |
 * | A                | 1             |
 * | B                | 1             |
 * <p>
 * The weight factors share the following relationship:
 * <p>
 * <blockquote><pre>
 * A_{vCpus} / A_{Weight} = B_{vCpus} / B_{Weight}
 * </pre></blockquote>
 * <p>
 * where <code>BvCpus</code> is the number of vCPUs allocated to jobs with share identifier <code>'B'</code>, and <code>B_weight</code> is the weight factor of <code>B</code>.
 * <p>
 * The total number of vCpus allocated to a share is equal to the amount of jobs in that share times the number of vCpus necessary for every job.
 * Let's say that each A job needs 32 VCpus (<code>A_requirement</code> = 32) and each B job needs 64 vCpus (<code>B_requirement</code> = 64):
 * <p>
 * <blockquote><pre>
 * A_{vCpus} = A_{Jobs} * A_{Requirement}
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * B_{vCpus} = B_{Jobs} * B_{Requirement}
 * </pre></blockquote>
 * <p>
 * We have:
 * <p>
 * <blockquote><pre>
 * A_{vCpus} / A_{Weight} = B_{vCpus} / B_{Weight}
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * A_{Jobs} * A_{Requirement} / A_{Weight} = B_{Jobs} * B_{Requirement} / B_{Weight}
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * A_{Jobs} * 32 / 1 = B_{Jobs} * 64 / 1
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * A_{Jobs} * 32 = B_{Jobs} * 64
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * A_{Jobs} = B_{Jobs} * 2
 * </pre></blockquote>
 * <p>
 * Thus the scheduler will schedule two <code>'A'</code> jobs for each <code>'B'</code> job.
 * <p>
 * You can control the weight factors to change these ratios, but note that
 * weight factors are inversely correlated with the vCpus allocated to the corresponding share.
 * <p>
 * This example would be configured like this:
 * <p>
 * <blockquote><pre>
 * FairshareSchedulingPolicy fairsharePolicy = new FairshareSchedulingPolicy(this, "myFairsharePolicy");
 * 
 * fairsharePolicy.addShare(Share.builder()
 *         .shareIdentifier("A")
 *         .weightFactor(1)
 *         .build());
 * fairsharePolicy.addShare(Share.builder()
 *         .shareIdentifier("B")
 *         .weightFactor(1)
 *         .build());
 * JobQueue.Builder.create(this, "JobQueue")
 *         .schedulingPolicy(fairsharePolicy)
 *         .build();
 * </pre></blockquote>
 * <p>
 * <em>Note</em>: The scheduler will only consider the current usage of the compute environment unless you specify <code>shareDecay</code>.
 * For example, a <code>shareDecay</code> of 5 minutes in the above example means that at any given point in time, twice as many <code>'A'</code> jobs
 * will be scheduled for each <code>'B'</code> job, but only for the past 5 minutes. If <code>'B'</code> jobs run longer than 5 minutes, then
 * the scheduler is allowed to put more than two <code>'A'</code> jobs for each <code>'B'</code> job, because the usage of those long-running
 * <code>'B'</code> jobs will no longer be considered after 5 minutes. <code>shareDecay</code> linearly decreases the usage of
 * long running jobs for calculation purposes. For example if share decay is 60 seconds,
 * then jobs that run for 30 seconds have their usage considered to be only 50% of what it actually is,
 * but after a whole minute the scheduler pretends they don't exist for fairness calculations.
 * <p>
 * The following code specifies a <code>shareDecay</code> of 5 minutes:
 * <p>
 * <blockquote><pre>
 * FairshareSchedulingPolicy fairsharePolicy = FairshareSchedulingPolicy.Builder.create(this, "myFairsharePolicy")
 *         .shareDecay(Duration.minutes(5))
 *         .build();
 * </pre></blockquote>
 * <p>
 * If you have high priority jobs that should always be executed as soon as they arrive,
 * you can define a <code>computeReservation</code> to specify the percentage of the
 * maximum vCPU capacity that should be reserved for shares that are <em>not in the queue</em>.
 * The actual reserved percentage is defined by Batch as:
 * <p>
 * <blockquote><pre>
 *  (\frac{computeReservation}{100}) ^ {ActiveFairShares}
 * </pre></blockquote>
 * <p>
 * where <code>ActiveFairShares</code> is the number of shares for which there exists
 * at least one job in the queue with a unique share identifier.
 * <p>
 * This is best illustrated with an example.
 * Suppose there are three shares with share identifiers <code>A</code>, <code>B</code> and <code>C</code> respectively
 * and we specify the <code>computeReservation</code> to be 75%. The queue is currently empty,
 * and no other shares exist.
 * <p>
 * There are no active fair shares, since the queue is empty.
 * Thus (75/100)^0 = 1 = 100% of the maximum vCpus are reserved for all shares.
 * <p>
 * A job with identifier <code>A</code> enters the queue.
 * <p>
 * The number of active fair shares is now 1, hence
 * (75/100)^1 = .75 = 75% of the maximum vCpus are reserved for all shares that do not have the identifier <code>A</code>;
 * for this example, this is <code>B</code> and <code>C</code>,
 * (but if jobs are submitted with a share identifier not covered by this fairshare policy, those would be considered just as <code>B</code> and <code>C</code> are).
 * <p>
 * Now a <code>B</code> job enters the queue. The number of active fair shares is now 2,
 * so (75/100)^2 = .5625 = 56.25% of the maximum vCpus are reserved for all shares that do not have the identifier <code>A</code> or <code>B</code>.
 * <p>
 * Now a second <code>A</code> job enters the queue. The number of active fair shares is still 2,
 * so the percentage reserved is still 56.25%
 * <p>
 * Now a <code>C</code> job enters the queue. The number of active fair shares is now 3,
 * so (75/100)^3 = .421875 = 42.1875% of the maximum vCpus are reserved for all shares that do not have the identifier <code>A</code>, <code>B</code>, or <code>C</code>.
 * <p>
 * If there are no other shares that your jobs can specify, this means that 42.1875% of your capacity will never be used!
 * <p>
 * Now, <code>A</code>, <code>B</code>, and <code>C</code> can only consume 100% - 42.1875% = 57.8125% of the maximum vCpus.
 * Note that the this percentage is <strong>not</strong> split between <code>A</code>, <code>B</code>, and <code>C</code>.
 * Instead, the scheduler will use their <code>weightFactor</code>s to decide which jobs to schedule;
 * the only difference is that instead of competing for 100% of the max capacity, jobs compete for 57.8125% of the max capacity.
 * <p>
 * This example specifies a <code>computeReservation</code> of 75% that will behave as explained in the example above:
 * <p>
 * <blockquote><pre>
 * FairshareSchedulingPolicy.Builder.create(this, "myFairsharePolicy")
 *         .computeReservation(75)
 *         .shares(List.of(Share.builder().weightFactor(1).shareIdentifier("A").build(), Share.builder().weightFactor(0.5).shareIdentifier("B").build(), Share.builder().weightFactor(2).shareIdentifier("C").build()))
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can specify a <code>priority</code> on your <code>JobDefinition</code>s to tell the scheduler to prioritize certain jobs that share the same share identifier.
 * <p>
 * <h3>Configuring Job Retry Policies</h3>
 * <p>
 * Certain workflows may result in Jobs failing due to intermittent issues.
 * Jobs can specify retry policies to respond to different failures with different actions.
 * There are three different ways information about the way a Job exited can be conveyed;
 * <p>
 * <ul>
 * <li><code>exitCode</code>: the exit code returned from the process executed by the container. Will only match non-zero exit codes.</li>
 * <li><code>reason</code>: any middleware errors, like your Docker registry being down.</li>
 * <li><code>statusReason</code>: infrastructure errors, most commonly your spot instance being reclaimed.</li>
 * </ul>
 * <p>
 * For most use cases, only one of these will be associated with a particular action at a time.
 * To specify common <code>exitCode</code>s, <code>reason</code>s, or <code>statusReason</code>s, use the corresponding value from
 * the <code>Reason</code> class. This example shows some common failure reasons:
 * <p>
 * <blockquote><pre>
 * EcsJobDefinition jobDefn = EcsJobDefinition.Builder.create(this, "JobDefn")
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "containerDefn")
 *                 .image(ContainerImage.fromRegistry("public.ecr.aws/amazonlinux/amazonlinux:latest"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .build())
 *         .retryAttempts(5)
 *         .retryStrategies(List.of(RetryStrategy.of(Action.EXIT, Reason.CANNOT_PULL_CONTAINER)))
 *         .build();
 * jobDefn.addRetryStrategy(RetryStrategy.of(Action.EXIT, Reason.SPOT_INSTANCE_RECLAIMED));
 * jobDefn.addRetryStrategy(RetryStrategy.of(Action.EXIT, Reason.CANNOT_PULL_CONTAINER));
 * jobDefn.addRetryStrategy(RetryStrategy.of(Action.EXIT, Reason.custom(CustomReason.builder()
 *         .onExitCode("40*")
 *         .onReason("some reason")
 *         .build())));
 * </pre></blockquote>
 * <p>
 * When specifying a custom reason,
 * you can specify a glob string to match each of these and react to different failures accordingly.
 * Up to five different retry strategies can be configured for each Job,
 * and each strategy can match against some or all of <code>exitCode</code>, <code>reason</code>, and <code>statusReason</code>.
 * You can optionally configure the number of times a job will be retried,
 * but you cannot configure different retry counts for different strategies; they all share the same count.
 * If multiple conditions are specified in a given retry strategy,
 * they must all match for the action to be taken; the conditions are ANDed together, not ORed.
 * <p>
 * <h3>Running single-container ECS workflows</h3>
 * <p>
 * Batch can run jobs on ECS or EKS. ECS jobs can be defined as single container or multinode.
 * This example creates a <code>JobDefinition</code> that runs a single container with ECS:
 * <p>
 * <blockquote><pre>
 * IFileSystem myFileSystem;
 * Role myJobRole;
 * 
 * myFileSystem.grantRead(myJobRole);
 * 
 * EcsJobDefinition jobDefn = EcsJobDefinition.Builder.create(this, "JobDefn")
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "containerDefn")
 *                 .image(ContainerImage.fromRegistry("public.ecr.aws/amazonlinux/amazonlinux:latest"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .volumes(List.of(EcsVolume.efs(EfsVolumeOptions.builder()
 *                         .name("myVolume")
 *                         .fileSystem(myFileSystem)
 *                         .containerPath("/Volumes/myVolume")
 *                         .useJobRole(true)
 *                         .build())))
 *                 .jobRole(myJobRole)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * For workflows that need persistent storage, batch supports mounting <code>Volume</code>s to the container.
 * You can both provision the volume and mount it to the container in a single operation:
 * <p>
 * <blockquote><pre>
 * IFileSystem myFileSystem;
 * EcsJobDefinition jobDefn;
 * 
 * 
 * jobDefn.container.addVolume(EcsVolume.efs(EfsVolumeOptions.builder()
 *         .name("myVolume")
 *         .fileSystem(myFileSystem)
 *         .containerPath("/Volumes/myVolume")
 *         .build()));
 * </pre></blockquote>
 * <p>
 * <h3>Running an ECS workflow with Fargate container</h3>
 * <p>
 * <blockquote><pre>
 * EcsJobDefinition jobDefn = EcsJobDefinition.Builder.create(this, "JobDefn")
 *         .container(EcsFargateContainerDefinition.Builder.create(this, "myFargateContainer")
 *                 .image(ContainerImage.fromRegistry("public.ecr.aws/amazonlinux/amazonlinux:latest"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .ephemeralStorageSize(Size.gibibytes(100))
 *                 .fargateCpuArchitecture(CpuArchitecture.ARM64)
 *                 .fargateOperatingSystemFamily(OperatingSystemFamily.LINUX)
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Secrets</h3>
 * <p>
 * You can expose SecretsManager Secret ARNs or SSM Parameters to your container as environment variables.
 * The following example defines the <code>MY_SECRET_ENV_VAR</code> environment variable that contains the
 * ARN of the Secret defined by <code>mySecret</code>:
 * <p>
 * <blockquote><pre>
 * ISecret mySecret;
 * 
 * 
 * EcsJobDefinition jobDefn = EcsJobDefinition.Builder.create(this, "JobDefn")
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "containerDefn")
 *                 .image(ContainerImage.fromRegistry("public.ecr.aws/amazonlinux/amazonlinux:latest"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .secrets(Map.of(
 *                         "MY_SECRET_ENV_VAR", Secret.fromSecretsManager(mySecret)))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Running Kubernetes Workflows</h3>
 * <p>
 * Batch also supports running workflows on EKS. The following example creates a <code>JobDefinition</code> that runs on EKS:
 * <p>
 * <blockquote><pre>
 * EksJobDefinition jobDefn = EksJobDefinition.Builder.create(this, "eksf2")
 *         .container(EksContainerDefinition.Builder.create(this, "container")
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .volumes(List.of(EksVolume.emptyDir(EmptyDirVolumeOptions.builder()
 *                         .name("myEmptyDirVolume")
 *                         .mountPath("/mount/path")
 *                         .medium(EmptyDirMediumType.MEMORY)
 *                         .readonly(true)
 *                         .sizeLimit(Size.mebibytes(2048))
 *                         .build())))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * You can mount <code>Volume</code>s to these containers in a single operation:
 * <p>
 * <blockquote><pre>
 * EksJobDefinition jobDefn;
 * 
 * jobDefn.container.addVolume(EksVolume.emptyDir(EmptyDirVolumeOptions.builder()
 *         .name("emptyDir")
 *         .mountPath("/Volumes/emptyDir")
 *         .build()));
 * jobDefn.container.addVolume(EksVolume.hostPath(HostPathVolumeOptions.builder()
 *         .name("hostPath")
 *         .hostPath("/sys")
 *         .mountPath("/Volumes/hostPath")
 *         .build()));
 * jobDefn.container.addVolume(EksVolume.secret(SecretPathVolumeOptions.builder()
 *         .name("secret")
 *         .optional(true)
 *         .mountPath("/Volumes/secret")
 *         .secretName("mySecret")
 *         .build()));
 * </pre></blockquote>
 * <p>
 * <h3>Running Distributed Workflows</h3>
 * <p>
 * Some workflows benefit from parallellization and are most powerful when run in a distributed environment,
 * such as certain numerical calculations or simulations. Batch offers <code>MultiNodeJobDefinition</code>s,
 * which allow a single job to run on multiple instances in parallel, for this purpose.
 * Message Passing Interface (MPI) is often used with these workflows.
 * You must configure your containers to use MPI properly,
 * but Batch allows different nodes running different containers to communicate easily with one another.
 * You must configure your containers to use certain environment variables that Batch will provide them,
 * which lets them know which one is the main node, among other information.
 * For an in-depth example on using MPI to perform numerical computations on Batch,
 * see this <a href="https://aws.amazon.com/blogs/compute/building-a-tightly-coupled-molecular-dynamics-workflow-with-multi-node-parallel-jobs-in-aws-batch/">blog post</a>
 * In particular, the environment variable that tells the containers which one is the main node can be configured on your <code>MultiNodeJobDefinition</code> as follows:
 * <p>
 * <blockquote><pre>
 * MultiNodeJobDefinition multiNodeJob = MultiNodeJobDefinition.Builder.create(this, "JobDefinition")
 *         .instanceType(InstanceType.of(InstanceClass.R4, InstanceSize.LARGE)) // optional, omit to let Batch choose the type for you
 *         .containers(List.of(MultiNodeContainer.builder()
 *                 .container(EcsEc2ContainerDefinition.Builder.create(this, "mainMPIContainer")
 *                         .image(ContainerImage.fromRegistry("yourregsitry.com/yourMPIImage:latest"))
 *                         .cpu(256)
 *                         .memory(Size.mebibytes(2048))
 *                         .build())
 *                 .startNode(0)
 *                 .endNode(5)
 *                 .build()))
 *         .build();
 * // convenience method
 * multiNodeJob.addContainer(MultiNodeContainer.builder()
 *         .startNode(6)
 *         .endNode(10)
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "multiContainer")
 *                 .image(ContainerImage.fromRegistry("amazon/amazon-ecs-sample"))
 *                 .cpu(256)
 *                 .memory(Size.mebibytes(2048))
 *                 .build())
 *         .build());
 * </pre></blockquote>
 * <p>
 * If you need to set the control node to an index other than 0, specify it in directly:
 * <p>
 * <blockquote><pre>
 * MultiNodeJobDefinition multiNodeJob = MultiNodeJobDefinition.Builder.create(this, "JobDefinition")
 *         .mainNode(5)
 *         .instanceType(InstanceType.of(InstanceClass.R4, InstanceSize.LARGE))
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Pass Parameters to a Job</h3>
 * <p>
 * Batch allows you define parameters in your <code>JobDefinition</code>, which can be referenced in the container command. For example:
 * <p>
 * <blockquote><pre>
 * EcsJobDefinition.Builder.create(this, "JobDefn")
 *         .parameters(Map.of("echoParam", "foobar"))
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "containerDefn")
 *                 .image(ContainerImage.fromRegistry("public.ecr.aws/amazonlinux/amazonlinux:latest"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .command(List.of("echo", "Ref::echoParam"))
 *                 .build())
 *         .build();
 * </pre></blockquote>
 * <p>
 * <h3>Understanding Progressive Allocation Strategies</h3>
 * <p>
 * AWS Batch uses an <a href="https://docs.aws.amazon.com/batch/latest/userguide/allocation-strategies.html">allocation strategy</a> to determine what compute resource will efficiently handle incoming job requests. By default, <strong>BEST_FIT</strong> will pick an available compute instance based on vCPU requirements. If none exist, the job will wait until resources become available. However, with this strategy, you may have jobs waiting in the queue unnecessarily despite having more powerful instances available. Below is an example of how that situation might look like:
 * <p>
 * <blockquote><pre>
 * Compute Environment:
 * 
 * 1. m5.xlarge =&gt; 4 vCPU
 * 2. m5.2xlarge =&gt; 8 vCPU
 * </pre></blockquote>
 * <p>
 * <blockquote><pre>
 * Job Queue:
 * ---------
 * | A | B |
 * ---------
 * 
 * Job Requirements:
 * A =&gt; 4 vCPU - ALLOCATED TO m5.xlarge
 * B =&gt; 2 vCPU - WAITING
 * </pre></blockquote>
 * <p>
 * In this situation, Batch will allocate <strong>Job A</strong> to compute resource #1 because it is the most cost efficient resource that matches the vCPU requirement. However, with this <code>BEST_FIT</code> strategy, <strong>Job B</strong> will not be allocated to our other available compute resource even though it is strong enough to handle it. Instead, it will wait until the first job is finished processing or wait a similar <code>m5.xlarge</code> resource to be provisioned.
 * <p>
 * The alternative would be to use the <code>BEST_FIT_PROGRESSIVE</code> strategy in order for the remaining job to be handled in larger containers regardless of vCPU requirement and costs.
 * <p>
 * <h3>Permissions</h3>
 * <p>
 * You can grant any Principal the <code>batch:submitJob</code> permission on both a job definition and a job queue like this:
 * <p>
 * <blockquote><pre>
 * IVpc vpc;
 * 
 * 
 * EcsJobDefinition ecsJob = EcsJobDefinition.Builder.create(this, "JobDefn")
 *         .container(EcsEc2ContainerDefinition.Builder.create(this, "containerDefn")
 *                 .image(ContainerImage.fromRegistry("public.ecr.aws/amazonlinux/amazonlinux:latest"))
 *                 .memory(Size.mebibytes(2048))
 *                 .cpu(256)
 *                 .build())
 *         .build();
 * 
 * JobQueue queue = JobQueue.Builder.create(this, "JobQueue")
 *         .computeEnvironments(List.of(OrderedComputeEnvironment.builder()
 *                 .computeEnvironment(ManagedEc2EcsComputeEnvironment.Builder.create(this, "managedEc2CE")
 *                         .vpc(vpc)
 *                         .build())
 *                 .order(1)
 *                 .build()))
 *         .priority(10)
 *         .build();
 * 
 * User user = new User(this, "MyUser");
 * ecsJob.grantSubmitJob(user, queue);
 * </pre></blockquote>
 */
package software.amazon.awscdk.services.batch;
