package pro.siper.adept.core

import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.recyclerview.widget.DiffUtil
import androidx.recyclerview.widget.RecyclerView
import pro.siper.adept.core.diff.DefaultDiffUtilCallback
import pro.siper.adept.core.diff.DiffUtilCallback
import pro.siper.adept.core.diff.DiffUtilCallbackCreator
import java.lang.IllegalArgumentException
import kotlin.reflect.KClass

class Adept : RecyclerView.Adapter<AdeptViewHolder>() {
    val renderers: MutableMap<KClass<*>, ItemViewRenderer<*>> = mutableMapOf()
    val layouts: MutableMap<KClass<*>, Int> = mutableMapOf()

    private val dataset: MutableList<Any?> = mutableListOf()
    private val viewTypes: MutableList<KClass<out Any>> = mutableListOf()
    private var useDiffUtil = false
    private var diffUtilCallback: DiffUtilCallback = DefaultDiffUtilCallback

    override fun getItemViewType(position: Int): Int {
        val type = dataset[position]!!::class
        if (viewTypes.indexOf(type) == -1) {
            viewTypes.add(type)
        }
        return viewTypes.indexOf(type)
    }

    override fun onCreateViewHolder(parent: ViewGroup, viewType: Int): AdeptViewHolder {
        val layout = layouts[viewTypes[viewType]]
                ?: throw IllegalArgumentException("Layout for this data type is not registered")
        val view = LayoutInflater
                .from(parent.context)
                .inflate(layout, parent, false)
        return AdeptViewHolder(view, viewType)
    }

    override fun getItemCount(): Int = dataset.size

    override fun onBindViewHolder(holder: AdeptViewHolder, position: Int) {
        val renderer = renderers[viewTypes[holder.viewType]]
                ?: throw IllegalArgumentException("There is no renderer for this data type")
        renderer.anyTypeRender(dataset[position], ViewBinder(holder))
    }

    fun updateDataset(dataset: List<Any?>) {
        if (useDiffUtil) {
            val diff = DiffUtilCallbackCreator(
                    this.dataset,
                    dataset,
                    diffUtilCallback
            )
            val diffResult = DiffUtil.calculateDiff(diff)
            this.dataset.clear()
            this.dataset.addAll(dataset)
            diffResult.dispatchUpdatesTo(this)
        } else {
            this.dataset.clear()
            this.dataset.addAll(dataset)
        }
    }

    inline fun <reified T> addRenderer(layoutId: Int, renderer: ItemViewRenderer<T>): Adept {
        if (layouts.containsKey(T::class)) {
            throw IllegalArgumentException("Renderer for this data type already registered")
        }
        layouts[T::class] = layoutId
        renderers[T::class] = renderer
        return this
    }

    inline fun <reified T> addRenderer(
            layoutId: Int,
            crossinline renderer: (data: T, viewBinder: ViewBinder) -> Unit): Adept {
        addRenderer(layoutId, object : ItemViewRenderer<T>() {
            override fun render(data: T, viewBinder: ViewBinder) {
                renderer.invoke(data, viewBinder)
            }
        })
        return this
    }

    fun attachTo(recyclerView: RecyclerView): Adept {
        recyclerView.adapter = this
        return this
    }

    fun useDiffUtil(diffUtilCallback: DiffUtilCallback = DefaultDiffUtilCallback): Adept {
        this.diffUtilCallback = diffUtilCallback
        this.useDiffUtil = true
        return this
    }
}