package pl.droidsonroids.gradle.jacoco.testkit

import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.provider.Provider
import org.gradle.api.tasks.TaskProvider
import org.gradle.api.tasks.WriteProperties
import org.gradle.testing.jacoco.plugins.JacocoTaskExtension
import org.gradle.util.GradleVersion

open class JacocoTestKitExtension(private val project: Project) {

    val jacocoRuntimePathProvider: Provider<String> =
        project.provider { project.configurations.getByName(Configurations.jacocoRuntime).asPath }

    fun applyTo(configurationRuntime: String, taskProvider: TaskProvider<Task>) {
        with(project) {
            val jacocoTestKitPropertiesTask = tasks.register(
                generatePropertiesTaskName(taskProvider.name),
                WriteProperties::class.java,
            ) {
                it.group = "verification"
                it.description = "Generates gradle.properties with JaCoCo java agent for TestKit"
                it.comment = "Generated by pl.droidsonroids.jacoco.testkit"

                val testKitPropertiesFile =
                    layout.buildDirectory.file("testkit/${taskProvider.name}/testkit-gradle.properties")
                if (GradleVersion.current() >= GradleVersion.version("8.1")) {
                    it.destinationFile.set(testKitPropertiesFile)
                } else {
                    @Suppress("DEPRECATION")
                    it.outputFile = testKitPropertiesFile.get().asFile
                }

                it.property(
                    "org.gradle.jvmargs",
                    taskProvider.map { task -> task.extensions.getByType(JacocoTaskExtension::class.java).destinationFile!! }
                        .map { destFile -> "\"-javaagent:${jacocoRuntimePathProvider.get()}=destfile=${destFile}\"" },
                )
            }

            dependencies.add(configurationRuntime, files(testKitDir(taskProvider.name)))
            taskProvider.configure { it.dependsOn(jacocoTestKitPropertiesTask) }
        }
    }

    fun applyTo(configurationRuntime: String, task: Task) =
        applyTo(configurationRuntime, project.provider { task } as TaskProvider<Task>)
}
