package pl.decerto.hyperon.persistence.sync.diff;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.smartparam.engine.util.Printer;

import pl.decerto.hyperon.persistence.model.def.EntityType;
import pl.decerto.hyperon.persistence.model.def.PropertyDef;
import pl.decerto.hyperon.persistence.model.value.EntityProperty;
import pl.decerto.hyperon.persistence.model.value.ValueProperty;
import pl.decerto.hyperon.persistence.sync.BundleSynchronizer;
import pl.decerto.hyperon.persistence.sync.SyncActions;

/**
 * @author przemek hertel
 */
public class BundleDiff {

	private static BundleSynchronizer synchronizer = new BundleSynchronizer();

	private final List<EntityProperty> toInsert;
	private final List<EntityProperty> toDelete;
	private final List<EntityDiff> toUpdate;

	public BundleDiff(SyncActions sync) {
		toInsert = new ArrayList<>(sync.getToInsert());
		toDelete = new ArrayList<>(sync.getToDelete());
		toUpdate = toEntityDiff(sync.getToUpdatePrev(), sync.getToUpdate());
	}

	private List<EntityDiff> toEntityDiff(List<EntityProperty> prevList, List<EntityProperty> nextList) {

		final int size = prevList.size();
		List<EntityDiff> diffList = new ArrayList<>(size);

		for (int i = 0; i < size; i++) {
			EntityProperty prev = prevList.get(i);
			EntityProperty next = nextList.get(i);

			// create entity diff result
			diffList.add(diff(prev, next));
		}

		return diffList;
	}

	private static EntityDiff diff(EntityProperty prev, EntityProperty next) {

		// entity definition
		EntityType def = prev.getType().getCompoundType();

		// comparision result
		EntityDiff result = new EntityDiff(def, prev.getId());

		// compare all defined value properties
		for (Map.Entry<String, PropertyDef> e : def.getProps().entrySet()) {
			String propName = e.getKey();
			PropertyDef propDef = e.getValue();

			if (propDef.isSimpleType()) {
				ValueProperty v1 = prev.getValue(propName);
				ValueProperty v2 = next.getValue(propName);

				if (synchronizer.diff(v1, v2)) {
					result.addValueDiff(propDef, propName, v1, v2);
				}
			}
		}

		return result;
	}

	public List<EntityProperty> getEntitiesToInsert() {
		return toInsert;
	}

	public List<EntityProperty> getEntitiesToDelete() {
		return toDelete;
	}

	public List<EntityDiff> getEntitiesToUpdate() {
		return toUpdate;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(64);
		sb.append("BundleDiff[");
		sb.append("insert:").append(toInsert.size());
		sb.append(", update:").append(toUpdate.size());
		sb.append(", delete:").append(toDelete.size());

		printInsert(sb, toInsert, "toInsert");
		printUpdate(sb, toUpdate, "toUpdate");
		printInsert(sb, toDelete, "toDelete");

		sb.append(']');
		return sb.toString();
	}

	private static void printUpdate(StringBuilder sb, List<EntityDiff> list, String title) {
		if (!list.isEmpty()) {
			sb.append(Printer.print(list, title));
		}
	}

	private static void printInsert(StringBuilder sb, List<EntityProperty> list, String name) {
		if (!list.isEmpty()) {
			sb.append(Printer.print(list, name, 100, o -> ((EntityProperty) o).printValues()));
		}
	}

}
