package pl.decerto.hyperon.persistence.model.value;

import java.util.Date;
import java.util.Map;
import java.util.Set;

import pl.decerto.hyperon.persistence.exception.HyperonPersistenceException;
import pl.decerto.hyperon.persistence.helper.BundleWalker;
import pl.decerto.hyperon.persistence.helper.IdentitySet;
import pl.decerto.hyperon.persistence.helper.PropertyVisitor;
import pl.decerto.hyperon.persistence.model.def.BundleDef;
import pl.decerto.hyperon.persistence.model.def.EntityType;

/**
 * This is main class, that developer can interact with using hyperon persistence engine. In general, this class should store all data,
 * that should be managed by persistence engine. Bundle is a root for Hyperon context. What's really important is that data, should match
 * {@link BundleDef} definition.
 * Bundle stores basic information like:
 * <ul>
 *	   <li>def - bundle definition, which holds type structure of created bundle</li>
 *     <li>revision - incremental number of updated bundle. By default, it starts with 0 on init, but on first persist it will store 1</li>
 *     <li>created - creation time, only setup on first persistence call</li>
 *	   <li>updated - update time, on first persistence call it is null, but for next revisions this field will be updated</li>
 *	   <li>lobHash - last hash code of serialized JSON</li>
 * </ul>
 * <br>
 * Example of bundle creation and setup some properties:
 * <pre>{@code
 * 	Bundle bundle = new Bundle(def); // def must match properties that are setup later
 *
 * 	Property steph = bundle.create("Client").set("age", 31).set("name", "steph");
 * 	Property dean = bundle.create("Client").set("age", 32).set("name", "dean");
 * 	Property alex = bundle.create("Client").set("age", 33).set("name", "alex");
 *
 * 	bundle.get("agent").set(steph);
 * 	bundle.get("policy.insurer").set(dean);
 * 	bundle.get("policy.insured").set(alex);
 * }</pre>
 * <p>
 * Bundle can be used with Hyperon context mechanism. The best way to do that is to work with {@link pl.decerto.hyperon.persistence.context.AppCtx}.
 * </p>
 *
 * @author przemek hertel
 * @see BundleDef
 * @see pl.decerto.hyperon.persistence.context.AppCtx
 */
public class Bundle extends EntityProperty {

	/**
	 * Bundle definition
	 */
	private final BundleDef def;

	/**
	 * Extra identity set for nested properties - for performance improvement.
	 */
	private final IdentitySet identitySet;

	/**
	 * bundle revision. On init, by default it is 0 and it is incremented during update of this bundle.
	 */
	private int revision;

	/**
	 * creation date of bundle
	 */
	private Date created;

	/**
	 * last update date on bundle
	 */
	private Date updated;

	/**
	 * Last LOB's hash.
	 */
	private int lobHash;

	/**
	 * Basic constructor to use with definition. Id of created bundle will be 0.
	 * This one should be used for bundle creation.
	 *
	 * @param def bundle definition
	 */
	public Bundle(BundleDef def) {
		super(def);
		this.def = def;
		this.bundle = this;
		this.state = EntityState.PERSISTENT;
		this.identitySet = new IdentitySet();
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 * Basic constructor with definition and id.
	 *
	 * @param def bundle definition
	 * @param id bundle id
	 */
	public Bundle(BundleDef def, long id) {
		this(def, id, 0, 64);
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 * Basic constructor with definition, id and revision.
	 *
	 * @param def bundle definition
	 * @param id bundle id
	 * @param revision bundle revision starting number
	 */
	public Bundle(BundleDef def, long id, int revision) {
		this(def, id, revision, 64);
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 * Advanced constructor with definition, id, revision and expected size of identity set(storage for nested properties).
	 *
	 * @param def bundle definition
	 * @param id bundle id
	 * @param revision bundle revision starting number
	 * @param expectedSize size of created identity set
	 */
	public Bundle(BundleDef def, long id, int revision, int expectedSize) {
		super(id, def);
		this.def = def;
		this.bundle = this;
		this.state = EntityState.PERSISTENT;
		this.revision = revision;
		this.identitySet = new IdentitySet(expectedSize);
	}

	/**
	 * Adds given {@code property} under provided {@code name}.
	 * It is another way of setting properties within bundle.
	 *
	 * @param name unique name
	 * @param prop property to store under name
	 * @return same bundle instance
	 * @see #set(Object)
	 * @see #set(String, Object)
	 */
	@Override
	public Bundle add(String name, Property prop) {
		super.add(name, prop);
		return this;
	}

	/**
	 * Since bundle is always a root, it will return {@code true}.
	 *
	 * @return {@code true}
	 */
	@Override
	public boolean isRoot() {
		return true;
	}

	/**
	 * Get bundle's revision.
	 *
	 * @return bundle's revision
	 */
	public int getRevision() {
		return revision;
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 *
	 * @param revision revision number
	 */
	public void setRevision(int revision) {
		this.revision = revision;
	}

	/**
	 * Get bundle's creation time.
	 *
	 * @return bundle's creation time
	 */
	public Date getCreated() {
		return created;
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 *
	 * @param created creation date
	 */
	public void setCreated(Date created) {
		this.created = created;
	}

	/**
	 * Get bundle's last update time
	 *
	 * @return bundle's last update time
	 */
	public Date getUpdated() {
		return updated;
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 *
	 * @param updated update date
	 */
	public void setUpdated(Date updated) {
		this.updated = updated;
	}

	/**
	 * Shouldn't be used outside of Hyperon persistence engine.
	 *
	 * @param lobHash lob hash
	 */
	public void setLobHash(int lobHash) {
		this.lobHash = lobHash;
	}

	/**
	 * Get last lob hash.
	 *
	 * @return last lob hash
	 */
	public int getLobHash() {
		return lobHash;
	}

	/**
	 * Get bundle's definition.
	 *
	 * @return bundle's definition
	 */
	public BundleDef getDef() {
		return def;
	}

	/**
	 * Checks if bundle was not saved yet.
	 *
	 * @return true if id == 0, false otherwise
	 */
	public boolean isNotSaved() {
		return id == 0;
	}

	/**
	 * Checks if bundle was saved already.
	 *
	 * @return true if id != 0, false otherwise
	 */
	public boolean isSaved() {
		return id != 0;
	}

	/**
	 * Create instance of property, that matches given {@code type} name from associated bundle definition.
	 *
	 * @param type from bundle definition
	 * @return created new EntityProperty instance
	 * @throws HyperonPersistenceException if type is not defined within bundle definition
	 */
	public Property create(String type) {
		EntityType entityType = def.findType(type);

		if (entityType != null) {
			return new EntityProperty(entityType);
		}

		throw new HyperonPersistenceException("unknown type: " + type);
	}

	/**
	 * Returns identity set with nested properties. It is exposed for {@link IdentityScanner} usage.
	 * This method shouldn't be used to access elements of bundle and especially modify it's content.
	 *
	 * @return identity set of properties
	 * @see IdentityScanner
	 */
	public IdentitySet identitySet() {
		return identitySet;
	}

	/**
	 * Checks if given {@code prop} property exists within bundle.
	 *
	 * @param prop property to check
	 * @return true if exists, false otherwise
	 */
	public boolean contains(Property prop) {
		return identitySet.contains(prop);
	}

	/**
	 * Get all properties within bundle.
	 *
	 * @return set of properties
	 */
	public Set<Property> getAll() {
		return identitySet.getAll();
	}

	/**
	 * Creates new bundle instance with copied:
	 * <ul>
	 *     <li>all bundle data</li>
	 *     <li>internal fields</li>
	 *     <li>references</li>
	 * </ul>
	 * It applies to all nested properties within bundle root.
	 * <br>
	 * Extra flag is responsible for resetting ids, if set to {@code true}.
	 *
	 * @param resetIds control flag, if true then bundle and all elements will have id's set to 0, if false, id will be copied
	 *
	 * @return new instance of bundle with copied properties
	 */
	@Override
	public Bundle deepcopy(boolean resetIds) {
		Bundle copy = new Bundle(def, resetIds ? 0 : id, revision, identitySet.size());
		copy.setCreated(copy(created));
		copy.setUpdated(copy(updated));
		copy.setLobHash(lobHash);
		copy.setState(state);

		// copy all nodes without refs
		deepcopyInternalFields(copy, resetIds);

		// reconstruct refs on copy
		copyRefs(copy);

		return copy;
	}

	/**
	 * Same as {@link #deepcopy(boolean)}, but with flag set to {@code true}.
	 *
	 * @return new instance of bundle with copied properties and reset ids
	 */
	@Override
	public Bundle deepcopy() {
		return deepcopy(true);
	}

	private Date copy(Date d) {
		return d != null ? new Date(d.getTime()) : null;
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof Bundle)) {
			return false;
		}
		if (!super.equals(o)) {
			return false;
		}

		Bundle other = (Bundle) o;
		return id == other.id && revision == other.revision;
	}

	@Override
	public int hashCode() {
		int result = super.hashCode();
		result = 31 * result + (def != null ? def.hashCode() : 0);
		result = 31 * result + revision;
		result = 31 * result + (created != null ? created.hashCode() : 0);
		return result;
	}

	/**
	 * This method allows to walk on each property using provided {@code visitor} and do some action.
	 *
	 * @param visitor property visitor implementation
	 *
	 * @see PropertyVisitor#visit(Property, ElementType)
	 */
	public void walk(PropertyVisitor visitor) {
		BundleWalker.walk(this, visitor);
	}

	/**
	 * @deprecated this method shouldn't be used anymore. Please use {@link #walk(PropertyVisitor)}.
	 *
	 * @param visitor property visitor implementation
	 * @param usePathsInVisitor control flag, if set to true it will trigger new implementation, false the old one.
	 */
	@Deprecated
	public void walk(PropertyVisitor visitor, boolean usePathsInVisitor) {
		BundleWalker.walk(this, visitor, usePathsInVisitor);
	}

	/**
	 * This method is not supported.
	 *
	 * @return nothing
	 * @throws UnsupportedOperationException this is not supported on root bundle level
	 */
	@Override
	public Property remove() {
		throw new UnsupportedOperationException("Bundle (root) cannot be removed");
	}

	/**
	 * This will print bundle with specific format.<br>
	 * Example:
	 * <pre>{@code
	 *   bundle D: Bundle[id=0, revision=0]
	 *   * agent  (Client#0  @60861)
	 *     - pesel = null  (string)
	 *     * age = 11  (integer)
	 *   * policy  (Policy#0  @12577)
	 *     * building  (Building#0  @23246)
	 *       * area = 99  (integer)
	 *     * clients  collection(Client)  size=1
	 *       * clients[0]  (Client#0  @31764)
	 *         - addr  (Address#0  @49659)
	 * }</pre>
	 *
	 * @return formatted string
	 * @see Property
	 * @see ValueProperty
	 * @see EntityProperty
	 * @see CollectionProperty
	 * @see RefProperty
	 */
	@Override
	public String print() {
		StringBuilder sb = new StringBuilder(64);

		// header
		write(sb, 0, "Bundle[id=", id, ", revision=", revision, "]");

		// down the tree
		for (Map.Entry<String, Property> entry : fields.entrySet()) {
			String name = entry.getKey();
			Property prop = entry.getValue();
			prop.print(sb, 1, name, name);
		}

		return sb.toString();
	}

	@Override
	public String toString() {
		return "Bundle[id=" + id + ", revision=" + revision + ", fields=" + fields.keySet() + "]";
	}

}
