package pl.decerto.hyperon.persistence.model.def;

import java.util.Objects;

/**
 * @author przemek hertel
 */
public class TypeDef {

	private final String simpleType;
	private final EntityType compoundType;

	public TypeDef(EntityType compoundType) {
		this.compoundType = compoundType;
		this.simpleType = null;
	}

	public TypeDef(String simpleType) {
		this.simpleType = simpleType;
		this.compoundType = null;
	}

	/**
	 * Create simple {@code TypeDef} based on given {@code simpleType}.
	 *
	 * @param simpleType must be supported by Hyperon Persistence Engine
	 * @return new type definition
	 */
	public static TypeDef buildSimpleType(String simpleType) {
		return new TypeDef(simpleType);
	}

	/**
	 * Create entity {@code TypeDef} based on given {@code compoundType}.
	 *
	 * @param compoundType entity (complex) type
	 * @return new type definition
	 */
	public static TypeDef buildCompoundType(EntityType compoundType) {
		return new TypeDef(compoundType);
	}

	public boolean isSimple() {
		return simpleType != null;
	}

	public boolean isCompound() {
		return compoundType != null;
	}

	public String getCode() {
		return compoundType != null ? compoundType.getName() : simpleType;
	}

	public EntityType getCompoundType() {
		return compoundType;
	}

	public String getSimpleType() {
		return simpleType;
	}

	@Override
	public String toString() {
		return getCode();
	}

	@Override
	public boolean equals(Object o) {
		if (this == o) {
			return true;
		}
		if (!(o instanceof TypeDef)) {
			return false;
		}

		TypeDef other = (TypeDef) o;
		return Objects.equals(simpleType, other.simpleType) && eqEntityType(compoundType, other.compoundType);
	}

	private boolean eqEntityType(EntityType t1, EntityType t2) {
		if (t1 == null || t2 == null) {
			return t1 == t2;
		}

		return Objects.equals(t1.getName(), t2.getName());
	}

	@Override
	public int hashCode() {
		String entityTypeName = compoundType != null ? compoundType.getName() : null;
		return Objects.hash(entityTypeName, simpleType);
	}
}
