package pl.decerto.hyperon.persistence.helper;

import java.util.List;
import java.util.Map;
import java.util.Objects;

import pl.decerto.hyperon.persistence.model.value.CollectionProperty;
import pl.decerto.hyperon.persistence.model.value.ElementType;
import pl.decerto.hyperon.persistence.model.value.EntityProperty;
import pl.decerto.hyperon.persistence.model.value.Property;

/**
 * @author przemek hertel
 */
public class BundleWalker {

	private BundleWalker() {
	}

	public static void walk(Property root, PropertyVisitor visitor) {
		walk(root, visitor, true);
	}

	public static void walk(Property root, PropertyVisitor visitor, boolean usePathsInVisitor) {
		if (Objects.nonNull(root)) {
			if (usePathsInVisitor) {
				internalWalk(root, visitor, null);
			} else {
				internalWalk(root, visitor);
			}
		}
	}

	private static void internalWalk(Property prop, PropertyVisitor visitor, String path) {

		// entity, collection, value or ref
		ElementType type = prop.getElementType();

		// enter this property
		visitor.visit(prop, path, type);

		// do not follow ref
		if (prop.isRef()) {
			return;
		}

		// follow entity fields
		if (type == ElementType.ENTITY) {
			EntityProperty ep = (EntityProperty) prop;

			for (Map.Entry<String, Property> entry : ep.getFields().entrySet()) {
				String name = entry.getKey();
				Property p = entry.getValue();
				internalWalk(p, visitor, toPath(path, name));
			}
		}

		// follow collection elements
		if (type == ElementType.COLLECTION) {
			CollectionProperty cp = (CollectionProperty) prop;

			final List<Property> elements = cp.getList();
			for (int i = 0; i < elements.size(); i++) {
				Property e = elements.get(i);
				internalWalk(e, visitor, toPath(path, i));
			}
		}
	}

	private static void internalWalk(Property prop, PropertyVisitor visitor) {
		// entity, collection, value or ref
		ElementType type = prop.getElementType();

		// enter this property
		visitor.visit(prop, type);

		// do not follow ref
		if (prop.isRef()) {
			return;
		}

		// follow entity fields
		if (type == ElementType.ENTITY) {
			EntityProperty ep = (EntityProperty) prop;

			for (Map.Entry<String, Property> entry : ep.getFields().entrySet()) {
				Property p = entry.getValue();
				internalWalk(p, visitor);
			}
		}

		// follow collection elements
		if (type == ElementType.COLLECTION) {
			CollectionProperty cp = (CollectionProperty) prop;

			final List<Property> elements = cp.getList();
			for (Property e : elements) {
				internalWalk(e, visitor);
			}
		}
	}

	private static String toPath(String path, String name) {
		return path != null ? path + "." + name : name;
	}

	private static String toPath(String path, int index) {
		return path + "[" + index + "]";
	}

}
