package pl.allegro.tech.build.axion.release.infrastructure.git

import org.ajoberstar.grgit.BranchStatus
import org.ajoberstar.grgit.Grgit
import org.ajoberstar.grgit.Status
import org.ajoberstar.grgit.operation.FetchOp
import org.eclipse.jgit.api.FetchCommand
import org.eclipse.jgit.api.LogCommand
import org.eclipse.jgit.api.PushCommand
import org.eclipse.jgit.api.TransportCommand
import org.eclipse.jgit.api.errors.NoHeadException
import org.eclipse.jgit.errors.RepositoryNotFoundException
import org.eclipse.jgit.lib.Config
import org.eclipse.jgit.lib.Constants
import org.eclipse.jgit.lib.ObjectId
import org.eclipse.jgit.revwalk.RevCommit
import org.eclipse.jgit.revwalk.RevSort
import org.eclipse.jgit.revwalk.RevWalk
import org.eclipse.jgit.transport.RemoteConfig
import org.eclipse.jgit.transport.TagOpt
import org.eclipse.jgit.transport.Transport
import org.eclipse.jgit.transport.URIish
import pl.allegro.tech.build.axion.release.domain.logging.ReleaseLogger
import pl.allegro.tech.build.axion.release.domain.scm.*

import java.util.regex.Pattern

class GitRepository implements ScmRepository {

    private static final ReleaseLogger logger = ReleaseLogger.Factory.logger(GitRepository)

    private static final String GIT_TAG_PREFIX = 'refs/tags/'

    private final TransportConfigFactory transportConfigFactory = new TransportConfigFactory()

    private final File repositoryDir

    private final Grgit repository

    private final ScmProperties properties

    GitRepository(ScmProperties properties) {
        try {
            this.repositoryDir = properties.directory
            repository = Grgit.open(dir: repositoryDir)
            this.properties = properties
        }
        catch(RepositoryNotFoundException exception) {
            throw new ScmRepositoryUnavailableException(exception)
        }

        if (properties.attachRemote) {
            this.attachRemote(properties.remote, properties.remoteUrl)
        }
        if (properties.fetchTags) {
            this.fetchTags(properties.identity, properties.remote)
        }
    }

    @Override
    void fetchTags(ScmIdentity identity, String remoteName) {
        identity.useDefault ? callFetch(remoteName) : callLowLevelFetch(identity, remoteName);
    }

    private void callFetch(String remoteName) {
        repository.fetch(remote: remoteName, tagMode: FetchOp.TagMode.ALL, refSpecs: [Transport.REFSPEC_TAGS])
    }

    private void callLowLevelFetch(ScmIdentity identity, String remoteName) {
        FetchCommand fetch = repository.repository.jgit.fetch()
        fetch.remote = remoteName
        fetch.tagOpt = TagOpt.FETCH_TAGS
        fetch.refSpecs = [Transport.REFSPEC_TAGS]
        setTransportOptions(identity, fetch)

        fetch.call()
    }

    @Override
    void tag(String tagName) {
        String headId = repository.repository.jgit.repository.resolve(Constants.HEAD).name()
        boolean isOnExistingTag = repository.tag.list().any({it.name == tagName && it.commit.id == headId})
        if (!isOnExistingTag) {
            repository.tag.add(name: tagName)
        } else {
            logger.debug("The head commit $headId already has the tag $tagName.")
        }
    }

    @Override
    void push(ScmIdentity identity, ScmPushOptions pushOptions) {
        push(identity, pushOptions, false)
    }

    void push(ScmIdentity identity, ScmPushOptions pushOptions, boolean all) {
        identity.useDefault ? callPush(pushOptions, all) : callLowLevelPush(identity, pushOptions, all)
    }

    private void callPush(ScmPushOptions pushOptions, boolean all) {
        if(!pushOptions.pushTagsOnly) {
            repository.push(remote: pushOptions.remote, all: all)
        }
        repository.push(remote: pushOptions.remote, tags: true, all: all)
    }

    private void callLowLevelPush(ScmIdentity identity, ScmPushOptions pushOptions, boolean all) {
        if(!pushOptions.pushTagsOnly) {
            pushCommand(identity, pushOptions.remote, all).call()
        }
        pushCommand(identity, pushOptions.remote, all).setPushTags().call()
    }

    private PushCommand pushCommand(ScmIdentity identity, String remoteName, boolean all) {
        PushCommand push = repository.repository.jgit.push()
        push.remote = remoteName

        if(all) {
            push.setPushAll()
        }

        setTransportOptions(identity, push)

        return push
    }

    private void setTransportOptions(ScmIdentity identity, TransportCommand command) {
        command.transportConfigCallback = transportConfigFactory.create(identity)
    }

    @Override
    void attachRemote(String remoteName, String remoteUrl) {
        Config config = repository.repository.jgit.repository.config

        RemoteConfig remote = new RemoteConfig(config, remoteName)
        // clear other push specs
        List<URIish> pushUris = new ArrayList<>(remote.pushURIs)
        for (URIish uri : pushUris) {
            remote.removePushURI(uri)
        }

        remote.addPushURI(new URIish(remoteUrl))
        remote.update(config)

        config.save()
    }

    @Override
    void commit(List patterns, String message) {
        if(!patterns.isEmpty()) {
            String canonicalPath = Pattern.quote(repositoryDir.canonicalPath + File.separatorChar)
            repository.add(patterns: patterns.collect { it.replaceFirst(canonicalPath, '') })
        }
        repository.commit(message: message)
    }

    String currentBranch() {
        return repository.branch.current.name
    }

    @Override
    ScmPosition currentPosition(Pattern pattern) {
        return currentPosition(pattern, Pattern.compile('$a^'), LAST_TAG_SELECTOR)
    }

    @Override
    ScmPosition currentPosition(Pattern pattern, Closure<String> tagSelector) {
        return currentPosition(pattern, Pattern.compile('$a^'), tagSelector)
    }

    @Override
    ScmPosition currentPosition(Pattern pattern, Pattern inversePattern, Closure<String> tagSelector) {
        if(!hasCommits()) {
            return ScmPosition.defaultPosition()
        }

        Map tags = repository.tag.list()
                .grep({ def tag = it.fullName.substring(GIT_TAG_PREFIX.length()); tag ==~ pattern && !(tag ==~ inversePattern) })
                .inject([:].withDefault {p -> []}, { map, entry ->
                    map[entry.commit.id] << entry.fullName.substring(GIT_TAG_PREFIX.length())
                    return map
                })

        ObjectId headId = repository.repository.jgit.repository.resolve(Constants.HEAD)
        String branch = repository.branch.current.name

        RevWalk walk = new RevWalk(repository.repository.jgit.repository)
        walk.sort(RevSort.TOPO)
        walk.sort(RevSort.COMMIT_TIME_DESC, true)
        RevCommit head = walk.parseCommit(headId)

        List<String> tagNameList = null

        walk.markStart(head)
        RevCommit commit
        for (commit = walk.next(); commit != null; commit = walk.next()) {
            tagNameList = tags[commit.id.name()]
            if (tagNameList) {
                break
            }
        }
        walk.dispose()
        String tagName = tagSelector(tagNameList)
        boolean onTag = (commit == null) ? null : commit.id.name() == headId.name()
        return new ScmPosition(branch, tagName, onTag, checkUncommittedChanges())
    }

    private boolean hasCommits() {
        LogCommand log = repository.repository.jgit.log()
        log.maxCount = 1

        try {
            log.call()
            return true
        }
        catch(NoHeadException exception) {
            return false
        }
    }

    @Override
    boolean remoteAttached(String remoteName) {
        Config config = repository.repository.jgit.repository.config

        return config.getSubsections('remote').any { it == remoteName }
    }

    @Override
    boolean checkUncommittedChanges() {
        return !repository.status().isClean()
    }

    @Override
    boolean checkAheadOfRemote() {
        BranchStatus status = repository.branch.status(branch: repository.branch.current.fullName)
        return status.aheadCount != 0 || status.behindCount != 0
    }

    void checkoutBranch(String branchName) {
        repository.checkout(branch: branchName, createBranch: true)
    }

    Status listChanges() {
        return repository.status()
    }

    @Override
    List<String> lastLogMessages(int messageCount) {
        return repository.log(maxCommits: messageCount)*.fullMessage
    }
}
