/*
 * See the NOTICE file distributed with this work for additional
 * information regarding copyright ownership.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.xwiki.velocity;

import java.util.Map;
import java.util.Set;

import javax.script.Bindings;
import javax.script.ScriptContext;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.runtime.directive.Scope;
import org.xwiki.stability.Unstable;

/**
 * Maintains the current ScriptContext in sync with any modification of the VelocityContext.
 * 
 * @version $Id: 9e9538840dbb91d6f48a14b951bbb1291716ae7b $
 * @since 15.9RC1
 */
@Unstable
public class ScriptVelocityContext extends XWikiVelocityContext
{
    private final Set<String> reservedBindings;

    private ScriptContext scriptContext;

    /**
     * @param parent the initial Velocity context
     * @param reservedBindings the binding that should not be synchronized
     */
    public ScriptVelocityContext(VelocityContext parent, Set<String> reservedBindings)
    {
        super(parent);

        this.reservedBindings = reservedBindings;
    }

    /**
     * @param parent the initial Velocity context
     * @param logDeprecated true if use of deprecated binding should be logged
     * @param reservedBindings the binding that should not be synchronized
     */
    public ScriptVelocityContext(VelocityContext parent, boolean logDeprecated, Set<String> reservedBindings)
    {
        super(parent, logDeprecated);

        this.reservedBindings = reservedBindings;
    }

    /**
     * @return the current script context
     */
    public ScriptContext getScriptContext()
    {
        return this.scriptContext;
    }

    /**
     * @param scriptContext the current script context
     */
    public void setScriptContext(ScriptContext scriptContext)
    {
        this.scriptContext = scriptContext;

        if (this.scriptContext != null) {
            copyScriptContext(ScriptContext.GLOBAL_SCOPE);
            copyScriptContext(ScriptContext.ENGINE_SCOPE);
        }
    }

    private void copyScriptContext(int scope)
    {
        Bindings bindings = this.scriptContext.getBindings(scope);
        if (bindings != null) {
            for (Map.Entry<String, Object> entry : bindings.entrySet()) {
                if (!this.reservedBindings.contains(entry.getKey())) {
                    Object currentValue = get(entry.getKey());
                    // Don't replace internal Velocity bindings
                    if (!(currentValue instanceof Scope)) {
                        put(entry.getKey(), entry.getValue());
                    }
                }
            }
        }
    }

    // VelocityContext

    @Override
    public Object internalPut(String key, Object value)
    {
        try {
            return super.internalPut(key, value);
        } finally {
            if (this.scriptContext != null && !this.reservedBindings.contains(key)) {
                this.scriptContext.setAttribute(key, value, ScriptContext.ENGINE_SCOPE);
            }
        }
    }

    @Override
    public Object internalRemove(String key)
    {
        try {
            return super.internalRemove(key);
        } finally {
            if (this.scriptContext != null && !this.reservedBindings.contains(key)) {
                this.scriptContext.removeAttribute(key, ScriptContext.ENGINE_SCOPE);
            }
        }
    }
}
