/*
 * Decompiled with CFR 0.152.
 */
package org.xwiki.store.blob.internal;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Locale;
import java.util.Set;
import java.util.stream.Stream;
import org.apache.commons.io.file.PathUtils;
import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.xwiki.store.blob.AbstractBlobStore;
import org.xwiki.store.blob.Blob;
import org.xwiki.store.blob.BlobAlreadyExistsException;
import org.xwiki.store.blob.BlobNotFoundException;
import org.xwiki.store.blob.BlobOption;
import org.xwiki.store.blob.BlobPath;
import org.xwiki.store.blob.BlobStore;
import org.xwiki.store.blob.BlobStoreException;
import org.xwiki.store.blob.BlobWriteMode;
import org.xwiki.store.blob.FileSystemBlobStoreProperties;
import org.xwiki.store.blob.internal.BlobOptionSupport;
import org.xwiki.store.blob.internal.FileSystemBlob;

public class FileSystemBlobStore
extends AbstractBlobStore<FileSystemBlobStoreProperties> {
    static final int NUM_ATTEMPTS = 5;
    private static final String SOURCE_AND_TARGET_SAME_ERROR = "source and target paths are the same";
    private static final Set<Class<? extends BlobOption>> SUPPORTED_COPY_MOVE_OPTIONS = Set.of(BlobWriteMode.class);
    private final Path basePath;

    public FileSystemBlobStore(String name, FileSystemBlobStoreProperties properties) {
        this.initialize(name, "filesystem", properties);
        this.basePath = properties.getRootDirectory();
    }

    public Blob getBlob(BlobPath path) throws BlobStoreException {
        Path blobFsPath = this.getBlobFilePath(path);
        return new FileSystemBlob(path, blobFsPath, this);
    }

    public Path getBlobFilePath(BlobPath blobPath) {
        Path currentPath = this.basePath;
        for (String segment : blobPath.getNames()) {
            currentPath = currentPath.resolve(segment);
        }
        return currentPath;
    }

    public Stream<Blob> listDescendants(BlobPath path) throws BlobStoreException {
        Path absolutePath = this.getBlobFilePath(path);
        if (!Files.exists(absolutePath, new LinkOption[0]) || !Files.isDirectory(absolutePath, new LinkOption[0])) {
            return Stream.empty();
        }
        try {
            Path normalizedAbsolutePath = absolutePath.normalize();
            return Files.walk(normalizedAbsolutePath, new FileVisitOption[0]).filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).map(p -> this.toBlobFromAbsolute(p.normalize(), normalizedAbsolutePath, path));
        }
        catch (IOException e) {
            throw new BlobStoreException("Failed to list blobs in directory: " + String.valueOf(absolutePath), (Throwable)e);
        }
    }

    private Blob toBlobFromAbsolute(Path normalizedPath, Path normalizedAbsolutePath, BlobPath basePath) {
        if (!normalizedPath.startsWith(normalizedAbsolutePath)) {
            throw new IllegalStateException("Found a file outside the expected directory: " + String.valueOf(normalizedPath));
        }
        Path relativePath = normalizedAbsolutePath.relativize(normalizedPath);
        ArrayList<String> segments = new ArrayList<String>(basePath.getNames());
        for (Path segment : relativePath) {
            segments.add(segment.toString());
        }
        return new FileSystemBlob(BlobPath.absolute(segments), normalizedPath, this);
    }

    public Blob copyBlob(BlobPath sourcePath, BlobPath targetPath, BlobOption ... options) throws BlobStoreException {
        if (sourcePath.equals((Object)targetPath)) {
            throw new BlobStoreException(SOURCE_AND_TARGET_SAME_ERROR);
        }
        BlobOptionSupport.validateSupportedOptions(SUPPORTED_COPY_MOVE_OPTIONS, (BlobOption[])options);
        BlobWriteMode writeMode = BlobWriteMode.resolve((BlobWriteMode)BlobWriteMode.CREATE_NEW, (BlobOption[])options);
        Path absoluteSourcePath = this.getBlobFilePath(sourcePath);
        this.transferBlobInternal(sourcePath, targetPath, absoluteSourcePath, TransferKind.COPY, writeMode);
        return this.getBlob(targetPath);
    }

    public Blob copyBlob(BlobStore sourceStore, BlobPath sourcePath, BlobPath targetPath, BlobOption ... options) throws BlobStoreException {
        BlobOptionSupport.validateSupportedOptions(SUPPORTED_COPY_MOVE_OPTIONS, (BlobOption[])options);
        BlobWriteMode writeMode = BlobWriteMode.resolve((BlobWriteMode)BlobWriteMode.CREATE_NEW, (BlobOption[])options);
        if (sourceStore instanceof FileSystemBlobStore) {
            FileSystemBlobStore fileSystemBlobStore = (FileSystemBlobStore)sourceStore;
            Path absoluteSourcePath = fileSystemBlobStore.getBlobFilePath(sourcePath);
            this.transferBlobInternal(sourcePath, targetPath, absoluteSourcePath, TransferKind.COPY, writeMode);
        } else {
            try (InputStream inputStream = sourceStore.getBlob(sourcePath).getStream();){
                Blob targetBlob = this.getBlob(targetPath);
                targetBlob.writeFromStream(inputStream, new BlobOption[]{writeMode});
            }
            catch (BlobStoreException e) {
                throw e;
            }
            catch (Exception e) {
                throw new BlobStoreException("Reading source blob failed", (Throwable)e);
            }
        }
        return this.getBlob(targetPath);
    }

    public Blob moveBlob(BlobPath sourcePath, BlobPath targetPath, BlobOption ... options) throws BlobStoreException {
        if (sourcePath.equals((Object)targetPath)) {
            throw new BlobStoreException(SOURCE_AND_TARGET_SAME_ERROR);
        }
        BlobOptionSupport.validateSupportedOptions(SUPPORTED_COPY_MOVE_OPTIONS, (BlobOption[])options);
        BlobWriteMode writeMode = BlobWriteMode.resolve((BlobWriteMode)BlobWriteMode.CREATE_NEW, (BlobOption[])options);
        Path absoluteSourcePath = this.getBlobFilePath(sourcePath);
        this.transferBlobInternal(sourcePath, targetPath, absoluteSourcePath, TransferKind.MOVE, writeMode);
        return this.getBlob(targetPath);
    }

    public Blob moveBlob(BlobStore sourceStore, BlobPath sourcePath, BlobPath targetPath, BlobOption ... options) throws BlobStoreException {
        if (sourceStore instanceof FileSystemBlobStore) {
            FileSystemBlobStore fileSystemBlobStore = (FileSystemBlobStore)sourceStore;
            BlobOptionSupport.validateSupportedOptions(SUPPORTED_COPY_MOVE_OPTIONS, (BlobOption[])options);
            Path absoluteSourcePath = fileSystemBlobStore.getBlobFilePath(sourcePath);
            BlobWriteMode writeMode = BlobWriteMode.resolve((BlobWriteMode)BlobWriteMode.CREATE_NEW, (BlobOption[])options);
            this.transferBlobInternal(sourcePath, targetPath, absoluteSourcePath, TransferKind.MOVE, writeMode);
            return this.getBlob(targetPath);
        }
        Blob targetBlob = this.copyBlob(sourceStore, sourcePath, targetPath, options);
        sourceStore.deleteBlob(sourcePath);
        return targetBlob;
    }

    public boolean hasDescendants(BlobPath path) throws BlobStoreException {
        try (Stream<Blob> stream = this.listDescendants(path);){
            boolean bl = stream.findFirst().isPresent();
            return bl;
        }
    }

    private void transferBlobInternal(BlobPath sourcePath, BlobPath targetPath, Path absoluteSourcePath, TransferKind kind, BlobWriteMode writeMode) throws BlobStoreException {
        Path absoluteTargetPath = this.getBlobFilePath(targetPath);
        String operationName = kind.toString().toLowerCase(Locale.ROOT);
        NoSuchFileException lastNoSuchFileException = null;
        for (int attempt = 0; attempt < 5; ++attempt) {
            try {
                this.createParents(absoluteTargetPath);
                CopyOption[] copyOptions = FileSystemBlobStore.resolveCopyOptions(writeMode);
                if (kind == TransferKind.MOVE) {
                    Files.move(absoluteSourcePath, absoluteTargetPath, copyOptions);
                    this.cleanUpParents(absoluteSourcePath);
                } else {
                    Files.copy(absoluteSourcePath, absoluteTargetPath, copyOptions);
                }
                return;
            }
            catch (NoSuchFileException e) {
                if (!Files.exists(absoluteSourcePath, new LinkOption[0])) {
                    this.cleanUpParents(absoluteTargetPath);
                    throw new BlobNotFoundException(sourcePath, (Throwable)e);
                }
                lastNoSuchFileException = e;
                continue;
            }
            catch (FileAlreadyExistsException e) {
                throw new BlobAlreadyExistsException(targetPath, (Throwable)e);
            }
            catch (IOException e) {
                this.cleanUpParents(absoluteTargetPath);
                throw new BlobStoreException(operationName + " blob failed", (Throwable)e);
            }
        }
        this.cleanUpParents(absoluteTargetPath);
        throw new BlobStoreException("%s blob failed after %d attempts".formatted(operationName, 5), (Throwable)lastNoSuchFileException);
    }

    private static CopyOption[] resolveCopyOptions(BlobWriteMode writeMode) {
        CopyOption[] copyOptionArray;
        if (writeMode == BlobWriteMode.REPLACE_EXISTING) {
            CopyOption[] copyOptionArray2 = new CopyOption[1];
            copyOptionArray = copyOptionArray2;
            copyOptionArray2[0] = StandardCopyOption.REPLACE_EXISTING;
        } else {
            copyOptionArray = new CopyOption[]{};
        }
        return copyOptionArray;
    }

    void createParents(Path absoluteTargetPath) throws IOException {
        Path targetParent = absoluteTargetPath.getParent();
        if (targetParent != null) {
            Files.createDirectories(targetParent, new FileAttribute[0]);
        }
    }

    public void deleteBlob(BlobPath path) throws BlobStoreException {
        try {
            Path fileSystemPath = this.getBlobFilePath(path);
            Files.deleteIfExists(fileSystemPath);
            this.cleanUpParents(fileSystemPath);
        }
        catch (IOException e) {
            throw new BlobStoreException("delete blob failed", (Throwable)e);
        }
    }

    void cleanUpParents(Path fileSystemPath) {
        try {
            for (Path parentPath = fileSystemPath.getParent(); parentPath != null && !this.basePath.equals(parentPath) && Files.isDirectory(parentPath, new LinkOption[0]) && PathUtils.isEmptyDirectory((Path)parentPath); parentPath = parentPath.getParent()) {
                Files.deleteIfExists(parentPath);
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    public void deleteDescendants(BlobPath path) throws BlobStoreException {
        try {
            Path absolutePath = this.getBlobFilePath(path);
            if (Files.exists(absolutePath, new LinkOption[0]) && Files.isDirectory(absolutePath, new LinkOption[0])) {
                PathUtils.deleteDirectory((Path)absolutePath);
                this.cleanUpParents(absolutePath);
            }
        }
        catch (IOException e) {
            throw new BlobStoreException("delete blobs failed", (Throwable)e);
        }
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof FileSystemBlobStore)) {
            return false;
        }
        FileSystemBlobStore blobStore = (FileSystemBlobStore)((Object)o);
        return new EqualsBuilder().append((Object)this.basePath, (Object)blobStore.basePath).isEquals();
    }

    public int hashCode() {
        return new HashCodeBuilder(17, 37).append((Object)this.basePath).toHashCode();
    }

    private static enum TransferKind {
        COPY,
        MOVE;

    }
}

