/*
 * Decompiled with CFR 0.152.
 */
package org.xwiki.store.blob.internal;

import jakarta.inject.Inject;
import jakarta.inject.Singleton;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.xwiki.component.annotation.Component;
import org.xwiki.store.blob.Blob;
import org.xwiki.store.blob.BlobAlreadyExistsException;
import org.xwiki.store.blob.BlobNotFoundException;
import org.xwiki.store.blob.BlobPath;
import org.xwiki.store.blob.BlobStore;
import org.xwiki.store.blob.BlobStoreException;
import org.xwiki.store.blob.BlobWriteMode;

@Component(roles={BlobStoreMigrator.class})
@Singleton
public class BlobStoreMigrator {
    private static final BlobPath MIGRATION_MARKER_PATH = BlobPath.absolute("_migration.txt", new String[0]);
    @Inject
    private Logger logger;

    public void migrate(BlobStore targetStore, BlobStore sourceStore) throws BlobStoreException {
        String storeName = targetStore.getName();
        BlobPath markerPath = MIGRATION_MARKER_PATH;
        Blob markerBlob = targetStore.getBlob(markerPath);
        try {
            this.writeMarker(markerBlob, storeName, sourceStore, targetStore);
            this.logger.info("Starting blob store migration for [{}] (source: [{}], target: [{}]).", new Object[]{storeName, sourceStore.getType(), targetStore.getType()});
        }
        catch (BlobAlreadyExistsException e) {
            this.logger.info("Detected existing migration marker [{}]; resuming migration of blob store [{}].", (Object)markerPath, (Object)storeName);
        }
        catch (BlobStoreException e) {
            throw new BlobStoreException("Failed to create migration marker for store [%s] at [%s]".formatted(storeName, markerPath), (Throwable)((Object)e));
        }
        this.migrateContent(storeName, targetStore, sourceStore);
        this.logger.info("Completed blob store migration for [{}].", (Object)storeName);
        try {
            targetStore.deleteBlob(markerPath);
        }
        catch (BlobStoreException e) {
            throw new BlobStoreException("Failed to delete migration marker [%s] after migrating store [%s]. Remove the marker manually once the migration is verified.".formatted(markerPath, storeName), (Throwable)((Object)e));
        }
    }

    public boolean isMigrationInProgress(BlobStore targetStore) throws BlobStoreException {
        return targetStore.getBlob(MIGRATION_MARKER_PATH).exists();
    }

    private void writeMarker(Blob markerBlob, String storeName, BlobStore sourceStore, BlobStore targetStore) throws BlobStoreException {
        String payload = "Store Name: %s\nSource Store: %s\nTarget Store: %s\nStarted At: %s\n".formatted(storeName, sourceStore.getType(), targetStore.getType(), Instant.now());
        try (ByteArrayInputStream stream = new ByteArrayInputStream(payload.getBytes(StandardCharsets.UTF_8));){
            markerBlob.writeFromStream(stream, BlobWriteMode.CREATE_NEW);
        }
        catch (IOException e) {
            throw new BlobStoreException("Failed to write marker for store [%s]".formatted(storeName), e);
        }
    }

    private void migrateContent(String storeName, BlobStore targetStore, BlobStore sourceStore) throws BlobStoreException {
        try (Stream<Blob> blobs = sourceStore.listDescendants(BlobPath.root());){
            for (Blob sourceBlob : blobs::iterator) {
                BlobPath path = sourceBlob.getPath();
                this.moveBlob(path, sourceStore, targetStore);
            }
        }
        catch (BlobStoreException e) {
            throw e;
        }
        catch (Exception e) {
            throw new BlobStoreException("Failed to list blobs from migration store [%s] during migration of [%s]".formatted(sourceStore.getType(), storeName), e);
        }
    }

    private void moveBlob(BlobPath path, BlobStore sourceStore, BlobStore targetStore) throws BlobStoreException {
        try {
            targetStore.moveBlob(sourceStore, path, path, BlobWriteMode.REPLACE_EXISTING);
        }
        catch (BlobNotFoundException e) {
            this.logger.debug("Skipping blob [{}] during migration of [{}] since it no longer exists in source store.", (Object)path, (Object)targetStore.getName());
        }
        catch (BlobStoreException e) {
            throw new BlobStoreException("Failed to move blob [%s] while migrating [%s] from [%s] to [%s]. Fix the issue or delete the source blob to unblock the migration.The migration will be resumed on the next attempt.".formatted(path, targetStore.getName(), sourceStore.getType(), targetStore.getType()), (Throwable)((Object)e));
        }
    }
}

