/*
 * JBoss, Home of Professional Open Source.
 * Copyright 2016 Red Hat, Inc., and individual contributors
 * as indicated by the @author tags.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.wildfly.transaction;

import org.wildfly.common.Assert;
import org.wildfly.security.permission.AbstractNameSetOnlyPermission;
import org.wildfly.security.util.StringEnumeration;
import org.wildfly.security.util.StringMapping;

/**
 * Permission for transaction operations.
 *
 * @author <a href="mailto:david.lloyd@redhat.com">David M. Lloyd</a>
 */
public final class TransactionPermission extends AbstractNameSetOnlyPermission<TransactionPermission> {
    private static final StringEnumeration NAMES = StringEnumeration.of(
        "getRecoveryInterface",
        "getTransactionResolver",
        "getProviderInterface",
        "registerAssociationListener",
        "registerCreationListener",
        "suspendRequests",
        "resumeRequests",
        "modifyUserTransactionAvailability",
        "getXATerminator",
        "setGlobalDefaultTransactionTimeout"
    );

    static final StringMapping<TransactionPermission> mapping = new StringMapping<>(NAMES, TransactionPermission::new);

    private static final TransactionPermission allPermission = new TransactionPermission("*");

    public TransactionPermission(final String name) {
        super(name, NAMES);
    }

    public TransactionPermission(final String name, final String action) {
        this(name);
    }

    public TransactionPermission withName(final String name) {
        return new TransactionPermission(name);
    }

    public static TransactionPermission forName(final String name) {
        Assert.checkNotNullParam("name", name);
        return "*".equals(name) ? allPermission : mapping.getItemByString(name);
    }
}
