/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.auth.realm.token.validator;

import jakarta.json.Json;
import jakarta.json.JsonArray;
import jakarta.json.JsonObject;
import jakarta.json.JsonReader;
import jakarta.json.JsonString;
import jakarta.json.JsonValue;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.PublicKey;
import java.security.Signature;
import java.security.SignatureException;
import java.util.Arrays;
import java.util.Base64;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import org.wildfly.common.Assert;
import org.wildfly.common.iteration.ByteIterator;
import org.wildfly.common.iteration.CodePointIterator;
import org.wildfly.security.auth.realm.token.TokenValidator;
import org.wildfly.security.auth.realm.token._private.ElytronMessages;
import org.wildfly.security.auth.realm.token.validator.JwkManager;
import org.wildfly.security.auth.server.RealmUnavailableException;
import org.wildfly.security.authz.Attributes;
import org.wildfly.security.evidence.BearerTokenEvidence;
import org.wildfly.security.json.util.JsonUtil;
import org.wildfly.security.pem.Pem;
import org.wildfly.security.pem.PemEntry;

public class JwtValidator
implements TokenValidator {
    private final Set<String> issuers;
    private final Set<String> audiences;
    private final Set<String> allowedJkuValues;
    private final JwkManager jwkManager;
    private final Map<String, PublicKey> namedKeys;
    private final PublicKey defaultPublicKey;

    public static Builder builder() {
        return new Builder();
    }

    JwtValidator(Builder configuration) {
        this.issuers = (Set)Assert.checkNotNullParam((String)"issuers", configuration.issuers);
        this.audiences = (Set)Assert.checkNotNullParam((String)"audience", configuration.audience);
        this.allowedJkuValues = (Set)Assert.checkNotNullParam((String)"allowedJkuValues", configuration.allowedJkuValues);
        this.defaultPublicKey = configuration.publicKey;
        this.namedKeys = configuration.namedKeys;
        if (configuration.sslContext != null) {
            this.jwkManager = new JwkManager(configuration.sslContext, configuration.hostnameVerifier != null ? configuration.hostnameVerifier : HttpsURLConnection.getDefaultHostnameVerifier(), configuration.updateTimeout, configuration.connectionTimeout, configuration.readTimeout, configuration.minTimeBetweenRequests, configuration.allowedJkuValues);
        } else {
            ElytronMessages.log.tokenRealmJwtNoSSLIgnoringJku();
            this.jwkManager = null;
        }
        if (this.defaultPublicKey == null && this.jwkManager == null && this.namedKeys.isEmpty()) {
            ElytronMessages.log.tokenRealmJwtWarnNoPublicKeyIgnoringSignatureCheck();
        }
        if (this.issuers.isEmpty()) {
            ElytronMessages.log.tokenRealmJwtWarnNoIssuerIgnoringIssuerCheck();
        }
        if (this.audiences.isEmpty()) {
            ElytronMessages.log.tokenRealmJwtWarnNoAudienceIgnoringAudienceCheck();
        }
        if (this.allowedJkuValues.isEmpty()) {
            ElytronMessages.log.allowedJkuValuesNotConfigured();
        }
    }

    @Override
    public Attributes validate(BearerTokenEvidence evidence) throws RealmUnavailableException {
        Assert.checkNotNullParam((String)"evidence", (Object)evidence);
        String jwt = evidence.getToken();
        String[] parts = jwt.split("\\.", -1);
        if (parts.length < 3) {
            throw ElytronMessages.log.tokenRealmJwtInvalidFormat();
        }
        String encodedHeader = parts[0];
        String encodedClaims = parts[1];
        String encodedSignature = parts[2];
        JsonObject claims = this.extractClaims(encodedClaims);
        if (this.verifySignature(encodedHeader, encodedClaims, encodedSignature) && this.hasValidIssuer(claims) && this.hasValidAudience(claims) && this.verifyTimeConstraints(claims)) {
            return JsonUtil.toAttributes(claims);
        }
        return null;
    }

    private boolean verifyTimeConstraints(JsonObject claims) {
        long currentTime = JwtValidator.currentTimeInSeconds();
        if (claims.containsKey((Object)"exp")) {
            boolean expired;
            boolean bl = expired = currentTime > claims.getJsonNumber("exp").longValue();
            if (expired) {
                ElytronMessages.log.debug("Token expired");
                return false;
            }
        }
        if (claims.containsKey((Object)"nbf")) {
            boolean notBefore;
            boolean bl = notBefore = currentTime >= claims.getJsonNumber("nbf").longValue();
            if (!notBefore) {
                ElytronMessages.log.debugf("Token is before [%s]", (Object)notBefore);
                return false;
            }
        }
        return true;
    }

    private JsonObject extractClaims(String encodedClaims) throws RealmUnavailableException {
        JsonObject retValue = null;
        try (JsonReader jsonReader = null;){
            Base64.Decoder urlDecoder = Base64.getUrlDecoder();
            CodePointIterator decodedClaims = CodePointIterator.ofUtf8Bytes((byte[])urlDecoder.decode(encodedClaims));
            jsonReader = Json.createReader((InputStream)decodedClaims.asUtf8().asInputStream());
            retValue = jsonReader.readObject();
        }
        return retValue;
    }

    private boolean verifySignature(String encodedHeader, String encodedClaims, String encodedSignature) throws RealmUnavailableException {
        if (this.defaultPublicKey == null && this.jwkManager == null && this.namedKeys.isEmpty()) {
            return true;
        }
        try {
            boolean verify;
            Base64.Decoder urlDecoder = Base64.getUrlDecoder();
            byte[] decodedSignature = urlDecoder.decode(encodedSignature);
            Signature signature = this.createSignature(encodedHeader, encodedClaims);
            boolean bl = verify = signature != null ? ByteIterator.ofBytes((byte[])decodedSignature).verify(signature) : false;
            if (!verify) {
                ElytronMessages.log.debug("Signature verification failed");
            }
            return verify;
        }
        catch (Exception cause) {
            throw ElytronMessages.log.tokenRealmJwtSignatureCheckFailed(cause);
        }
    }

    private boolean hasValidAudience(JsonObject claims) throws RealmUnavailableException {
        if (this.audiences.isEmpty()) {
            return true;
        }
        JsonValue audience = (JsonValue)claims.get((Object)"aud");
        if (audience == null) {
            ElytronMessages.log.debug("Token does not contain an audience claim");
            return false;
        }
        JsonArray audClaimArray = JsonValue.ValueType.STRING.equals((Object)audience.getValueType()) ? Json.createArrayBuilder().add(audience).build() : (JsonArray)audience;
        boolean valid = audClaimArray.stream().map(jsonValue -> (JsonString)jsonValue).anyMatch(audience1 -> this.audiences.contains(audience1.getString()));
        if (!valid) {
            ElytronMessages.log.debugf("Audience check failed. Provided [%s] but was expected [%s].", (Object)audClaimArray.toArray(), (Object)this.audiences);
        }
        return valid;
    }

    private boolean hasValidIssuer(JsonObject claims) throws RealmUnavailableException {
        if (this.issuers.isEmpty()) {
            return true;
        }
        String issuer = claims.getString("iss", null);
        if (issuer == null) {
            ElytronMessages.log.debug("Token does not contain an issuer claim");
            return false;
        }
        boolean valid = this.issuers.contains(issuer);
        if (!valid) {
            ElytronMessages.log.debugf("Issuer check failed. Provided [%s] but was expected [%s].", (Object)issuer, (Object)this.issuers);
        }
        return valid;
    }

    private Signature createSignature(String encodedHeader, String encodedClaims) throws NoSuchAlgorithmException, SignatureException, RealmUnavailableException {
        byte[] headerDecoded = Base64.getUrlDecoder().decode(encodedHeader);
        JsonObject headers = null;
        try (JsonReader jsonReader = Json.createReader((InputStream)ByteIterator.ofBytes((byte[])headerDecoded).asInputStream());){
            headers = jsonReader.readObject();
        }
        String headerAlg = this.resolveAlgorithm(headers);
        Signature signature = Signature.getInstance(headerAlg);
        try {
            PublicKey publicKey = this.resolvePublicKey(headers);
            if (publicKey == null) {
                ElytronMessages.log.debug("Public key could not be resolved.");
                return null;
            }
            signature.initVerify(publicKey);
        }
        catch (InvalidKeyException e) {
            e.printStackTrace();
            return null;
        }
        signature.update((encodedHeader + "." + encodedClaims).getBytes());
        return signature;
    }

    private String resolveAlgorithm(JsonObject headers) {
        JsonString algClaim = (JsonString)headers.get((Object)"alg");
        if (algClaim == null) {
            throw ElytronMessages.log.tokenRealmJwtSignatureInvalidAlgorithm("not_provided");
        }
        String algorithm = algClaim.getString();
        ElytronMessages.log.debugf("Token is using algorithm [%s]", (Object)algorithm);
        switch (algorithm) {
            case "RS256": {
                return "SHA256withRSA";
            }
            case "RS384": {
                return "SHA384withRSA";
            }
            case "RS512": {
                return "SHA512withRSA";
            }
        }
        throw ElytronMessages.log.tokenRealmJwtSignatureInvalidAlgorithm(algorithm);
    }

    private PublicKey resolvePublicKey(JsonObject headers) {
        JsonString kid = headers.getJsonString("kid");
        JsonString jku = headers.getJsonString("jku");
        if (kid == null) {
            if (this.defaultPublicKey == null) {
                ElytronMessages.log.debug("Default public key not configured. Cannot validate token without kid claim.");
                return null;
            }
            return this.defaultPublicKey;
        }
        if (jku != null) {
            if (this.jwkManager == null) {
                ElytronMessages.log.debugf("Cannot validate token with jku [%s]. SSL is not configured and jku claim is not supported.", (Object)jku);
                return null;
            }
            if (!this.allowedJkuValues.contains(jku.getString())) {
                ElytronMessages.log.debug("Cannot validate token, jku value is not allowed");
                return null;
            }
            try {
                return this.jwkManager.getPublicKey(kid.getString(), new URL(jku.getString()));
            }
            catch (MalformedURLException e) {
                ElytronMessages.log.debug("Invalid jku URL.");
                return null;
            }
        }
        if (this.namedKeys.isEmpty()) {
            ElytronMessages.log.debug("Cannot validate token with kid claim.");
            return null;
        }
        PublicKey res = this.namedKeys.get(kid.getString());
        if (res == null) {
            ElytronMessages.log.debug("Unknown kid.");
        }
        return res;
    }

    private static long currentTimeInSeconds() {
        return System.currentTimeMillis() / 1000L;
    }

    public static class Builder {
        private static final int CONNECTION_TIMEOUT = 2000;
        private static final int MIN_TIME_BETWEEN_REQUESTS = 10000;
        private Set<String> issuers = new LinkedHashSet<String>();
        private Set<String> audience = new LinkedHashSet<String>();
        private Set<String> allowedJkuValues = new LinkedHashSet<String>();
        private PublicKey publicKey;
        private Map<String, PublicKey> namedKeys = new LinkedHashMap<String, PublicKey>();
        private HostnameVerifier hostnameVerifier;
        private SSLContext sslContext;
        private long updateTimeout = 120000L;
        private int connectionTimeout = 2000;
        private int readTimeout = 2000;
        private int minTimeBetweenRequests = 10000;

        private Builder() {
        }

        public Builder issuer(String ... issuer) {
            this.issuers.addAll(Arrays.asList(issuer));
            return this;
        }

        public Builder audience(String ... audience) {
            this.audience.addAll(Arrays.asList(audience));
            return this;
        }

        public Builder publicKey(byte[] publicKeyPem) {
            Iterator<PemEntry<?>> pemEntryIterator = Pem.parsePemContent(CodePointIterator.ofUtf8Bytes((byte[])publicKeyPem));
            PublicKey publicKey = pemEntryIterator.next().tryCast(PublicKey.class);
            if (publicKey == null) {
                throw ElytronMessages.log.tokenRealmJwtInvalidPublicKeyPem();
            }
            this.publicKey = publicKey;
            return this;
        }

        public Builder publicKey(PublicKey publicKey) {
            this.publicKey = publicKey;
            return this;
        }

        public Builder publicKeys(Map<String, PublicKey> namedKeys) {
            this.namedKeys.putAll(namedKeys);
            return this;
        }

        public Builder useSslContext(SSLContext sslContext) {
            this.sslContext = sslContext;
            return this;
        }

        public Builder useSslHostnameVerifier(HostnameVerifier hostnameVerifier) {
            this.hostnameVerifier = hostnameVerifier;
            return this;
        }

        public Builder setJkuTimeout(long timeout) {
            this.updateTimeout = timeout;
            return this;
        }

        public Builder connectionTimeout(int connectionTimeout) {
            this.connectionTimeout = connectionTimeout;
            return this;
        }

        public Builder readTimeout(int readTimeout) {
            this.readTimeout = readTimeout;
            return this;
        }

        public Builder setJkuMinTimeBetweenRequests(int minTimeBetweenRequests) {
            this.minTimeBetweenRequests = minTimeBetweenRequests;
            return this;
        }

        public Builder setAllowedJkuValues(String ... allowedJkuValues) {
            this.allowedJkuValues.addAll(Arrays.asList(allowedJkuValues));
            return this;
        }

        public JwtValidator build() {
            return new JwtValidator(this);
        }
    }
}

