/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.pem;

import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.util.Iterator;
import java.util.NoSuchElementException;
import java.util.function.BiFunction;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.pem.PemEntry;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.ByteStringBuilder;
import org.wildfly.security.util.CodePointIterator;

public final class Pem {
    private static final Pattern VALID_LABEL = Pattern.compile("[^ -~&&[^-]]");

    public static <R> R parsePemContent(CodePointIterator pemContent, BiFunction<String, ByteIterator, R> contentFunction) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"pemContent", (Object)pemContent);
        Assert.checkNotNullParam((String)"contentFunction", contentFunction);
        if (!pemContent.contentEquals("-----BEGIN ")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.offset());
        }
        String type = pemContent.delimitedBy(45).drainToString().trim();
        Matcher matcher = VALID_LABEL.matcher(type);
        if (matcher.find()) {
            throw ElytronMessages.log.malformedPemContent(matcher.start() + 11);
        }
        if (!pemContent.contentEquals("-----")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.offset());
        }
        ByteIterator byteIterator = pemContent.delimitedBy(45).base64Decode();
        R result = contentFunction.apply(type, byteIterator);
        if (!pemContent.contentEquals("-----END ")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.offset());
        }
        if (!pemContent.contentEquals(type)) {
            throw ElytronMessages.log.malformedPemContent(pemContent.offset());
        }
        if (!pemContent.contentEquals("-----")) {
            throw ElytronMessages.log.malformedPemContent(pemContent.offset());
        }
        return result;
    }

    public static Iterator<PemEntry<?>> parsePemContent(final CodePointIterator pemContent) {
        return new Iterator<PemEntry<?>>(){
            private PemEntry<?> next;

            @Override
            public boolean hasNext() {
                if (this.next == null) {
                    if (!pemContent.hasNext()) {
                        return false;
                    }
                    this.next = Pem.parsePemContent(pemContent, (label, byteIterator) -> {
                        switch (label) {
                            case "CERTIFICATE": {
                                X509Certificate x509Certificate = Pem.parsePemX509CertificateContent(label, byteIterator);
                                return new PemEntry<X509Certificate>(x509Certificate);
                            }
                        }
                        throw ElytronMessages.log.malformedPemContent(pemContent.offset());
                    });
                }
                return true;
            }

            @Override
            public PemEntry<?> next() {
                if (!this.hasNext()) {
                    throw new NoSuchElementException();
                }
                try {
                    PemEntry<?> pemEntry = this.next;
                    return pemEntry;
                }
                finally {
                    this.next = null;
                }
            }
        };
    }

    public static void generatePemContent(ByteStringBuilder target, String type, ByteIterator content) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"target", (Object)target);
        Assert.checkNotNullParam((String)"type", (Object)type);
        Assert.checkNotNullParam((String)"content", (Object)content);
        Matcher matcher = VALID_LABEL.matcher(type);
        if (matcher.find()) {
            throw ElytronMessages.log.invalidPemType("<any valid PEM type>", type);
        }
        target.append("-----BEGIN ").append(type).append("-----\n");
        target.append(content.base64Encode().drainToString(10, 64));
        target.append("\n-----END ").append(type).append("-----\n");
    }

    private static X509Certificate parsePemX509CertificateContent(String type, ByteIterator content) throws IllegalArgumentException {
        if (!type.equals("CERTIFICATE")) {
            throw ElytronMessages.log.invalidPemType("CERTIFICATE", type);
        }
        try {
            CertificateFactory instance = CertificateFactory.getInstance("X.509");
            return (X509Certificate)instance.generateCertificate(content.asInputStream());
        }
        catch (CertificateException e) {
            throw ElytronMessages.log.certificateParseError(e);
        }
    }

    public static X509Certificate parsePemX509Certificate(CodePointIterator pemContent) throws IllegalArgumentException {
        Assert.checkNotNullParam((String)"pemContent", (Object)pemContent);
        return Pem.parsePemContent(pemContent, Pem::parsePemX509CertificateContent);
    }

    public static void generatePemX509Certificate(ByteStringBuilder target, X509Certificate certificate) {
        Assert.checkNotNullParam((String)"target", (Object)target);
        Assert.checkNotNullParam((String)"certificate", (Object)certificate);
        try {
            Pem.generatePemContent(target, "CERTIFICATE", ByteIterator.ofBytes(certificate.getEncoded()));
        }
        catch (CertificateEncodingException e) {
            throw ElytronMessages.log.certificateParseError(e);
        }
    }
}

