/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.cache;

import java.security.Principal;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.LockSupport;
import org.wildfly.common.Assert;
import org.wildfly.security.auth.server.RealmIdentity;
import org.wildfly.security.cache.RealmIdentityCache;

public final class LRURealmIdentityCache
implements RealmIdentityCache {
    private static final float DEFAULT_LOAD_FACTOR = 0.75f;
    private final Map<Principal, CacheEntry> identityCache;
    private final Map<Principal, Set<Principal>> domainPrincipalMap;
    private final AtomicBoolean writing = new AtomicBoolean(false);
    private final long maxAge;

    public LRURealmIdentityCache(int maxEntries) {
        this(maxEntries, -1L);
    }

    public LRURealmIdentityCache(final int maxEntries, long maxAge) {
        Assert.checkMinimumParameter("maxEntries", 1, maxEntries);
        Assert.checkMinimumParameter("maxAge", -1L, maxAge);
        this.identityCache = new LinkedHashMap<Principal, CacheEntry>(16, 0.75f, true){

            @Override
            protected boolean removeEldestEntry(Map.Entry eldest) {
                return LRURealmIdentityCache.this.identityCache.size() > maxEntries;
            }
        };
        this.domainPrincipalMap = new HashMap<Principal, Set<Principal>>(16);
        this.maxAge = maxAge;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(Principal key, RealmIdentity newValue) {
        try {
            if (this.parkForWriteAndCheckInterrupt()) {
                return;
            }
            CacheEntry entry = this.identityCache.computeIfAbsent(key, (? super K principal) -> {
                this.domainPrincipalMap.computeIfAbsent(newValue.getRealmIdentityPrincipal(), (? super K principal1) -> {
                    HashSet<Principal> principals = new HashSet<Principal>();
                    principals.add(key);
                    return principals;
                });
                return new CacheEntry(key, newValue, this.maxAge);
            });
            if (entry != null) {
                this.domainPrincipalMap.get(entry.value().getRealmIdentityPrincipal()).add(key);
            }
        }
        finally {
            this.writing.lazySet(false);
        }
    }

    @Override
    public RealmIdentity get(Principal key) {
        if (this.parkForReadAndCheckInterrupt()) {
            return null;
        }
        CacheEntry cached = this.identityCache.get(key);
        if (cached != null) {
            return this.removeIfExpired(cached);
        }
        Set<Principal> domainPrincipal = this.domainPrincipalMap.get(key);
        if (domainPrincipal != null) {
            return this.removeIfExpired(this.identityCache.get(domainPrincipal.iterator().next()));
        }
        return null;
    }

    @Override
    public void remove(Principal key) {
        try {
            if (this.parkForWriteAndCheckInterrupt()) {
                return;
            }
            if (this.identityCache.containsKey(key)) {
                this.domainPrincipalMap.remove(this.identityCache.remove(key).value().getRealmIdentityPrincipal()).forEach(this.identityCache::remove);
            } else if (this.domainPrincipalMap.containsKey(key)) {
                this.domainPrincipalMap.remove(key).forEach(this.identityCache::remove);
            }
        }
        finally {
            this.writing.lazySet(false);
        }
    }

    @Override
    public void clear() {
        try {
            this.parkForWriteAndCheckInterrupt();
            this.identityCache.clear();
            this.domainPrincipalMap.clear();
        }
        finally {
            this.writing.lazySet(false);
        }
    }

    private RealmIdentity removeIfExpired(CacheEntry cached) {
        if (cached == null) {
            return null;
        }
        if (cached.isExpired()) {
            this.remove(cached.key());
            return null;
        }
        return cached.value();
    }

    private boolean parkForWriteAndCheckInterrupt() {
        while (!this.writing.compareAndSet(false, true)) {
            LockSupport.parkNanos(1L);
            if (!Thread.interrupted()) continue;
            return true;
        }
        return false;
    }

    private boolean parkForReadAndCheckInterrupt() {
        while (this.writing.get()) {
            LockSupport.parkNanos(1L);
            if (!Thread.interrupted()) continue;
            return true;
        }
        return false;
    }

    private static final class CacheEntry {
        final Principal key;
        final RealmIdentity value;
        final long expiration;

        CacheEntry(Principal key, RealmIdentity value, long maxAge) {
            this.key = key;
            this.value = value;
            this.expiration = maxAge == -1L ? -1L : System.currentTimeMillis() + maxAge;
        }

        Principal key() {
            return this.key;
        }

        RealmIdentity value() {
            return this.value;
        }

        boolean isExpired() {
            return this.expiration != -1L ? System.currentTimeMillis() > this.expiration : false;
        }
    }
}

