/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.util;

import java.nio.charset.StandardCharsets;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.Key;
import java.security.NoSuchAlgorithmException;
import java.security.Provider;
import java.security.Security;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidParameterSpecException;
import java.text.Normalizer;
import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.PBEKeySpec;
import javax.crypto.spec.PBEParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.wildfly.common.Assert;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.util.Alphabet;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.CodePointIterator;

public final class PasswordBasedEncryptionUtil {
    private static final String DEFAULT_PBE_ALGORITHM = "PBEWithHmacSHA1andAES_128";
    private final Cipher cipher;
    private final AlgorithmParameters algorithmParameters;
    private final Alphabet alphabet;

    PasswordBasedEncryptionUtil(Cipher cipher, AlgorithmParameters algorithmParameters, Alphabet alphabet) {
        this.cipher = cipher;
        this.alphabet = alphabet;
        this.algorithmParameters = algorithmParameters;
    }

    public String encryptAndEncode(char[] payload) throws GeneralSecurityException {
        return this.encodeUsingAlphabet(this.encrypt(PasswordBasedEncryptionUtil.charArrayEncode(payload)));
    }

    public char[] decodeAndDecrypt(String encodedPayload) throws GeneralSecurityException {
        return PasswordBasedEncryptionUtil.byteArrayDecode(this.decrypt(this.decodeUsingAlphabet(encodedPayload)));
    }

    public AlgorithmParameters getAlgorithmParameters() {
        return this.algorithmParameters;
    }

    public String getEncodedIV() {
        if (this.algorithmParameters != null) {
            try {
                PBEParameterSpec spec = this.algorithmParameters.getParameterSpec(PBEParameterSpec.class);
                AlgorithmParameterSpec algSpec = spec.getParameterSpec();
                if (algSpec instanceof IvParameterSpec) {
                    return this.encodeUsingAlphabet(((IvParameterSpec)algSpec).getIV());
                }
            }
            catch (InvalidParameterSpecException e) {
                return null;
            }
        }
        return null;
    }

    private byte[] decodeUsingAlphabet(String payload) {
        ByteIterator byteIterator = PasswordBasedEncryptionUtil.isBase64(this.alphabet) ? CodePointIterator.ofString(payload).base64Decode(PasswordBasedEncryptionUtil.getAlphabet64(this.alphabet)) : CodePointIterator.ofString(payload).base32Decode(PasswordBasedEncryptionUtil.getAlphabet32(this.alphabet));
        return byteIterator.drain();
    }

    private String encodeUsingAlphabet(byte[] payload) {
        CodePointIterator codePointIteratorIterator = PasswordBasedEncryptionUtil.isBase64(this.alphabet) ? ByteIterator.ofBytes(payload).base64Encode(PasswordBasedEncryptionUtil.getAlphabet64(this.alphabet)) : ByteIterator.ofBytes(payload).base32Encode(PasswordBasedEncryptionUtil.getAlphabet32(this.alphabet));
        return codePointIteratorIterator.drainToString();
    }

    private static boolean isBase64(Alphabet alphabet) {
        return alphabet instanceof Alphabet.Base64Alphabet;
    }

    private static Alphabet.Base64Alphabet getAlphabet64(Alphabet alphabet) {
        return (Alphabet.Base64Alphabet)alphabet;
    }

    private static Alphabet.Base32Alphabet getAlphabet32(Alphabet alphabet) {
        return (Alphabet.Base32Alphabet)alphabet;
    }

    private byte[] encrypt(byte[] payload) throws GeneralSecurityException {
        return this.cipher.doFinal(payload);
    }

    private byte[] decrypt(byte[] payload) throws GeneralSecurityException {
        return this.cipher.doFinal(payload);
    }

    private static char[] byteArrayDecode(byte[] buffer) {
        return new String(buffer, StandardCharsets.UTF_8).toCharArray();
    }

    private static byte[] charArrayEncode(char[] buffer) {
        return Normalizer.normalize(new String(buffer), Normalizer.Form.NFKC).getBytes(StandardCharsets.UTF_8);
    }

    public static class Builder {
        private String keyAlgorithm;
        private String transformation;
        private String parametersAlgorithm;
        private int iteration = -1;
        private byte[] salt;
        private int keyLength = 0;
        private char[] password;
        private int cipherMode;
        private int cipherIteration = -1;
        private byte[] cipherSalt;
        private Provider provider;
        private Alphabet alphabet = Alphabet.Base64Alphabet.STANDARD;
        private IvParameterSpec ivSpec;
        private String encodedIV;
        private AlgorithmParameters algorithmParameters;

        public Builder password(char[] password) {
            this.password = password;
            return this;
        }

        public Builder password(String password) {
            this.password = password.toCharArray();
            return this;
        }

        public Builder iv(byte[] iv) {
            this.ivSpec = new IvParameterSpec(iv);
            return this;
        }

        public Builder iv(String encodedIV) {
            this.encodedIV = encodedIV;
            return this;
        }

        public Builder transformation(String transformation) {
            this.transformation = transformation;
            return this;
        }

        public Builder parametersAlgorithm(String parametersAlgorithm) {
            this.parametersAlgorithm = parametersAlgorithm;
            return this;
        }

        public Builder salt(String salt) {
            this.salt = salt.getBytes(StandardCharsets.UTF_8);
            return this;
        }

        public Builder salt(byte[] salt) {
            this.salt = salt;
            return this;
        }

        public Builder iteration(int iteration) {
            this.iteration = iteration;
            return this;
        }

        public Builder keyAlgorithm(String keyAlgorithm) {
            this.keyAlgorithm = keyAlgorithm;
            return this;
        }

        public Builder keyLength(int keyLength) {
            this.keyLength = keyLength;
            return this;
        }

        public Builder cipherIteration(int cipherIteration) {
            this.cipherIteration = cipherIteration;
            return this;
        }

        public Builder cipherSalt(byte[] cipherSalt) {
            this.cipherSalt = cipherSalt;
            return this;
        }

        public Builder cipherSalt(String cipherSalt) {
            this.cipherSalt = cipherSalt.getBytes(StandardCharsets.UTF_8);
            return this;
        }

        public Builder provider(Provider provider) {
            this.provider = provider;
            return this;
        }

        public Builder provider(String providerName) {
            Assert.checkNotNullParam("providerName", providerName);
            this.provider = Security.getProvider(providerName);
            if (this.provider == null) {
                throw ElytronMessages.log.securityProviderDoesnExist(providerName);
            }
            return this;
        }

        public Builder alphabet(Alphabet alphabet) {
            this.alphabet = alphabet;
            return this;
        }

        public Builder encryptMode() {
            this.cipherMode = 1;
            return this;
        }

        public Builder decryptMode() {
            this.cipherMode = 2;
            return this;
        }

        public Builder algorithmParameters(AlgorithmParameters algorithmParameters) {
            if (this.algorithmParameters == null) {
                this.algorithmParameters = algorithmParameters;
            }
            return this;
        }

        private Cipher createAndInitCipher(SecretKey secretKey) throws GeneralSecurityException {
            Cipher cipher;
            Cipher cipher2 = cipher = this.provider == null ? Cipher.getInstance(this.transformation) : Cipher.getInstance(this.transformation, this.provider);
            if (this.cipherMode == 1) {
                cipher.init(this.cipherMode, (Key)secretKey, Builder.generateAlgorithmParameters(this.parametersAlgorithm, this.cipherIteration, this.cipherSalt, null, this.provider));
                this.algorithmParameters = cipher.getParameters();
            } else if (this.algorithmParameters != null) {
                cipher.init(this.cipherMode, (Key)secretKey, this.algorithmParameters);
            } else {
                cipher.init(this.cipherMode, (Key)secretKey, Builder.generateAlgorithmParameters(this.parametersAlgorithm, this.cipherIteration, this.cipherSalt, this.ivSpec, this.provider));
            }
            return cipher;
        }

        private static AlgorithmParameters generateAlgorithmParameters(String algorithm, int iterationCount, byte[] salt, IvParameterSpec ivSpec, Provider provider) throws GeneralSecurityException {
            AlgorithmParameters tempParams = provider == null ? AlgorithmParameters.getInstance(algorithm) : AlgorithmParameters.getInstance(algorithm, provider);
            PBEParameterSpec pbeParameterSpec = ivSpec != null ? new PBEParameterSpec(salt, iterationCount, ivSpec) : new PBEParameterSpec(salt, iterationCount);
            tempParams.init(pbeParameterSpec);
            return tempParams;
        }

        private SecretKey deriveSecretKey() throws GeneralSecurityException {
            SecretKeyFactory secretKeyFactory;
            try {
                secretKeyFactory = this.provider != null ? SecretKeyFactory.getInstance(this.keyAlgorithm, this.provider) : SecretKeyFactory.getInstance(this.keyAlgorithm);
            }
            catch (NoSuchAlgorithmException e) {
                throw ElytronMessages.log.noSuchKeyAlgorithm(this.keyAlgorithm, e);
            }
            PBEKeySpec pbeKeySpec = this.keyLength == 0 ? new PBEKeySpec(this.password, this.salt, this.iteration) : new PBEKeySpec(this.password, this.salt, this.iteration, this.keyLength);
            SecretKey partialKey = secretKeyFactory.generateSecret(pbeKeySpec);
            return new SecretKeySpec(partialKey.getEncoded(), this.transformation);
        }

        public PasswordBasedEncryptionUtil build() throws GeneralSecurityException {
            if (this.iteration <= -1) {
                throw ElytronMessages.log.iterationCountNotSpecified();
            }
            if (this.salt == null) {
                throw ElytronMessages.log.saltNotSpecified();
            }
            if (this.password == null || this.password.length == 0) {
                throw ElytronMessages.log.initialKeyNotSpecified();
            }
            if (this.keyAlgorithm == null) {
                this.keyAlgorithm = PasswordBasedEncryptionUtil.DEFAULT_PBE_ALGORITHM;
            }
            if (this.transformation == null) {
                this.transformation = this.keyAlgorithm;
            }
            if (this.parametersAlgorithm == null) {
                this.parametersAlgorithm = this.keyAlgorithm;
            }
            if (this.cipherSalt == null) {
                this.cipherSalt = this.salt;
            }
            if (this.cipherIteration == -1) {
                this.cipherIteration = this.iteration;
            }
            if (this.ivSpec == null && this.encodedIV != null) {
                ByteIterator byteIterator = PasswordBasedEncryptionUtil.isBase64(this.alphabet) ? CodePointIterator.ofString(this.encodedIV).base64Decode(PasswordBasedEncryptionUtil.getAlphabet64(this.alphabet)) : CodePointIterator.ofString(this.encodedIV).base32Decode(PasswordBasedEncryptionUtil.getAlphabet32(this.alphabet));
                this.ivSpec = new IvParameterSpec(byteIterator.drain());
            }
            return new PasswordBasedEncryptionUtil(this.createAndInitCipher(this.deriveSecretKey()), this.algorithmParameters, this.alphabet);
        }
    }
}

