/**
 * Copyright 2013-2014 Facebook, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @providesModule shouldUpdateReactComponent
 * @typechecks static-only
 */

"use strict";

/**
 * Given a `prevComponentInstance` and `nextComponent`, determines if
 * `prevComponentInstance` should be updated as opposed to being destroyed or
 * replaced by a new instance. The second argument is a descriptor. Future
 * versions of the reconciler should only compare descriptors to other
 * descriptors.
 *
 * @param {?object} prevComponentInstance
 * @param {?object} nextDescriptor
 * @return {boolean} True if `prevComponentInstance` should be updated.
 * @protected
 */
function shouldUpdateReactComponent(prevComponentInstance, nextDescriptor) {
  // TODO: Remove warning after a release.
  if (prevComponentInstance && nextDescriptor &&
      prevComponentInstance.constructor === nextDescriptor.constructor && (
        (prevComponentInstance.props && prevComponentInstance.props.key) ===
        (nextDescriptor.props && nextDescriptor.props.key)
      )) {
    if (prevComponentInstance._owner === nextDescriptor._owner) {
      return true;
    } else {
      if (__DEV__) {
        if (prevComponentInstance.state) {
          console.warn(
            'A recent change to React has been found to impact your code. ' +
            'A mounted component will now be unmounted and replaced by a ' +
            'component (of the same class) if their owners are different. ' +
            'Previously, ownership was not considered when updating.',
            prevComponentInstance,
            nextDescriptor
          );
        }
      }
    }
  }
  return false;
}

module.exports = shouldUpdateReactComponent;
