/**
 * Copyright 2013-2014 Facebook, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * @providesModule instantiateReactComponent
 * @typechecks static-only
 */

"use strict";

var warning = require('warning');

/**
 * Validate a `componentDescriptor`. This should be exposed publicly in a follow
 * up diff.
 *
 * @param {object} descriptor
 * @return {boolean} Returns true if this is a valid descriptor of a Component.
 */
function isValidComponentDescriptor(descriptor) {
  return (
    typeof descriptor.constructor === 'function' &&
    typeof descriptor.constructor.prototype.construct === 'function' &&
    typeof descriptor.constructor.prototype.mountComponent === 'function' &&
    typeof descriptor.constructor.prototype.receiveComponent === 'function'
  );
}

/**
 * Given a `componentDescriptor` create an instance that will actually be
 * mounted. Currently it just extracts an existing clone from composite
 * components but this is an implementation detail which will change.
 *
 * @param {object} descriptor
 * @return {object} A new instance of componentDescriptor's constructor.
 * @protected
 */
function instantiateReactComponent(descriptor) {
  if (__DEV__) {
    warning(
      isValidComponentDescriptor(descriptor),
      'Only React Components are valid for mounting.'
    );
    // We use the clone of a composite component instead of the original
    // instance. This allows us to warn you if you're are accessing the wrong
    // instance.
    var instance = descriptor.__realComponentInstance || descriptor;
    instance._descriptor = descriptor;
    return instance;
  }
  // In prod we don't clone, we simply use the same instance for unaffected
  // behavior. We have to keep the descriptor around for comparison later on.
  // This should ideally be accepted in the constructor of the instance but
  // since that is currently overloaded, we just manually attach it here.
  descriptor._descriptor = descriptor;
  return descriptor;
}

module.exports = instantiateReactComponent;
