/**
 * @license
 * Copyright 2018 Google Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import MDCComponent from '@material/base/component';
import * as util from './util';
import {MDCMenuSurfaceFoundation, AnchorMargin} from './foundation';
import {MDCMenuSurfaceAdapter} from './adapter';
import {Corner, CornerBit, strings, cssClasses} from './constants';

/**
 * @extends MDCComponent<!MDCMenuSurfaceFoundation>
 */
class MDCMenuSurface extends MDCComponent {
  /** @param {...?} args */
  constructor(...args) {
    super(...args);

    /** @private {!Element} */
    this.previousFocus_;
    /** @private {!Element} */
    this.anchorElement;
    /** @private {Element} */
    this.firstFocusableElement_;
    /** @private {Element} */
    this.lastFocusableElement_;
    /** @private {!Function} */
    this.handleKeydown_;
    /** @private {!Function} */
    this.handleBodyClick_;
    /** @private {!Function} */
    this.registerBodyClickListener_;
    /** @private {!Function} */
    this.deregisterBodyClickListener_;
  }

  /**
   * @param {!Element} root
   * @return {!MDCMenuSurface}
   */
  static attachTo(root) {
    return new MDCMenuSurface(root);
  }

  initialSyncWithDOM() {
    if (this.root_.parentElement && this.root_.parentElement.classList.contains(cssClasses.ANCHOR)) {
      this.anchorElement = this.root_.parentElement;
    }

    if (this.root_.classList.contains(cssClasses.FIXED)) {
      this.setFixedPosition(true);
    }

    this.handleKeydown_ = (evt) => this.foundation_.handleKeydown(evt);
    this.handleBodyClick_ = (evt) => this.foundation_.handleBodyClick(evt);

    this.registerBodyClickListener_ = () => document.body.addEventListener('click', this.handleBodyClick_);
    this.deregisterBodyClickListener_ = () => document.body.removeEventListener('click', this.handleBodyClick_);

    this.root_.addEventListener('keydown', this.handleKeydown_);
    this.root_.addEventListener(strings.OPENED_EVENT, this.registerBodyClickListener_);
    this.root_.addEventListener(strings.CLOSED_EVENT, this.deregisterBodyClickListener_);
  }

  destroy() {
    this.root_.removeEventListener('keydown', this.handleKeydown_);
    this.root_.removeEventListener(strings.OPENED_EVENT, this.registerBodyClickListener_);
    this.root_.removeEventListener(strings.CLOSED_EVENT, this.deregisterBodyClickListener_);
    super.destroy();
  }

  /** @return {boolean} */
  get open() {
    return this.foundation_.isOpen();
  }

  /** @param {boolean} value */
  set open(value) {
    if (value) {
      const focusableElements = this.root_.querySelectorAll(strings.FOCUSABLE_ELEMENTS);
      this.firstFocusableElement_ = focusableElements.length > 0 ? focusableElements[0] : null;
      this.lastFocusableElement_ = focusableElements.length > 0 ?
        focusableElements[focusableElements.length - 1] : null;
      this.foundation_.open();
    } else {
      this.foundation_.close();
    }
  }

  /**
   * Removes the menu-surface from it's current location and appends it to the
   * body to overcome any overflow:hidden issues.
   */
  hoistMenuToBody() {
    document.body.appendChild(this.root_.parentElement.removeChild(this.root_));
    this.setIsHoisted(true);
  }

  /**
   * Sets the foundation to use page offsets for an positioning when the menu
   * is hoisted to the body.
   * @param {boolean} isHoisted
   */
  setIsHoisted(isHoisted) {
    this.foundation_.setIsHoisted(isHoisted);
  }

  /**
   * Sets the element that the menu-surface is anchored to.
   * @param {!Element} element
   */
  setMenuSurfaceAnchorElement(element) {
    this.anchorElement = element;
  }

  /**
   * Sets the menu-surface to position: fixed.
   * @param {boolean} isFixed
   */
  setFixedPosition(isFixed) {
    if (isFixed) {
      this.root_.classList.add(cssClasses.FIXED);
    } else {
      this.root_.classList.remove(cssClasses.FIXED);
    }

    this.foundation_.setFixedPosition(isFixed);
  }

  /**
   * Sets the absolute x/y position to position based on. Requires the menu to be hoisted.
   * @param {number} x
   * @param {number} y
   */
  setAbsolutePosition(x, y) {
    this.foundation_.setAbsolutePosition(x, y);
    this.setIsHoisted(true);
  }

  /**
   * @param {!Corner} corner Default anchor corner alignment of top-left
   *     surface corner.
   */
  setAnchorCorner(corner) {
    this.foundation_.setAnchorCorner(corner);
  }

  /**
   * @param {!AnchorMargin} margin
   */
  setAnchorMargin(margin) {
    this.foundation_.setAnchorMargin(margin);
  }

  /** @param {boolean} quickOpen */
  set quickOpen(quickOpen) {
    this.foundation_.setQuickOpen(quickOpen);
  }

  /** @return {!MDCMenuSurfaceFoundation} */
  getDefaultFoundation() {
    return new MDCMenuSurfaceFoundation(
      /** @type {!MDCMenuSurfaceAdapter} */ (Object.assign({
        addClass: (className) => this.root_.classList.add(className),
        removeClass: (className) => this.root_.classList.remove(className),
        hasClass: (className) => this.root_.classList.contains(className),
        hasAnchor: () => !!this.anchorElement,
        notifyClose: () => this.emit(MDCMenuSurfaceFoundation.strings.CLOSED_EVENT, {}),
        notifyOpen: () => this.emit(MDCMenuSurfaceFoundation.strings.OPENED_EVENT, {}),
        isElementInContainer: (el) => this.root_ === el || this.root_.contains(el),
        isRtl: () => getComputedStyle(this.root_).getPropertyValue('direction') === 'rtl',
        setTransformOrigin: (origin) => {
          this.root_.style[`${util.getTransformPropertyName(window)}-origin`] = origin;
        },
      },
      this.getFocusAdapterMethods_(),
      this.getDimensionAdapterMethods_())
      ));
  }

  /**
   * @return {!{
   * isFocused: function(): boolean,
   * saveFocus: function(),
   * restoreFocus: function(),
   * isFirstElementFocused: function(): boolean,
   * isLastElementFocused: function(): boolean,
   * focusFirstElement: function(),
   * focusLastElement: function(),
   * }}
   * @private
   */
  getFocusAdapterMethods_() {
    return {
      isFocused: () => document.activeElement === this.root_,
      saveFocus: () => {
        this.previousFocus_ = document.activeElement;
      },
      restoreFocus: () => {
        if (this.root_.contains(document.activeElement)) {
          if (this.previousFocus_ && this.previousFocus_.focus) {
            this.previousFocus_.focus();
          }
        }
      },
      isFirstElementFocused: () =>
        this.firstFocusableElement_ && this.firstFocusableElement_ === document.activeElement,
      isLastElementFocused: () =>
        this.lastFocusableElement_ && this.lastFocusableElement_ === document.activeElement,
      focusFirstElement: () =>
        this.firstFocusableElement_ && this.firstFocusableElement_.focus && this.firstFocusableElement_.focus(),
      focusLastElement: () =>
        this.lastFocusableElement_ && this.lastFocusableElement_.focus && this.lastFocusableElement_.focus(),
    };
  }

  /**
   * @return {!{
   * getInnerDimensions: function(),
   * getAnchorDimensions: function(): (HTMLElement | null | * | ClientRect),
   * getWindowDimensions: function(),
   * setPosition: function(*),
   * setMaxHeight: function(string)}}
   * @private
   */
  getDimensionAdapterMethods_() {
    return {
      getInnerDimensions: () => {
        return {width: this.root_.offsetWidth, height: this.root_.offsetHeight};
      },
      getAnchorDimensions: () => this.anchorElement && this.anchorElement.getBoundingClientRect(),
      getWindowDimensions: () => {
        return {width: window.innerWidth, height: window.innerHeight};
      },
      getBodyDimensions: () => {
        return {width: document.body.clientWidth, height: document.body.clientHeight};
      },
      getWindowScroll: () => {
        return {x: window.pageXOffset, y: window.pageYOffset};
      },
      setPosition: (position) => {
        this.root_.style.left = 'left' in position ? position.left : null;
        this.root_.style.right = 'right' in position ? position.right : null;
        this.root_.style.top = 'top' in position ? position.top : null;
        this.root_.style.bottom = 'bottom' in position ? position.bottom : null;
      },
      setMaxHeight: (height) => {
        this.root_.style.maxHeight = height;
      },
    };
  }
}

export {MDCMenuSurfaceFoundation, MDCMenuSurface, AnchorMargin, Corner, CornerBit, util};
