/*
 * Copyright (c) Highsoft AS. All rights reserved.
 */

import * as globals from "./globals";
/**
 * The horizontal alignment of an element.
 */
export type AlignType = ("center"|"left"|"right");
/**
 * Options for crosshairs on axes.
 */
export type AxisCrosshairOptions = (XAxisCrosshairOptions|YAxisCrosshairOptions);
export type AxisEventCallbackFunction = (this: Axis) => void;
/**
 * Options for axes.
 */
export type AxisOptions = (XAxisOptions|YAxisOptions|ZAxisOptions);
/**
 * Options for plot band labels on axes.
 */
export type AxisPlotBandsLabelOptions = (XAxisPlotBandsLabelOptions|YAxisPlotBandsLabelOptions|ZAxisPlotBandsLabelOptions);
/**
 * Options for plot bands on axes.
 */
export type AxisPlotBandsOptions = (XAxisPlotBandsOptions|YAxisPlotBandsOptions|ZAxisPlotBandsOptions);
/**
 * Options for plot line labels on axes.
 */
export type AxisPlotLinesLabelOptions = (XAxisPlotLinesLabelOptions|YAxisPlotLinesLabelOptions|ZAxisPlotLinesLabelOptions);
/**
 * Options for plot lines on axes.
 */
export type AxisPlotLinesOptions = (XAxisPlotLinesOptions|YAxisPlotLinesOptions|ZAxisPlotLinesOptions);
/**
 *
 */
export type AxisPointBreakEventCallbackFunction = (this: Axis, event: AxisPointBreakEventObject) => void;
/**
 *
 */
export type AxisSetExtremesEventCallbackFunction = (this: Axis, event: AxisSetExtremesEventObject) => void;
export type AxisTickPositionerCallbackFunction = (this: Axis) => Array<number>;
/**
 * Gets fired when a series is added to the chart after load time, using the
 * `addSeries` method. Returning `false` prevents the series from being added.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartAddSeriesCallbackFunction = (this: Chart, event: ChartAddSeriesEventObject) => void;
/**
 * Callback for chart constructors.
 *
 * @param chart
 *        Created chart.
 */
export type ChartCallbackFunction = (chart: Chart) => void;
/**
 * Gets fired when clicking on the plot background.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartClickCallbackFunction = (this: Chart, event: PointerEventObject) => void;
/**
 * Gets fired when the chart is finished loading.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartLoadCallbackFunction = (this: Chart, event: Event) => void;
/**
 * Fires when the chart is redrawn, either after a call to `chart.redraw()` or
 * after an axis, series or point is modified with the `redraw` option set to
 * `true`.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartRedrawCallbackFunction = (this: Chart, event: Event) => void;
/**
 * Gets fired after initial load of the chart (directly after the `load` event),
 * and after each redraw (directly after the `redraw` event).
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ChartRenderCallbackFunction = (this: Chart, event: Event) => void;
/**
 * Gets fired when an area of the chart has been selected. The default action
 * for the selection event is to zoom the chart to the selected area. It can be
 * prevented by calling `event.preventDefault()` or return false.
 *
 * @param this
 *        The chart on which the event occured.
 *
 * @param event
 *        Event informations
 *
 * @return Return false to prevent the default action, usually zoom.
 */
export type ChartSelectionCallbackFunction = (this: Chart, event: Event) => (boolean|undefined);
/**
 * A clipping rectangle that can be applied to one or more SVGElement instances.
 * It is instanciated with the SVGRenderer#clipRect function and applied with
 * the SVGElement#clip function.
 */
export type ClipRectElement = SVGElement;
/**
 * A valid color to be parsed and handled by Highcharts. Highcharts internally
 * supports hex colors like `#ffffff`, rgb colors like `rgb(255,255,255)` and
 * rgba colors like `rgba(255,255,255,1)`. Other colors may be supported by the
 * browsers and displayed correctly, but Highcharts is not able to process them
 * and apply concepts like opacity and brightening.
 */
export type ColorString = string;
/**
 * All possible cursor styles.
 */
export type CursorType = ("alias"|"all-scroll"|"auto"|"cell"|"col-resize"|"context-menu"|"copy"|"crosshair"|"default"|"e-resize"|"ew-resize"|"grab"|"grabbing"|"help"|"move"|"n-resize"|"ne-resize"|
"nesw-resize"|"no-drop"|"none"|"not-allowed"|"ns-resize"|"nw-resize"|"nwse-resize"|"pointer"|"progress"|"row-resize"|"s-resize"|"se-resize"|"sw-resize"|"text"|"vertical-text"|"w-resize"|"wait"|
"zoom-in"|"zoom-out");
/**
 * All possible dash styles.
 */
export type DashStyleType = ("Dash"|"DashDot"|"Dot"|"LongDash"|"LongDashDot"|"LongDashDotDot"|"ShortDash"|"ShortDashDot"|"ShortDashDotDot"|"ShortDot"|"Solid");
/**
 * Callback function to modify the CSV before parsing it by the data module.
 *
 * @param csv
 *        The CSV to modify.
 *
 * @return The CSV to parse.
 */
export type DataBeforeParseCallbackFunction = (csv: string) => string;
/**
 * Callback function that gets called after parsing data.
 *
 * @param chartOptions
 *        The chart options that were used.
 */
export type DataCompleteCallbackFunction = (chartOptions: ChartOptions) => void;
/**
 * Callback function to parse string representations of dates into JavaScript
 * timestamps (milliseconds since 1.1.1970).
 *
 * @return Timestamp (milliseconds since 1.1.1970) as integer for Date class.
 */
export type DataParseDateCallbackFunction = (dateValue: string) => number;
/**
 * Callback function to access the parsed columns, the two-dimentional input
 * data array directly, before they are interpreted into series data and
 * categories.
 *
 * @param columns
 *        The parsed columns by the data module.
 *
 * @return Return `false` to stop completion, or call `this.complete()` to
 *         continue async.
 */
export type DataParsedCallbackFunction = (columns: Array<Array<any>>) => (boolean|undefined);
/**
 * Gets fired when a drilldown point is clicked, before the new series is added.
 * Note that when clicking a category label to trigger multiple series
 * drilldown, one `drilldown` event is triggered per point in the category.
 *
 * @param this
 *        The chart where the event occurs.
 *
 * @param e
 *        The drilldown event.
 */
export type DrilldownCallbackFunction = (this: Chart, e: DrilldownEventObject) => void;
/**
 * This gets fired after all the series have been drilled up. This is especially
 * usefull in a chart with multiple drilldown series.
 *
 * @param this
 *        The chart where the event occurs.
 *
 * @param e
 *        The final drillup event.
 */
export type DrillupAllCallbackFunction = (this: Chart, e: DrillupAllEventObject) => void;
/**
 * Gets fired when drilling up from a drilldown series.
 *
 * @param this
 *        The chart where the event occurs.
 *
 * @param e
 *        The drillup event.
 */
export type DrillupCallbackFunction = (this: Chart, e: DrillupEventObject) => void;
/**
 * The function callback to execute when the event is fired. The `this` context
 * contains the instance, that fired the event.
 *
 *
 * @param eventArguments
 *        Event arguments.
 */
export type EventCallbackFunction<T> = (this: T, eventArguments?: Dictionary<any>) => void;
/**
 * Gets fired after a chart is printed through the context menu item or the
 * Chart.print method.
 *
 * @param chart
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ExportingAfterPrintCallbackFunction = (chart: Chart, event: Event) => void;
/**
 * Gets fired before a chart is printed through the context menu item or the
 * Chart.print method.
 *
 * @param chart
 *        The chart on which the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type ExportingBeforePrintCallbackFunction = (chart: Chart, event: Event) => void;
/**
 * Function to call if the offline-exporting module fails to export a chart on
 * the client side.
 *
 * @param options
 *        The exporting options.
 *
 * @param err
 *        The error from the module.
 */
export type ExportingErrorCallbackFunction = (options: ExportingOptions, err: Error) => void;
/**
 * Formats data as a string. Usually the data is accessible throught the `this`
 * keyword.
 */
export type FormatterCallbackFunction<T> = (this: T) => string;
/**
 * An object of key-value pairs for HTML attributes.
 */
export type HTMLAttributes = Dictionary<(boolean|number|string)>;
/**
 * An HTML DOM element. The type is a reference to the regular HTMLElement in
 * the global scope.
 */
export type HTMLDOMElement = HTMLElement;
/**
 * The iterator callback.
 *
 * @param value
 *        The property value.
 *
 * @param key
 *        The property key.
 *
 * @param obj
 *        The object that objectEach is being applied to.
 */
export type ObjectEachCallbackFunction = (value: any, key: string, obj: any) => void;
/**
 * Callback function to react on button clicks.
 *
 * @param e
 *        Event arguments.
 *
 * @param Return
 *        false to cancel the default button event.
 */
export type RangeSelectorClickCallbackFunction = (e: Event, Return: (boolean|undefined)) => void;
/**
 * Callback function to parse values entered in the input boxes and return a
 * valid JavaScript time as milliseconds since 1970.
 *
 * @param value
 *        Input value to parse.
 *
 * @return Parsed JavaScript time value.
 */
export type RangeSelectorParseCallbackFunction = (value: string) => number;
/**
 * If a number is given, it defines the pixel length. If a percentage string is
 * given, like for example `'50%'`, the setting defines a length relative to a
 * base size, for example the size of a container.
 */
export type RelativeSize = (number|string);
/**
 * A callback function to gain complete control on when the responsive rule
 * applies.
 *
 * @return Return `true` if it applies.
 */
export type ResponsiveCallbackFunction = () => boolean;
/**
 * Function callback when a series has been animated.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesAfterAnimateCallbackFunction = (this: Series, event: SeriesAfterAnimateEventObject) => void;
/**
 * Function callback when the checkbox next to the series' name in the legend is
 * clicked.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesCheckboxClickCallbackFunction = (this: Series, event: SeriesCheckboxClickEventObject) => void;
/**
 * Function callback when a series is clicked. Return false to cancel toogle
 * actions.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesClickCallbackFunction = (this: Series, event: SeriesClickEventObject) => void;
/**
 * Gets fired when the series is hidden after chart generation time, either by
 * clicking the legend item or by calling `.hide()`.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type SeriesHideCallbackFunction = (this: Series, event: Event) => void;
/**
 * Gets fired when the legend item belonging to the series is clicked. The
 * default action is to toggle the visibility of the series. This can be
 * prevented by returning `false` or calling `event.preventDefault()`.
 *
 * @param this
 *        The series where the event occured.
 *
 * @param event
 *        The event that occured.
 */
export type SeriesLegendItemClickCallbackFunction = (this: Series, event: SeriesLegendItemClickEventObject) => void;
/**
 * Gets fired when the mouse leaves the graph.
 *
 * @param this
 *        Series where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesMouseOutCallbackFunction = (this: Series, event: Event) => void;
/**
 * Gets fired when the mouse enters the graph.
 *
 * @param this
 *        Series where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesMouseOverCallbackFunction = (this: Series, event: Event) => void;
/**
 * The possible types of series options.
 */
export type SeriesOptionsType = (SeriesAbandsOptions|SeriesAdOptions|SeriesAoOptions|SeriesApoOptions|SeriesAreaOptions|SeriesArearangeOptions|SeriesAreasplineOptions|SeriesAreasplinerangeOptions|
SeriesAroonOptions|SeriesAroonoscillatorOptions|SeriesAtrOptions|SeriesBarOptions|SeriesBbOptions|SeriesBellcurveOptions|SeriesBoxplotOptions|SeriesBubbleOptions|SeriesBulletOptions|
SeriesCandlestickOptions|SeriesCciOptions|SeriesChaikinOptions|SeriesCmfOptions|SeriesColumnOptions|SeriesColumnpyramidOptions|SeriesColumnrangeOptions|SeriesCylinderOptions|SeriesDemaOptions|
SeriesDpoOptions|SeriesEmaOptions|SeriesErrorbarOptions|SeriesFlagsOptions|SeriesFunnelOptions|SeriesGanttOptions|SeriesGaugeOptions|SeriesHeatmapOptions|SeriesHistogramOptions|SeriesIkhOptions|
SeriesKeltnerchannelsOptions|SeriesLinearregressionangleOptions|SeriesLinearregressioninterceptOptions|SeriesLinearregressionOptions|SeriesLinearregressionslopeOptions|SeriesLineOptions|
SeriesMacdOptions|SeriesMapbubbleOptions|SeriesMaplineOptions|SeriesMapOptions|SeriesMappointOptions|SeriesMfiOptions|SeriesMomentumOptions|SeriesNatrOptions|SeriesNetworkgraphOptions|
SeriesOhlcOptions|SeriesPackedbubbleOptions|SeriesParetoOptions|SeriesPcOptions|SeriesPieOptions|SeriesPivotpointsOptions|SeriesPolygonOptions|SeriesPpoOptions|SeriesPriceenvelopesOptions|
SeriesPsarOptions|SeriesPyramidOptions|SeriesRocOptions|SeriesRsiOptions|SeriesSankeyOptions|SeriesScatter3dOptions|SeriesScatterOptions|SeriesSmaOptions|SeriesSolidgaugeOptions|SeriesSplineOptions|
SeriesStochasticOptions|SeriesStreamgraphOptions|SeriesSunburstOptions|SeriesSupertrendOptions|SeriesTemaOptions|SeriesTilemapOptions|SeriesTreemapOptions|SeriesTrixOptions|SeriesVariablepieOptions|
SeriesVariwideOptions|SeriesVbpOptions|SeriesVectorOptions|SeriesVennOptions|SeriesVwapOptions|SeriesWaterfallOptions|SeriesWilliamsrOptions|SeriesWindbarbOptions|SeriesWmaOptions|
SeriesWordcloudOptions|SeriesXrangeOptions|SeriesZigzagOptions);
/**
 * Function callback when a series point is clicked. Return false to cancel the
 * action.
 *
 * @param this
 *        The point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesPointClickCallbackFunction = (this: Point, event: SeriesPointClickEventObject) => void;
/**
 * Function callback to execute while series points are dragged. Return false to
 * stop the default drag action.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesPointDragCallbackFunction = (this: Point, event: SeriesPointDragEventObject) => void;
/**
 * Function callback to execute when a series point is dragged.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesPointDragStartCallbackFunction = (this: Point, event: SeriesPointDragStartEventObject) => void;
/**
 * Function callback to execute when series points are dropped.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event arguments.
 */
export type SeriesPointDropCallbackFunction = (this: Point, event: SeriesPointDropEventObject) => void;
/**
 * Gets fired when the mouse leaves the area close to the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesPointMouseOutCallbackFunction = (this: Point, event: Event) => void;
/**
 * Gets fired when the mouse enters the area close to the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesPointMouseOverCallbackFunction = (this: Point, event: Event) => void;
/**
 * Gets fired when the point is removed using the `.remove()` method.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesPointRemoveCallbackFunction = (this: Point, event: Event) => void;
/**
 * Gets fired when the point is selected either programmatically or following a
 * click on the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesPointSelectCallbackFunction = (this: Point, event: SeriesPointSelectEventObject) => void;
/**
 * Fires when the point is unselected either programmatically or following a
 * click on the point.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesPointUnselectCallbackFunction = (this: Point, event: SeriesPointUnselectEventObject) => void;
/**
 * Gets fired when the point is updated programmatically through the `.update()`
 * method.
 *
 * @param this
 *        Point where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesPointUpdateCallbackFunction = (this: Point, event: SeriesPointUpdateEventObject) => void;
/**
 * Gets fired when the series is shown after chart generation time, either by
 * clicking the legend item or by calling `.show()`.
 *
 * @param this
 *        Series where the event occured.
 *
 * @param event
 *        Event that occured.
 */
export type SeriesShowCallbackFunction = (this: Series, event: Event) => void;
/**
 * An SVG DOM element. The type is a reference to the regular SVGElement in the
 * global scope.
 */
export type SVGDOMElement = globals.GlobalSVGElement;
/**
 * Array of path commands, that will go into the `d` attribute of an SVG
 * element.
 */
export type SVGPathArray = Array<(number|SVGPathCommand)>;
/**
 * Possible path commands in a SVG path array.
 */
export type SVGPathCommand = ("a"|"c"|"h"|"l"|"m"|"q"|"s"|"t"|"v"|"z"|"A"|"C"|"H"|"L"|"M"|"Q"|"S"|"T"|"V"|"Z");
/**
 * An extendable collection of functions for defining symbol paths.
 */
export type SymbolDictionary = {
    [key in SymbolKey]: (() => void|undefined);
};
/**
 * Can be one of `arc`, `callout`, `circle`, `diamond`, `square`, `triangle`,
 * `triangle-down`. Symbols are used internally for point markers, button and
 * label borders and backgrounds, or custom shapes. Extendable by adding to
 * SVGRenderer#symbols.
 */
export type SymbolKey = ("arc"|"callout"|"circle"|"diamond"|"square"|"triangle"|"triangle-down");
/**
 * Function of an additional date format specifier.
 *
 * @param timestamp
 *        The time to format.
 *
 * @return The formatted portion of the date.
 */
export type TimeFormatCallbackFunction = (timestamp: number) => string;
/**
 * A callback function to place the tooltip in a specific position.
 *
 * @param labelWidth
 *        Width of the tooltip.
 *
 * @param labelHeight
 *        Height of the tooltip.
 *
 * @param point
 *        Point information for positioning a tooltip.
 *
 * @return New position for the tooltip.
 */
export type TooltipPositionerCallbackFunction = (labelWidth: number, labelHeight: number, point: TooltipPositionerPointObject) => PositionObject;
/**
 * The vertical alignment of an element.
 */
export type VerticalAlignType = ("bottom"|"middle"|"top");
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the focus border drawn
 * around elements while navigating through them.
 *
 * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.focusBorder
 * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.focusBorder
 * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.focusBorder
 * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.focusBorder
 */
export interface AccessibilityKeyboardNavigationFocusBorderOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable/disable focus border for
     * chart.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.focusBorder.enabled
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.focusBorder.enabled
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.focusBorder.enabled
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.focusBorder.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Hide the browser's default focus
     * indicator.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.focusBorder.hideBrowserFocusOutline
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.focusBorder.hideBrowserFocusOutline
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.focusBorder.hideBrowserFocusOutline
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.focusBorder.hideBrowserFocusOutline
     */
    hideBrowserFocusOutline?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Focus border margin around the
     * elements.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.focusBorder.margin
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.focusBorder.margin
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.focusBorder.margin
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.focusBorder.margin
     */
    margin?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Style options for the focus
     * border drawn around elements while navigating through them. Note that
     * some browsers in addition draw their own borders for focused elements.
     * These automatic borders can not be styled by Highcharts.
     *
     * In styled mode, the border is given the `.highcharts-focus-border` class.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.focusBorder.style
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.focusBorder.style
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.focusBorder.style
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.focusBorder.style
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for keyboard navigation.
 *
 * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation
 * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation
 * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation
 * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation
 */
export interface AccessibilityKeyboardNavigationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable keyboard navigation for
     * the chart.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.enabled
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.enabled
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.enabled
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the focus border
     * drawn around elements while navigating through them.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.focusBorder
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.focusBorder
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.focusBorder
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.focusBorder
     */
    focusBorder?: AccessibilityKeyboardNavigationFocusBorderOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set the keyboard navigation mode
     * for the chart. Can be "normal" or "serialize". In normal mode, left/right
     * arrow keys move between points in a series, while up/down arrow keys move
     * between series. Up/down navigation acts intelligently to figure out which
     * series makes sense to move to from any given point.
     *
     * In "serialize" mode, points are instead navigated as a single list.
     * Left/right behaves as in "normal" mode. Up/down arrow keys will behave
     * like left/right. This is useful for unifying navigation behavior
     * with/without screen readers enabled.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.mode
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.mode
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.mode
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.mode
     */
    mode?: ("normal"|"serialize");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Skip null points when navigating
     * through points with the keyboard.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation.skipNullPoints
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation.skipNullPoints
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation.skipNullPoints
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation.skipNullPoints
     */
    skipNullPoints?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring
 * accessibility for the chart. Requires the accessibility module to be loaded.
 * For a description of the module and information on its features, see
 * Highcharts Accessibility.
 *
 * @see https://api.highcharts.com/highcharts/accessibility
 * @see https://api.highcharts.com/highstock/accessibility
 * @see https://api.highcharts.com/highmaps/accessibility
 * @see https://api.highcharts.com/gantt/accessibility
 */
export interface AccessibilityOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether or not to add series
     * descriptions to charts with a single series.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.describeSingleSeries
     * @see https://api.highcharts.com/highstock/accessibility.describeSingleSeries
     * @see https://api.highcharts.com/highmaps/accessibility.describeSingleSeries
     * @see https://api.highcharts.com/gantt/accessibility.describeSingleSeries
     */
    describeSingleSeries?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable accessibility features
     * for the chart.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.enabled
     * @see https://api.highcharts.com/highstock/accessibility.enabled
     * @see https://api.highcharts.com/highmaps/accessibility.enabled
     * @see https://api.highcharts.com/gantt/accessibility.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for keyboard navigation.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.keyboardNavigation
     * @see https://api.highcharts.com/highstock/accessibility.keyboardNavigation
     * @see https://api.highcharts.com/highmaps/accessibility.keyboardNavigation
     * @see https://api.highcharts.com/gantt/accessibility.keyboardNavigation
     */
    keyboardNavigation?: AccessibilityKeyboardNavigationOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Function to run upon clicking
     * the "View as Data Table" link in the screen reader region.
     *
     * By default Highcharts will insert and set focus to a data table
     * representation of the chart.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.onTableAnchorClick
     * @see https://api.highcharts.com/highstock/accessibility.onTableAnchorClick
     * @see https://api.highcharts.com/highmaps/accessibility.onTableAnchorClick
     * @see https://api.highcharts.com/gantt/accessibility.onTableAnchorClick
     */
    onTableAnchorClick?: () => void;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Date format to use for points on
     * datetime axes when describing them to screen reader users.
     *
     * Defaults to the same format as in tooltip.
     *
     * For an overview of the replacement codes, see dateFormat.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.pointDateFormat
     * @see https://api.highcharts.com/highstock/accessibility.pointDateFormat
     * @see https://api.highcharts.com/highmaps/accessibility.pointDateFormat
     * @see https://api.highcharts.com/gantt/accessibility.pointDateFormat
     */
    pointDateFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter function to determine
     * the date/time format used with points on datetime axes when describing
     * them to screen reader users. Receives one argument, `point`, referring to
     * the point to describe. Should return a date format string compatible with
     * dateFormat.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.pointDateFormatter
     * @see https://api.highcharts.com/highstock/accessibility.pointDateFormatter
     * @see https://api.highcharts.com/highmaps/accessibility.pointDateFormatter
     * @see https://api.highcharts.com/gantt/accessibility.pointDateFormatter
     */
    pointDateFormatter?: () => void;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter function to use
     * instead of the default for point descriptions. Receives one argument,
     * `point`, referring to the point to describe. Should return a String with
     * the description of the point for a screen reader user.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.pointDescriptionFormatter
     * @see https://api.highcharts.com/highstock/accessibility.pointDescriptionFormatter
     * @see https://api.highcharts.com/highmaps/accessibility.pointDescriptionFormatter
     * @see https://api.highcharts.com/gantt/accessibility.pointDescriptionFormatter
     */
    pointDescriptionFormatter?: () => void;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When a series contains more
     * points than this, we no longer expose information about individual points
     * to screen readers.
     *
     * Set to `false` to disable.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.pointDescriptionThreshold
     * @see https://api.highcharts.com/highstock/accessibility.pointDescriptionThreshold
     * @see https://api.highcharts.com/highmaps/accessibility.pointDescriptionThreshold
     * @see https://api.highcharts.com/gantt/accessibility.pointDescriptionThreshold
     */
    pointDescriptionThreshold?: (false|number);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A formatter function to create
     * the HTML contents of the hidden screen reader information region.
     * Receives one argument, `chart`, referring to the chart object. Should
     * return a String with the HTML content of the region.
     *
     * The link to view the chart as a data table will be added automatically
     * after the custom HTML content.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.screenReaderSectionFormatter
     * @see https://api.highcharts.com/highstock/accessibility.screenReaderSectionFormatter
     * @see https://api.highcharts.com/highmaps/accessibility.screenReaderSectionFormatter
     * @see https://api.highcharts.com/gantt/accessibility.screenReaderSectionFormatter
     */
    screenReaderSectionFormatter?: () => void;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter function to use
     * instead of the default for series descriptions. Receives one argument,
     * `series`, referring to the series to describe. Should return a String
     * with the description of the series for a screen reader user.
     *
     * @see https://api.highcharts.com/highcharts/accessibility.seriesDescriptionFormatter
     * @see https://api.highcharts.com/highstock/accessibility.seriesDescriptionFormatter
     * @see https://api.highcharts.com/highmaps/accessibility.seriesDescriptionFormatter
     * @see https://api.highcharts.com/gantt/accessibility.seriesDescriptionFormatter
     */
    seriesDescriptionFormatter?: () => void;
}
/**
 * Options to align the element relative to the chart or another box.
 */
export interface AlignObject {
    /**
     * Horizontal alignment. Can be one of `left`, `center` and `right`.
     */
    align?: AlignType;
    /**
     * Use the `transform` attribute with translateX and translateY custom
     * attributes to align this elements rather than `x` and `y` attributes.
     */
    alignByTranslate?: boolean;
    /**
     * Vertical alignment. Can be one of `top`, `middle` and `bottom`.
     */
    verticalAlign?: VerticalAlignType;
    /**
     * Horizontal pixel offset from alignment.
     */
    x?: number;
    /**
     * Vertical pixel offset from alignment.
     */
    y?: number;
}
/**
 * An animation configuration. Animation configurations can also be defined as
 * booleans, where `false` turns off animation and `true` defaults to a duration
 * of 500ms.
 */
export interface AnimationOptionsObject {
    /**
     * A callback function to exectute when the animation finishes.
     */
    complete?: () => void;
    /**
     * The animation duration in milliseconds.
     */
    duration: number;
    /**
     * The name of an easing function as defined on the `Math` object.
     */
    easing?: string;
    /**
     * A callback function to execute on each step of each attribute or CSS
     * property that's being animated. The first argument contains information
     * about the animation and progress.
     */
    step?: () => void;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's labels.
 * Each label inherits options from the labelOptions object. An option from the
 * labelOptions can be overwritten by config for a specific label.
 *
 * @see https://api.highcharts.com/highcharts/annotations.labelOptions
 * @see https://api.highcharts.com/highstock/annotations.labelOptions
 * @see https://api.highcharts.com/highmaps/annotations.labelOptions
 * @see https://api.highcharts.com/gantt/annotations.labelOptions
 */
export interface AnnotationsLabelOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the
     * annotation's label. If right, the right side of the label should be
     * touching the point.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.align
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.align
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.align
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow the
     * annotation's labels to overlap. To make the labels less sensitive for
     * overlapping, the can be set to 0.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.allowOverlap
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.allowOverlap
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.allowOverlap
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.backgroundColor
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.backgroundColor
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.backgroundColor
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the
     * annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.borderColor
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.borderColor
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.borderColor
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the annotaiton's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.borderRadius
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.borderRadius
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.borderRadius
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the annotation's label
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.borderWidth
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.borderWidth
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.borderWidth
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for styling by CSS.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.className
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.className
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.className
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide the annotation's
     * label that is outside the plot area.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.crop
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.crop
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.crop
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.crop
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The label's pixel distance from
     * the point.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.distance
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.distance
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.distance
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.distance
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.format
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.format
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.format
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.format
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the annotation's label. Note that if a `format` or `text` are
     * defined, the format or text take precedence and the formatter is ignored.
     * `This` refers to a point object.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.formatter
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.formatter
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.formatter
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.formatter
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle the annotation's
     * label that flow outside the plot area. The justify option aligns the
     * label inside the plot area.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.overflow
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.overflow
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.overflow
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the borderWidth or
     * the backgroundColor is set, this is the padding within the box.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.padding
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.padding
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.padding
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.padding
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. The
     * shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.shadow
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.shadow
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.shadow
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.shape
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.shape
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.shape
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.shape
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the annotation's
     * label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.style
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.style
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.style
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alias for the format option.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.text
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.text
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.text
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.useHTML
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.useHTML
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.useHTML
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.verticalAlign
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.verticalAlign
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.verticalAlign
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.x
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.x
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.x
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions.y
     * @see https://api.highcharts.com/highstock/annotations.labelOptions.y
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions.y
     * @see https://api.highcharts.com/gantt/annotations.labelOptions.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of labels for the
 * annotation. For options that apply to multiple labels, they can be added to
 * the labelOptions.
 *
 * @see https://api.highcharts.com/highcharts/annotations.labels
 * @see https://api.highcharts.com/highstock/annotations.labels
 * @see https://api.highcharts.com/highmaps/annotations.labels
 * @see https://api.highcharts.com/gantt/annotations.labels
 */
export interface AnnotationsLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The alignment of the
     * annotation's label. If right, the right side of the label should be
     * touching the point.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.align
     * @see https://api.highcharts.com/highstock/annotations.labels.align
     * @see https://api.highcharts.com/highmaps/annotations.labels.align
     * @see https://api.highcharts.com/gantt/annotations.labels.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to allow the
     * annotation's labels to overlap. To make the labels less sensitive for
     * overlapping, the can be set to 0.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.allowOverlap
     * @see https://api.highcharts.com/highstock/annotations.labels.allowOverlap
     * @see https://api.highcharts.com/highmaps/annotations.labels.allowOverlap
     * @see https://api.highcharts.com/gantt/annotations.labels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.backgroundColor
     * @see https://api.highcharts.com/highstock/annotations.labels.backgroundColor
     * @see https://api.highcharts.com/highmaps/annotations.labels.backgroundColor
     * @see https://api.highcharts.com/gantt/annotations.labels.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border color for the
     * annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.borderColor
     * @see https://api.highcharts.com/highstock/annotations.labels.borderColor
     * @see https://api.highcharts.com/highmaps/annotations.labels.borderColor
     * @see https://api.highcharts.com/gantt/annotations.labels.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius in pixels for
     * the annotaiton's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.borderRadius
     * @see https://api.highcharts.com/highstock/annotations.labels.borderRadius
     * @see https://api.highcharts.com/highmaps/annotations.labels.borderRadius
     * @see https://api.highcharts.com/gantt/annotations.labels.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border width in pixels for
     * the annotation's label
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.borderWidth
     * @see https://api.highcharts.com/highstock/annotations.labels.borderWidth
     * @see https://api.highcharts.com/highmaps/annotations.labels.borderWidth
     * @see https://api.highcharts.com/gantt/annotations.labels.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A class name for styling by CSS.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.className
     * @see https://api.highcharts.com/highstock/annotations.labels.className
     * @see https://api.highcharts.com/highmaps/annotations.labels.className
     * @see https://api.highcharts.com/gantt/annotations.labels.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to hide the annotation's
     * label that is outside the plot area.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.crop
     * @see https://api.highcharts.com/highstock/annotations.labels.crop
     * @see https://api.highcharts.com/highmaps/annotations.labels.crop
     * @see https://api.highcharts.com/gantt/annotations.labels.crop
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The label's pixel distance from
     * the point.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.distance
     * @see https://api.highcharts.com/highstock/annotations.labels.distance
     * @see https://api.highcharts.com/highmaps/annotations.labels.distance
     * @see https://api.highcharts.com/gantt/annotations.labels.distance
     */
    distance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for the data
     * label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.format
     * @see https://api.highcharts.com/highstock/annotations.labels.format
     * @see https://api.highcharts.com/highmaps/annotations.labels.format
     * @see https://api.highcharts.com/gantt/annotations.labels.format
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback JavaScript function to
     * format the annotation's label. Note that if a `format` or `text` are
     * defined, the format or text take precedence and the formatter is ignored.
     * `This` refers to a point object.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.formatter
     * @see https://api.highcharts.com/highstock/annotations.labels.formatter
     * @see https://api.highcharts.com/highmaps/annotations.labels.formatter
     * @see https://api.highcharts.com/gantt/annotations.labels.formatter
     */
    formatter?: FormatterCallbackFunction<Point>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to handle the annotation's
     * label that flow outside the plot area. The justify option aligns the
     * label inside the plot area.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.overflow
     * @see https://api.highcharts.com/highstock/annotations.labels.overflow
     * @see https://api.highcharts.com/highmaps/annotations.labels.overflow
     * @see https://api.highcharts.com/gantt/annotations.labels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When either the borderWidth or
     * the backgroundColor is set, this is the padding within the box.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.padding
     * @see https://api.highcharts.com/highstock/annotations.labels.padding
     * @see https://api.highcharts.com/highmaps/annotations.labels.padding
     * @see https://api.highcharts.com/gantt/annotations.labels.padding
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
     * which the label will be connected. It can be either the point which
     * exists in the series - it is referenced by the point's id - or a new
     * point with defined x, y properies and optionally axes.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.point
     * @see https://api.highcharts.com/highstock/annotations.labels.point
     * @see https://api.highcharts.com/highmaps/annotations.labels.point
     * @see https://api.highcharts.com/gantt/annotations.labels.point
     */
    point?: (string|AnnotationsLabelsPointOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The shadow of the box. The
     * shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.shadow
     * @see https://api.highcharts.com/highstock/annotations.labels.shadow
     * @see https://api.highcharts.com/highmaps/annotations.labels.shadow
     * @see https://api.highcharts.com/gantt/annotations.labels.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The name of a symbol to use for
     * the border around the label. Symbols are predefined functions on the
     * Renderer object.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.shape
     * @see https://api.highcharts.com/highstock/annotations.labels.shape
     * @see https://api.highcharts.com/highmaps/annotations.labels.shape
     * @see https://api.highcharts.com/gantt/annotations.labels.shape
     */
    shape?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styles for the annotation's
     * label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.style
     * @see https://api.highcharts.com/highstock/annotations.labels.style
     * @see https://api.highcharts.com/highmaps/annotations.labels.style
     * @see https://api.highcharts.com/gantt/annotations.labels.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Alias for the format option.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.text
     * @see https://api.highcharts.com/highstock/annotations.labels.text
     * @see https://api.highcharts.com/highmaps/annotations.labels.text
     * @see https://api.highcharts.com/gantt/annotations.labels.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.useHTML
     * @see https://api.highcharts.com/highstock/annotations.labels.useHTML
     * @see https://api.highcharts.com/highmaps/annotations.labels.useHTML
     * @see https://api.highcharts.com/gantt/annotations.labels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * annotation's label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.verticalAlign
     * @see https://api.highcharts.com/highstock/annotations.labels.verticalAlign
     * @see https://api.highcharts.com/highmaps/annotations.labels.verticalAlign
     * @see https://api.highcharts.com/gantt/annotations.labels.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.x
     * @see https://api.highcharts.com/highstock/annotations.labels.x
     * @see https://api.highcharts.com/highmaps/annotations.labels.x
     * @see https://api.highcharts.com/gantt/annotations.labels.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position offset of the
     * label relative to the point. Note that if a `distance` is defined, the
     * distance takes precedence over `x` and `y` options.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.y
     * @see https://api.highcharts.com/highstock/annotations.labels.y
     * @see https://api.highcharts.com/highmaps/annotations.labels.y
     * @see https://api.highcharts.com/gantt/annotations.labels.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
 * which the label will be connected. It can be either the point which exists in
 * the series - it is referenced by the point's id - or a new point with defined
 * x, y properies and optionally axes.
 *
 * @see https://api.highcharts.com/highcharts/annotations.labels.point
 * @see https://api.highcharts.com/highstock/annotations.labels.point
 * @see https://api.highcharts.com/highmaps/annotations.labels.point
 * @see https://api.highcharts.com/gantt/annotations.labels.point
 */
export interface AnnotationsLabelsPointOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the point.
     * Units can be either in axis or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.point.x
     * @see https://api.highcharts.com/highstock/annotations.labels.point.x
     * @see https://api.highcharts.com/highmaps/annotations.labels.point.x
     * @see https://api.highcharts.com/gantt/annotations.labels.point.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which xAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the xAxis array. If the option is not configured or the
     * axis is not found the point's x coordinate refers to the chart pixels.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.point.xAxis
     * @see https://api.highcharts.com/highstock/annotations.labels.point.xAxis
     * @see https://api.highcharts.com/highmaps/annotations.labels.point.xAxis
     * @see https://api.highcharts.com/gantt/annotations.labels.point.xAxis
     */
    xAxis?: (number|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the point.
     * Units can be either in axis or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.point.y
     * @see https://api.highcharts.com/highstock/annotations.labels.point.y
     * @see https://api.highcharts.com/highmaps/annotations.labels.point.y
     * @see https://api.highcharts.com/gantt/annotations.labels.point.y
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which yAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the yAxis array. If the option is not configured or the
     * axis is not found the point's y coordinate refers to the chart pixels.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels.point.yAxis
     * @see https://api.highcharts.com/highstock/annotations.labels.point.yAxis
     * @see https://api.highcharts.com/highmaps/annotations.labels.point.yAxis
     * @see https://api.highcharts.com/gantt/annotations.labels.point.yAxis
     */
    yAxis?: (number|string);
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring annotations,
 * for example labels, arrows or shapes. Annotations can be tied to points, axis
 * coordinates or chart pixel coordinates.
 *
 * @see https://api.highcharts.com/highcharts/annotations
 * @see https://api.highcharts.com/highstock/annotations
 * @see https://api.highcharts.com/highmaps/annotations
 * @see https://api.highcharts.com/gantt/annotations
 */
export interface AnnotationsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Sets an ID for an annotation.
     * Can be user later when removing an annotation in
     * Chart.removeAnnotation(id) method.
     *
     * @see https://api.highcharts.com/highcharts/annotations.id
     * @see https://api.highcharts.com/highstock/annotations.id
     * @see https://api.highcharts.com/highmaps/annotations.id
     * @see https://api.highcharts.com/gantt/annotations.id
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's labels.
     * Each label inherits options from the labelOptions object. An option from
     * the labelOptions can be overwritten by config for a specific label.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labelOptions
     * @see https://api.highcharts.com/highstock/annotations.labelOptions
     * @see https://api.highcharts.com/highmaps/annotations.labelOptions
     * @see https://api.highcharts.com/gantt/annotations.labelOptions
     */
    labelOptions?: AnnotationsLabelOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of labels for the
     * annotation. For options that apply to multiple labels, they can be added
     * to the labelOptions.
     *
     * @see https://api.highcharts.com/highcharts/annotations.labels
     * @see https://api.highcharts.com/highstock/annotations.labels
     * @see https://api.highcharts.com/highmaps/annotations.labels
     * @see https://api.highcharts.com/gantt/annotations.labels
     */
    labels?: Array<AnnotationsLabelsOptions>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's shapes.
     * Each shape inherits options from the shapeOptions object. An option from
     * the shapeOptions can be overwritten by config for a specific shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions
     */
    shapeOptions?: AnnotationsShapeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of shapes for the
     * annotation. For options that apply to multiple shapes, then can be added
     * to the shapeOptions.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes
     * @see https://api.highcharts.com/highstock/annotations.shapes
     * @see https://api.highcharts.com/highmaps/annotations.shapes
     * @see https://api.highcharts.com/gantt/annotations.shapes
     */
    shapes?: Array<AnnotationsShapesOptions>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether the annotation is
     * visible.
     *
     * @see https://api.highcharts.com/highcharts/annotations.visible
     * @see https://api.highcharts.com/highstock/annotations.visible
     * @see https://api.highcharts.com/highmaps/annotations.visible
     * @see https://api.highcharts.com/gantt/annotations.visible
     */
    visible?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index of the annotation.
     *
     * @see https://api.highcharts.com/highcharts/annotations.zIndex
     * @see https://api.highcharts.com/highstock/annotations.zIndex
     * @see https://api.highcharts.com/highmaps/annotations.zIndex
     * @see https://api.highcharts.com/gantt/annotations.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for annotation's shapes.
 * Each shape inherits options from the shapeOptions object. An option from the
 * shapeOptions can be overwritten by config for a specific shape.
 *
 * @see https://api.highcharts.com/highcharts/annotations.shapeOptions
 * @see https://api.highcharts.com/highstock/annotations.shapeOptions
 * @see https://api.highcharts.com/highmaps/annotations.shapeOptions
 * @see https://api.highcharts.com/gantt/annotations.shapeOptions
 */
export interface AnnotationsShapeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's fill.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.fill
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.fill
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.fill
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.fill
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The height of the shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.height
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.height
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.height
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.height
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The radius of the shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.r
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.r
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.r
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.r
     */
    r?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's stroke.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.stroke
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.stroke
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.stroke
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.stroke
     */
    stroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.strokeWidth
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.strokeWidth
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.strokeWidth
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.strokeWidth
     */
    strokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The type of the shape, e.g.
     * circle or rectangle.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.type
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.type
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.type
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.type
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapeOptions.width
     * @see https://api.highcharts.com/highstock/annotations.shapeOptions.width
     * @see https://api.highcharts.com/highmaps/annotations.shapeOptions.width
     * @see https://api.highcharts.com/gantt/annotations.shapeOptions.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of shapes for the
 * annotation. For options that apply to multiple shapes, then can be added to
 * the shapeOptions.
 *
 * @see https://api.highcharts.com/highcharts/annotations.shapes
 * @see https://api.highcharts.com/highstock/annotations.shapes
 * @see https://api.highcharts.com/highmaps/annotations.shapes
 * @see https://api.highcharts.com/gantt/annotations.shapes
 */
export interface AnnotationsShapesOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's fill.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.fill
     * @see https://api.highcharts.com/highstock/annotations.shapes.fill
     * @see https://api.highcharts.com/highmaps/annotations.shapes.fill
     * @see https://api.highcharts.com/gantt/annotations.shapes.fill
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The height of the shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.height
     * @see https://api.highcharts.com/highstock/annotations.shapes.height
     * @see https://api.highcharts.com/highmaps/annotations.shapes.height
     * @see https://api.highcharts.com/gantt/annotations.shapes.height
     */
    height?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Id of the marker which will be
     * drawn at the final vertex of the path. Custom markers can be defined in
     * defs property.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.markerEnd
     * @see https://api.highcharts.com/highstock/annotations.shapes.markerEnd
     * @see https://api.highcharts.com/highmaps/annotations.shapes.markerEnd
     * @see https://api.highcharts.com/gantt/annotations.shapes.markerEnd
     */
    markerEnd?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Id of the marker which will be
     * drawn at the first vertex of the path. Custom markers can be defined in
     * defs property.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.markerStart
     * @see https://api.highcharts.com/highstock/annotations.shapes.markerStart
     * @see https://api.highcharts.com/highmaps/annotations.shapes.markerStart
     * @see https://api.highcharts.com/gantt/annotations.shapes.markerStart
     */
    markerStart?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
     * which the shape will be connected. It can be either the point which
     * exists in the series - it is referenced by the point's id - or a new
     * point with defined x, y properties and optionally axes.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.point
     * @see https://api.highcharts.com/highstock/annotations.shapes.point
     * @see https://api.highcharts.com/highmaps/annotations.shapes.point
     * @see https://api.highcharts.com/gantt/annotations.shapes.point
     */
    point?: (string|AnnotationsShapesPointOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array of points for the
     * shape. This option is available for shapes which can use multiple points
     * such as path. A point can be either a point object or a point's id.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.points
     * @see https://api.highcharts.com/highstock/annotations.shapes.points
     * @see https://api.highcharts.com/highmaps/annotations.shapes.points
     * @see https://api.highcharts.com/gantt/annotations.shapes.points
     */
    points?: Array<AnnotationsShapesPointsOptions>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The radius of the shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.r
     * @see https://api.highcharts.com/highstock/annotations.shapes.r
     * @see https://api.highcharts.com/highmaps/annotations.shapes.r
     * @see https://api.highcharts.com/gantt/annotations.shapes.r
     */
    r?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the shape's stroke.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.stroke
     * @see https://api.highcharts.com/highstock/annotations.shapes.stroke
     * @see https://api.highcharts.com/highmaps/annotations.shapes.stroke
     * @see https://api.highcharts.com/gantt/annotations.shapes.stroke
     */
    stroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.strokeWidth
     * @see https://api.highcharts.com/highstock/annotations.shapes.strokeWidth
     * @see https://api.highcharts.com/highmaps/annotations.shapes.strokeWidth
     * @see https://api.highcharts.com/gantt/annotations.shapes.strokeWidth
     */
    strokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The type of the shape, e.g.
     * circle or rectangle.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.type
     * @see https://api.highcharts.com/highstock/annotations.shapes.type
     * @see https://api.highcharts.com/highmaps/annotations.shapes.type
     * @see https://api.highcharts.com/gantt/annotations.shapes.type
     */
    type?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the shape.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.width
     * @see https://api.highcharts.com/highstock/annotations.shapes.width
     * @see https://api.highcharts.com/highmaps/annotations.shapes.width
     * @see https://api.highcharts.com/gantt/annotations.shapes.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) This option defines the point to
 * which the shape will be connected. It can be either the point which exists in
 * the series - it is referenced by the point's id - or a new point with defined
 * x, y properties and optionally axes.
 *
 * @see https://api.highcharts.com/highcharts/annotations.shapes.point
 * @see https://api.highcharts.com/highstock/annotations.shapes.point
 * @see https://api.highcharts.com/highmaps/annotations.shapes.point
 * @see https://api.highcharts.com/gantt/annotations.shapes.point
 */
export interface AnnotationsShapesPointOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the point.
     * Units can be either in axis or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.point.x
     * @see https://api.highcharts.com/highstock/annotations.shapes.point.x
     * @see https://api.highcharts.com/highmaps/annotations.shapes.point.x
     * @see https://api.highcharts.com/gantt/annotations.shapes.point.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which xAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the xAxis array. If the option is not configured or the
     * axis is not found the point's x coordinate refers to the chart pixels.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.point.xAxis
     * @see https://api.highcharts.com/highstock/annotations.shapes.point.xAxis
     * @see https://api.highcharts.com/highmaps/annotations.shapes.point.xAxis
     * @see https://api.highcharts.com/gantt/annotations.shapes.point.xAxis
     */
    xAxis?: (number|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the point.
     * Units can be either in axis or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.point.y
     * @see https://api.highcharts.com/highstock/annotations.shapes.point.y
     * @see https://api.highcharts.com/highmaps/annotations.shapes.point.y
     * @see https://api.highcharts.com/gantt/annotations.shapes.point.y
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which yAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the yAxis array. If the option is not configured or the
     * axis is not found the point's y coordinate refers to the chart pixels.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.point.yAxis
     * @see https://api.highcharts.com/highstock/annotations.shapes.point.yAxis
     * @see https://api.highcharts.com/highmaps/annotations.shapes.point.yAxis
     * @see https://api.highcharts.com/gantt/annotations.shapes.point.yAxis
     */
    yAxis?: (number|string);
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An array of points for the shape.
 * This option is available for shapes which can use multiple points such as
 * path. A point can be either a point object or a point's id.
 *
 * @see https://api.highcharts.com/highcharts/annotations.shapes.points
 * @see https://api.highcharts.com/highstock/annotations.shapes.points
 * @see https://api.highcharts.com/highmaps/annotations.shapes.points
 * @see https://api.highcharts.com/gantt/annotations.shapes.points
 */
export interface AnnotationsShapesPointsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x position of the point.
     * Units can be either in axis or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.points.x
     * @see https://api.highcharts.com/highstock/annotations.shapes.points.x
     * @see https://api.highcharts.com/highmaps/annotations.shapes.points.x
     * @see https://api.highcharts.com/gantt/annotations.shapes.points.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which xAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the xAxis array. If the option is not configured or the
     * axis is not found the point's x coordinate refers to the chart pixels.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.points.xAxis
     * @see https://api.highcharts.com/highstock/annotations.shapes.points.xAxis
     * @see https://api.highcharts.com/highmaps/annotations.shapes.points.xAxis
     * @see https://api.highcharts.com/gantt/annotations.shapes.points.xAxis
     */
    xAxis?: (number|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The y position of the point.
     * Units can be either in axis or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.points.y
     * @see https://api.highcharts.com/highstock/annotations.shapes.points.y
     * @see https://api.highcharts.com/highmaps/annotations.shapes.points.y
     * @see https://api.highcharts.com/gantt/annotations.shapes.points.y
     */
    y?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This number defines which yAxis
     * the point is connected to. It refers to either the axis id or the index
     * of the axis in the yAxis array. If the option is not configured or the
     * axis is not found the point's y coordinate refers to the chart pixels.
     *
     * @see https://api.highcharts.com/highcharts/annotations.shapes.points.yAxis
     * @see https://api.highcharts.com/highstock/annotations.shapes.points.yAxis
     * @see https://api.highcharts.com/highmaps/annotations.shapes.points.yAxis
     * @see https://api.highcharts.com/gantt/annotations.shapes.points.yAxis
     */
    yAxis?: (number|string);
}
export interface AxisLabelsFormatterContextObject {
    axis: Axis;
    chart: Chart;
    isFirst: boolean;
    isLast: boolean;
    value: number;
}
export interface AxisPointBreakEventObject {
    brk: Dictionary<number>;
    point: Point;
    preventDefault: () => void;
    target: SVGElement;
    type: ("pointBreak"|"pointInBreak");
}
export interface AxisSetExtremesEventObject {
    dataMax: number;
    dataMin: number;
    max: number;
    min: number;
    preventDefault: () => void;
    target: SVGElement;
    trigger: string;
    type: "setExtremes";
    userMax: number;
    userMin: number;
}
/**
 * Bounding box of an element.
 */
export interface BBoxObject {
    /**
     * Height of the bounding box.
     */
    height: number;
    /**
     * Width of the bounding box.
     */
    width: number;
    /**
     * Horizontal position of the bounding box.
     */
    x: number;
    /**
     * Vertical position of the bounding box.
     */
    y: number;
}
/**
 * (Highcharts, Highstock) Debugging options for boost. Useful for benchmarking,
 * and general timing.
 *
 * @see https://api.highcharts.com/highcharts/boost.debug
 * @see https://api.highcharts.com/highstock/boost.debug
 */
export interface BoostDebugOptions {
    /**
     * (Highcharts, Highstock) Show the number of points skipped through
     * culling.
     *
     * When set to true, the number of points skipped in series processing is
     * outputted. Points are skipped if they are closer than 1 pixel from each
     * other.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug.showSkipSummary
     * @see https://api.highcharts.com/highstock/boost.debug.showSkipSummary
     */
    showSkipSummary?: boolean;
    /**
     * (Highcharts, Highstock) Time the WebGL to SVG buffer copy
     *
     * After rendering, the result is copied to an image which is injected into
     * the SVG.
     *
     * If this property is set to true, the time it takes for the buffer copy to
     * complete is outputted.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug.timeBufferCopy
     * @see https://api.highcharts.com/highstock/boost.debug.timeBufferCopy
     */
    timeBufferCopy?: boolean;
    /**
     * (Highcharts, Highstock) Time the building of the k-d tree.
     *
     * This outputs the time spent building the k-d tree used for markers etc.
     *
     * Note that the k-d tree is built async, and runs post-rendering.
     * Following, it does not affect the performance of the rendering itself.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug.timeKDTree
     * @see https://api.highcharts.com/highstock/boost.debug.timeKDTree
     */
    timeKDTree?: boolean;
    /**
     * (Highcharts, Highstock) Time the series rendering.
     *
     * This outputs the time spent on actual rendering in the console when set
     * to true.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug.timeRendering
     * @see https://api.highcharts.com/highstock/boost.debug.timeRendering
     */
    timeRendering?: boolean;
    /**
     * (Highcharts, Highstock) Time the series processing.
     *
     * This outputs the time spent on transforming the series data to vertex
     * buffers when set to true.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug.timeSeriesProcessing
     * @see https://api.highcharts.com/highstock/boost.debug.timeSeriesProcessing
     */
    timeSeriesProcessing?: boolean;
    /**
     * (Highcharts, Highstock) Time the the WebGL setup.
     *
     * This outputs the time spent on setting up the WebGL context, creating
     * shaders, and textures.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug.timeSetup
     * @see https://api.highcharts.com/highstock/boost.debug.timeSetup
     */
    timeSetup?: boolean;
}
/**
 * (Highcharts, Highstock) Options for the Boost module. The Boost module allows
 * certain series types to be rendered by WebGL instead of the default SVG. This
 * allows hundreds of thousands of data points to be rendered in milliseconds.
 * In addition to the WebGL rendering it saves time by skipping processing and
 * inspection of the data wherever possible. This introduces some limitations to
 * what features are available in Boost mode. See the docs for details.
 *
 * In addition to the global `boost` option, each series has a boostThreshold
 * that defines when the boost should kick in.
 *
 * Requires the `modules/boost.js` module.
 *
 * @see https://api.highcharts.com/highcharts/boost
 * @see https://api.highcharts.com/highstock/boost
 */
export interface BoostOptions {
    /**
     * (Highcharts, Highstock) If set to true, the whole chart will be boosted
     * if one of the series crosses its threshold, and all the series can be
     * boosted.
     *
     * @see https://api.highcharts.com/highcharts/boost.allowForce
     * @see https://api.highcharts.com/highstock/boost.allowForce
     */
    allowForce?: boolean;
    /**
     * (Highcharts, Highstock) Debugging options for boost. Useful for
     * benchmarking, and general timing.
     *
     * @see https://api.highcharts.com/highcharts/boost.debug
     * @see https://api.highcharts.com/highstock/boost.debug
     */
    debug?: BoostDebugOptions;
    /**
     * (Highcharts, Highstock) Enable or disable boost on a chart.
     *
     * @see https://api.highcharts.com/highcharts/boost.enabled
     * @see https://api.highcharts.com/highstock/boost.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Set the series threshold for when the boost
     * should kick in globally.
     *
     * Setting to e.g. 20 will cause the whole chart to enter boost mode if
     * there are 20 or more series active. When the chart is in boost mode,
     * every series in it will be rendered to a common canvas. This offers a
     * significant speed improvment in charts with a very high amount of series.
     *
     * @see https://api.highcharts.com/highcharts/boost.seriesThreshold
     * @see https://api.highcharts.com/highstock/boost.seriesThreshold
     */
    seriesThreshold?: (number|null);
    /**
     * (Highcharts, Highstock) Enable or disable GPU translations. GPU
     * translations are faster than doing the translation in JavaScript.
     *
     * This option may cause rendering issues with certain datasets. Namely, if
     * your dataset has large numbers with small increments (such as
     * timestamps), it won't work correctly. This is due to floating point
     * precission.
     *
     * @see https://api.highcharts.com/highcharts/boost.useGPUTranslations
     * @see https://api.highcharts.com/highstock/boost.useGPUTranslations
     */
    useGPUTranslations?: boolean;
    /**
     * (Highcharts, Highstock) Enable or disable pre-allocation of vertex
     * buffers.
     *
     * Enabling this will make it so that the binary data arrays required for
     * storing the series data will be allocated prior to transforming the data
     * to a WebGL-compatible format.
     *
     * This saves a copy operation on the order of O(n) and so is significantly
     * more performant. However, this is currently an experimental option, and
     * may cause visual artifacts with some datasets.
     *
     * As such, care should be taken when using this setting to make sure that
     * it doesn't cause any rendering glitches with the given use-case.
     *
     * @see https://api.highcharts.com/highcharts/boost.usePreallocated
     * @see https://api.highcharts.com/highstock/boost.usePreallocated
     */
    usePreallocated?: boolean;
}
/**
 * (Highcharts) The back side of the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.back
 */
export interface Chart3dFrameBackOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.back.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.back.size
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.back.visible
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The bottom of the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.bottom
 */
export interface Chart3dFrameBottomOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.bottom.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.bottom.size
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.bottom.visible
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The front of the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.front
 */
export interface Chart3dFrameFrontOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.front.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.front.size
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.front.visible
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) The left side of the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.left
 */
export interface Chart3dFrameLeftOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.left.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.left.size
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.left.visible
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) Provides the option to draw a frame around the charts by
 * defining a bottom, front and back panel.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame
 */
export interface Chart3dFrameOptions {
    /**
     * (Highcharts) The back side of the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.back
     */
    back?: Chart3dFrameBackOptions;
    /**
     * (Highcharts) The bottom of the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.bottom
     */
    bottom?: Chart3dFrameBottomOptions;
    /**
     * (Highcharts) The front of the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.front
     */
    front?: Chart3dFrameFrontOptions;
    /**
     * (Highcharts) The left side of the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.left
     */
    left?: Chart3dFrameLeftOptions;
    /**
     * (Highcharts) The right of the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.right
     */
    right?: Chart3dFrameRightOptions;
    /**
     * (Highcharts) Note: As of v5.0.12, `frame.left` or `frame.right` should be
     * used instead.
     *
     * The side for the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.side
     */
    side?: Chart3dFrameSideOptions;
    /**
     * (Highcharts) General pixel thickness for the frame faces.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.size
     */
    size?: number;
    /**
     * (Highcharts) The top of the frame around a 3D chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.top
     */
    top?: Chart3dFrameTopOptions;
    /**
     * (Highcharts) Whether the frames are visible.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.visible
     */
    visible?: string;
}
/**
 * (Highcharts) The right of the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.right
 */
export interface Chart3dFrameRightOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.right.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.right.size
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.right.visible
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) Note: As of v5.0.12, `frame.left` or `frame.right` should be
 * used instead.
 *
 * The side for the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.side
 */
export interface Chart3dFrameSideOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.side.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.side.size
     */
    size?: number;
}
/**
 * (Highcharts) The top of the frame around a 3D chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d.frame.top
 */
export interface Chart3dFrameTopOptions {
    /**
     * (Highcharts) The color of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.top.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts) The thickness of the panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.top.size
     */
    size?: number;
    /**
     * (Highcharts) Whether to display the frame. Possible values are `true`,
     * `false`, `"auto"` to display only the frames behind the data, and
     * `"default"` to display faces behind the data based on the axis layout,
     * ignoring the point of view.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame.top.visible
     */
    visible?: ("auto"|"default"|boolean);
}
/**
 * (Highcharts) Options to render charts in 3 dimensions. This feature requires
 * `highcharts-3d.js`, found in the download package or online at
 * code.highcharts.com/highcharts-3d.js.
 *
 * @see https://api.highcharts.com/highcharts/chart.options3d
 */
export interface Chart3dOptions {
    /**
     * (Highcharts) One of the two rotation angles for the chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.alpha
     */
    alpha?: number;
    /**
     * (Highcharts) Set it to `"auto"` to automatically move the labels to the
     * best edge.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.axisLabelPosition
     */
    axisLabelPosition?: ("auto"|null);
    /**
     * (Highcharts) One of the two rotation angles for the chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.beta
     */
    beta?: number;
    /**
     * (Highcharts) The total depth of the chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.depth
     */
    depth?: number;
    /**
     * (Highcharts) Wether to render the chart using the 3D functionality.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts) Whether the 3d box should automatically adjust to the chart
     * plot area.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.fitToPlot
     */
    fitToPlot?: boolean;
    /**
     * (Highcharts) Provides the option to draw a frame around the charts by
     * defining a bottom, front and back panel.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.frame
     */
    frame?: Chart3dFrameOptions;
    /**
     * (Highcharts) Defines the distance the viewer is standing in front of the
     * chart, this setting is important to calculate the perspective effect in
     * column and scatter charts. It is not used for 3D pie charts.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d.viewDistance
     */
    viewDistance?: number;
}
/**
 * Conaints common event information. Through the `options` property you can
 * access the series options that were passed to the `addSeries` method.
 */
export interface ChartAddSeriesEventObject {
    /**
     * The series options that were passed to the `addSeries` method.
     */
    options: SeriesOptionsType;
}
/**
 * Contains an axes of the clicked spot.
 */
export interface ChartClickEventAxisObject {
    /**
     * Axis at the clicked spot.
     */
    axis: Axis;
    /**
     * Axis value at the clicked spot.
     */
    value: number;
}
/**
 * Contains information about the clicked spot on the chart. Remember the unit
 * of a datetime axis is milliseconds since 1970-01-01 00:00:00.
 */
export interface ChartClickEventObject extends PointerEventObject {
    /**
     * Information about the x-axis on the clicked spot.
     */
    xAxis: Array<ChartClickEventAxisObject>;
    /**
     * Information about the y-axis on the clicked spot.
     */
    yAxis: Array<ChartClickEventAxisObject>;
    /**
     * Information about the z-axis on the clicked spot.
     */
    zAxis?: Array<ChartClickEventAxisObject>;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Event listeners for the chart.
 *
 * @see https://api.highcharts.com/highcharts/chart.events
 * @see https://api.highcharts.com/highstock/chart.events
 * @see https://api.highcharts.com/highmaps/chart.events
 * @see https://api.highcharts.com/gantt/chart.events
 */
export interface ChartEventsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when a series is added to
     * the chart after load time, using the `addSeries` method. One parameter,
     * `event`, is passed to the function, containing common event information.
     * Through `event.options` you can access the series options that were
     * passed to the `addSeries` method. Returning false prevents the series
     * from being added.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.addSeries
     * @see https://api.highcharts.com/highstock/chart.events.addSeries
     * @see https://api.highcharts.com/highmaps/chart.events.addSeries
     * @see https://api.highcharts.com/gantt/chart.events.addSeries
     */
    addSeries?: ChartAddSeriesCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires after a chart is printed
     * through the context menu item or the `Chart.print` method. Requires the
     * exporting module.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.afterPrint
     * @see https://api.highcharts.com/highstock/chart.events.afterPrint
     * @see https://api.highcharts.com/highmaps/chart.events.afterPrint
     * @see https://api.highcharts.com/gantt/chart.events.afterPrint
     */
    afterPrint?: ExportingAfterPrintCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires before a chart is printed
     * through the context menu item or the `Chart.print` method. Requires the
     * exporting module.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.beforePrint
     * @see https://api.highcharts.com/highstock/chart.events.beforePrint
     * @see https://api.highcharts.com/highmaps/chart.events.beforePrint
     * @see https://api.highcharts.com/gantt/chart.events.beforePrint
     */
    beforePrint?: ExportingBeforePrintCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when clicking on the plot
     * background. One parameter, `event`, is passed to the function, containing
     * common event information.
     *
     * Information on the clicked spot can be found through `event.xAxis` and
     * `event.yAxis`, which are arrays containing the axes of each dimension and
     * each axis' value at the clicked spot. The primary axes are
     * `event.xAxis[0]` and `event.yAxis[0]`. Remember the unit of a datetime
     * axis is milliseconds since 1970-01-01 00:00:00.
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.events.click
     * @see https://api.highcharts.com/highstock/chart.events.click
     * @see https://api.highcharts.com/highmaps/chart.events.click
     * @see https://api.highcharts.com/gantt/chart.events.click
     */
    click?: ChartClickCallbackFunction;
    /**
     * (Highcharts, Highmaps) Fires when a drilldown point is clicked, before
     * the new series is added. This event is also utilized for async drilldown,
     * where the seriesOptions are not added by option, but rather loaded async.
     * Note that when clicking a category label to trigger multiple series
     * drilldown, one `drilldown` event is triggered per point in the category.
     *
     * Event arguments:
     *
     * - `category`: If a category label was clicked, which index.</dd>
     *
     * - `originalEvent`: The original browser event (usually click) that
     * triggered the drilldown.
     *
     * - `point`: The originating point.
     *
     * - `points`: If a category label was clicked, this array holds all points
     * corresponing to the category.</dd>
     *
     * - `seriesOptions`: Options for the new series.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.drilldown
     * @see https://api.highcharts.com/highmaps/chart.events.drilldown
     */
    drilldown?: DrilldownCallbackFunction;
    /**
     * (Highcharts, Highmaps) Fires when drilling up from a drilldown series.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.drillup
     * @see https://api.highcharts.com/highmaps/chart.events.drillup
     */
    drillup?: DrillupCallbackFunction;
    /**
     * (Highcharts, Highmaps) In a chart with multiple drilldown series, this
     * event fires after all the series have been drilled up.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.drillupall
     * @see https://api.highcharts.com/highmaps/chart.events.drillupall
     */
    drillupall?: DrillupAllCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when the chart is finished
     * loading. Since v4.2.2, it also waits for images to be loaded, for example
     * from point markers. One parameter, `event`, is passed to the function,
     * containing common event information.
     *
     * There is also a second parameter to the chart constructor where a
     * callback function can be passed to be executed on chart.load.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.load
     * @see https://api.highcharts.com/highstock/chart.events.load
     * @see https://api.highcharts.com/highmaps/chart.events.load
     * @see https://api.highcharts.com/gantt/chart.events.load
     */
    load?: ChartLoadCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when the chart is redrawn,
     * either after a call to `chart.redraw()` or after an axis, series or point
     * is modified with the `redraw` option set to `true`. One parameter,
     * `event`, is passed to the function, containing common event information.
     *
     * @see https://api.highcharts.com/highcharts/chart.events.redraw
     * @see https://api.highcharts.com/highstock/chart.events.redraw
     * @see https://api.highcharts.com/highmaps/chart.events.redraw
     * @see https://api.highcharts.com/gantt/chart.events.redraw
     */
    redraw?: ChartRedrawCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires after initial load of the
     * chart (directly after the `load` event), and after each redraw (directly
     * after the `redraw` event).
     *
     * @see https://api.highcharts.com/highcharts/chart.events.render
     * @see https://api.highcharts.com/highstock/chart.events.render
     * @see https://api.highcharts.com/highmaps/chart.events.render
     * @see https://api.highcharts.com/gantt/chart.events.render
     */
    render?: ChartRenderCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fires when an area of the chart
     * has been selected. Selection is enabled by setting the chart's zoomType.
     * One parameter, `event`, is passed to the function, containing common
     * event information. The default action for the selection event is to zoom
     * the chart to the selected area. It can be prevented by calling
     * `event.preventDefault()` or return false.
     *
     * Information on the selected area can be found through `event.xAxis` and
     * `event.yAxis`, which are arrays containing the axes of each dimension and
     * each axis' min and max values. The primary axes are `event.xAxis[0]` and
     * `event.yAxis[0]`. Remember the unit of a datetime axis is milliseconds
     * since 1970-01-01 00:00:00.
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.events.selection
     * @see https://api.highcharts.com/highstock/chart.events.selection
     * @see https://api.highcharts.com/highmaps/chart.events.selection
     * @see https://api.highcharts.com/gantt/chart.events.selection
     */
    selection?: ChartSelectionCallbackFunction;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) General options for the chart.
 *
 * @see https://api.highcharts.com/highcharts/chart
 * @see https://api.highcharts.com/highstock/chart
 * @see https://api.highcharts.com/highmaps/chart
 * @see https://api.highcharts.com/gantt/chart
 */
export interface ChartOptions {
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     *
     * @see https://api.highcharts.com/highcharts/chart.alignTicks
     * @see https://api.highcharts.com/highstock/chart.alignTicks
     * @see https://api.highcharts.com/gantt/chart.alignTicks
     */
    alignTicks?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set the overall animation for
     * all chart updating. Animation can be disabled throughout the chart by
     * setting it to false here. It can be overridden for each individual API
     * method as a function parameter. The only animation not affected by this
     * option is the initial series animation, see plotOptions.series.animation.
     *
     * The animation can either be set as a boolean or a configuration object.
     * If `true`, it will use the 'swing' jQuery easing and a duration of 500
     * ms. If used as a configuration object, the following properties are
     * supported:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.animation
     * @see https://api.highcharts.com/highstock/chart.animation
     * @see https://api.highcharts.com/highmaps/chart.animation
     * @see https://api.highcharts.com/gantt/chart.animation
     */
    animation?: (boolean|AnimationOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the outer chart area.
     *
     * @see https://api.highcharts.com/highcharts/chart.backgroundColor
     * @see https://api.highcharts.com/highstock/chart.backgroundColor
     * @see https://api.highcharts.com/highmaps/chart.backgroundColor
     * @see https://api.highcharts.com/gantt/chart.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the outer chart
     * border.
     *
     * @see https://api.highcharts.com/highcharts/chart.borderColor
     * @see https://api.highcharts.com/highstock/chart.borderColor
     * @see https://api.highcharts.com/highmaps/chart.borderColor
     * @see https://api.highcharts.com/gantt/chart.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The corner radius of the outer
     * chart border.
     *
     * @see https://api.highcharts.com/highcharts/chart.borderRadius
     * @see https://api.highcharts.com/highstock/chart.borderRadius
     * @see https://api.highcharts.com/highmaps/chart.borderRadius
     * @see https://api.highcharts.com/gantt/chart.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the outer
     * chart border.
     *
     * @see https://api.highcharts.com/highcharts/chart.borderWidth
     * @see https://api.highcharts.com/highstock/chart.borderWidth
     * @see https://api.highcharts.com/highmaps/chart.borderWidth
     * @see https://api.highcharts.com/gantt/chart.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A CSS class name to apply to the
     * charts container `div`, allowing unique CSS styling for each chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.className
     * @see https://api.highcharts.com/highstock/chart.className
     * @see https://api.highcharts.com/highmaps/chart.className
     * @see https://api.highcharts.com/gantt/chart.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In styled mode, this sets how
     * many colors the class names should rotate between. With ten colors,
     * series (or points) are given class names like `highcharts-color-0`,
     * `highcharts-color-0` [...] `highcharts-color-9`. The equivalent in
     * non-styled mode is to set colors using the colors setting.
     *
     * @see https://api.highcharts.com/highcharts/chart.colorCount
     * @see https://api.highcharts.com/highstock/chart.colorCount
     * @see https://api.highcharts.com/highmaps/chart.colorCount
     * @see https://api.highcharts.com/gantt/chart.colorCount
     */
    colorCount?: number;
    /**
     * (Highcharts) Alias of `type`.
     *
     * @see https://api.highcharts.com/highcharts/chart.defaultSeriesType
     */
    defaultSeriesType?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text description of the chart.
     *
     * If the Accessibility module is loaded, this is included by default as a
     * long description of the chart and its contents in the hidden screen
     * reader information region.
     *
     * @see https://api.highcharts.com/highcharts/chart.description
     * @see https://api.highcharts.com/highstock/chart.description
     * @see https://api.highcharts.com/highmaps/chart.description
     * @see https://api.highcharts.com/gantt/chart.description
     */
    description?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to display errors on the
     * chart. When `false`, the errors will be shown only in the console.
     *
     * Requires `debugger.js` module.
     *
     * @see https://api.highcharts.com/highcharts/chart.displayErrors
     * @see https://api.highcharts.com/highstock/chart.displayErrors
     * @see https://api.highcharts.com/highmaps/chart.displayErrors
     * @see https://api.highcharts.com/gantt/chart.displayErrors
     */
    displayErrors?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Event listeners for the chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.events
     * @see https://api.highcharts.com/highstock/chart.events
     * @see https://api.highcharts.com/highmaps/chart.events
     * @see https://api.highcharts.com/gantt/chart.events
     */
    events?: ChartEventsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An explicit height for the
     * chart. If a _number_, the height is given in pixels. If given a
     * _percentage string_ (for example `'56%'`), the height is given as the
     * percentage of the actual chart width. This allows for preserving the
     * aspect ratio across responsive sizes.
     *
     * By default (when `null`) the height is calculated from the offset height
     * of the containing element, or 400 pixels if the containing element's
     * height is 0.
     *
     * @see https://api.highcharts.com/highcharts/chart.height
     * @see https://api.highcharts.com/highstock/chart.height
     * @see https://api.highcharts.com/highmaps/chart.height
     * @see https://api.highcharts.com/gantt/chart.height
     */
    height?: (number|string|null);
    /**
     * (Highcharts, Highstock, Gantt) If true, the axes will scale to the
     * remaining visible series once one series is hidden. If false, hiding and
     * showing a series will not affect the axes or the other series. For
     * stacks, once one series within the stack is hidden, the rest of the stack
     * will close in around it even if the axis is not affected.
     *
     * @see https://api.highcharts.com/highcharts/chart.ignoreHiddenSeries
     * @see https://api.highcharts.com/highstock/chart.ignoreHiddenSeries
     * @see https://api.highcharts.com/gantt/chart.ignoreHiddenSeries
     */
    ignoreHiddenSeries?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to invert the axes so that the x
     * axis is vertical and y axis is horizontal. When `true`, the x axis is
     * reversed by default.
     *
     * @see https://api.highcharts.com/highcharts/chart.inverted
     * @see https://api.highcharts.com/highstock/chart.inverted
     * @see https://api.highcharts.com/gantt/chart.inverted
     */
    inverted?: boolean;
    /**
     * (Highmaps) Default `mapData` for all series. If set to a string, it
     * functions as an index into the `Highcharts.maps` array. Otherwise it is
     * interpreted s map data.
     *
     * @see https://api.highcharts.com/highmaps/chart.map
     */
    map?: (string|Array<any>);
    /**
     * (Highmaps) Set lat/lon transformation definitions for the chart. If not
     * defined, these are extracted from the map data.
     *
     * @see https://api.highcharts.com/highmaps/chart.mapTransforms
     */
    mapTransforms?: any;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the outer
     * edge of the chart and the plot area. The numbers in the array designate
     * top, right, bottom and left respectively. Use the options `marginTop`,
     * `marginRight`, `marginBottom` and `marginLeft` for shorthand setting of
     * one option.
     *
     * By default there is no margin. The actual space is dynamically calculated
     * from the offset of axis labels, axis title, title, subtitle and legend in
     * addition to the `spacingTop`, `spacingRight`, `spacingBottom` and
     * `spacingLeft` options.
     *
     * @see https://api.highcharts.com/highcharts/chart.margin
     * @see https://api.highcharts.com/highstock/chart.margin
     * @see https://api.highcharts.com/highmaps/chart.margin
     * @see https://api.highcharts.com/gantt/chart.margin
     */
    margin?: (number|Array<number>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the bottom
     * outer edge of the chart and the plot area. Use this to set a fixed pixel
     * value for the margin as opposed to the default dynamic margin. See also
     * `spacingBottom`.
     *
     * @see https://api.highcharts.com/highcharts/chart.marginBottom
     * @see https://api.highcharts.com/highstock/chart.marginBottom
     * @see https://api.highcharts.com/highmaps/chart.marginBottom
     * @see https://api.highcharts.com/gantt/chart.marginBottom
     */
    marginBottom?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the left
     * outer edge of the chart and the plot area. Use this to set a fixed pixel
     * value for the margin as opposed to the default dynamic margin. See also
     * `spacingLeft`.
     *
     * @see https://api.highcharts.com/highcharts/chart.marginLeft
     * @see https://api.highcharts.com/highstock/chart.marginLeft
     * @see https://api.highcharts.com/highmaps/chart.marginLeft
     * @see https://api.highcharts.com/gantt/chart.marginLeft
     */
    marginLeft?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the right
     * outer edge of the chart and the plot area. Use this to set a fixed pixel
     * value for the margin as opposed to the default dynamic margin. See also
     * `spacingRight`.
     *
     * @see https://api.highcharts.com/highcharts/chart.marginRight
     * @see https://api.highcharts.com/highstock/chart.marginRight
     * @see https://api.highcharts.com/highmaps/chart.marginRight
     * @see https://api.highcharts.com/gantt/chart.marginRight
     */
    marginRight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The margin between the top outer
     * edge of the chart and the plot area. Use this to set a fixed pixel value
     * for the margin as opposed to the default dynamic margin. See also
     * `spacingTop`.
     *
     * @see https://api.highcharts.com/highcharts/chart.marginTop
     * @see https://api.highcharts.com/highstock/chart.marginTop
     * @see https://api.highcharts.com/highmaps/chart.marginTop
     * @see https://api.highcharts.com/gantt/chart.marginTop
     */
    marginTop?: number;
    /**
     * (Highcharts) Options to render charts in 3 dimensions. This feature
     * requires `highcharts-3d.js`, found in the download package or online at
     * code.highcharts.com/highcharts-3d.js.
     *
     * @see https://api.highcharts.com/highcharts/chart.options3d
     */
    options3d?: Chart3dOptions;
    /**
     * (Highcharts, Gantt) Allows setting a key to switch between zooming and
     * panning. Can be one of `alt`, `ctrl`, `meta` (the command key on Mac and
     * Windows key on Windows) or `shift`. The keys are mapped directly to the
     * key properties of the click event argument (`event.altKey`,
     * `event.ctrlKey`, `event.metaKey` and `event.shiftKey`).
     *
     * @see https://api.highcharts.com/highcharts/chart.panKey
     * @see https://api.highcharts.com/gantt/chart.panKey
     */
    panKey?: ("alt"|"ctrl"|"meta"|"shift");
    /**
     * (Highcharts, Highstock, Gantt) Allow panning in a chart. Best used with
     * panKey to combine zooming and panning.
     *
     * On touch devices, when the tooltip.followTouchMove option is `true`
     * (default), panning requires two fingers. To allow panning with one
     * finger, set `followTouchMove` to `false`.
     *
     * @see https://api.highcharts.com/highcharts/chart.panning
     * @see https://api.highcharts.com/highstock/chart.panning
     * @see https://api.highcharts.com/gantt/chart.panning
     */
    panning?: boolean;
    /**
     * (Highcharts) Common options for all yAxes rendered in a parallel
     * coordinates plot. This feature requires
     * `modules/parallel-coordinates.js`.
     *
     * The default options are: (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes
     */
    parallelAxes?: ChartParallelAxesOptions;
    /**
     * (Highcharts) Flag to render charts as a parallel coordinates plot. In a
     * parallel coordinates plot (||-coords) by default all required yAxes are
     * generated and the legend is disabled. This feature requires
     * `modules/parallel-coordinates.js`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelCoordinates
     */
    parallelCoordinates?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Equivalent to zoomType, but for multitouch
     * gestures only. By default, the `pinchType` is the same as the `zoomType`
     * setting. However, pinching can be enabled separately in some cases, for
     * example in stock charts where a mouse drag pans the chart, while pinching
     * is enabled. When tooltip.followTouchMove is true, pinchType only applies
     * to two-finger touches.
     *
     * @see https://api.highcharts.com/highcharts/chart.pinchType
     * @see https://api.highcharts.com/highstock/chart.pinchType
     * @see https://api.highcharts.com/gantt/chart.pinchType
     */
    pinchType?: ("x"|"xy"|"y");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color or gradient
     * for the plot area.
     *
     * @see https://api.highcharts.com/highcharts/chart.plotBackgroundColor
     * @see https://api.highcharts.com/highstock/chart.plotBackgroundColor
     * @see https://api.highcharts.com/highmaps/chart.plotBackgroundColor
     * @see https://api.highcharts.com/gantt/chart.plotBackgroundColor
     */
    plotBackgroundColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for an image to use as
     * the plot background. To set an image as the background for the entire
     * chart, set a CSS background image to the container element. Note that for
     * the image to be applied to exported charts, its URL needs to be
     * accessible by the export server.
     *
     * @see https://api.highcharts.com/highcharts/chart.plotBackgroundImage
     * @see https://api.highcharts.com/highstock/chart.plotBackgroundImage
     * @see https://api.highcharts.com/highmaps/chart.plotBackgroundImage
     * @see https://api.highcharts.com/gantt/chart.plotBackgroundImage
     */
    plotBackgroundImage?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the inner chart or
     * plot area border.
     *
     * @see https://api.highcharts.com/highcharts/chart.plotBorderColor
     * @see https://api.highcharts.com/highstock/chart.plotBorderColor
     * @see https://api.highcharts.com/highmaps/chart.plotBorderColor
     * @see https://api.highcharts.com/gantt/chart.plotBorderColor
     */
    plotBorderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the plot area
     * border.
     *
     * @see https://api.highcharts.com/highcharts/chart.plotBorderWidth
     * @see https://api.highcharts.com/highstock/chart.plotBorderWidth
     * @see https://api.highcharts.com/highmaps/chart.plotBorderWidth
     * @see https://api.highcharts.com/gantt/chart.plotBorderWidth
     */
    plotBorderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply a drop shadow
     * to the plot area. Requires that plotBackgroundColor be set. The shadow
     * can be an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highcharts/chart.plotShadow
     * @see https://api.highcharts.com/highstock/chart.plotShadow
     * @see https://api.highcharts.com/highmaps/chart.plotShadow
     * @see https://api.highcharts.com/gantt/chart.plotShadow
     */
    plotShadow?: (boolean|CSSObject);
    /**
     * (Highcharts) When true, cartesian charts like line, spline, area and
     * column are transformed into the polar coordinate system. This produces
     * _polar charts_, also known as _radar charts_. Requires
     * `highcharts-more.js`.
     *
     * @see https://api.highcharts.com/highcharts/chart.polar
     */
    polar?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to reflow the chart to
     * fit the width of the container div on resizing the window.
     *
     * @see https://api.highcharts.com/highcharts/chart.reflow
     * @see https://api.highcharts.com/highstock/chart.reflow
     * @see https://api.highcharts.com/highmaps/chart.reflow
     * @see https://api.highcharts.com/gantt/chart.reflow
     */
    reflow?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The HTML element where the chart
     * will be rendered. If it is a string, the element by that id is used. The
     * HTML element can also be passed by direct reference, or as the first
     * argument of the chart constructor, in which case the option is not
     * needed.
     *
     * @see https://api.highcharts.com/highcharts/chart.renderTo
     * @see https://api.highcharts.com/highstock/chart.renderTo
     * @see https://api.highcharts.com/highmaps/chart.renderTo
     * @see https://api.highcharts.com/gantt/chart.renderTo
     */
    renderTo?: (string|HTMLDOMElement);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The button that appears after a
     * selection zoom, allowing the user to reset zoom.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton
     */
    resetZoomButton?: ChartResetZoomButtonOptions;
    /**
     * (Highcharts, Gantt) Options for a scrollable plot area. This feature
     * provides a minimum width for the plot area of the chart. If the width
     * gets smaller than this, typically on mobile devices, a native browser
     * scrollbar is presented below the chart. This scrollbar provides smooth
     * scrolling for the contents of the plot area, whereas the title, legend
     * and axes are fixed.
     *
     * @see https://api.highcharts.com/highcharts/chart.scrollablePlotArea
     * @see https://api.highcharts.com/gantt/chart.scrollablePlotArea
     */
    scrollablePlotArea?: ChartScrollablePlotAreaOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color of the
     * marker square when selecting (zooming in on) an area of the chart.
     *
     * @see https://api.highcharts.com/highcharts/chart.selectionMarkerFill
     * @see https://api.highcharts.com/highstock/chart.selectionMarkerFill
     * @see https://api.highcharts.com/highmaps/chart.selectionMarkerFill
     * @see https://api.highcharts.com/gantt/chart.selectionMarkerFill
     */
    selectionMarkerFill?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply a drop shadow
     * to the outer chart area. Requires that backgroundColor be set. The shadow
     * can be an object configuration containing `color`, `offsetX`, `offsetY`,
     * `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highcharts/chart.shadow
     * @see https://api.highcharts.com/highstock/chart.shadow
     * @see https://api.highcharts.com/highmaps/chart.shadow
     * @see https://api.highcharts.com/gantt/chart.shadow
     */
    shadow?: (boolean|CSSObject);
    /**
     * (Highcharts, Gantt) Whether to show the axes initially. This only applies
     * to empty charts where series are added dynamically, as axes are
     * automatically added to cartesian series.
     *
     * @see https://api.highcharts.com/highcharts/chart.showAxes
     * @see https://api.highcharts.com/gantt/chart.showAxes
     */
    showAxes?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The distance between the outer
     * edge of the chart and the content, like title or legend, or axis title
     * and labels if present. The numbers in the array designate top, right,
     * bottom and left respectively. Use the options spacingTop, spacingRight,
     * spacingBottom and spacingLeft options for shorthand setting of one
     * option.
     *
     * @see https://api.highcharts.com/highcharts/chart.spacing
     * @see https://api.highcharts.com/highstock/chart.spacing
     * @see https://api.highcharts.com/highmaps/chart.spacing
     * @see https://api.highcharts.com/gantt/chart.spacing
     */
    spacing?: Array<number>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the bottom
     * edge of the chart and the content (plot area, axis title and labels,
     * title, subtitle or legend in top position).
     *
     * @see https://api.highcharts.com/highcharts/chart.spacingBottom
     * @see https://api.highcharts.com/highstock/chart.spacingBottom
     * @see https://api.highcharts.com/highmaps/chart.spacingBottom
     * @see https://api.highcharts.com/gantt/chart.spacingBottom
     */
    spacingBottom?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the left edge
     * of the chart and the content (plot area, axis title and labels, title,
     * subtitle or legend in top position).
     *
     * @see https://api.highcharts.com/highcharts/chart.spacingLeft
     * @see https://api.highcharts.com/highstock/chart.spacingLeft
     * @see https://api.highcharts.com/highmaps/chart.spacingLeft
     * @see https://api.highcharts.com/gantt/chart.spacingLeft
     */
    spacingLeft?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the right edge
     * of the chart and the content (plot area, axis title and labels, title,
     * subtitle or legend in top position).
     *
     * @see https://api.highcharts.com/highcharts/chart.spacingRight
     * @see https://api.highcharts.com/highstock/chart.spacingRight
     * @see https://api.highcharts.com/highmaps/chart.spacingRight
     * @see https://api.highcharts.com/gantt/chart.spacingRight
     */
    spacingRight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The space between the top edge
     * of the chart and the content (plot area, axis title and labels, title,
     * subtitle or legend in top position).
     *
     * @see https://api.highcharts.com/highcharts/chart.spacingTop
     * @see https://api.highcharts.com/highstock/chart.spacingTop
     * @see https://api.highcharts.com/highmaps/chart.spacingTop
     * @see https://api.highcharts.com/gantt/chart.spacingTop
     */
    spacingTop?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Additional CSS styles to apply
     * inline to the container `div`. Note that since the default font styles
     * are applied in the renderer, it is ignorant of the individual chart
     * options and must be set globally.
     *
     * @see https://api.highcharts.com/highcharts/chart.style
     * @see https://api.highcharts.com/highstock/chart.style
     * @see https://api.highcharts.com/highmaps/chart.style
     * @see https://api.highcharts.com/gantt/chart.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply styled mode.
     * When in styled mode, no presentational attributes or CSS are applied to
     * the chart SVG. Instead, CSS rules are required to style the chart. The
     * default style sheet is available from
     * `https://code.highcharts.com/css/highcharts.css`.
     *
     * @see https://api.highcharts.com/highcharts/chart.styledMode
     * @see https://api.highcharts.com/highstock/chart.styledMode
     * @see https://api.highcharts.com/highmaps/chart.styledMode
     * @see https://api.highcharts.com/gantt/chart.styledMode
     */
    styledMode?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default series type for the
     * chart. Can be any of the chart types listed under plotOptions.
     *
     * In TypeScript instead the `type` option must always be set in the series.
     *
     * @see https://api.highcharts.com/highcharts/chart.type
     * @see https://api.highcharts.com/highstock/chart.type
     * @see https://api.highcharts.com/highmaps/chart.type
     * @see https://api.highcharts.com/gantt/chart.type
     */
    type?: ("area"|"arearange"|"areaspline"|"areasplinerange"|"bar"|"boxplot"|"bubble"|"column"|"columnrange"|"errorbar"|"funnel"|"gauge"|"heatmap"|"line"|"pie"|"polygon"|"pyramid"|"scatter"|
"solidgauge"|"spline"|"treemap"|"waterfall");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text description of the chart
     * type.
     *
     * If the Accessibility module is loaded, this will be included in the
     * description of the chart in the screen reader information region.
     *
     * Highcharts will by default attempt to guess the chart type, but for more
     * complex charts it is recommended to specify this property for clarity.
     *
     * @see https://api.highcharts.com/highcharts/chart.typeDescription
     * @see https://api.highcharts.com/highstock/chart.typeDescription
     * @see https://api.highcharts.com/highmaps/chart.typeDescription
     * @see https://api.highcharts.com/gantt/chart.typeDescription
     */
    typeDescription?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An explicit width for the chart.
     * By default (when `null`) the width is calculated from the offset width of
     * the containing element.
     *
     * @see https://api.highcharts.com/highcharts/chart.width
     * @see https://api.highcharts.com/highstock/chart.width
     * @see https://api.highcharts.com/highmaps/chart.width
     * @see https://api.highcharts.com/gantt/chart.width
     */
    width?: (number|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Set a key to hold when dragging
     * to zoom the chart. Requires the draggable-points module. This is useful
     * to avoid zooming while moving points. Should be set different than
     * chart.panKey.
     *
     * @see https://api.highcharts.com/highcharts/chart.zoomKey
     * @see https://api.highcharts.com/highstock/chart.zoomKey
     * @see https://api.highcharts.com/highmaps/chart.zoomKey
     * @see https://api.highcharts.com/gantt/chart.zoomKey
     */
    zoomKey?: ("alt"|"ctrl"|"meta"|"shift");
    /**
     * (Highcharts, Highstock, Gantt) Decides in what dimensions the user can
     * zoom by dragging the mouse. Can be one of `x`, `y` or `xy`.
     *
     * @see https://api.highcharts.com/highcharts/chart.zoomType
     * @see https://api.highcharts.com/highstock/chart.zoomType
     * @see https://api.highcharts.com/gantt/chart.zoomType
     */
    zoomType?: ("x"|"xy"|"y");
}
/**
 * (Highstock) A label on the axis next to the crosshair.
 *
 * In styled mode, the label is styled with the `.highcharts-crosshair-label`
 * class.
 *
 * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label
 */
export interface ChartParallelAxesCrosshairLabelOptions {
    /**
     * (Highstock) Alignment of the label compared to the axis. Defaults to
     * `left` for right-side axes, `right` for left-side axes and `center` for
     * horizontal axes.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.align
     */
    align?: string;
    /**
     * (Highstock) The background color for the label. Defaults to the related
     * series color, or `#666666` if that is not available.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the crosshair label
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border corner radius of the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width for the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A format string for the crosshair label. Defaults to
     * `{value}` for numeric axes and `{value:%b %d, %Y}` for datetime axes.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.format
     */
    format?: string;
    /**
     * (Highstock) Formatter function for the label text.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.formatter
     */
    formatter?: FormatterCallbackFunction<object>;
    /**
     * (Highstock) Padding inside the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.padding
     */
    padding?: number;
    /**
     * (Highstock) The shape to use for the label box.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.shape
     */
    shape?: string;
    /**
     * (Highstock) Text styles for the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label.style
     */
    style?: CSSObject;
}
/**
 * (Highcharts) Configure a crosshair that follows either the mouse pointer or
 * the hovered point.
 *
 * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
 * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
 *
 * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair
 */
export interface ChartParallelAxesCrosshairOptions {
    /**
     * (Highcharts) A class name for the crosshair, especially as a hook for
     * styling.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair.className
     */
    className?: string;
    /**
     * (Highcharts) The color of the crosshair. Defaults to `#cccccc` for
     * numeric and datetime axes, and `rgba(204,214,235,0.25)` for category
     * axes, where the crosshair by default highlights the whole category.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair.color
     */
    color?: ColorString;
    /**
     * (Highcharts) The dash style for the crosshair. See series.dashStyle for
     * possible values.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highstock) A label on the axis next to the crosshair.
     *
     * In styled mode, the label is styled with the
     * `.highcharts-crosshair-label` class.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.crosshair.label
     */
    label?: ChartParallelAxesCrosshairLabelOptions;
    /**
     * (Highcharts) Whether the crosshair should snap to the point or follow the
     * pointer independent of points.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair.snap
     */
    snap?: boolean;
    /**
     * (Highcharts) The pixel width of the crosshair. Defaults to 1 for numeric
     * or datetime axes, and for one category width for category axes.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair.width
     */
    width?: number;
    /**
     * (Highcharts) The Z index of the crosshair. Higher Z indices allow drawing
     * the crosshair on top of the series or behind the grid lines.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair.zIndex
     */
    zIndex?: number;
}
export interface ChartParallelAxesDateTimeLabelFormatsDayOptions {
    main?: string;
}
export interface ChartParallelAxesDateTimeLabelFormatsHourOptions {
    main?: string;
    range?: boolean;
}
export interface ChartParallelAxesDateTimeLabelFormatsMillisecondOptions {
    main?: string;
    range?: boolean;
}
export interface ChartParallelAxesDateTimeLabelFormatsMinuteOptions {
    main?: string;
    range?: boolean;
}
export interface ChartParallelAxesDateTimeLabelFormatsMonthOptions {
    main?: string;
}
/**
 * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
 * automatically adjust to the appropriate unit. This member gives the default
 * string representations used for each unit. For intermediate values, different
 * units may be used, for example the `day` unit can be used on midnight and
 * `hour` unit be used for intermediate values on the same axis. For an overview
 * of the replacement codes, see dateFormat. Defaults to:
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highcharts/chart.parallelAxes.dateTimeLabelFormats
 * @see https://api.highcharts.com/highstock/chart.parallelAxes.dateTimeLabelFormats
 * @see https://api.highcharts.com/gantt/chart.parallelAxes.dateTimeLabelFormats
 */
export interface ChartParallelAxesDateTimeLabelFormatsOptions {
    day?: ChartParallelAxesDateTimeLabelFormatsDayOptions;
    hour?: ChartParallelAxesDateTimeLabelFormatsHourOptions;
    millisecond?: ChartParallelAxesDateTimeLabelFormatsMillisecondOptions;
    minute?: ChartParallelAxesDateTimeLabelFormatsMinuteOptions;
    month?: ChartParallelAxesDateTimeLabelFormatsMonthOptions;
    second?: ChartParallelAxesDateTimeLabelFormatsSecondOptions;
    week?: ChartParallelAxesDateTimeLabelFormatsWeekOptions;
    year?: ChartParallelAxesDateTimeLabelFormatsYearOptions;
}
export interface ChartParallelAxesDateTimeLabelFormatsSecondOptions {
    main?: string;
    range?: boolean;
}
export interface ChartParallelAxesDateTimeLabelFormatsWeekOptions {
    main?: string;
}
export interface ChartParallelAxesDateTimeLabelFormatsYearOptions {
    main?: string;
}
/**
 * (Highcharts) Event handlers for the axis.
 *
 * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events
 */
export interface ChartParallelAxesEventsOptions {
    /**
     * (Highcharts, Gantt) An event fired after the breaks have rendered.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events.afterBreaks
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.events.afterBreaks
     */
    afterBreaks?: AxisEventCallbackFunction;
    /**
     * (Highcharts) As opposed to the `setExtremes` event, this event fires
     * after the final min and max values are computed and corrected for
     * `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events.afterSetExtremes
     */
    afterSetExtremes?: AxisEventCallbackFunction;
    /**
     * (Highcharts, Gantt) An event fired when a break from this axis occurs on
     * a point.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events.pointBreak
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.events.pointBreak
     */
    pointBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) An event fired when a point falls inside a
     * break from this axis.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events.pointInBreak
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.events.pointInBreak
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.events.pointInBreak
     */
    pointInBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts) Fires when the minimum and maximum is set for the axis,
     * either by calling the `.setExtremes()` method or by selecting an area in
     * the chart. One parameter, `event`, is passed to the function, containing
     * common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events.setExtremes
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 *
 * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid
 */
export interface ChartParallelAxesGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid.borderWidth
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid.cellHeight
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid.columns
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid.enabled
     */
    enabled?: boolean;
}
/**
 * (Gantt) Set options on specific levels in a tree grid axis. Takes precedence
 * over labels options.
 *
 * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.levels
 */
export interface ChartParallelAxesLabelsLevelsOptions {
    /**
     * (Gantt) Specify the level which the options within this object applies
     * to.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.levels.level
     */
    level?: number;
    style?: CSSObject;
}
/**
 * (Highcharts) The axis labels show the number or category for each tick.
 *
 * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels
 */
export interface ChartParallelAxesLabelsOptions {
    /**
     * (Highcharts) What part of the string the given position is anchored to.
     * Can be one of `"left"`, `"center"` or `"right"`. The exact position also
     * depends on the `labels.x` setting.
     *
     * Angular gauges and solid gauges defaults to `center`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.align
     */
    align?: ("center"|"left"|"right");
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.autoRotation
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.labels.autoRotation
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.autoRotation
     */
    autoRotation?: Array<number>;
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.autoRotationLimit
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.autoRotationLimit
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts) Angular gauges and solid gauges only. The label's pixel
     * distance from the perimeter of the plot area.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.distance
     */
    distance?: number;
    /**
     * (Highcharts) Enable or disable the axis labels.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts) A format string for the axis label.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.format
     */
    format?: string;
    /**
     * (Highcharts) Callback JavaScript function to format the label. The value
     * is given by `this.value`. Additional properties for `this` are `axis`,
     * `chart`, `isFirst` and `isLast`. The value of the default label formatter
     * can be retrieved by calling `this.axis.defaultLabelFormatter.call(this)`
     * within the function.
     *
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.formatter
     */
    formatter?: FormatterCallbackFunction<AxisLabelsFormatterContextObject>;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.indentation
     */
    indentation?: number;
    /**
     * (Gantt) Set options on specific levels in a tree grid axis. Takes
     * precedence over labels options.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.levels
     */
    levels?: Array<ChartParallelAxesLabelsLevelsOptions>;
    /**
     * (Highcharts) Horizontal axis only. When `staggerLines` is not set,
     * `maxStaggerLines` defines how many lines the axis is allowed to add to
     * automatically avoid overlapping X labels. Set to `1` to disable overlap
     * detection.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.maxStaggerLines
     */
    maxStaggerLines?: number;
    /**
     * (Highcharts) How to handle overflowing labels on horizontal axis. If set
     * to `"allow"`, it will not be aligned at all. By default it `"justify"`
     * labels inside the chart area. If there is room to move it, it will be
     * aligned to the edge, else it will be removed.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.padding
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.padding
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.position3d
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho");
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.reserveSpace
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.reserveSpace
     */
    reserveSpace?: boolean;
    /**
     * (Highcharts) Rotation of the labels in degrees.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.skew3d
     */
    skew3d?: boolean;
    /**
     * (Highcharts) Horizontal axes only. The number of lines to spread the
     * labels over to make room or tighter labels.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.staggerLines
     */
    staggerLines?: number;
    /**
     * (Highcharts) To show only every _n_'th label on the axis, set the step to
     * _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.step
     */
    step?: number;
    /**
     * (Highcharts) CSS styles for the label. Use `whiteSpace: 'nowrap'` to
     * prevent wrapping of category labels. Use `textOverflow: 'none'` to
     * prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.style
     */
    style?: CSSObject;
    /**
     * (Gantt) The symbol for the collapse and expand icon in a treegrid.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.symbol
     */
    symbol?: ChartParallelAxesLabelsSymbolOptions;
    /**
     * (Highcharts) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts) The x position offset of the label relative to the tick
     * position on the axis. Defaults to -15 for left axis, 15 for right axis.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.x
     */
    x?: number;
    /**
     * (Highcharts) The y position offset of the label relative to the tick
     * position on the axis.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.y
     */
    y?: number;
    /**
     * (Highcharts) The Z index for the axis labels.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels.zIndex
     */
    zIndex?: number;
}
/**
 * (Gantt) The symbol for the collapse and expand icon in a treegrid.
 *
 * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.symbol
 */
export interface ChartParallelAxesLabelsSymbolOptions {
    height?: number;
    padding?: number;
    /**
     * (Gantt) The symbol type. Points to a definition function in the
     * `Highcharts.Renderer.symbols` collection.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.labels.symbol.type
     */
    type?: ("arc"|"circle"|"diamond"|"square"|"triangle"|"triangle-down");
    width?: number;
    x?: number;
    y?: number;
}
/**
 * (Highcharts) Common options for all yAxes rendered in a parallel coordinates
 * plot. This feature requires `modules/parallel-coordinates.js`.
 *
 * The default options are: (see online documentation for example)
 *
 * @see https://api.highcharts.com/highcharts/chart.parallelAxes
 */
export interface ChartParallelAxesOptions {
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.alignTicks
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.alignTicks
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.alignTicks
     */
    alignTicks?: boolean;
    /**
     * (Highcharts) Whether to allow decimals in this axis' ticks. When counting
     * integers, like persons or hits on a web page, decimals should be avoided
     * in the labels.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.allowDecimals
     */
    allowDecimals?: boolean;
    /**
     * (Highcharts, Gantt) If categories are present for the xAxis, names are
     * used instead of numbers for that axis. Since Highcharts 3.0, categories
     * can also be extracted by giving each point a name and setting axis type
     * to `category`. However, if you have multiple series, best practice
     * remains defining the `categories` array.
     *
     * Example:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.categories
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.categories
     */
    categories?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.ceiling
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.ceiling
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.ceiling
     */
    ceiling?: number;
    /**
     * (Highcharts) A class name that opens for styling the axis by CSS,
     * especially in Highcharts styled mode. The class name is applied to group
     * elements for the grid, axis elements and labels.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.className
     */
    className?: string;
    /**
     * (Highcharts) Configure a crosshair that follows either the mouse pointer
     * or the hovered point.
     *
     * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
     * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.crosshair
     */
    crosshair?: (boolean|ChartParallelAxesCrosshairOptions);
    /**
     * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
     * automatically adjust to the appropriate unit. This member gives the
     * default string representations used for each unit. For intermediate
     * values, different units may be used, for example the `day` unit can be
     * used on midnight and `hour` unit be used for intermediate values on the
     * same axis. For an overview of the replacement codes, see dateFormat.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.dateTimeLabelFormats
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.dateTimeLabelFormats
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: ChartParallelAxesDateTimeLabelFormatsOptions;
    /**
     * (Highcharts) _Requires Accessibility module_
     *
     * Description of the axis to screen reader users.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.description
     */
    description?: string;
    /**
     * (Highcharts) Whether to force the axis to end on a tick. Use this option
     * with the `maxPadding` option to control the axis end.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.endOnTick
     */
    endOnTick?: boolean;
    /**
     * (Highcharts) Event handlers for the axis.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.events
     */
    events?: ChartParallelAxesEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.floor
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.floor
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.floor
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.grid
     */
    grid?: ChartParallelAxesGridOptions;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.gridZIndex
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.gridZIndex
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.gridZIndex
     */
    gridZIndex?: number;
    /**
     * (Highstock) The height of the Y axis. If it's a number, it is interpreted
     * as pixels.
     *
     * Since Highstock 2: If it's a percentage string, it is interpreted as
     * percentages of the total plot height.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.height
     */
    height?: (number|string);
    /**
     * (Highcharts) The axis labels show the number or category for each tick.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.labels
     */
    labels?: ChartParallelAxesLabelsOptions;
    /**
     * (Highcharts) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts) The width of the line marking the axis itself.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Index of another axis that this axis is
     * linked to. When an axis is linked to a master axis, it will take the same
     * extremes as the master, but as assigned by min or max or by setExtremes.
     * It can be used to show additional info, or to ease reading the chart by
     * duplicating the scales.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.linkedTo
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.linkedTo
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.linkedTo
     */
    linkedTo?: number;
    /**
     * (Highcharts) The maximum value of the axis. If `null`, the max value is
     * automatically calculated.
     *
     * If the endOnTick option is true, the `max` value might be rounded up.
     *
     * If a tickAmount is set, the axis may be extended beyond the set max in
     * order to reach the given number of ticks. The same may happen in a chart
     * with multiple axes, determined by chart. alignTicks, where a `tickAmount`
     * is applied internally.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.max
     */
    max?: number;
    /**
     * (Highstock) Maximal size of a resizable axis. Could be set as a percent
     * of plot area or pixel size.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.maxLength
     */
    maxLength?: (number|string);
    /**
     * (Highcharts, Highstock, Gantt) Padding of the max value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the highest data value to appear on
     * the edge of the plot area. When the axis' `max` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.maxPadding
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.maxPadding
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.maxPadding
     */
    maxPadding?: number;
    /**
     * (Highstock) Maximum range which can be set using the navigator's handles.
     * Opposite of xAxis.minRange.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.maxRange
     */
    maxRange?: number;
    /**
     * (Highcharts) The minimum value of the axis. If `null` the min value is
     * automatically calculated.
     *
     * If the startOnTick option is true (default), the `min` value might be
     * rounded down.
     *
     * The automatically calculated minimum value is also affected by floor,
     * softMin, minPadding, minRange as well as series.threshold and
     * series.softThreshold.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.min
     */
    min?: number;
    /**
     * (Highstock) Minimal size of a resizable axis. Could be set as a percent
     * of plot area or pixel size.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.minLength
     */
    minLength?: (number|string);
    /**
     * (Highcharts) Color for the minor tick marks.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minorTickColor
     */
    minorTickColor?: ColorString;
    /**
     * (Highcharts) Specific tick interval in axis units for the minor ticks. On
     * a linear axis, if `"auto"`, the minor tick interval is calculated as a
     * fifth of the tickInterval. If `null` or `undefined`, minor ticks are not
     * shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minorTickInterval
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highcharts) The pixel length of the minor tick marks.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minorTickLength
     */
    minorTickLength?: number;
    /**
     * (Highcharts) The position of the minor tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minorTickPosition
     */
    minorTickPosition?: ("inside"|"outside");
    /**
     * (Highcharts) Enable or disable minor ticks. Unless minorTickInterval is
     * set, the tick interval is calculated as a fifth of the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minorTicks
     */
    minorTicks?: boolean;
    /**
     * (Highcharts) The pixel width of the minor tick mark.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minorTickWidth
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the min value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the lowest data value to appear on the
     * edge of the plot area. When the axis' `min` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minPadding
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.minPadding
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.minPadding
     */
    minPadding?: number;
    /**
     * (Highcharts) The minimum range to display on this axis. The entire axis
     * will not be allowed to span over a smaller interval than this. For
     * example, for a datetime axis the main unit is milliseconds. If minRange
     * is set to 3600000, you can't zoom in more than to one hour.
     *
     * The default minRange for the x axis is five times the smallest interval
     * between any of the data points.
     *
     * On a logarithmic axis, the unit for the minimum range is the power. So a
     * minRange of 1 means that the axis can be zoomed to 10-100, 100-1000,
     * 1000-10000 etc.
     *
     * Note that the `minPadding`, `maxPadding`, `startOnTick` and `endOnTick`
     * settings also affect how the extremes of the axis are computed.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minRange
     */
    minRange?: number;
    /**
     * (Highcharts) The minimum tick interval allowed in axis values. For
     * example on zooming in on an axis with daily data, this can be used to
     * prevent the axis from showing hours. Defaults to the closest distance
     * between two points on the axis.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.minTickInterval
     */
    minTickInterval?: number;
    /**
     * (Highcharts) The distance in pixels from the plot area to the axis line.
     * A positive offset moves the axis with it's line, labels and ticks away
     * from the plot area. This is typically used when two or more axes are
     * displayed on the same side of the plot. With multiple axes the offset is
     * dynamically adjusted to avoid collision, this can be overridden by
     * setting offset explicitly.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.offset
     */
    offset?: number;
    /**
     * (Highstock, Highcharts, Gantt) Whether to display the axis on the
     * opposite side of the normal. The normal is on the left side for vertical
     * axes and bottom for horizontal, so the opposite sides will be right and
     * top respectively. This is typically used with dual or multiple axes.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.opposite
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.opposite
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.opposite
     */
    opposite?: boolean;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.pane
     */
    pane?: number;
    /**
     * (Highstock) The zoomed range to display when only defining one or none of
     * `min` or `max`. For example, to show the latest month, a range of one
     * month can be set.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.range
     */
    range?: number;
    /**
     * (Highstock) Options for axis resizing. This feature requires the
     * `drag-panes.js` - classic or styled mode - module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize
     */
    resize?: ChartParallelAxesResizeOptions;
    /**
     * (Highcharts) Whether to reverse the axis so that the highest number is
     * closest to the origin.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.reversed
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) If `true`, the first series in a stack will be
     * drawn on top in a positive, non-reversed Y axis. If `false`, the first
     * series is in the base of the stack.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.reversedStacks
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.reversedStacks
     */
    reversedStacks?: boolean;
    /**
     * (Highcharts) Whether to show the axis line and title when the axis has no
     * data.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.showEmpty
     */
    showEmpty?: boolean;
    /**
     * (Highcharts) Whether to show the first tick label.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.showFirstLabel
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.showLastLabel
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.showLastLabel
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.showLastLabel
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.softMax
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.softMax
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.softMax
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.softMin
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.softMin
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.softMin
     */
    softMin?: number;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.startOfWeek
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.startOfWeek
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.startOfWeek
     */
    startOfWeek?: number;
    /**
     * (Highcharts, Highstock, Gantt) Whether to force the axis to start on a
     * tick. Use this option with the `maxPadding` option to control the axis
     * start.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.startOnTick
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.startOnTick
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.startOnTick
     */
    startOnTick?: boolean;
    /**
     * (Gantt) For vertical axes only. Setting the static scale ensures that
     * each tick unit is translated into a fixed pixel height. For example,
     * setting the static scale to 24 results in each Y axis category taking up
     * 24 pixels, and the height of the chart adjusts. Adding or removing items
     * will make the chart resize.
     *
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.staticScale
     */
    staticScale?: number;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickAmount
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.tickAmount
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.tickAmount
     */
    tickAmount?: number;
    /**
     * (Highcharts) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickColor
     */
    tickColor?: ColorString;
    /**
     * (Highcharts) The interval of the tick marks in axis units. When
     * `undefined`, the tick interval is computed to approximately follow the
     * tickPixelInterval on linear and datetime axes. On categorized axes, a
     * `undefined` tickInterval will default to 1, one category. Note that
     * datetime axes are based on milliseconds, so for example an interval of
     * one day is expressed as `24 * 3600 * 1000`.
     *
     * On logarithmic axes, the tickInterval is based on powers, so a
     * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
     * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
     * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
     * etc.
     *
     * If the tickInterval is too dense for labels to be drawn, Highcharts may
     * remove ticks.
     *
     * If the chart has multiple axes, the alignTicks option may interfere with
     * the `tickInterval` setting.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickInterval
     */
    tickInterval?: number;
    /**
     * (Highcharts) The pixel length of the main tick marks.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickLength
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickmarkPlacement
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.tickmarkPlacement
     */
    tickmarkPlacement?: ("between"|"on");
    /**
     * (Highcharts) If tickInterval is `null` this option sets the approximate
     * pixel interval of the tick marks. Not applicable to categorized axis.
     *
     * The tick interval is also influenced by the minTickInterval option, that,
     * by default prevents ticks from being denser than the data points.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickPixelInterval
     */
    tickPixelInterval?: number;
    /**
     * (Highcharts) The position of the major tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickPosition
     */
    tickPosition?: ("inside"|"outside");
    /**
     * (Highcharts) A callback function returning array defining where the ticks
     * are laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickPositioner
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highcharts) An array defining where the ticks are laid out on the axis.
     * This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickPositions
     */
    tickPositions?: Array<number>;
    /**
     * (Highcharts, Highstock, Gantt) The pixel width of the major tick marks.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tickWidth
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.tickWidth
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.tickWidth
     */
    tickWidth?: number;
    /**
     * (Highcharts) Titles for yAxes are taken from xAxis.categories. All
     * options for `xAxis.labels` applies to parallel coordinates titles. For
     * example, to style categories, use xAxis.labels.style.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.title
     */
    title?: ChartParallelAxesTitleOptions;
    /**
     * (Highcharts) Parallel coordinates only. Format that will be used for
     * point.y and available in tooltip.pointFormat as `{point.formattedValue}`.
     * If not set, `{point.formattedValue}` will use other options, in this
     * order:
     *
     * 1. yAxis.labels.format will be used if set
     *
     * 2. If yAxis is a category, then category name will be displayed
     *
     * 3. If yAxis is a datetime, then value will use the same format as yAxis
     * labels
     *
     * 4. If yAxis is linear/logarithmic type, then simple value will be used
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.tooltipValueFormat
     */
    tooltipValueFormat?: string;
    /**
     * (Highstock) The top position of the Y axis. If it's a number, it is
     * interpreted as pixel position relative to the chart.
     *
     * Since Highstock 2: If it's a percentage string, it is interpreted as
     * percentages of the plot height, offset from plot area top.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.top
     */
    top?: (number|string);
    /**
     * (Highcharts, Gantt) The type of axis. Can be one of `linear`,
     * `logarithmic`, `datetime`, `category` or `treegrid`. Defaults to
     * `treegrid` for Gantt charts, `linear` for other chart types.
     *
     * In a datetime axis, the numbers are given in milliseconds, and tick marks
     * are placed on appropriate values, like full hours or days. In a category
     * or treegrid axis, the point names of the chart's series are used for
     * categories, if a categories array is not defined.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.type
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.type
     */
    type?: ("category"|"datetime"|"linear"|"logarithmic"|"treegrid");
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.uniqueNames
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.uniqueNames
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Datetime axis only. An array determining
     * what time intervals the ticks are allowed to fall on. Each array item is
     * an array where the first value is the time unit and the second value
     * another array of allowed multiples. Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.units
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.units
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.units
     */
    units?: Array<[string, (Array<number>|null)]>;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.visible
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.visible
     * @see https://api.highcharts.com/gantt/chart.parallelAxes.visible
     */
    visible?: boolean;
}
/**
 * (Highstock) Contains two arrays of axes that are controlled by control line
 * of the axis.
 *
 * This feature requires the `drag-panes.js` module.
 *
 * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.controlledAxis
 */
export interface ChartParallelAxesResizeControlledAxisOptions {
    /**
     * (Highstock) Array of axes that should move out of the way of resizing
     * being done for the current axis. If not set, the next axis will be used.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.controlledAxis.next
     */
    next?: Array<(string|number)>;
    /**
     * (Highstock) Array of axes that should move with the current axis while
     * resizing.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.controlledAxis.prev
     */
    prev?: Array<(string|number)>;
}
/**
 * (Highstock) Options for axis resizing. This feature requires the
 * `drag-panes.js` - classic or styled mode - module.
 *
 * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize
 */
export interface ChartParallelAxesResizeOptions {
    /**
     * (Highstock) Contains two arrays of axes that are controlled by control
     * line of the axis.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.controlledAxis
     */
    controlledAxis?: ChartParallelAxesResizeControlledAxisOptions;
    /**
     * (Highstock) Cursor style for the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.cursor
     */
    cursor?: string;
    /**
     * (Highstock) Enable or disable resize by drag for the axis.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) Color of the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) Dash style of the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.lineDashStyle
     */
    lineDashStyle?: string;
    /**
     * (Highstock) Width of the control line.
     *
     * In styled mode use class `highcharts-axis-resizer` instead.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Horizontal offset of the control line.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.x
     */
    x?: number;
    /**
     * (Highstock) Vertical offset of the control line.
     *
     * This feature requires the `drag-panes.js` module.
     *
     * @see https://api.highcharts.com/highstock/chart.parallelAxes.resize.y
     */
    y?: number;
}
/**
 * (Highcharts) Titles for yAxes are taken from xAxis.categories. All options
 * for `xAxis.labels` applies to parallel coordinates titles. For example, to
 * style categories, use xAxis.labels.style.
 *
 * @see https://api.highcharts.com/highcharts/chart.parallelAxes.title
 */
export interface ChartParallelAxesTitleOptions {
    reserveSpace?: boolean;
    text?: string;
    /**
     * (Highcharts) Alignment of the text, can be `"left"`, `"right"` or
     * `"center"`. Default alignment depends on the title.align:
     *
     * Horizontal axes:
     *
     * - for `align` = `"low"`, `textAlign` is set to `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"`, `textAlign` is set to `right`
     *
     * Vertical axes:
     *
     * - for `align` = `"low"` and `opposite` = `true`, `textAlign` is set to
     * `right`
     *
     * - for `align` = `"low"` and `opposite` = `false`, `textAlign` is set to
     * `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"` and `opposite` = `true` `textAlign` is set to
     * `left`
     *
     * - for `align` = `"high"` and `opposite` = `false` `textAlign` is set to
     * `right`
     *
     * @see https://api.highcharts.com/highcharts/chart.parallelAxes.title.textAlign
     */
    textAlign?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The button that appears after a
 * selection zoom, allowing the user to reset zoom.
 *
 * @see https://api.highcharts.com/highcharts/chart.resetZoomButton
 * @see https://api.highcharts.com/highstock/chart.resetZoomButton
 * @see https://api.highcharts.com/highmaps/chart.resetZoomButton
 * @see https://api.highcharts.com/gantt/chart.resetZoomButton
 */
export interface ChartResetZoomButtonOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The position of the button.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.position
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.position
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.position
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.position
     */
    position?: (AlignObject|ChartResetZoomButtonPositionOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) What frame the button should be
     * placed related to. Can be either `plot` or `chart`
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.relativeTo
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.relativeTo
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.relativeTo
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.relativeTo
     */
    relativeTo?: ("chart"|"plot");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of attributes for
     * the button. The object takes SVG attributes like `fill`, `stroke`,
     * `stroke-width` or `r`, the border radius. The theme also supports
     * `style`, a collection of CSS properties for the text. Equivalent
     * attributes for the hover state are given in `theme.states.hover`.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.theme
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.theme
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.theme
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.theme
     */
    theme?: ChartResetZoomButtonThemeOptions;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The position of the button.
 *
 * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.position
 * @see https://api.highcharts.com/highstock/chart.resetZoomButton.position
 * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.position
 * @see https://api.highcharts.com/gantt/chart.resetZoomButton.position
 */
export interface ChartResetZoomButtonPositionOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal alignment of the
     * button.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.position.align
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.position.align
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.position.align
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.position.align
     */
    align?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * button.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.position.verticalAlign
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.position.verticalAlign
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.position.verticalAlign
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.position.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal offset of the
     * button.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.position.x
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.position.x
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.position.x
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.position.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * button.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.position.y
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.position.y
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.position.y
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.position.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A collection of attributes for the
 * button. The object takes SVG attributes like `fill`, `stroke`, `stroke-width`
 * or `r`, the border radius. The theme also supports `style`, a collection of
 * CSS properties for the text. Equivalent attributes for the hover state are
 * given in `theme.states.hover`.
 *
 * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.theme
 * @see https://api.highcharts.com/highstock/chart.resetZoomButton.theme
 * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.theme
 * @see https://api.highcharts.com/gantt/chart.resetZoomButton.theme
 */
export interface ChartResetZoomButtonThemeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Z index for the reset zoom
     * button. The default value places it below the tooltip that has Z index 7.
     *
     * @see https://api.highcharts.com/highcharts/chart.resetZoomButton.theme.zIndex
     * @see https://api.highcharts.com/highstock/chart.resetZoomButton.theme.zIndex
     * @see https://api.highcharts.com/highmaps/chart.resetZoomButton.theme.zIndex
     * @see https://api.highcharts.com/gantt/chart.resetZoomButton.theme.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts, Gantt) Options for a scrollable plot area. This feature provides
 * a minimum width for the plot area of the chart. If the width gets smaller
 * than this, typically on mobile devices, a native browser scrollbar is
 * presented below the chart. This scrollbar provides smooth scrolling for the
 * contents of the plot area, whereas the title, legend and axes are fixed.
 *
 * @see https://api.highcharts.com/highcharts/chart.scrollablePlotArea
 * @see https://api.highcharts.com/gantt/chart.scrollablePlotArea
 */
export interface ChartScrollablePlotAreaOptions {
    /**
     * (Highcharts, Gantt) The minimum width for the plot area. If it gets
     * smaller than this, the plot area will become scrollable.
     *
     * @see https://api.highcharts.com/highcharts/chart.scrollablePlotArea.minWidth
     * @see https://api.highcharts.com/gantt/chart.scrollablePlotArea.minWidth
     */
    minWidth?: number;
    /**
     * (Highcharts, Gantt) The initial scrolling position of the scrollable plot
     * area. Ranges from 0 to 1, where 0 aligns the plot area to the left and 1
     * aligns it to the right. Typically we would use 1 if the chart has right
     * aligned Y axes.
     *
     * @see https://api.highcharts.com/highcharts/chart.scrollablePlotArea.scrollPositionX
     * @see https://api.highcharts.com/gantt/chart.scrollablePlotArea.scrollPositionX
     */
    scrollPositionX?: number;
}
/**
 * (Gantt) Text labels for the plot bands
 *
 * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label
 */
export interface ColorAxisCurrentDateIndicatorLabelOptions {
    /**
     * (Gantt) Horizontal alignment of the label. Can be one of "left", "center"
     * or "right".
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.align
     */
    align?: AlignType;
    /**
     * (Gantt) Rotation of the text label in degrees. Defaults to 0 for
     * horizontal plot lines and 90 for vertical lines.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.rotation
     */
    rotation?: number;
    /**
     * (Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.style
     */
    style?: CSSObject;
    /**
     * (Gantt) The text itself. A subset of HTML is supported.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.text
     */
    text?: string;
    /**
     * (Gantt) The text alignment for the label. While `align` determines where
     * the texts anchor point is placed within the plot band, `textAlign`
     * determines how the text is aligned against its anchor point. Possible
     * values are "left", "center" and "right". Defaults to the same as the
     * `align` option.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Gantt) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.useHTML
     */
    useHTML?: boolean;
    /**
     * (Gantt) Vertical alignment of the label relative to the plot line. Can be
     * one of "top", "middle" or "bottom".
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.verticalAlign
     */
    verticalAlign?: ("bottom"|"middle"|"top");
    /**
     * (Gantt) Horizontal position relative the alignment. Default varies by
     * orientation.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.x
     */
    x?: number;
    /**
     * (Gantt) Vertical position of the text baseline relative to the alignment.
     * Default varies by orientation.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label.y
     */
    y?: number;
}
/**
 * (Gantt) Show an indicator on the axis for the current date and time. Can be a
 * boolean or a configuration object similar to xAxis.plotLines.
 *
 * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator
 */
export interface ColorAxisCurrentDateIndicatorOptions {
    /**
     * (Gantt) A custom class name, in addition to the default
     * `highcharts-plot-line`, to apply to each individual line.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.className
     */
    className?: string;
    /**
     * (Gantt) The color of the line.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.color
     */
    color?: ColorString;
    /**
     * (Gantt) The dashing or dot style for the plot line. For possible values
     * see this overview.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Gantt) An object defining mouse events for the plot line. Supported
     * properties are `click`, `mouseover`, `mouseout`, `mousemove`.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.events
     */
    events?: any;
    /**
     * (Gantt) An id used for identifying the plot line in Axis.removePlotLine.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.id
     */
    id?: string;
    /**
     * (Gantt) Text labels for the plot bands
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.label
     */
    label?: ColorAxisCurrentDateIndicatorLabelOptions;
    /**
     * (Gantt) The width or thickness of the plot line.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.width
     */
    width?: number;
    /**
     * (Gantt) The z index of the plot line within the chart.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highmaps) An array of data classes or ranges for the choropleth
 * map. If none given, the color axis is scalar and values are distributed as a
 * gradient between the minimum and maximum colors.
 *
 * @see https://api.highcharts.com/highcharts/colorAxis.dataClasses
 * @see https://api.highcharts.com/highmaps/colorAxis.dataClasses
 */
export interface ColorAxisDataClassesOptions {
    /**
     * (Highcharts, Highmaps) The color of each data class. If not set, the
     * color is pulled from the global or chart-specific colors array. In styled
     * mode, this option is ignored. Instead, use colors defined in CSS.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.dataClasses.color
     * @see https://api.highcharts.com/highmaps/colorAxis.dataClasses.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highmaps) The start of the value range that the data class
     * represents, relating to the point value.
     *
     * The range of each `dataClass` is closed in both ends, but can be
     * overridden by the next `dataClass`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.dataClasses.from
     * @see https://api.highcharts.com/highmaps/colorAxis.dataClasses.from
     */
    from?: number;
    /**
     * (Highcharts, Highmaps) The name of the data class as it appears in the
     * legend. If no name is given, it is automatically created based on the
     * `from` and `to` values. For full programmatic control,
     * legend.labelFormatter can be used. In the formatter, `this.from` and
     * `this.to` can be accessed.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.dataClasses.name
     * @see https://api.highcharts.com/highmaps/colorAxis.dataClasses.name
     */
    name?: string;
    /**
     * (Highcharts, Highmaps) The end of the value range that the data class
     * represents, relating to the point value.
     *
     * The range of each `dataClass` is closed in both ends, but can be
     * overridden by the next `dataClass`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.dataClasses.to
     * @see https://api.highcharts.com/highmaps/colorAxis.dataClasses.to
     */
    to?: number;
}
/**
 * (Highcharts, Highmaps) Event handlers for the axis.
 *
 * @see https://api.highcharts.com/highcharts/colorAxis.events
 * @see https://api.highcharts.com/highmaps/colorAxis.events
 */
export interface ColorAxisEventsOptions {
    /**
     * (Highcharts, Highmaps) As opposed to the `setExtremes` event, this event
     * fires after the final min and max values are computed and corrected for
     * `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.events.afterSetExtremes
     * @see https://api.highcharts.com/highmaps/colorAxis.events.afterSetExtremes
     */
    afterSetExtremes?: AxisEventCallbackFunction;
    /**
     * (Highcharts, Highmaps) Fires when the legend item belonging to the
     * colorAxis is clicked. One parameter, `event`, is passed to the function.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.events.legendItemClick
     * @see https://api.highcharts.com/highmaps/colorAxis.events.legendItemClick
     */
    legendItemClick?: () => void;
    /**
     * (Highcharts, Highmaps) Fires when the minimum and maximum is set for the
     * axis, either by calling the `.setExtremes()` method or by selecting an
     * area in the chart. One parameter, `event`, is passed to the function,
     * containing common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.events.setExtremes
     * @see https://api.highcharts.com/highmaps/colorAxis.events.setExtremes
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 *
 * @see https://api.highcharts.com/gantt/colorAxis.grid
 */
export interface ColorAxisGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.grid.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.grid.borderWidth
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.grid.cellHeight
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.grid.columns
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.grid.enabled
     */
    enabled?: boolean;
}
/**
 * (Highcharts, Highmaps) The axis labels show the number for each tick.
 *
 * For more live examples on label options, see xAxis.labels in the Highcharts
 * API.
 *
 * @see https://api.highcharts.com/highcharts/colorAxis.labels
 * @see https://api.highcharts.com/highmaps/colorAxis.labels
 */
export interface ColorAxisLabelsOptions {
    /**
     * (Highcharts, Highmaps) What part of the string the given position is
     * anchored to. If `left`, the left side of the string is at the axis
     * position. Can be one of `"left"`, `"center"` or `"right"`. Defaults to an
     * intelligent guess based on which side of the chart the axis is on and the
     * rotation of the label.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.align
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.align
     */
    align?: ("center"|"left"|"right");
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.autoRotation
     * @see https://api.highcharts.com/highstock/colorAxis.labels.autoRotation
     * @see https://api.highcharts.com/gantt/colorAxis.labels.autoRotation
     */
    autoRotation?: Array<number>;
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.autoRotationLimit
     * @see https://api.highcharts.com/gantt/colorAxis.labels.autoRotationLimit
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts, Gantt) Polar charts only. The label's pixel distance from
     * the perimeter of the plot area.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.distance
     * @see https://api.highcharts.com/gantt/colorAxis.labels.distance
     */
    distance?: number;
    /**
     * (Highcharts, Highmaps) Enable or disable the axis labels.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.enabled
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highmaps) A format string for the axis label.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.format
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.format
     */
    format?: string;
    /**
     * (Highcharts, Highmaps) Callback JavaScript function to format the label.
     * The value is given by `this.value`. Additional properties for `this` are
     * `axis`, `chart`, `isFirst` and `isLast`. The value of the default label
     * formatter can be retrieved by calling
     * `this.axis.defaultLabelFormatter.call(this)` within the function.
     *
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.formatter
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.formatter
     */
    formatter?: FormatterCallbackFunction<AxisLabelsFormatterContextObject>;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.labels.indentation
     */
    indentation?: number;
    /**
     * (Highcharts, Highmaps) Horizontal axis only. When `staggerLines` is not
     * set, `maxStaggerLines` defines how many lines the axis is allowed to add
     * to automatically avoid overlapping X labels. Set to `1` to disable
     * overlap detection.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.maxStaggerLines
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.maxStaggerLines
     */
    maxStaggerLines?: number;
    /**
     * (Highcharts, Highmaps) How to handle overflowing labels on horizontal
     * color axis. Can be undefined or "justify". If "justify", labels will not
     * render outside the legend area. If there is room to move it, it will be
     * aligned to the edge, else it will be removed.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.overflow
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.padding
     * @see https://api.highcharts.com/gantt/colorAxis.labels.padding
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.position3d
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho");
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.reserveSpace
     * @see https://api.highcharts.com/gantt/colorAxis.labels.reserveSpace
     */
    reserveSpace?: boolean;
    /**
     * (Highcharts, Highmaps) Rotation of the labels in degrees.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.rotation
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.skew3d
     */
    skew3d?: boolean;
    /**
     * (Highcharts, Highmaps) Horizontal axes only. The number of lines to
     * spread the labels over to make room or tighter labels.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.staggerLines
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.staggerLines
     */
    staggerLines?: number;
    /**
     * (Highcharts, Highmaps) To show only every _n_'th label on the axis, set
     * the step to _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.step
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.step
     */
    step?: number;
    /**
     * (Highcharts, Highmaps) CSS styles for the label. Use `whiteSpace:
     * 'nowrap'` to prevent wrapping of category labels. Use `textOverflow:
     * 'none'` to prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.style
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highmaps) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.useHTML
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highmaps) The x position offset of the label relative to the
     * tick position on the axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.x
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.x
     */
    x?: number;
    /**
     * (Highcharts, Highmaps) The y position offset of the label relative to the
     * tick position on the axis. The default makes it adapt to the font size on
     * bottom axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.y
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.y
     */
    y?: number;
    /**
     * (Highcharts, Highmaps) The Z index for the axis labels.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels.zIndex
     * @see https://api.highcharts.com/highmaps/colorAxis.labels.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highmaps) The triangular marker on a scalar color axis that
 * points to the value of the hovered area. To disable the marker, set `marker:
 * null`.
 *
 * @see https://api.highcharts.com/highcharts/colorAxis.marker
 * @see https://api.highcharts.com/highmaps/colorAxis.marker
 */
export interface ColorAxisMarkerOptions {
    /**
     * (Highcharts, Highmaps) Animation for the marker as it moves between
     * values. Set to `false` to disable animation. Defaults to `{ duration: 50
     * }`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.marker.animation
     * @see https://api.highcharts.com/highmaps/colorAxis.marker.animation
     */
    animation?: (boolean|AnimationOptionsObject);
    /**
     * (Highcharts, Highmaps) The color of the marker.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.marker.color
     * @see https://api.highcharts.com/highmaps/colorAxis.marker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
}
/**
 * (Highcharts, Highmaps) A color axis for choropleth maps and heat maps.
 * Visually, the color axis will appear as a gradient or as separate items
 * inside the legend, depending on whether the axis is scalar or based on data
 * classes.
 *
 * For supported color formats, see the docs article about colors.
 *
 * A scalar color axis is represented by a gradient. The colors either range
 * between the minColor and the maxColor, or for more fine grained control the
 * colors can be defined in stops. Often times, the color axis needs to be
 * adjusted to get the right color spread for the data. In addition to stops,
 * consider using a logarithmic axis type, or setting min and max to avoid the
 * colors being determined by outliers.
 *
 * When dataClasses are used, the ranges are subdivided into separate classes
 * like categories based on their values. This can be used for ranges between
 * two values, but also for a true category. However, when your data is
 * categorized, it may be as convenient to add each category to a separate
 * series.
 *
 * See the Axis object for programmatic access to the axis.
 *
 * @see https://api.highcharts.com/highcharts/colorAxis
 * @see https://api.highcharts.com/highmaps/colorAxis
 */
export interface ColorAxisOptions {
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.alignTicks
     * @see https://api.highcharts.com/highstock/colorAxis.alignTicks
     * @see https://api.highcharts.com/gantt/colorAxis.alignTicks
     */
    alignTicks?: boolean;
    /**
     * (Highcharts, Highmaps) Whether to allow decimals on the color axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.allowDecimals
     * @see https://api.highcharts.com/highmaps/colorAxis.allowDecimals
     */
    allowDecimals?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.ceiling
     * @see https://api.highcharts.com/highstock/colorAxis.ceiling
     * @see https://api.highcharts.com/gantt/colorAxis.ceiling
     */
    ceiling?: number;
    /**
     * (Highcharts, Highmaps) A class name that opens for styling the axis by
     * CSS, especially in Highcharts styled mode. The class name is applied to
     * group elements for the grid, axis elements and labels.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.className
     * @see https://api.highcharts.com/highmaps/colorAxis.className
     */
    className?: string;
    /**
     * (Gantt) Show an indicator on the axis for the current date and time. Can
     * be a boolean or a configuration object similar to xAxis.plotLines.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.currentDateIndicator
     */
    currentDateIndicator?: (boolean|ColorAxisCurrentDateIndicatorOptions);
    /**
     * (Highcharts, Highmaps) Determines how to set each data class' color if no
     * individual color is set. The default value, `tween`, computes
     * intermediate colors between `minColor` and `maxColor`. The other possible
     * value, `category`, pulls colors from the global or chart specific colors
     * array.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.dataClassColor
     * @see https://api.highcharts.com/highmaps/colorAxis.dataClassColor
     */
    dataClassColor?: ("category"|"tween");
    /**
     * (Highcharts, Highmaps) An array of data classes or ranges for the
     * choropleth map. If none given, the color axis is scalar and values are
     * distributed as a gradient between the minimum and maximum colors.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.dataClasses
     * @see https://api.highcharts.com/highmaps/colorAxis.dataClasses
     */
    dataClasses?: Array<ColorAxisDataClassesOptions>;
    /**
     * (Highcharts, Highmaps) _Requires Accessibility module_
     *
     * Description of the axis to screen reader users.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.description
     * @see https://api.highcharts.com/highmaps/colorAxis.description
     */
    description?: string;
    /**
     * (Highcharts, Highmaps) Whether to force the axis to end on a tick. Use
     * this option with the maxPadding option to control the axis end.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.endOnTick
     * @see https://api.highcharts.com/highmaps/colorAxis.endOnTick
     */
    endOnTick?: boolean;
    /**
     * (Highcharts, Highmaps) Event handlers for the axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.events
     * @see https://api.highcharts.com/highmaps/colorAxis.events
     */
    events?: ColorAxisEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.floor
     * @see https://api.highcharts.com/highstock/colorAxis.floor
     * @see https://api.highcharts.com/gantt/colorAxis.floor
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     *
     * @see https://api.highcharts.com/gantt/colorAxis.grid
     */
    grid?: ColorAxisGridOptions;
    /**
     * (Highcharts, Highmaps) Color of the grid lines extending from the axis
     * across the gradient.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.gridLineColor
     * @see https://api.highcharts.com/highmaps/colorAxis.gridLineColor
     */
    gridLineColor?: ColorString;
    /**
     * (Highcharts, Highmaps) The dash or dot style of the grid lines. For
     * possible values, see this demonstration.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.gridLineDashStyle
     * @see https://api.highcharts.com/highmaps/colorAxis.gridLineDashStyle
     */
    gridLineDashStyle?: DashStyleType;
    /**
     * (Highcharts, Highmaps) The width of the grid lines extending from the
     * axis across the gradient of a scalar color axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.gridLineWidth
     * @see https://api.highcharts.com/highmaps/colorAxis.gridLineWidth
     */
    gridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.gridZIndex
     * @see https://api.highcharts.com/highstock/colorAxis.gridZIndex
     * @see https://api.highcharts.com/gantt/colorAxis.gridZIndex
     */
    gridZIndex?: number;
    /**
     * (Highcharts, Highmaps) An id for the axis. This can be used after render
     * time to get a pointer to the axis object through `chart.get()`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.id
     * @see https://api.highcharts.com/highmaps/colorAxis.id
     */
    id?: string;
    /**
     * (Highcharts, Highmaps) The axis labels show the number for each tick.
     *
     * For more live examples on label options, see xAxis.labels in the
     * Highcharts API.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.labels
     * @see https://api.highcharts.com/highmaps/colorAxis.labels
     */
    labels?: ColorAxisLabelsOptions;
    /**
     * (Highcharts, Highmaps) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.lineColor
     * @see https://api.highcharts.com/highmaps/colorAxis.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highmaps) The triangular marker on a scalar color axis that
     * points to the value of the hovered area. To disable the marker, set
     * `marker: null`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.marker
     * @see https://api.highcharts.com/highmaps/colorAxis.marker
     */
    marker?: ColorAxisMarkerOptions;
    /**
     * (Highcharts, Highmaps) The maximum value of the axis in terms of map
     * point values. If `null`, the max value is automatically calculated. If
     * the `endOnTick` option is true, the max value might be rounded up.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.max
     * @see https://api.highcharts.com/highmaps/colorAxis.max
     */
    max?: number;
    /**
     * (Highcharts, Highmaps) The color to represent the maximum of the color
     * axis. Unless dataClasses or stops are set, the gradient ends at this
     * value.
     *
     * If dataClasses are set, the color is based on minColor and maxColor
     * unless a color is set for each data class, or the dataClassColor is set.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.maxColor
     * @see https://api.highcharts.com/highmaps/colorAxis.maxColor
     */
    maxColor?: ColorString;
    /**
     * (Highcharts, Highmaps) Padding of the max value relative to the length of
     * the axis. A padding of 0.05 will make a 100px axis 5px longer.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.maxPadding
     * @see https://api.highcharts.com/highmaps/colorAxis.maxPadding
     */
    maxPadding?: number;
    /**
     * (Highstock) Maximum range which can be set using the navigator's handles.
     * Opposite of xAxis.minRange.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.maxRange
     */
    maxRange?: number;
    /**
     * (Highcharts, Highmaps) The minimum value of the axis in terms of map
     * point values. If `null`, the min value is automatically calculated. If
     * the `startOnTick` option is true, the min value might be rounded down.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.min
     * @see https://api.highcharts.com/highmaps/colorAxis.min
     */
    min?: number;
    /**
     * (Highcharts, Highmaps) The color to represent the minimum of the color
     * axis. Unless dataClasses or stops are set, the gradient starts at this
     * value.
     *
     * If dataClasses are set, the color is based on minColor and maxColor
     * unless a color is set for each data class, or the dataClassColor is set.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minColor
     * @see https://api.highcharts.com/highmaps/colorAxis.minColor
     */
    minColor?: ColorString;
    /**
     * (Highcharts, Highmaps) Color of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the
     * `.highcharts-minor-grid-line` class.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorGridLineColor
     * @see https://api.highcharts.com/highmaps/colorAxis.minorGridLineColor
     */
    minorGridLineColor?: ColorString;
    /**
     * (Highcharts, Highmaps) The dash or dot style of the minor grid lines. For
     * possible values, see this demonstration.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorGridLineDashStyle
     * @see https://api.highcharts.com/highmaps/colorAxis.minorGridLineDashStyle
     */
    minorGridLineDashStyle?: DashStyleType;
    /**
     * (Highcharts, Highmaps) Width of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorGridLineWidth
     * @see https://api.highcharts.com/highmaps/colorAxis.minorGridLineWidth
     */
    minorGridLineWidth?: number;
    /**
     * (Highcharts, Highmaps) Color for the minor tick marks.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorTickColor
     * @see https://api.highcharts.com/highmaps/colorAxis.minorTickColor
     */
    minorTickColor?: ColorString;
    /**
     * (Highcharts, Highmaps) Specific tick interval in axis units for the minor
     * ticks. On a linear axis, if `"auto"`, the minor tick interval is
     * calculated as a fifth of the tickInterval. If `null` or `undefined`,
     * minor ticks are not shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorTickInterval
     * @see https://api.highcharts.com/highmaps/colorAxis.minorTickInterval
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highcharts, Highmaps) The pixel length of the minor tick marks.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorTickLength
     * @see https://api.highcharts.com/highmaps/colorAxis.minorTickLength
     */
    minorTickLength?: number;
    /**
     * (Highcharts, Highmaps) The position of the minor tick marks relative to
     * the axis line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorTickPosition
     * @see https://api.highcharts.com/highmaps/colorAxis.minorTickPosition
     */
    minorTickPosition?: ("inside"|"outside");
    /**
     * (Highcharts, Highmaps) Enable or disable minor ticks. Unless
     * minorTickInterval is set, the tick interval is calculated as a fifth of
     * the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorTicks
     * @see https://api.highcharts.com/highmaps/colorAxis.minorTicks
     */
    minorTicks?: boolean;
    /**
     * (Highcharts, Highmaps) The pixel width of the minor tick mark.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minorTickWidth
     * @see https://api.highcharts.com/highmaps/colorAxis.minorTickWidth
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highmaps) Padding of the min value relative to the length of
     * the axis. A padding of 0.05 will make a 100px axis 5px longer.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.minPadding
     * @see https://api.highcharts.com/highmaps/colorAxis.minPadding
     */
    minPadding?: number;
    /**
     * (Highstock) In an ordinal axis, the points are equally spaced in the
     * chart regardless of the actual time or x distance between them. This
     * means that missing data periods (e.g. nights or weekends for a stock
     * chart) will not take up space in the chart. Having `ordinal: false` will
     * show any gaps created by the `gapSize` setting proportionate to their
     * duration.
     *
     * In stock charts the X axis is ordinal by default, unless the boost module
     * is used and at least one of the series' data length exceeds the
     * boostThreshold.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.ordinal
     */
    ordinal?: boolean;
    /**
     * (Highstock) Additional range on the right side of the xAxis. Works
     * similar to `xAxis.maxPadding`, but value is set in milliseconds. Can be
     * set for both main `xAxis` and the navigator's `xAxis`.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.overscroll
     */
    overscroll?: number;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.pane
     */
    pane?: number;
    /**
     * (Highstock) The zoomed range to display when only defining one or none of
     * `min` or `max`. For example, to show the latest month, a range of one
     * month can be set.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.range
     */
    range?: number;
    /**
     * (Highcharts, Highmaps) Whether to reverse the axis so that the highest
     * number is closest to the origin. Defaults to `false` in a horizontal
     * legend and `true` in a vertical legend, where the smallest value starts
     * on top.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.reversed
     * @see https://api.highcharts.com/highmaps/colorAxis.reversed
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) This option determines how stacks should be
     * ordered within a group. For example reversed xAxis also reverses stacks,
     * so first series comes last in a group. To keep order like for
     * non-reversed xAxis enable this option.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.reversedStacks
     * @see https://api.highcharts.com/highstock/colorAxis.reversedStacks
     */
    reversedStacks?: boolean;
    /**
     * (Highstock) An optional scrollbar to display on the X axis in response to
     * limiting the minimum and maximum of the axis values.
     *
     * In styled mode, all the presentational options for the scrollbar are
     * replaced by the classes `.highcharts-scrollbar-thumb`,
     * `.highcharts-scrollbar-arrow`, `.highcharts-scrollbar-button`,
     * `.highcharts-scrollbar-rifles` and `.highcharts-scrollbar-track`.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar
     */
    scrollbar?: ColorAxisScrollbarOptions;
    /**
     * (Highcharts, Highmaps) Whether to show the first tick label.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.showFirstLabel
     * @see https://api.highcharts.com/highmaps/colorAxis.showFirstLabel
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highmaps) Whether to display the colorAxis in the legend.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.showInLegend
     * @see https://api.highcharts.com/highmaps/colorAxis.showInLegend
     */
    showInLegend?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.showLastLabel
     * @see https://api.highcharts.com/highstock/colorAxis.showLastLabel
     * @see https://api.highcharts.com/gantt/colorAxis.showLastLabel
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.softMax
     * @see https://api.highcharts.com/highstock/colorAxis.softMax
     * @see https://api.highcharts.com/gantt/colorAxis.softMax
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.softMin
     * @see https://api.highcharts.com/highstock/colorAxis.softMin
     * @see https://api.highcharts.com/gantt/colorAxis.softMin
     */
    softMin?: number;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.startOfWeek
     * @see https://api.highcharts.com/highstock/colorAxis.startOfWeek
     * @see https://api.highcharts.com/gantt/colorAxis.startOfWeek
     */
    startOfWeek?: number;
    /**
     * (Highcharts, Highmaps) Whether to force the axis to start on a tick. Use
     * this option with the `maxPadding` option to control the axis start.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.startOnTick
     * @see https://api.highcharts.com/highmaps/colorAxis.startOnTick
     */
    startOnTick?: boolean;
    /**
     * (Highcharts, Highmaps) Color stops for the gradient of a scalar color
     * axis. Use this in cases where a linear gradient between a `minColor` and
     * `maxColor` is not sufficient. The stops is an array of tuples, where the
     * first item is a float between 0 and 1 assigning the relative position in
     * the gradient, and the second item is the color.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.stops
     * @see https://api.highcharts.com/highmaps/colorAxis.stops
     */
    stops?: Array<[number, ColorString]>;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickAmount
     * @see https://api.highcharts.com/highstock/colorAxis.tickAmount
     * @see https://api.highcharts.com/gantt/colorAxis.tickAmount
     */
    tickAmount?: number;
    /**
     * (Highcharts, Highmaps) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickColor
     * @see https://api.highcharts.com/highmaps/colorAxis.tickColor
     */
    tickColor?: ColorString;
    /**
     * (Highcharts, Highmaps) The interval of the tick marks in axis units. When
     * `null`, the tick interval is computed to approximately follow the
     * `tickPixelInterval`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickInterval
     * @see https://api.highcharts.com/highmaps/colorAxis.tickInterval
     */
    tickInterval?: number;
    /**
     * (Highcharts, Highmaps) The pixel length of the main tick marks on the
     * color axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickLength
     * @see https://api.highcharts.com/highmaps/colorAxis.tickLength
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickmarkPlacement
     * @see https://api.highcharts.com/gantt/colorAxis.tickmarkPlacement
     */
    tickmarkPlacement?: ("between"|"on");
    /**
     * (Highcharts, Highmaps) If tickInterval is `null` this option sets the
     * approximate pixel interval of the tick marks.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickPixelInterval
     * @see https://api.highcharts.com/highmaps/colorAxis.tickPixelInterval
     */
    tickPixelInterval?: number;
    /**
     * (Highcharts, Highmaps) The position of the major tick marks relative to
     * the axis line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickPosition
     * @see https://api.highcharts.com/highmaps/colorAxis.tickPosition
     */
    tickPosition?: ("inside"|"outside");
    /**
     * (Highcharts, Highmaps) A callback function returning array defining where
     * the ticks are laid out on the axis. This overrides the default behaviour
     * of tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickPositioner
     * @see https://api.highcharts.com/highmaps/colorAxis.tickPositioner
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highcharts, Highmaps) An array defining where the ticks are laid out on
     * the axis. This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickPositions
     * @see https://api.highcharts.com/highmaps/colorAxis.tickPositions
     */
    tickPositions?: Array<number>;
    /**
     * (Highcharts, Highmaps) The pixel width of the major tick marks.
     *
     * In styled mode, the stroke width is given in the `.highcharts-tick`
     * class.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.tickWidth
     * @see https://api.highcharts.com/highmaps/colorAxis.tickWidth
     */
    tickWidth?: number;
    /**
     * (Highcharts, Highmaps) The type of interpolation to use for the color
     * axis. Can be `linear` or `logarithmic`.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.type
     * @see https://api.highcharts.com/highmaps/colorAxis.type
     */
    type?: ("linear"|"logarithmic");
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.uniqueNames
     * @see https://api.highcharts.com/gantt/colorAxis.uniqueNames
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Datetime axis only. An array determining
     * what time intervals the ticks are allowed to fall on. Each array item is
     * an array where the first value is the time unit and the second value
     * another array of allowed multiples. Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.units
     * @see https://api.highcharts.com/highstock/colorAxis.units
     * @see https://api.highcharts.com/gantt/colorAxis.units
     */
    units?: Array<[string, (Array<number>|null)]>;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis.visible
     * @see https://api.highcharts.com/highstock/colorAxis.visible
     * @see https://api.highcharts.com/gantt/colorAxis.visible
     */
    visible?: boolean;
}
/**
 * (Highstock) An optional scrollbar to display on the X axis in response to
 * limiting the minimum and maximum of the axis values.
 *
 * In styled mode, all the presentational options for the scrollbar are replaced
 * by the classes `.highcharts-scrollbar-thumb`, `.highcharts-scrollbar-arrow`,
 * `.highcharts-scrollbar-button`, `.highcharts-scrollbar-rifles` and
 * `.highcharts-scrollbar-track`.
 *
 * @see https://api.highcharts.com/highstock/colorAxis.scrollbar
 */
export interface ColorAxisScrollbarOptions {
    /**
     * (Highstock) The background color of the scrollbar itself.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.barBackgroundColor
     */
    barBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the scrollbar's border.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.barBorderColor
     */
    barBorderColor?: ColorString;
    /**
     * (Highstock) The border rounding radius of the bar.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.barBorderRadius
     */
    barBorderRadius?: number;
    /**
     * (Highstock) The width of the bar's border.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.barBorderWidth
     */
    barBorderWidth?: number;
    /**
     * (Highstock) The color of the small arrow inside the scrollbar buttons.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.buttonArrowColor
     */
    buttonArrowColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of scrollbar buttons.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.buttonBackgroundColor
     */
    buttonBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the border of the scrollbar buttons.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.buttonBorderColor
     */
    buttonBorderColor?: ColorString;
    /**
     * (Highstock) The corner radius of the scrollbar buttons.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.buttonBorderRadius
     */
    buttonBorderRadius?: number;
    /**
     * (Highstock) The border width of the scrollbar buttons.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.buttonBorderWidth
     */
    buttonBorderWidth?: number;
    /**
     * (Highstock) Enable or disable the scrollbar.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The height of the scrollbar. The height also applies to the
     * width of the scroll arrows so that they are always squares. Defaults to
     * 20 for touch devices and 14 for mouse devices.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.height
     */
    height?: number;
    /**
     * (Highstock) Whether to redraw the main chart as the scrollbar or the
     * navigator zoomed window is moved. Defaults to `true` for modern browsers
     * and `false` for legacy IE browsers as well as mobile devices.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.liveRedraw
     */
    liveRedraw?: boolean;
    /**
     * (Highstock) The margin between the scrollbar and its axis when the
     * scrollbar is applied directly to an axis.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.margin
     */
    margin?: number;
    /**
     * (Highstock) The minimum width of the scrollbar.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.minWidth
     */
    minWidth?: number;
    /**
     * (Highstock) The color of the small rifles in the middle of the scrollbar.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.rifleColor
     */
    rifleColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Whether to show or hide the scrollbar when the scrolled
     * content is zoomed out to it full extent.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.showFull
     */
    showFull?: boolean;
    step?: number;
    /**
     * (Highstock) The color of the track background.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.trackBackgroundColor
     */
    trackBackgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the border of the scrollbar track.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.trackBorderColor
     */
    trackBorderColor?: ColorString;
    /**
     * (Highstock) The corner radius of the border of the scrollbar track.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.trackBorderRadius
     */
    trackBorderRadius?: number;
    /**
     * (Highstock) The width of the border of the scrollbar track.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.trackBorderWidth
     */
    trackBorderWidth?: number;
    /**
     * (Highstock) The z index of the scrollbar group.
     *
     * @see https://api.highcharts.com/highstock/colorAxis.scrollbar.zIndex
     */
    zIndex?: number;
}
/**
 * (Gantt) Marker options specific to the end markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/connectors.endMarker
 */
export interface ConnectorsEndMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector end markers.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker.width
     */
    width?: number;
}
/**
 * (Gantt) Marker options for this chart's Pathfinder connectors. Note that this
 * option is overridden by the `startMarker` and `endMarker` options.
 *
 * @see https://api.highcharts.com/gantt/connectors.marker
 */
export interface ConnectorsMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.radius
     */
    radius?: number;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker.width
     */
    width?: number;
}
/**
 * (Gantt) The Pathfinder module allows you to define connections between any
 * two points, represented as lines - optionally with markers for the start
 * and/or end points. Multiple algorithms are available for calculating how the
 * connecting lines are drawn.
 *
 * Connector functionality requires Highcharts Gantt to be loaded. In Gantt
 * charts, the connectors are used to draw dependencies between tasks.
 *
 * @see https://api.highcharts.com/gantt/connectors
 */
export interface ConnectorsOptions {
    /**
     * (Gantt) Set the default pathfinder margin to use, in pixels. Some
     * Pathfinder algorithms attempt to avoid obstacles, such as other points in
     * the chart. These algorithms use this margin to determine how close lines
     * can be to an obstacle. The default is to compute this automatically from
     * the size of the obstacles in the chart.
     *
     * To draw connecting lines close to existing points, set this to a low
     * number. For more space around existing points, set this number higher.
     *
     * @see https://api.highcharts.com/gantt/connectors.algorithmMargin
     */
    algorithmMargin?: number;
    /**
     * (Gantt) Set the default dash style for this chart's connecting lines.
     *
     * @see https://api.highcharts.com/gantt/connectors.dashStyle
     */
    dashStyle?: string;
    /**
     * (Gantt) Enable connectors for this chart. Requires Highcharts Gantt.
     *
     * @see https://api.highcharts.com/gantt/connectors.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Marker options specific to the end markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/connectors.endMarker
     */
    endMarker?: ConnectorsEndMarkerOptions;
    /**
     * (Gantt) Set the default color for this chart's Pathfinder connecting
     * lines. Defaults to the color of the point being connected.
     *
     * @see https://api.highcharts.com/gantt/connectors.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the default pixel width for this chart's Pathfinder
     * connecting lines.
     *
     * @see https://api.highcharts.com/gantt/connectors.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Marker options for this chart's Pathfinder connectors. Note that
     * this option is overridden by the `startMarker` and `endMarker` options.
     *
     * @see https://api.highcharts.com/gantt/connectors.marker
     */
    marker?: ConnectorsMarkerOptions;
    /**
     * (Gantt) Marker options specific to the start markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker
     */
    startMarker?: ConnectorsStartMarkerOptions;
    /**
     * (Gantt) Set the default pathfinder algorithm to use for this chart. It is
     * possible to define your own algorithms by adding them to the
     * Highcharts.Pathfinder.prototype.algorithms object before the chart has
     * been created.
     *
     * The default algorithms are as follows:
     *
     * `straight`: Draws a straight line between the connecting points. Does not
     * avoid other points when drawing.
     *
     * `simpleConnect`: Finds a path between the points using right angles only.
     * Takes only starting/ending points into account, and will not avoid other
     * points.
     *
     * `fastAvoid`: Finds a path between the points using right angles only.
     * Will attempt to avoid other points, but its focus is performance over
     * accuracy. Works well with less dense datasets.
     *
     * Default value: `straight` is used as default for most series types, while
     * `simpleConnect` is used as default for Gantt series, to show dependencies
     * between points.
     *
     * @see https://api.highcharts.com/gantt/connectors.type
     */
    type?: ("straight"|"fastAvoid"|"simpleConnect");
}
/**
 * (Gantt) Marker options specific to the start markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/connectors.startMarker
 */
export interface ConnectorsStartMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector start markers.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/connectors.startMarker.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Highchart by default puts a credits
 * label in the lower right corner of the chart. This can be changed using these
 * options.
 *
 * @see https://api.highcharts.com/highcharts/credits
 * @see https://api.highcharts.com/highstock/credits
 * @see https://api.highcharts.com/highmaps/credits
 * @see https://api.highcharts.com/gantt/credits
 */
export interface CreditsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to show the credits
     * text.
     *
     * @see https://api.highcharts.com/highcharts/credits.enabled
     * @see https://api.highcharts.com/highstock/credits.enabled
     * @see https://api.highcharts.com/highmaps/credits.enabled
     * @see https://api.highcharts.com/gantt/credits.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for the credits label.
     *
     * @see https://api.highcharts.com/highcharts/credits.href
     * @see https://api.highcharts.com/highstock/credits.href
     * @see https://api.highcharts.com/highmaps/credits.href
     * @see https://api.highcharts.com/gantt/credits.href
     */
    href?: string;
    /**
     * (Highmaps) Credits for map source to be concatenated with conventional
     * credit text. By default this is a format string that collects copyright
     * information from the map if available.
     *
     * @see https://api.highcharts.com/highmaps/credits.mapText
     */
    mapText?: string;
    /**
     * (Highmaps) Detailed credits for map source to be displayed on hover of
     * credits text. By default this is a format string that collects copyright
     * information from the map if available.
     *
     * @see https://api.highcharts.com/highmaps/credits.mapTextFull
     */
    mapTextFull?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Position configuration for the
     * credits label.
     *
     * @see https://api.highcharts.com/highcharts/credits.position
     * @see https://api.highcharts.com/highstock/credits.position
     * @see https://api.highcharts.com/highmaps/credits.position
     * @see https://api.highcharts.com/gantt/credits.position
     */
    position?: (AlignObject|CreditsPositionOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the credits
     * label.
     *
     * @see https://api.highcharts.com/highcharts/credits.style
     * @see https://api.highcharts.com/highstock/credits.style
     * @see https://api.highcharts.com/highmaps/credits.style
     * @see https://api.highcharts.com/gantt/credits.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the credits label.
     *
     * @see https://api.highcharts.com/highcharts/credits.text
     * @see https://api.highcharts.com/highstock/credits.text
     * @see https://api.highcharts.com/highmaps/credits.text
     * @see https://api.highcharts.com/gantt/credits.text
     */
    text?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Position configuration for the
 * credits label.
 *
 * @see https://api.highcharts.com/highcharts/credits.position
 * @see https://api.highcharts.com/highstock/credits.position
 * @see https://api.highcharts.com/highmaps/credits.position
 * @see https://api.highcharts.com/gantt/credits.position
 */
export interface CreditsPositionOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Horizontal alignment of the
     * credits.
     *
     * @see https://api.highcharts.com/highcharts/credits.position.align
     * @see https://api.highcharts.com/highstock/credits.position.align
     * @see https://api.highcharts.com/highmaps/credits.position.align
     * @see https://api.highcharts.com/gantt/credits.position.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Vertical alignment of the
     * credits.
     *
     * @see https://api.highcharts.com/highcharts/credits.position.verticalAlign
     * @see https://api.highcharts.com/highstock/credits.position.verticalAlign
     * @see https://api.highcharts.com/highmaps/credits.position.verticalAlign
     * @see https://api.highcharts.com/gantt/credits.position.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Horizontal pixel offset of the
     * credits.
     *
     * @see https://api.highcharts.com/highcharts/credits.position.x
     * @see https://api.highcharts.com/highstock/credits.position.x
     * @see https://api.highcharts.com/highmaps/credits.position.x
     * @see https://api.highcharts.com/gantt/credits.position.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Vertical pixel offset of the
     * credits.
     *
     * @see https://api.highcharts.com/highcharts/credits.position.y
     * @see https://api.highcharts.com/highstock/credits.position.y
     * @see https://api.highcharts.com/highmaps/credits.position.y
     * @see https://api.highcharts.com/gantt/credits.position.y
     */
    y?: number;
}
/**
 * A style object with camel case property names to define visual appearance of
 * a SVG element or HTML element. The properties can be whatever styles are
 * supported on the given SVG or HTML element.
 */
export interface CSSObject {
    [key: string]: (boolean|number|string|undefined);
    /**
     * Background style for the element.
     */
    background?: string;
    /**
     * Background color of the element.
     */
    backgroundColor?: ColorString;
    /**
     * Border style for the element.
     */
    border?: string;
    /**
     * Radius of the element border.
     */
    borderRadius?: number;
    /**
     * Color used in the element. The "contrast" option is a Highcharts custom
     * property that results in black or white, depending on the background of
     * the element.
     */
    color?: ("contrast"|ColorString);
    /**
     * Style of the mouse cursor when resting over the element.
     */
    cursor?: CursorType;
    /**
     * Font family of the element text. Multiple values have to be in decreasing
     * preference order and separated by comma.
     */
    fontFamily?: string;
    /**
     * Font size of the element text.
     */
    fontSize?: string;
    /**
     * Font weight of the element text.
     */
    fontWeight?: string;
    /**
     * Height of the element.
     */
    height?: number;
    /**
     * Width of the element border.
     */
    lineWidth?: number;
    /**
     * Opacity of the element.
     */
    opacity?: number;
    /**
     * Space around the element content.
     */
    padding?: string;
    /**
     * Behaviour of the element when the mouse cursor rests over it.
     */
    pointerEvents?: string;
    /**
     * Positioning of the element.
     */
    position?: string;
    /**
     * Alignment of the element text.
     */
    textAlign?: string;
    /**
     * Additional decoration of the element text.
     */
    textDecoration?: string;
    /**
     * Outline style of the element text.
     */
    textOutline?: string;
    /**
     * Line break style of the element text. Highcharts SVG elements support
     * `ellipsis` when a `width` is set.
     */
    textOverflow?: string;
    /**
     * Top spacing of the element relative to the parent element.
     */
    top?: string;
    /**
     * Animated transition of selected element properties.
     */
    transition?: string;
    /**
     * Line break style of the element text.
     */
    whiteSpace?: string;
    /**
     * Width of the element.
     */
    width?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The Data module provides a
 * simplified interface for adding data to a chart from sources like CVS, HTML
 * tables or grid views. See also the tutorial article on the Data module.
 *
 * It requires the `modules/data.js` file to be loaded.
 *
 * Please note that the default way of adding data in Highcharts, without the
 * need of a module, is through the series.data option.
 *
 * @see https://api.highcharts.com/highcharts/data
 * @see https://api.highcharts.com/highstock/data
 * @see https://api.highcharts.com/highmaps/data
 * @see https://api.highcharts.com/gantt/data
 */
export interface DataOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A callback function to modify
     * the CSV before parsing it. Return the modified string.
     *
     * @see https://api.highcharts.com/highcharts/data.beforeParse
     * @see https://api.highcharts.com/highstock/data.beforeParse
     * @see https://api.highcharts.com/highmaps/data.beforeParse
     * @see https://api.highcharts.com/gantt/data.beforeParse
     */
    beforeParse?: DataBeforeParseCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A two-dimensional array
     * representing the input data on tabular form. This input can be used when
     * the data is already parsed, for example from a grid view component. Each
     * cell can be a string or number. If not switchRowsAndColumns is set, the
     * columns are interpreted as series.
     *
     * @see https://api.highcharts.com/highcharts/data.columns
     * @see https://api.highcharts.com/highstock/data.columns
     * @see https://api.highcharts.com/highmaps/data.columns
     * @see https://api.highcharts.com/gantt/data.columns
     */
    columns?: Array<Array<any>>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A URL to a remote JSON dataset,
     * structured as a column array. Will be fetched when the chart is created
     * using Ajax.
     *
     * @see https://api.highcharts.com/highcharts/data.columnsURL
     * @see https://api.highcharts.com/highstock/data.columnsURL
     * @see https://api.highcharts.com/highmaps/data.columnsURL
     * @see https://api.highcharts.com/gantt/data.columnsURL
     */
    columnsURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The callback that is evaluated
     * when the data is finished loading, optionally from an external source,
     * and parsed. The first argument passed is a finished chart options object,
     * containing the series. These options can be extended with additional
     * options and passed directly to the chart constructor.
     *
     * @see https://api.highcharts.com/highcharts/data.complete
     * @see https://api.highcharts.com/highstock/data.complete
     * @see https://api.highcharts.com/highmaps/data.complete
     * @see https://api.highcharts.com/gantt/data.complete
     */
    complete?: DataCompleteCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A comma delimited string to be
     * parsed. Related options are startRow, endRow, startColumn and endColumn
     * to delimit what part of the table is used. The lineDelimiter and
     * itemDelimiter options define the CSV delimiter formats.
     *
     * The built-in CSV parser doesn't support all flavours of CSV, so in some
     * cases it may be necessary to use an external CSV parser. See this example
     * of parsing CSV through the MIT licensed Papa Parse library.
     *
     * @see https://api.highcharts.com/highcharts/data.csv
     * @see https://api.highcharts.com/highstock/data.csv
     * @see https://api.highcharts.com/highmaps/data.csv
     * @see https://api.highcharts.com/gantt/data.csv
     */
    csv?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An URL to a remote CSV dataset.
     * Will be fetched when the chart is created using Ajax.
     *
     * @see https://api.highcharts.com/highcharts/data.csvURL
     * @see https://api.highcharts.com/highstock/data.csvURL
     * @see https://api.highcharts.com/highmaps/data.csvURL
     * @see https://api.highcharts.com/gantt/data.csvURL
     */
    csvURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Sets the refresh rate for data
     * polling when importing remote dataset by setting data.csvURL,
     * data.rowsURL, data.columnsURL, or data.googleSpreadsheetKey.
     *
     * Note that polling must be enabled by setting data.enablePolling to true.
     *
     * The value is the number of seconds between pollings. It cannot be set to
     * less than 1 second.
     *
     * @see https://api.highcharts.com/highcharts/data.dataRefreshRate
     * @see https://api.highcharts.com/highstock/data.dataRefreshRate
     * @see https://api.highcharts.com/highmaps/data.dataRefreshRate
     * @see https://api.highcharts.com/gantt/data.dataRefreshRate
     */
    dataRefreshRate?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Which of the predefined date
     * formats in Date.prototype.dateFormats to use to parse date values.
     * Defaults to a best guess based on what format gives valid and ordered
     * dates. Valid options include: `YYYY/mm/dd`, `dd/mm/YYYY`, `mm/dd/YYYY`,
     * `dd/mm/YY`, `mm/dd/YY`.
     *
     * @see https://api.highcharts.com/highcharts/data.dateFormat
     * @see https://api.highcharts.com/highstock/data.dateFormat
     * @see https://api.highcharts.com/highmaps/data.dateFormat
     * @see https://api.highcharts.com/gantt/data.dateFormat
     */
    dateFormat?: ("dd/mm/YY"|"dd/mm/YYYY"|"dd/mm/YYYY"|"mm/dd/YY"|"mm/dd/YYYY"|"YYYY/mm/dd");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The decimal point used for
     * parsing numbers in the CSV.
     *
     * If both this and data.delimiter is set to `undefined`, the parser will
     * attempt to deduce the decimal point automatically.
     *
     * @see https://api.highcharts.com/highcharts/data.decimalPoint
     * @see https://api.highcharts.com/highstock/data.decimalPoint
     * @see https://api.highcharts.com/highmaps/data.decimalPoint
     * @see https://api.highcharts.com/gantt/data.decimalPoint
     */
    decimalPoint?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enables automatic refetching of
     * remote datasets every _n_ seconds (defined by setting
     * data.dataRefreshRate).
     *
     * Only works when either data.csvURL, data.rowsURL, data.columnsURL, or
     * data.googleSpreadsheetKey.
     *
     * @see https://api.highcharts.com/highcharts/data.enablePolling
     * @see https://api.highcharts.com/highstock/data.enablePolling
     * @see https://api.highcharts.com/highmaps/data.enablePolling
     * @see https://api.highcharts.com/gantt/data.enablePolling
     */
    enablePolling?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the last
     * column (indexed by 0) to use. Defaults to the last column containing
     * data.
     *
     * @see https://api.highcharts.com/highcharts/data.endColumn
     * @see https://api.highcharts.com/highstock/data.endColumn
     * @see https://api.highcharts.com/highmaps/data.endColumn
     * @see https://api.highcharts.com/gantt/data.endColumn
     */
    endColumn?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the last
     * row (indexed by 0) to use. Defaults to the last row containing data.
     *
     * @see https://api.highcharts.com/highcharts/data.endRow
     * @see https://api.highcharts.com/highstock/data.endRow
     * @see https://api.highcharts.com/highmaps/data.endRow
     * @see https://api.highcharts.com/gantt/data.endRow
     */
    endRow?: number;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the first row in the data
     * set as series names.
     *
     * @see https://api.highcharts.com/highcharts/data.firstRowAsNames
     * @see https://api.highcharts.com/highstock/data.firstRowAsNames
     * @see https://api.highcharts.com/gantt/data.firstRowAsNames
     */
    firstRowAsNames?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The key for a Google Spreadsheet
     * to load. See general information on GS.
     *
     * @see https://api.highcharts.com/highcharts/data.googleSpreadsheetKey
     * @see https://api.highcharts.com/highstock/data.googleSpreadsheetKey
     * @see https://api.highcharts.com/highmaps/data.googleSpreadsheetKey
     * @see https://api.highcharts.com/gantt/data.googleSpreadsheetKey
     */
    googleSpreadsheetKey?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Google Spreadsheet worksheet
     * to use in combination with googleSpreadsheetKey. The available id's from
     * your sheet can be read from
     * `https://spreadsheets.google.com/feeds/worksheets/{key}/public/basic`.
     *
     * @see https://api.highcharts.com/highcharts/data.googleSpreadsheetWorksheet
     * @see https://api.highcharts.com/highstock/data.googleSpreadsheetWorksheet
     * @see https://api.highcharts.com/highmaps/data.googleSpreadsheetWorksheet
     * @see https://api.highcharts.com/gantt/data.googleSpreadsheetWorksheet
     */
    googleSpreadsheetWorksheet?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Item or cell delimiter for
     * parsing CSV. Defaults to the tab character `\t` if a tab character is
     * found in the CSV string, if not it defaults to `,`.
     *
     * If this is set to false or undefined, the parser will attempt to deduce
     * the delimiter automatically.
     *
     * @see https://api.highcharts.com/highcharts/data.itemDelimiter
     * @see https://api.highcharts.com/highstock/data.itemDelimiter
     * @see https://api.highcharts.com/highmaps/data.itemDelimiter
     * @see https://api.highcharts.com/gantt/data.itemDelimiter
     */
    itemDelimiter?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Line delimiter for parsing CSV.
     *
     * @see https://api.highcharts.com/highcharts/data.lineDelimiter
     * @see https://api.highcharts.com/highstock/data.lineDelimiter
     * @see https://api.highcharts.com/highmaps/data.lineDelimiter
     * @see https://api.highcharts.com/gantt/data.lineDelimiter
     */
    lineDelimiter?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A callback function to access
     * the parsed columns, the two-dimentional input data array directly, before
     * they are interpreted into series data and categories. Return `false` to
     * stop completion, or call `this.complete()` to continue async.
     *
     * @see https://api.highcharts.com/highcharts/data.parsed
     * @see https://api.highcharts.com/highstock/data.parsed
     * @see https://api.highcharts.com/highmaps/data.parsed
     * @see https://api.highcharts.com/gantt/data.parsed
     */
    parsed?: DataParsedCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A callback function to parse
     * string representations of dates into JavaScript timestamps. Should return
     * an integer timestamp on success.
     *
     * @see https://api.highcharts.com/highcharts/data.parseDate
     * @see https://api.highcharts.com/highstock/data.parseDate
     * @see https://api.highcharts.com/highmaps/data.parseDate
     * @see https://api.highcharts.com/gantt/data.parseDate
     */
    parseDate?: DataParseDateCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The same as the columns input
     * option, but defining rows intead of columns.
     *
     * @see https://api.highcharts.com/highcharts/data.rows
     * @see https://api.highcharts.com/highstock/data.rows
     * @see https://api.highcharts.com/highmaps/data.rows
     * @see https://api.highcharts.com/gantt/data.rows
     */
    rows?: Array<Array<any>>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A URL to a remote JSON dataset,
     * structured as a row array. Will be fetched when the chart is created
     * using Ajax.
     *
     * @see https://api.highcharts.com/highcharts/data.rowsURL
     * @see https://api.highcharts.com/highstock/data.rowsURL
     * @see https://api.highcharts.com/highmaps/data.rowsURL
     * @see https://api.highcharts.com/gantt/data.rowsURL
     */
    rowsURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing dictionaries
     * for each series. A dictionary exists of Point property names as the key
     * and the CSV column index as the value.
     *
     * @see https://api.highcharts.com/highcharts/data.seriesMapping
     * @see https://api.highcharts.com/highstock/data.seriesMapping
     * @see https://api.highcharts.com/highmaps/data.seriesMapping
     * @see https://api.highcharts.com/gantt/data.seriesMapping
     */
    seriesMapping?: Array<Dictionary<number>>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the first
     * column (indexed by 0) to use.
     *
     * @see https://api.highcharts.com/highcharts/data.startColumn
     * @see https://api.highcharts.com/highstock/data.startColumn
     * @see https://api.highcharts.com/highmaps/data.startColumn
     * @see https://api.highcharts.com/gantt/data.startColumn
     */
    startColumn?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In tabular input data, the first
     * row (indexed by 0) to use.
     *
     * @see https://api.highcharts.com/highcharts/data.startRow
     * @see https://api.highcharts.com/highstock/data.startRow
     * @see https://api.highcharts.com/highmaps/data.startRow
     * @see https://api.highcharts.com/gantt/data.startRow
     */
    startRow?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Switch rows and columns of the
     * input data, so that `this.columns` effectively becomes the rows of the
     * data set, and the rows are interpreted as series.
     *
     * @see https://api.highcharts.com/highcharts/data.switchRowsAndColumns
     * @see https://api.highcharts.com/highstock/data.switchRowsAndColumns
     * @see https://api.highcharts.com/highmaps/data.switchRowsAndColumns
     * @see https://api.highcharts.com/gantt/data.switchRowsAndColumns
     */
    switchRowsAndColumns?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An HTML table or the id of such
     * to be parsed as input data. Related options are `startRow`, `endRow`,
     * `startColumn` and `endColumn` to delimit what part of the table is used.
     *
     * @see https://api.highcharts.com/highcharts/data.table
     * @see https://api.highcharts.com/highstock/data.table
     * @see https://api.highcharts.com/highmaps/data.table
     * @see https://api.highcharts.com/gantt/data.table
     */
    table?: (string|HTMLElement);
}
/**
 * Generic dictionary in TypeScript notation.
 */
export interface Dictionary<T> {
    [key: string]: T;
}
/**
 * (Highcharts, Highmaps) Additional styles to apply to the data label of a
 * point that has drilldown data. By default it is underlined and blue to invite
 * to interaction.
 *
 * In styled mode, active data label styles can be applied with the
 * `.highcharts-drilldown-data-label` class.
 *
 * @see https://api.highcharts.com/highcharts/drilldown.activeDataLabelStyle
 * @see https://api.highcharts.com/highmaps/drilldown.activeDataLabelStyle
 */
export interface DrilldownActiveDataLabelStyleOptions {
    color?: string;
    cursor?: string;
    fontWeight?: string;
    textDecoration?: string;
}
/**
 * (Highcharts, Highmaps) Options for the drill up button that appears when
 * drilling down on a series. The text for the button is defined in
 * lang.drillUpText.
 *
 * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton
 * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton
 */
export interface DrilldownDrillUpButtonOptions {
    /**
     * (Highcharts, Highmaps) Positioning options for the button within the
     * `relativeTo` box. Available properties are `x`, `y`, `align` and
     * `verticalAlign`.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.position
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.position
     */
    position?: (AlignObject|DrilldownDrillUpButtonPositionOptions);
    /**
     * (Highcharts, Highmaps) What box to align the button to. Can be either
     * `plotBox` or `spacingBox`.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.relativeTo
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.relativeTo
     */
    relativeTo?: ("plotBox"|"spacingBox");
    /**
     * (Highcharts, Highmaps) A collection of attributes for the button. The
     * object takes SVG attributes like `fill`, `stroke`, `stroke-width` or `r`,
     * the border radius. The theme also supports `style`, a collection of CSS
     * properties for the text. Equivalent attributes for the hover state are
     * given in `theme.states.hover`.
     *
     * In styled mode, drill-up button styles can be applied with the
     * `.highcharts-drillup-button` class.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.theme
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.theme
     */
    theme?: object;
}
/**
 * (Highcharts, Highmaps) Positioning options for the button within the
 * `relativeTo` box. Available properties are `x`, `y`, `align` and
 * `verticalAlign`.
 *
 * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.position
 * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.position
 */
export interface DrilldownDrillUpButtonPositionOptions {
    /**
     * (Highcharts, Highmaps) Horizontal alignment.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.position.align
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.position.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highmaps) Vertical alignment of the button.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.position.verticalAlign
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.position.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highmaps) The X offset of the button.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.position.x
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.position.x
     */
    x?: number;
    /**
     * (Highcharts, Highmaps) The Y offset of the button.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton.position.y
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton.position.y
     */
    y?: number;
}
/**
 * The event arguments when a drilldown point is clicked.
 */
export interface DrilldownEventObject {
    /**
     * If a category label was clicked, which index.
     */
    category?: number;
    /**
     * The original browser event (usually click) that triggered the drilldown.
     */
    originalEvent?: Event;
    /**
     * The originating point.
     */
    point: Point;
    /**
     * If a category label was clicked, this array holds all points corresponing
     * to the category. Otherwise it is set to false.
     */
    points?: (boolean|Array<Point>);
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: () => void;
    /**
     * Options for the new series. If the event is utilized for async drilldown,
     * the seriesOptions are not added, but rather loaded async.
     */
    seriesOptions?: SeriesOptionsType;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "drilldown";
}
/**
 * (Highcharts, Highstock, Highmaps) Options for drill down, the concept of
 * inspecting increasingly high resolution data through clicking on chart items
 * like columns or pie slices.
 *
 * The drilldown feature requires the drilldown.js file to be loaded, found in
 * the modules directory of the download package, or online at
 * code.highcharts.com/modules/drilldown.js.
 *
 * @see https://api.highcharts.com/highcharts/drilldown
 * @see https://api.highcharts.com/highstock/drilldown
 * @see https://api.highcharts.com/highmaps/drilldown
 */
export interface DrilldownOptions {
    /**
     * (Highcharts, Highmaps) Additional styles to apply to the X axis label for
     * a point that has drilldown data. By default it is underlined and blue to
     * invite to interaction.
     *
     * In styled mode, active label styles can be set with the
     * `.highcharts-drilldown-axis-label` class.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.activeAxisLabelStyle
     * @see https://api.highcharts.com/highmaps/drilldown.activeAxisLabelStyle
     */
    activeAxisLabelStyle?: CSSObject;
    /**
     * (Highcharts, Highmaps) Additional styles to apply to the data label of a
     * point that has drilldown data. By default it is underlined and blue to
     * invite to interaction.
     *
     * In styled mode, active data label styles can be applied with the
     * `.highcharts-drilldown-data-label` class.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.activeDataLabelStyle
     * @see https://api.highcharts.com/highmaps/drilldown.activeDataLabelStyle
     */
    activeDataLabelStyle?: (CSSObject|DrilldownActiveDataLabelStyleOptions);
    /**
     * (Highcharts) When this option is false, clicking a single point will
     * drill down all points in the same category, equivalent to clicking the X
     * axis label.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.allowPointDrilldown
     */
    allowPointDrilldown?: boolean;
    /**
     * (Highcharts, Highmaps) Set the animation for all drilldown animations.
     * Animation of a drilldown occurs when drilling between a column point and
     * a column series, or a pie slice and a full pie series. Drilldown can
     * still be used between series and points of different types, but animation
     * will not occur.
     *
     * The animation can either be set as a boolean or a configuration object.
     * If `true`, it will use the 'swing' jQuery easing and a duration of 500
     * ms. If used as a configuration object, the following properties are
     * supported:
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: A string reference to an easing function set on the `Math`
     * object. See the easing demo.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.animation
     * @see https://api.highcharts.com/highmaps/drilldown.animation
     */
    animation?: (boolean|AnimationOptionsObject);
    /**
     * (Highcharts, Highmaps) Options for the drill up button that appears when
     * drilling down on a series. The text for the button is defined in
     * lang.drillUpText.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.drillUpButton
     * @see https://api.highcharts.com/highmaps/drilldown.drillUpButton
     */
    drillUpButton?: DrilldownDrillUpButtonOptions;
    /**
     * (Highcharts, Highmaps) An array of series configurations for the drill
     * down. Each series configuration uses the same syntax as the series option
     * set. These drilldown series are hidden by default. The drilldown series
     * is linked to the parent series' point by its `id`.
     *
     * @see https://api.highcharts.com/highcharts/drilldown.series
     * @see https://api.highcharts.com/highmaps/drilldown.series
     */
    series?: Array<SeriesOptionsType>;
}
/**
 * The event arguments when all the series have been drilled up.
 */
export interface DrillupAllEventObject {
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: () => void;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "drillupall";
}
/**
 * The event arguments when drilling up from a drilldown series.
 */
export interface DrillupEventObject {
    /**
     * Prevents the default behaviour of the event.
     */
    preventDefault: () => void;
    /**
     * Options for the new series.
     */
    seriesOptions?: SeriesOptionsType;
    /**
     * The event target.
     */
    target: Chart;
    /**
     * The event type.
     */
    type: "drillup";
}
/**
 * The event options for adding function callback.
 */
export interface EventOptionsObject {
    /**
     * The order the event handler should be called. This opens for having one
     * handler be called before another, independent of in which order they were
     * added.
     */
    order: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the export button.
 *
 * In styled mode, export button styles can be applied with the
 * `.highcharts-contextbutton` class.
 *
 * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton
 * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton
 * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton
 * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton
 */
export interface ExportingButtonsContextButtonOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option is deprecated, use
     * titleKey instead.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton._titleKey
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton._titleKey
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton._titleKey
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton._titleKey
     */
    _titleKey?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The class name of the context
     * button.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.className
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.className
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.className
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable buttons.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.enabled
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.enabled
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.enabled
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The class name of the menu
     * appearing from the button.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.menuClassName
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.menuClassName
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.menuClassName
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.menuClassName
     */
    menuClassName?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of strings pointing
     * to config options for the menu items. The config options are defined in
     * the `menuItemDefinitions` option.
     *
     * By default, there is the "Print" menu item plus one menu item for each of
     * the available export types.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.menuItems
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.menuItems
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.menuItems
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.menuItems
     */
    menuItems?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A click handler callback to use
     * on the button directly instead of the popup menu.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.onclick
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.onclick
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.onclick
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.onclick
     */
    onclick?: () => void;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The symbol for the button.
     * Points to a definition function in the `Highcharts.Renderer.symbols`
     * collection. The default `exportIcon` function is part of the exporting
     * module. Possible values are "circle", "square", "diamond", "triangle",
     * "triangle-down", "menu", "menuball" or custom shape.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.symbol
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.symbol
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.symbol
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.symbol
     */
    symbol?: ("circle"|"diamond"|"menu"|"menuball"|"square"|"triangle"|"triangle-down"|"exportIcon");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) See
     * navigation.buttonOptions.symbolFill.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.symbolFill
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.symbolFill
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.symbolFill
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.symbolFill
     */
    symbolFill?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the symbol's stroke
     * or line.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.symbolStroke
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.symbolStroke
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.symbolStroke
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.symbolStroke
     */
    symbolStroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * symbol on the button.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.symbolStrokeWidth
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.symbolStrokeWidth
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.symbolStrokeWidth
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.symbolStrokeWidth
     */
    symbolStrokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text string to add to the
     * individual button.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.text
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.text
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.text
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
     * button theme. The object accepts SVG properties like `stroke-width`,
     * `stroke` and `fill`. Tri-state button styles are supported by the
     * `states.hover` and `states.select` objects.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.theme
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.theme
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.theme
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.theme
     */
    theme?: ExportingButtonsContextButtonThemeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The key to a lang option setting
     * that is used for the button's title tooltip. When the key is
     * `contextButtonTitle`, it refers to lang.contextButtonTitle that defaults
     * to "Chart context menu".
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.titleKey
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.titleKey
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.titleKey
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.titleKey
     */
    titleKey?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal position of the
     * button relative to the `align` option.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.x
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.x
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.x
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * button's position relative to its `verticalAlign`.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.y
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.y
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.y
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
 * button theme. The object accepts SVG properties like `stroke-width`, `stroke`
 * and `fill`. Tri-state button styles are supported by the `states.hover` and
 * `states.select` objects.
 *
 * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.theme
 * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.theme
 * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.theme
 * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.theme
 */
export interface ExportingButtonsContextButtonThemeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default fill exists only to
     * capture hover events.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.theme.fill
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.theme.fill
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.theme.fill
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.theme.fill
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Padding for the button.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.theme.padding
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.theme.padding
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.theme.padding
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.theme.padding
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default stroke for the buttons.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton.theme.stroke
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton.theme.stroke
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton.theme.stroke
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton.theme.stroke
     */
    stroke?: ColorString;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the export related
 * buttons, print and export. In addition to the default buttons listed here,
 * custom buttons can be added. See navigation.buttonOptions for general
 * options.
 *
 * @see https://api.highcharts.com/highcharts/exporting.buttons
 * @see https://api.highcharts.com/highstock/exporting.buttons
 * @see https://api.highcharts.com/highmaps/exporting.buttons
 * @see https://api.highcharts.com/gantt/exporting.buttons
 */
export interface ExportingButtonsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the export button.
     *
     * In styled mode, export button styles can be applied with the
     * `.highcharts-contextbutton` class.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons.contextButton
     * @see https://api.highcharts.com/highstock/exporting.buttons.contextButton
     * @see https://api.highcharts.com/highmaps/exporting.buttons.contextButton
     * @see https://api.highcharts.com/gantt/exporting.buttons.contextButton
     */
    contextButton?: ExportingButtonsContextButtonOptions;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for exporting data to CSV or
 * ExCel, or displaying the data in a HTML table or a JavaScript structure.
 * Requires the `export-data.js` module. This module adds data export options to
 * the export menu and provides functions like `Chart.getCSV`, `Chart.getTable`,
 * `Chart.getDataRows` and `Chart.viewData`.
 *
 * The XLS converter is limited and only creates a HTML string that is passed
 * for download, which works but creates a warning before opening. The
 * workaround for this is to use a third party XLSX converter, as demonstrated
 * in the sample below.
 *
 * @see https://api.highcharts.com/highcharts/exporting.csv
 * @see https://api.highcharts.com/highstock/exporting.csv
 * @see https://api.highcharts.com/highmaps/exporting.csv
 * @see https://api.highcharts.com/gantt/exporting.csv
 */
export interface ExportingCsvOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Formatter callback for the
     * column headers. Parameters are:
     *
     * - `item` - The series or axis object)
     *
     * - `key` - The point key, for example y or z
     *
     * - `keyLength` - The amount of value keys for this item, for example a
     * range series has the keys `low` and `high` so the key length is 2.
     *
     * If useMultiLevelHeaders is true, columnHeaderFormatter by default returns
     * an object with columnTitle and topLevelColumnTitle for each key. Columns
     * with the same topLevelColumnTitle have their titles merged into a single
     * cell with colspan for table/Excel export.
     *
     * If `useMultiLevelHeaders` is false, or for CSV export, it returns the
     * series name, followed by the key if there is more than one key.
     *
     * For the axis it returns the axis title or "Category" or "DateTime" by
     * default.
     *
     * Return `false` to use Highcharts' proposed header.
     *
     * @see https://api.highcharts.com/highcharts/exporting.csv.columnHeaderFormatter
     * @see https://api.highcharts.com/highstock/exporting.csv.columnHeaderFormatter
     * @see https://api.highcharts.com/highmaps/exporting.csv.columnHeaderFormatter
     * @see https://api.highcharts.com/gantt/exporting.csv.columnHeaderFormatter
     */
    columnHeaderFormatter?: (() => void|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Which date format to use for
     * exported dates on a datetime X axis. See `Highcharts.dateFormat`.
     *
     * @see https://api.highcharts.com/highcharts/exporting.csv.dateFormat
     * @see https://api.highcharts.com/highstock/exporting.csv.dateFormat
     * @see https://api.highcharts.com/highmaps/exporting.csv.dateFormat
     * @see https://api.highcharts.com/gantt/exporting.csv.dateFormat
     */
    dateFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Which decimal point to use for
     * exported CSV. Defaults to the same as the browser locale, typically `.`
     * (English) or `,` (German, French etc).
     *
     * @see https://api.highcharts.com/highcharts/exporting.csv.decimalPoint
     * @see https://api.highcharts.com/highstock/exporting.csv.decimalPoint
     * @see https://api.highcharts.com/highmaps/exporting.csv.decimalPoint
     * @see https://api.highcharts.com/gantt/exporting.csv.decimalPoint
     */
    decimalPoint?: (string|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The item delimiter in the
     * exported data. Use `;` for direct exporting to Excel. Defaults to a best
     * guess based on the browser locale. If the locale _decimal point_ is `,`,
     * the `itemDelimiter` defaults to `;`, otherwise the `itemDelimiter`
     * defaults to `,`.
     *
     * @see https://api.highcharts.com/highcharts/exporting.csv.itemDelimiter
     * @see https://api.highcharts.com/highstock/exporting.csv.itemDelimiter
     * @see https://api.highcharts.com/highmaps/exporting.csv.itemDelimiter
     * @see https://api.highcharts.com/gantt/exporting.csv.itemDelimiter
     */
    itemDelimiter?: (string|null);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The line delimiter in the
     * exported data, defaults to a newline.
     *
     * @see https://api.highcharts.com/highcharts/exporting.csv.lineDelimiter
     * @see https://api.highcharts.com/highstock/exporting.csv.lineDelimiter
     * @see https://api.highcharts.com/highmaps/exporting.csv.lineDelimiter
     * @see https://api.highcharts.com/gantt/exporting.csv.lineDelimiter
     */
    lineDelimiter?: string;
}
/**
 * Definition for a menu item in the context menu.
 */
export interface ExportingMenuObject {
    /**
     * The click handler for the menu item.
     */
    onclick?: () => void;
    /**
     * Indicates a separator line instead of an item.
     */
    separator?: boolean;
    /**
     * The text for the menu item.
     */
    text?: string;
    /**
     * If internationalization is required, the key to a language string.
     */
    textKey?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the exporting module.
 * For an overview on the matter, see the docs.
 *
 * @see https://api.highcharts.com/highcharts/exporting
 * @see https://api.highcharts.com/highstock/exporting
 * @see https://api.highcharts.com/highmaps/exporting
 * @see https://api.highcharts.com/gantt/exporting
 */
export interface ExportingOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Experimental setting to allow
     * HTML inside the chart (added through the `useHTML` options), directly in
     * the exported image. This allows you to preserve complicated HTML
     * structures like tables or bi-directional text in exported charts.
     *
     * Disclaimer: The HTML is rendered in a `foreignObject` tag in the
     * generated SVG. The official export server is based on PhantomJS, which
     * supports this, but other SVG clients, like Batik, does not support it.
     * This also applies to downloaded SVG that you want to open in a desktop
     * client.
     *
     * @see https://api.highcharts.com/highcharts/exporting.allowHTML
     * @see https://api.highcharts.com/highstock/exporting.allowHTML
     * @see https://api.highcharts.com/highmaps/exporting.allowHTML
     * @see https://api.highcharts.com/gantt/exporting.allowHTML
     */
    allowHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the export related
     * buttons, print and export. In addition to the default buttons listed
     * here, custom buttons can be added. See navigation.buttonOptions for
     * general options.
     *
     * @see https://api.highcharts.com/highcharts/exporting.buttons
     * @see https://api.highcharts.com/highstock/exporting.buttons
     * @see https://api.highcharts.com/highmaps/exporting.buttons
     * @see https://api.highcharts.com/gantt/exporting.buttons
     */
    buttons?: (ExportingButtonsOptions|Dictionary<ExportingButtonsContextButtonOptions>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Additional chart options to be
     * merged into an exported chart. For example, a common use case is to add
     * data labels to improve readability of the exported chart, or to add a
     * printer-friendly color scheme.
     *
     * @see https://api.highcharts.com/highcharts/exporting.chartOptions
     * @see https://api.highcharts.com/highstock/exporting.chartOptions
     * @see https://api.highcharts.com/highmaps/exporting.chartOptions
     * @see https://api.highcharts.com/gantt/exporting.chartOptions
     */
    chartOptions?: Options;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for exporting data to
     * CSV or ExCel, or displaying the data in a HTML table or a JavaScript
     * structure. Requires the `export-data.js` module. This module adds data
     * export options to the export menu and provides functions like
     * `Chart.getCSV`, `Chart.getTable`, `Chart.getDataRows` and
     * `Chart.viewData`.
     *
     * The XLS converter is limited and only creates a HTML string that is
     * passed for download, which works but creates a warning before opening.
     * The workaround for this is to use a third party XLSX converter, as
     * demonstrated in the sample below.
     *
     * @see https://api.highcharts.com/highcharts/exporting.csv
     * @see https://api.highcharts.com/highstock/exporting.csv
     * @see https://api.highcharts.com/highmaps/exporting.csv
     * @see https://api.highcharts.com/gantt/exporting.csv
     */
    csv?: ExportingCsvOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable the exporting
     * module. Disabling the module will hide the context button, but API
     * methods will still be available.
     *
     * @see https://api.highcharts.com/highcharts/exporting.enabled
     * @see https://api.highcharts.com/highstock/exporting.enabled
     * @see https://api.highcharts.com/highmaps/exporting.enabled
     * @see https://api.highcharts.com/gantt/exporting.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Function to call if the
     * offline-exporting module fails to export a chart on the client side, and
     * fallbackToExportServer is disabled. If left undefined, an exception is
     * thrown instead. Receives two parameters, the exporting options, and the
     * error from the module.
     *
     * @see https://api.highcharts.com/highcharts/exporting.error
     * @see https://api.highcharts.com/highstock/exporting.error
     * @see https://api.highcharts.com/highmaps/exporting.error
     * @see https://api.highcharts.com/gantt/exporting.error
     */
    error?: ExportingErrorCallbackFunction;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether or not to fall back to
     * the export server if the offline-exporting module is unable to export the
     * chart on the client side. This happens for certain browsers, and certain
     * features (e.g. allowHTML), depending on the image type exporting to. For
     * very complex charts, it is possible that export can fail in browsers that
     * don't support Blob objects, due to data URL length limits. It is
     * recommended to define the exporting.error handler if disabling fallback,
     * in order to notify users in case export fails.
     *
     * @see https://api.highcharts.com/highcharts/exporting.fallbackToExportServer
     * @see https://api.highcharts.com/highstock/exporting.fallbackToExportServer
     * @see https://api.highcharts.com/highmaps/exporting.fallbackToExportServer
     * @see https://api.highcharts.com/gantt/exporting.fallbackToExportServer
     */
    fallbackToExportServer?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The filename, without extension,
     * to use for the exported chart.
     *
     * @see https://api.highcharts.com/highcharts/exporting.filename
     * @see https://api.highcharts.com/highstock/exporting.filename
     * @see https://api.highcharts.com/highmaps/exporting.filename
     * @see https://api.highcharts.com/gantt/exporting.filename
     */
    filename?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An object containing additional
     * key value data for the POST form that sends the SVG to the export server.
     * For example, a `target` can be set to make sure the generated image is
     * received in another frame, or a custom `enctype` or `encoding` can be
     * set.
     *
     * @see https://api.highcharts.com/highcharts/exporting.formAttributes
     * @see https://api.highcharts.com/highstock/exporting.formAttributes
     * @see https://api.highcharts.com/highmaps/exporting.formAttributes
     * @see https://api.highcharts.com/gantt/exporting.formAttributes
     */
    formAttributes?: any;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Path where Highcharts will look
     * for export module dependencies to load on demand if they don't already
     * exist on `window`. Should currently point to location of CanVG library,
     * RGBColor.js, jsPDF and svg2pdf.js, required for client side export in
     * certain browsers.
     *
     * @see https://api.highcharts.com/highcharts/exporting.libURL
     * @see https://api.highcharts.com/highstock/exporting.libURL
     * @see https://api.highcharts.com/highmaps/exporting.libURL
     * @see https://api.highcharts.com/gantt/exporting.libURL
     */
    libURL?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An object consisting of
     * definitions for the menu items in the context menu. Each key value pair
     * has a `key` that is referenced in the menuItems setting, and a `value`,
     * which is an object with the following properties:
     *
     * - **onclick:** The click handler for the menu item
     *
     * - **text:** The text for the menu item
     *
     * - **textKey:** If internationalization is required, the key to a language
     * string
     *
     * @see https://api.highcharts.com/highcharts/exporting.menuItemDefinitions
     * @see https://api.highcharts.com/highstock/exporting.menuItemDefinitions
     * @see https://api.highcharts.com/highmaps/exporting.menuItemDefinitions
     * @see https://api.highcharts.com/gantt/exporting.menuItemDefinitions
     */
    menuItemDefinitions?: Dictionary<ExportingMenuObject>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When printing the chart from the
     * menu item in the burger menu, if the on-screen chart exceeds this width,
     * it is resized. After printing or cancelled, it is restored. The default
     * width makes the chart fit into typical paper format. Note that this does
     * not affect the chart when printing the web page as a whole.
     *
     * @see https://api.highcharts.com/highcharts/exporting.printMaxWidth
     * @see https://api.highcharts.com/highstock/exporting.printMaxWidth
     * @see https://api.highcharts.com/highmaps/exporting.printMaxWidth
     * @see https://api.highcharts.com/gantt/exporting.printMaxWidth
     */
    printMaxWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Defines the scale or zoom factor
     * for the exported image compared to the on-screen display. While for
     * instance a 600px wide chart may look good on a website, it will look bad
     * in print. The default scale of 2 makes this chart export to a 1200px PNG
     * or JPG.
     *
     * @see https://api.highcharts.com/highcharts/exporting.scale
     * @see https://api.highcharts.com/highstock/exporting.scale
     * @see https://api.highcharts.com/highmaps/exporting.scale
     * @see https://api.highcharts.com/gantt/exporting.scale
     */
    scale?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module required.
     * Show a HTML table below the chart with the chart's current data.
     *
     * @see https://api.highcharts.com/highcharts/exporting.showTable
     * @see https://api.highcharts.com/highstock/exporting.showTable
     * @see https://api.highcharts.com/highmaps/exporting.showTable
     * @see https://api.highcharts.com/gantt/exporting.showTable
     */
    showTable?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Analogous to sourceWidth.
     *
     * @see https://api.highcharts.com/highcharts/exporting.sourceHeight
     * @see https://api.highcharts.com/highstock/exporting.sourceHeight
     * @see https://api.highcharts.com/highmaps/exporting.sourceHeight
     * @see https://api.highcharts.com/gantt/exporting.sourceHeight
     */
    sourceHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the original chart
     * when exported, unless an explicit chart.width is set, or a pixel width is
     * set on the container. The width exported raster image is then multiplied
     * by scale.
     *
     * @see https://api.highcharts.com/highcharts/exporting.sourceWidth
     * @see https://api.highcharts.com/highstock/exporting.sourceWidth
     * @see https://api.highcharts.com/highmaps/exporting.sourceWidth
     * @see https://api.highcharts.com/gantt/exporting.sourceWidth
     */
    sourceWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module required.
     * Caption for the data table. Same as chart title by default. Set to
     * `false` to disable.
     *
     * @see https://api.highcharts.com/highcharts/exporting.tableCaption
     * @see https://api.highcharts.com/highstock/exporting.tableCaption
     * @see https://api.highcharts.com/highmaps/exporting.tableCaption
     * @see https://api.highcharts.com/gantt/exporting.tableCaption
     */
    tableCaption?: (boolean|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default MIME type for exporting
     * if `chart.exportChart()` is called without specifying a `type` option.
     * Possible values are `image/png`, `image/jpeg`, `application/pdf` and
     * `image/svg+xml`.
     *
     * @see https://api.highcharts.com/highcharts/exporting.type
     * @see https://api.highcharts.com/highstock/exporting.type
     * @see https://api.highcharts.com/highmaps/exporting.type
     * @see https://api.highcharts.com/gantt/exporting.type
     */
    type?: ("application/pdf"|"image/jpeg"|"image/png"|"image/svg+xml");
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The URL for the server module
     * converting the SVG string to an image format. By default this points to
     * Highchart's free web service.
     *
     * @see https://api.highcharts.com/highcharts/exporting.url
     * @see https://api.highcharts.com/highstock/exporting.url
     * @see https://api.highcharts.com/highmaps/exporting.url
     * @see https://api.highcharts.com/gantt/exporting.url
     */
    url?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module required. Use
     * multi level headers in data table. If csv.columnHeaderFormatter is
     * defined, it has to return objects in order for multi level headers to
     * work.
     *
     * @see https://api.highcharts.com/highcharts/exporting.useMultiLevelHeaders
     * @see https://api.highcharts.com/highstock/exporting.useMultiLevelHeaders
     * @see https://api.highcharts.com/highmaps/exporting.useMultiLevelHeaders
     * @see https://api.highcharts.com/gantt/exporting.useMultiLevelHeaders
     */
    useMultiLevelHeaders?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module required. If
     * using multi level table headers, use rowspans for headers that have only
     * one level.
     *
     * @see https://api.highcharts.com/highcharts/exporting.useRowspanHeaders
     * @see https://api.highcharts.com/highstock/exporting.useRowspanHeaders
     * @see https://api.highcharts.com/highmaps/exporting.useRowspanHeaders
     * @see https://api.highcharts.com/gantt/exporting.useRowspanHeaders
     */
    useRowspanHeaders?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of charts
     * exported to PNG or JPG. As of Highcharts 3.0, the default pixel width is
     * a function of the chart.width or exporting.sourceWidth and the
     * exporting.scale.
     *
     * @see https://api.highcharts.com/highcharts/exporting.width
     * @see https://api.highcharts.com/highstock/exporting.width
     * @see https://api.highcharts.com/highmaps/exporting.width
     * @see https://api.highcharts.com/gantt/exporting.width
     */
    width?: number;
}
/**
 * The returned object literal from the Highcharts.Axis#getExtremes function.
 */
export interface ExtremesObject {
    /**
     * The maximum value of the axis' associated series.
     */
    dataMax: number;
    /**
     * The minimum value of the axis' associated series.
     */
    dataMin: number;
    /**
     * The maximum axis value, either automatic or set manually. If the `max`
     * option is not set, `maxPadding` is 0 and `endOnTick` is false, this value
     * will be the same as `dataMax`.
     */
    max: number;
    /**
     * The minimum axis value, either automatic or set manually. If the `min`
     * option is not set, `minPadding` is 0 and `startOnTick` is false, this
     * value will be the same as `dataMin`.
     */
    min: number;
    /**
     * The user defined maximum, either from the `max` option or from a zoom or
     * `setExtremes` action.
     */
    userMax: number;
    /**
     * The user defined minimum, either from the `min` option or from a zoom or
     * `setExtremes` action.
     */
    userMin: number;
}
/**
 * The font metrics.
 */
export interface FontMetricsObject {
    /**
     * The baseline relative to the top of the box.
     */
    b: number;
    /**
     * The font size.
     */
    f: number;
    /**
     * The line height.
     */
    h: number;
}
export interface GlobalOptions {
    /**
     * (Highcharts, Highmaps) _Canvg rendering for Android 2.x is removed as of
     * Highcharts 5.0\. Use the libURL option to configure exporting._
     *
     * The URL to the additional file to lazy load for Android 2.x devices.
     * These devices don't support SVG, so we download a helper file that
     * contains canvg, its dependency rbcolor, and our own CanVG Renderer class.
     * To avoid hotlinking to our site, you can install canvas-tools.js on your
     * own server and change this option accordingly.
     *
     * @see https://api.highcharts.com/highcharts/global.canvasToolsURL
     * @see https://api.highcharts.com/highmaps/global.canvasToolsURL
     */
    canvasToolsURL?: string;
    /**
     * (Highcharts, Highstock) This option is deprecated since v6.0.5. Instead,
     * use time.Date that supports individual time settings per chart.
     *
     * @see https://api.highcharts.com/highcharts/global.Date
     * @see https://api.highcharts.com/highstock/global.Date
     */
    Date?: () => void;
    /**
     * (Highcharts, Highstock) This option is deprecated since v6.0.5. Instead,
     * use time.getTimezoneOffset that supports individual time settings per
     * chart.
     *
     * @see https://api.highcharts.com/highcharts/global.getTimezoneOffset
     * @see https://api.highcharts.com/highstock/global.getTimezoneOffset
     */
    getTimezoneOffset?: () => void;
    /**
     * (Highcharts, Highstock) This option is deprecated since v6.0.5. Instead,
     * use time.timezone that supports individual time settings per chart.
     *
     * @see https://api.highcharts.com/highcharts/global.timezone
     * @see https://api.highcharts.com/highstock/global.timezone
     */
    timezone?: string;
    /**
     * (Highcharts, Highstock) This option is deprecated since v6.0.5. Instead,
     * use time.timezoneOffset that supports individual time settings per chart.
     *
     * @see https://api.highcharts.com/highcharts/global.timezoneOffset
     * @see https://api.highcharts.com/highstock/global.timezoneOffset
     */
    timezoneOffset?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) This option is deprecated since
     * v6.0.5. Instead, use time.useUTC that supports individual time settings
     * per chart.
     *
     * @see https://api.highcharts.com/highcharts/global.useUTC
     * @see https://api.highcharts.com/highstock/global.useUTC
     * @see https://api.highcharts.com/highmaps/global.useUTC
     * @see https://api.highcharts.com/gantt/global.useUTC
     */
    useUTC?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Path to the pattern image
     * required by VML browsers in order to draw radial gradients.
     *
     * @see https://api.highcharts.com/highcharts/global.VMLRadialGradientURL
     * @see https://api.highcharts.com/highstock/global.VMLRadialGradientURL
     * @see https://api.highcharts.com/highmaps/global.VMLRadialGradientURL
     * @see https://api.highcharts.com/gantt/global.VMLRadialGradientURL
     */
    VMLRadialGradientURL?: string;
}
/**
 * Gradient options instead of a solid color.
 */
export interface GradientColorObject {
    /**
     * Holds an object that defines the start position and the end position
     * relative to the shape.
     */
    linearGradient?: LinearGradientColorObject;
    /**
     * Holds an object that defines the center position and the radius.
     */
    radialGradient?: RadialGradientColorObject;
    /**
     * The first item in each tuple is the position in the gradient, where 0 is
     * the start of the gradient and 1 is the end of the gradient. Multiple
     * stops can be applied. The second item is the color for each stop. This
     * color can also be given in the rgba format.
     */
    stops?: Array<[number, ColorString]>;
}
/**
 * Containing the position of a box that should be avoided by labels.
 */
export interface LabelIntersectBoxObject {
    bottom: number;
    left: number;
    right: number;
    top: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) An HTML label that can be positioned
 * anywhere in the chart area.
 *
 * @see https://api.highcharts.com/highcharts/labels.items
 * @see https://api.highcharts.com/highstock/labels.items
 * @see https://api.highcharts.com/highmaps/labels.items
 * @see https://api.highcharts.com/gantt/labels.items
 */
export interface LabelsItemsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Inner HTML or text for the
     * label.
     *
     * @see https://api.highcharts.com/highcharts/labels.items.html
     * @see https://api.highcharts.com/highstock/labels.items.html
     * @see https://api.highcharts.com/highmaps/labels.items.html
     * @see https://api.highcharts.com/gantt/labels.items.html
     */
    html?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each label. To
     * position the label, use left and top like this:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/labels.items.style
     * @see https://api.highcharts.com/highstock/labels.items.style
     * @see https://api.highcharts.com/highmaps/labels.items.style
     * @see https://api.highcharts.com/gantt/labels.items.style
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) HTML labels that can be positioned
 * anywhere in the chart area.
 *
 * @see https://api.highcharts.com/highcharts/labels
 * @see https://api.highcharts.com/highstock/labels
 * @see https://api.highcharts.com/highmaps/labels
 * @see https://api.highcharts.com/gantt/labels
 */
export interface LabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An HTML label that can be
     * positioned anywhere in the chart area.
     *
     * @see https://api.highcharts.com/highcharts/labels.items
     * @see https://api.highcharts.com/highstock/labels.items
     * @see https://api.highcharts.com/highmaps/labels.items
     * @see https://api.highcharts.com/gantt/labels.items
     */
    items?: Array<LabelsItemsOptions>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Shared CSS styles for all
     * labels.
     *
     * @see https://api.highcharts.com/highcharts/labels.style
     * @see https://api.highcharts.com/highstock/labels.style
     * @see https://api.highcharts.com/highmaps/labels.style
     * @see https://api.highcharts.com/gantt/labels.style
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Axis description format strings.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility.axis
 * @see https://api.highcharts.com/highstock/lang.accessibility.axis
 * @see https://api.highcharts.com/highmaps/lang.accessibility.axis
 * @see https://api.highcharts.com/gantt/lang.accessibility.axis
 */
export interface LangAccessibilityAxisOptions {
    xAxisDescriptionPlural?: string;
    xAxisDescriptionSingular?: string;
    yAxisDescriptionPlural?: string;
    yAxisDescriptionSingular?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Chart type description strings. This
 * is added to the chart information region.
 *
 * If there is only a single series type used in the chart, we use the format
 * string for the series type, or default if missing. There is one format string
 * for cases where there is only a single series in the chart, and one for
 * multiple series of the same type.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility.chartTypes
 * @see https://api.highcharts.com/highstock/lang.accessibility.chartTypes
 * @see https://api.highcharts.com/highmaps/lang.accessibility.chartTypes
 * @see https://api.highcharts.com/gantt/lang.accessibility.chartTypes
 */
export interface LangAccessibilityChartTypesOptions {
    barMultiple?: string;
    barSingle?: string;
    boxplotMultiple?: string;
    boxplotSingle?: string;
    bubbleMultiple?: string;
    bubbleSingle?: string;
    columnMultiple?: string;
    columnSingle?: string;
    combinationChart?: string;
    defaultMultiple?: string;
    defaultSingle?: string;
    emptyChart?: string;
    lineMultiple?: string;
    lineSingle?: string;
    mapTypeDescription?: string;
    pieMultiple?: string;
    pieSingle?: string;
    scatterMultiple?: string;
    scatterSingle?: string;
    splineMultiple?: string;
    splineSingle?: string;
    unknownMap?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Exporting menu format strings for
 * accessibility module.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility.exporting
 * @see https://api.highcharts.com/highstock/lang.accessibility.exporting
 * @see https://api.highcharts.com/highmaps/lang.accessibility.exporting
 * @see https://api.highcharts.com/gantt/lang.accessibility.exporting
 */
export interface LangAccessibilityExportingOptions {
    chartMenuLabel?: string;
    exportRegionLabel?: string;
    menuButtonLabel?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Configure the accessibility strings
 * in the chart. Requires the accessibility module to be loaded. For a
 * description of the module and information on its features, see Highcharts
 * Accessibility.
 *
 * For more dynamic control over the accessibility functionality, see
 * accessibility.pointDescriptionFormatter,
 * accessibility.seriesDescriptionFormatter, and
 * accessibility.screenReaderSectionFormatter.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility
 * @see https://api.highcharts.com/highstock/lang.accessibility
 * @see https://api.highcharts.com/highmaps/lang.accessibility
 * @see https://api.highcharts.com/gantt/lang.accessibility
 */
export interface LangAccessibilityOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Axis description format strings.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.axis
     * @see https://api.highcharts.com/highstock/lang.accessibility.axis
     * @see https://api.highcharts.com/highmaps/lang.accessibility.axis
     * @see https://api.highcharts.com/gantt/lang.accessibility.axis
     */
    axis?: LangAccessibilityAxisOptions;
    chartContainerLabel?: string;
    chartHeading?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Chart type description strings.
     * This is added to the chart information region.
     *
     * If there is only a single series type used in the chart, we use the
     * format string for the series type, or default if missing. There is one
     * format string for cases where there is only a single series in the chart,
     * and one for multiple series of the same type.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.chartTypes
     * @see https://api.highcharts.com/highstock/lang.accessibility.chartTypes
     * @see https://api.highcharts.com/highmaps/lang.accessibility.chartTypes
     * @see https://api.highcharts.com/gantt/lang.accessibility.chartTypes
     */
    chartTypes?: LangAccessibilityChartTypesOptions;
    defaultChartTitle?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting menu format strings
     * for accessibility module.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.exporting
     * @see https://api.highcharts.com/highstock/lang.accessibility.exporting
     * @see https://api.highcharts.com/highmaps/lang.accessibility.exporting
     * @see https://api.highcharts.com/gantt/lang.accessibility.exporting
     */
    exporting?: LangAccessibilityExportingOptions;
    legendItem?: string;
    longDescriptionHeading?: string;
    mapZoomIn?: string;
    mapZoomOut?: string;
    navigationHint?: string;
    noDescription?: string;
    rangeSelectorButton?: string;
    rangeSelectorMaxInput?: string;
    rangeSelectorMinInput?: string;
    screenReaderRegionLabel?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for different
     * series types. For more dynamic control over the series element
     * descriptions, see accessibility.seriesDescriptionFormatter.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.series
     * @see https://api.highcharts.com/highstock/lang.accessibility.series
     * @see https://api.highcharts.com/highmaps/lang.accessibility.series
     * @see https://api.highcharts.com/gantt/lang.accessibility.series
     */
    series?: LangAccessibilitySeriesOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Descriptions of lesser known
     * series types. The relevant description is added to the screen reader
     * information region when these series types are used.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.seriesTypeDescriptions
     * @see https://api.highcharts.com/highstock/lang.accessibility.seriesTypeDescriptions
     * @see https://api.highcharts.com/highmaps/lang.accessibility.seriesTypeDescriptions
     * @see https://api.highcharts.com/gantt/lang.accessibility.seriesTypeDescriptions
     */
    seriesTypeDescriptions?: LangAccessibilitySeriesTypeDescriptionsOptions;
    structureHeading?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Title element text for the chart
     * SVG element. Leave this empty to disable adding the title element.
     * Browsers will display this content when hovering over elements in the
     * chart. Assistive technology may use this element to label the chart.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.svgContainerTitle
     * @see https://api.highcharts.com/highstock/lang.accessibility.svgContainerTitle
     * @see https://api.highcharts.com/highmaps/lang.accessibility.svgContainerTitle
     * @see https://api.highcharts.com/gantt/lang.accessibility.svgContainerTitle
     */
    svgContainerTitle?: string;
    tableSummary?: string;
    viewAsDataTable?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for different
 * series types. For more dynamic control over the series element descriptions,
 * see accessibility.seriesDescriptionFormatter.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility.series
 * @see https://api.highcharts.com/highstock/lang.accessibility.series
 * @see https://api.highcharts.com/highmaps/lang.accessibility.series
 * @see https://api.highcharts.com/gantt/lang.accessibility.series
 */
export interface LangAccessibilitySeriesOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) User supplied description text.
     * This is added after the main summary if present.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.series.description
     * @see https://api.highcharts.com/highstock/lang.accessibility.series.description
     * @see https://api.highcharts.com/highmaps/lang.accessibility.series.description
     * @see https://api.highcharts.com/gantt/lang.accessibility.series.description
     */
    description?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for the
     * series main summary. Each series type has two modes:
     *
     * 1. This series type is the only series type used in the chart
     *
     * 2. This is a combination chart with multiple series types
     *
     * If a definition does not exist for the specific series type and mode, the
     * 'default' lang definitions are used.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.series.summary
     * @see https://api.highcharts.com/highstock/lang.accessibility.series.summary
     * @see https://api.highcharts.com/highmaps/lang.accessibility.series.summary
     * @see https://api.highcharts.com/gantt/lang.accessibility.series.summary
     */
    summary?: LangAccessibilitySeriesSummaryOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) xAxis description for series if
     * there are multiple xAxes in the chart.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.series.xAxisDescription
     * @see https://api.highcharts.com/highstock/lang.accessibility.series.xAxisDescription
     * @see https://api.highcharts.com/highmaps/lang.accessibility.series.xAxisDescription
     * @see https://api.highcharts.com/gantt/lang.accessibility.series.xAxisDescription
     */
    xAxisDescription?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) yAxis description for series if
     * there are multiple yAxes in the chart.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility.series.yAxisDescription
     * @see https://api.highcharts.com/highstock/lang.accessibility.series.yAxisDescription
     * @see https://api.highcharts.com/highmaps/lang.accessibility.series.yAxisDescription
     * @see https://api.highcharts.com/gantt/lang.accessibility.series.yAxisDescription
     */
    yAxisDescription?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Lang configuration for the series
 * main summary. Each series type has two modes:
 *
 * 1. This series type is the only series type used in the chart
 *
 * 2. This is a combination chart with multiple series types
 *
 * If a definition does not exist for the specific series type and mode, the
 * 'default' lang definitions are used.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility.series.summary
 * @see https://api.highcharts.com/highstock/lang.accessibility.series.summary
 * @see https://api.highcharts.com/highmaps/lang.accessibility.series.summary
 * @see https://api.highcharts.com/gantt/lang.accessibility.series.summary
 */
export interface LangAccessibilitySeriesSummaryOptions {
    bar?: string;
    barCombination?: string;
    boxplot?: string;
    boxplotCombination?: string;
    bubble?: string;
    bubbleCombination?: string;
    column?: string;
    columnCombination?: string;
    default?: string;
    defaultCombination?: string;
    line?: string;
    lineCombination?: string;
    map?: string;
    mapbubble?: string;
    mapbubbleCombination?: string;
    mapCombination?: string;
    mapline?: string;
    maplineCombination?: string;
    pie?: string;
    pieCombination?: string;
    scatter?: string;
    scatterCombination?: string;
    spline?: string;
    splineCombination?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Descriptions of lesser known series
 * types. The relevant description is added to the screen reader information
 * region when these series types are used.
 *
 * @see https://api.highcharts.com/highcharts/lang.accessibility.seriesTypeDescriptions
 * @see https://api.highcharts.com/highstock/lang.accessibility.seriesTypeDescriptions
 * @see https://api.highcharts.com/highmaps/lang.accessibility.seriesTypeDescriptions
 * @see https://api.highcharts.com/gantt/lang.accessibility.seriesTypeDescriptions
 */
export interface LangAccessibilitySeriesTypeDescriptionsOptions {
    arearange?: string;
    areasplinerange?: string;
    boxplot?: string;
    bubble?: string;
    columnrange?: string;
    errorbar?: string;
    funnel?: string;
    pyramid?: string;
    waterfall?: string;
}
/**
 * (Highcharts, Highstock) Configure the Popup strings in the chart. Requires
 * the `annotations.js` or `annotations-advanced.src.js` module to be loaded.
 *
 * @see https://api.highcharts.com/highcharts/lang.navigation
 * @see https://api.highcharts.com/highstock/lang.navigation
 */
export interface LangNavigationOptions {
    /**
     * (Highcharts, Highstock) Translations for all field names used in popup.
     *
     * @see https://api.highcharts.com/highcharts/lang.navigation.popup
     * @see https://api.highcharts.com/highstock/lang.navigation.popup
     */
    popup?: (object|LangNavigationPopupOptions);
}
/**
 * (Highcharts, Highstock) Translations for all field names used in popup.
 *
 * @see https://api.highcharts.com/highcharts/lang.navigation.popup
 * @see https://api.highcharts.com/highstock/lang.navigation.popup
 */
export interface LangNavigationPopupOptions {
    addButton?: string;
    arrowLine?: string;
    arrowRay?: string;
    arrowSegment?: string;
    background?: string;
    backgroundColor?: string;
    backgroundColors?: string;
    borderColor?: string;
    borderRadius?: string;
    borderWidth?: string;
    circle?: string;
    color?: string;
    connector?: string;
    crooked3?: string;
    crooked5?: string;
    crosshairX?: string;
    crosshairY?: string;
    editButton?: string;
    elliott3?: string;
    elliott5?: string;
    fibonacci?: string;
    fill?: string;
    flags?: string;
    fontSize?: string;
    format?: string;
    height?: string;
    horizontalLine?: string;
    infinityLine?: string;
    innerBackground?: string;
    label?: string;
    labelOptions?: string;
    labels?: string;
    line?: string;
    lines?: string;
    measure?: string;
    measureX?: string;
    measureXY?: string;
    measureY?: string;
    name?: string;
    outerBackground?: string;
    padding?: string;
    parallelChannel?: string;
    pitchfork?: string;
    ray?: string;
    rectangle?: string;
    removeButton?: string;
    saveButton?: string;
    segment?: string;
    series?: string;
    shapeOptions?: string;
    shapes?: string;
    simpleShapes?: string;
    stroke?: string;
    strokeWidth?: string;
    style?: string;
    title?: string;
    tunnel?: string;
    typeOptions?: string;
    verticalArrow?: string;
    verticalCounter?: string;
    verticalLabel?: string;
    verticalLine?: string;
    volume?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Language object. The language object
 * is global and it can't be set on each chart initiation. Instead, use
 * `Highcharts.setOptions` to set it before any chart is initialized.
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highcharts/lang
 * @see https://api.highcharts.com/highstock/lang
 * @see https://api.highcharts.com/highmaps/lang
 * @see https://api.highcharts.com/gantt/lang
 */
export interface LangOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Configure the accessibility
     * strings in the chart. Requires the accessibility module to be loaded. For
     * a description of the module and information on its features, see
     * Highcharts Accessibility.
     *
     * For more dynamic control over the accessibility functionality, see
     * accessibility.pointDescriptionFormatter,
     * accessibility.seriesDescriptionFormatter, and
     * accessibility.screenReaderSectionFormatter.
     *
     * @see https://api.highcharts.com/highcharts/lang.accessibility
     * @see https://api.highcharts.com/highstock/lang.accessibility
     * @see https://api.highcharts.com/highmaps/lang.accessibility
     * @see https://api.highcharts.com/gantt/lang.accessibility
     */
    accessibility?: LangAccessibilityOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module menu. The
     * tooltip title for the context menu holding print and export menu items.
     *
     * @see https://api.highcharts.com/highcharts/lang.contextButtonTitle
     * @see https://api.highcharts.com/highstock/lang.contextButtonTitle
     * @see https://api.highcharts.com/highmaps/lang.contextButtonTitle
     * @see https://api.highcharts.com/gantt/lang.contextButtonTitle
     */
    contextButtonTitle?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default decimal point used
     * in the `Highcharts.numberFormat` method unless otherwise specified in the
     * function arguments.
     *
     * @see https://api.highcharts.com/highcharts/lang.decimalPoint
     * @see https://api.highcharts.com/highstock/lang.decimalPoint
     * @see https://api.highcharts.com/highmaps/lang.decimalPoint
     * @see https://api.highcharts.com/gantt/lang.decimalPoint
     */
    decimalPoint?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module only. The
     * text for the menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.downloadCSV
     * @see https://api.highcharts.com/highstock/lang.downloadCSV
     * @see https://api.highcharts.com/highmaps/lang.downloadCSV
     * @see https://api.highcharts.com/gantt/lang.downloadCSV
     */
    downloadCSV?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the JPEG download menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.downloadJPEG
     * @see https://api.highcharts.com/highstock/lang.downloadJPEG
     * @see https://api.highcharts.com/highmaps/lang.downloadJPEG
     * @see https://api.highcharts.com/gantt/lang.downloadJPEG
     */
    downloadJPEG?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the PDF download menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.downloadPDF
     * @see https://api.highcharts.com/highstock/lang.downloadPDF
     * @see https://api.highcharts.com/highmaps/lang.downloadPDF
     * @see https://api.highcharts.com/gantt/lang.downloadPDF
     */
    downloadPDF?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the PNG download menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.downloadPNG
     * @see https://api.highcharts.com/highstock/lang.downloadPNG
     * @see https://api.highcharts.com/highmaps/lang.downloadPNG
     * @see https://api.highcharts.com/gantt/lang.downloadPNG
     */
    downloadPNG?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the SVG download menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.downloadSVG
     * @see https://api.highcharts.com/highstock/lang.downloadSVG
     * @see https://api.highcharts.com/highmaps/lang.downloadSVG
     * @see https://api.highcharts.com/gantt/lang.downloadSVG
     */
    downloadSVG?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module only. The
     * text for the menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.downloadXLS
     * @see https://api.highcharts.com/highstock/lang.downloadXLS
     * @see https://api.highcharts.com/highmaps/lang.downloadXLS
     * @see https://api.highcharts.com/gantt/lang.downloadXLS
     */
    downloadXLS?: string;
    /**
     * (Highcharts, Highmaps) The text for the button that appears when drilling
     * down, linking back to the parent series. The parent series' name is
     * inserted for `{series.name}`.
     *
     * @see https://api.highcharts.com/highcharts/lang.drillUpText
     * @see https://api.highcharts.com/highmaps/lang.drillUpText
     */
    drillUpText?: string;
    /**
     * (Highcharts, Highstock) What to show in a date field for invalid dates.
     * Defaults to an empty string.
     *
     * @see https://api.highcharts.com/highcharts/lang.invalidDate
     * @see https://api.highcharts.com/highstock/lang.invalidDate
     */
    invalidDate?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The loading text that appears
     * when the chart is set into the loading state following a call to
     * `chart.showLoading`.
     *
     * @see https://api.highcharts.com/highcharts/lang.loading
     * @see https://api.highcharts.com/highstock/lang.loading
     * @see https://api.highcharts.com/highmaps/lang.loading
     * @see https://api.highcharts.com/gantt/lang.loading
     */
    loading?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the months
     * names. Corresponds to the `%B` format in `Highcharts.dateFormat()`.
     *
     * @see https://api.highcharts.com/highcharts/lang.months
     * @see https://api.highcharts.com/highstock/lang.months
     * @see https://api.highcharts.com/highmaps/lang.months
     * @see https://api.highcharts.com/gantt/lang.months
     */
    months?: Array<string>;
    /**
     * (Highcharts, Highstock) Configure the Popup strings in the chart.
     * Requires the `annotations.js` or `annotations-advanced.src.js` module to
     * be loaded.
     *
     * @see https://api.highcharts.com/highcharts/lang.navigation
     * @see https://api.highcharts.com/highstock/lang.navigation
     */
    navigation?: (object|LangNavigationOptions);
    /**
     * (Highcharts, Highstock) The text to display when the chart contains no
     * data. Requires the no-data module, see noData.
     *
     * @see https://api.highcharts.com/highcharts/lang.noData
     * @see https://api.highcharts.com/highstock/lang.noData
     */
    noData?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The magnitude of numericSymbols
     * replacements. Use 10000 for Japanese, Korean and various Chinese locales,
     * which use symbols for 10^4, 10^8 and 10^12.
     *
     * @see https://api.highcharts.com/highcharts/lang.numericSymbolMagnitude
     * @see https://api.highcharts.com/highstock/lang.numericSymbolMagnitude
     * @see https://api.highcharts.com/highmaps/lang.numericSymbolMagnitude
     * @see https://api.highcharts.com/gantt/lang.numericSymbolMagnitude
     */
    numericSymbolMagnitude?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Metric prefixes used to shorten
     * high numbers in axis labels. Replacing any of the positions with `null`
     * causes the full number to be written. Setting `numericSymbols` to `null`
     * disables shortening altogether.
     *
     * @see https://api.highcharts.com/highcharts/lang.numericSymbols
     * @see https://api.highcharts.com/highstock/lang.numericSymbols
     * @see https://api.highcharts.com/highmaps/lang.numericSymbols
     * @see https://api.highcharts.com/gantt/lang.numericSymbols
     */
    numericSymbols?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module only. The
     * text for the menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.openInCloud
     * @see https://api.highcharts.com/highstock/lang.openInCloud
     * @see https://api.highcharts.com/highmaps/lang.openInCloud
     * @see https://api.highcharts.com/gantt/lang.openInCloud
     */
    openInCloud?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Exporting module only. The text
     * for the menu item to print the chart.
     *
     * @see https://api.highcharts.com/highcharts/lang.printChart
     * @see https://api.highcharts.com/highstock/lang.printChart
     * @see https://api.highcharts.com/highmaps/lang.printChart
     * @see https://api.highcharts.com/gantt/lang.printChart
     */
    printChart?: string;
    /**
     * (Highstock) The text for the label for the "from" input box in the range
     * selector.
     *
     * @see https://api.highcharts.com/highstock/lang.rangeSelectorFrom
     */
    rangeSelectorFrom?: string;
    /**
     * (Highstock) The text for the label for the "to" input box in the range
     * selector.
     *
     * @see https://api.highcharts.com/highstock/lang.rangeSelectorTo
     */
    rangeSelectorTo?: string;
    /**
     * (Highstock) The text for the label for the range selector buttons.
     *
     * @see https://api.highcharts.com/highstock/lang.rangeSelectorZoom
     */
    rangeSelectorZoom?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The text for the label appearing
     * when a chart is zoomed.
     *
     * @see https://api.highcharts.com/highcharts/lang.resetZoom
     * @see https://api.highcharts.com/highstock/lang.resetZoom
     * @see https://api.highcharts.com/highmaps/lang.resetZoom
     * @see https://api.highcharts.com/gantt/lang.resetZoom
     */
    resetZoom?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The tooltip title for the label
     * appearing when a chart is zoomed.
     *
     * @see https://api.highcharts.com/highcharts/lang.resetZoomTitle
     * @see https://api.highcharts.com/highstock/lang.resetZoomTitle
     * @see https://api.highcharts.com/highmaps/lang.resetZoomTitle
     * @see https://api.highcharts.com/gantt/lang.resetZoomTitle
     */
    resetZoomTitle?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the months
     * names in abbreviated form. Corresponds to the `%b` format in
     * `Highcharts.dateFormat()`.
     *
     * @see https://api.highcharts.com/highcharts/lang.shortMonths
     * @see https://api.highcharts.com/highstock/lang.shortMonths
     * @see https://api.highcharts.com/highmaps/lang.shortMonths
     * @see https://api.highcharts.com/gantt/lang.shortMonths
     */
    shortMonths?: Array<string>;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Short week days, starting
     * Sunday. If not specified, Highcharts uses the first three letters of the
     * `lang.weekdays` option.
     *
     * @see https://api.highcharts.com/highcharts/lang.shortWeekdays
     * @see https://api.highcharts.com/highstock/lang.shortWeekdays
     * @see https://api.highcharts.com/highmaps/lang.shortWeekdays
     * @see https://api.highcharts.com/gantt/lang.shortWeekdays
     */
    shortWeekdays?: Array<string>;
    /**
     * (Highstock) Configure the stockTools GUI titles(hints) in the chart.
     * Requires the `stock-tools.js` module to be loaded.
     *
     * @see https://api.highcharts.com/highstock/lang.stockTools
     */
    stockTools?: (object|LangStockToolsOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default thousands separator
     * used in the `Highcharts.numberFormat` method unless otherwise specified
     * in the function arguments. Defaults to a single space character, which is
     * recommended in ISO 31-0 and works across Anglo-American and continental
     * European languages.
     *
     * @see https://api.highcharts.com/highcharts/lang.thousandsSep
     * @see https://api.highcharts.com/highstock/lang.thousandsSep
     * @see https://api.highcharts.com/highmaps/lang.thousandsSep
     * @see https://api.highcharts.com/gantt/lang.thousandsSep
     */
    thousandsSep?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Export-data module only. The
     * text for the menu item.
     *
     * @see https://api.highcharts.com/highcharts/lang.viewData
     * @see https://api.highcharts.com/highstock/lang.viewData
     * @see https://api.highcharts.com/highmaps/lang.viewData
     * @see https://api.highcharts.com/gantt/lang.viewData
     */
    viewData?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the weekday
     * names.
     *
     * @see https://api.highcharts.com/highcharts/lang.weekdays
     * @see https://api.highcharts.com/highstock/lang.weekdays
     * @see https://api.highcharts.com/highmaps/lang.weekdays
     * @see https://api.highcharts.com/gantt/lang.weekdays
     */
    weekdays?: Array<string>;
    /**
     * (Highmaps) The title appearing on hovering the zoom in button. The text
     * itself defaults to "+" and can be changed in the button options.
     *
     * @see https://api.highcharts.com/highmaps/lang.zoomIn
     */
    zoomIn?: string;
    /**
     * (Highmaps) The title appearing on hovering the zoom out button. The text
     * itself defaults to "-" and can be changed in the button options.
     *
     * @see https://api.highcharts.com/highmaps/lang.zoomOut
     */
    zoomOut?: string;
}
export interface LangStockToolsGuiOptions {
    advanced?: string;
    arrowLine?: string;
    arrowRay?: string;
    arrowSegment?: string;
    circle?: string;
    crooked3?: string;
    crooked5?: string;
    crookedLines?: string;
    currentPriceIndicator?: string;
    elliott3?: string;
    elliott5?: string;
    fibonacci?: string;
    flagCirclepin?: string;
    flagDiamondpin?: string;
    flags?: string;
    flagSimplepin?: string;
    flagSquarepin?: string;
    fullScreen?: string;
    horizontalLine?: string;
    indicators?: string;
    infinityLine?: string;
    label?: string;
    line?: string;
    lines?: string;
    measure?: string;
    measureX?: string;
    measureXY?: string;
    measureY?: string;
    parallelChannel?: string;
    pitchfork?: string;
    ray?: string;
    rectangle?: string;
    saveChart?: string;
    segment?: string;
    simpleShapes?: string;
    toggleAnnotations?: string;
    typeCandlestick?: string;
    typeChange?: string;
    typeLine?: string;
    typeOHLC?: string;
    verticalArrow?: string;
    verticalCounter?: string;
    verticalLabel?: string;
    verticalLabels?: string;
    verticalLine?: string;
    zoomChange?: string;
    zoomX?: string;
    zoomXY?: string;
    zoomY?: string;
}
/**
 * (Highstock) Configure the stockTools GUI titles(hints) in the chart. Requires
 * the `stock-tools.js` module to be loaded.
 *
 * @see https://api.highcharts.com/highstock/lang.stockTools
 */
export interface LangStockToolsOptions {
    gui?: LangStockToolsGuiOptions;
}
export interface LegendBubbleLegendFormatterContextObject {
    /**
     * The center y position of the range.
     */
    center: number;
    /**
     * The radius of the bubble range.
     */
    radius: number;
    /**
     * The bubble value.
     */
    value: number;
}
/**
 * (Highcharts, Highstock, Highmaps) Options for the bubble legend labels.
 *
 * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels
 * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels
 * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels
 */
export interface LegendBubbleLegendLabelsOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The alignment of the labels compared to
     * the bubble legend. Can be one of `left`, `center` or `right`.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.align
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.align
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.align
     */
    align?: ("center"|"left"|"right");
    /**
     * (Highcharts, Highstock, Highmaps) Whether to allow data labels to
     * overlap.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.allowOverlap
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.allowOverlap
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) An additional class name to apply to
     * the bubble legend label graphical elements. This option does not replace
     * default class names of the graphical element.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.className
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.className
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps) A format string for the bubble legend
     * labels. Available variables are the same as for `formatter`.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.format
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.format
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.format
     */
    format?: string;
    /**
     * (Highcharts, Highstock, Highmaps) Available `this` properties are:
     *
     * - `this.value`: The bubble value.
     *
     * - `this.radius`: The radius of the bubble range.
     *
     * - `this.center`: The center y position of the range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.formatter
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.formatter
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.formatter
     */
    formatter?: FormatterCallbackFunction<LegendBubbleLegendFormatterContextObject>;
    /**
     * (Highcharts, Highstock, Highmaps) CSS styles for the labels.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.style
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.style
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps) The x position offset of the label
     * relative to the connector.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.x
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.x
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The y position offset of the label
     * relative to the connector.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels.y
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels.y
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) The bubble legend is an additional element
 * in legend which presents the scale of the bubble series. Individual bubble
 * ranges can be defined by user or calculated from series. In the case of
 * automatically calculated ranges, a 1px margin of error is permitted. Requires
 * `highcharts-more.js`.
 *
 * @see https://api.highcharts.com/highcharts/legend.bubbleLegend
 * @see https://api.highcharts.com/highstock/legend.bubbleLegend
 * @see https://api.highcharts.com/highmaps/legend.bubbleLegend
 */
export interface LegendBubbleLegendOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The color of the ranges borders, can be
     * also defined for an individual range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.borderColor
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.borderColor
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps) The width of the ranges borders in
     * pixels, can be also defined for an individual range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.borderWidth
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.borderWidth
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) An additional class name to apply to
     * the bubble legend' circle graphical elements. This option does not
     * replace default class names of the graphical element.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.className
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.className
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Highmaps) The main color of the bubble legend.
     * Applies to ranges, if individual color is not defined.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.color
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.color
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) An additional class name to apply to
     * the bubble legend's connector graphical elements. This option does not
     * replace default class names of the graphical element.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.connectorClassName
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.connectorClassName
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.connectorClassName
     */
    connectorClassName?: string;
    /**
     * (Highcharts, Highstock, Highmaps) The color of the connector, can be also
     * defined for an individual range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.connectorColor
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.connectorColor
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.connectorColor
     */
    connectorColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps) The length of the connectors in pixels.
     * If labels are centered, the distance is reduced to 0.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.connectorDistance
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.connectorDistance
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.connectorDistance
     */
    connectorDistance?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The width of the connectors in pixels.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.connectorWidth
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.connectorWidth
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.connectorWidth
     */
    connectorWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Enable or disable the bubble legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.enabled
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.enabled
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Options for the bubble legend labels.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.labels
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.labels
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.labels
     */
    labels?: LegendBubbleLegendLabelsOptions;
    /**
     * (Highcharts, Highstock, Highmaps) The position of the bubble legend in
     * the legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.legendIndex
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.legendIndex
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.legendIndex
     */
    legendIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Miximum bubble legend range size. If
     * values for ranges are not specified, the `minSize` and the `maxSize` are
     * calculated from bubble series.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.maxSize
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.maxSize
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.maxSize
     */
    maxSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Minimum bubble legend range size. If
     * values for ranges are not specified, the `minSize` and the `maxSize` are
     * calculated from bubble series.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.minSize
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.minSize
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.minSize
     */
    minSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Options for specific range. One range
     * consists of bubble, label and connector.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.ranges
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.ranges
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.ranges
     */
    ranges?: Array<LegendBubbleLegendRangesOptions>;
    /**
     * (Highcharts, Highstock, Highmaps) Whether the bubble legend range value
     * should be represented by the area or the width of the bubble. The
     * default, area, corresponds best to the human perception of the size of
     * each bubble.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.sizeBy
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.sizeBy
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.sizeBy
     */
    sizeBy?: ("area"|"width");
    /**
     * (Highcharts, Highstock, Highmaps) When this is true, the absolute value
     * of z determines the size of the bubble. This means that with the default
     * zThreshold of 0, a bubble of value -1 will have the same size as a bubble
     * of value 1, while a bubble of value 0 will have a smaller size according
     * to minSize.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.sizeByAbsoluteValue
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.sizeByAbsoluteValue
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.sizeByAbsoluteValue
     */
    sizeByAbsoluteValue?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps) Define the visual z index of the bubble
     * legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.zIndex
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.zIndex
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.zIndex
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock, Highmaps) Ranges with with lower value than
     * zThreshold, are skipped.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.zThreshold
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.zThreshold
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.zThreshold
     */
    zThreshold?: number;
}
/**
 * (Highcharts, Highstock, Highmaps) Options for specific range. One range
 * consists of bubble, label and connector.
 *
 * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.ranges
 * @see https://api.highcharts.com/highstock/legend.bubbleLegend.ranges
 * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.ranges
 */
export interface LegendBubbleLegendRangesOptions {
    /**
     * (Highcharts, Highstock, Highmaps) The color of the border for individual
     * range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.ranges.borderColor
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.ranges.borderColor
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.ranges.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps) The color of the bubble for individual
     * range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.ranges.color
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.ranges.color
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.ranges.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps) The color of the connector for
     * individual range.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.ranges.connectorColor
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.ranges.connectorColor
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.ranges.connectorColor
     */
    connectorColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps) Range size value, similar to bubble Z
     * data.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend.ranges.value
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend.ranges.value
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend.ranges.value
     */
    value?: object;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Default styling for the checkbox
 * next to a legend item when `showCheckbox` is true.
 *
 * @see https://api.highcharts.com/highcharts/legend.itemCheckboxStyle
 * @see https://api.highcharts.com/highstock/legend.itemCheckboxStyle
 * @see https://api.highcharts.com/highmaps/legend.itemCheckboxStyle
 * @see https://api.highcharts.com/gantt/legend.itemCheckboxStyle
 */
export interface LegendItemCheckboxStyleOptions {
    height?: string;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Keyboard navigation for the legend.
 * Requires the Accessibility module.
 *
 * @see https://api.highcharts.com/highcharts/legend.keyboardNavigation
 * @see https://api.highcharts.com/highstock/legend.keyboardNavigation
 * @see https://api.highcharts.com/highmaps/legend.keyboardNavigation
 * @see https://api.highcharts.com/gantt/legend.keyboardNavigation
 */
export interface LegendKeyboardNavigationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable/disable keyboard
     * navigation for the legend. Requires the Accessibility module.
     *
     * @see https://api.highcharts.com/highcharts/legend.keyboardNavigation.enabled
     * @see https://api.highcharts.com/highstock/legend.keyboardNavigation.enabled
     * @see https://api.highcharts.com/highmaps/legend.keyboardNavigation.enabled
     * @see https://api.highcharts.com/gantt/legend.keyboardNavigation.enabled
     */
    enabled?: boolean;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) Options for the paging or navigation
 * appearing when the legend is overflown. Navigation works well on screen, but
 * not in static exported images. One way of working around that is to increase
 * the chart height in export.
 *
 * @see https://api.highcharts.com/highcharts/legend.navigation
 * @see https://api.highcharts.com/highstock/legend.navigation
 * @see https://api.highcharts.com/highmaps/legend.navigation
 * @see https://api.highcharts.com/gantt/legend.navigation
 */
export interface LegendNavigationOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color for the active up or
     * down arrow in the legend page navigation.
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation.activeColor
     * @see https://api.highcharts.com/highstock/legend.navigation.activeColor
     * @see https://api.highcharts.com/highmaps/legend.navigation.activeColor
     * @see https://api.highcharts.com/gantt/legend.navigation.activeColor
     */
    activeColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) How to animate the pages when
     * navigating up or down. A value of `true` applies the default navigation
     * given in the `chart.animation` option. Additional options can be given as
     * an object containing values for easing and duration.
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation.animation
     * @see https://api.highcharts.com/highstock/legend.navigation.animation
     * @see https://api.highcharts.com/highmaps/legend.navigation.animation
     * @see https://api.highcharts.com/gantt/legend.navigation.animation
     */
    animation?: (boolean|AnimationOptionsObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel size of the up and
     * down arrows in the legend paging navigation.
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation.arrowSize
     * @see https://api.highcharts.com/highstock/legend.navigation.arrowSize
     * @see https://api.highcharts.com/highmaps/legend.navigation.arrowSize
     * @see https://api.highcharts.com/gantt/legend.navigation.arrowSize
     */
    arrowSize?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable the legend
     * navigation. In most cases, disabling the navigation results in an
     * unwanted overflow.
     *
     * See also the adapt chart to legend plugin for a solution to extend the
     * chart height to make room for the legend, optionally in exported charts
     * only.
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation.enabled
     * @see https://api.highcharts.com/highstock/legend.navigation.enabled
     * @see https://api.highcharts.com/highmaps/legend.navigation.enabled
     * @see https://api.highcharts.com/gantt/legend.navigation.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the inactive up or
     * down arrow in the legend page navigation. .
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation.inactiveColor
     * @see https://api.highcharts.com/highstock/legend.navigation.inactiveColor
     * @see https://api.highcharts.com/highmaps/legend.navigation.inactiveColor
     * @see https://api.highcharts.com/gantt/legend.navigation.inactiveColor
     */
    inactiveColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Text styles for the legend page
     * navigation.
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation.style
     * @see https://api.highcharts.com/highstock/legend.navigation.style
     * @see https://api.highcharts.com/highmaps/legend.navigation.style
     * @see https://api.highcharts.com/gantt/legend.navigation.style
     */
    style?: CSSObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The legend is a box containing a
 * symbol and name for each series item or point item in the chart. Each series
 * (or points in case of pie charts) is represented by a symbol and its name in
 * the legend.
 *
 * It is possible to override the symbol creator function and create custom
 * legend symbols.
 *
 * @see https://api.highcharts.com/highcharts/legend
 * @see https://api.highcharts.com/highstock/legend
 * @see https://api.highcharts.com/highmaps/legend
 * @see https://api.highcharts.com/gantt/legend
 */
export interface LegendOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The horizontal alignment of the
     * legend box within the chart area. Valid values are `left`, `center` and
     * `right`.
     *
     * In the case that the legend is aligned in a corner position, the `layout`
     * option will determine whether to place it above/below or on the side of
     * the plot area.
     *
     * @see https://api.highcharts.com/highcharts/legend.align
     * @see https://api.highcharts.com/highstock/legend.align
     * @see https://api.highcharts.com/highmaps/legend.align
     * @see https://api.highcharts.com/gantt/legend.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) If the layout is `horizontal`
     * and the legend items span over two lines or more, whether to align the
     * items into vertical columns. Setting this to `false` makes room for more
     * items, but will look more messy.
     *
     * @see https://api.highcharts.com/highcharts/legend.alignColumns
     * @see https://api.highcharts.com/highstock/legend.alignColumns
     * @see https://api.highcharts.com/highmaps/legend.alignColumns
     * @see https://api.highcharts.com/gantt/legend.alignColumns
     */
    alignColumns?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The background color of the
     * legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.backgroundColor
     * @see https://api.highcharts.com/highstock/legend.backgroundColor
     * @see https://api.highcharts.com/highmaps/legend.backgroundColor
     * @see https://api.highcharts.com/gantt/legend.backgroundColor
     */
    backgroundColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the drawn border
     * around the legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.borderColor
     * @see https://api.highcharts.com/highstock/legend.borderColor
     * @see https://api.highcharts.com/highmaps/legend.borderColor
     * @see https://api.highcharts.com/gantt/legend.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border corner radius of the
     * legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.borderRadius
     * @see https://api.highcharts.com/highstock/legend.borderRadius
     * @see https://api.highcharts.com/highmaps/legend.borderRadius
     * @see https://api.highcharts.com/gantt/legend.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the drawn border
     * around the legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.borderWidth
     * @see https://api.highcharts.com/highstock/legend.borderWidth
     * @see https://api.highcharts.com/highmaps/legend.borderWidth
     * @see https://api.highcharts.com/gantt/legend.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps) The bubble legend is an additional
     * element in legend which presents the scale of the bubble series.
     * Individual bubble ranges can be defined by user or calculated from
     * series. In the case of automatically calculated ranges, a 1px margin of
     * error is permitted. Requires `highcharts-more.js`.
     *
     * @see https://api.highcharts.com/highcharts/legend.bubbleLegend
     * @see https://api.highcharts.com/highstock/legend.bubbleLegend
     * @see https://api.highcharts.com/highmaps/legend.bubbleLegend
     */
    bubbleLegend?: LegendBubbleLegendOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Enable or disable the legend.
     * There is also a series-specific option, showInLegend, that can hide the
     * series from the legend. In some series types this is `false` by default,
     * so it must set to `true` in order to show the legend for the series.
     *
     * @see https://api.highcharts.com/highcharts/legend.enabled
     * @see https://api.highcharts.com/highstock/legend.enabled
     * @see https://api.highcharts.com/highmaps/legend.enabled
     * @see https://api.highcharts.com/gantt/legend.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When the legend is floating, the
     * plot area ignores it and is allowed to be placed below it.
     *
     * @see https://api.highcharts.com/highcharts/legend.floating
     * @see https://api.highcharts.com/highstock/legend.floating
     * @see https://api.highcharts.com/highmaps/legend.floating
     * @see https://api.highcharts.com/gantt/legend.floating
     */
    floating?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default styling for the checkbox
     * next to a legend item when `showCheckbox` is true.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemCheckboxStyle
     * @see https://api.highcharts.com/highstock/legend.itemCheckboxStyle
     * @see https://api.highcharts.com/highmaps/legend.itemCheckboxStyle
     * @see https://api.highcharts.com/gantt/legend.itemCheckboxStyle
     */
    itemCheckboxStyle?: (CSSObject|LegendItemCheckboxStyleOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) In a legend with horizontal
     * layout, the itemDistance defines the pixel distance between each item.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemDistance
     * @see https://api.highcharts.com/highstock/legend.itemDistance
     * @see https://api.highcharts.com/highmaps/legend.itemDistance
     * @see https://api.highcharts.com/gantt/legend.itemDistance
     */
    itemDistance?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each legend item
     * when the corresponding series or point is hidden. Only a subset of CSS is
     * supported, notably those options related to text. Properties are
     * inherited from `style` unless overridden here.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemHiddenStyle
     * @see https://api.highcharts.com/highstock/legend.itemHiddenStyle
     * @see https://api.highcharts.com/highmaps/legend.itemHiddenStyle
     * @see https://api.highcharts.com/gantt/legend.itemHiddenStyle
     */
    itemHiddenStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each legend item
     * in hover mode. Only a subset of CSS is supported, notably those options
     * related to text. Properties are inherited from `style` unless overridden
     * here.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemHoverStyle
     * @see https://api.highcharts.com/highstock/legend.itemHoverStyle
     * @see https://api.highcharts.com/highmaps/legend.itemHoverStyle
     * @see https://api.highcharts.com/gantt/legend.itemHoverStyle
     */
    itemHoverStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel bottom margin for each
     * legend item.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemMarginBottom
     * @see https://api.highcharts.com/highstock/legend.itemMarginBottom
     * @see https://api.highcharts.com/highmaps/legend.itemMarginBottom
     * @see https://api.highcharts.com/gantt/legend.itemMarginBottom
     */
    itemMarginBottom?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel top margin for each
     * legend item.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemMarginTop
     * @see https://api.highcharts.com/highstock/legend.itemMarginTop
     * @see https://api.highcharts.com/highmaps/legend.itemMarginTop
     * @see https://api.highcharts.com/gantt/legend.itemMarginTop
     */
    itemMarginTop?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for each legend item.
     * Only a subset of CSS is supported, notably those options related to text.
     * The default `textOverflow` property makes long texts truncate. Set it to
     * `undefined` to wrap text instead. A `width` property can be added to
     * control the text width.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemStyle
     * @see https://api.highcharts.com/highstock/legend.itemStyle
     * @see https://api.highcharts.com/highmaps/legend.itemStyle
     * @see https://api.highcharts.com/gantt/legend.itemStyle
     */
    itemStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width for each legend item.
     * By default the items are laid out successively. In a horizontal layout,
     * if the items are laid out across two rows or more, they will be
     * vertically aligned depending on the legend.alignColumns option.
     *
     * @see https://api.highcharts.com/highcharts/legend.itemWidth
     * @see https://api.highcharts.com/highstock/legend.itemWidth
     * @see https://api.highcharts.com/highmaps/legend.itemWidth
     * @see https://api.highcharts.com/gantt/legend.itemWidth
     */
    itemWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Keyboard navigation for the
     * legend. Requires the Accessibility module.
     *
     * @see https://api.highcharts.com/highcharts/legend.keyboardNavigation
     * @see https://api.highcharts.com/highstock/legend.keyboardNavigation
     * @see https://api.highcharts.com/highmaps/legend.keyboardNavigation
     * @see https://api.highcharts.com/gantt/legend.keyboardNavigation
     */
    keyboardNavigation?: LegendKeyboardNavigationOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A format string for each legend
     * label. Available variables relates to properties on the series, or the
     * point in case of pies.
     *
     * @see https://api.highcharts.com/highcharts/legend.labelFormat
     * @see https://api.highcharts.com/highstock/legend.labelFormat
     * @see https://api.highcharts.com/highmaps/legend.labelFormat
     * @see https://api.highcharts.com/gantt/legend.labelFormat
     */
    labelFormat?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Callback function to format each
     * of the series' labels. The `this` keyword refers to the series object, or
     * the point object in case of pie charts. By default the series or point
     * name is printed.
     *
     * @see https://api.highcharts.com/highcharts/legend.labelFormatter
     * @see https://api.highcharts.com/highstock/legend.labelFormatter
     * @see https://api.highcharts.com/highmaps/legend.labelFormatter
     * @see https://api.highcharts.com/gantt/legend.labelFormatter
     */
    labelFormatter?: object;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The layout of the legend items.
     * Can be one of `horizontal` or `vertical` or `proximate`. When
     * `proximate`, the legend items will be placed as close as possible to the
     * graphs they're representing, except in inverted charts or when the legend
     * position doesn't allow it.
     *
     * @see https://api.highcharts.com/highcharts/legend.layout
     * @see https://api.highcharts.com/highstock/legend.layout
     * @see https://api.highcharts.com/highmaps/legend.layout
     * @see https://api.highcharts.com/gantt/legend.layout
     */
    layout?: ("horizontal"|"proximate"|"vertical");
    /**
     * (Highcharts, Gantt) Line height for the legend items. Deprecated as of
     * 2.1\. Instead, the line height for each item can be set using
     * itemStyle.lineHeight, and the padding between items using `itemMarginTop`
     * and `itemMarginBottom`.
     *
     * @see https://api.highcharts.com/highcharts/legend.lineHeight
     * @see https://api.highcharts.com/gantt/legend.lineHeight
     */
    lineHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) If the plot area sized is
     * calculated automatically and the legend is not floating, the legend
     * margin is the space between the legend and the axis labels or plot area.
     *
     * @see https://api.highcharts.com/highcharts/legend.margin
     * @see https://api.highcharts.com/highstock/legend.margin
     * @see https://api.highcharts.com/highmaps/legend.margin
     * @see https://api.highcharts.com/gantt/legend.margin
     */
    margin?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Maximum pixel height for the
     * legend. When the maximum height is extended, navigation will show.
     *
     * @see https://api.highcharts.com/highcharts/legend.maxHeight
     * @see https://api.highcharts.com/highstock/legend.maxHeight
     * @see https://api.highcharts.com/highmaps/legend.maxHeight
     * @see https://api.highcharts.com/gantt/legend.maxHeight
     */
    maxHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the paging or
     * navigation appearing when the legend is overflown. Navigation works well
     * on screen, but not in static exported images. One way of working around
     * that is to increase the chart height in export.
     *
     * @see https://api.highcharts.com/highcharts/legend.navigation
     * @see https://api.highcharts.com/highstock/legend.navigation
     * @see https://api.highcharts.com/highmaps/legend.navigation
     * @see https://api.highcharts.com/gantt/legend.navigation
     */
    navigation?: LegendNavigationOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The inner padding of the legend
     * box.
     *
     * @see https://api.highcharts.com/highcharts/legend.padding
     * @see https://api.highcharts.com/highstock/legend.padding
     * @see https://api.highcharts.com/highmaps/legend.padding
     * @see https://api.highcharts.com/gantt/legend.padding
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to reverse the order of
     * the legend items compared to the order of the series or points as defined
     * in the configuration object.
     *
     * @see https://api.highcharts.com/highcharts/legend.reversed
     * @see https://api.highcharts.com/highstock/legend.reversed
     * @see https://api.highcharts.com/highmaps/legend.reversed
     * @see https://api.highcharts.com/gantt/legend.reversed
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to show the symbol on
     * the right side of the text rather than the left side. This is common in
     * Arabic and Hebraic.
     *
     * @see https://api.highcharts.com/highcharts/legend.rtl
     * @see https://api.highcharts.com/highstock/legend.rtl
     * @see https://api.highcharts.com/highmaps/legend.rtl
     * @see https://api.highcharts.com/gantt/legend.rtl
     */
    rtl?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to apply a drop shadow
     * to the legend. A `backgroundColor` also needs to be applied for this to
     * take effect. The shadow can be an object configuration containing
     * `color`, `offsetX`, `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highcharts/legend.shadow
     * @see https://api.highcharts.com/highstock/legend.shadow
     * @see https://api.highcharts.com/highmaps/legend.shadow
     * @see https://api.highcharts.com/gantt/legend.shadow
     */
    shadow?: (boolean|CSSObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) When this is true, the legend
     * symbol width will be the same as the symbol height, which in turn
     * defaults to the font size of the legend items.
     *
     * @see https://api.highcharts.com/highcharts/legend.squareSymbol
     * @see https://api.highcharts.com/highstock/legend.squareSymbol
     * @see https://api.highcharts.com/highmaps/legend.squareSymbol
     * @see https://api.highcharts.com/gantt/legend.squareSymbol
     */
    squareSymbol?: boolean;
    /**
     * (Highcharts, Highstock) CSS styles for the legend area. In the 1.x
     * versions the position of the legend area was determined by CSS. In 2.x,
     * the position is determined by properties like `align`, `verticalAlign`,
     * `x` and `y`, but the styles are still parsed for backwards compatibility.
     *
     * @see https://api.highcharts.com/highcharts/legend.style
     * @see https://api.highcharts.com/highstock/legend.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel height of the symbol
     * for series types that use a rectangle in the legend. Defaults to the font
     * size of legend items.
     *
     * @see https://api.highcharts.com/highcharts/legend.symbolHeight
     * @see https://api.highcharts.com/highstock/legend.symbolHeight
     * @see https://api.highcharts.com/highmaps/legend.symbolHeight
     * @see https://api.highcharts.com/gantt/legend.symbolHeight
     */
    symbolHeight?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel padding between the
     * legend item symbol and the legend item text.
     *
     * @see https://api.highcharts.com/highcharts/legend.symbolPadding
     * @see https://api.highcharts.com/highstock/legend.symbolPadding
     * @see https://api.highcharts.com/highmaps/legend.symbolPadding
     * @see https://api.highcharts.com/gantt/legend.symbolPadding
     */
    symbolPadding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The border radius of the symbol
     * for series types that use a rectangle in the legend. Defaults to half the
     * `symbolHeight`.
     *
     * @see https://api.highcharts.com/highcharts/legend.symbolRadius
     * @see https://api.highcharts.com/highstock/legend.symbolRadius
     * @see https://api.highcharts.com/highmaps/legend.symbolRadius
     * @see https://api.highcharts.com/gantt/legend.symbolRadius
     */
    symbolRadius?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel width of the legend
     * item symbol. When the `squareSymbol` option is set, this defaults to the
     * `symbolHeight`, otherwise 16.
     *
     * @see https://api.highcharts.com/highcharts/legend.symbolWidth
     * @see https://api.highcharts.com/highstock/legend.symbolWidth
     * @see https://api.highcharts.com/highmaps/legend.symbolWidth
     * @see https://api.highcharts.com/gantt/legend.symbolWidth
     */
    symbolWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A title to be added on top of
     * the legend.
     *
     * @see https://api.highcharts.com/highcharts/legend.title
     * @see https://api.highcharts.com/highstock/legend.title
     * @see https://api.highcharts.com/highmaps/legend.title
     * @see https://api.highcharts.com/gantt/legend.title
     */
    title?: LegendTitleOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to use HTML to render
     * the legend item texts.
     *
     * Prior to 4.1.7, when using HTML, legend.navigation was disabled.
     *
     * @see https://api.highcharts.com/highcharts/legend.useHTML
     * @see https://api.highcharts.com/highstock/legend.useHTML
     * @see https://api.highcharts.com/highmaps/legend.useHTML
     * @see https://api.highcharts.com/gantt/legend.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical alignment of the
     * legend box. Can be one of `top`, `middle` or `bottom`. Vertical position
     * can be further determined by the `y` option.
     *
     * In the case that the legend is aligned in a corner position, the `layout`
     * option will determine whether to place it above/below or on the side of
     * the plot area.
     *
     * When the layout option is `proximate`, the `verticalAlign` option doesn't
     * apply.
     *
     * @see https://api.highcharts.com/highcharts/legend.verticalAlign
     * @see https://api.highcharts.com/highstock/legend.verticalAlign
     * @see https://api.highcharts.com/highmaps/legend.verticalAlign
     * @see https://api.highcharts.com/gantt/legend.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The width of the legend box. If
     * a number is set, it translates to pixels. Since v7.0.2 it allows setting
     * a percent string of the full chart width, for example `40%`.
     *
     * Defaults to the full chart width from legends below or above the chart,
     * half the chart width for legends to the left and right.
     *
     * @see https://api.highcharts.com/highcharts/legend.width
     * @see https://api.highcharts.com/highstock/legend.width
     * @see https://api.highcharts.com/highmaps/legend.width
     * @see https://api.highcharts.com/gantt/legend.width
     */
    width?: (number|string);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The x offset of the legend
     * relative to its horizontal alignment `align` within chart.spacingLeft and
     * chart.spacingRight. Negative x moves it to the left, positive x moves it
     * to the right.
     *
     * @see https://api.highcharts.com/highcharts/legend.x
     * @see https://api.highcharts.com/highstock/legend.x
     * @see https://api.highcharts.com/highmaps/legend.x
     * @see https://api.highcharts.com/gantt/legend.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * legend relative to it's vertical alignment `verticalAlign` within
     * chart.spacingTop and chart.spacingBottom. Negative y moves it up,
     * positive y moves it down.
     *
     * @see https://api.highcharts.com/highcharts/legend.y
     * @see https://api.highcharts.com/highstock/legend.y
     * @see https://api.highcharts.com/highmaps/legend.y
     * @see https://api.highcharts.com/gantt/legend.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A title to be added on top of the
 * legend.
 *
 * @see https://api.highcharts.com/highcharts/legend.title
 * @see https://api.highcharts.com/highstock/legend.title
 * @see https://api.highcharts.com/highmaps/legend.title
 * @see https://api.highcharts.com/gantt/legend.title
 */
export interface LegendTitleOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Generic CSS styles for the
     * legend title.
     *
     * @see https://api.highcharts.com/highcharts/legend.title.style
     * @see https://api.highcharts.com/highstock/legend.title.style
     * @see https://api.highcharts.com/highmaps/legend.title.style
     * @see https://api.highcharts.com/gantt/legend.title.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text or HTML string for the
     * title.
     *
     * @see https://api.highcharts.com/highcharts/legend.title.text
     * @see https://api.highcharts.com/highstock/legend.title.text
     * @see https://api.highcharts.com/highmaps/legend.title.text
     * @see https://api.highcharts.com/gantt/legend.title.text
     */
    text?: string;
}
/**
 * Defines the start position and the end position for a gradient relative to
 * the shape. Start position (x1, y1) and end position (x2, y2) are relative to
 * the shape, where 0 means top/left and 1 is bottom/right.
 */
export interface LinearGradientColorObject {
    /**
     * Start horizontal position of the gradient. Float ranges 0-1.
     */
    x1: number;
    /**
     * End horizontal position of the gradient. Float ranges 0-1.
     */
    x2: number;
    /**
     * Start vertical position of the gradient. Float ranges 0-1.
     */
    y1: number;
    /**
     * End vertical position of the gradient. Float ranges 0-1.
     */
    y2: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) The loading options control the
 * appearance of the loading screen that covers the plot area on chart
 * operations. This screen only appears after an explicit call to
 * `chart.showLoading()`. It is a utility for developers to communicate to the
 * end user that something is going on, for example while retrieving new data
 * via an XHR connection. The "Loading..." text itself is not part of this
 * configuration object, but part of the `lang` object.
 *
 * @see https://api.highcharts.com/highcharts/loading
 * @see https://api.highcharts.com/highstock/loading
 * @see https://api.highcharts.com/highmaps/loading
 * @see https://api.highcharts.com/gantt/loading
 */
export interface LoadingOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The duration in milliseconds of
     * the fade out effect.
     *
     * @see https://api.highcharts.com/highcharts/loading.hideDuration
     * @see https://api.highcharts.com/highstock/loading.hideDuration
     * @see https://api.highcharts.com/highmaps/loading.hideDuration
     * @see https://api.highcharts.com/gantt/loading.hideDuration
     */
    hideDuration?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the loading label
     * `span`.
     *
     * @see https://api.highcharts.com/highcharts/loading.labelStyle
     * @see https://api.highcharts.com/highstock/loading.labelStyle
     * @see https://api.highcharts.com/highmaps/loading.labelStyle
     * @see https://api.highcharts.com/gantt/loading.labelStyle
     */
    labelStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The duration in milliseconds of
     * the fade in effect.
     *
     * @see https://api.highcharts.com/highcharts/loading.showDuration
     * @see https://api.highcharts.com/highstock/loading.showDuration
     * @see https://api.highcharts.com/highmaps/loading.showDuration
     * @see https://api.highcharts.com/gantt/loading.showDuration
     */
    showDuration?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the loading
     * screen that covers the plot area.
     *
     * In styled mode, the loading label is styled with the
     * `.highcharts-loading` class.
     *
     * @see https://api.highcharts.com/highcharts/loading.style
     * @see https://api.highcharts.com/highstock/loading.style
     * @see https://api.highcharts.com/highmaps/loading.style
     * @see https://api.highcharts.com/gantt/loading.style
     */
    style?: CSSObject;
}
/**
 * Map data object.
 */
export interface MapDataObject {
    /**
     * The name of the data.
     */
    name?: string;
    /**
     * The SVG path.
     */
    path: SVGPathArray;
    /**
     * The GeoJSON meta data.
     */
    properties?: object;
}
/**
 * (Highmaps) General options for the map navigation buttons. Individual options
 * can be given from the mapNavigation.buttons option set.
 *
 * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions
 */
export interface MapNavigationButtonOptions {
    /**
     * (Highmaps) The alignment of the navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.align
     */
    align?: AlignType;
    /**
     * (Highmaps) What box to align the buttons to. Possible values are
     * `plotBox` and `spacingBox`.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.alignTo
     */
    alignTo?: ("plotBox"|"spacingBox");
    /**
     * (Highmaps) The pixel height of the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.height
     */
    height?: number;
    /**
     * (Highmaps) Padding for the navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.padding
     */
    padding?: number;
    /**
     * (Highmaps) Text styles for the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.style
     */
    style?: CSSObject;
    /**
     * (Highmaps) A configuration object for the button theme. The object
     * accepts SVG properties like `stroke-width`, `stroke` and `fill`.
     * Tri-state button styles are supported by the `states.hover` and
     * `states.select` objects.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.theme
     */
    theme?: SVGAttributes;
    /**
     * (Highmaps) The vertical alignment of the buttons. Individual alignment
     * can be adjusted by each button's `y` offset.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highmaps) The width of the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.width
     */
    width?: number;
    /**
     * (Highmaps) The X offset of the buttons relative to its `align` setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions.x
     */
    x?: number;
}
/**
 * (Highmaps) The individual buttons for the map navigation. This usually
 * includes the zoom in and zoom out buttons. Properties for each button is
 * inherited from mapNavigation.buttonOptions, while individual options can be
 * overridden. But default, the `onclick`, `text` and `y` options are
 * individual.
 *
 * @see https://api.highcharts.com/highmaps/mapNavigation.buttons
 */
export interface MapNavigationButtonsOptions {
    /**
     * (Highmaps) Options for the zoom in button. Properties for the zoom in and
     * zoom out buttons are inherited from mapNavigation.buttonOptions, while
     * individual options can be overridden. By default, the `onclick`, `text`
     * and `y` options are individual.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn
     */
    zoomIn?: MapNavigationButtonsZoomInOptions;
    /**
     * (Highmaps) Options for the zoom out button. Properties for the zoom in
     * and zoom out buttons are inherited from mapNavigation.buttonOptions,
     * while individual options can be overridden. By default, the `onclick`,
     * `text` and `y` options are individual.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut
     */
    zoomOut?: MapNavigationButtonsZoomOutOptions;
}
/**
 * (Highmaps) Options for the zoom in button. Properties for the zoom in and
 * zoom out buttons are inherited from mapNavigation.buttonOptions, while
 * individual options can be overridden. By default, the `onclick`, `text` and
 * `y` options are individual.
 *
 * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn
 */
export interface MapNavigationButtonsZoomInOptions {
    /**
     * (Highmaps) The alignment of the navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.align
     */
    align?: AlignType;
    /**
     * (Highmaps) What box to align the buttons to. Possible values are
     * `plotBox` and `spacingBox`.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.alignTo
     */
    alignTo?: ("plotBox"|"spacingBox");
    /**
     * (Highmaps) The pixel height of the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.height
     */
    height?: number;
    /**
     * (Highmaps) Click handler for the button.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.onclick
     */
    onclick?: () => void;
    /**
     * (Highmaps) Padding for the navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.padding
     */
    padding?: number;
    /**
     * (Highmaps) Text styles for the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.style
     */
    style?: CSSObject;
    /**
     * (Highmaps) The text for the button. The tooltip (title) is a language
     * option given by lang.zoomIn.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.text
     */
    text?: string;
    /**
     * (Highmaps) A configuration object for the button theme. The object
     * accepts SVG properties like `stroke-width`, `stroke` and `fill`.
     * Tri-state button styles are supported by the `states.hover` and
     * `states.select` objects.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.theme
     */
    theme?: SVGAttributes;
    /**
     * (Highmaps) The vertical alignment of the buttons. Individual alignment
     * can be adjusted by each button's `y` offset.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highmaps) The width of the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.width
     */
    width?: number;
    /**
     * (Highmaps) The X offset of the buttons relative to its `align` setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.x
     */
    x?: number;
    /**
     * (Highmaps) The position of the zoomIn button relative to the vertical
     * alignment.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomIn.y
     */
    y?: number;
}
/**
 * (Highmaps) Options for the zoom out button. Properties for the zoom in and
 * zoom out buttons are inherited from mapNavigation.buttonOptions, while
 * individual options can be overridden. By default, the `onclick`, `text` and
 * `y` options are individual.
 *
 * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut
 */
export interface MapNavigationButtonsZoomOutOptions {
    /**
     * (Highmaps) The alignment of the navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.align
     */
    align?: AlignType;
    /**
     * (Highmaps) What box to align the buttons to. Possible values are
     * `plotBox` and `spacingBox`.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.alignTo
     */
    alignTo?: ("plotBox"|"spacingBox");
    /**
     * (Highmaps) The pixel height of the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.height
     */
    height?: number;
    /**
     * (Highmaps) Click handler for the button.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.onclick
     */
    onclick?: () => void;
    /**
     * (Highmaps) Padding for the navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.padding
     */
    padding?: number;
    /**
     * (Highmaps) Text styles for the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.style
     */
    style?: CSSObject;
    /**
     * (Highmaps) The text for the button. The tooltip (title) is a language
     * option given by lang.zoomOut.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.text
     */
    text?: string;
    /**
     * (Highmaps) A configuration object for the button theme. The object
     * accepts SVG properties like `stroke-width`, `stroke` and `fill`.
     * Tri-state button styles are supported by the `states.hover` and
     * `states.select` objects.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.theme
     */
    theme?: SVGAttributes;
    /**
     * (Highmaps) The vertical alignment of the buttons. Individual alignment
     * can be adjusted by each button's `y` offset.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highmaps) The width of the map navigation buttons.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.width
     */
    width?: number;
    /**
     * (Highmaps) The X offset of the buttons relative to its `align` setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.x
     */
    x?: number;
    /**
     * (Highmaps) The position of the zoomOut button relative to the vertical
     * alignment.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons.zoomOut.y
     */
    y?: number;
}
export interface MapNavigationOptions {
    /**
     * (Highmaps) General options for the map navigation buttons. Individual
     * options can be given from the mapNavigation.buttons option set.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttonOptions
     */
    buttonOptions?: MapNavigationButtonOptions;
    /**
     * (Highmaps) The individual buttons for the map navigation. This usually
     * includes the zoom in and zoom out buttons. Properties for each button is
     * inherited from mapNavigation.buttonOptions, while individual options can
     * be overridden. But default, the `onclick`, `text` and `y` options are
     * individual.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.buttons
     */
    buttons?: MapNavigationButtonsOptions;
    /**
     * (Highmaps) Whether to enable navigation buttons. By default it inherits
     * the enabled setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.enableButtons
     */
    enableButtons?: boolean;
    /**
     * (Highmaps) Whether to enable map navigation. The default is not to enable
     * navigation, as many choropleth maps are simple and don't need it.
     * Additionally, when touch zoom and mousewheel zoom is enabled, it breaks
     * the default behaviour of these interactions in the website, and the
     * implementer should be aware of this.
     *
     * Individual interactions can be enabled separately, namely buttons,
     * multitouch zoom, double click zoom, double click zoom to element and
     * mousewheel zoom.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.enabled
     */
    enabled?: boolean;
    /**
     * (Highmaps) Enables zooming in on an area on double clicking in the map.
     * By default it inherits the enabled setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.enableDoubleClickZoom
     */
    enableDoubleClickZoom?: boolean;
    /**
     * (Highmaps) Whether to zoom in on an area when that area is double
     * clicked.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.enableDoubleClickZoomTo
     */
    enableDoubleClickZoomTo?: boolean;
    /**
     * (Highmaps) Enables zooming by mouse wheel. By default it inherits the
     * enabled setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.enableMouseWheelZoom
     */
    enableMouseWheelZoom?: boolean;
    /**
     * (Highmaps) Whether to enable multitouch zooming. Note that if the chart
     * covers the viewport, this prevents the user from using multitouch and
     * touchdrag on the web page, so you should make sure the user is not
     * trapped inside the chart. By default it inherits the enabled setting.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.enableTouchZoom
     */
    enableTouchZoom?: boolean;
    /**
     * (Highmaps) Sensitivity of mouse wheel or trackpad scrolling. 1 is no
     * sensitivity, while with 2, one mousewheel delta will zoom in 50%.
     *
     * @see https://api.highcharts.com/highmaps/mapNavigation.mouseWheelSensitivity
     */
    mouseWheelSensitivity?: number;
}
/**
 * (Highcharts, Highstock) Bindings definitions for custom HTML buttons. Each
 * binding implements simple event-driven interface:
 *
 * - `className`: classname used to bind event to
 *
 * - `init`: initial event, fired on button click
 *
 * - `start`: fired on first click on a chart
 *
 * - `steps`: array of sequential events fired one after another on each of
 * users clicks
 *
 * - `end`: last event to be called after last step event
 *
 * @see https://api.highcharts.com/highcharts/navigation.bindings
 * @see https://api.highcharts.com/highstock/navigation.bindings
 */
export interface NavigationBindingsOptions {
    /**
     * (Highstock) A line with arrow annotation. Includes `start` and one event
     * in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.arrowInfinityLine
     */
    arrowInfinityLine?: StockToolsBindingsObject;
    /**
     * (Highstock) A ray with an arrow annotation bindings. Includes `start` and
     * one event in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.arrowRay
     */
    arrowRay?: StockToolsBindingsObject;
    /**
     * (Highstock) A segment with an arrow annotation bindings. Includes `start`
     * and one event in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.arrowSegment
     */
    arrowSegment?: StockToolsBindingsObject;
    /**
     * (Highcharts, Highstock) A circle annotation bindings. Includes `start`
     * and one event in `steps` array.
     *
     * @see https://api.highcharts.com/highcharts/navigation.bindings.circleAnnotation
     * @see https://api.highcharts.com/highstock/navigation.bindings.circleAnnotation
     */
    circleAnnotation?: StockToolsBindingsObject;
    /**
     * (Highstock) Crooked line (three points) annotation bindings. Includes
     * `start` and two events in `steps` (for second and third points in crooked
     * line) array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.crooked3
     */
    crooked3?: StockToolsBindingsObject;
    /**
     * (Highstock) Crooked line (five points) annotation bindings. Includes
     * `start` and four events in `steps` (for all consequent points in crooked
     * line) array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.crooked5
     */
    crooked5?: StockToolsBindingsObject;
    /**
     * (Highstock) Hides/shows two price indicators:
     *
     * - last price in the dataset
     *
     * - last price in the selected range
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.currentPriceIndicator
     */
    currentPriceIndicator?: StockToolsBindingsObject;
    /**
     * (Highstock) Elliott wave (three points) annotation bindings. Includes
     * `start` and two events in `steps` (for second and third points) array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.elliott3
     */
    elliott3?: StockToolsBindingsObject;
    /**
     * (Highstock) Elliott wave (five points) annotation bindings. Includes
     * `start` and four event in `steps` (for all consequent points in Elliott
     * wave) array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.elliott5
     */
    elliott5?: StockToolsBindingsObject;
    /**
     * (Highstock) A fibonacci annotation bindings. Includes `start` and two
     * events in `steps` array (updates second point, then height).
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.fibonacci
     */
    fibonacci?: StockToolsBindingsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag with `'circlepin'`
     * shape.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.flagCirclepin
     */
    flagCirclepin?: StockToolsBindingsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag with `'diamondpin'`
     * shape.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.flagDiamondpin
     */
    flagDiamondpin?: StockToolsBindingsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag without pin shape.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.flagSimplepin
     */
    flagSimplepin?: StockToolsBindingsObject;
    /**
     * (Highstock) A flag series bindings. Includes `start` event. On click,
     * finds the closest point and marks it with a flag with `'squarepin'`
     * shape.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.flagSquarepin
     */
    flagSquarepin?: StockToolsBindingsObject;
    /**
     * (Highstock) Displays chart in fullscreen.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.fullScreen
     */
    fullScreen?: StockToolsBindingsObject;
    /**
     * (Highstock) A horizontal line annotation. Includes `start` event.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.horizontalLine
     */
    horizontalLine?: StockToolsBindingsObject;
    /**
     * (Highstock) Indicators bindings. Includes `init` event to show a popup.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.indicators
     */
    indicators?: StockToolsBindingsObject;
    /**
     * (Highstock) A line annotation. Includes `start` and one event in `steps`
     * array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.infinityLine
     */
    infinityLine?: StockToolsBindingsObject;
    /**
     * (Highcharts, Highstock) A label annotation bindings. Includes `start`
     * event only.
     *
     * @see https://api.highcharts.com/highcharts/navigation.bindings.labelAnnotation
     * @see https://api.highcharts.com/highstock/navigation.bindings.labelAnnotation
     */
    labelAnnotation?: StockToolsBindingsObject;
    /**
     * (Highstock) A measure (x-dimension) annotation bindings. Includes `start`
     * and one event in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.measureX
     */
    measureX?: StockToolsBindingsObject;
    /**
     * (Highstock) A measure (xy-dimension) annotation bindings. Includes
     * `start` and one event in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.measureXY
     */
    measureXY?: StockToolsBindingsObject;
    /**
     * (Highstock) A measure (y-dimension) annotation bindings. Includes `start`
     * and one event in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.measureY
     */
    measureY?: StockToolsBindingsObject;
    /**
     * (Highstock) A parallel channel (tunnel) annotation bindings. Includes
     * `start` and two events in `steps` array (updates second point, then
     * height).
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.parallelChannel
     */
    parallelChannel?: StockToolsBindingsObject;
    /**
     * (Highstock) An Andrew's pitchfork annotation bindings. Includes `start`
     * and two events in `steps` array (sets second and third control points).
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.pitchfork
     */
    pitchfork?: StockToolsBindingsObject;
    /**
     * (Highstock) A ray annotation bindings. Includes `start` and one event in
     * `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.ray
     */
    ray?: StockToolsBindingsObject;
    /**
     * (Highcharts, Highstock) A rectangle annotation bindings. Includes `start`
     * and one event in `steps` array.
     *
     * @see https://api.highcharts.com/highcharts/navigation.bindings.rectangleAnnotation
     * @see https://api.highcharts.com/highstock/navigation.bindings.rectangleAnnotation
     */
    rectangleAnnotation?: StockToolsBindingsObject;
    /**
     * (Highstock) Save a chart in localStorage under `highcharts-chart` key.
     * Stored items:
     *
     * - annotations
     *
     * - indicators (with yAxes)
     *
     * - flags
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.saveChart
     */
    saveChart?: StockToolsBindingsObject;
    /**
     * (Highstock) A segment annotation bindings. Includes `start` and one event
     * in `steps` array.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.segment
     */
    segment?: StockToolsBindingsObject;
    /**
     * (Highstock) Changes main series to `'candlestick'` type.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.seriesTypeCandlestick
     */
    seriesTypeCandlestick?: StockToolsBindingsObject;
    /**
     * (Highstock) Changes main series to `'line'` type.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.seriesTypeLine
     */
    seriesTypeLine?: StockToolsBindingsObject;
    /**
     * (Highstock) Changes main series to `'ohlc'` type.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.seriesTypeOhlc
     */
    seriesTypeOhlc?: StockToolsBindingsObject;
    /**
     * (Highstock) Hides/shows all annotations on a chart.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.toggleAnnotations
     */
    toggleAnnotations?: StockToolsBindingsObject;
    /**
     * (Highstock) A vertical arrow annotation bindings. Includes `start` event.
     * On click, finds the closest point and marks it with an arrow. Green arrow
     * when pointing from above, red when pointing from below the point.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.verticalArrow
     */
    verticalArrow?: StockToolsBindingsObject;
    /**
     * (Highstock) A vertical counter annotation bindings. Includes `start`
     * event. On click, finds the closest point and marks it with a numeric
     * annotation - incrementing counter on each add.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.verticalCounter
     */
    verticalCounter?: StockToolsBindingsObject;
    /**
     * (Highstock) A vertical arrow annotation bindings. Includes `start` event.
     * On click, finds the closest point and marks it with an arrow and a label
     * with value.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.verticalLabel
     */
    verticalLabel?: StockToolsBindingsObject;
    /**
     * (Highstock) A vertical line annotation. Includes `start` event.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.verticalLine
     */
    verticalLine?: StockToolsBindingsObject;
    /**
     * (Highstock) Enables zooming in xAxis on a chart. Includes `start` event
     * which changes chart.zoomType.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.zoomX
     */
    zoomX?: StockToolsBindingsObject;
    /**
     * (Highstock) Enables zooming in xAxis and yAxis on a chart. Includes
     * `start` event which changes chart.zoomType.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.zoomXY
     */
    zoomXY?: StockToolsBindingsObject;
    /**
     * (Highstock) Enables zooming in yAxis on a chart. Includes `start` event
     * which changes chart.zoomType.
     *
     * @see https://api.highcharts.com/highstock/navigation.bindings.zoomY
     */
    zoomY?: StockToolsBindingsObject;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for buttons
 * appearing in the exporting module.
 *
 * In styled mode, the buttons are styled with the `.highcharts-contextbutton`
 * and `.highcharts-button-symbol` classes.
 *
 * @see https://api.highcharts.com/highcharts/navigation.buttonOptions
 * @see https://api.highcharts.com/highstock/navigation.buttonOptions
 * @see https://api.highcharts.com/highmaps/navigation.buttonOptions
 * @see https://api.highcharts.com/gantt/navigation.buttonOptions
 */
export interface NavigationButtonOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Whether to enable buttons.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.enabled
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.enabled
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.enabled
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Fill color for the symbol within
     * the button.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.symbolFill
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.symbolFill
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.symbolFill
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.symbolFill
     */
    symbolFill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The color of the symbol's stroke
     * or line.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.symbolStroke
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.symbolStroke
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.symbolStroke
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.symbolStroke
     */
    symbolStroke?: ColorString;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The pixel stroke width of the
     * symbol on the button.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.symbolStrokeWidth
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.symbolStrokeWidth
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.symbolStrokeWidth
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.symbolStrokeWidth
     */
    symbolStrokeWidth?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A text string to add to the
     * individual button.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.text
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.text
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.text
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
     * button theme. The object accepts SVG properties like `stroke-width`,
     * `stroke` and `fill`. Tri-state button styles are supported by the
     * `states.hover` and `states.select` objects.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.theme
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.theme
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.theme
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.theme
     */
    theme?: NavigationButtonThemeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The vertical offset of the
     * button's position relative to its `verticalAlign`.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.y
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.y
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.y
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A configuration object for the
 * button theme. The object accepts SVG properties like `stroke-width`, `stroke`
 * and `fill`. Tri-state button styles are supported by the `states.hover` and
 * `states.select` objects.
 *
 * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.theme
 * @see https://api.highcharts.com/highstock/navigation.buttonOptions.theme
 * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.theme
 * @see https://api.highcharts.com/gantt/navigation.buttonOptions.theme
 */
export interface NavigationButtonThemeOptions {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The default fill exists only to
     * capture hover events.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.theme.fill
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.theme.fill
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.theme.fill
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.theme.fill
     */
    fill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Padding for the button.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.theme.padding
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.theme.padding
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.theme.padding
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.theme.padding
     */
    padding?: number;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Default stroke for the buttons.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions.theme.stroke
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions.theme.stroke
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions.theme.stroke
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions.theme.stroke
     */
    stroke?: ColorString;
}
/**
 * (Highcharts, Highstock) Events to communicate between Stock Tools and custom
 * GUI.
 *
 * @see https://api.highcharts.com/highcharts/navigation.events
 * @see https://api.highcharts.com/highstock/navigation.events
 */
export interface NavigationEventsOptions {
    /**
     * (Highcharts, Highstock) Event fired when button state should change, for
     * example after adding an annotation.
     *
     * @see https://api.highcharts.com/highcharts/navigation.events.deselectButton
     * @see https://api.highcharts.com/highstock/navigation.events.deselectButton
     */
    deselectButton?: () => void;
    /**
     * (Highcharts, Highstock) A `hidePopop` event. Fired when Popup should be
     * hidden, for exampole when clicking on an annotation again.
     *
     * @see https://api.highcharts.com/highcharts/navigation.events.hidePopup
     * @see https://api.highcharts.com/highstock/navigation.events.hidePopup
     */
    hidePopup?: () => void;
    /**
     * (Highcharts, Highstock) Event fired on a button click.
     *
     * @see https://api.highcharts.com/highcharts/navigation.events.selectButton
     * @see https://api.highcharts.com/highstock/navigation.events.selectButton
     */
    selectButton?: () => void;
    /**
     * (Highcharts, Highstock) A `showPopup` event. Fired when selecting for
     * example an annotation.
     *
     * @see https://api.highcharts.com/highcharts/navigation.events.showPopup
     * @see https://api.highcharts.com/highstock/navigation.events.showPopup
     */
    showPopup?: () => void;
}
/**
 * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for buttons
 * and menus appearing in the exporting module.
 *
 * @see https://api.highcharts.com/highcharts/navigation
 * @see https://api.highcharts.com/highstock/navigation
 * @see https://api.highcharts.com/highmaps/navigation
 * @see https://api.highcharts.com/gantt/navigation
 */
export interface NavigationOptions {
    /**
     * (Highcharts, Highstock) Bindings definitions for custom HTML buttons.
     * Each binding implements simple event-driven interface:
     *
     * - `className`: classname used to bind event to
     *
     * - `init`: initial event, fired on button click
     *
     * - `start`: fired on first click on a chart
     *
     * - `steps`: array of sequential events fired one after another on each of
     * users clicks
     *
     * - `end`: last event to be called after last step event
     *
     * @see https://api.highcharts.com/highcharts/navigation.bindings
     * @see https://api.highcharts.com/highstock/navigation.bindings
     */
    bindings?: (NavigationBindingsOptions|Dictionary<StockToolsBindingsObject>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A CSS class name where all
     * bindings will be attached to. Multiple charts on the same page should
     * have separate class names to prevent duplicating events.
     *
     * @see https://api.highcharts.com/highcharts/navigation.bindingsClassName
     * @see https://api.highcharts.com/highstock/navigation.bindingsClassName
     * @see https://api.highcharts.com/highmaps/navigation.bindingsClassName
     * @see https://api.highcharts.com/gantt/navigation.bindingsClassName
     */
    bindingsClassName?: string;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for
     * buttons appearing in the exporting module.
     *
     * In styled mode, the buttons are styled with the
     * `.highcharts-contextbutton` and `.highcharts-button-symbol` classes.
     *
     * @see https://api.highcharts.com/highcharts/navigation.buttonOptions
     * @see https://api.highcharts.com/highstock/navigation.buttonOptions
     * @see https://api.highcharts.com/highmaps/navigation.buttonOptions
     * @see https://api.highcharts.com/gantt/navigation.buttonOptions
     */
    buttonOptions?: NavigationButtonOptions;
    /**
     * (Highcharts, Highstock) Events to communicate between Stock Tools and
     * custom GUI.
     *
     * @see https://api.highcharts.com/highcharts/navigation.events
     * @see https://api.highcharts.com/highstock/navigation.events
     */
    events?: NavigationEventsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the hover state
     * of the individual items within the popup menu appearing by default when
     * the export icon is clicked. The menu items are rendered in HTML.
     *
     * @see https://api.highcharts.com/highcharts/navigation.menuItemHoverStyle
     * @see https://api.highcharts.com/highstock/navigation.menuItemHoverStyle
     * @see https://api.highcharts.com/highmaps/navigation.menuItemHoverStyle
     * @see https://api.highcharts.com/gantt/navigation.menuItemHoverStyle
     */
    menuItemHoverStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the individual
     * items within the popup menu appearing by default when the export icon is
     * clicked. The menu items are rendered in HTML. Font size defaults to
     * `11px` on desktop and `14px` on touch devices.
     *
     * @see https://api.highcharts.com/highcharts/navigation.menuItemStyle
     * @see https://api.highcharts.com/highstock/navigation.menuItemStyle
     * @see https://api.highcharts.com/highmaps/navigation.menuItemStyle
     * @see https://api.highcharts.com/gantt/navigation.menuItemStyle
     */
    menuItemStyle?: CSSObject;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) CSS styles for the popup menu
     * appearing by default when the export icon is clicked. This menu is
     * rendered in HTML.
     *
     * @see https://api.highcharts.com/highcharts/navigation.menuStyle
     * @see https://api.highcharts.com/highstock/navigation.menuStyle
     * @see https://api.highcharts.com/highmaps/navigation.menuStyle
     * @see https://api.highcharts.com/gantt/navigation.menuStyle
     */
    menuStyle?: CSSObject;
}
/**
 * (Highstock) Options for the handles for dragging the zoomed area.
 *
 * @see https://api.highcharts.com/highstock/navigator.handles
 */
export interface NavigatorHandlesOptions {
    /**
     * (Highstock) The fill for the handle.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The stroke for the handle border and the stripes inside.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) Allows to enable/disable handles.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) Height for handles.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.height
     */
    height?: number;
    /**
     * (Highstock) The width for the handle border and the stripes inside.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Array to define shapes of handles. 0-index for left, 1-index
     * for right.
     *
     * Additionally, the URL to a graphic can be given on this form:
     * `url(graphic.png)`. Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.symbols
     */
    symbols?: Array<string>;
    /**
     * (Highstock) Width for handles.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles.width
     */
    width?: number;
}
/**
 * (Highstock) The navigator is a small series below the main series, displaying
 * a view of the entire data set. It provides tools to zoom in and out on parts
 * of the data as well as panning across the dataset.
 *
 * @see https://api.highcharts.com/highstock/navigator
 */
export interface NavigatorOptions {
    /**
     * (Highstock) Whether the navigator and scrollbar should adapt to updated
     * data in the base X axis. When loading data async, as in the demo below,
     * this should be `false`. Otherwise new data will trigger navigator redraw,
     * which will cause unwanted looping. In the demo below, the data in the
     * navigator is set only once. On navigating, only the main chart content is
     * updated.
     *
     * @see https://api.highcharts.com/highstock/navigator.adaptToUpdatedData
     */
    adaptToUpdatedData?: boolean;
    /**
     * (Highstock) An integer identifying the index to use for the base series,
     * or a string representing the id of the series.
     *
     * **Note**: As of Highcharts 5.0, this is now a deprecated option. Prefer
     * series.showInNavigator.
     *
     * @see https://api.highcharts.com/highstock/navigator.baseSeries
     */
    baseSeries?: any;
    /**
     * (Highstock) Enable or disable the navigator.
     *
     * @see https://api.highcharts.com/highstock/navigator.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) Options for the handles for dragging the zoomed area.
     *
     * @see https://api.highcharts.com/highstock/navigator.handles
     */
    handles?: NavigatorHandlesOptions;
    /**
     * (Highstock) The height of the navigator.
     *
     * @see https://api.highcharts.com/highstock/navigator.height
     */
    height?: number;
    /**
     * (Highstock) The distance from the nearest element, the X axis or X axis
     * labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.margin
     */
    margin?: number;
    /**
     * (Highstock) The color of the mask covering the areas of the navigator
     * series that are currently not visible in the main series. The default
     * color is bluish with an opacity of 0.3 to see the series below.
     *
     * @see https://api.highcharts.com/highstock/navigator.maskFill
     */
    maskFill?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Whether the mask should be inside the range marking the
     * zoomed range, or outside. In Highstock 1.x it was always `false`.
     *
     * @see https://api.highcharts.com/highstock/navigator.maskInside
     */
    maskInside?: boolean;
    /**
     * (Highstock) When the chart is inverted, whether to draw the navigator on
     * the opposite side.
     *
     * @see https://api.highcharts.com/highstock/navigator.opposite
     */
    opposite?: boolean;
    /**
     * (Highstock) The color of the line marking the currently zoomed area in
     * the navigator.
     *
     * @see https://api.highcharts.com/highstock/navigator.outlineColor
     */
    outlineColor?: ColorString;
    /**
     * (Highstock) The width of the line marking the currently zoomed area in
     * the navigator.
     *
     * @see https://api.highcharts.com/highstock/navigator.outlineWidth
     */
    outlineWidth?: number;
    /**
     * (Highstock) Options for the navigator series. Available options are the
     * same as any series, documented at plotOptions and series.
     *
     * Unless data is explicitly defined on navigator.series, the data is
     * borrowed from the first series in the chart.
     *
     * Default series options for the navigator series are:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/navigator.series
     */
    series?: NavigatorSeriesOptions;
    /**
     * (Highstock) Options for the navigator X axis. Default series options for
     * the navigator xAxis are:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis
     */
    xAxis?: NavigatorXAxisOptions;
    /**
     * (Highstock) Options for the navigator Y axis. Default series options for
     * the navigator yAxis are:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis
     */
    yAxis?: NavigatorYAxisOptions;
}
/**
 * (Highstock) Data grouping options for the navigator series.
 *
 * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping
 */
export interface NavigatorSeriesDataGroupingOptions {
    /**
     * (Highstock) The method of approximation inside a group. When for example
     * 30 days are grouped into one month, this determines what value should
     * represent the group. Possible values are "average", "averages", "open",
     * "high", "low", "close" and "sum". For OHLC and candlestick series the
     * approximation is "ohlc" by default, which finds the open, high, low and
     * close values within all the grouped data. For ranges, the approximation
     * is "range", which finds the low and high values. For multi-dimensional
     * data, like ranges and OHLC, "averages" will compute the average for each
     * dimension.
     *
     * Custom aggregate methods can be added by assigning a callback function as
     * the approximation. This function takes a numeric array as the argument
     * and should return a single numeric value or `null`. Note that the numeric
     * array will never contain null values, only true numbers. Instead, if null
     * values are present in the raw data, the numeric array will have an
     * `.hasNulls` property set to `true`. For single-value data sets the data
     * is available in the first argument of the callback function. For OHLC
     * data sets, all the open values are in the first argument, all high values
     * in the second etc.
     *
     * Since v4.2.7, grouping meta data is available in the approximation
     * callback from `this.dataGroupInfo`. It can be used to extract information
     * from the raw data.
     *
     * Defaults to `average` for line-type series, `sum` for columns, `range`
     * for range series and `ohlc` for OHLC and candlestick.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.approximation
     */
    approximation?: ("average"|"averages"|"close"|"high"|"low"|"open"|"sum");
    /**
     * (Highstock) Datetime formats for the header of the tooltip in a stock
     * chart. The format can vary within a chart depending on the currently
     * selected time range and the current data grouping.
     *
     * The default formats are:
     *
     * (see online documentation for example)
     *
     * For each of these array definitions, the first item is the format used
     * when the active time span is one unit. For instance, if the current data
     * applies to one week, the first item of the week array is used. The second
     * and third items are used when the active time span is more than two
     * units. For instance, if the current data applies to two weeks, the second
     * and third item of the week array are used, and applied to the start and
     * end date of the time span.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: object;
    /**
     * (Highstock) Enable or disable data grouping.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) When data grouping is forced, it runs no matter how small the
     * intervals are. This can be handy for example when the sum should be
     * calculated for values appearing at random times within each hour.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.forced
     */
    forced?: boolean;
    /**
     * (Highstock) By default only points within the visible range are grouped.
     * Enabling this option will force data grouping to calculate all grouped
     * points for a given dataset. That option prevents for example a column
     * series from calculating a grouped point partially. The effect is similar
     * to Series.getExtremesFromAll but does not affect yAxis extremes.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.groupAll
     */
    groupAll?: boolean;
    /**
     * (Highstock) The approximate pixel width of each group. If for example a
     * series with 30 points is displayed over a 600 pixel wide plot area, no
     * grouping is performed. If however the series contains so many points that
     * the spacing is less than the groupPixelWidth, Highcharts will try to
     * group it into appropriate groups so that each is more or less two pixels
     * wide. If multiple series with different group pixel widths are drawn on
     * the same x axis, all series will take the greatest width. For example,
     * line series have 2px default group width, while column series have 10px.
     * If combined, both the line and the column will have 10px by default.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.groupPixelWidth
     */
    groupPixelWidth?: number;
    /**
     * (Highstock) Normally, a group is indexed by the start of that group, so
     * for example when 30 daily values are grouped into one month, that month's
     * x value will be the 1st of the month. This apparently shifts the data to
     * the left. When the smoothed option is true, this is compensated for. The
     * data is shifted to the middle of the group, and min and max values are
     * preserved. Internally, this is used in the Navigator series.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.smoothed
     */
    smoothed?: boolean;
    /**
     * (Highstock) An array determining what time intervals the data is allowed
     * to be grouped to. Each array item is an array where the first value is
     * the time unit and the second value another array of allowed multiples.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping.units
     */
    units?: Array<[string, (Array<number>|null)]>;
}
/**
 * (Highstock) A declarative filter for which data labels to display. The
 * declarative filter is designed for use when callback functions are not
 * available, like when the chart options require a pure JSON structure or for
 * use with graphical editors. For programmatic control, use the `formatter`
 * instead, and return `undefined` to disable a single data label.
 *
 * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.filter
 */
export interface NavigatorSeriesDataLabelsFilterOptions {
    /**
     * (Highstock) The operator to compare by. Can be one of `>`, `<`, `>=`,
     * `<=`, `==`, and `===`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.filter.operator
     */
    operator?: ("=="|"==="|">"|">="|"<"|"<=");
    /**
     * (Highstock) The point property to filter by. Point options are passed
     * directly to properties, additionally there are `y` value, `percentage`
     * and others listed under Point members.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.filter.property
     */
    property?: string;
    /**
     * (Highstock) The value to compare against.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.filter.value
     */
    value?: any;
}
/**
 * (Highstock) Data label options for the navigator series. Data labels are
 * disabled by default on the navigator series.
 *
 * @see https://api.highcharts.com/highstock/navigator.series.dataLabels
 */
export interface NavigatorSeriesDataLabelsOptions {
    /**
     * (Highstock) The alignment of the data label compared to the point. If
     * `right`, the right side of the label should be touching the point. For
     * points with an extent, like columns, the alignments also dictates how to
     * align it inside the box, as given with the inside option. Can be one of
     * `left`, `center` or `right`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.align
     */
    align?: AlignType;
    /**
     * (Highstock) Whether to allow data labels to overlap. To make the labels
     * less sensitive for overlapping, the dataLabels.padding can be set to 0.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the data label.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the data label. Defaults to `undefined`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for the data label. Particularly in styled mode,
     * this can be used to give each series' or point's data label unique
     * styling. In addition to this option, a default color class name is added
     * so that we can give the labels a contrast text shadow.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.className
     */
    className?: string;
    /**
     * (Highstock) The text color for the data labels. Defaults to `undefined`.
     * For certain series types, like column or map, the data labels can be
     * drawn inside the points. In this case the data label will be drawn with
     * maximum contrast by default. Additionally, it will be given a
     * `text-outline` style with the opposite color, to further increase the
     * contrast. This can be overridden by setting the `text-outline` style to
     * `none` in the `dataLabels.style` option.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.color
     */
    color?: ColorString;
    /**
     * (Highstock) Whether to hide data labels that are outside the plot area.
     * By default, the data label is moved inside the plot area according to the
     * overflow option.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.crop
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished.
     *
     * @see https://api.highcharts.com/highcharts/navigator.series.dataLabels.defer
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.defer
     * @see https://api.highcharts.com/gantt/navigator.series.dataLabels.defer
     */
    defer?: boolean;
    /**
     * (Highstock) Enable or disable the data labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) A declarative filter for which data labels to display. The
     * declarative filter is designed for use when callback functions are not
     * available, like when the chart options require a pure JSON structure or
     * for use with graphical editors. For programmatic control, use the
     * `formatter` instead, and return `undefined` to disable a single data
     * label.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.filter
     */
    filter?: NavigatorSeriesDataLabelsFilterOptions;
    /**
     * (Highstock) A format string for the data label. Available variables are
     * the same as for `formatter`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.format
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the data label. Note
     * that if a `format` is defined, the format takes precedence and the
     * formatter is ignored. Available data are:
     *
     * - `this.percentage`: Stacked series and pies only. The point's percentage
     * of the total.
     *
     * - `this.point`: The point object. The point name, if defined, is
     * available through `this.point.name`.
     *
     * - `this.series`: The series object. The series name is available
     * through`this.series.name`.
     *
     * - `this.total`: Stacked series only. The total value at this point's x
     * value.
     *
     * - `this.x`: The x value.
     *
     * - `this.y`: The y value.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.formatter
     */
    formatter?: FormatterCallbackFunction<SeriesDataLabelsFormatterContextObject>;
    /**
     * (Highstock) For points with an extent, like columns or map areas, whether
     * to align the data label inside the box or to the actual value point.
     * Defaults to `false` in most cases, `true` in stacked columns.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.inside
     */
    inside?: boolean;
    /**
     * (Highstock) How to handle data labels that flow outside the plot area.
     * The default is `"justify"`, which aligns them inside the plot area. For
     * columns and bars, this means it will be moved inside the bar. To display
     * data labels outside the plot area, set `crop` to `false` and `overflow`
     * to `"allow"`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highstock) When either the `borderWidth` or the `backgroundColor` is
     * set, this is the padding within the box.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.padding
     */
    padding?: number;
    /**
     * (Highstock) Text rotation in degrees. Note that due to a more complex
     * structure, backgrounds, borders and padding will be lost on a rotated
     * data label.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.rotation
     */
    rotation?: number;
    /**
     * (Highstock) The shadow of the box. Works best with `borderWidth` or
     * `backgroundColor`. Since 2.3 the shadow can be an object configuration
     * containing `color`, `offsetX`, `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.shape
     */
    shape?: string;
    /**
     * (Highstock) Styles for the label. The default `color` setting is
     * `"contrast"`, which is a pseudo color that Highcharts picks up and
     * applies the maximum contrast to the underlying point item, for example
     * the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.style
     */
    style?: CSSObject;
    /**
     * (Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of a data label. Can be one of `top`,
     * `middle` or `bottom`. The default value depends on the data, for instance
     * in a column chart, the label is above positive values and below negative
     * values.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highstock) The x position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.x
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.y
     */
    y?: number;
    /**
     * (Highstock) The Z index of the data labels. The default Z index puts it
     * above the series. Use a Z index of 2 to display it behind the series.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels.zIndex
     */
    zIndex?: number;
}
export interface NavigatorSeriesMarkerOptions {
    enabled?: boolean;
}
/**
 * (Highstock) Options for the navigator series. Available options are the same
 * as any series, documented at plotOptions and series.
 *
 * Unless data is explicitly defined on navigator.series, the data is borrowed
 * from the first series in the chart.
 *
 * Default series options for the navigator series are:
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highstock/navigator.series
 */
export interface NavigatorSeriesOptions {
    className?: string;
    /**
     * (Highstock) Data grouping options for the navigator series.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataGrouping
     */
    dataGrouping?: NavigatorSeriesDataGroupingOptions;
    /**
     * (Highstock) Data label options for the navigator series. Data labels are
     * disabled by default on the navigator series.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.dataLabels
     */
    dataLabels?: NavigatorSeriesDataLabelsOptions;
    /**
     * (Highstock) The fill opacity of the navigator series.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.fillOpacity
     */
    fillOpacity?: number;
    id?: string;
    /**
     * (Highstock) Line color for the navigator series. Allows setting the color
     * while disallowing the default candlestick setting.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.lineColor
     */
    lineColor?: (ColorString|null);
    /**
     * (Highstock) The pixel line width of the navigator series.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.lineWidth
     */
    lineWidth?: number;
    marker?: NavigatorSeriesMarkerOptions;
    pointRange?: number;
    /**
     * (Highstock) The threshold option. Setting it to 0 will make the default
     * navigator area series draw its area from the 0 value and up.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.threshold
     */
    threshold?: (number|null);
    /**
     * (Highstock) The type of the navigator series. Defaults to `areaspline` if
     * defined, otherwise `line`.
     *
     * @see https://api.highcharts.com/highstock/navigator.series.type
     */
    type?: string;
}
/**
 * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
 * sections defined will be left out and all the points shifted closer to each
 * other.
 *
 * @see https://api.highcharts.com/highcharts/navigator.xAxis.breaks
 * @see https://api.highcharts.com/highstock/navigator.xAxis.breaks
 * @see https://api.highcharts.com/gantt/navigator.xAxis.breaks
 */
export interface NavigatorXAxisBreaksOptions {
    /**
     * (Highcharts, Highstock, Gantt) A number indicating how much space should
     * be left between the start and the end of the break. The break size is
     * given in axis units, so for instance on a `datetime` axis, a break size
     * of 3600000 would indicate the equivalent of an hour.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.breaks.breakSize
     * @see https://api.highcharts.com/highstock/navigator.xAxis.breaks.breakSize
     * @see https://api.highcharts.com/gantt/navigator.xAxis.breaks.breakSize
     */
    breakSize?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break starts.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.breaks.from
     * @see https://api.highcharts.com/highstock/navigator.xAxis.breaks.from
     * @see https://api.highcharts.com/gantt/navigator.xAxis.breaks.from
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) Defines an interval after which the break
     * appears again. By default the breaks do not repeat.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.breaks.repeat
     * @see https://api.highcharts.com/highstock/navigator.xAxis.breaks.repeat
     * @see https://api.highcharts.com/gantt/navigator.xAxis.breaks.repeat
     */
    repeat?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break ends.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.breaks.to
     * @see https://api.highcharts.com/highstock/navigator.xAxis.breaks.to
     * @see https://api.highcharts.com/gantt/navigator.xAxis.breaks.to
     */
    to?: number;
}
/**
 * (Highstock) A label on the axis next to the crosshair.
 *
 * In styled mode, the label is styled with the `.highcharts-crosshair-label`
 * class.
 *
 * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label
 */
export interface NavigatorXAxisCrosshairLabelOptions {
    /**
     * (Highstock) Alignment of the label compared to the axis. Defaults to
     * `left` for right-side axes, `right` for left-side axes and `center` for
     * horizontal axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.align
     */
    align?: string;
    /**
     * (Highstock) The background color for the label. Defaults to the related
     * series color, or `#666666` if that is not available.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the crosshair label
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border corner radius of the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width for the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A format string for the crosshair label. Defaults to
     * `{value}` for numeric axes and `{value:%b %d, %Y}` for datetime axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.format
     */
    format?: string;
    /**
     * (Highstock) Formatter function for the label text.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.formatter
     */
    formatter?: FormatterCallbackFunction<object>;
    /**
     * (Highstock) Padding inside the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.padding
     */
    padding?: number;
    /**
     * (Highstock) The shape to use for the label box.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.shape
     */
    shape?: string;
    /**
     * (Highstock) Text styles for the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label.style
     */
    style?: CSSObject;
}
/**
 * (Highstock) Configure a crosshair that follows either the mouse pointer or
 * the hovered point.
 *
 * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
 * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
 *
 * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair
 */
export interface NavigatorXAxisCrosshairOptions {
    /**
     * (Highstock) A class name for the crosshair, especially as a hook for
     * styling.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.className
     */
    className?: string;
    /**
     * (Highstock) The color of the crosshair. Defaults to `#cccccc` for numeric
     * and datetime axes, and `rgba(204,214,235,0.25)` for category axes, where
     * the crosshair by default highlights the whole category.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.color
     */
    color?: ColorString;
    /**
     * (Highstock) The dash style for the crosshair. See series.dashStyle for
     * possible values.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highstock) A label on the axis next to the crosshair.
     *
     * In styled mode, the label is styled with the
     * `.highcharts-crosshair-label` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.label
     */
    label?: NavigatorXAxisCrosshairLabelOptions;
    /**
     * (Highstock) Whether the crosshair should snap to the point or follow the
     * pointer independent of points.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.snap
     */
    snap?: boolean;
    /**
     * (Highstock) The pixel width of the crosshair. Defaults to 1 for numeric
     * or datetime axes, and for one category width for category axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.width
     */
    width?: number;
    /**
     * (Highstock) The Z index of the crosshair. Higher Z indices allow drawing
     * the crosshair on top of the series or behind the grid lines.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair.zIndex
     */
    zIndex?: number;
}
/**
 * (Gantt) Text labels for the plot bands
 *
 * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label
 */
export interface NavigatorXAxisCurrentDateIndicatorLabelOptions {
    /**
     * (Gantt) Horizontal alignment of the label. Can be one of "left", "center"
     * or "right".
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.align
     */
    align?: AlignType;
    /**
     * (Gantt) Rotation of the text label in degrees. Defaults to 0 for
     * horizontal plot lines and 90 for vertical lines.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.rotation
     */
    rotation?: number;
    /**
     * (Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.style
     */
    style?: CSSObject;
    /**
     * (Gantt) The text itself. A subset of HTML is supported.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.text
     */
    text?: string;
    /**
     * (Gantt) The text alignment for the label. While `align` determines where
     * the texts anchor point is placed within the plot band, `textAlign`
     * determines how the text is aligned against its anchor point. Possible
     * values are "left", "center" and "right". Defaults to the same as the
     * `align` option.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Gantt) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.useHTML
     */
    useHTML?: boolean;
    /**
     * (Gantt) Vertical alignment of the label relative to the plot line. Can be
     * one of "top", "middle" or "bottom".
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.verticalAlign
     */
    verticalAlign?: ("bottom"|"middle"|"top");
    /**
     * (Gantt) Horizontal position relative the alignment. Default varies by
     * orientation.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.x
     */
    x?: number;
    /**
     * (Gantt) Vertical position of the text baseline relative to the alignment.
     * Default varies by orientation.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label.y
     */
    y?: number;
}
/**
 * (Gantt) Show an indicator on the axis for the current date and time. Can be a
 * boolean or a configuration object similar to xAxis.plotLines.
 *
 * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator
 */
export interface NavigatorXAxisCurrentDateIndicatorOptions {
    /**
     * (Gantt) A custom class name, in addition to the default
     * `highcharts-plot-line`, to apply to each individual line.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.className
     */
    className?: string;
    /**
     * (Gantt) The color of the line.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.color
     */
    color?: ColorString;
    /**
     * (Gantt) The dashing or dot style for the plot line. For possible values
     * see this overview.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Gantt) An object defining mouse events for the plot line. Supported
     * properties are `click`, `mouseover`, `mouseout`, `mousemove`.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.events
     */
    events?: any;
    /**
     * (Gantt) An id used for identifying the plot line in Axis.removePlotLine.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.id
     */
    id?: string;
    /**
     * (Gantt) Text labels for the plot bands
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.label
     */
    label?: NavigatorXAxisCurrentDateIndicatorLabelOptions;
    /**
     * (Gantt) The width or thickness of the plot line.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.width
     */
    width?: number;
    /**
     * (Gantt) The z index of the plot line within the chart.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator.zIndex
     */
    zIndex?: number;
}
export interface NavigatorXAxisDateTimeLabelFormatsDayOptions {
    main?: string;
}
export interface NavigatorXAxisDateTimeLabelFormatsHourOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorXAxisDateTimeLabelFormatsMillisecondOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorXAxisDateTimeLabelFormatsMinuteOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorXAxisDateTimeLabelFormatsMonthOptions {
    main?: string;
}
/**
 * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
 * automatically adjust to the appropriate unit. This member gives the default
 * string representations used for each unit. For intermediate values, different
 * units may be used, for example the `day` unit can be used on midnight and
 * `hour` unit be used for intermediate values on the same axis. For an overview
 * of the replacement codes, see dateFormat. Defaults to:
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highcharts/navigator.xAxis.dateTimeLabelFormats
 * @see https://api.highcharts.com/highstock/navigator.xAxis.dateTimeLabelFormats
 * @see https://api.highcharts.com/gantt/navigator.xAxis.dateTimeLabelFormats
 */
export interface NavigatorXAxisDateTimeLabelFormatsOptions {
    day?: NavigatorXAxisDateTimeLabelFormatsDayOptions;
    hour?: NavigatorXAxisDateTimeLabelFormatsHourOptions;
    millisecond?: NavigatorXAxisDateTimeLabelFormatsMillisecondOptions;
    minute?: NavigatorXAxisDateTimeLabelFormatsMinuteOptions;
    month?: NavigatorXAxisDateTimeLabelFormatsMonthOptions;
    second?: NavigatorXAxisDateTimeLabelFormatsSecondOptions;
    week?: NavigatorXAxisDateTimeLabelFormatsWeekOptions;
    year?: NavigatorXAxisDateTimeLabelFormatsYearOptions;
}
export interface NavigatorXAxisDateTimeLabelFormatsSecondOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorXAxisDateTimeLabelFormatsWeekOptions {
    main?: string;
}
export interface NavigatorXAxisDateTimeLabelFormatsYearOptions {
    main?: string;
}
/**
 * (Highstock) Event handlers for the axis.
 *
 * @see https://api.highcharts.com/highstock/navigator.xAxis.events
 */
export interface NavigatorXAxisEventsOptions {
    /**
     * (Highcharts, Gantt) An event fired after the breaks have rendered.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.events.afterBreaks
     * @see https://api.highcharts.com/gantt/navigator.xAxis.events.afterBreaks
     */
    afterBreaks?: AxisEventCallbackFunction;
    /**
     * (Highstock) As opposed to the `setExtremes` event, this event fires after
     * the final min and max values are computed and corrected for `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.events.afterSetExtremes
     */
    afterSetExtremes?: AxisEventCallbackFunction;
    /**
     * (Highcharts, Gantt) An event fired when a break from this axis occurs on
     * a point.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.events.pointBreak
     * @see https://api.highcharts.com/gantt/navigator.xAxis.events.pointBreak
     */
    pointBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) An event fired when a point falls inside a
     * break from this axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.events.pointInBreak
     * @see https://api.highcharts.com/highstock/navigator.xAxis.events.pointInBreak
     * @see https://api.highcharts.com/gantt/navigator.xAxis.events.pointInBreak
     */
    pointInBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highstock) Fires when the minimum and maximum is set for the axis,
     * either by calling the `.setExtremes()` method or by selecting an area in
     * the chart. One parameter, `event`, is passed to the function, containing
     * common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.events.setExtremes
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 *
 * @see https://api.highcharts.com/gantt/navigator.xAxis.grid
 */
export interface NavigatorXAxisGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.grid.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.grid.borderWidth
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.grid.cellHeight
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.grid.columns
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.grid.enabled
     */
    enabled?: boolean;
}
/**
 * (Highstock) The axis labels show the number or category for each tick.
 *
 * @see https://api.highcharts.com/highstock/navigator.xAxis.labels
 */
export interface NavigatorXAxisLabelsOptions {
    /**
     * (Highstock) What part of the string the given position is anchored to. If
     * `left`, the left side of the string is at the axis position. Can be one
     * of `"left"`, `"center"` or `"right"`. Defaults to an intelligent guess
     * based on which side of the chart the axis is on and the rotation of the
     * label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.align
     */
    align?: ("center"|"left"|"right");
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.autoRotation
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.autoRotation
     * @see https://api.highcharts.com/gantt/navigator.xAxis.labels.autoRotation
     */
    autoRotation?: Array<number>;
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.autoRotationLimit
     * @see https://api.highcharts.com/gantt/navigator.xAxis.labels.autoRotationLimit
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts, Gantt) Polar charts only. The label's pixel distance from
     * the perimeter of the plot area.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.distance
     * @see https://api.highcharts.com/gantt/navigator.xAxis.labels.distance
     */
    distance?: number;
    /**
     * (Highstock) Enable or disable the axis labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) A format string for the axis label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.format
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the label. The value
     * is given by `this.value`. Additional properties for `this` are `axis`,
     * `chart`, `isFirst` and `isLast`. The value of the default label formatter
     * can be retrieved by calling `this.axis.defaultLabelFormatter.call(this)`
     * within the function.
     *
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.formatter
     */
    formatter?: FormatterCallbackFunction<AxisLabelsFormatterContextObject>;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.labels.indentation
     */
    indentation?: number;
    /**
     * (Highstock) Horizontal axis only. When `staggerLines` is not set,
     * `maxStaggerLines` defines how many lines the axis is allowed to add to
     * automatically avoid overlapping X labels. Set to `1` to disable overlap
     * detection.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.maxStaggerLines
     */
    maxStaggerLines?: number;
    /**
     * (Highstock) How to handle overflowing labels on horizontal axis. If set
     * to `"allow"`, it will not be aligned at all. By default it `"justify"`
     * labels inside the chart area. If there is room to move it, it will be
     * aligned to the edge, else it will be removed.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.padding
     * @see https://api.highcharts.com/gantt/navigator.xAxis.labels.padding
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.position3d
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho");
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.reserveSpace
     * @see https://api.highcharts.com/gantt/navigator.xAxis.labels.reserveSpace
     */
    reserveSpace?: boolean;
    /**
     * (Highstock) Rotation of the labels in degrees.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.labels.skew3d
     */
    skew3d?: boolean;
    /**
     * (Highstock) Horizontal axes only. The number of lines to spread the
     * labels over to make room or tighter labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.staggerLines
     */
    staggerLines?: number;
    /**
     * (Highstock) To show only every _n_'th label on the axis, set the step to
     * _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.step
     */
    step?: number;
    /**
     * (Highstock) CSS styles for the label. Use `whiteSpace: 'nowrap'` to
     * prevent wrapping of category labels. Use `textOverflow: 'none'` to
     * prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.style
     */
    style?: CSSObject;
    /**
     * (Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highstock) The x position offset of the label relative to the tick
     * position on the axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.x
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the tick
     * position on the axis. The default makes it adapt to the font size on
     * bottom axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.y
     */
    y?: number;
    /**
     * (Highstock) The Z index for the axis labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Options for the navigator X axis. Default series options for the
 * navigator xAxis are:
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highstock/navigator.xAxis
 */
export interface NavigatorXAxisOptions {
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.alignTicks
     * @see https://api.highcharts.com/highstock/navigator.xAxis.alignTicks
     * @see https://api.highcharts.com/gantt/navigator.xAxis.alignTicks
     */
    alignTicks?: boolean;
    /**
     * (Highstock) Whether to allow decimals in this axis' ticks. When counting
     * integers, like persons or hits on a web page, decimals should be avoided
     * in the labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.allowDecimals
     */
    allowDecimals?: boolean;
    /**
     * (Highstock) When using an alternate grid color, a band is painted across
     * the plot area between every other grid line.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.alternateGridColor
     */
    alternateGridColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
     * sections defined will be left out and all the points shifted closer to
     * each other.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.breaks
     * @see https://api.highcharts.com/highstock/navigator.xAxis.breaks
     * @see https://api.highcharts.com/gantt/navigator.xAxis.breaks
     */
    breaks?: Array<NavigatorXAxisBreaksOptions>;
    /**
     * (Highcharts, Gantt) If categories are present for the xAxis, names are
     * used instead of numbers for that axis. Since Highcharts 3.0, categories
     * can also be extracted by giving each point a name and setting axis type
     * to `category`. However, if you have multiple series, best practice
     * remains defining the `categories` array.
     *
     * Example:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.categories
     * @see https://api.highcharts.com/gantt/navigator.xAxis.categories
     */
    categories?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.ceiling
     * @see https://api.highcharts.com/highstock/navigator.xAxis.ceiling
     * @see https://api.highcharts.com/gantt/navigator.xAxis.ceiling
     */
    ceiling?: number;
    /**
     * (Highstock) A class name that opens for styling the axis by CSS,
     * especially in Highcharts styled mode. The class name is applied to group
     * elements for the grid, axis elements and labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.className
     */
    className?: string;
    /**
     * (Highstock) Configure a crosshair that follows either the mouse pointer
     * or the hovered point.
     *
     * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
     * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.crosshair
     */
    crosshair?: (boolean|NavigatorXAxisCrosshairOptions);
    /**
     * (Gantt) Show an indicator on the axis for the current date and time. Can
     * be a boolean or a configuration object similar to xAxis.plotLines.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.currentDateIndicator
     */
    currentDateIndicator?: (boolean|NavigatorXAxisCurrentDateIndicatorOptions);
    /**
     * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
     * automatically adjust to the appropriate unit. This member gives the
     * default string representations used for each unit. For intermediate
     * values, different units may be used, for example the `day` unit can be
     * used on midnight and `hour` unit be used for intermediate values on the
     * same axis. For an overview of the replacement codes, see dateFormat.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.dateTimeLabelFormats
     * @see https://api.highcharts.com/highstock/navigator.xAxis.dateTimeLabelFormats
     * @see https://api.highcharts.com/gantt/navigator.xAxis.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: NavigatorXAxisDateTimeLabelFormatsOptions;
    /**
     * (Highstock) _Requires Accessibility module_
     *
     * Description of the axis to screen reader users.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.description
     */
    description?: string;
    /**
     * (Highstock) Whether to force the axis to end on a tick. Use this option
     * with the `maxPadding` option to control the axis end.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.endOnTick
     */
    endOnTick?: boolean;
    /**
     * (Highstock) Event handlers for the axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.events
     */
    events?: NavigatorXAxisEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.floor
     * @see https://api.highcharts.com/highstock/navigator.xAxis.floor
     * @see https://api.highcharts.com/gantt/navigator.xAxis.floor
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     *
     * @see https://api.highcharts.com/gantt/navigator.xAxis.grid
     */
    grid?: NavigatorXAxisGridOptions;
    /**
     * (Highstock) Color of the grid lines extending the ticks across the plot
     * area.
     *
     * In styled mode, the stroke is given in the `.highcharts-grid-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.gridLineColor
     */
    gridLineColor?: string;
    /**
     * (Highstock) The dash or dot style of the grid lines. For possible values,
     * see this demonstration.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.gridLineDashStyle
     */
    gridLineDashStyle?: DashStyleType;
    /**
     * (Highstock) The width of the grid lines extending the ticks across the
     * plot area.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.gridLineWidth
     */
    gridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.gridZIndex
     * @see https://api.highcharts.com/highstock/navigator.xAxis.gridZIndex
     * @see https://api.highcharts.com/gantt/navigator.xAxis.gridZIndex
     */
    gridZIndex?: number;
    /**
     * (Highstock) An id for the axis. This can be used after render time to get
     * a pointer to the axis object through `chart.get()`.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.id
     */
    id?: string;
    /**
     * (Highstock) The axis labels show the number or category for each tick.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.labels
     */
    labels?: NavigatorXAxisLabelsOptions;
    /**
     * (Highstock) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the line marking the axis itself.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The maximum value of the axis. If `null`, the max value is
     * automatically calculated.
     *
     * If the endOnTick option is true, the `max` value might be rounded up.
     *
     * If a tickAmount is set, the axis may be extended beyond the set max in
     * order to reach the given number of ticks. The same may happen in a chart
     * with multiple axes, determined by chart. alignTicks, where a `tickAmount`
     * is applied internally.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.max
     */
    max?: number;
    /**
     * (Highstock) Padding of the max value relative to the length of the axis.
     * A padding of 0.05 will make a 100px axis 5px longer. This is useful when
     * you don't want the highest data value to appear on the edge of the plot
     * area. When the axis' `max` option is set or a max extreme is set using
     * `axis.setExtremes()`, the maxPadding will be ignored.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.maxPadding
     */
    maxPadding?: number;
    /**
     * (Highstock) The minimum value of the axis. If `null` the min value is
     * automatically calculated.
     *
     * If the startOnTick option is true (default), the `min` value might be
     * rounded down.
     *
     * The automatically calculated minimum value is also affected by floor,
     * softMin, minPadding, minRange as well as series.threshold and
     * series.softThreshold.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.min
     */
    min?: number;
    /**
     * (Highstock) Color of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the
     * `.highcharts-minor-grid-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorGridLineColor
     */
    minorGridLineColor?: ColorString;
    /**
     * (Highstock) The dash or dot style of the minor grid lines. For possible
     * values, see this demonstration.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorGridLineDashStyle
     */
    minorGridLineDashStyle?: DashStyleType;
    /**
     * (Highstock) Width of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorGridLineWidth
     */
    minorGridLineWidth?: number;
    /**
     * (Highstock) Color for the minor tick marks.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorTickColor
     */
    minorTickColor?: ColorString;
    /**
     * (Highstock) Specific tick interval in axis units for the minor ticks. On
     * a linear axis, if `"auto"`, the minor tick interval is calculated as a
     * fifth of the tickInterval. If `null` or `undefined`, minor ticks are not
     * shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorTickInterval
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highstock) The pixel length of the minor tick marks.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorTickLength
     */
    minorTickLength?: number;
    /**
     * (Highstock) The position of the minor tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorTickPosition
     */
    minorTickPosition?: ("inside"|"outside");
    /**
     * (Highstock) Enable or disable minor ticks. Unless minorTickInterval is
     * set, the tick interval is calculated as a fifth of the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorTicks
     */
    minorTicks?: boolean;
    /**
     * (Highstock) The pixel width of the minor tick mark.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minorTickWidth
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the min value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the lowest data value to appear on the
     * edge of the plot area. When the axis' `min` option is set or a min
     * extreme is set using `axis.setExtremes()`, the minPadding will be
     * ignored.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.minPadding
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minPadding
     * @see https://api.highcharts.com/gantt/navigator.xAxis.minPadding
     */
    minPadding?: number;
    /**
     * (Highstock) The minimum tick interval allowed in axis values. For example
     * on zooming in on an axis with daily data, this can be used to prevent the
     * axis from showing hours. Defaults to the closest distance between two
     * points on the axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.minTickInterval
     */
    minTickInterval?: number;
    /**
     * (Highstock) The distance in pixels from the plot area to the axis line. A
     * positive offset moves the axis with it's line, labels and ticks away from
     * the plot area. This is typically used when two or more axes are displayed
     * on the same side of the plot. With multiple axes the offset is
     * dynamically adjusted to avoid collision, this can be overridden by
     * setting offset explicitly.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.offset
     */
    offset?: number;
    /**
     * (Highstock) In an ordinal axis, the points are equally spaced in the
     * chart regardless of the actual time or x distance between them. This
     * means that missing data periods (e.g. nights or weekends for a stock
     * chart) will not take up space in the chart. Having `ordinal: false` will
     * show any gaps created by the `gapSize` setting proportionate to their
     * duration.
     *
     * In stock charts the X axis is ordinal by default, unless the boost module
     * is used and at least one of the series' data length exceeds the
     * boostThreshold.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.ordinal
     */
    ordinal?: boolean;
    /**
     * (Highstock) Additional range on the right side of the xAxis. Works
     * similar to xAxis.maxPadding, but value is set in milliseconds. Can be set
     * for both, main xAxis and navigator's xAxis.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.overscroll
     */
    overscroll?: number;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.pane
     */
    pane?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array of colored bands stretching
     * across the plot area marking an interval on the axis.
     *
     * In styled mode, the plot bands are styled by the `.highcharts-plot-band`
     * class in addition to the `className` option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands
     */
    plotBands?: Array<NavigatorXAxisPlotBandsOptions>;
    /**
     * (Highcharts, Highstock, Gantt) An array of lines stretching across the
     * plot area, marking a specific value on one of the axes.
     *
     * In styled mode, the plot lines are styled by the `.highcharts-plot-line`
     * class in addition to the `className` option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines
     */
    plotLines?: Array<NavigatorXAxisPlotLinesOptions>;
    /**
     * (Highstock) Whether to reverse the axis so that the highest number is
     * closest to the origin. If the chart is inverted, the x axis is reversed
     * by default.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.reversed
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) This option determines how stacks should be
     * ordered within a group. For example reversed xAxis also reverses stacks,
     * so first series comes last in a group. To keep order like for
     * non-reversed xAxis enable this option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.reversedStacks
     * @see https://api.highcharts.com/highstock/navigator.xAxis.reversedStacks
     */
    reversedStacks?: boolean;
    /**
     * (Highstock) Whether to show the first tick label.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.showFirstLabel
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.showLastLabel
     * @see https://api.highcharts.com/highstock/navigator.xAxis.showLastLabel
     * @see https://api.highcharts.com/gantt/navigator.xAxis.showLastLabel
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.softMax
     * @see https://api.highcharts.com/highstock/navigator.xAxis.softMax
     * @see https://api.highcharts.com/gantt/navigator.xAxis.softMax
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.softMin
     * @see https://api.highcharts.com/highstock/navigator.xAxis.softMin
     * @see https://api.highcharts.com/gantt/navigator.xAxis.softMin
     */
    softMin?: number;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.startOfWeek
     * @see https://api.highcharts.com/highstock/navigator.xAxis.startOfWeek
     * @see https://api.highcharts.com/gantt/navigator.xAxis.startOfWeek
     */
    startOfWeek?: number;
    /**
     * (Highstock) Whether to force the axis to start on a tick. Use this option
     * with the `minPadding` option to control the axis start.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.startOnTick
     */
    startOnTick?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.tickAmount
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickAmount
     * @see https://api.highcharts.com/gantt/navigator.xAxis.tickAmount
     */
    tickAmount?: number;
    /**
     * (Highstock) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickColor
     */
    tickColor?: ColorString;
    /**
     * (Highstock) The interval of the tick marks in axis units. When
     * `undefined`, the tick interval is computed to approximately follow the
     * tickPixelInterval on linear and datetime axes. On categorized axes, a
     * `undefined` tickInterval will default to 1, one category. Note that
     * datetime axes are based on milliseconds, so for example an interval of
     * one day is expressed as `24 * 3600 * 1000`.
     *
     * On logarithmic axes, the tickInterval is based on powers, so a
     * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
     * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
     * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
     * etc.
     *
     * If the tickInterval is too dense for labels to be drawn, Highcharts may
     * remove ticks.
     *
     * If the chart has multiple axes, the alignTicks option may interfere with
     * the `tickInterval` setting.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickInterval
     */
    tickInterval?: number;
    /**
     * (Highstock) The pixel length of the main tick marks.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickLength
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.tickmarkPlacement
     * @see https://api.highcharts.com/gantt/navigator.xAxis.tickmarkPlacement
     */
    tickmarkPlacement?: ("between"|"on");
    /**
     * (Highstock) If tickInterval is `null` this option sets the approximate
     * pixel interval of the tick marks. Not applicable to categorized axis.
     *
     * The tick interval is also influenced by the minTickInterval option, that,
     * by default prevents ticks from being denser than the data points.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickPixelInterval
     */
    tickPixelInterval?: number;
    /**
     * (Highstock) The position of the major tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickPosition
     */
    tickPosition?: ("inside"|"outside");
    /**
     * (Highstock) A callback function returning array defining where the ticks
     * are laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickPositioner
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highstock) An array defining where the ticks are laid out on the axis.
     * This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickPositions
     */
    tickPositions?: Array<number>;
    /**
     * (Highstock) The pixel width of the major tick marks.
     *
     * In styled mode, the stroke width is given in the `.highcharts-tick`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.tickWidth
     */
    tickWidth?: number;
    /**
     * (Highstock) The axis title, showing next to the axis line.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title
     */
    title?: NavigatorXAxisTitleOptions;
    /**
     * (Highcharts, Gantt) The type of axis. Can be one of `linear`,
     * `logarithmic`, `datetime` or `category`. In a datetime axis, the numbers
     * are given in milliseconds, and tick marks are placed on appropriate
     * values like full hours or days. In a category axis, the point names of
     * the chart's series are used for categories, if not a categories array is
     * defined.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.type
     * @see https://api.highcharts.com/gantt/navigator.xAxis.type
     */
    type?: ("category"|"datetime"|"linear"|"logarithmic");
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.uniqueNames
     * @see https://api.highcharts.com/gantt/navigator.xAxis.uniqueNames
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Datetime axis only. An array determining
     * what time intervals the ticks are allowed to fall on. Each array item is
     * an array where the first value is the time unit and the second value
     * another array of allowed multiples. Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.units
     * @see https://api.highcharts.com/highstock/navigator.xAxis.units
     * @see https://api.highcharts.com/gantt/navigator.xAxis.units
     */
    units?: Array<[string, (Array<number>|null)]>;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.visible
     * @see https://api.highcharts.com/highstock/navigator.xAxis.visible
     * @see https://api.highcharts.com/gantt/navigator.xAxis.visible
     */
    visible?: boolean;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 *
 * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label
 * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label
 * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label
 */
export interface NavigatorXAxisPlotBandsLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.align
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.align
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees .
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.rotation
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.rotation
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.rotation
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-band-label` class.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.style
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.style
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The string text itself. A subset of HTML
     * is supported.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.text
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.text
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.textAlign
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.textAlign
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.useHTML
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.useHTML
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot band. Can be one of "top", "middle" or "bottom".
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.verticalAlign
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.verticalAlign
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.x
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.x
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label.y
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label.y
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of colored bands stretching across
 * the plot area marking an interval on the axis.
 *
 * In styled mode, the plot bands are styled by the `.highcharts-plot-band`
 * class in addition to the `className` option.
 *
 * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands
 * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands
 * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands
 */
export interface NavigatorXAxisPlotBandsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Border color for the plot band. Also
     * requires `borderWidth` to be set.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.borderColor
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.borderColor
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) Border width for the plot band. Also
     * requires `borderColor` to be set.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.borderWidth
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.borderWidth
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-band`, to apply to each individual band.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.className
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.className
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the plot band.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.color
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.color
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot band. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.events
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.events
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.events
     */
    events?: object;
    /**
     * (Highcharts, Highstock, Gantt) The start position of the plot band in
     * axis units.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.from
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.from
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.from
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot band
     * in Axis.removePlotBand.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.id
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.id
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.id
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.label
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.label
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.label
     */
    label?: NavigatorXAxisPlotBandsLabelOptions;
    /**
     * (Highcharts, Highstock, Gantt) The end position of the plot band in axis
     * units.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.to
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.to
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.to
     */
    to?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot band within the
     * chart, relative to other elements. Using the same z index as another
     * element may give unpredictable results, as the last rendered element will
     * be on top. Values from 0 to 20 make sense.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotBands.zIndex
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotBands.zIndex
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotBands.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 *
 * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label
 * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label
 * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label
 */
export interface NavigatorXAxisPlotLinesLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.align
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.align
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees.
     * Defaults to 0 for horizontal plot lines and 90 for vertical lines.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.rotation
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.rotation
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.rotation
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.style
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.style
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The text itself. A subset of HTML is
     * supported.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.text
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.text
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.textAlign
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.textAlign
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.useHTML
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.useHTML
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot line. Can be one of "top", "middle" or "bottom".
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.verticalAlign
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.verticalAlign
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.verticalAlign
     */
    verticalAlign?: ("bottom"|"middle"|"top");
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.x
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.x
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label.y
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label.y
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of lines stretching across the plot
 * area, marking a specific value on one of the axes.
 *
 * In styled mode, the plot lines are styled by the `.highcharts-plot-line`
 * class in addition to the `className` option.
 *
 * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines
 * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines
 * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines
 */
export interface NavigatorXAxisPlotLinesOptions {
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-line`, to apply to each individual line.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.className
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.className
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the line.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.color
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.color
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.color
     */
    color?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) The dashing or dot style for the plot
     * line. For possible values see this overview.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.dashStyle
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.dashStyle
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot line. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.events
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.events
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.events
     */
    events?: any;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot line
     * in Axis.removePlotLine.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.id
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.id
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.id
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.label
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.label
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.label
     */
    label?: NavigatorXAxisPlotLinesLabelOptions;
    /**
     * (Highcharts, Highstock, Gantt) The position of the line in axis units.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.value
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.value
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.value
     */
    value?: number;
    /**
     * (Highcharts, Highstock, Gantt) The width or thickness of the plot line.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.width
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.width
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.width
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot line within the
     * chart.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.plotLines.zIndex
     * @see https://api.highcharts.com/highstock/navigator.xAxis.plotLines.zIndex
     * @see https://api.highcharts.com/gantt/navigator.xAxis.plotLines.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) The axis title, showing next to the axis line.
 *
 * @see https://api.highcharts.com/highstock/navigator.xAxis.title
 */
export interface NavigatorXAxisTitleOptions {
    /**
     * (Highstock) Alignment of the title relative to the axis values. Possible
     * values are "low", "middle" or "high".
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.align
     */
    align?: ("high"|"low"|"middle");
    /**
     * (Highcharts) Deprecated. Set the `text` to `null` to disable the title.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.enabled
     */
    enabled?: string;
    /**
     * (Highstock) The pixel distance between the axis labels or line and the
     * title. Defaults to 0 for horizontal axes, 10 for vertical
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.margin
     */
    margin?: number;
    /**
     * (Highstock) The distance of the axis title from the axis line. By
     * default, this distance is computed from the offset width of the labels,
     * the labels' distance from the axis and the title's margin. However when
     * the offset option is set, it overrides all this.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.offset
     */
    offset?: number;
    /**
     * (Highcharts) Defines how the title is repositioned according to the 3D
     * chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * - `undefined`: Will use the config from `labels.position3d`
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.position3d
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho"|null);
    /**
     * (Highcharts, Highstock, Gantt) Whether to reserve space for the title
     * when laying out the axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.reserveSpace
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.reserveSpace
     * @see https://api.highcharts.com/gantt/navigator.xAxis.title.reserveSpace
     */
    reserveSpace?: boolean;
    /**
     * (Highstock) The rotation of the text in degrees. 0 is horizontal, 270 is
     * vertical reading from bottom to top.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis title will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `title.position3d`.
     *
     * A `null` value will use the config from `labels.skew3d`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.skew3d
     */
    skew3d?: (boolean|null);
    /**
     * (Highstock) CSS styles for the title. If the title text is longer than
     * the axis length, it will wrap to multiple lines by default. This can be
     * customized by setting `textOverflow: 'ellipsis'`, by setting a specific
     * `width` or by setting `whiteSpace: 'nowrap'`.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-title`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.style
     */
    style?: CSSObject;
    /**
     * (Highstock) The actual text of the axis title. It can contain basic HTML
     * text markup like <b>, <i> and spans with style.
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.text
     */
    text?: (string|null);
    /**
     * (Highstock) Alignment of the text, can be `"left"`, `"right"` or
     * `"center"`. Default alignment depends on the title.align:
     *
     * Horizontal axes:
     *
     * - for `align` = `"low"`, `textAlign` is set to `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"`, `textAlign` is set to `right`
     *
     * Vertical axes:
     *
     * - for `align` = `"low"` and `opposite` = `true`, `textAlign` is set to
     * `right`
     *
     * - for `align` = `"low"` and `opposite` = `false`, `textAlign` is set to
     * `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"` and `opposite` = `true` `textAlign` is set to
     * `left`
     *
     * - for `align` = `"high"` and `opposite` = `false` `textAlign` is set to
     * `right`
     *
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.textAlign
     */
    textAlign?: string;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the axis
     * title.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.useHTML
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.useHTML
     * @see https://api.highcharts.com/gantt/navigator.xAxis.title.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal pixel offset of the title
     * position.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.x
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.x
     * @see https://api.highcharts.com/gantt/navigator.xAxis.title.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical pixel offset of the title
     * position.
     *
     * @see https://api.highcharts.com/highcharts/navigator.xAxis.title.y
     * @see https://api.highcharts.com/highstock/navigator.xAxis.title.y
     * @see https://api.highcharts.com/gantt/navigator.xAxis.title.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
 * sections defined will be left out and all the points shifted closer to each
 * other.
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.breaks
 * @see https://api.highcharts.com/highstock/navigator.yAxis.breaks
 * @see https://api.highcharts.com/gantt/navigator.yAxis.breaks
 */
export interface NavigatorYAxisBreaksOptions {
    /**
     * (Highcharts, Highstock, Gantt) A number indicating how much space should
     * be left between the start and the end of the break. The break size is
     * given in axis units, so for instance on a `datetime` axis, a break size
     * of 3600000 would indicate the equivalent of an hour.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.breaks.breakSize
     * @see https://api.highcharts.com/highstock/navigator.yAxis.breaks.breakSize
     * @see https://api.highcharts.com/gantt/navigator.yAxis.breaks.breakSize
     */
    breakSize?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break starts.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.breaks.from
     * @see https://api.highcharts.com/highstock/navigator.yAxis.breaks.from
     * @see https://api.highcharts.com/gantt/navigator.yAxis.breaks.from
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) Defines an interval after which the break
     * appears again. By default the breaks do not repeat.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.breaks.repeat
     * @see https://api.highcharts.com/highstock/navigator.yAxis.breaks.repeat
     * @see https://api.highcharts.com/gantt/navigator.yAxis.breaks.repeat
     */
    repeat?: number;
    /**
     * (Highcharts, Highstock, Gantt) The point where the break ends.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.breaks.to
     * @see https://api.highcharts.com/highstock/navigator.yAxis.breaks.to
     * @see https://api.highcharts.com/gantt/navigator.yAxis.breaks.to
     */
    to?: number;
}
/**
 * (Highstock) A label on the axis next to the crosshair.
 *
 * In styled mode, the label is styled with the `.highcharts-crosshair-label`
 * class.
 *
 * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label
 */
export interface NavigatorYAxisCrosshairLabelOptions {
    /**
     * (Highstock) Alignment of the label compared to the axis. Defaults to
     * `left` for right-side axes, `right` for left-side axes and `center` for
     * horizontal axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.align
     */
    align?: string;
    /**
     * (Highstock) The background color for the label. Defaults to the related
     * series color, or `#666666` if that is not available.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the crosshair label
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border corner radius of the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width for the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A format string for the crosshair label. Defaults to
     * `{value}` for numeric axes and `{value:%b %d, %Y}` for datetime axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.format
     */
    format?: string;
    /**
     * (Highstock) Formatter function for the label text.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.formatter
     */
    formatter?: FormatterCallbackFunction<object>;
    /**
     * (Highstock) Padding inside the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.padding
     */
    padding?: number;
    /**
     * (Highstock) The shape to use for the label box.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.shape
     */
    shape?: string;
    /**
     * (Highstock) Text styles for the crosshair label.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label.style
     */
    style?: CSSObject;
}
/**
 * (Highstock) Configure a crosshair that follows either the mouse pointer or
 * the hovered point.
 *
 * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
 * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
 *
 * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair
 */
export interface NavigatorYAxisCrosshairOptions {
    /**
     * (Highstock) A class name for the crosshair, especially as a hook for
     * styling.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.className
     */
    className?: string;
    /**
     * (Highstock) The color of the crosshair. Defaults to `#cccccc` for numeric
     * and datetime axes, and `rgba(204,214,235,0.25)` for category axes, where
     * the crosshair by default highlights the whole category.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.color
     */
    color?: ColorString;
    /**
     * (Highstock) The dash style for the crosshair. See series.dashStyle for
     * possible values.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highstock) A label on the axis next to the crosshair.
     *
     * In styled mode, the label is styled with the
     * `.highcharts-crosshair-label` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.label
     */
    label?: NavigatorYAxisCrosshairLabelOptions;
    /**
     * (Highstock) Whether the crosshair should snap to the point or follow the
     * pointer independent of points.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.snap
     */
    snap?: boolean;
    /**
     * (Highstock) The pixel width of the crosshair. Defaults to 1 for numeric
     * or datetime axes, and for one category width for category axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.width
     */
    width?: number;
    /**
     * (Highstock) The Z index of the crosshair. Higher Z indices allow drawing
     * the crosshair on top of the series or behind the grid lines.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair.zIndex
     */
    zIndex?: number;
}
export interface NavigatorYAxisDateTimeLabelFormatsDayOptions {
    main?: string;
}
export interface NavigatorYAxisDateTimeLabelFormatsHourOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorYAxisDateTimeLabelFormatsMillisecondOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorYAxisDateTimeLabelFormatsMinuteOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorYAxisDateTimeLabelFormatsMonthOptions {
    main?: string;
}
/**
 * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
 * automatically adjust to the appropriate unit. This member gives the default
 * string representations used for each unit. For intermediate values, different
 * units may be used, for example the `day` unit can be used on midnight and
 * `hour` unit be used for intermediate values on the same axis. For an overview
 * of the replacement codes, see dateFormat. Defaults to:
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.dateTimeLabelFormats
 * @see https://api.highcharts.com/highstock/navigator.yAxis.dateTimeLabelFormats
 * @see https://api.highcharts.com/gantt/navigator.yAxis.dateTimeLabelFormats
 */
export interface NavigatorYAxisDateTimeLabelFormatsOptions {
    day?: NavigatorYAxisDateTimeLabelFormatsDayOptions;
    hour?: NavigatorYAxisDateTimeLabelFormatsHourOptions;
    millisecond?: NavigatorYAxisDateTimeLabelFormatsMillisecondOptions;
    minute?: NavigatorYAxisDateTimeLabelFormatsMinuteOptions;
    month?: NavigatorYAxisDateTimeLabelFormatsMonthOptions;
    second?: NavigatorYAxisDateTimeLabelFormatsSecondOptions;
    week?: NavigatorYAxisDateTimeLabelFormatsWeekOptions;
    year?: NavigatorYAxisDateTimeLabelFormatsYearOptions;
}
export interface NavigatorYAxisDateTimeLabelFormatsSecondOptions {
    main?: string;
    range?: boolean;
}
export interface NavigatorYAxisDateTimeLabelFormatsWeekOptions {
    main?: string;
}
export interface NavigatorYAxisDateTimeLabelFormatsYearOptions {
    main?: string;
}
/**
 * (Highstock) Event handlers for the axis.
 *
 * @see https://api.highcharts.com/highstock/navigator.yAxis.events
 */
export interface NavigatorYAxisEventsOptions {
    /**
     * (Highcharts, Gantt) An event fired after the breaks have rendered.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.events.afterBreaks
     * @see https://api.highcharts.com/gantt/navigator.yAxis.events.afterBreaks
     */
    afterBreaks?: AxisEventCallbackFunction;
    /**
     * (Highstock) As opposed to the `setExtremes` event, this event fires after
     * the final min and max values are computed and corrected for `minRange`.
     *
     * Fires when the minimum and maximum is set for the axis, either by calling
     * the `.setExtremes()` method or by selecting an area in the chart. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in axis
     * values. The actual data extremes are found in `event.dataMin` and
     * `event.dataMax`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.events.afterSetExtremes
     */
    afterSetExtremes?: AxisEventCallbackFunction;
    /**
     * (Highcharts, Gantt) An event fired when a break from this axis occurs on
     * a point.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.events.pointBreak
     * @see https://api.highcharts.com/gantt/navigator.yAxis.events.pointBreak
     */
    pointBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highcharts, Highstock, Gantt) An event fired when a point falls inside a
     * break from this axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.events.pointInBreak
     * @see https://api.highcharts.com/highstock/navigator.yAxis.events.pointInBreak
     * @see https://api.highcharts.com/gantt/navigator.yAxis.events.pointInBreak
     */
    pointInBreak?: AxisPointBreakEventCallbackFunction;
    /**
     * (Highstock) Fires when the minimum and maximum is set for the axis,
     * either by calling the `.setExtremes()` method or by selecting an area in
     * the chart. One parameter, `event`, is passed to the function, containing
     * common event information.
     *
     * The new user set minimum and maximum values can be found by `event.min`
     * and `event.max`. These reflect the axis minimum and maximum in data
     * values. When an axis is zoomed all the way out from the "Reset zoom"
     * button, `event.min` and `event.max` are null, and the new extremes are
     * set based on `this.dataMin` and `this.dataMax`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.events.setExtremes
     */
    setExtremes?: AxisSetExtremesEventCallbackFunction;
}
/**
 * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
 *
 * @see https://api.highcharts.com/gantt/navigator.yAxis.grid
 */
export interface NavigatorYAxisGridOptions {
    /**
     * (Gantt) Set border color for the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.grid.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Gantt) Set border width of the label grid lines.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.grid.borderWidth
     */
    borderWidth?: number;
    /**
     * (Gantt) Set cell height for grid axis labels. By default this is
     * calculated from font size.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.grid.cellHeight
     */
    cellHeight?: number;
    /**
     * (Gantt) Set specific options for each column (or row for horizontal axes)
     * in the grid. Each extra column/row is its own axis, and the axis options
     * can be set here.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.grid.columns
     */
    columns?: Array<XAxisOptions>;
    /**
     * (Gantt) Enable grid on the axis labels. Defaults to true for Gantt
     * charts.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.grid.enabled
     */
    enabled?: boolean;
}
/**
 * (Gantt) Set options on specific levels in a tree grid axis. Takes precedence
 * over labels options.
 *
 * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.levels
 */
export interface NavigatorYAxisLabelsLevelsOptions {
    /**
     * (Gantt) Specify the level which the options within this object applies
     * to.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.levels.level
     */
    level?: number;
    style?: CSSObject;
}
/**
 * (Highstock) The axis labels show the number or category for each tick.
 *
 * @see https://api.highcharts.com/highstock/navigator.yAxis.labels
 */
export interface NavigatorYAxisLabelsOptions {
    /**
     * (Highstock) What part of the string the given position is anchored to.
     * Can be one of `"left"`, `"center"` or `"right"`. The exact position also
     * depends on the `labels.x` setting.
     *
     * Angular gauges and solid gauges defaults to `center`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.align
     */
    align?: ("center"|"left"|"right");
    /**
     * (Highcharts, Highstock, Gantt) For horizontal axes, the allowed degrees
     * of label rotation to prevent overlapping labels. If there is enough
     * space, labels are not rotated. As the chart gets narrower, it will start
     * rotating the labels -45 degrees, then remove every second label and try
     * again with rotations 0 and -45 etc. Set it to `false` to disable
     * rotation, which will cause the labels to word-wrap if possible.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.autoRotation
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.autoRotation
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.autoRotation
     */
    autoRotation?: Array<number>;
    /**
     * (Highcharts, Gantt) When each category width is more than this many
     * pixels, we don't apply auto rotation. Instead, we lay out the axis label
     * with word wrap. A lower limit makes sense when the label contains
     * multiple short words that don't extend the available horizontal space for
     * each label.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.autoRotationLimit
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.autoRotationLimit
     */
    autoRotationLimit?: number;
    /**
     * (Highcharts) Angular gauges and solid gauges only. The label's pixel
     * distance from the perimeter of the plot area.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.distance
     */
    distance?: number;
    /**
     * (Highstock) Enable or disable the axis labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) A format string for the axis label.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.format
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the label. The value
     * is given by `this.value`. Additional properties for `this` are `axis`,
     * `chart`, `isFirst` and `isLast`. The value of the default label formatter
     * can be retrieved by calling `this.axis.defaultLabelFormatter.call(this)`
     * within the function.
     *
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.formatter
     */
    formatter?: FormatterCallbackFunction<AxisLabelsFormatterContextObject>;
    /**
     * (Gantt) The number of pixels to indent the labels per level in a treegrid
     * axis.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.indentation
     */
    indentation?: number;
    /**
     * (Gantt) Set options on specific levels in a tree grid axis. Takes
     * precedence over labels options.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.levels
     */
    levels?: Array<NavigatorYAxisLabelsLevelsOptions>;
    /**
     * (Highstock) Horizontal axis only. When `staggerLines` is not set,
     * `maxStaggerLines` defines how many lines the axis is allowed to add to
     * automatically avoid overlapping X labels. Set to `1` to disable overlap
     * detection.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.maxStaggerLines
     */
    maxStaggerLines?: number;
    /**
     * (Highstock) How to handle overflowing labels on horizontal axis. If set
     * to `"allow"`, it will not be aligned at all. By default it `"justify"`
     * labels inside the chart area. If there is room to move it, it will be
     * aligned to the edge, else it will be removed.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highcharts, Gantt) The pixel padding for axis labels, to ensure white
     * space between them.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.padding
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.padding
     */
    padding?: number;
    /**
     * (Highcharts) Defines how the labels are be repositioned according to the
     * 3D chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.position3d
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho");
    /**
     * (Highcharts, Gantt) Whether to reserve space for the labels. By default,
     * space is reserved for the labels in these cases:
     *
     * * On all horizontal axes.
     *
     * * On vertical axes if `label.align` is `right` on a left-side axis or
     * `left` on a right-side axis.
     *
     * * On vertical axes if `label.align` is `center`.
     *
     * This can be turned off when for example the labels are rendered inside
     * the plot area instead of outside.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.reserveSpace
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.reserveSpace
     */
    reserveSpace?: boolean;
    /**
     * (Highstock) Rotation of the labels in degrees.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis labels will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `labels.position3d`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.labels.skew3d
     */
    skew3d?: boolean;
    /**
     * (Highstock) Horizontal axes only. The number of lines to spread the
     * labels over to make room or tighter labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.staggerLines
     */
    staggerLines?: number;
    /**
     * (Highstock) To show only every _n_'th label on the axis, set the step to
     * _n_. Setting the step to 2 shows every other label.
     *
     * By default, the step is calculated automatically to avoid overlap. To
     * prevent this, set it to 1\. This usually only happens on a category axis,
     * and is often a sign that you have chosen the wrong axis type.
     *
     * Read more at Axis docs => What axis should I use?
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.step
     */
    step?: number;
    /**
     * (Highstock) CSS styles for the label. Use `whiteSpace: 'nowrap'` to
     * prevent wrapping of category labels. Use `textOverflow: 'none'` to
     * prevent ellipsis (dots).
     *
     * In styled mode, the labels are styled with the `.highcharts-axis-labels`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.style
     */
    style?: CSSObject;
    /**
     * (Gantt) The symbol for the collapse and expand icon in a treegrid.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.symbol
     */
    symbol?: NavigatorYAxisLabelsSymbolOptions;
    /**
     * (Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highstock) The x position offset of the label relative to the tick
     * position on the axis. Defaults to -15 for left axis, 15 for right axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.x
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the tick
     * position on the axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.y
     */
    y?: number;
    /**
     * (Highstock) The Z index for the axis labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels.zIndex
     */
    zIndex?: number;
}
/**
 * (Gantt) The symbol for the collapse and expand icon in a treegrid.
 *
 * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.symbol
 */
export interface NavigatorYAxisLabelsSymbolOptions {
    height?: number;
    padding?: number;
    /**
     * (Gantt) The symbol type. Points to a definition function in the
     * `Highcharts.Renderer.symbols` collection.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.labels.symbol.type
     */
    type?: ("arc"|"circle"|"diamond"|"square"|"triangle"|"triangle-down");
    width?: number;
    x?: number;
    y?: number;
}
/**
 * (Highstock) Options for the navigator Y axis. Default series options for the
 * navigator yAxis are:
 *
 * (see online documentation for example)
 *
 * @see https://api.highcharts.com/highstock/navigator.yAxis
 */
export interface NavigatorYAxisOptions {
    /**
     * (Highcharts, Highstock, Gantt) When using multiple axis, the ticks of two
     * or more opposite axes will automatically be aligned by adding ticks to
     * the axis or axes with the least ticks, as if `tickAmount` were specified.
     *
     * This can be prevented by setting `alignTicks` to false. If the grid lines
     * look messy, it's a good idea to hide them for the secondary axis by
     * setting `gridLineWidth` to 0.
     *
     * If `startOnTick` or `endOnTick` in an Axis options are set to false, then
     * the `alignTicks ` will be disabled for the Axis.
     *
     * Disabled for logarithmic axes.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.alignTicks
     * @see https://api.highcharts.com/highstock/navigator.yAxis.alignTicks
     * @see https://api.highcharts.com/gantt/navigator.yAxis.alignTicks
     */
    alignTicks?: boolean;
    /**
     * (Highstock) Whether to allow decimals in this axis' ticks. When counting
     * integers, like persons or hits on a web page, decimals should be avoided
     * in the labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.allowDecimals
     */
    allowDecimals?: boolean;
    /**
     * (Highstock) When using an alternate grid color, a band is painted across
     * the plot area between every other grid line.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.alternateGridColor
     */
    alternateGridColor?: ColorString;
    /**
     * (Highcharts) In a polar chart, this is the angle of the Y axis in
     * degrees, where 0 is up and 90 is right. The angle determines the position
     * of the axis line and the labels, though the coordinate system is
     * unaffected.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.angle
     */
    angle?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array defining breaks in the axis, the
     * sections defined will be left out and all the points shifted closer to
     * each other.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.breaks
     * @see https://api.highcharts.com/highstock/navigator.yAxis.breaks
     * @see https://api.highcharts.com/gantt/navigator.yAxis.breaks
     */
    breaks?: Array<NavigatorYAxisBreaksOptions>;
    /**
     * (Highcharts, Gantt) If categories are present for the xAxis, names are
     * used instead of numbers for that axis. Since Highcharts 3.0, categories
     * can also be extracted by giving each point a name and setting axis type
     * to `category`. However, if you have multiple series, best practice
     * remains defining the `categories` array.
     *
     * Example:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.categories
     * @see https://api.highcharts.com/gantt/navigator.yAxis.categories
     */
    categories?: Array<string>;
    /**
     * (Highcharts, Highstock, Gantt) The highest allowed value for
     * automatically computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.ceiling
     * @see https://api.highcharts.com/highstock/navigator.yAxis.ceiling
     * @see https://api.highcharts.com/gantt/navigator.yAxis.ceiling
     */
    ceiling?: number;
    /**
     * (Highstock) A class name that opens for styling the axis by CSS,
     * especially in Highcharts styled mode. The class name is applied to group
     * elements for the grid, axis elements and labels.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.className
     */
    className?: string;
    /**
     * (Highstock) Configure a crosshair that follows either the mouse pointer
     * or the hovered point.
     *
     * In styled mode, the crosshairs are styled in the `.highcharts-crosshair`,
     * `.highcharts-crosshair-thin` or `.highcharts-xaxis-category` classes.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.crosshair
     */
    crosshair?: (boolean|NavigatorYAxisCrosshairOptions);
    /**
     * (Highcharts, Highstock, Gantt) For a datetime axis, the scale will
     * automatically adjust to the appropriate unit. This member gives the
     * default string representations used for each unit. For intermediate
     * values, different units may be used, for example the `day` unit can be
     * used on midnight and `hour` unit be used for intermediate values on the
     * same axis. For an overview of the replacement codes, see dateFormat.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.dateTimeLabelFormats
     * @see https://api.highcharts.com/highstock/navigator.yAxis.dateTimeLabelFormats
     * @see https://api.highcharts.com/gantt/navigator.yAxis.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: NavigatorYAxisDateTimeLabelFormatsOptions;
    /**
     * (Highstock) _Requires Accessibility module_
     *
     * Description of the axis to screen reader users.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.description
     */
    description?: string;
    /**
     * (Highstock) Whether to force the axis to end on a tick. Use this option
     * with the `maxPadding` option to control the axis end.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.endOnTick
     */
    endOnTick?: boolean;
    /**
     * (Highstock) Event handlers for the axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.events
     */
    events?: NavigatorYAxisEventsOptions;
    /**
     * (Highcharts, Highstock, Gantt) The lowest allowed value for automatically
     * computed axis extremes.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.floor
     * @see https://api.highcharts.com/highstock/navigator.yAxis.floor
     * @see https://api.highcharts.com/gantt/navigator.yAxis.floor
     */
    floor?: number;
    /**
     * (Gantt) Set grid options for the axis labels. Requires Highcharts Gantt.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.grid
     */
    grid?: NavigatorYAxisGridOptions;
    /**
     * (Highstock) Color of the grid lines extending the ticks across the plot
     * area.
     *
     * In styled mode, the stroke is given in the `.highcharts-grid-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.gridLineColor
     */
    gridLineColor?: ColorString;
    /**
     * (Highstock) The dash or dot style of the grid lines. For possible values,
     * see this demonstration.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.gridLineDashStyle
     */
    gridLineDashStyle?: DashStyleType;
    /**
     * (Highcharts) Polar charts only. Whether the grid lines should draw as a
     * polygon with straight lines between categories, or as circles. Can be
     * either `circle` or `polygon`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.gridLineInterpolation
     */
    gridLineInterpolation?: ("circle"|"polygon");
    /**
     * (Highstock) The width of the grid lines extending the ticks across the
     * plot area.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.gridLineWidth
     */
    gridLineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The Z index of the grid lines.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.gridZIndex
     * @see https://api.highcharts.com/highstock/navigator.yAxis.gridZIndex
     * @see https://api.highcharts.com/gantt/navigator.yAxis.gridZIndex
     */
    gridZIndex?: number;
    /**
     * (Highstock) An id for the axis. This can be used after render time to get
     * a pointer to the axis object through `chart.get()`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.id
     */
    id?: string;
    /**
     * (Highstock) The axis labels show the number or category for each tick.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.labels
     */
    labels?: NavigatorYAxisLabelsOptions;
    /**
     * (Highstock) The color of the line marking the axis itself.
     *
     * In styled mode, the line stroke is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the line marking the axis itself.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-line`
     * or `.highcharts-xaxis-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The maximum value of the axis. If `null`, the max value is
     * automatically calculated.
     *
     * If the endOnTick option is true, the `max` value might be rounded up.
     *
     * If a tickAmount is set, the axis may be extended beyond the set max in
     * order to reach the given number of ticks. The same may happen in a chart
     * with multiple axes, determined by chart. alignTicks, where a `tickAmount`
     * is applied internally.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.max
     */
    max?: number;
    /**
     * (Highcharts) Solid gauge only. Unless stops are set, the color to
     * represent the maximum value of the Y axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.maxColor
     */
    maxColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the max value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the highest data value to appear on
     * the edge of the plot area. When the axis' `max` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.maxPadding
     * @see https://api.highcharts.com/highstock/navigator.yAxis.maxPadding
     * @see https://api.highcharts.com/gantt/navigator.yAxis.maxPadding
     */
    maxPadding?: number;
    /**
     * (Highstock) The minimum value of the axis. If `null` the min value is
     * automatically calculated.
     *
     * If the startOnTick option is true (default), the `min` value might be
     * rounded down.
     *
     * The automatically calculated minimum value is also affected by floor,
     * softMin, minPadding, minRange as well as series.threshold and
     * series.softThreshold.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.min
     */
    min?: number;
    /**
     * (Highcharts) Solid gauge only. Unless stops are set, the color to
     * represent the minimum value of the Y axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.minColor
     */
    minColor?: ColorString;
    /**
     * (Highstock) Color of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the
     * `.highcharts-minor-grid-line` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorGridLineColor
     */
    minorGridLineColor?: ColorString;
    /**
     * (Highstock) The dash or dot style of the minor grid lines. For possible
     * values, see this demonstration.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorGridLineDashStyle
     */
    minorGridLineDashStyle?: DashStyleType;
    /**
     * (Highstock) Width of the minor, secondary grid lines.
     *
     * In styled mode, the stroke width is given in the `.highcharts-grid-line`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorGridLineWidth
     */
    minorGridLineWidth?: number;
    /**
     * (Highstock) Color for the minor tick marks.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorTickColor
     */
    minorTickColor?: ColorString;
    /**
     * (Highstock) Specific tick interval in axis units for the minor ticks. On
     * a linear axis, if `"auto"`, the minor tick interval is calculated as a
     * fifth of the tickInterval. If `null` or `undefined`, minor ticks are not
     * shown.
     *
     * On logarithmic axes, the unit is the power of the value. For example,
     * setting the minorTickInterval to 1 puts one tick on each of 0.1, 1, 10,
     * 100 etc. Setting the minorTickInterval to 0.1 produces 9 ticks between 1
     * and 10, 10 and 100 etc.
     *
     * If user settings dictate minor ticks to become too dense, they don't make
     * sense, and will be ignored to prevent performance problems.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorTickInterval
     */
    minorTickInterval?: (number|string|null);
    /**
     * (Highstock) The pixel length of the minor tick marks.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorTickLength
     */
    minorTickLength?: number;
    /**
     * (Highstock) The position of the minor tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorTickPosition
     */
    minorTickPosition?: ("inside"|"outside");
    /**
     * (Highstock) Enable or disable minor ticks. Unless minorTickInterval is
     * set, the tick interval is calculated as a fifth of the `tickInterval`.
     *
     * On a logarithmic axis, minor ticks are laid out based on a best guess,
     * attempting to enter approximately 5 minor ticks between each major tick.
     *
     * Prior to v6.0.0, ticks were unabled in auto layout by setting
     * `minorTickInterval` to `"auto"`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorTicks
     */
    minorTicks?: boolean;
    /**
     * (Highstock) The pixel width of the minor tick mark.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minorTickWidth
     */
    minorTickWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) Padding of the min value relative to the
     * length of the axis. A padding of 0.05 will make a 100px axis 5px longer.
     * This is useful when you don't want the lowest data value to appear on the
     * edge of the plot area. When the axis' `min` option is set or a max
     * extreme is set using `axis.setExtremes()`, the maxPadding will be
     * ignored.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.minPadding
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minPadding
     * @see https://api.highcharts.com/gantt/navigator.yAxis.minPadding
     */
    minPadding?: number;
    /**
     * (Highstock) The minimum tick interval allowed in axis values. For example
     * on zooming in on an axis with daily data, this can be used to prevent the
     * axis from showing hours. Defaults to the closest distance between two
     * points on the axis.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.minTickInterval
     */
    minTickInterval?: number;
    /**
     * (Highstock) The distance in pixels from the plot area to the axis line. A
     * positive offset moves the axis with it's line, labels and ticks away from
     * the plot area. This is typically used when two or more axes are displayed
     * on the same side of the plot. With multiple axes the offset is
     * dynamically adjusted to avoid collision, this can be overridden by
     * setting offset explicitly.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.offset
     */
    offset?: number;
    /**
     * (Highstock, Highcharts, Gantt) Whether to display the axis on the
     * opposite side of the normal. The normal is on the left side for vertical
     * axes and bottom for horizontal, so the opposite sides will be right and
     * top respectively. This is typically used with dual or multiple axes.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.opposite
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.opposite
     * @see https://api.highcharts.com/gantt/navigator.yAxis.opposite
     */
    opposite?: boolean;
    /**
     * (Highcharts) Refers to the index in the panes array. Used for circular
     * gauges and polar charts. When the option is not set then first pane will
     * be used.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.pane
     */
    pane?: number;
    /**
     * (Highcharts, Highstock, Gantt) An array of objects defining plot bands on
     * the Y axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands
     */
    plotBands?: Array<NavigatorYAxisPlotBandsOptions>;
    /**
     * (Highcharts, Highstock, Gantt) An array of objects representing plot
     * lines on the X axis
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines
     */
    plotLines?: Array<NavigatorYAxisPlotLinesOptions>;
    /**
     * (Highstock) Whether to reverse the axis so that the highest number is
     * closest to the origin.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.reversed
     */
    reversed?: boolean;
    /**
     * (Highcharts, Highstock) If `true`, the first series in a stack will be
     * drawn on top in a positive, non-reversed Y axis. If `false`, the first
     * series is in the base of the stack.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.reversedStacks
     * @see https://api.highcharts.com/highstock/navigator.yAxis.reversedStacks
     */
    reversedStacks?: boolean;
    /**
     * (Highstock) Whether to show the first tick label.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.showFirstLabel
     */
    showFirstLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to show the last tick label.
     * Defaults to `true` on cartesian charts, and `false` on polar charts.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.showLastLabel
     * @see https://api.highcharts.com/highstock/navigator.yAxis.showLastLabel
     * @see https://api.highcharts.com/gantt/navigator.yAxis.showLastLabel
     */
    showLastLabel?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) A soft maximum for the axis. If the series
     * data maximum is less than this, the axis will stay at this maximum, but
     * if the series data maximum is higher, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.softMax
     * @see https://api.highcharts.com/highstock/navigator.yAxis.softMax
     * @see https://api.highcharts.com/gantt/navigator.yAxis.softMax
     */
    softMax?: number;
    /**
     * (Highcharts, Highstock, Gantt) A soft minimum for the axis. If the series
     * data minimum is greater than this, the axis will stay at this minimum,
     * but if the series data minimum is lower, the axis will flex to show all
     * data.
     *
     * **Note**: The series.softThreshold option takes precedence over this
     * option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.softMin
     * @see https://api.highcharts.com/highstock/navigator.yAxis.softMin
     * @see https://api.highcharts.com/gantt/navigator.yAxis.softMin
     */
    softMin?: number;
    /**
     * (Highcharts) The stack labels show the total value for each bar in a
     * stacked column or bar chart. The label will be placed on top of positive
     * columns and below negative columns. In case of an inverted column chart
     * or a bar chart the label is placed to the right of positive bars and to
     * the left of negative bars.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels
     */
    stackLabels?: NavigatorYAxisStackLabelsOptions;
    /**
     * (Highcharts, Highstock, Gantt) For datetime axes, this decides where to
     * put the tick between weeks. 0 = Sunday, 1 = Monday.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.startOfWeek
     * @see https://api.highcharts.com/highstock/navigator.yAxis.startOfWeek
     * @see https://api.highcharts.com/gantt/navigator.yAxis.startOfWeek
     */
    startOfWeek?: number;
    /**
     * (Highcharts, Highstock, Gantt) Whether to force the axis to start on a
     * tick. Use this option with the `maxPadding` option to control the axis
     * start.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.startOnTick
     * @see https://api.highcharts.com/highstock/navigator.yAxis.startOnTick
     * @see https://api.highcharts.com/gantt/navigator.yAxis.startOnTick
     */
    startOnTick?: boolean;
    /**
     * (Gantt) For vertical axes only. Setting the static scale ensures that
     * each tick unit is translated into a fixed pixel height. For example,
     * setting the static scale to 24 results in each Y axis category taking up
     * 24 pixels, and the height of the chart adjusts. Adding or removing items
     * will make the chart resize.
     *
     * @see https://api.highcharts.com/gantt/navigator.yAxis.staticScale
     */
    staticScale?: number;
    /**
     * (Highcharts) Solid gauge series only. Color stops for the solid gauge.
     * Use this in cases where a linear gradient between a `minColor` and
     * `maxColor` is not sufficient. The stops is an array of tuples, where the
     * first item is a float between 0 and 1 assigning the relative position in
     * the gradient, and the second item is the color.
     *
     * For solid gauges, the Y axis also inherits the concept of data classes
     * from the Highmaps color axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stops
     */
    stops?: Array<[number, ColorString]>;
    /**
     * (Highcharts, Highstock, Gantt) The amount of ticks to draw on the axis.
     * This opens up for aligning the ticks of multiple charts or panes within a
     * chart. This option overrides the `tickPixelInterval` option.
     *
     * This option only has an effect on linear axes. Datetime, logarithmic or
     * category axes are not affected.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.tickAmount
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickAmount
     * @see https://api.highcharts.com/gantt/navigator.yAxis.tickAmount
     */
    tickAmount?: number;
    /**
     * (Highstock) Color for the main tick marks.
     *
     * In styled mode, the stroke is given in the `.highcharts-tick` class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickColor
     */
    tickColor?: ColorString;
    /**
     * (Highstock) The interval of the tick marks in axis units. When
     * `undefined`, the tick interval is computed to approximately follow the
     * tickPixelInterval on linear and datetime axes. On categorized axes, a
     * `undefined` tickInterval will default to 1, one category. Note that
     * datetime axes are based on milliseconds, so for example an interval of
     * one day is expressed as `24 * 3600 * 1000`.
     *
     * On logarithmic axes, the tickInterval is based on powers, so a
     * tickInterval of 1 means one tick on each of 0.1, 1, 10, 100 etc. A
     * tickInterval of 2 means a tick of 0.1, 10, 1000 etc. A tickInterval of
     * 0.2 puts a tick on 0.1, 0.2, 0.4, 0.6, 0.8, 1, 2, 4, 6, 8, 10, 20, 40
     * etc.
     *
     * If the tickInterval is too dense for labels to be drawn, Highcharts may
     * remove ticks.
     *
     * If the chart has multiple axes, the alignTicks option may interfere with
     * the `tickInterval` setting.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickInterval
     */
    tickInterval?: number;
    /**
     * (Highstock) The pixel length of the main tick marks.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickLength
     */
    tickLength?: number;
    /**
     * (Highcharts, Gantt) For categorized axes only. If `on` the tick mark is
     * placed in the center of the category, if `between` the tick mark is
     * placed between categories. The default is `between` if the `tickInterval`
     * is 1, else `on`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.tickmarkPlacement
     * @see https://api.highcharts.com/gantt/navigator.yAxis.tickmarkPlacement
     */
    tickmarkPlacement?: ("between"|"on");
    /**
     * (Highstock) If tickInterval is `null` this option sets the approximate
     * pixel interval of the tick marks. Not applicable to categorized axis.
     *
     * The tick interval is also influenced by the minTickInterval option, that,
     * by default prevents ticks from being denser than the data points.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickPixelInterval
     */
    tickPixelInterval?: number;
    /**
     * (Highstock) The position of the major tick marks relative to the axis
     * line. Can be one of `inside` and `outside`.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickPosition
     */
    tickPosition?: ("inside"|"outside");
    /**
     * (Highstock) A callback function returning array defining where the ticks
     * are laid out on the axis. This overrides the default behaviour of
     * tickPixelInterval and tickInterval. The automatic tick positions are
     * accessible through `this.tickPositions` and can be modified by the
     * callback.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickPositioner
     */
    tickPositioner?: AxisTickPositionerCallbackFunction;
    /**
     * (Highstock) An array defining where the ticks are laid out on the axis.
     * This overrides the default behaviour of tickPixelInterval and
     * tickInterval.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickPositions
     */
    tickPositions?: Array<number>;
    /**
     * (Highcharts, Highstock, Gantt) The pixel width of the major tick marks.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.tickWidth
     * @see https://api.highcharts.com/highstock/navigator.yAxis.tickWidth
     * @see https://api.highcharts.com/gantt/navigator.yAxis.tickWidth
     */
    tickWidth?: number;
    /**
     * (Highstock) The axis title, showing next to the axis line.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title
     */
    title?: NavigatorYAxisTitleOptions;
    /**
     * (Highcharts) Parallel coordinates only. Format that will be used for
     * point.y and available in tooltip.pointFormat as `{point.formattedValue}`.
     * If not set, `{point.formattedValue}` will use other options, in this
     * order:
     *
     * 1. yAxis.labels.format will be used if set
     *
     * 2. If yAxis is a category, then category name will be displayed
     *
     * 3. If yAxis is a datetime, then value will use the same format as yAxis
     * labels
     *
     * 4. If yAxis is linear/logarithmic type, then simple value will be used
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.tooltipValueFormat
     */
    tooltipValueFormat?: string;
    /**
     * (Highcharts, Gantt) The type of axis. Can be one of `linear`,
     * `logarithmic`, `datetime`, `category` or `treegrid`. Defaults to
     * `treegrid` for Gantt charts, `linear` for other chart types.
     *
     * In a datetime axis, the numbers are given in milliseconds, and tick marks
     * are placed on appropriate values, like full hours or days. In a category
     * or treegrid axis, the point names of the chart's series are used for
     * categories, if a categories array is not defined.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.type
     * @see https://api.highcharts.com/gantt/navigator.yAxis.type
     */
    type?: ("category"|"datetime"|"linear"|"logarithmic"|"treegrid");
    /**
     * (Highcharts, Gantt) Applies only when the axis `type` is `category`. When
     * `uniqueNames` is true, points are placed on the X axis according to their
     * names. If the same point name is repeated in the same or another series,
     * the point is placed on the same X position as other points of the same
     * name. When `uniqueNames` is false, the points are laid out in increasing
     * X positions regardless of their names, and the X axis category will take
     * the name of the last point in each position.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.uniqueNames
     * @see https://api.highcharts.com/gantt/navigator.yAxis.uniqueNames
     */
    uniqueNames?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether axis, including axis title, line,
     * ticks and labels, should be visible.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.visible
     * @see https://api.highcharts.com/highstock/navigator.yAxis.visible
     * @see https://api.highcharts.com/gantt/navigator.yAxis.visible
     */
    visible?: boolean;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label
 * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label
 * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label
 */
export interface NavigatorYAxisPlotBandsLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.align
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.align
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees .
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.rotation
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.rotation
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.rotation
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-band-label` class.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.style
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.style
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The string text itself. A subset of HTML
     * is supported.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.text
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.text
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.textAlign
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.textAlign
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.useHTML
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.useHTML
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot band. Can be one of "top", "middle" or "bottom".
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.verticalAlign
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.verticalAlign
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.x
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.x
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label.y
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label.y
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of objects defining plot bands on the
 * Y axis.
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands
 * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands
 * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands
 */
export interface NavigatorYAxisPlotBandsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Border color for the plot band. Also
     * requires `borderWidth` to be set.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.borderColor
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.borderColor
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) Border width for the plot band. Also
     * requires `borderColor` to be set.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.borderWidth
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.borderWidth
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-band`, to apply to each individual band.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.className
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.className
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the plot band.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.color
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.color
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot band. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.events
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.events
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.events
     */
    events?: object;
    /**
     * (Highcharts, Highstock, Gantt) The start position of the plot band in
     * axis units.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.from
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.from
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.from
     */
    from?: number;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot band
     * in Axis.removePlotBand.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.id
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.id
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.id
     */
    id?: string;
    /**
     * (Highcharts) In a gauge chart, this option determines the inner radius of
     * the plot band that stretches along the perimeter. It can be given as a
     * percentage string, like `"100%"`, or as a pixel number, like `100`. By
     * default, the inner radius is controlled by the thickness option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.innerRadius
     */
    innerRadius?: (number|string);
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.label
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.label
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.label
     */
    label?: NavigatorYAxisPlotBandsLabelOptions;
    /**
     * (Highcharts) In a gauge chart, this option determines the outer radius of
     * the plot band that stretches along the perimeter. It can be given as a
     * percentage string, like `"100%"`, or as a pixel number, like `100`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.outerRadius
     */
    outerRadius?: (number|string);
    /**
     * (Highcharts) In a gauge chart, this option sets the width of the plot
     * band stretching along the perimeter. It can be given as a percentage
     * string, like `"10%"`, or as a pixel number, like `10`. The default value
     * 10 is the same as the default tickLength, thus making the plot band act
     * as a background for the tick markers.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.thickness
     */
    thickness?: (number|string);
    /**
     * (Highcharts, Highstock, Gantt) The end position of the plot band in axis
     * units.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.to
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.to
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.to
     */
    to?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot band within the
     * chart, relative to other elements. Using the same z index as another
     * element may give unpredictable results, as the last rendered element will
     * be on top. Values from 0 to 20 make sense.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotBands.zIndex
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotBands.zIndex
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotBands.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Text labels for the plot bands
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label
 * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label
 * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label
 */
export interface NavigatorYAxisPlotLinesLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label. Can be
     * one of "left", "center" or "right".
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.align
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.align
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Rotation of the text label in degrees.
     * Defaults to 0 for horizontal plot lines and 90 for vertical lines.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.rotation
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.rotation
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.rotation
     */
    rotation?: number;
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the text label.
     *
     * In styled mode, the labels are styled by the
     * `.highcharts-plot-line-label` class.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.style
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.style
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The text itself. A subset of HTML is
     * supported.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.text
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.text
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.text
     */
    text?: string;
    /**
     * (Highcharts, Highstock, Gantt) The text alignment for the label. While
     * `align` determines where the texts anchor point is placed within the plot
     * band, `textAlign` determines how the text is aligned against its anchor
     * point. Possible values are "left", "center" and "right". Defaults to the
     * same as the `align` option.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.textAlign
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.textAlign
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.useHTML
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.useHTML
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label relative
     * to the plot line. Can be one of "top", "middle" or "bottom".
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.verticalAlign
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.verticalAlign
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.verticalAlign
     */
    verticalAlign?: ("bottom"|"middle"|"top");
    /**
     * (Highcharts, Highstock, Gantt) Horizontal position relative the
     * alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.x
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.x
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical position of the text baseline
     * relative to the alignment. Default varies by orientation.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label.y
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label.y
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) An array of objects representing plot lines on
 * the X axis
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines
 * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines
 * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines
 */
export interface NavigatorYAxisPlotLinesOptions {
    /**
     * (Highcharts, Highstock, Gantt) A custom class name, in addition to the
     * default `highcharts-plot-line`, to apply to each individual line.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.className
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.className
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock, Gantt) The color of the line.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.color
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.color
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.color
     */
    color?: ColorString;
    /**
     * (Highcharts, Highstock, Gantt) The dashing or dot style for the plot
     * line. For possible values see this overview.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.dashStyle
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.dashStyle
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highcharts, Highstock, Gantt) An object defining mouse events for the
     * plot line. Supported properties are `click`, `mouseover`, `mouseout`,
     * `mousemove`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.events
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.events
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.events
     */
    events?: any;
    /**
     * (Highcharts, Highstock, Gantt) An id used for identifying the plot line
     * in Axis.removePlotLine.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.id
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.id
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.id
     */
    id?: string;
    /**
     * (Highcharts, Highstock, Gantt) Text labels for the plot bands
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.label
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.label
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.label
     */
    label?: NavigatorYAxisPlotLinesLabelOptions;
    /**
     * (Highcharts, Highstock, Gantt) The position of the line in axis units.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.value
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.value
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.value
     */
    value?: number;
    /**
     * (Highcharts, Highstock, Gantt) The width or thickness of the plot line.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.width
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.width
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.width
     */
    width?: number;
    /**
     * (Highcharts, Highstock, Gantt) The z index of the plot line within the
     * chart.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.plotLines.zIndex
     * @see https://api.highcharts.com/highstock/navigator.yAxis.plotLines.zIndex
     * @see https://api.highcharts.com/gantt/navigator.yAxis.plotLines.zIndex
     */
    zIndex?: number;
}
/**
 * (Highcharts) The stack labels show the total value for each bar in a stacked
 * column or bar chart. The label will be placed on top of positive columns and
 * below negative columns. In case of an inverted column chart or a bar chart
 * the label is placed to the right of positive bars and to the left of negative
 * bars.
 *
 * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels
 */
export interface NavigatorYAxisStackLabelsOptions {
    /**
     * (Highcharts) Defines the horizontal alignment of the stack total label.
     * Can be one of `"left"`, `"center"` or `"right"`. The default value is
     * calculated at runtime and depends on orientation and whether the stack is
     * positive or negative.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.align
     */
    align?: AlignType;
    /**
     * (Highcharts) Allow the stack labels to overlap.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highcharts) Enable or disable the stack total labels.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) A format string for the data label. Available
     * variables are the same as for `formatter`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.format
     * @see https://api.highcharts.com/highstock/navigator.yAxis.stackLabels.format
     */
    format?: string;
    /**
     * (Highcharts) Callback JavaScript function to format the label. The value
     * is given by `this.total`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.formatter
     */
    formatter?: FormatterCallbackFunction<object>;
    /**
     * (Highcharts) Rotation of the labels in degrees.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) CSS styles for the label.
     *
     * In styled mode, the styles are set in the `.highcharts-stack-label`
     * class.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.style
     */
    style?: CSSObject;
    /**
     * (Highcharts) The text alignment for the label. While `align` determines
     * where the texts anchor point is placed with regards to the stack,
     * `textAlign` determines how the text is aligned against its anchor point.
     * Possible values are `"left"`, `"center"` and `"right"`. The default value
     * is calculated at runtime and depends on orientation and whether the stack
     * is positive or negative.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.textAlign
     */
    textAlign?: AlignType;
    /**
     * (Highcharts, Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.useHTML
     * @see https://api.highcharts.com/highstock/navigator.yAxis.stackLabels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts) Defines the vertical alignment of the stack total label. Can
     * be one of `"top"`, `"middle"` or `"bottom"`. The default value is
     * calculated at runtime and depends on orientation and whether the stack is
     * positive or negative.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts) The x position offset of the label relative to the left of
     * the stacked bar. The default value is calculated at runtime and depends
     * on orientation and whether the stack is positive or negative.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.x
     */
    x?: number;
    /**
     * (Highcharts) The y position offset of the label relative to the tick
     * position on the axis. The default value is calculated at runtime and
     * depends on orientation and whether the stack is positive or negative.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.stackLabels.y
     */
    y?: number;
}
/**
 * (Highstock) The axis title, showing next to the axis line.
 *
 * @see https://api.highcharts.com/highstock/navigator.yAxis.title
 */
export interface NavigatorYAxisTitleOptions {
    /**
     * (Highstock) Alignment of the title relative to the axis values. Possible
     * values are "low", "middle" or "high".
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.align
     */
    align?: ("high"|"low"|"middle");
    /**
     * (Highcharts) Deprecated. Set the `text` to `null` to disable the title.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.enabled
     */
    enabled?: string;
    /**
     * (Highstock) The pixel distance between the axis labels and the title.
     * Positive values are outside the axis line, negative are inside.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.margin
     */
    margin?: number;
    /**
     * (Highstock) The distance of the axis title from the axis line. By
     * default, this distance is computed from the offset width of the labels,
     * the labels' distance from the axis and the title's margin. However when
     * the offset option is set, it overrides all this.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.offset
     */
    offset?: number;
    /**
     * (Highcharts) Defines how the title is repositioned according to the 3D
     * chart orientation.
     *
     * - `'offset'`: Maintain a fixed horizontal/vertical distance from the tick
     * marks, despite the chart orientation. This is the backwards compatible
     * behavior, and causes skewing of X and Z axes.
     *
     * - `'chart'`: Preserve 3D position relative to the chart. This looks nice,
     * but hard to read if the text isn't forward-facing.
     *
     * - `'flap'`: Rotated text along the axis to compensate for the chart
     * orientation. This tries to maintain text as legible as possible on all
     * orientations.
     *
     * - `'ortho'`: Rotated text along the axis direction so that the labels are
     * orthogonal to the axis. This is very similar to `'flap'`, but prevents
     * skewing the labels (X and Y scaling are still present).
     *
     * - `undefined`: Will use the config from `labels.position3d`
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.position3d
     */
    position3d?: ("chart"|"flap"|"offset"|"ortho"|null);
    /**
     * (Highcharts, Highstock, Gantt) Whether to reserve space for the title
     * when laying out the axis.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.reserveSpace
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.reserveSpace
     * @see https://api.highcharts.com/gantt/navigator.yAxis.title.reserveSpace
     */
    reserveSpace?: boolean;
    /**
     * (Highstock) The rotation of the text in degrees. 0 is horizontal, 270 is
     * vertical reading from bottom to top.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.rotation
     */
    rotation?: number;
    /**
     * (Highcharts) If enabled, the axis title will skewed to follow the
     * perspective.
     *
     * This will fix overlapping labels and titles, but texts become less
     * legible due to the distortion.
     *
     * The final appearance depends heavily on `title.position3d`.
     *
     * A `null` value will use the config from `labels.skew3d`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.skew3d
     */
    skew3d?: (boolean|null);
    /**
     * (Highstock) CSS styles for the title. If the title text is longer than
     * the axis length, it will wrap to multiple lines by default. This can be
     * customized by setting `textOverflow: 'ellipsis'`, by setting a specific
     * `width` or by setting `whiteSpace: 'nowrap'`.
     *
     * In styled mode, the stroke width is given in the `.highcharts-axis-title`
     * class.
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) The actual text of the axis title.
     * Horizontal texts can contain HTML, but rotated texts are painted using
     * vector techniques and must be clean text. The Y axis title is disabled by
     * setting the `text` option to `undefined`.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.text
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.text
     * @see https://api.highcharts.com/gantt/navigator.yAxis.title.text
     */
    text?: any;
    /**
     * (Highstock) Alignment of the text, can be `"left"`, `"right"` or
     * `"center"`. Default alignment depends on the title.align:
     *
     * Horizontal axes:
     *
     * - for `align` = `"low"`, `textAlign` is set to `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"`, `textAlign` is set to `right`
     *
     * Vertical axes:
     *
     * - for `align` = `"low"` and `opposite` = `true`, `textAlign` is set to
     * `right`
     *
     * - for `align` = `"low"` and `opposite` = `false`, `textAlign` is set to
     * `left`
     *
     * - for `align` = `"middle"`, `textAlign` is set to `center`
     *
     * - for `align` = `"high"` and `opposite` = `true` `textAlign` is set to
     * `left`
     *
     * - for `align` = `"high"` and `opposite` = `false` `textAlign` is set to
     * `right`
     *
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.textAlign
     */
    textAlign?: string;
    /**
     * (Highcharts, Highstock, Gantt) Whether to use HTML to render the axis
     * title.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.useHTML
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.useHTML
     * @see https://api.highcharts.com/gantt/navigator.yAxis.title.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal pixel offset of the title
     * position.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.x
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.x
     * @see https://api.highcharts.com/gantt/navigator.yAxis.title.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical pixel offset of the title
     * position.
     *
     * @see https://api.highcharts.com/highcharts/navigator.yAxis.title.y
     * @see https://api.highcharts.com/highstock/navigator.yAxis.title.y
     * @see https://api.highcharts.com/gantt/navigator.yAxis.title.y
     */
    y?: number;
}
/**
 * (Highcharts, Highstock, Gantt) Options for displaying a message like "No data
 * to display". This feature requires the file no-data-to-display.js to be
 * loaded in the page. The actual text to display is set in the lang.noData
 * option.
 *
 * @see https://api.highcharts.com/highcharts/noData
 * @see https://api.highcharts.com/highstock/noData
 * @see https://api.highcharts.com/gantt/noData
 */
export interface NoDataOptions {
    /**
     * (Highcharts, Highstock, Gantt) An object of additional SVG attributes for
     * the no-data label.
     *
     * @see https://api.highcharts.com/highcharts/noData.attr
     * @see https://api.highcharts.com/highstock/noData.attr
     * @see https://api.highcharts.com/gantt/noData.attr
     */
    attr?: SVGAttributes;
    /**
     * (Highcharts, Highstock, Gantt) The position of the no-data label,
     * relative to the plot area.
     *
     * @see https://api.highcharts.com/highcharts/noData.position
     * @see https://api.highcharts.com/highstock/noData.position
     * @see https://api.highcharts.com/gantt/noData.position
     */
    position?: (AlignObject|NoDataPositionOptions);
    /**
     * (Highcharts, Highstock, Gantt) CSS styles for the no-data label.
     *
     * @see https://api.highcharts.com/highcharts/noData.style
     * @see https://api.highcharts.com/highstock/noData.style
     * @see https://api.highcharts.com/gantt/noData.style
     */
    style?: CSSObject;
    /**
     * (Highcharts, Highstock, Gantt) Whether to insert the label as HTML, or as
     * pseudo-HTML rendered with SVG.
     *
     * @see https://api.highcharts.com/highcharts/noData.useHTML
     * @see https://api.highcharts.com/highstock/noData.useHTML
     * @see https://api.highcharts.com/gantt/noData.useHTML
     */
    useHTML?: boolean;
}
/**
 * (Highcharts, Highstock, Gantt) The position of the no-data label, relative to
 * the plot area.
 *
 * @see https://api.highcharts.com/highcharts/noData.position
 * @see https://api.highcharts.com/highstock/noData.position
 * @see https://api.highcharts.com/gantt/noData.position
 */
export interface NoDataPositionOptions {
    /**
     * (Highcharts, Highstock, Gantt) Horizontal alignment of the label.
     *
     * @see https://api.highcharts.com/highcharts/noData.position.align
     * @see https://api.highcharts.com/highstock/noData.position.align
     * @see https://api.highcharts.com/gantt/noData.position.align
     */
    align?: AlignType;
    /**
     * (Highcharts, Highstock, Gantt) Vertical alignment of the label.
     *
     * @see https://api.highcharts.com/highcharts/noData.position.verticalAlign
     * @see https://api.highcharts.com/highstock/noData.position.verticalAlign
     * @see https://api.highcharts.com/gantt/noData.position.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highcharts, Highstock, Gantt) Horizontal offset of the label, in pixels.
     *
     * @see https://api.highcharts.com/highcharts/noData.position.x
     * @see https://api.highcharts.com/highstock/noData.position.x
     * @see https://api.highcharts.com/gantt/noData.position.x
     */
    x?: number;
    /**
     * (Highcharts, Highstock, Gantt) Vertical offset of the label, in pixels.
     *
     * @see https://api.highcharts.com/highcharts/noData.position.y
     * @see https://api.highcharts.com/highstock/noData.position.y
     * @see https://api.highcharts.com/gantt/noData.position.y
     */
    y?: number;
}
/**
 * Normalized interval.
 */
export interface NormalizedIntervalObject {
    /**
     * The count.
     */
    count: number;
    /**
     * The interval in axis values (ms).
     */
    unitRange: number;
}
/**
 * An object containing `left` and `top` properties for the position in the
 * page.
 */
export interface OffsetObject {
    /**
     * Left distance to the page border.
     */
    left: number;
    /**
     * Top distance to the page border.
     */
    top: number;
}
/**
 * The option tree for every chart.
 */
export interface Options {
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring
     * accessibility for the chart. Requires the accessibility module to be
     * loaded. For a description of the module and information on its features,
     * see Highcharts Accessibility.
     *
     * @see https://api.highcharts.com/highcharts/accessibility
     * @see https://api.highcharts.com/highstock/accessibility
     * @see https://api.highcharts.com/highmaps/accessibility
     * @see https://api.highcharts.com/gantt/accessibility
     */
    accessibility?: AccessibilityOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for configuring
     * annotations, for example labels, arrows or shapes. Annotations can be
     * tied to points, axis coordinates or chart pixel coordinates.
     *
     * @see https://api.highcharts.com/highcharts/annotations
     * @see https://api.highcharts.com/highstock/annotations
     * @see https://api.highcharts.com/highmaps/annotations
     * @see https://api.highcharts.com/gantt/annotations
     */
    annotations?: Array<AnnotationsOptions>;
    /**
     * (Highcharts, Highstock) Options for the Boost module. The Boost module
     * allows certain series types to be rendered by WebGL instead of the
     * default SVG. This allows hundreds of thousands of data points to be
     * rendered in milliseconds. In addition to the WebGL rendering it saves
     * time by skipping processing and inspection of the data wherever possible.
     * This introduces some limitations to what features are available in Boost
     * mode. See the docs for details.
     *
     * In addition to the global `boost` option, each series has a
     * boostThreshold that defines when the boost should kick in.
     *
     * Requires the `modules/boost.js` module.
     *
     * @see https://api.highcharts.com/highcharts/boost
     * @see https://api.highcharts.com/highstock/boost
     */
    boost?: BoostOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) General options for the chart.
     *
     * @see https://api.highcharts.com/highcharts/chart
     * @see https://api.highcharts.com/highstock/chart
     * @see https://api.highcharts.com/highmaps/chart
     * @see https://api.highcharts.com/gantt/chart
     */
    chart?: ChartOptions;
    /**
     * (Highcharts, Highmaps) A color axis for choropleth maps and heat maps.
     * Visually, the color axis will appear as a gradient or as separate items
     * inside the legend, depending on whether the axis is scalar or based on
     * data classes.
     *
     * For supported color formats, see the docs article about colors.
     *
     * A scalar color axis is represented by a gradient. The colors either range
     * between the minColor and the maxColor, or for more fine grained control
     * the colors can be defined in stops. Often times, the color axis needs to
     * be adjusted to get the right color spread for the data. In addition to
     * stops, consider using a logarithmic axis type, or setting min and max to
     * avoid the colors being determined by outliers.
     *
     * When dataClasses are used, the ranges are subdivided into separate
     * classes like categories based on their values. This can be used for
     * ranges between two values, but also for a true category. However, when
     * your data is categorized, it may be as convenient to add each category to
     * a separate series.
     *
     * See the Axis object for programmatic access to the axis.
     *
     * @see https://api.highcharts.com/highcharts/colorAxis
     * @see https://api.highcharts.com/highmaps/colorAxis
     */
    colorAxis?: ColorAxisOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) An array containing the default
     * colors for the chart's series. When all colors are used, new colors are
     * pulled from the start again.
     *
     * Default colors can also be set on a series or series.type basis, see
     * column.colors, pie.colors.
     *
     * In styled mode, the colors option doesn't exist. Instead, colors are
     * defined in CSS and applied either through series or point class names, or
     * through the chart.colorCount option.
     *
     * ### Legacy
     *
     * In Highcharts 3.x, the default colors were:
     *
     * (see online documentation for example)
     *
     * In Highcharts 2.x, the default colors were:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/colors
     * @see https://api.highcharts.com/highstock/colors
     * @see https://api.highcharts.com/highmaps/colors
     * @see https://api.highcharts.com/gantt/colors
     */
    colors?: Array<ColorString>;
    /**
     * (Gantt) The Pathfinder module allows you to define connections between
     * any two points, represented as lines - optionally with markers for the
     * start and/or end points. Multiple algorithms are available for
     * calculating how the connecting lines are drawn.
     *
     * Connector functionality requires Highcharts Gantt to be loaded. In Gantt
     * charts, the connectors are used to draw dependencies between tasks.
     *
     * @see https://api.highcharts.com/gantt/connectors
     */
    connectors?: ConnectorsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Highchart by default puts a
     * credits label in the lower right corner of the chart. This can be changed
     * using these options.
     *
     * @see https://api.highcharts.com/highcharts/credits
     * @see https://api.highcharts.com/highstock/credits
     * @see https://api.highcharts.com/highmaps/credits
     * @see https://api.highcharts.com/gantt/credits
     */
    credits?: CreditsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Data module provides a
     * simplified interface for adding data to a chart from sources like CVS,
     * HTML tables or grid views. See also the tutorial article on the Data
     * module.
     *
     * It requires the `modules/data.js` file to be loaded.
     *
     * Please note that the default way of adding data in Highcharts, without
     * the need of a module, is through the series.data option.
     *
     * @see https://api.highcharts.com/highcharts/data
     * @see https://api.highcharts.com/highstock/data
     * @see https://api.highcharts.com/highmaps/data
     * @see https://api.highcharts.com/gantt/data
     */
    data?: DataOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Styled mode only. Configuration
     * object for adding SVG definitions for reusable elements. See gradients,
     * shadows and patterns for more information and code examples.
     *
     * @see https://api.highcharts.com/highcharts/defs
     * @see https://api.highcharts.com/highstock/defs
     * @see https://api.highcharts.com/highmaps/defs
     * @see https://api.highcharts.com/gantt/defs
     */
    defs?: any;
    /**
     * (Highcharts, Highstock, Highmaps) Options for drill down, the concept of
     * inspecting increasingly high resolution data through clicking on chart
     * items like columns or pie slices.
     *
     * The drilldown feature requires the drilldown.js file to be loaded, found
     * in the modules directory of the download package, or online at
     * code.highcharts.com/modules/drilldown.js.
     *
     * @see https://api.highcharts.com/highcharts/drilldown
     * @see https://api.highcharts.com/highstock/drilldown
     * @see https://api.highcharts.com/highmaps/drilldown
     */
    drilldown?: DrilldownOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the exporting
     * module. For an overview on the matter, see the docs.
     *
     * @see https://api.highcharts.com/highcharts/exporting
     * @see https://api.highcharts.com/highstock/exporting
     * @see https://api.highcharts.com/highmaps/exporting
     * @see https://api.highcharts.com/gantt/exporting
     */
    exporting?: ExportingOptions;
    global?: GlobalOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) HTML labels that can be
     * positioned anywhere in the chart area.
     *
     * @see https://api.highcharts.com/highcharts/labels
     * @see https://api.highcharts.com/highstock/labels
     * @see https://api.highcharts.com/highmaps/labels
     * @see https://api.highcharts.com/gantt/labels
     */
    labels?: LabelsOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Language object. The language
     * object is global and it can't be set on each chart initiation. Instead,
     * use `Highcharts.setOptions` to set it before any chart is initialized.
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highcharts/lang
     * @see https://api.highcharts.com/highstock/lang
     * @see https://api.highcharts.com/highmaps/lang
     * @see https://api.highcharts.com/gantt/lang
     */
    lang?: LangOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The legend is a box containing a
     * symbol and name for each series item or point item in the chart. Each
     * series (or points in case of pie charts) is represented by a symbol and
     * its name in the legend.
     *
     * It is possible to override the symbol creator function and create custom
     * legend symbols.
     *
     * @see https://api.highcharts.com/highcharts/legend
     * @see https://api.highcharts.com/highstock/legend
     * @see https://api.highcharts.com/highmaps/legend
     * @see https://api.highcharts.com/gantt/legend
     */
    legend?: LegendOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The loading options control the
     * appearance of the loading screen that covers the plot area on chart
     * operations. This screen only appears after an explicit call to
     * `chart.showLoading()`. It is a utility for developers to communicate to
     * the end user that something is going on, for example while retrieving new
     * data via an XHR connection. The "Loading..." text itself is not part of
     * this configuration object, but part of the `lang` object.
     *
     * @see https://api.highcharts.com/highcharts/loading
     * @see https://api.highcharts.com/highstock/loading
     * @see https://api.highcharts.com/highmaps/loading
     * @see https://api.highcharts.com/gantt/loading
     */
    loading?: LoadingOptions;
    mapNavigation?: MapNavigationOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) A collection of options for
     * buttons and menus appearing in the exporting module.
     *
     * @see https://api.highcharts.com/highcharts/navigation
     * @see https://api.highcharts.com/highstock/navigation
     * @see https://api.highcharts.com/highmaps/navigation
     * @see https://api.highcharts.com/gantt/navigation
     */
    navigation?: NavigationOptions;
    /**
     * (Highstock) The navigator is a small series below the main series,
     * displaying a view of the entire data set. It provides tools to zoom in
     * and out on parts of the data as well as panning across the dataset.
     *
     * @see https://api.highcharts.com/highstock/navigator
     */
    navigator?: NavigatorOptions;
    /**
     * (Highcharts, Highstock, Gantt) Options for displaying a message like "No
     * data to display". This feature requires the file no-data-to-display.js to
     * be loaded in the page. The actual text to display is set in the
     * lang.noData option.
     *
     * @see https://api.highcharts.com/highcharts/noData
     * @see https://api.highcharts.com/highstock/noData
     * @see https://api.highcharts.com/gantt/noData
     */
    noData?: NoDataOptions;
    /**
     * (Highcharts) The pane serves as a container for axes and backgrounds for
     * circular gauges and polar charts.
     *
     * @see https://api.highcharts.com/highcharts/pane
     */
    pane?: PaneOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The plotOptions is a wrapper
     * object for config objects for each series type. The config objects for
     * each series can also be overridden for each series item as given in the
     * series array.
     *
     * Configuration options for the series are given in three levels. Options
     * for all series in a chart are given in the plotOptions.series object.
     * Then options for all series of a specific type are given in the
     * plotOptions of that type, for example `plotOptions.line`. Next, options
     * for one single series are given in the series array.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions
     * @see https://api.highcharts.com/highstock/plotOptions
     * @see https://api.highcharts.com/highmaps/plotOptions
     * @see https://api.highcharts.com/gantt/plotOptions
     */
    plotOptions?: PlotOptions;
    /**
     * (Highstock) The range selector is a tool for selecting ranges to display
     * within the chart. It provides buttons to select preconfigured ranges in
     * the chart, like 1 day, 1 week, 1 month etc. It also provides input boxes
     * where min and max dates can be manually input.
     *
     * @see https://api.highcharts.com/highstock/rangeSelector
     */
    rangeSelector?: RangeSelectorOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Allows setting a set of rules to
     * apply for different screen or chart sizes. Each rule specifies additional
     * chart options.
     *
     * @see https://api.highcharts.com/highcharts/responsive
     * @see https://api.highcharts.com/highstock/responsive
     * @see https://api.highcharts.com/highmaps/responsive
     * @see https://api.highcharts.com/gantt/responsive
     */
    responsive?: ResponsiveOptions;
    /**
     * (Highstock) The scrollbar is a means of panning over the X axis of a
     * stock chart. Scrollbars can also be applied to other types of axes.
     *
     * Another approach to scrollable charts is the chart.scrollablePlotArea
     * option that is especially suitable for simpler cartesian charts on
     * mobile.
     *
     * In styled mode, all the presentational options for the scrollbar are
     * replaced by the classes `.highcharts-scrollbar-thumb`,
     * `.highcharts-scrollbar-arrow`, `.highcharts-scrollbar-button`,
     * `.highcharts-scrollbar-rifles` and `.highcharts-scrollbar-track`.
     *
     * @see https://api.highcharts.com/highstock/scrollbar
     */
    scrollbar?: ScrollbarOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Series options for specific data
     * and the data itself. In TypeScript you have to cast the series options to
     * specific series types, to get all possible options for a series.
     *
     * @see https://api.highcharts.com/highcharts/series
     * @see https://api.highcharts.com/highstock/series
     * @see https://api.highcharts.com/highmaps/series
     * @see https://api.highcharts.com/gantt/series
     */
    series?: Array<SeriesOptionsType>;
    /**
     * (Highstock) Configure the stockTools gui strings in the chart. Requires
     * the [stockTools module]() to be loaded. For a description of the module
     * and information on its features, see [Highcharts StockTools]().
     *
     * @see https://api.highcharts.com/highstock/stockTools
     */
    stockTools?: (object|StockToolsOptions);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The chart's subtitle. This can
     * be used both to display a subtitle below the main title, and to display
     * random text anywhere in the chart. The subtitle can be updated after
     * chart initialization through the `Chart.setTitle` method.
     *
     * @see https://api.highcharts.com/highcharts/subtitle
     * @see https://api.highcharts.com/highstock/subtitle
     * @see https://api.highcharts.com/highmaps/subtitle
     * @see https://api.highcharts.com/gantt/subtitle
     */
    subtitle?: SubtitleOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Time options that can apply
     * globally or to individual charts. These settings affect how `datetime`
     * axes are laid out, how tooltips are formatted, how series
     * pointIntervalUnit works and how the Highstock range selector handles
     * time.
     *
     * The common use case is that all charts in the same Highcharts object
     * share the same time settings, in which case the global settings are set
     * using `setOptions`.(see online documentation for example)
     *
     * Since v6.0.5, the time options were moved from the `global` obect to the
     * `time` object, and time options can be set on each individual chart.
     *
     * @see https://api.highcharts.com/highcharts/time
     * @see https://api.highcharts.com/highstock/time
     * @see https://api.highcharts.com/highmaps/time
     * @see https://api.highcharts.com/gantt/time
     */
    time?: TimeOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The chart's main title.
     *
     * @see https://api.highcharts.com/highcharts/title
     * @see https://api.highcharts.com/highstock/title
     * @see https://api.highcharts.com/highmaps/title
     * @see https://api.highcharts.com/gantt/title
     */
    title?: TitleOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) Options for the tooltip that
     * appears when the user hovers over a series or point.
     *
     * @see https://api.highcharts.com/highcharts/tooltip
     * @see https://api.highcharts.com/highstock/tooltip
     * @see https://api.highcharts.com/highmaps/tooltip
     * @see https://api.highcharts.com/gantt/tooltip
     */
    tooltip?: TooltipOptions;
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The X axis or category axis.
     * Normally this is the horizontal axis, though if the chart is inverted
     * this is the vertical axis. In case of multiple axes, the xAxis node is an
     * array of configuration objects.
     *
     * See the Axis class for programmatic access to the axis.
     *
     * @see https://api.highcharts.com/highcharts/xAxis
     * @see https://api.highcharts.com/highstock/xAxis
     * @see https://api.highcharts.com/highmaps/xAxis
     * @see https://api.highcharts.com/gantt/xAxis
     */
    xAxis?: (XAxisOptions|Array<XAxisOptions>);
    /**
     * (Highcharts, Highstock, Highmaps, Gantt) The Y axis or value axis.
     * Normally this is the vertical axis, though if the chart is inverted this
     * is the horizontal axis. In case of multiple axes, the yAxis node is an
     * array of configuration objects.
     *
     * See the Axis object for programmatic access to the axis.
     *
     * @see https://api.highcharts.com/highcharts/yAxis
     * @see https://api.highcharts.com/highstock/yAxis
     * @see https://api.highcharts.com/highmaps/yAxis
     * @see https://api.highcharts.com/gantt/yAxis
     */
    yAxis?: (YAxisOptions|Array<YAxisOptions>);
    /**
     * (Highcharts) The Z axis or depth axis for 3D plots.
     *
     * See the Axis class for programmatic access to the axis.
     *
     * @see https://api.highcharts.com/highcharts/zAxis
     */
    zAxis?: (ZAxisOptions|Array<ZAxisOptions>);
}
/**
 * (Highcharts) An array of background items for the pane.
 *
 * @see https://api.highcharts.com/highcharts/pane.background
 */
export interface PaneBackgroundOptions {
    /**
     * (Highcharts) The background color or gradient for the pane.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.backgroundColor
     */
    backgroundColor?: GradientColorObject;
    /**
     * (Highcharts) The pane background border color.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highcharts) The pixel border width of the pane background.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highcharts) The class name for this background.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.className
     */
    className?: string;
    /**
     * (Highcharts) The inner radius of the pane background. Can be either
     * numeric (pixels) or a percentage string.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.innerRadius
     */
    innerRadius?: (number|string);
    /**
     * (Highcharts) The outer radius of the circular pane background. Can be
     * either numeric (pixels) or a percentage string.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.outerRadius
     */
    outerRadius?: (number|string);
    /**
     * (Highcharts) The shape of the pane background. When `solid`, the
     * background is circular. When `arc`, the background extends only from the
     * min to the max of the value axis.
     *
     * @see https://api.highcharts.com/highcharts/pane.background.shape
     */
    shape?: ("arc"|"circle"|"solid");
}
/**
 * (Highcharts) The pane serves as a container for axes and backgrounds for
 * circular gauges and polar charts.
 *
 * @see https://api.highcharts.com/highcharts/pane
 */
export interface PaneOptions {
    /**
     * (Highcharts) An array of background items for the pane.
     *
     * @see https://api.highcharts.com/highcharts/pane.background
     */
    background?: Array<PaneBackgroundOptions>;
    /**
     * (Highcharts) The center of a polar chart or angular gauge, given as an
     * array of [x, y] positions. Positions can be given as integers that
     * transform to pixels, or as percentages of the plot area size.
     *
     * @see https://api.highcharts.com/highcharts/pane.center
     */
    center?: Array<(string|number)>;
    /**
     * (Highcharts) The end angle of the polar X axis or gauge value axis, given
     * in degrees where 0 is north. Defaults to startAngle
     *
     * + 360.
     *
     * @see https://api.highcharts.com/highcharts/pane.endAngle
     */
    endAngle?: number;
    /**
     * (Highcharts) The size of the pane, either as a number defining pixels, or
     * a percentage defining a percentage of the plot are.
     *
     * @see https://api.highcharts.com/highcharts/pane.size
     */
    size?: (number|string);
    /**
     * (Highcharts) The start angle of the polar X axis or gauge axis, given in
     * degrees where 0 is north. Defaults to 0.
     *
     * @see https://api.highcharts.com/highcharts/pane.startAngle
     */
    startAngle?: number;
}
/**
 * Holds a pattern definition.
 */
export interface PatternObject {
    /**
     * Animation options for the image pattern loading.
     */
    animation?: (boolean|AnimationOptionsObject);
    /**
     * Pattern options
     */
    pattern: PatternOptionsObject;
}
/**
 * Pattern options
 */
export interface PatternOptionsObject {
    /**
     * For automatically calculated width and height on images, it is possible
     * to set an aspect ratio. The image will be zoomed to fill the bounding
     * box, maintaining the aspect ratio defined.
     */
    aspectRatio: number;
    /**
     * Pattern color, used as default path stroke.
     */
    color: ColorString;
    /**
     * Analogous to pattern.width.
     */
    height: number;
    /**
     * ID to assign to the pattern. This is automatically computed if not added,
     * and identical patterns are reused. To refer to an existing pattern for a
     * Highcharts color, use `color: "url(#pattern-id)"`.
     */
    id: string;
    /**
     * URL to an image to use as the pattern.
     */
    image: string;
    /**
     * Opacity of the pattern as a float value from 0 to 1.
     */
    opacity: number;
    /**
     * Either an SVG path as string, or an object. As an object, supply the path
     * string in the `path.d` property. Other supported properties are standard
     * SVG attributes like `path.stroke` and `path.fill`. If a path is supplied
     * for the pattern, the `image` property is ignored.
     */
    path: (string|SVGAttributes);
    /**
     * Width of the pattern. For images this is automatically set to the width
     * of the element bounding box if not supplied. For non-image patterns the
     * default is 32px. Note that automatic resizing of image patterns to fill a
     * bounding box dynamically is only supported for patterns with an
     * automatically calculated ID.
     */
    with: number;
    /**
     * Horizontal offset of the pattern. Defaults to 0.
     */
    x?: number;
    /**
     * Vertical offset of the pattern. Defaults to 0.
     */
    y?: number;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.animation
 */
export interface PlotAbandsAnimationOptions {
    duration?: number;
}
export interface PlotAbandsBottomLineOptions {
    styles?: PlotAbandsBottomLineStylesOptions;
}
export interface PlotAbandsBottomLineStylesOptions {
    /**
     * (Highstock) Pixel width of the line.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.bottomLine.styles.lineWidth
     */
    lineWidth?: number;
}
/**
 * (Gantt) Marker options specific to the end markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker
 */
export interface PlotAbandsConnectorsEndMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector end markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker.width
     */
    width?: number;
}
/**
 * (Gantt) Marker options for this chart's Pathfinder connectors. Note that this
 * option is overridden by the `startMarker` and `endMarker` options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker
 */
export interface PlotAbandsConnectorsMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.radius
     */
    radius?: number;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker.width
     */
    width?: number;
}
/**
 * (Gantt) Override Pathfinder connector options for a series. Requires
 * Highcharts Gantt to be loaded.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors
 */
export interface PlotAbandsConnectorsOptions {
    /**
     * (Gantt) Set the default dash style for this chart's connecting lines.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.dashStyle
     */
    dashStyle?: string;
    /**
     * (Gantt) Marker options specific to the end markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.endMarker
     */
    endMarker?: PlotAbandsConnectorsEndMarkerOptions;
    /**
     * (Gantt) Set the default color for this chart's Pathfinder connecting
     * lines. Defaults to the color of the point being connected.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the default pixel width for this chart's Pathfinder
     * connecting lines.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Marker options for this chart's Pathfinder connectors. Note that
     * this option is overridden by the `startMarker` and `endMarker` options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.marker
     */
    marker?: PlotAbandsConnectorsMarkerOptions;
    /**
     * (Gantt) Marker options specific to the start markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker
     */
    startMarker?: PlotAbandsConnectorsStartMarkerOptions;
    /**
     * (Gantt) Set the default pathfinder algorithm to use for this chart. It is
     * possible to define your own algorithms by adding them to the
     * Highcharts.Pathfinder.prototype.algorithms object before the chart has
     * been created.
     *
     * The default algorithms are as follows:
     *
     * `straight`: Draws a straight line between the connecting points. Does not
     * avoid other points when drawing.
     *
     * `simpleConnect`: Finds a path between the points using right angles only.
     * Takes only starting/ending points into account, and will not avoid other
     * points.
     *
     * `fastAvoid`: Finds a path between the points using right angles only.
     * Will attempt to avoid other points, but its focus is performance over
     * accuracy. Works well with less dense datasets.
     *
     * Default value: `straight` is used as default for most series types, while
     * `simpleConnect` is used as default for Gantt series, to show dependencies
     * between points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.type
     */
    type?: ("straight"|"fastAvoid"|"simpleConnect");
}
/**
 * (Gantt) Marker options specific to the start markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker
 */
export interface PlotAbandsConnectorsStartMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector start markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors.startMarker.width
     */
    width?: number;
}
/**
 * (Highstock) Data grouping is the concept of sampling the data values into
 * larger blocks in order to ease readability and increase performance of the
 * JavaScript charts. Highstock by default applies data grouping when the points
 * become closer than a certain pixel value, determined by the `groupPixelWidth`
 * option.
 *
 * If data grouping is applied, the grouping information of grouped points can
 * be read from the Point.dataGroup. If point options other than the data itself
 * are set, for example `name` or `color` or custom properties, the grouping
 * logic doesn't know how to group it. In this case the options of the first
 * point instance are copied over to the group point. This can be altered
 * through a custom `approximation` callback function.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping
 */
export interface PlotAbandsDataGroupingOptions {
    /**
     * (Highstock) The method of approximation inside a group. When for example
     * 30 days are grouped into one month, this determines what value should
     * represent the group. Possible values are "average", "averages", "open",
     * "high", "low", "close" and "sum". For OHLC and candlestick series the
     * approximation is "ohlc" by default, which finds the open, high, low and
     * close values within all the grouped data. For ranges, the approximation
     * is "range", which finds the low and high values. For multi-dimensional
     * data, like ranges and OHLC, "averages" will compute the average for each
     * dimension.
     *
     * Custom aggregate methods can be added by assigning a callback function as
     * the approximation. This function takes a numeric array as the argument
     * and should return a single numeric value or `null`. Note that the numeric
     * array will never contain null values, only true numbers. Instead, if null
     * values are present in the raw data, the numeric array will have an
     * `.hasNulls` property set to `true`. For single-value data sets the data
     * is available in the first argument of the callback function. For OHLC
     * data sets, all the open values are in the first argument, all high values
     * in the second etc.
     *
     * Since v4.2.7, grouping meta data is available in the approximation
     * callback from `this.dataGroupInfo`. It can be used to extract information
     * from the raw data.
     *
     * Defaults to `average` for line-type series, `sum` for columns, `range`
     * for range series and `ohlc` for OHLC and candlestick.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.approximation
     */
    approximation?: ("average"|"averages"|"close"|"high"|"low"|"open"|"sum");
    /**
     * (Highstock) Datetime formats for the header of the tooltip in a stock
     * chart. The format can vary within a chart depending on the currently
     * selected time range and the current data grouping.
     *
     * The default formats are:
     *
     * (see online documentation for example)
     *
     * For each of these array definitions, the first item is the format used
     * when the active time span is one unit. For instance, if the current data
     * applies to one week, the first item of the week array is used. The second
     * and third items are used when the active time span is more than two
     * units. For instance, if the current data applies to two weeks, the second
     * and third item of the week array are used, and applied to the start and
     * end date of the time span.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: object;
    /**
     * (Highstock) Enable or disable data grouping.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) When data grouping is forced, it runs no matter how small the
     * intervals are. This can be handy for example when the sum should be
     * calculated for values appearing at random times within each hour.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.forced
     */
    forced?: boolean;
    /**
     * (Highstock) By default only points within the visible range are grouped.
     * Enabling this option will force data grouping to calculate all grouped
     * points for a given dataset. That option prevents for example a column
     * series from calculating a grouped point partially. The effect is similar
     * to Series.getExtremesFromAll but does not affect yAxis extremes.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.groupAll
     */
    groupAll?: boolean;
    /**
     * (Highstock) The approximate pixel width of each group. If for example a
     * series with 30 points is displayed over a 600 pixel wide plot area, no
     * grouping is performed. If however the series contains so many points that
     * the spacing is less than the groupPixelWidth, Highcharts will try to
     * group it into appropriate groups so that each is more or less two pixels
     * wide. If multiple series with different group pixel widths are drawn on
     * the same x axis, all series will take the greatest width. For example,
     * line series have 2px default group width, while column series have 10px.
     * If combined, both the line and the column will have 10px by default.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.groupPixelWidth
     */
    groupPixelWidth?: number;
    /**
     * (Highstock) Normally, a group is indexed by the start of that group, so
     * for example when 30 daily values are grouped into one month, that month's
     * x value will be the 1st of the month. This apparently shifts the data to
     * the left. When the smoothed option is true, this is compensated for. The
     * data is shifted to the middle of the group, and min and max values are
     * preserved. Internally, this is used in the Navigator series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.smoothed
     */
    smoothed?: boolean;
    /**
     * (Highstock) An array determining what time intervals the data is allowed
     * to be grouped to. Each array item is an array where the first value is
     * the time unit and the second value another array of allowed multiples.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping.units
     */
    units?: Array<[string, (Array<number>|null)]>;
}
/**
 * (Highstock) A declarative filter for which data labels to display. The
 * declarative filter is designed for use when callback functions are not
 * available, like when the chart options require a pure JSON structure or for
 * use with graphical editors. For programmatic control, use the `formatter`
 * instead, and return `undefined` to disable a single data label.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.filter
 */
export interface PlotAbandsDataLabelsFilterOptions {
    /**
     * (Highstock) The operator to compare by. Can be one of `>`, `<`, `>=`,
     * `<=`, `==`, and `===`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.filter.operator
     */
    operator?: ("=="|"==="|">"|">="|"<"|"<=");
    /**
     * (Highstock) The point property to filter by. Point options are passed
     * directly to properties, additionally there are `y` value, `percentage`
     * and others listed under Point members.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.filter.property
     */
    property?: string;
    /**
     * (Highstock) The value to compare against.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.filter.value
     */
    value?: any;
}
/**
 * (Highstock) Options for the series data labels, appearing next to each data
 * point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels
 */
export interface PlotAbandsDataLabelsOptions {
    /**
     * (Highstock) The alignment of the data label compared to the point. If
     * `right`, the right side of the label should be touching the point. For
     * points with an extent, like columns, the alignments also dictates how to
     * align it inside the box, as given with the inside option. Can be one of
     * `left`, `center` or `right`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.align
     */
    align?: AlignType;
    /**
     * (Highstock) Whether to allow data labels to overlap. To make the labels
     * less sensitive for overlapping, the dataLabels.padding can be set to 0.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the data label. Defaults to `undefined`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for the data label. Particularly in styled mode,
     * this can be used to give each series' or point's data label unique
     * styling. In addition to this option, a default color class name is added
     * so that we can give the labels a contrast text shadow.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.className
     */
    className?: string;
    /**
     * (Highstock) The text color for the data labels. Defaults to `undefined`.
     * For certain series types, like column or map, the data labels can be
     * drawn inside the points. In this case the data label will be drawn with
     * maximum contrast by default. Additionally, it will be given a
     * `text-outline` style with the opposite color, to further increase the
     * contrast. This can be overridden by setting the `text-outline` style to
     * `none` in the `dataLabels.style` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.color
     */
    color?: ColorString;
    /**
     * (Highstock) Whether to hide data labels that are outside the plot area.
     * By default, the data label is moved inside the plot area according to the
     * overflow option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.crop
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.dataLabels.defer
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.defer
     * @see https://api.highcharts.com/gantt/plotOptions.abands.dataLabels.defer
     */
    defer?: boolean;
    /**
     * (Highstock) Enable or disable the data labels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) A declarative filter for which data labels to display. The
     * declarative filter is designed for use when callback functions are not
     * available, like when the chart options require a pure JSON structure or
     * for use with graphical editors. For programmatic control, use the
     * `formatter` instead, and return `undefined` to disable a single data
     * label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.filter
     */
    filter?: PlotAbandsDataLabelsFilterOptions;
    /**
     * (Highstock) A format string for the data label. Available variables are
     * the same as for `formatter`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.format
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the data label. Note
     * that if a `format` is defined, the format takes precedence and the
     * formatter is ignored. Available data are:
     *
     * - `this.percentage`: Stacked series and pies only. The point's percentage
     * of the total.
     *
     * - `this.point`: The point object. The point name, if defined, is
     * available through `this.point.name`.
     *
     * - `this.series`: The series object. The series name is available
     * through`this.series.name`.
     *
     * - `this.total`: Stacked series only. The total value at this point's x
     * value.
     *
     * - `this.x`: The x value.
     *
     * - `this.y`: The y value.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.formatter
     */
    formatter?: FormatterCallbackFunction<SeriesDataLabelsFormatterContextObject>;
    /**
     * (Highstock) For points with an extent, like columns or map areas, whether
     * to align the data label inside the box or to the actual value point.
     * Defaults to `false` in most cases, `true` in stacked columns.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.inside
     */
    inside?: boolean;
    /**
     * (Highstock) How to handle data labels that flow outside the plot area.
     * The default is `"justify"`, which aligns them inside the plot area. For
     * columns and bars, this means it will be moved inside the bar. To display
     * data labels outside the plot area, set `crop` to `false` and `overflow`
     * to `"allow"`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highstock) When either the `borderWidth` or the `backgroundColor` is
     * set, this is the padding within the box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.padding
     */
    padding?: number;
    /**
     * (Highstock) Text rotation in degrees. Note that due to a more complex
     * structure, backgrounds, borders and padding will be lost on a rotated
     * data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.rotation
     */
    rotation?: number;
    /**
     * (Highstock) The shadow of the box. Works best with `borderWidth` or
     * `backgroundColor`. Since 2.3 the shadow can be an object configuration
     * containing `color`, `offsetX`, `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.shape
     */
    shape?: string;
    /**
     * (Highstock) Styles for the label. The default `color` setting is
     * `"contrast"`, which is a pseudo color that Highcharts picks up and
     * applies the maximum contrast to the underlying point item, for example
     * the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.style
     */
    style?: CSSObject;
    /**
     * (Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of a data label. Can be one of `top`,
     * `middle` or `bottom`. The default value depends on the data, for instance
     * in a column chart, the label is above positive values and below negative
     * values.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highstock) The x position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.x
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.y
     */
    y?: number;
    /**
     * (Highstock) The Z index of the data labels. The default Z index puts it
     * above the series. Use a Z index of 2 to display it behind the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Options for the drag handles.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle
 */
export interface PlotAbandsDragDropDragHandleOptions {
    /**
     * (Highstock) The class name of the drag handles. Defaults to
     * `highcharts-drag-handle`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.className
     */
    className?: string;
    /**
     * (Highstock) The fill color of the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The mouse cursor to use for the drag handles. By default this
     * is intelligently switching between `ew-resize` and `ns-resize` depending
     * on the direction the point is being dragged.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.cursor
     */
    cursor?: string;
    /**
     * (Highstock) The line color of the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The line width for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Function to define the SVG path to use for the drag handles.
     * Takes the point as argument. Should return an SVG path in array format.
     * The SVG path is automatically positioned on the point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.pathFormatter
     */
    pathFormatter?: () => void;
    /**
     * (Highstock) The z index for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Style options for the guide box default state.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default
 */
export interface PlotAbandsDragDropGuideBoxDefaultOptions {
    /**
     * (Highstock) CSS class name of the guide box in this state. Defaults to
     * `highcharts-drag-box-default`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default.className
     */
    className?: string;
    /**
     * (Highstock) Guide box fill color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Guide box cursor.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default.cursor
     */
    cursor?: string;
    /**
     * (Highstock) Color of the border around the guide box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) Width of the line around the guide box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Guide box zIndex.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Style options for the guide box. The guide box has one state by
 * default, the `default` state.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox
 */
export interface PlotAbandsDragDropGuideBoxOptions {
    /**
     * (Highstock) Style options for the guide box default state.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox.default
     */
    default?: PlotAbandsDragDropGuideBoxDefaultOptions;
}
/**
 * (Highstock) The draggable-points module allows points to be moved around or
 * modified in the chart. In addition to the options mentioned under the
 * `dragDrop` API structure, the module fires three events, point.dragStart,
 * point.drag and point.drop.
 *
 * It requires the `modules/draggable-points.js` file to be loaded.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop
 */
export interface PlotAbandsDragDropOptions {
    /**
     * (Highstock) Enable dragging in the X dimension.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.draggableX
     */
    draggableX?: boolean;
    /**
     * (Highstock) Enable dragging in the Y dimension. Note that this is not
     * supported for TreeGrid axes (the default axis type in Gantt charts).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.draggableY
     */
    draggableY?: boolean;
    /**
     * (Highstock) Options for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragHandle
     */
    dragHandle?: PlotAbandsDragDropDragHandleOptions;
    /**
     * (Highstock) Set the maximum X value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragMaxX
     */
    dragMaxX?: number;
    /**
     * (Highstock) Set the maximum Y value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragMaxY
     */
    dragMaxY?: number;
    /**
     * (Highstock) Set the minimum X value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragMinX
     */
    dragMinX?: number;
    /**
     * (Highstock) Set the minimum Y value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragMinY
     */
    dragMinY?: number;
    /**
     * (Highstock) The X precision value to drag to for this series. Set to 0 to
     * disable. By default this is disabled, except for category axes, where the
     * default is 1.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragPrecisionX
     */
    dragPrecisionX?: number;
    /**
     * (Highstock) The Y precision value to drag to for this series. Set to 0 to
     * disable. By default this is disabled, except for category axes, where the
     * default is 1.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragPrecisionY
     */
    dragPrecisionY?: number;
    /**
     * (Highstock) The amount of pixels to drag the pointer before it counts as
     * a drag operation. This prevents drag/drop to fire when just clicking or
     * selecting points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.dragSensitivity
     */
    dragSensitivity?: number;
    /**
     * (Highstock) Group the points by a property. Points with the same property
     * value will be grouped together when moving.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.groupBy
     */
    groupBy?: string;
    /**
     * (Highstock) Style options for the guide box. The guide box has one state
     * by default, the `default` state.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.guideBox
     */
    guideBox?: (PlotAbandsDragDropGuideBoxOptions|Dictionary<PlotSeriesDragDropGuideBoxDefaultOptions>);
    /**
     * (Highstock) Update points as they are dragged. If false, a guide box is
     * drawn to illustrate the new point size.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop.liveRedraw
     */
    liveRedraw?: boolean;
}
/**
 * (Highstock) General event handlers for the series items. These event hooks
 * can also be attached to the series at run time using the
 * `Highcharts.addEvent` function.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.events
 */
export interface PlotAbandsEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Fires after the series has finished its
     * initial animation, or in case animation is disabled, immediately as the
     * series is displayed.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.events.afterAnimate
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.afterAnimate
     * @see https://api.highcharts.com/gantt/plotOptions.abands.events.afterAnimate
     */
    afterAnimate?: SeriesAfterAnimateCallbackFunction;
    /**
     * (Highstock) Fires when the checkbox next to the series' name in the
     * legend is clicked. One parameter, `event`, is passed to the function. The
     * state of the checkbox is found by `event.checked`. The checked item is
     * found by `event.item`. Return `false` to prevent the default action which
     * is to toggle the select state of the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.checkboxClick
     */
    checkboxClick?: SeriesCheckboxClickCallbackFunction;
    /**
     * (Highstock) Fires when the series is clicked. One parameter, `event`, is
     * passed to the function, containing common event information.
     * Additionally, `event.point` holds a pointer to the nearest point on the
     * graph.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.click
     */
    click?: SeriesClickCallbackFunction;
    /**
     * (Highstock) Fires when the series is hidden after chart generation time,
     * either by clicking the legend item or by calling `.hide()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.hide
     */
    hide?: SeriesHideCallbackFunction;
    /**
     * (Highstock) Fires when the legend item belonging to the series is
     * clicked. One parameter, `event`, is passed to the function. The default
     * action is to toggle the visibility of the series. This can be prevented
     * by returning `false` or calling `event.preventDefault()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.legendItemClick
     */
    legendItemClick?: SeriesLegendItemClickCallbackFunction;
    /**
     * (Highstock) Fires when the mouse leaves the graph. One parameter,
     * `event`, is passed to the function, containing common event information.
     * If the stickyTracking option is true, `mouseOut` doesn't happen before
     * the mouse enters another graph or leaves the plot area.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.mouseOut
     */
    mouseOut?: SeriesMouseOutCallbackFunction;
    /**
     * (Highstock) Fires when the mouse enters the graph. One parameter,
     * `event`, is passed to the function, containing common event information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.mouseOver
     */
    mouseOver?: SeriesMouseOverCallbackFunction;
    /**
     * (Highstock) Fires when the series is shown after chart generation time,
     * either by clicking the legend item or by calling `.show()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events.show
     */
    show?: SeriesShowCallbackFunction;
}
/**
 * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
 * series as possible in a natural way, seeking to avoid other series. The goal
 * of this feature is to make the chart more easily readable, like if a human
 * designer placed the labels in the optimal position.
 *
 * The series labels currently work with series types having a `graph` or an
 * `area`.
 *
 * Requires the `series-label.js` module.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.label
 * @see https://api.highcharts.com/highstock/plotOptions.abands.label
 * @see https://api.highcharts.com/gantt/plotOptions.abands.label
 */
export interface PlotAbandsLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) An array of boxes to avoid when laying out
     * the labels. Each item has a `left`, `right`, `top` and `bottom` property.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.boxesToAvoid
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.boxesToAvoid
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.boxesToAvoid
     */
    boxesToAvoid?: Array<LabelIntersectBoxObject>;
    /**
     * (Highcharts, Highstock, Gantt) Allow labels to be placed distant to the
     * graph if necessary, and draw a connector line to the graph. Setting this
     * option to true may decrease the performance significantly, since the
     * algorithm with systematically search for open spaces in the whole plot
     * area. Visually, it may also result in a more cluttered chart, though more
     * of the series will be labeled.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.connectorAllowed
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.connectorAllowed
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.connectorAllowed
     */
    connectorAllowed?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) If the label is closer than this to a
     * neighbour graph, draw a connector.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.connectorNeighbourDistance
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.connectorNeighbourDistance
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.connectorNeighbourDistance
     */
    connectorNeighbourDistance?: number;
    /**
     * (Highcharts, Highstock, Gantt) Enable the series label per series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.enabled
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) For area-like series, allow the font size
     * to vary so that small areas get a smaller font size. The default applies
     * this effect to area-like series but not line-like series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.maxFontSize
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.maxFontSize
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.maxFontSize
     */
    maxFontSize?: (number|null);
    /**
     * (Highcharts, Highstock, Gantt) For area-like series, allow the font size
     * to vary so that small areas get a smaller font size. The default applies
     * this effect to area-like series but not line-like series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.minFontSize
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.minFontSize
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.minFontSize
     */
    minFontSize?: (number|null);
    /**
     * (Highcharts, Highstock, Gantt) Draw the label on the area of an area
     * series. By default it is drawn on the area. Set it to `false` to draw it
     * next to the graph instead.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.onArea
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.onArea
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.onArea
     */
    onArea?: (boolean|null);
    /**
     * (Highcharts, Highstock, Gantt) Styles for the series label. The color
     * defaults to the series color, or a contrast color if `onArea`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label.style
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label.style
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label.style
     */
    style?: CSSObject;
}
/**
 * (Highstock) The line marks the last price from all points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.lastPrice
 */
export interface PlotAbandsLastPriceOptions {
    /**
     * (Highstock) The color of the line of last price.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lastPrice.color
     */
    color?: string;
    /**
     * (Highstock) Enable or disable the indicator.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lastPrice.enabled
     */
    enabled?: boolean;
}
export interface PlotAbandsLastVisiblePriceLabelOptions {
    /**
     * (Highstock) Enable or disable the label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lastVisiblePrice.label.enabled
     */
    enabled?: boolean;
}
/**
 * (Highstock) The line marks the last price from visible range of points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.lastVisiblePrice
 */
export interface PlotAbandsLastVisiblePriceOptions {
    /**
     * (Highstock) Enable or disable the indicator.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lastVisiblePrice.enabled
     */
    enabled?: boolean;
    label?: PlotAbandsLastVisiblePriceLabelOptions;
}
/**
 * (Highstock) Options for the point markers of line-like series. Properties
 * like `fillColor`, `lineColor` and `lineWidth` define the visual appearance of
 * the markers. Other series types, like column series, don't have markers, but
 * have visual options on the series level instead.
 *
 * In styled mode, the markers can be styled with the `.highcharts-point`,
 * `.highcharts-point-hover` and `.highcharts-point-select` class names.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.marker
 */
export interface PlotAbandsMarkerOptions {
    /**
     * (Highstock) Enable or disable the point marker. If `undefined`, the
     * markers are hidden when the data is dense, and shown for more widespread
     * data points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The threshold for how dense the point markers should be
     * before they are hidden, given that `enabled` is not defined. The number
     * indicates the horizontal distance between the two closest points in the
     * series, as multiples of the `marker.radius`. In other words, the default
     * value of 2 means points are hidden if overlapping horizontally.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.enabledThreshold
     */
    enabledThreshold?: number;
    /**
     * (Highstock) The fill color of the point marker. When `undefined`, the
     * series' or point's color is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Image markers only. Set the image width explicitly. When
     * using this option, a `width` must also be set.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.height
     */
    height?: number;
    /**
     * (Highstock) The color of the point marker's outline. When `undefined`,
     * the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The radius of the point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.radius
     */
    radius?: number;
    /**
     * (Highstock) States for a single point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states
     */
    states?: PlotAbandsMarkerStatesOptions;
    /**
     * (Highstock) A predefined shape or symbol for the marker. When undefined,
     * the symbol is pulled from options.symbols. Other possible values are
     * "circle", "square", "diamond", "triangle" and "triangle-down".
     *
     * Additionally, the URL to a graphic can be given on this form:
     * "url(graphic.png)". Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.symbol
     */
    symbol?: string;
    /**
     * (Highstock) Image markers only. Set the image width explicitly. When
     * using this option, a `height` must also be set.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.width
     */
    width?: number;
}
/**
 * (Highstock) Animation when hovering over the marker.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.animation
 */
export interface PlotAbandsMarkerStatesHoverAnimationOptions {
    duration?: number;
}
/**
 * (Highstock) The hover state for a single point marker.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover
 */
export interface PlotAbandsMarkerStatesHoverOptions {
    /**
     * (Highstock) Animation when hovering over the marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAbandsMarkerStatesHoverAnimationOptions);
    /**
     * (Highstock) Enable or disable the point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The fill color of the marker in hover state. When
     * `undefined`, the series' or point's fillColor for normal state is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the point marker's outline. When `undefined`,
     * the series' or point's lineColor for normal state is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the point marker's outline. When `undefined`,
     * the series' or point's lineWidth for normal state is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The additional line width for a hovered point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highstock) The radius of the point marker. In hover state, it defaults
     * to the normal state's radius + 2 as per the radiusPlus option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.radius
     */
    radius?: number;
    /**
     * (Highstock) The number of pixels to increase the radius of the hovered
     * point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover.radiusPlus
     */
    radiusPlus?: number;
}
/**
 * (Highstock) The normal state of a single point marker. Currently only used
 * for setting animation when returning to normal state from hover.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.normal
 */
export interface PlotAbandsMarkerStatesNormalOptions {
    /**
     * (Highstock) Animation when returning to normal state after hovering.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highstock) States for a single point marker.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states
 */
export interface PlotAbandsMarkerStatesOptions {
    /**
     * (Highstock) The hover state for a single point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.hover
     */
    hover?: PlotAbandsMarkerStatesHoverOptions;
    /**
     * (Highstock) The normal state of a single point marker. Currently only
     * used for setting animation when returning to normal state from hover.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.normal
     */
    normal?: PlotAbandsMarkerStatesNormalOptions;
    /**
     * (Highstock) The appearance of the point marker when selected. In order to
     * allow a point to be selected, set the `series.allowPointSelect` option to
     * true.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select
     */
    select?: PlotAbandsMarkerStatesSelectOptions;
}
/**
 * (Highstock) The appearance of the point marker when selected. In order to
 * allow a point to be selected, set the `series.allowPointSelect` option to
 * true.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select
 */
export interface PlotAbandsMarkerStatesSelectOptions {
    /**
     * (Highstock) Enable or disable visible feedback for selection.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The fill color of the point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the point marker's outline. When `undefined`,
     * the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The radius of the point marker. In hover state, it defaults
     * to the normal state's radius + 2.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker.states.select.radius
     */
    radius?: number;
}
/**
 * (Highstock) Acceleration bands (ABANDS). This series requires the `linkedTo`
 * option to be set and should be loaded after the
 * `stock/indicators/indicators.js`.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `abands` series are defined in plotOptions.abands.
 *
 * 3. Options for one single series are given in the series instance array.(see
 * online documentation for example)
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands
 */
export interface PlotAbandsOptions {
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.allowPointSelect
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAbandsAnimationOptions);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.animationLimit
     */
    animationLimit?: number;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     *
     * Requires `modules/boost.js`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.boostThreshold
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.borderWidth
     */
    borderWidth?: number;
    bottomLine?: PlotAbandsBottomLineOptions;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.className
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.clip
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.colorIndex
     */
    colorIndex?: number;
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point (see online
     * documentation for example) the range. In other words, this flag
     * determines if first point within the visible range will have 0%
     * (`compareStart=true`) or should have been already calculated according to
     * the previous point (`compareStart=false`).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.compareStart
     */
    compareStart?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.connectEnds
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.connectNulls
     * @see https://api.highcharts.com/highstock/plotOptions.abands.connectNulls
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.abands.connectors
     */
    connectors?: PlotAbandsConnectorsOptions;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.cropThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.abands.cropThreshold
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.cursor
     */
    cursor?: (string|CursorType);
    /**
     * (Highstock) A name for the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataGrouping
     */
    dataGrouping?: PlotAbandsDataGroupingOptions;
    /**
     * (Highstock) Options for the series data labels, appearing next to each
     * data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dataLabels
     */
    dataLabels?: PlotAbandsDataLabelsOptions;
    /**
     * (Highstock) Requires the Accessibility module.
     *
     * A description of the series to add to the screen reader information about
     * the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.description
     */
    description?: string;
    /**
     * (Highstock) The draggable-points module allows points to be moved around
     * or modified in the chart. In addition to the options mentioned under the
     * `dragDrop` API structure, the module fires three events, point.dragStart,
     * point.drag and point.drop.
     *
     * It requires the `modules/draggable-points.js` file to be loaded.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.dragDrop
     */
    dragDrop?: PlotAbandsDragDropOptions;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.enableMouseTracking
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.events
     */
    events?: PlotAbandsEventsOptions;
    /**
     * (Highstock) By default, series are exposed to screen readers as regions.
     * By enabling this option, the series element itself will be exposed in the
     * same way as the data points. This is useful if the series is not used as
     * a grouping entity in the chart, but you still want to attach a
     * description to the series.
     *
     * Requires the Accessibility module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.exposeElementToA11y
     */
    exposeElementToA11y?: boolean;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.findNearestPointBy
     */
    findNearestPointBy?: ("x"|"xy");
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.gapSize
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `relative` (default), a gap size of 5 means that if
     * the distance between two points is greater than five times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `value`, the gap is based on absolute axis values,
     * which on a datetime axis is milliseconds. This also applies to the
     * navigator series that inherits gap options from the base series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.gapUnit
     */
    gapUnit?: ("relative"|"value");
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.getExtremesFromAll
     * @see https://api.highcharts.com/highstock/plotOptions.abands.getExtremesFromAll
     * @see https://api.highcharts.com/gantt/plotOptions.abands.getExtremesFromAll
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     *
     * Requires the `series-label.js` module.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.label
     * @see https://api.highcharts.com/highstock/plotOptions.abands.label
     * @see https://api.highcharts.com/gantt/plotOptions.abands.label
     */
    label?: PlotAbandsLabelOptions;
    /**
     * (Highstock) The line marks the last price from all points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lastPrice
     */
    lastPrice?: PlotAbandsLastPriceOptions;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lastVisiblePrice
     */
    lastVisiblePrice?: PlotAbandsLastVisiblePriceOptions;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.linecap
     * @see https://api.highcharts.com/highstock/plotOptions.abands.linecap
     */
    linecap?: ("butt"|"round"|"square");
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.linkedTo
     * @see https://api.highcharts.com/highstock/plotOptions.abands.linkedTo
     * @see https://api.highcharts.com/gantt/plotOptions.abands.linkedTo
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.marker
     */
    marker?: PlotAbandsMarkerOptions;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.name
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.negativeColor
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Paramters used in calculation of regression series' points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.params
     */
    params?: PlotAbandsParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point
     */
    point?: PlotAbandsPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.pointDescriptionFormatter
     */
    pointDescriptionFormatter?: () => void;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.selected
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.showCheckbox
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. The default value is `true` for standalone series, `false`
     * for linked series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.showInLegend
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.skipKeyboardNavigation
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.softThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.abands.softThreshold
     */
    softThreshold?: boolean;
    /**
     * (Highstock) A wrapper object for all the series options in specific
     * states.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states
     */
    states?: PlotAbandsStatesOptions;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.step
     * @see https://api.highcharts.com/highstock/plotOptions.abands.step
     */
    step?: ("center"|"left"|"right");
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.stickyTracking
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.threshold
     * @see https://api.highcharts.com/highstock/plotOptions.abands.threshold
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip
     */
    tooltip?: PlotAbandsTooltipOptions;
    topLine?: PlotAbandsTopLineOptions;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.turboThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.abands.turboThreshold
     * @see https://api.highcharts.com/gantt/plotOptions.abands.turboThreshold
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.visible
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.zIndex
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zoneAxis
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zoneAxis
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zones
     */
    zones?: Array<PlotAbandsZonesOptions>;
}
/**
 * (Highstock) Paramters used in calculation of regression series' points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.params
 */
export interface PlotAbandsParamsOptions {
    /**
     * (Highstock) The algorithms factor value used to calculate bands.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.params.factor
     */
    factor?: number;
    /**
     * (Highstock) The point index which indicator calculations will base. For
     * example using OHLC data, index=2 means the indicator will be calculated
     * using Low values.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.params.index
     */
    index?: number;
    /**
     * (Highstock) The base period for indicator calculations. This is the
     * number of data points which are taken into account for the indicator
     * calculations.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.params.period
     */
    period?: number;
}
/**
 * (Highstock) Events for each single point.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events
 */
export interface PlotAbandsPointEventsOptions {
    /**
     * (Highstock) Fires when a point is clicked. One parameter, `event`, is
     * passed to the function, containing common event information.
     *
     * If the `series.allowPointSelect` option is true, the default action for
     * the point's click event is to toggle the point's select state. Returning
     * `false` cancels this action.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.click
     */
    click?: SeriesPointClickCallbackFunction;
    /**
     * (Highstock) Callback that fires while dragging a point. The mouse event
     * is passed in as parameter. The original data can be accessed from
     * `e.origin`, and the new point values can be accessed from `e.newPoints`.
     * If there is only a single point being updated, it can be accessed from
     * `e.newPoint` for simplicity, and its ID can be accessed from
     * `e.newPointId`. The `this` context is the point being dragged. To stop
     * the default drag action, return false. See drag and drop options.
     *
     * Requires the `draggable-points` module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.drag
     */
    drag?: SeriesPointDragCallbackFunction;
    /**
     * (Highstock) Callback that fires when starting to drag a point. The mouse
     * event object is passed in as an argument. If a drag handle is used,
     * `e.updateProp` is set to the data property being dragged. The `this`
     * context is the point. See drag and drop options.
     *
     * Requires the `draggable-points` module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.dragStart
     */
    dragStart?: SeriesPointDragStartCallbackFunction;
    /**
     * (Highstock) Callback that fires when the point is dropped. The parameters
     * passed are the same as for drag. To stop the default drop action, return
     * false. See drag and drop options.
     *
     * Requires the `draggable-points` module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.drop
     */
    drop?: SeriesPointDropCallbackFunction;
    /**
     * (Highstock) Fires when the mouse leaves the area close to the point. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.mouseOut
     */
    mouseOut?: SeriesPointMouseOutCallbackFunction;
    /**
     * (Highstock) Fires when the mouse enters the area close to the point. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.mouseOver
     */
    mouseOver?: SeriesPointMouseOverCallbackFunction;
    /**
     * (Highstock) Fires when the point is removed using the `.remove()` method.
     * One parameter, `event`, is passed to the function. Returning `false`
     * cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.remove
     */
    remove?: SeriesPointRemoveCallbackFunction;
    /**
     * (Highstock) Fires when the point is selected either programmatically or
     * following a click on the point. One parameter, `event`, is passed to the
     * function. Returning `false` cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.select
     */
    select?: SeriesPointSelectCallbackFunction;
    /**
     * (Highstock) Fires when the point is unselected either programmatically or
     * following a click on the point. One parameter, `event`, is passed to the
     * function. Returning `false` cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.unselect
     */
    unselect?: SeriesPointUnselectCallbackFunction;
    /**
     * (Highstock) Fires when the point is updated programmatically through the
     * `.update()` method. One parameter, `event`, is passed to the function.
     * The new point options can be accessed through `event.options`. Returning
     * `false` cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events.update
     */
    update?: SeriesPointUpdateCallbackFunction;
}
/**
 * (Highstock) Properties for each single point.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.point
 */
export interface PlotAbandsPointOptions {
    /**
     * (Highstock) Events for each single point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.point.events
     */
    events?: PlotAbandsPointEventsOptions;
}
/**
 * (Highcharts) Animation setting for hovering the graph in line-type series.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.animation
 */
export interface PlotAbandsStatesHoverAnimationOptions {
    /**
     * (Highcharts) The duration of the hover animation in milliseconds. By
     * default the hover state animates quickly in, and slowly back to normal.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.animation.duration
     */
    duration?: number;
}
/**
 * (Highcharts, Highstock) Options for the halo appearing around the hovered
 * point in line-type series as well as outside the hovered slice in pie charts.
 * By default the halo is filled by the current point or series color with an
 * opacity of 0.25\. The halo can be disabled by setting the `halo` option to
 * `false`.
 *
 * In styled mode, the halo is styled with the `.highcharts-halo` class, with
 * colors inherited from `.highcharts-color-{n}`.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.halo
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.halo
 */
export interface PlotAbandsStatesHoverHaloOptions {
    /**
     * (Highcharts, Highstock) A collection of SVG attributes to override the
     * appearance of the halo, for example `fill`, `stroke` and `stroke-width`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.halo.attributes
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.halo.attributes
     */
    attributes?: SVGAttributes;
    /**
     * (Highcharts, Highstock) Opacity for the halo unless a specific fill is
     * overridden using the `attributes` setting. Note that Highcharts is only
     * able to apply opacity to colors of hex or rgb(a) formats.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.halo.opacity
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.halo.opacity
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) The pixel size of the halo. For point markers
     * this is the radius of the halo. For pie slices it is the width of the
     * halo outside the slice. For bubbles it defaults to 5 and is the width of
     * the halo outside the bubble.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.halo.size
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.halo.size
     */
    size?: number;
}
/**
 * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
 * belonging to the hovered series. For settings on the hover state of the
 * individual point, see marker.states.hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker
 */
export interface PlotAbandsStatesHoverMarkerOptions {
    /**
     * (Highcharts, Highstock) Enable or disable the point marker. If
     * `undefined`, the markers are hidden when the data is dense, and shown for
     * more widespread data points.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The threshold for how dense the point markers
     * should be before they are hidden, given that `enabled` is not defined.
     * The number indicates the horizontal distance between the two closest
     * points in the series, as multiples of the `marker.radius`. In other
     * words, the default value of 2 means points are hidden if overlapping
     * horizontally.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.enabledThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.enabledThreshold
     */
    enabledThreshold?: number;
    /**
     * (Highcharts, Highstock) The fill color of the point marker. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `width` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.height
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.height
     */
    height?: number;
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.radius
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) States for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states
     */
    states?: PlotAbandsStatesHoverMarkerStatesOptions;
    /**
     * (Highcharts, Highstock) A predefined shape or symbol for the marker. When
     * undefined, the symbol is pulled from options.symbols. Other possible
     * values are "circle", "square", "diamond", "triangle" and "triangle-down".
     *
     * Additionally, the URL to a graphic can be given on this form:
     * "url(graphic.png)". Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.symbol
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.symbol
     */
    symbol?: string;
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `height` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.width
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock) Animation when hovering over the marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.animation
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.animation
 */
export interface PlotAbandsStatesHoverMarkerStatesHoverAnimationOptions {
    duration?: number;
}
/**
 * (Highcharts, Highstock) The hover state for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover
 */
export interface PlotAbandsStatesHoverMarkerStatesHoverOptions {
    /**
     * (Highcharts, Highstock) Animation when hovering over the marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.animation
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAbandsStatesHoverMarkerStatesHoverAnimationOptions);
    /**
     * (Highcharts, Highstock) Enable or disable the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the marker in hover state. When
     * `undefined`, the series' or point's fillColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's lineColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline. When
     * `undefined`, the series' or point's lineWidth for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for a hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2 as per the radiusPlus
     * option.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.radius
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) The number of pixels to increase the radius of
     * the hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover.radiusPlus
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover.radiusPlus
     */
    radiusPlus?: number;
}
/**
 * (Highcharts, Highstock) The normal state of a single point marker. Currently
 * only used for setting animation when returning to normal state from hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.normal
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.normal
 */
export interface PlotAbandsStatesHoverMarkerStatesNormalOptions {
    /**
     * (Highcharts, Highstock) Animation when returning to normal state after
     * hovering.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.normal.animation
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highcharts, Highstock) States for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states
 */
export interface PlotAbandsStatesHoverMarkerStatesOptions {
    /**
     * (Highcharts, Highstock) The hover state for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.hover
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.hover
     */
    hover?: PlotAbandsStatesHoverMarkerStatesHoverOptions;
    /**
     * (Highcharts, Highstock) The normal state of a single point marker.
     * Currently only used for setting animation when returning to normal state
     * from hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.normal
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.normal
     */
    normal?: PlotAbandsStatesHoverMarkerStatesNormalOptions;
    /**
     * (Highcharts, Highstock) The appearance of the point marker when selected.
     * In order to allow a point to be selected, set the
     * `series.allowPointSelect` option to true.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select
     */
    select?: PlotAbandsStatesHoverMarkerStatesSelectOptions;
}
/**
 * (Highcharts, Highstock) The appearance of the point marker when selected. In
 * order to allow a point to be selected, set the `series.allowPointSelect`
 * option to true.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select
 */
export interface PlotAbandsStatesHoverMarkerStatesSelectOptions {
    /**
     * (Highcharts, Highstock) Enable or disable visible feedback for selection.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker.states.select.radius
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker.states.select.radius
     */
    radius?: number;
}
/**
 * (Highstock) Options for the hovered series. These settings override the
 * normal state options when a series is moused over or touched.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover
 */
export interface PlotAbandsStatesHoverOptions {
    /**
     * (Highcharts) Animation setting for hovering the graph in line-type
     * series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAbandsStatesHoverAnimationOptions);
    /**
     * (Highmaps) The border color of the point in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.hover.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highmaps) The border width of the point in this state
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.hover.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highmaps) The relative brightness of the point when hovered, relative to
     * the normal point color.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.hover.brightness
     */
    brightness?: number;
    /**
     * (Highmaps) The color of the shape in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.hover.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Enable separate styles for the hovered series to visualize
     * that the user hovers either the series itself or the legend.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Options for the halo appearing around the hovered
     * point in line-type series as well as outside the hovered slice in pie
     * charts. By default the halo is filled by the current point or series
     * color with an opacity of 0.25\. The halo can be disabled by setting the
     * `halo` option to `false`.
     *
     * In styled mode, the halo is styled with the `.highcharts-halo` class,
     * with colors inherited from `.highcharts-color-{n}`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.halo
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.halo
     */
    halo?: PlotAbandsStatesHoverHaloOptions;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line. By default this
     * property is undefined, and the `lineWidthPlus` property dictates how much
     * to increase the linewidth from normal state.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for the graph of a
     * hovered series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
     * belonging to the hovered series. For settings on the hover state of the
     * individual point, see marker.states.hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.hover.marker
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover.marker
     */
    marker?: PlotAbandsStatesHoverMarkerOptions;
}
/**
 * (Highmaps) Overrides for the normal state.
 *
 * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.normal
 */
export interface PlotAbandsStatesNormalOptions {
    /**
     * (Highmaps) Animation options for the fill color when returning from hover
     * state to normal state. The animation adds some latency in order to reduce
     * the effect of flickering when hovering in and out of for example an
     * uneven coastline.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highstock) A wrapper object for all the series options in specific states.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states
 */
export interface PlotAbandsStatesOptions {
    /**
     * (Highstock) Options for the hovered series. These settings override the
     * normal state options when a series is moused over or touched.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.hover
     */
    hover?: PlotAbandsStatesHoverOptions;
    /**
     * (Highmaps) Overrides for the normal state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.normal
     */
    normal?: PlotAbandsStatesNormalOptions;
    /**
     * (Highmaps) Specific options for point in selected states, after being
     * selected by allowPointSelect or programmatically.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.select
     */
    select?: PlotAbandsStatesSelectOptions;
}
/**
 * (Highcharts) Animation setting for hovering the graph in line-type series.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.animation
 */
export interface PlotAbandsStatesSelectAnimationOptions {
    /**
     * (Highcharts) The duration of the hover animation in milliseconds. By
     * default the hover state animates quickly in, and slowly back to normal.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.animation.duration
     */
    duration?: number;
}
/**
 * (Highcharts, Highstock) Options for the halo appearing around the hovered
 * point in line-type series as well as outside the hovered slice in pie charts.
 * By default the halo is filled by the current point or series color with an
 * opacity of 0.25\. The halo can be disabled by setting the `halo` option to
 * `false`.
 *
 * In styled mode, the halo is styled with the `.highcharts-halo` class, with
 * colors inherited from `.highcharts-color-{n}`.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.halo
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.halo
 */
export interface PlotAbandsStatesSelectHaloOptions {
    /**
     * (Highcharts, Highstock) A collection of SVG attributes to override the
     * appearance of the halo, for example `fill`, `stroke` and `stroke-width`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.halo.attributes
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.halo.attributes
     */
    attributes?: SVGAttributes;
    /**
     * (Highcharts, Highstock) Opacity for the halo unless a specific fill is
     * overridden using the `attributes` setting. Note that Highcharts is only
     * able to apply opacity to colors of hex or rgb(a) formats.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.halo.opacity
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.halo.opacity
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) The pixel size of the halo. For point markers
     * this is the radius of the halo. For pie slices it is the width of the
     * halo outside the slice. For bubbles it defaults to 5 and is the width of
     * the halo outside the bubble.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.halo.size
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.halo.size
     */
    size?: number;
}
/**
 * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
 * belonging to the hovered series. For settings on the hover state of the
 * individual point, see marker.states.hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker
 */
export interface PlotAbandsStatesSelectMarkerOptions {
    /**
     * (Highcharts, Highstock) Enable or disable the point marker. If
     * `undefined`, the markers are hidden when the data is dense, and shown for
     * more widespread data points.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The threshold for how dense the point markers
     * should be before they are hidden, given that `enabled` is not defined.
     * The number indicates the horizontal distance between the two closest
     * points in the series, as multiples of the `marker.radius`. In other
     * words, the default value of 2 means points are hidden if overlapping
     * horizontally.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.enabledThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.enabledThreshold
     */
    enabledThreshold?: number;
    /**
     * (Highcharts, Highstock) The fill color of the point marker. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `width` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.height
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.height
     */
    height?: number;
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.radius
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) States for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states
     */
    states?: PlotAbandsStatesSelectMarkerStatesOptions;
    /**
     * (Highcharts, Highstock) A predefined shape or symbol for the marker. When
     * undefined, the symbol is pulled from options.symbols. Other possible
     * values are "circle", "square", "diamond", "triangle" and "triangle-down".
     *
     * Additionally, the URL to a graphic can be given on this form:
     * "url(graphic.png)". Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.symbol
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.symbol
     */
    symbol?: string;
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `height` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.width
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock) Animation when hovering over the marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.animation
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.animation
 */
export interface PlotAbandsStatesSelectMarkerStatesHoverAnimationOptions {
    duration?: number;
}
/**
 * (Highcharts, Highstock) The hover state for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover
 */
export interface PlotAbandsStatesSelectMarkerStatesHoverOptions {
    /**
     * (Highcharts, Highstock) Animation when hovering over the marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.animation
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAbandsStatesSelectMarkerStatesHoverAnimationOptions);
    /**
     * (Highcharts, Highstock) Enable or disable the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the marker in hover state. When
     * `undefined`, the series' or point's fillColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's lineColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline. When
     * `undefined`, the series' or point's lineWidth for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for a hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2 as per the radiusPlus
     * option.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.radius
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) The number of pixels to increase the radius of
     * the hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover.radiusPlus
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover.radiusPlus
     */
    radiusPlus?: number;
}
/**
 * (Highcharts, Highstock) The normal state of a single point marker. Currently
 * only used for setting animation when returning to normal state from hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.normal
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.normal
 */
export interface PlotAbandsStatesSelectMarkerStatesNormalOptions {
    /**
     * (Highcharts, Highstock) Animation when returning to normal state after
     * hovering.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.normal.animation
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highcharts, Highstock) States for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states
 */
export interface PlotAbandsStatesSelectMarkerStatesOptions {
    /**
     * (Highcharts, Highstock) The hover state for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.hover
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.hover
     */
    hover?: PlotAbandsStatesSelectMarkerStatesHoverOptions;
    /**
     * (Highcharts, Highstock) The normal state of a single point marker.
     * Currently only used for setting animation when returning to normal state
     * from hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.normal
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.normal
     */
    normal?: PlotAbandsStatesSelectMarkerStatesNormalOptions;
    /**
     * (Highcharts, Highstock) The appearance of the point marker when selected.
     * In order to allow a point to be selected, set the
     * `series.allowPointSelect` option to true.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select
     */
    select?: PlotAbandsStatesSelectMarkerStatesSelectOptions;
}
/**
 * (Highcharts, Highstock) The appearance of the point marker when selected. In
 * order to allow a point to be selected, set the `series.allowPointSelect`
 * option to true.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select
 * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select
 */
export interface PlotAbandsStatesSelectMarkerStatesSelectOptions {
    /**
     * (Highcharts, Highstock) Enable or disable visible feedback for selection.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker.states.select.radius
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker.states.select.radius
     */
    radius?: number;
}
/**
 * (Highmaps) Specific options for point in selected states, after being
 * selected by allowPointSelect or programmatically.
 *
 * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.select
 */
export interface PlotAbandsStatesSelectOptions {
    /**
     * (Highcharts) Animation setting for hovering the graph in line-type
     * series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.animation
     */
    animation?: PlotAbandsStatesSelectAnimationOptions;
    /**
     * (Highmaps) The border color of the point in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.select.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highmaps) The border width of the point in this state
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.select.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highmaps) The color of the shape in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.select.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) Enable separate styles for the hovered series to visualize
     * that the user hovers either the series itself or the legend.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.abands.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Options for the halo appearing around the hovered
     * point in line-type series as well as outside the hovered slice in pie
     * charts. By default the halo is filled by the current point or series
     * color with an opacity of 0.25\. The halo can be disabled by setting the
     * `halo` option to `false`.
     *
     * In styled mode, the halo is styled with the `.highcharts-halo` class,
     * with colors inherited from `.highcharts-color-{n}`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.halo
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.halo
     */
    halo?: PlotAbandsStatesSelectHaloOptions;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line. By default this
     * property is undefined, and the `lineWidthPlus` property dictates how much
     * to increase the linewidth from normal state.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for the graph of a
     * hovered series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
     * belonging to the hovered series. For settings on the hover state of the
     * individual point, see marker.states.hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.states.select.marker
     * @see https://api.highcharts.com/highstock/plotOptions.abands.states.select.marker
     */
    marker?: PlotAbandsStatesSelectMarkerOptions;
}
/**
 * (Highcharts, Highstock, Gantt) For series on a datetime axes, the date format
 * in the tooltip's header will by default be guessed based on the closest data
 * points. This member gives the default string representations used for each
 * unit. For an overview of the replacement codes, see dateFormat.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.tooltip.dateTimeLabelFormats
 * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.dateTimeLabelFormats
 * @see https://api.highcharts.com/gantt/plotOptions.abands.tooltip.dateTimeLabelFormats
 */
export interface PlotAbandsTooltipDateTimeLabelFormatsOptions {
    day?: string;
    hour?: string;
    millisecond?: string;
    minute?: string;
    month?: string;
    second?: string;
    week?: string;
    year?: string;
}
/**
 * (Highstock) A configuration object for the tooltip rendering of each single
 * series. Properties are inherited from tooltip, but only the following
 * properties can be defined on a series level.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip
 */
export interface PlotAbandsTooltipOptions {
    /**
     * (Highstock) How many decimals to show for the `point.change` value when
     * the `series.compare` option is set. This is overridable in each series'
     * tooltip options object. The default is to preserve all decimals.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.changeDecimals
     */
    changeDecimals?: number;
    /**
     * (Highcharts, Highstock, Gantt) For series on a datetime axes, the date
     * format in the tooltip's header will by default be guessed based on the
     * closest data points. This member gives the default string representations
     * used for each unit. For an overview of the replacement codes, see
     * dateFormat.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.tooltip.dateTimeLabelFormats
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.dateTimeLabelFormats
     * @see https://api.highcharts.com/gantt/plotOptions.abands.tooltip.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: (PlotAbandsTooltipDateTimeLabelFormatsOptions|Dictionary<string>);
    /**
     * (Highstock) Whether the tooltip should follow the mouse as it moves
     * across columns, pie slices and other point types with an extent. By
     * default it behaves this way for scatter, bubble and pie series by
     * override in the `plotOptions` for those series types.
     *
     * For touch moves to behave the same way, followTouchMove must be `true`
     * also.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.followPointer
     */
    followPointer?: boolean;
    /**
     * (Highstock) Whether the tooltip should update as the finger moves on a
     * touch device. If this is `true` and chart.panning is
     * set,`followTouchMove` will take over one-finger touches, so the user
     * needs to use two fingers for zooming and panning.
     *
     * Note the difference to followPointer that only defines the _position_ of
     * the tooltip. If `followPointer` is false in for example a column series,
     * the tooltip will show above or below the column, but as `followTouchMove`
     * is true, the tooltip will jump from column to column as the user swipes
     * across the plot area.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.followTouchMove
     */
    followTouchMove?: boolean;
    /**
     * (Highstock) A string to append to the tooltip format.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.footerFormat
     */
    footerFormat?: string;
    /**
     * (Highstock) The HTML of the tooltip header line. Variables are enclosed
     * by curly brackets. Available variables are `point.key`, `series.name`,
     * `series.color` and other members from the `point` and `series` objects.
     * The `point.key` variable contains the category name, x value or datetime
     * string depending on the type of axis. For datetime axes, the `point.key`
     * date format can be set using `tooltip.xDateFormat`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.headerFormat
     */
    headerFormat?: string;
    /**
     * (Highstock) The name of a symbol to use for the border around the tooltip
     * header. Applies only when tooltip.split is enabled.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols` the same way as for
     * series.marker.symbol.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.headerShape
     */
    headerShape?: ("callout"|"square");
    /**
     * (Highstock) The number of milliseconds to wait until the tooltip is
     * hidden when mouse out from a point or chart.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.hideDelay
     */
    hideDelay?: number;
    /**
     * (Highstock) Whether to allow the tooltip to render outside the chart's
     * SVG element box. By default (`false`), the tooltip is rendered within the
     * chart's SVG element, which results in the tooltip being aligned inside
     * the chart area. For small charts, this may result in clipping or
     * overlapping. When `true`, a separate SVG element is created and overlaid
     * on the page, allowing the tooltip to be aligned inside the page itself.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.outside
     */
    outside?: boolean;
    /**
     * (Highstock) Padding inside the tooltip, in pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.padding
     */
    padding?: number;
    /**
     * (Highstock) The HTML of the point's line in the tooltip. Variables are
     * enclosed by curly brackets. Available variables are point.x, point.y,
     * series. name and series.color and other properties on the same form.
     * Furthermore, `point.y` can be extended by the `tooltip.valuePrefix` and
     * `tooltip.valueSuffix` variables. This can also be overridden for each
     * series, which makes it a good hook for displaying units.
     *
     * In styled mode, the dot is colored by a class name rather than the point
     * color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.pointFormat
     */
    pointFormat?: string;
    /**
     * (Highstock) A callback function for formatting the HTML output for a
     * single point in the tooltip. Like the `pointFormat` string, but with more
     * flexibility.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.pointFormatter
     */
    pointFormatter?: () => void;
    /**
     * (Highcharts, Highstock) Split the tooltip into one label per series, with
     * the header close to the axis. This is recommended over shared tooltips
     * for charts with multiple line series, generally making them easier to
     * read. This option takes precedence over `tooltip.shared`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.tooltip.split
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.split
     */
    split?: boolean;
    /**
     * (Highstock) Number of decimals in indicator series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.valueDecimals
     */
    valueDecimals?: number;
    /**
     * (Highstock) A string to prepend to each series' y value. Overridable in
     * each series' tooltip options object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.valuePrefix
     */
    valuePrefix?: string;
    /**
     * (Highstock) A string to append to each series' y value. Overridable in
     * each series' tooltip options object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.valueSuffix
     */
    valueSuffix?: string;
    /**
     * (Highcharts, Highstock, Gantt) The format for the date in the tooltip
     * header if the X axis is a datetime axis. The default is a best guess
     * based on the smallest distance between points in the chart.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.tooltip.xDateFormat
     * @see https://api.highcharts.com/highstock/plotOptions.abands.tooltip.xDateFormat
     * @see https://api.highcharts.com/gantt/plotOptions.abands.tooltip.xDateFormat
     */
    xDateFormat?: string;
}
export interface PlotAbandsTopLineOptions {
    styles?: PlotAbandsTopLineStylesOptions;
}
export interface PlotAbandsTopLineStylesOptions {
    /**
     * (Highstock) Pixel width of the line.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.abands.topLine.styles.lineWidth
     */
    lineWidth?: number;
}
/**
 * (Highcharts, Highstock) An array defining zones within a series. Zones can be
 * applied to the X axis, Y axis or Z axis for bubbles, according to the
 * `zoneAxis` option. The zone definitions have to be in ascending order
 * regarding to the value.
 *
 * In styled mode, the color zones are styled with the `.highcharts-zone-{n}`
 * class, or custom classed from the `className` option (view live demo).
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones
 * @see https://api.highcharts.com/highstock/plotOptions.abands.zones
 */
export interface PlotAbandsZonesOptions {
    /**
     * (Highcharts, Highstock) Styled mode only. A custom class name for the
     * zone.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones.className
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zones.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock) Defines the color of the series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones.color
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zones.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) A name for the dash style to use for the graph.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones.dashStyle
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zones.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highcharts, Highstock) Defines the fill color for the series (in area
     * type series)
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zones.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The value up to where the zone extends, if
     * undefined the zones stretches to the last value in the series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.abands.zones.value
     * @see https://api.highcharts.com/highstock/plotOptions.abands.zones.value
     */
    value?: number;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.animation
 */
export interface PlotAdAnimationOptions {
    duration?: number;
}
/**
 * (Gantt) Marker options specific to the end markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker
 */
export interface PlotAdConnectorsEndMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector end markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker.width
     */
    width?: number;
}
/**
 * (Gantt) Marker options for this chart's Pathfinder connectors. Note that this
 * option is overridden by the `startMarker` and `endMarker` options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker
 */
export interface PlotAdConnectorsMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.radius
     */
    radius?: number;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker.width
     */
    width?: number;
}
/**
 * (Gantt) Override Pathfinder connector options for a series. Requires
 * Highcharts Gantt to be loaded.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors
 */
export interface PlotAdConnectorsOptions {
    /**
     * (Gantt) Set the default dash style for this chart's connecting lines.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.dashStyle
     */
    dashStyle?: string;
    /**
     * (Gantt) Marker options specific to the end markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.endMarker
     */
    endMarker?: PlotAdConnectorsEndMarkerOptions;
    /**
     * (Gantt) Set the default color for this chart's Pathfinder connecting
     * lines. Defaults to the color of the point being connected.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the default pixel width for this chart's Pathfinder
     * connecting lines.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Marker options for this chart's Pathfinder connectors. Note that
     * this option is overridden by the `startMarker` and `endMarker` options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.marker
     */
    marker?: PlotAdConnectorsMarkerOptions;
    /**
     * (Gantt) Marker options specific to the start markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker
     */
    startMarker?: PlotAdConnectorsStartMarkerOptions;
    /**
     * (Gantt) Set the default pathfinder algorithm to use for this chart. It is
     * possible to define your own algorithms by adding them to the
     * Highcharts.Pathfinder.prototype.algorithms object before the chart has
     * been created.
     *
     * The default algorithms are as follows:
     *
     * `straight`: Draws a straight line between the connecting points. Does not
     * avoid other points when drawing.
     *
     * `simpleConnect`: Finds a path between the points using right angles only.
     * Takes only starting/ending points into account, and will not avoid other
     * points.
     *
     * `fastAvoid`: Finds a path between the points using right angles only.
     * Will attempt to avoid other points, but its focus is performance over
     * accuracy. Works well with less dense datasets.
     *
     * Default value: `straight` is used as default for most series types, while
     * `simpleConnect` is used as default for Gantt series, to show dependencies
     * between points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.type
     */
    type?: ("straight"|"fastAvoid"|"simpleConnect");
}
/**
 * (Gantt) Marker options specific to the start markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker
 */
export interface PlotAdConnectorsStartMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector start markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors.startMarker.width
     */
    width?: number;
}
/**
 * (Highstock) Data grouping is the concept of sampling the data values into
 * larger blocks in order to ease readability and increase performance of the
 * JavaScript charts. Highstock by default applies data grouping when the points
 * become closer than a certain pixel value, determined by the `groupPixelWidth`
 * option.
 *
 * If data grouping is applied, the grouping information of grouped points can
 * be read from the Point.dataGroup. If point options other than the data itself
 * are set, for example `name` or `color` or custom properties, the grouping
 * logic doesn't know how to group it. In this case the options of the first
 * point instance are copied over to the group point. This can be altered
 * through a custom `approximation` callback function.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping
 */
export interface PlotAdDataGroupingOptions {
    /**
     * (Highstock) The method of approximation inside a group. When for example
     * 30 days are grouped into one month, this determines what value should
     * represent the group. Possible values are "average", "averages", "open",
     * "high", "low", "close" and "sum". For OHLC and candlestick series the
     * approximation is "ohlc" by default, which finds the open, high, low and
     * close values within all the grouped data. For ranges, the approximation
     * is "range", which finds the low and high values. For multi-dimensional
     * data, like ranges and OHLC, "averages" will compute the average for each
     * dimension.
     *
     * Custom aggregate methods can be added by assigning a callback function as
     * the approximation. This function takes a numeric array as the argument
     * and should return a single numeric value or `null`. Note that the numeric
     * array will never contain null values, only true numbers. Instead, if null
     * values are present in the raw data, the numeric array will have an
     * `.hasNulls` property set to `true`. For single-value data sets the data
     * is available in the first argument of the callback function. For OHLC
     * data sets, all the open values are in the first argument, all high values
     * in the second etc.
     *
     * Since v4.2.7, grouping meta data is available in the approximation
     * callback from `this.dataGroupInfo`. It can be used to extract information
     * from the raw data.
     *
     * Defaults to `average` for line-type series, `sum` for columns, `range`
     * for range series and `ohlc` for OHLC and candlestick.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.approximation
     */
    approximation?: ("average"|"averages"|"close"|"high"|"low"|"open"|"sum");
    /**
     * (Highstock) Datetime formats for the header of the tooltip in a stock
     * chart. The format can vary within a chart depending on the currently
     * selected time range and the current data grouping.
     *
     * The default formats are:
     *
     * (see online documentation for example)
     *
     * For each of these array definitions, the first item is the format used
     * when the active time span is one unit. For instance, if the current data
     * applies to one week, the first item of the week array is used. The second
     * and third items are used when the active time span is more than two
     * units. For instance, if the current data applies to two weeks, the second
     * and third item of the week array are used, and applied to the start and
     * end date of the time span.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: object;
    /**
     * (Highstock) Enable or disable data grouping.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) When data grouping is forced, it runs no matter how small the
     * intervals are. This can be handy for example when the sum should be
     * calculated for values appearing at random times within each hour.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.forced
     */
    forced?: boolean;
    /**
     * (Highstock) By default only points within the visible range are grouped.
     * Enabling this option will force data grouping to calculate all grouped
     * points for a given dataset. That option prevents for example a column
     * series from calculating a grouped point partially. The effect is similar
     * to Series.getExtremesFromAll but does not affect yAxis extremes.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.groupAll
     */
    groupAll?: boolean;
    /**
     * (Highstock) The approximate pixel width of each group. If for example a
     * series with 30 points is displayed over a 600 pixel wide plot area, no
     * grouping is performed. If however the series contains so many points that
     * the spacing is less than the groupPixelWidth, Highcharts will try to
     * group it into appropriate groups so that each is more or less two pixels
     * wide. If multiple series with different group pixel widths are drawn on
     * the same x axis, all series will take the greatest width. For example,
     * line series have 2px default group width, while column series have 10px.
     * If combined, both the line and the column will have 10px by default.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.groupPixelWidth
     */
    groupPixelWidth?: number;
    /**
     * (Highstock) Normally, a group is indexed by the start of that group, so
     * for example when 30 daily values are grouped into one month, that month's
     * x value will be the 1st of the month. This apparently shifts the data to
     * the left. When the smoothed option is true, this is compensated for. The
     * data is shifted to the middle of the group, and min and max values are
     * preserved. Internally, this is used in the Navigator series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.smoothed
     */
    smoothed?: boolean;
    /**
     * (Highstock) An array determining what time intervals the data is allowed
     * to be grouped to. Each array item is an array where the first value is
     * the time unit and the second value another array of allowed multiples.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping.units
     */
    units?: Array<[string, (Array<number>|null)]>;
}
/**
 * (Highstock) A declarative filter for which data labels to display. The
 * declarative filter is designed for use when callback functions are not
 * available, like when the chart options require a pure JSON structure or for
 * use with graphical editors. For programmatic control, use the `formatter`
 * instead, and return `undefined` to disable a single data label.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.filter
 */
export interface PlotAdDataLabelsFilterOptions {
    /**
     * (Highstock) The operator to compare by. Can be one of `>`, `<`, `>=`,
     * `<=`, `==`, and `===`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.filter.operator
     */
    operator?: ("=="|"==="|">"|">="|"<"|"<=");
    /**
     * (Highstock) The point property to filter by. Point options are passed
     * directly to properties, additionally there are `y` value, `percentage`
     * and others listed under Point members.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.filter.property
     */
    property?: string;
    /**
     * (Highstock) The value to compare against.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.filter.value
     */
    value?: any;
}
/**
 * (Highstock) Options for the series data labels, appearing next to each data
 * point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels
 */
export interface PlotAdDataLabelsOptions {
    /**
     * (Highstock) The alignment of the data label compared to the point. If
     * `right`, the right side of the label should be touching the point. For
     * points with an extent, like columns, the alignments also dictates how to
     * align it inside the box, as given with the inside option. Can be one of
     * `left`, `center` or `right`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.align
     */
    align?: AlignType;
    /**
     * (Highstock) Whether to allow data labels to overlap. To make the labels
     * less sensitive for overlapping, the dataLabels.padding can be set to 0.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the data label. Defaults to `undefined`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for the data label. Particularly in styled mode,
     * this can be used to give each series' or point's data label unique
     * styling. In addition to this option, a default color class name is added
     * so that we can give the labels a contrast text shadow.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.className
     */
    className?: string;
    /**
     * (Highstock) The text color for the data labels. Defaults to `undefined`.
     * For certain series types, like column or map, the data labels can be
     * drawn inside the points. In this case the data label will be drawn with
     * maximum contrast by default. Additionally, it will be given a
     * `text-outline` style with the opposite color, to further increase the
     * contrast. This can be overridden by setting the `text-outline` style to
     * `none` in the `dataLabels.style` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.color
     */
    color?: ColorString;
    /**
     * (Highstock) Whether to hide data labels that are outside the plot area.
     * By default, the data label is moved inside the plot area according to the
     * overflow option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.crop
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.dataLabels.defer
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.defer
     * @see https://api.highcharts.com/gantt/plotOptions.ad.dataLabels.defer
     */
    defer?: boolean;
    /**
     * (Highstock) Enable or disable the data labels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) A declarative filter for which data labels to display. The
     * declarative filter is designed for use when callback functions are not
     * available, like when the chart options require a pure JSON structure or
     * for use with graphical editors. For programmatic control, use the
     * `formatter` instead, and return `undefined` to disable a single data
     * label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.filter
     */
    filter?: PlotAdDataLabelsFilterOptions;
    /**
     * (Highstock) A format string for the data label. Available variables are
     * the same as for `formatter`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.format
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the data label. Note
     * that if a `format` is defined, the format takes precedence and the
     * formatter is ignored. Available data are:
     *
     * - `this.percentage`: Stacked series and pies only. The point's percentage
     * of the total.
     *
     * - `this.point`: The point object. The point name, if defined, is
     * available through `this.point.name`.
     *
     * - `this.series`: The series object. The series name is available
     * through`this.series.name`.
     *
     * - `this.total`: Stacked series only. The total value at this point's x
     * value.
     *
     * - `this.x`: The x value.
     *
     * - `this.y`: The y value.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.formatter
     */
    formatter?: FormatterCallbackFunction<SeriesDataLabelsFormatterContextObject>;
    /**
     * (Highstock) For points with an extent, like columns or map areas, whether
     * to align the data label inside the box or to the actual value point.
     * Defaults to `false` in most cases, `true` in stacked columns.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.inside
     */
    inside?: boolean;
    /**
     * (Highstock) How to handle data labels that flow outside the plot area.
     * The default is `"justify"`, which aligns them inside the plot area. For
     * columns and bars, this means it will be moved inside the bar. To display
     * data labels outside the plot area, set `crop` to `false` and `overflow`
     * to `"allow"`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highstock) When either the `borderWidth` or the `backgroundColor` is
     * set, this is the padding within the box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.padding
     */
    padding?: number;
    /**
     * (Highstock) Text rotation in degrees. Note that due to a more complex
     * structure, backgrounds, borders and padding will be lost on a rotated
     * data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.rotation
     */
    rotation?: number;
    /**
     * (Highstock) The shadow of the box. Works best with `borderWidth` or
     * `backgroundColor`. Since 2.3 the shadow can be an object configuration
     * containing `color`, `offsetX`, `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.shape
     */
    shape?: string;
    /**
     * (Highstock) Styles for the label. The default `color` setting is
     * `"contrast"`, which is a pseudo color that Highcharts picks up and
     * applies the maximum contrast to the underlying point item, for example
     * the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.style
     */
    style?: CSSObject;
    /**
     * (Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of a data label. Can be one of `top`,
     * `middle` or `bottom`. The default value depends on the data, for instance
     * in a column chart, the label is above positive values and below negative
     * values.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highstock) The x position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.x
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.y
     */
    y?: number;
    /**
     * (Highstock) The Z index of the data labels. The default Z index puts it
     * above the series. Use a Z index of 2 to display it behind the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Options for the drag handles.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle
 */
export interface PlotAdDragDropDragHandleOptions {
    /**
     * (Highstock) The class name of the drag handles. Defaults to
     * `highcharts-drag-handle`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.className
     */
    className?: string;
    /**
     * (Highstock) The fill color of the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The mouse cursor to use for the drag handles. By default this
     * is intelligently switching between `ew-resize` and `ns-resize` depending
     * on the direction the point is being dragged.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.cursor
     */
    cursor?: string;
    /**
     * (Highstock) The line color of the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The line width for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Function to define the SVG path to use for the drag handles.
     * Takes the point as argument. Should return an SVG path in array format.
     * The SVG path is automatically positioned on the point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.pathFormatter
     */
    pathFormatter?: () => void;
    /**
     * (Highstock) The z index for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Style options for the guide box default state.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default
 */
export interface PlotAdDragDropGuideBoxDefaultOptions {
    /**
     * (Highstock) CSS class name of the guide box in this state. Defaults to
     * `highcharts-drag-box-default`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default.className
     */
    className?: string;
    /**
     * (Highstock) Guide box fill color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Guide box cursor.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default.cursor
     */
    cursor?: string;
    /**
     * (Highstock) Color of the border around the guide box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) Width of the line around the guide box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Guide box zIndex.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Style options for the guide box. The guide box has one state by
 * default, the `default` state.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox
 */
export interface PlotAdDragDropGuideBoxOptions {
    /**
     * (Highstock) Style options for the guide box default state.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox.default
     */
    default?: PlotAdDragDropGuideBoxDefaultOptions;
}
/**
 * (Highstock) The draggable-points module allows points to be moved around or
 * modified in the chart. In addition to the options mentioned under the
 * `dragDrop` API structure, the module fires three events, point.dragStart,
 * point.drag and point.drop.
 *
 * It requires the `modules/draggable-points.js` file to be loaded.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop
 */
export interface PlotAdDragDropOptions {
    /**
     * (Highstock) Enable dragging in the X dimension.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.draggableX
     */
    draggableX?: boolean;
    /**
     * (Highstock) Enable dragging in the Y dimension. Note that this is not
     * supported for TreeGrid axes (the default axis type in Gantt charts).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.draggableY
     */
    draggableY?: boolean;
    /**
     * (Highstock) Options for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragHandle
     */
    dragHandle?: PlotAdDragDropDragHandleOptions;
    /**
     * (Highstock) Set the maximum X value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragMaxX
     */
    dragMaxX?: number;
    /**
     * (Highstock) Set the maximum Y value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragMaxY
     */
    dragMaxY?: number;
    /**
     * (Highstock) Set the minimum X value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragMinX
     */
    dragMinX?: number;
    /**
     * (Highstock) Set the minimum Y value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragMinY
     */
    dragMinY?: number;
    /**
     * (Highstock) The X precision value to drag to for this series. Set to 0 to
     * disable. By default this is disabled, except for category axes, where the
     * default is 1.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragPrecisionX
     */
    dragPrecisionX?: number;
    /**
     * (Highstock) The Y precision value to drag to for this series. Set to 0 to
     * disable. By default this is disabled, except for category axes, where the
     * default is 1.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragPrecisionY
     */
    dragPrecisionY?: number;
    /**
     * (Highstock) The amount of pixels to drag the pointer before it counts as
     * a drag operation. This prevents drag/drop to fire when just clicking or
     * selecting points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.dragSensitivity
     */
    dragSensitivity?: number;
    /**
     * (Highstock) Group the points by a property. Points with the same property
     * value will be grouped together when moving.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.groupBy
     */
    groupBy?: string;
    /**
     * (Highstock) Style options for the guide box. The guide box has one state
     * by default, the `default` state.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.guideBox
     */
    guideBox?: (PlotAdDragDropGuideBoxOptions|Dictionary<PlotSeriesDragDropGuideBoxDefaultOptions>);
    /**
     * (Highstock) Update points as they are dragged. If false, a guide box is
     * drawn to illustrate the new point size.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop.liveRedraw
     */
    liveRedraw?: boolean;
}
/**
 * (Highstock) General event handlers for the series items. These event hooks
 * can also be attached to the series at run time using the
 * `Highcharts.addEvent` function.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.events
 */
export interface PlotAdEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Fires after the series has finished its
     * initial animation, or in case animation is disabled, immediately as the
     * series is displayed.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.events.afterAnimate
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.afterAnimate
     * @see https://api.highcharts.com/gantt/plotOptions.ad.events.afterAnimate
     */
    afterAnimate?: SeriesAfterAnimateCallbackFunction;
    /**
     * (Highstock) Fires when the checkbox next to the series' name in the
     * legend is clicked. One parameter, `event`, is passed to the function. The
     * state of the checkbox is found by `event.checked`. The checked item is
     * found by `event.item`. Return `false` to prevent the default action which
     * is to toggle the select state of the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.checkboxClick
     */
    checkboxClick?: SeriesCheckboxClickCallbackFunction;
    /**
     * (Highstock) Fires when the series is clicked. One parameter, `event`, is
     * passed to the function, containing common event information.
     * Additionally, `event.point` holds a pointer to the nearest point on the
     * graph.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.click
     */
    click?: SeriesClickCallbackFunction;
    /**
     * (Highstock) Fires when the series is hidden after chart generation time,
     * either by clicking the legend item or by calling `.hide()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.hide
     */
    hide?: SeriesHideCallbackFunction;
    /**
     * (Highstock) Fires when the legend item belonging to the series is
     * clicked. One parameter, `event`, is passed to the function. The default
     * action is to toggle the visibility of the series. This can be prevented
     * by returning `false` or calling `event.preventDefault()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.legendItemClick
     */
    legendItemClick?: SeriesLegendItemClickCallbackFunction;
    /**
     * (Highstock) Fires when the mouse leaves the graph. One parameter,
     * `event`, is passed to the function, containing common event information.
     * If the stickyTracking option is true, `mouseOut` doesn't happen before
     * the mouse enters another graph or leaves the plot area.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.mouseOut
     */
    mouseOut?: SeriesMouseOutCallbackFunction;
    /**
     * (Highstock) Fires when the mouse enters the graph. One parameter,
     * `event`, is passed to the function, containing common event information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.mouseOver
     */
    mouseOver?: SeriesMouseOverCallbackFunction;
    /**
     * (Highstock) Fires when the series is shown after chart generation time,
     * either by clicking the legend item or by calling `.show()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events.show
     */
    show?: SeriesShowCallbackFunction;
}
/**
 * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
 * series as possible in a natural way, seeking to avoid other series. The goal
 * of this feature is to make the chart more easily readable, like if a human
 * designer placed the labels in the optimal position.
 *
 * The series labels currently work with series types having a `graph` or an
 * `area`.
 *
 * Requires the `series-label.js` module.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.label
 * @see https://api.highcharts.com/highstock/plotOptions.ad.label
 * @see https://api.highcharts.com/gantt/plotOptions.ad.label
 */
export interface PlotAdLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) An array of boxes to avoid when laying out
     * the labels. Each item has a `left`, `right`, `top` and `bottom` property.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.boxesToAvoid
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.boxesToAvoid
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.boxesToAvoid
     */
    boxesToAvoid?: Array<LabelIntersectBoxObject>;
    /**
     * (Highcharts, Highstock, Gantt) Allow labels to be placed distant to the
     * graph if necessary, and draw a connector line to the graph. Setting this
     * option to true may decrease the performance significantly, since the
     * algorithm with systematically search for open spaces in the whole plot
     * area. Visually, it may also result in a more cluttered chart, though more
     * of the series will be labeled.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.connectorAllowed
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.connectorAllowed
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.connectorAllowed
     */
    connectorAllowed?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) If the label is closer than this to a
     * neighbour graph, draw a connector.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.connectorNeighbourDistance
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.connectorNeighbourDistance
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.connectorNeighbourDistance
     */
    connectorNeighbourDistance?: number;
    /**
     * (Highcharts, Highstock, Gantt) Enable the series label per series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.enabled
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) For area-like series, allow the font size
     * to vary so that small areas get a smaller font size. The default applies
     * this effect to area-like series but not line-like series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.maxFontSize
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.maxFontSize
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.maxFontSize
     */
    maxFontSize?: (number|null);
    /**
     * (Highcharts, Highstock, Gantt) For area-like series, allow the font size
     * to vary so that small areas get a smaller font size. The default applies
     * this effect to area-like series but not line-like series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.minFontSize
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.minFontSize
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.minFontSize
     */
    minFontSize?: (number|null);
    /**
     * (Highcharts, Highstock, Gantt) Draw the label on the area of an area
     * series. By default it is drawn on the area. Set it to `false` to draw it
     * next to the graph instead.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.onArea
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.onArea
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.onArea
     */
    onArea?: (boolean|null);
    /**
     * (Highcharts, Highstock, Gantt) Styles for the series label. The color
     * defaults to the series color, or a contrast color if `onArea`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label.style
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label.style
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label.style
     */
    style?: CSSObject;
}
/**
 * (Highstock) The line marks the last price from all points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.lastPrice
 */
export interface PlotAdLastPriceOptions {
    /**
     * (Highstock) The color of the line of last price.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lastPrice.color
     */
    color?: string;
    /**
     * (Highstock) Enable or disable the indicator.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lastPrice.enabled
     */
    enabled?: boolean;
}
export interface PlotAdLastVisiblePriceLabelOptions {
    /**
     * (Highstock) Enable or disable the label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lastVisiblePrice.label.enabled
     */
    enabled?: boolean;
}
/**
 * (Highstock) The line marks the last price from visible range of points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.lastVisiblePrice
 */
export interface PlotAdLastVisiblePriceOptions {
    /**
     * (Highstock) Enable or disable the indicator.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lastVisiblePrice.enabled
     */
    enabled?: boolean;
    label?: PlotAdLastVisiblePriceLabelOptions;
}
/**
 * (Highstock) Options for the point markers of line-like series. Properties
 * like `fillColor`, `lineColor` and `lineWidth` define the visual appearance of
 * the markers. Other series types, like column series, don't have markers, but
 * have visual options on the series level instead.
 *
 * In styled mode, the markers can be styled with the `.highcharts-point`,
 * `.highcharts-point-hover` and `.highcharts-point-select` class names.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.marker
 */
export interface PlotAdMarkerOptions {
    /**
     * (Highstock) Enable or disable the point marker. If `undefined`, the
     * markers are hidden when the data is dense, and shown for more widespread
     * data points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The threshold for how dense the point markers should be
     * before they are hidden, given that `enabled` is not defined. The number
     * indicates the horizontal distance between the two closest points in the
     * series, as multiples of the `marker.radius`. In other words, the default
     * value of 2 means points are hidden if overlapping horizontally.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.enabledThreshold
     */
    enabledThreshold?: number;
    /**
     * (Highstock) The fill color of the point marker. When `undefined`, the
     * series' or point's color is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Image markers only. Set the image width explicitly. When
     * using this option, a `width` must also be set.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.height
     */
    height?: number;
    /**
     * (Highstock) The color of the point marker's outline. When `undefined`,
     * the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The radius of the point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.radius
     */
    radius?: number;
    /**
     * (Highstock) States for a single point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states
     */
    states?: PlotAdMarkerStatesOptions;
    /**
     * (Highstock) A predefined shape or symbol for the marker. When undefined,
     * the symbol is pulled from options.symbols. Other possible values are
     * "circle", "square", "diamond", "triangle" and "triangle-down".
     *
     * Additionally, the URL to a graphic can be given on this form:
     * "url(graphic.png)". Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.symbol
     */
    symbol?: string;
    /**
     * (Highstock) Image markers only. Set the image width explicitly. When
     * using this option, a `height` must also be set.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.width
     */
    width?: number;
}
/**
 * (Highstock) Animation when hovering over the marker.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.animation
 */
export interface PlotAdMarkerStatesHoverAnimationOptions {
    duration?: number;
}
/**
 * (Highstock) The hover state for a single point marker.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover
 */
export interface PlotAdMarkerStatesHoverOptions {
    /**
     * (Highstock) Animation when hovering over the marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAdMarkerStatesHoverAnimationOptions);
    /**
     * (Highstock) Enable or disable the point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The fill color of the marker in hover state. When
     * `undefined`, the series' or point's fillColor for normal state is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the point marker's outline. When `undefined`,
     * the series' or point's lineColor for normal state is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the point marker's outline. When `undefined`,
     * the series' or point's lineWidth for normal state is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The additional line width for a hovered point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highstock) The radius of the point marker. In hover state, it defaults
     * to the normal state's radius + 2 as per the radiusPlus option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.radius
     */
    radius?: number;
    /**
     * (Highstock) The number of pixels to increase the radius of the hovered
     * point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover.radiusPlus
     */
    radiusPlus?: number;
}
/**
 * (Highstock) The normal state of a single point marker. Currently only used
 * for setting animation when returning to normal state from hover.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.normal
 */
export interface PlotAdMarkerStatesNormalOptions {
    /**
     * (Highstock) Animation when returning to normal state after hovering.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highstock) States for a single point marker.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states
 */
export interface PlotAdMarkerStatesOptions {
    /**
     * (Highstock) The hover state for a single point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.hover
     */
    hover?: PlotAdMarkerStatesHoverOptions;
    /**
     * (Highstock) The normal state of a single point marker. Currently only
     * used for setting animation when returning to normal state from hover.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.normal
     */
    normal?: PlotAdMarkerStatesNormalOptions;
    /**
     * (Highstock) The appearance of the point marker when selected. In order to
     * allow a point to be selected, set the `series.allowPointSelect` option to
     * true.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select
     */
    select?: PlotAdMarkerStatesSelectOptions;
}
/**
 * (Highstock) The appearance of the point marker when selected. In order to
 * allow a point to be selected, set the `series.allowPointSelect` option to
 * true.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select
 */
export interface PlotAdMarkerStatesSelectOptions {
    /**
     * (Highstock) Enable or disable visible feedback for selection.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) The fill color of the point marker.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The color of the point marker's outline. When `undefined`,
     * the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) The radius of the point marker. In hover state, it defaults
     * to the normal state's radius + 2.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker.states.select.radius
     */
    radius?: number;
}
/**
 * (Highstock) Accumulation Distribution (AD). This series requires `linkedTo`
 * option to be set.
 *
 * In TypeScript the type option must always be set.
 *
 * Configuration options for the series are given in three levels:
 *
 * 1. Options for all series in a chart are defined in the plotOptions.series
 * object.
 *
 * 2. Options for all `ad` series are defined in plotOptions.ad.
 *
 * 3. Options for one single series are given in the series instance array.(see
 * online documentation for example)
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad
 */
export interface PlotAdOptions {
    /**
     * (Highstock) Allow this series' points to be selected by clicking on the
     * graphic (columns, point markers, pie slices, map areas etc).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.allowPointSelect
     */
    allowPointSelect?: boolean;
    /**
     * (Highstock) Enable or disable the initial animation when a series is
     * displayed. The animation can also be set as a configuration object.
     * Please note that this option only applies to the initial animation of the
     * series itself. For other animations, see chart.animation and the
     * animation parameter under the API methods. The following properties are
     * supported:
     *
     * - `duration`: The duration of the animation in milliseconds.
     *
     * - `easing`: Can be a string reference to an easing function set on the
     * `Math` object or a function. See the _Custom easing function_ demo below.
     *
     * Due to poor performance, animation is disabled in old IE browsers for
     * several chart types.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAdAnimationOptions);
    /**
     * (Highstock) For some series, there is a limit that shuts down initial
     * animation by default when the total number of points in the chart is too
     * high. For example, for a column chart and its derivatives, animation does
     * not run if there is more than 250 points totally. To disable this cap,
     * set `animationLimit` to `Infinity`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.animationLimit
     */
    animationLimit?: number;
    /**
     * (Highstock) Set the point threshold for when a series should enter boost
     * mode.
     *
     * Setting it to e.g. 2000 will cause the series to enter boost mode when
     * there are 2000 or more points in the series.
     *
     * To disable boosting on the series, set the `boostThreshold` to 0. Setting
     * it to 1 will force boosting.
     *
     * Note that the cropThreshold also affects this setting. When zooming in on
     * a series that has fewer points than the `cropThreshold`, all points are
     * rendered although outside the visible plot area, and the `boostThreshold`
     * won't take effect.
     *
     * Requires `modules/boost.js`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.boostThreshold
     */
    boostThreshold?: number;
    /**
     * (Highmaps) The border color of the map areas.
     *
     * In styled mode, the border stroke is given in the `.highcharts-point`
     * class.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highmaps) The border width of each map area.
     *
     * In styled mode, the border stroke width is given in the
     * `.highcharts-point` class.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) An additional class name to apply to the series' graphical
     * elements. This option does not replace default class names of the
     * graphical element.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.className
     */
    className?: string;
    /**
     * (Highstock) Disable this option to allow series rendering in the whole
     * plotting area.
     *
     * **Note:** Clipping should be always enabled when chart.zoomType is set
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.clip
     */
    clip?: boolean;
    /**
     * (Highstock) The main color of the series. In line type series it applies
     * to the line and the point markers unless otherwise specified. In bar type
     * series it applies to the bars unless a color is specified per point. The
     * default value is pulled from the `options.colors` array.
     *
     * In styled mode, the color can be defined by the colorIndex option. Also,
     * the series color can be set with the `.highcharts-series`,
     * `.highcharts-color-{n}`, `.highcharts-{type}-series` or
     * `.highcharts-series-{n}` class, or individual classes given by the
     * `className` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Styled mode only. A specific color index to use for the
     * series, so its graphic representations are given the class name
     * `highcharts-color-{n}`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.colorIndex
     */
    colorIndex?: number;
    /**
     * (Highstock) Defines if comparison should start from the first point
     * within the visible range or should start from the first point (see online
     * documentation for example) the range. In other words, this flag
     * determines if first point within the visible range will have 0%
     * (`compareStart=true`) or should have been already calculated according to
     * the previous point (`compareStart=false`).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.compareStart
     */
    compareStart?: boolean;
    /**
     * (Highcharts) Polar charts only. Whether to connect the ends of a line
     * series plot across the extremes.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.connectEnds
     */
    connectEnds?: boolean;
    /**
     * (Highcharts, Highstock) Whether to connect a graph line across null
     * points, or render a gap between the two points on either side of the
     * null.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.connectNulls
     * @see https://api.highcharts.com/highstock/plotOptions.ad.connectNulls
     */
    connectNulls?: boolean;
    /**
     * (Gantt) Override Pathfinder connector options for a series. Requires
     * Highcharts Gantt to be loaded.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ad.connectors
     */
    connectors?: PlotAdConnectorsOptions;
    /**
     * (Highcharts, Highstock) When the series contains less points than the
     * crop threshold, all points are drawn, even if the points fall outside the
     * visible plot area at the current zoom. The advantage of drawing all
     * points (including markers and columns), is that animation is performed on
     * updates. On the other hand, when the series contains more points than the
     * crop threshold, the series data is cropped to only contain points that
     * fall within the plot area. The advantage of cropping away invisible
     * points is to increase performance on large series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.cropThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.ad.cropThreshold
     */
    cropThreshold?: number;
    /**
     * (Highstock) You can set the cursor to "pointer" if you have click events
     * attached to the series, to signal to the user that the points and lines
     * can be clicked.
     *
     * In styled mode, the series cursor can be set with the same classes as
     * listed under series.color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.cursor
     */
    cursor?: (string|CursorType);
    /**
     * (Highstock) A name for the dash style to use for the graph, or for some
     * series types the outline of each shape.
     *
     * In styled mode, the stroke dash-array can be set with the same classes as
     * listed under series.color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highstock) Data grouping is the concept of sampling the data values into
     * larger blocks in order to ease readability and increase performance of
     * the JavaScript charts. Highstock by default applies data grouping when
     * the points become closer than a certain pixel value, determined by the
     * `groupPixelWidth` option.
     *
     * If data grouping is applied, the grouping information of grouped points
     * can be read from the Point.dataGroup. If point options other than the
     * data itself are set, for example `name` or `color` or custom properties,
     * the grouping logic doesn't know how to group it. In this case the options
     * of the first point instance are copied over to the group point. This can
     * be altered through a custom `approximation` callback function.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataGrouping
     */
    dataGrouping?: PlotAdDataGroupingOptions;
    /**
     * (Highstock) Options for the series data labels, appearing next to each
     * data point.
     *
     * Since v6.2.0, multiple data labels can be applied to each single point by
     * defining them as an array of configs.
     *
     * In styled mode, the data labels can be styled with the
     * `.highcharts-data-label-box` and `.highcharts-data-label` class names
     * (see example).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dataLabels
     */
    dataLabels?: PlotAdDataLabelsOptions;
    /**
     * (Highstock) Requires the Accessibility module.
     *
     * A description of the series to add to the screen reader information about
     * the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.description
     */
    description?: string;
    /**
     * (Highstock) The draggable-points module allows points to be moved around
     * or modified in the chart. In addition to the options mentioned under the
     * `dragDrop` API structure, the module fires three events, point.dragStart,
     * point.drag and point.drop.
     *
     * It requires the `modules/draggable-points.js` file to be loaded.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.dragDrop
     */
    dragDrop?: PlotAdDragDropOptions;
    /**
     * (Highstock) Enable or disable the mouse tracking for a specific series.
     * This includes point tooltips and click events on graphs and points. For
     * large datasets it improves performance.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.enableMouseTracking
     */
    enableMouseTracking?: boolean;
    /**
     * (Highstock) General event handlers for the series items. These event
     * hooks can also be attached to the series at run time using the
     * `Highcharts.addEvent` function.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.events
     */
    events?: PlotAdEventsOptions;
    /**
     * (Highstock) By default, series are exposed to screen readers as regions.
     * By enabling this option, the series element itself will be exposed in the
     * same way as the data points. This is useful if the series is not used as
     * a grouping entity in the chart, but you still want to attach a
     * description to the series.
     *
     * Requires the Accessibility module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.exposeElementToA11y
     */
    exposeElementToA11y?: boolean;
    /**
     * (Highstock) Determines whether the series should look for the nearest
     * point in both dimensions or just the x-dimension when hovering the
     * series. Defaults to `'xy'` for scatter series and `'x'` for most other
     * series. If the data has duplicate x-values, it is recommended to set this
     * to `'xy'` to allow hovering over all points.
     *
     * Applies only to series types using nearest neighbor search (not direct
     * hover) for tooltip.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.findNearestPointBy
     */
    findNearestPointBy?: ("x"|"xy");
    /**
     * (Highstock) Defines when to display a gap in the graph, together with the
     * gapUnit option.
     *
     * In case when `dataGrouping` is enabled, points can be grouped into a
     * larger time span. This can make the grouped points to have a greater
     * distance than the absolute value of `gapSize` property, which will result
     * in disappearing graph completely. To prevent this situation the mentioned
     * distance between grouped points is used instead of previously defined
     * `gapSize`.
     *
     * In practice, this option is most often used to visualize gaps in time
     * series. In a stock chart, intraday data is available for daytime hours,
     * while gaps will appear in nights and weekends.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.gapSize
     */
    gapSize?: number;
    /**
     * (Highstock) Together with gapSize, this option defines where to draw gaps
     * in the graph.
     *
     * When the `gapUnit` is `relative` (default), a gap size of 5 means that if
     * the distance between two points is greater than five times that of the
     * two closest points, the graph will be broken.
     *
     * When the `gapUnit` is `value`, the gap is based on absolute axis values,
     * which on a datetime axis is milliseconds. This also applies to the
     * navigator series that inherits gap options from the base series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.gapUnit
     */
    gapUnit?: ("relative"|"value");
    /**
     * (Highcharts, Highstock, Gantt) Whether to use the Y extremes of the total
     * chart width or only the zoomed area when zooming in on parts of the X
     * axis. By default, the Y axis adjusts to the min and max of the visible
     * data. Cartesian series only.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.getExtremesFromAll
     * @see https://api.highcharts.com/highstock/plotOptions.ad.getExtremesFromAll
     * @see https://api.highcharts.com/gantt/plotOptions.ad.getExtremesFromAll
     */
    getExtremesFromAll?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
     * series as possible in a natural way, seeking to avoid other series. The
     * goal of this feature is to make the chart more easily readable, like if a
     * human designer placed the labels in the optimal position.
     *
     * The series labels currently work with series types having a `graph` or an
     * `area`.
     *
     * Requires the `series-label.js` module.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.label
     * @see https://api.highcharts.com/highstock/plotOptions.ad.label
     * @see https://api.highcharts.com/gantt/plotOptions.ad.label
     */
    label?: PlotAdLabelOptions;
    /**
     * (Highstock) The line marks the last price from all points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lastPrice
     */
    lastPrice?: PlotAdLastPriceOptions;
    /**
     * (Highstock) The line marks the last price from visible range of points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lastVisiblePrice
     */
    lastVisiblePrice?: PlotAdLastVisiblePriceOptions;
    /**
     * (Highcharts, Highstock) The SVG value used for the `stroke-linecap` and
     * `stroke-linejoin` of a line graph. Round means that lines are rounded in
     * the ends and bends.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.linecap
     * @see https://api.highcharts.com/highstock/plotOptions.ad.linecap
     */
    linecap?: ("butt"|"round"|"square");
    /**
     * (Highcharts, Highstock) Pixel width of the graph line.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock, Gantt) The main series ID that indicator will be
     * based on. Required for this indicator.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.linkedTo
     * @see https://api.highcharts.com/highstock/plotOptions.ad.linkedTo
     * @see https://api.highcharts.com/gantt/plotOptions.ad.linkedTo
     */
    linkedTo?: string;
    /**
     * (Highstock) Options for the point markers of line-like series. Properties
     * like `fillColor`, `lineColor` and `lineWidth` define the visual
     * appearance of the markers. Other series types, like column series, don't
     * have markers, but have visual options on the series level instead.
     *
     * In styled mode, the markers can be styled with the `.highcharts-point`,
     * `.highcharts-point-hover` and `.highcharts-point-select` class names.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.marker
     */
    marker?: PlotAdMarkerOptions;
    /**
     * (Highstock) The name of the series as shown in the legend, tooltip etc.
     * If not set, it will be based on a technical indicator type and default
     * params.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.name
     */
    name?: string;
    /**
     * (Highstock) The color for the parts of the graph or points that are below
     * the threshold.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.negativeColor
     */
    negativeColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Paramters used in calculation of regression series' points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.params
     */
    params?: PlotAdParamsOptions;
    /**
     * (Highstock) Properties for each single point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point
     */
    point?: PlotAdPointOptions;
    /**
     * (Highstock) Same as accessibility.pointDescriptionFormatter, but for an
     * individual series. Overrides the chart wide configuration.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.pointDescriptionFormatter
     */
    pointDescriptionFormatter?: () => void;
    /**
     * (Highstock) Whether to select the series initially. If `showCheckbox` is
     * true, the checkbox next to the series name in the legend will be checked
     * for a selected series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.selected
     */
    selected?: boolean;
    /**
     * (Highstock) Whether to apply a drop shadow to the graph line. Since 2.3
     * the shadow can be an object configuration containing `color`, `offsetX`,
     * `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) If true, a checkbox is displayed next to the legend item to
     * allow selecting the series. The state of the checkbox is determined by
     * the `selected` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.showCheckbox
     */
    showCheckbox?: boolean;
    /**
     * (Highstock) Whether to display this particular series or series type in
     * the legend. The default value is `true` for standalone series, `false`
     * for linked series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.showInLegend
     */
    showInLegend?: boolean;
    /**
     * (Highstock) If set to `true`, the accessibility module will skip past the
     * points in this series for keyboard navigation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.skipKeyboardNavigation
     */
    skipKeyboardNavigation?: boolean;
    /**
     * (Highcharts, Highstock) When this is true, the series will not cause the
     * Y axis to cross the zero plane (or threshold option) unless the data
     * actually crosses the plane.
     *
     * For example, if `softThreshold` is `false`, a series of 0, 1, 2, 3 will
     * make the Y axis show negative values according to the `minPadding`
     * option. If `softThreshold` is `true`, the Y axis starts at 0.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.softThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.ad.softThreshold
     */
    softThreshold?: boolean;
    /**
     * (Highstock) A wrapper object for all the series options in specific
     * states.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states
     */
    states?: PlotAdStatesOptions;
    /**
     * (Highcharts, Highstock) Whether to apply steps to the line. Possible
     * values are `left`, `center` and `right`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.step
     * @see https://api.highcharts.com/highstock/plotOptions.ad.step
     */
    step?: ("center"|"left"|"right");
    /**
     * (Highstock) Sticky tracking of mouse events. When true, the `mouseOut`
     * event on a series isn't triggered until the mouse moves over another
     * series, or out of the plot area. When false, the `mouseOut` event on a
     * series is triggered when the mouse leaves the area around the series'
     * graph or markers. This also implies the tooltip when not shared. When
     * `stickyTracking` is false and `tooltip.shared` is false, the tooltip will
     * be hidden when moving the mouse between series. Defaults to true for line
     * and area type series, but to false for columns, pies etc.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.stickyTracking
     */
    stickyTracking?: boolean;
    /**
     * (Highcharts, Highstock) The threshold, also called zero level or base
     * level. For line type series this is only used in conjunction with
     * negativeColor.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.threshold
     * @see https://api.highcharts.com/highstock/plotOptions.ad.threshold
     */
    threshold?: number;
    /**
     * (Highstock) A configuration object for the tooltip rendering of each
     * single series. Properties are inherited from tooltip, but only the
     * following properties can be defined on a series level.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip
     */
    tooltip?: PlotAdTooltipOptions;
    /**
     * (Highcharts, Highstock, Gantt) When a series contains a data array that
     * is longer than this, only one dimensional arrays of numbers, or two
     * dimensional arrays with x and y values are allowed. Also, only the first
     * point is tested, and the rest are assumed to be the same format. This
     * saves expensive data checking and indexing in long series. Set it to `0`
     * disable.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.turboThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.ad.turboThreshold
     * @see https://api.highcharts.com/gantt/plotOptions.ad.turboThreshold
     */
    turboThreshold?: number;
    /**
     * (Highstock) Set the initial visibility of the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.visible
     */
    visible?: boolean;
    /**
     * (Highmaps) Define the z index of the series.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.zIndex
     */
    zIndex?: number;
    /**
     * (Highcharts, Highstock) Defines the Axis on which the zones are applied.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zoneAxis
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zoneAxis
     */
    zoneAxis?: string;
    /**
     * (Highcharts, Highstock) An array defining zones within a series. Zones
     * can be applied to the X axis, Y axis or Z axis for bubbles, according to
     * the `zoneAxis` option. The zone definitions have to be in ascending order
     * regarding to the value.
     *
     * In styled mode, the color zones are styled with the
     * `.highcharts-zone-{n}` class, or custom classed from the `className`
     * option (view live demo).
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zones
     */
    zones?: Array<PlotAdZonesOptions>;
}
/**
 * (Highstock) Paramters used in calculation of regression series' points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.params
 */
export interface PlotAdParamsOptions {
    /**
     * (Highstock) The point index which indicator calculations will base. For
     * example using OHLC data, index=2 means the indicator will be calculated
     * using Low values.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.params.index
     */
    index?: number;
    /**
     * (Highstock) The base period for indicator calculations. This is the
     * number of data points which are taken into account for the indicator
     * calculations.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.params.period
     */
    period?: number;
    /**
     * (Highstock) The id of volume series which is mandatory. For example using
     * OHLC data, volumeSeriesID='volume' means the indicator will be calculated
     * using OHLC and volume values.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.params.volumeSeriesID
     */
    volumeSeriesID?: string;
}
/**
 * (Highstock) Events for each single point.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events
 */
export interface PlotAdPointEventsOptions {
    /**
     * (Highstock) Fires when a point is clicked. One parameter, `event`, is
     * passed to the function, containing common event information.
     *
     * If the `series.allowPointSelect` option is true, the default action for
     * the point's click event is to toggle the point's select state. Returning
     * `false` cancels this action.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.click
     */
    click?: SeriesPointClickCallbackFunction;
    /**
     * (Highstock) Callback that fires while dragging a point. The mouse event
     * is passed in as parameter. The original data can be accessed from
     * `e.origin`, and the new point values can be accessed from `e.newPoints`.
     * If there is only a single point being updated, it can be accessed from
     * `e.newPoint` for simplicity, and its ID can be accessed from
     * `e.newPointId`. The `this` context is the point being dragged. To stop
     * the default drag action, return false. See drag and drop options.
     *
     * Requires the `draggable-points` module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.drag
     */
    drag?: SeriesPointDragCallbackFunction;
    /**
     * (Highstock) Callback that fires when starting to drag a point. The mouse
     * event object is passed in as an argument. If a drag handle is used,
     * `e.updateProp` is set to the data property being dragged. The `this`
     * context is the point. See drag and drop options.
     *
     * Requires the `draggable-points` module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.dragStart
     */
    dragStart?: SeriesPointDragStartCallbackFunction;
    /**
     * (Highstock) Callback that fires when the point is dropped. The parameters
     * passed are the same as for drag. To stop the default drop action, return
     * false. See drag and drop options.
     *
     * Requires the `draggable-points` module.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.drop
     */
    drop?: SeriesPointDropCallbackFunction;
    /**
     * (Highstock) Fires when the mouse leaves the area close to the point. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.mouseOut
     */
    mouseOut?: SeriesPointMouseOutCallbackFunction;
    /**
     * (Highstock) Fires when the mouse enters the area close to the point. One
     * parameter, `event`, is passed to the function, containing common event
     * information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.mouseOver
     */
    mouseOver?: SeriesPointMouseOverCallbackFunction;
    /**
     * (Highstock) Fires when the point is removed using the `.remove()` method.
     * One parameter, `event`, is passed to the function. Returning `false`
     * cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.remove
     */
    remove?: SeriesPointRemoveCallbackFunction;
    /**
     * (Highstock) Fires when the point is selected either programmatically or
     * following a click on the point. One parameter, `event`, is passed to the
     * function. Returning `false` cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.select
     */
    select?: SeriesPointSelectCallbackFunction;
    /**
     * (Highstock) Fires when the point is unselected either programmatically or
     * following a click on the point. One parameter, `event`, is passed to the
     * function. Returning `false` cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.unselect
     */
    unselect?: SeriesPointUnselectCallbackFunction;
    /**
     * (Highstock) Fires when the point is updated programmatically through the
     * `.update()` method. One parameter, `event`, is passed to the function.
     * The new point options can be accessed through `event.options`. Returning
     * `false` cancels the operation.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events.update
     */
    update?: SeriesPointUpdateCallbackFunction;
}
/**
 * (Highstock) Properties for each single point.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.point
 */
export interface PlotAdPointOptions {
    /**
     * (Highstock) Events for each single point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.point.events
     */
    events?: PlotAdPointEventsOptions;
}
/**
 * (Highcharts) Animation setting for hovering the graph in line-type series.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.animation
 */
export interface PlotAdStatesHoverAnimationOptions {
    /**
     * (Highcharts) The duration of the hover animation in milliseconds. By
     * default the hover state animates quickly in, and slowly back to normal.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.animation.duration
     */
    duration?: number;
}
/**
 * (Highcharts, Highstock) Options for the halo appearing around the hovered
 * point in line-type series as well as outside the hovered slice in pie charts.
 * By default the halo is filled by the current point or series color with an
 * opacity of 0.25\. The halo can be disabled by setting the `halo` option to
 * `false`.
 *
 * In styled mode, the halo is styled with the `.highcharts-halo` class, with
 * colors inherited from `.highcharts-color-{n}`.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.halo
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.halo
 */
export interface PlotAdStatesHoverHaloOptions {
    /**
     * (Highcharts, Highstock) A collection of SVG attributes to override the
     * appearance of the halo, for example `fill`, `stroke` and `stroke-width`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.halo.attributes
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.halo.attributes
     */
    attributes?: SVGAttributes;
    /**
     * (Highcharts, Highstock) Opacity for the halo unless a specific fill is
     * overridden using the `attributes` setting. Note that Highcharts is only
     * able to apply opacity to colors of hex or rgb(a) formats.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.halo.opacity
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.halo.opacity
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) The pixel size of the halo. For point markers
     * this is the radius of the halo. For pie slices it is the width of the
     * halo outside the slice. For bubbles it defaults to 5 and is the width of
     * the halo outside the bubble.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.halo.size
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.halo.size
     */
    size?: number;
}
/**
 * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
 * belonging to the hovered series. For settings on the hover state of the
 * individual point, see marker.states.hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker
 */
export interface PlotAdStatesHoverMarkerOptions {
    /**
     * (Highcharts, Highstock) Enable or disable the point marker. If
     * `undefined`, the markers are hidden when the data is dense, and shown for
     * more widespread data points.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The threshold for how dense the point markers
     * should be before they are hidden, given that `enabled` is not defined.
     * The number indicates the horizontal distance between the two closest
     * points in the series, as multiples of the `marker.radius`. In other
     * words, the default value of 2 means points are hidden if overlapping
     * horizontally.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.enabledThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.enabledThreshold
     */
    enabledThreshold?: number;
    /**
     * (Highcharts, Highstock) The fill color of the point marker. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `width` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.height
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.height
     */
    height?: number;
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.radius
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) States for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states
     */
    states?: PlotAdStatesHoverMarkerStatesOptions;
    /**
     * (Highcharts, Highstock) A predefined shape or symbol for the marker. When
     * undefined, the symbol is pulled from options.symbols. Other possible
     * values are "circle", "square", "diamond", "triangle" and "triangle-down".
     *
     * Additionally, the URL to a graphic can be given on this form:
     * "url(graphic.png)". Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.symbol
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.symbol
     */
    symbol?: string;
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `height` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.width
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock) Animation when hovering over the marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.animation
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.animation
 */
export interface PlotAdStatesHoverMarkerStatesHoverAnimationOptions {
    duration?: number;
}
/**
 * (Highcharts, Highstock) The hover state for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover
 */
export interface PlotAdStatesHoverMarkerStatesHoverOptions {
    /**
     * (Highcharts, Highstock) Animation when hovering over the marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.animation
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAdStatesHoverMarkerStatesHoverAnimationOptions);
    /**
     * (Highcharts, Highstock) Enable or disable the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the marker in hover state. When
     * `undefined`, the series' or point's fillColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's lineColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline. When
     * `undefined`, the series' or point's lineWidth for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for a hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2 as per the radiusPlus
     * option.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.radius
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) The number of pixels to increase the radius of
     * the hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover.radiusPlus
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover.radiusPlus
     */
    radiusPlus?: number;
}
/**
 * (Highcharts, Highstock) The normal state of a single point marker. Currently
 * only used for setting animation when returning to normal state from hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.normal
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.normal
 */
export interface PlotAdStatesHoverMarkerStatesNormalOptions {
    /**
     * (Highcharts, Highstock) Animation when returning to normal state after
     * hovering.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.normal.animation
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highcharts, Highstock) States for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states
 */
export interface PlotAdStatesHoverMarkerStatesOptions {
    /**
     * (Highcharts, Highstock) The hover state for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.hover
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.hover
     */
    hover?: PlotAdStatesHoverMarkerStatesHoverOptions;
    /**
     * (Highcharts, Highstock) The normal state of a single point marker.
     * Currently only used for setting animation when returning to normal state
     * from hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.normal
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.normal
     */
    normal?: PlotAdStatesHoverMarkerStatesNormalOptions;
    /**
     * (Highcharts, Highstock) The appearance of the point marker when selected.
     * In order to allow a point to be selected, set the
     * `series.allowPointSelect` option to true.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select
     */
    select?: PlotAdStatesHoverMarkerStatesSelectOptions;
}
/**
 * (Highcharts, Highstock) The appearance of the point marker when selected. In
 * order to allow a point to be selected, set the `series.allowPointSelect`
 * option to true.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select
 */
export interface PlotAdStatesHoverMarkerStatesSelectOptions {
    /**
     * (Highcharts, Highstock) Enable or disable visible feedback for selection.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker.states.select.radius
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker.states.select.radius
     */
    radius?: number;
}
/**
 * (Highstock) Options for the hovered series. These settings override the
 * normal state options when a series is moused over or touched.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover
 */
export interface PlotAdStatesHoverOptions {
    /**
     * (Highcharts) Animation setting for hovering the graph in line-type
     * series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAdStatesHoverAnimationOptions);
    /**
     * (Highmaps) The border color of the point in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.hover.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highmaps) The border width of the point in this state
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.hover.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highmaps) The relative brightness of the point when hovered, relative to
     * the normal point color.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.hover.brightness
     */
    brightness?: number;
    /**
     * (Highmaps) The color of the shape in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.hover.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Enable separate styles for the hovered series to visualize
     * that the user hovers either the series itself or the legend.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Options for the halo appearing around the hovered
     * point in line-type series as well as outside the hovered slice in pie
     * charts. By default the halo is filled by the current point or series
     * color with an opacity of 0.25\. The halo can be disabled by setting the
     * `halo` option to `false`.
     *
     * In styled mode, the halo is styled with the `.highcharts-halo` class,
     * with colors inherited from `.highcharts-color-{n}`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.halo
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.halo
     */
    halo?: PlotAdStatesHoverHaloOptions;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line. By default this
     * property is undefined, and the `lineWidthPlus` property dictates how much
     * to increase the linewidth from normal state.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for the graph of a
     * hovered series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
     * belonging to the hovered series. For settings on the hover state of the
     * individual point, see marker.states.hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.hover.marker
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover.marker
     */
    marker?: PlotAdStatesHoverMarkerOptions;
}
/**
 * (Highmaps) Overrides for the normal state.
 *
 * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.normal
 */
export interface PlotAdStatesNormalOptions {
    /**
     * (Highmaps) Animation options for the fill color when returning from hover
     * state to normal state. The animation adds some latency in order to reduce
     * the effect of flickering when hovering in and out of for example an
     * uneven coastline.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highstock) A wrapper object for all the series options in specific states.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states
 */
export interface PlotAdStatesOptions {
    /**
     * (Highstock) Options for the hovered series. These settings override the
     * normal state options when a series is moused over or touched.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.hover
     */
    hover?: PlotAdStatesHoverOptions;
    /**
     * (Highmaps) Overrides for the normal state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.normal
     */
    normal?: PlotAdStatesNormalOptions;
    /**
     * (Highmaps) Specific options for point in selected states, after being
     * selected by allowPointSelect or programmatically.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.select
     */
    select?: PlotAdStatesSelectOptions;
}
/**
 * (Highcharts) Animation setting for hovering the graph in line-type series.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.animation
 */
export interface PlotAdStatesSelectAnimationOptions {
    /**
     * (Highcharts) The duration of the hover animation in milliseconds. By
     * default the hover state animates quickly in, and slowly back to normal.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.animation.duration
     */
    duration?: number;
}
/**
 * (Highcharts, Highstock) Options for the halo appearing around the hovered
 * point in line-type series as well as outside the hovered slice in pie charts.
 * By default the halo is filled by the current point or series color with an
 * opacity of 0.25\. The halo can be disabled by setting the `halo` option to
 * `false`.
 *
 * In styled mode, the halo is styled with the `.highcharts-halo` class, with
 * colors inherited from `.highcharts-color-{n}`.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.halo
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.halo
 */
export interface PlotAdStatesSelectHaloOptions {
    /**
     * (Highcharts, Highstock) A collection of SVG attributes to override the
     * appearance of the halo, for example `fill`, `stroke` and `stroke-width`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.halo.attributes
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.halo.attributes
     */
    attributes?: SVGAttributes;
    /**
     * (Highcharts, Highstock) Opacity for the halo unless a specific fill is
     * overridden using the `attributes` setting. Note that Highcharts is only
     * able to apply opacity to colors of hex or rgb(a) formats.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.halo.opacity
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.halo.opacity
     */
    opacity?: number;
    /**
     * (Highcharts, Highstock) The pixel size of the halo. For point markers
     * this is the radius of the halo. For pie slices it is the width of the
     * halo outside the slice. For bubbles it defaults to 5 and is the width of
     * the halo outside the bubble.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.halo.size
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.halo.size
     */
    size?: number;
}
/**
 * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
 * belonging to the hovered series. For settings on the hover state of the
 * individual point, see marker.states.hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker
 */
export interface PlotAdStatesSelectMarkerOptions {
    /**
     * (Highcharts, Highstock) Enable or disable the point marker. If
     * `undefined`, the markers are hidden when the data is dense, and shown for
     * more widespread data points.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The threshold for how dense the point markers
     * should be before they are hidden, given that `enabled` is not defined.
     * The number indicates the horizontal distance between the two closest
     * points in the series, as multiples of the `marker.radius`. In other
     * words, the default value of 2 means points are hidden if overlapping
     * horizontally.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.enabledThreshold
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.enabledThreshold
     */
    enabledThreshold?: number;
    /**
     * (Highcharts, Highstock) The fill color of the point marker. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `width` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.height
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.height
     */
    height?: number;
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.radius
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) States for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states
     */
    states?: PlotAdStatesSelectMarkerStatesOptions;
    /**
     * (Highcharts, Highstock) A predefined shape or symbol for the marker. When
     * undefined, the symbol is pulled from options.symbols. Other possible
     * values are "circle", "square", "diamond", "triangle" and "triangle-down".
     *
     * Additionally, the URL to a graphic can be given on this form:
     * "url(graphic.png)". Note that for the image to be applied to exported
     * charts, its URL needs to be accessible by the export server.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols`. The callback is then used by
     * its method name, as shown in the demo.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.symbol
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.symbol
     */
    symbol?: string;
    /**
     * (Highcharts, Highstock) Image markers only. Set the image width
     * explicitly. When using this option, a `height` must also be set.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.width
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.width
     */
    width?: number;
}
/**
 * (Highcharts, Highstock) Animation when hovering over the marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.animation
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.animation
 */
export interface PlotAdStatesSelectMarkerStatesHoverAnimationOptions {
    duration?: number;
}
/**
 * (Highcharts, Highstock) The hover state for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover
 */
export interface PlotAdStatesSelectMarkerStatesHoverOptions {
    /**
     * (Highcharts, Highstock) Animation when hovering over the marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.animation
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.animation
     */
    animation?: (boolean|AnimationOptionsObject|PlotAdStatesSelectMarkerStatesHoverAnimationOptions);
    /**
     * (Highcharts, Highstock) Enable or disable the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the marker in hover state. When
     * `undefined`, the series' or point's fillColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's lineColor for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline. When
     * `undefined`, the series' or point's lineWidth for normal state is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for a hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2 as per the radiusPlus
     * option.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.radius
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.radius
     */
    radius?: number;
    /**
     * (Highcharts, Highstock) The number of pixels to increase the radius of
     * the hovered point.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover.radiusPlus
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover.radiusPlus
     */
    radiusPlus?: number;
}
/**
 * (Highcharts, Highstock) The normal state of a single point marker. Currently
 * only used for setting animation when returning to normal state from hover.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.normal
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.normal
 */
export interface PlotAdStatesSelectMarkerStatesNormalOptions {
    /**
     * (Highcharts, Highstock) Animation when returning to normal state after
     * hovering.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.normal.animation
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.normal.animation
     */
    animation?: (boolean|AnimationOptionsObject);
}
/**
 * (Highcharts, Highstock) States for a single point marker.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states
 */
export interface PlotAdStatesSelectMarkerStatesOptions {
    /**
     * (Highcharts, Highstock) The hover state for a single point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.hover
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.hover
     */
    hover?: PlotAdStatesSelectMarkerStatesHoverOptions;
    /**
     * (Highcharts, Highstock) The normal state of a single point marker.
     * Currently only used for setting animation when returning to normal state
     * from hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.normal
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.normal
     */
    normal?: PlotAdStatesSelectMarkerStatesNormalOptions;
    /**
     * (Highcharts, Highstock) The appearance of the point marker when selected.
     * In order to allow a point to be selected, set the
     * `series.allowPointSelect` option to true.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select
     */
    select?: PlotAdStatesSelectMarkerStatesSelectOptions;
}
/**
 * (Highcharts, Highstock) The appearance of the point marker when selected. In
 * order to allow a point to be selected, set the `series.allowPointSelect`
 * option to true.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select
 * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select
 */
export interface PlotAdStatesSelectMarkerStatesSelectOptions {
    /**
     * (Highcharts, Highstock) Enable or disable visible feedback for selection.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) The fill color of the point marker.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The color of the point marker's outline. When
     * `undefined`, the series' or point's color is used.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select.lineColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highcharts, Highstock) The width of the point marker's outline.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The radius of the point marker. In hover state,
     * it defaults to the normal state's radius + 2.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker.states.select.radius
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker.states.select.radius
     */
    radius?: number;
}
/**
 * (Highmaps) Specific options for point in selected states, after being
 * selected by allowPointSelect or programmatically.
 *
 * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.select
 */
export interface PlotAdStatesSelectOptions {
    /**
     * (Highcharts) Animation setting for hovering the graph in line-type
     * series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.animation
     */
    animation?: PlotAdStatesSelectAnimationOptions;
    /**
     * (Highmaps) The border color of the point in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.select.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highmaps) The border width of the point in this state
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.select.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highmaps) The color of the shape in this state.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.select.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highmaps) Enable separate styles for the hovered series to visualize
     * that the user hovers either the series itself or the legend.
     *
     * @see https://api.highcharts.com/highmaps/plotOptions.ad.states.select.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock) Options for the halo appearing around the hovered
     * point in line-type series as well as outside the hovered slice in pie
     * charts. By default the halo is filled by the current point or series
     * color with an opacity of 0.25\. The halo can be disabled by setting the
     * `halo` option to `false`.
     *
     * In styled mode, the halo is styled with the `.highcharts-halo` class,
     * with colors inherited from `.highcharts-color-{n}`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.halo
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.halo
     */
    halo?: PlotAdStatesSelectHaloOptions;
    /**
     * (Highcharts, Highstock) Pixel width of the graph line. By default this
     * property is undefined, and the `lineWidthPlus` property dictates how much
     * to increase the linewidth from normal state.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.lineWidth
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highcharts, Highstock) The additional line width for the graph of a
     * hovered series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.lineWidthPlus
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.lineWidthPlus
     */
    lineWidthPlus?: number;
    /**
     * (Highcharts, Highstock) In Highcharts 1.0, the appearance of all markers
     * belonging to the hovered series. For settings on the hover state of the
     * individual point, see marker.states.hover.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.states.select.marker
     * @see https://api.highcharts.com/highstock/plotOptions.ad.states.select.marker
     */
    marker?: PlotAdStatesSelectMarkerOptions;
}
/**
 * (Highcharts, Highstock, Gantt) For series on a datetime axes, the date format
 * in the tooltip's header will by default be guessed based on the closest data
 * points. This member gives the default string representations used for each
 * unit. For an overview of the replacement codes, see dateFormat.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.tooltip.dateTimeLabelFormats
 * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.dateTimeLabelFormats
 * @see https://api.highcharts.com/gantt/plotOptions.ad.tooltip.dateTimeLabelFormats
 */
export interface PlotAdTooltipDateTimeLabelFormatsOptions {
    day?: string;
    hour?: string;
    millisecond?: string;
    minute?: string;
    month?: string;
    second?: string;
    week?: string;
    year?: string;
}
/**
 * (Highstock) A configuration object for the tooltip rendering of each single
 * series. Properties are inherited from tooltip, but only the following
 * properties can be defined on a series level.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip
 */
export interface PlotAdTooltipOptions {
    /**
     * (Highstock) How many decimals to show for the `point.change` value when
     * the `series.compare` option is set. This is overridable in each series'
     * tooltip options object. The default is to preserve all decimals.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.changeDecimals
     */
    changeDecimals?: number;
    /**
     * (Highcharts, Highstock, Gantt) For series on a datetime axes, the date
     * format in the tooltip's header will by default be guessed based on the
     * closest data points. This member gives the default string representations
     * used for each unit. For an overview of the replacement codes, see
     * dateFormat.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.tooltip.dateTimeLabelFormats
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.dateTimeLabelFormats
     * @see https://api.highcharts.com/gantt/plotOptions.ad.tooltip.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: (PlotAdTooltipDateTimeLabelFormatsOptions|Dictionary<string>);
    /**
     * (Highstock) Whether the tooltip should follow the mouse as it moves
     * across columns, pie slices and other point types with an extent. By
     * default it behaves this way for scatter, bubble and pie series by
     * override in the `plotOptions` for those series types.
     *
     * For touch moves to behave the same way, followTouchMove must be `true`
     * also.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.followPointer
     */
    followPointer?: boolean;
    /**
     * (Highstock) Whether the tooltip should update as the finger moves on a
     * touch device. If this is `true` and chart.panning is
     * set,`followTouchMove` will take over one-finger touches, so the user
     * needs to use two fingers for zooming and panning.
     *
     * Note the difference to followPointer that only defines the _position_ of
     * the tooltip. If `followPointer` is false in for example a column series,
     * the tooltip will show above or below the column, but as `followTouchMove`
     * is true, the tooltip will jump from column to column as the user swipes
     * across the plot area.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.followTouchMove
     */
    followTouchMove?: boolean;
    /**
     * (Highstock) A string to append to the tooltip format.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.footerFormat
     */
    footerFormat?: string;
    /**
     * (Highstock) The HTML of the tooltip header line. Variables are enclosed
     * by curly brackets. Available variables are `point.key`, `series.name`,
     * `series.color` and other members from the `point` and `series` objects.
     * The `point.key` variable contains the category name, x value or datetime
     * string depending on the type of axis. For datetime axes, the `point.key`
     * date format can be set using `tooltip.xDateFormat`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.headerFormat
     */
    headerFormat?: string;
    /**
     * (Highstock) The name of a symbol to use for the border around the tooltip
     * header. Applies only when tooltip.split is enabled.
     *
     * Custom callbacks for symbol path generation can also be added to
     * `Highcharts.SVGRenderer.prototype.symbols` the same way as for
     * series.marker.symbol.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.headerShape
     */
    headerShape?: ("callout"|"square");
    /**
     * (Highstock) The number of milliseconds to wait until the tooltip is
     * hidden when mouse out from a point or chart.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.hideDelay
     */
    hideDelay?: number;
    /**
     * (Highstock) Whether to allow the tooltip to render outside the chart's
     * SVG element box. By default (`false`), the tooltip is rendered within the
     * chart's SVG element, which results in the tooltip being aligned inside
     * the chart area. For small charts, this may result in clipping or
     * overlapping. When `true`, a separate SVG element is created and overlaid
     * on the page, allowing the tooltip to be aligned inside the page itself.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.outside
     */
    outside?: boolean;
    /**
     * (Highstock) Padding inside the tooltip, in pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.padding
     */
    padding?: number;
    /**
     * (Highstock) The HTML of the point's line in the tooltip. Variables are
     * enclosed by curly brackets. Available variables are point.x, point.y,
     * series. name and series.color and other properties on the same form.
     * Furthermore, `point.y` can be extended by the `tooltip.valuePrefix` and
     * `tooltip.valueSuffix` variables. This can also be overridden for each
     * series, which makes it a good hook for displaying units.
     *
     * In styled mode, the dot is colored by a class name rather than the point
     * color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.pointFormat
     */
    pointFormat?: string;
    /**
     * (Highstock) A callback function for formatting the HTML output for a
     * single point in the tooltip. Like the `pointFormat` string, but with more
     * flexibility.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.pointFormatter
     */
    pointFormatter?: () => void;
    /**
     * (Highcharts, Highstock) Split the tooltip into one label per series, with
     * the header close to the axis. This is recommended over shared tooltips
     * for charts with multiple line series, generally making them easier to
     * read. This option takes precedence over `tooltip.shared`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.tooltip.split
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.split
     */
    split?: boolean;
    /**
     * (Highstock) Number of decimals in indicator series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.valueDecimals
     */
    valueDecimals?: number;
    /**
     * (Highstock) A string to prepend to each series' y value. Overridable in
     * each series' tooltip options object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.valuePrefix
     */
    valuePrefix?: string;
    /**
     * (Highstock) A string to append to each series' y value. Overridable in
     * each series' tooltip options object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.valueSuffix
     */
    valueSuffix?: string;
    /**
     * (Highcharts, Highstock, Gantt) The format for the date in the tooltip
     * header if the X axis is a datetime axis. The default is a best guess
     * based on the smallest distance between points in the chart.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.tooltip.xDateFormat
     * @see https://api.highcharts.com/highstock/plotOptions.ad.tooltip.xDateFormat
     * @see https://api.highcharts.com/gantt/plotOptions.ad.tooltip.xDateFormat
     */
    xDateFormat?: string;
}
/**
 * (Highcharts, Highstock) An array defining zones within a series. Zones can be
 * applied to the X axis, Y axis or Z axis for bubbles, according to the
 * `zoneAxis` option. The zone definitions have to be in ascending order
 * regarding to the value.
 *
 * In styled mode, the color zones are styled with the `.highcharts-zone-{n}`
 * class, or custom classed from the `className` option (view live demo).
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones
 * @see https://api.highcharts.com/highstock/plotOptions.ad.zones
 */
export interface PlotAdZonesOptions {
    /**
     * (Highcharts, Highstock) Styled mode only. A custom class name for the
     * zone.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones.className
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zones.className
     */
    className?: string;
    /**
     * (Highcharts, Highstock) Defines the color of the series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones.color
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zones.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) A name for the dash style to use for the graph.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones.dashStyle
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zones.dashStyle
     */
    dashStyle?: DashStyleType;
    /**
     * (Highcharts, Highstock) Defines the fill color for the series (in area
     * type series)
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones.fillColor
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zones.fillColor
     */
    fillColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highcharts, Highstock) The value up to where the zone extends, if
     * undefined the zones stretches to the last value in the series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ad.zones.value
     * @see https://api.highcharts.com/highstock/plotOptions.ad.zones.value
     */
    value?: number;
}
/**
 * (Highstock) Enable or disable the initial animation when a series is
 * displayed. The animation can also be set as a configuration object. Please
 * note that this option only applies to the initial animation of the series
 * itself. For other animations, see chart.animation and the animation parameter
 * under the API methods. The following properties are supported:
 *
 * - `duration`: The duration of the animation in milliseconds.
 *
 * - `easing`: Can be a string reference to an easing function set on the `Math`
 * object or a function. See the _Custom easing function_ demo below.
 *
 * Due to poor performance, animation is disabled in old IE browsers for several
 * chart types.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.animation
 */
export interface PlotAoAnimationOptions {
    duration?: number;
}
/**
 * (Gantt) Marker options specific to the end markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker
 */
export interface PlotAoConnectorsEndMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector end markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker.width
     */
    width?: number;
}
/**
 * (Gantt) Marker options for this chart's Pathfinder connectors. Note that this
 * option is overridden by the `startMarker` and `endMarker` options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker
 */
export interface PlotAoConnectorsMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.radius
     */
    radius?: number;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker.width
     */
    width?: number;
}
/**
 * (Gantt) Override Pathfinder connector options for a series. Requires
 * Highcharts Gantt to be loaded.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors
 */
export interface PlotAoConnectorsOptions {
    /**
     * (Gantt) Set the default dash style for this chart's connecting lines.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.dashStyle
     */
    dashStyle?: string;
    /**
     * (Gantt) Marker options specific to the end markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.endMarker
     */
    endMarker?: PlotAoConnectorsEndMarkerOptions;
    /**
     * (Gantt) Set the default color for this chart's Pathfinder connecting
     * lines. Defaults to the color of the point being connected.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the default pixel width for this chart's Pathfinder
     * connecting lines.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Marker options for this chart's Pathfinder connectors. Note that
     * this option is overridden by the `startMarker` and `endMarker` options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.marker
     */
    marker?: PlotAoConnectorsMarkerOptions;
    /**
     * (Gantt) Marker options specific to the start markers for this chart's
     * Pathfinder connectors. Overrides the generic marker options.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker
     */
    startMarker?: PlotAoConnectorsStartMarkerOptions;
    /**
     * (Gantt) Set the default pathfinder algorithm to use for this chart. It is
     * possible to define your own algorithms by adding them to the
     * Highcharts.Pathfinder.prototype.algorithms object before the chart has
     * been created.
     *
     * The default algorithms are as follows:
     *
     * `straight`: Draws a straight line between the connecting points. Does not
     * avoid other points when drawing.
     *
     * `simpleConnect`: Finds a path between the points using right angles only.
     * Takes only starting/ending points into account, and will not avoid other
     * points.
     *
     * `fastAvoid`: Finds a path between the points using right angles only.
     * Will attempt to avoid other points, but its focus is performance over
     * accuracy. Works well with less dense datasets.
     *
     * Default value: `straight` is used as default for most series types, while
     * `simpleConnect` is used as default for Gantt series, to show dependencies
     * between points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.type
     */
    type?: ("straight"|"fastAvoid"|"simpleConnect");
}
/**
 * (Gantt) Marker options specific to the start markers for this chart's
 * Pathfinder connectors. Overrides the generic marker options.
 *
 * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker
 */
export interface PlotAoConnectorsStartMarkerOptions {
    /**
     * (Gantt) Horizontal alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.align
     */
    align?: AlignType;
    /**
     * (Gantt) Set the color of the connector markers. By default this is the
     * same as the connector color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Gantt) Enable markers for the connectors.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.enabled
     */
    enabled?: boolean;
    /**
     * (Gantt) Set the height of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.height
     */
    height?: number;
    /**
     * (Gantt) Whether or not to draw the markers inside the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.inside
     */
    inside?: boolean;
    /**
     * (Gantt) Set the line/border color of the connector markers. By default
     * this is the same as the marker color.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Gantt) Set the line/border width of the pathfinder markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.lineWidth
     */
    lineWidth?: number;
    /**
     * (Gantt) Set the radius of the connector markers. The default is
     * automatically computed based on the algorithmMargin setting.
     *
     * Setting marker.width and marker.height will override this setting.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.radius
     */
    radius?: number;
    /**
     * (Gantt) Set the symbol of the connector start markers.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.symbol
     */
    symbol?: string;
    /**
     * (Gantt) Vertical alignment of the markers relative to the points.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Gantt) Set the width of the connector markers. If not supplied, this is
     * inferred from the marker radius.
     *
     * @see https://api.highcharts.com/gantt/plotOptions.ao.connectors.startMarker.width
     */
    width?: number;
}
/**
 * (Highstock) Data grouping is the concept of sampling the data values into
 * larger blocks in order to ease readability and increase performance of the
 * JavaScript charts. Highstock by default applies data grouping when the points
 * become closer than a certain pixel value, determined by the `groupPixelWidth`
 * option.
 *
 * If data grouping is applied, the grouping information of grouped points can
 * be read from the Point.dataGroup. If point options other than the data itself
 * are set, for example `name` or `color` or custom properties, the grouping
 * logic doesn't know how to group it. In this case the options of the first
 * point instance are copied over to the group point. This can be altered
 * through a custom `approximation` callback function.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping
 */
export interface PlotAoDataGroupingOptions {
    /**
     * (Highstock) The method of approximation inside a group. When for example
     * 30 days are grouped into one month, this determines what value should
     * represent the group. Possible values are "average", "averages", "open",
     * "high", "low", "close" and "sum". For OHLC and candlestick series the
     * approximation is "ohlc" by default, which finds the open, high, low and
     * close values within all the grouped data. For ranges, the approximation
     * is "range", which finds the low and high values. For multi-dimensional
     * data, like ranges and OHLC, "averages" will compute the average for each
     * dimension.
     *
     * Custom aggregate methods can be added by assigning a callback function as
     * the approximation. This function takes a numeric array as the argument
     * and should return a single numeric value or `null`. Note that the numeric
     * array will never contain null values, only true numbers. Instead, if null
     * values are present in the raw data, the numeric array will have an
     * `.hasNulls` property set to `true`. For single-value data sets the data
     * is available in the first argument of the callback function. For OHLC
     * data sets, all the open values are in the first argument, all high values
     * in the second etc.
     *
     * Since v4.2.7, grouping meta data is available in the approximation
     * callback from `this.dataGroupInfo`. It can be used to extract information
     * from the raw data.
     *
     * Defaults to `average` for line-type series, `sum` for columns, `range`
     * for range series and `ohlc` for OHLC and candlestick.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.approximation
     */
    approximation?: ("average"|"averages"|"close"|"high"|"low"|"open"|"sum");
    /**
     * (Highstock) Datetime formats for the header of the tooltip in a stock
     * chart. The format can vary within a chart depending on the currently
     * selected time range and the current data grouping.
     *
     * The default formats are:
     *
     * (see online documentation for example)
     *
     * For each of these array definitions, the first item is the format used
     * when the active time span is one unit. For instance, if the current data
     * applies to one week, the first item of the week array is used. The second
     * and third items are used when the active time span is more than two
     * units. For instance, if the current data applies to two weeks, the second
     * and third item of the week array are used, and applied to the start and
     * end date of the time span.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.dateTimeLabelFormats
     */
    dateTimeLabelFormats?: object;
    /**
     * (Highstock) Enable or disable data grouping.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) When data grouping is forced, it runs no matter how small the
     * intervals are. This can be handy for example when the sum should be
     * calculated for values appearing at random times within each hour.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.forced
     */
    forced?: boolean;
    /**
     * (Highstock) By default only points within the visible range are grouped.
     * Enabling this option will force data grouping to calculate all grouped
     * points for a given dataset. That option prevents for example a column
     * series from calculating a grouped point partially. The effect is similar
     * to Series.getExtremesFromAll but does not affect yAxis extremes.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.groupAll
     */
    groupAll?: boolean;
    /**
     * (Highstock) The approximate pixel width of each group. If for example a
     * series with 30 points is displayed over a 600 pixel wide plot area, no
     * grouping is performed. If however the series contains so many points that
     * the spacing is less than the groupPixelWidth, Highcharts will try to
     * group it into appropriate groups so that each is more or less two pixels
     * wide. If multiple series with different group pixel widths are drawn on
     * the same x axis, all series will take the greatest width. For example,
     * line series have 2px default group width, while column series have 10px.
     * If combined, both the line and the column will have 10px by default.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.groupPixelWidth
     */
    groupPixelWidth?: number;
    /**
     * (Highstock) Normally, a group is indexed by the start of that group, so
     * for example when 30 daily values are grouped into one month, that month's
     * x value will be the 1st of the month. This apparently shifts the data to
     * the left. When the smoothed option is true, this is compensated for. The
     * data is shifted to the middle of the group, and min and max values are
     * preserved. Internally, this is used in the Navigator series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.smoothed
     */
    smoothed?: boolean;
    /**
     * (Highstock) An array determining what time intervals the data is allowed
     * to be grouped to. Each array item is an array where the first value is
     * the time unit and the second value another array of allowed multiples.
     * Defaults to:
     *
     * (see online documentation for example)
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataGrouping.units
     */
    units?: Array<[string, (Array<number>|null)]>;
}
/**
 * (Highstock) A declarative filter for which data labels to display. The
 * declarative filter is designed for use when callback functions are not
 * available, like when the chart options require a pure JSON structure or for
 * use with graphical editors. For programmatic control, use the `formatter`
 * instead, and return `undefined` to disable a single data label.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.filter
 */
export interface PlotAoDataLabelsFilterOptions {
    /**
     * (Highstock) The operator to compare by. Can be one of `>`, `<`, `>=`,
     * `<=`, `==`, and `===`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.filter.operator
     */
    operator?: ("=="|"==="|">"|">="|"<"|"<=");
    /**
     * (Highstock) The point property to filter by. Point options are passed
     * directly to properties, additionally there are `y` value, `percentage`
     * and others listed under Point members.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.filter.property
     */
    property?: string;
    /**
     * (Highstock) The value to compare against.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.filter.value
     */
    value?: any;
}
/**
 * (Highstock) Options for the series data labels, appearing next to each data
 * point.
 *
 * Since v6.2.0, multiple data labels can be applied to each single point by
 * defining them as an array of configs.
 *
 * In styled mode, the data labels can be styled with the
 * `.highcharts-data-label-box` and `.highcharts-data-label` class names (see
 * example).
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels
 */
export interface PlotAoDataLabelsOptions {
    /**
     * (Highstock) The alignment of the data label compared to the point. If
     * `right`, the right side of the label should be touching the point. For
     * points with an extent, like columns, the alignments also dictates how to
     * align it inside the box, as given with the inside option. Can be one of
     * `left`, `center` or `right`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.align
     */
    align?: AlignType;
    /**
     * (Highstock) Whether to allow data labels to overlap. To make the labels
     * less sensitive for overlapping, the dataLabels.padding can be set to 0.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.allowOverlap
     */
    allowOverlap?: boolean;
    /**
     * (Highstock) The background color or gradient for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.backgroundColor
     */
    backgroundColor?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The border color for the data label. Defaults to `undefined`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.borderColor
     */
    borderColor?: ColorString;
    /**
     * (Highstock) The border radius in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.borderRadius
     */
    borderRadius?: number;
    /**
     * (Highstock) The border width in pixels for the data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.borderWidth
     */
    borderWidth?: number;
    /**
     * (Highstock) A class name for the data label. Particularly in styled mode,
     * this can be used to give each series' or point's data label unique
     * styling. In addition to this option, a default color class name is added
     * so that we can give the labels a contrast text shadow.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.className
     */
    className?: string;
    /**
     * (Highstock) The text color for the data labels. Defaults to `undefined`.
     * For certain series types, like column or map, the data labels can be
     * drawn inside the points. In this case the data label will be drawn with
     * maximum contrast by default. Additionally, it will be given a
     * `text-outline` style with the opposite color, to further increase the
     * contrast. This can be overridden by setting the `text-outline` style to
     * `none` in the `dataLabels.style` option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.color
     */
    color?: ColorString;
    /**
     * (Highstock) Whether to hide data labels that are outside the plot area.
     * By default, the data label is moved inside the plot area according to the
     * overflow option.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.crop
     */
    crop?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) Whether to defer displaying the data
     * labels until the initial series animation has finished.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.dataLabels.defer
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.defer
     * @see https://api.highcharts.com/gantt/plotOptions.ao.dataLabels.defer
     */
    defer?: boolean;
    /**
     * (Highstock) Enable or disable the data labels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.enabled
     */
    enabled?: boolean;
    /**
     * (Highstock) A declarative filter for which data labels to display. The
     * declarative filter is designed for use when callback functions are not
     * available, like when the chart options require a pure JSON structure or
     * for use with graphical editors. For programmatic control, use the
     * `formatter` instead, and return `undefined` to disable a single data
     * label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.filter
     */
    filter?: PlotAoDataLabelsFilterOptions;
    /**
     * (Highstock) A format string for the data label. Available variables are
     * the same as for `formatter`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.format
     */
    format?: string;
    /**
     * (Highstock) Callback JavaScript function to format the data label. Note
     * that if a `format` is defined, the format takes precedence and the
     * formatter is ignored. Available data are:
     *
     * - `this.percentage`: Stacked series and pies only. The point's percentage
     * of the total.
     *
     * - `this.point`: The point object. The point name, if defined, is
     * available through `this.point.name`.
     *
     * - `this.series`: The series object. The series name is available
     * through`this.series.name`.
     *
     * - `this.total`: Stacked series only. The total value at this point's x
     * value.
     *
     * - `this.x`: The x value.
     *
     * - `this.y`: The y value.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.formatter
     */
    formatter?: FormatterCallbackFunction<SeriesDataLabelsFormatterContextObject>;
    /**
     * (Highstock) For points with an extent, like columns or map areas, whether
     * to align the data label inside the box or to the actual value point.
     * Defaults to `false` in most cases, `true` in stacked columns.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.inside
     */
    inside?: boolean;
    /**
     * (Highstock) How to handle data labels that flow outside the plot area.
     * The default is `"justify"`, which aligns them inside the plot area. For
     * columns and bars, this means it will be moved inside the bar. To display
     * data labels outside the plot area, set `crop` to `false` and `overflow`
     * to `"allow"`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.overflow
     */
    overflow?: ("allow"|"justify");
    /**
     * (Highstock) When either the `borderWidth` or the `backgroundColor` is
     * set, this is the padding within the box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.padding
     */
    padding?: number;
    /**
     * (Highstock) Text rotation in degrees. Note that due to a more complex
     * structure, backgrounds, borders and padding will be lost on a rotated
     * data label.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.rotation
     */
    rotation?: number;
    /**
     * (Highstock) The shadow of the box. Works best with `borderWidth` or
     * `backgroundColor`. Since 2.3 the shadow can be an object configuration
     * containing `color`, `offsetX`, `offsetY`, `opacity` and `width`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.shadow
     */
    shadow?: (boolean|ShadowOptionsObject);
    /**
     * (Highstock) The name of a symbol to use for the border around the label.
     * Symbols are predefined functions on the Renderer object.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.shape
     */
    shape?: string;
    /**
     * (Highstock) Styles for the label. The default `color` setting is
     * `"contrast"`, which is a pseudo color that Highcharts picks up and
     * applies the maximum contrast to the underlying point item, for example
     * the bar in a bar chart.
     *
     * The `textOutline` is a pseudo property that applies an outline of the
     * given width with the given color, which by default is the maximum
     * contrast to the text. So a bright text color will result in a black text
     * outline for maximum readability on a mixed background. In some cases,
     * especially with grayscale text, the text outline doesn't work well, in
     * which cases it can be disabled by setting it to `"none"`. When `useHTML`
     * is true, the `textOutline` will not be picked up. In this, case, the same
     * effect can be acheived through the `text-shadow` CSS property.
     *
     * For some series types, where each point has an extent, like for example
     * tree maps, the data label may overflow the point. There are two
     * strategies for handling overflow. By default, the text will wrap to
     * multiple lines. The other strategy is to set `style.textOverflow` to
     * `ellipsis`, which will keep the text on one line plus it will break
     * inside long words.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.style
     */
    style?: CSSObject;
    /**
     * (Highstock) Whether to use HTML to render the labels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.useHTML
     */
    useHTML?: boolean;
    /**
     * (Highstock) The vertical alignment of a data label. Can be one of `top`,
     * `middle` or `bottom`. The default value depends on the data, for instance
     * in a column chart, the label is above positive values and below negative
     * values.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.verticalAlign
     */
    verticalAlign?: VerticalAlignType;
    /**
     * (Highstock) The x position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.x
     */
    x?: number;
    /**
     * (Highstock) The y position offset of the label relative to the point in
     * pixels.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.y
     */
    y?: number;
    /**
     * (Highstock) The Z index of the data labels. The default Z index puts it
     * above the series. Use a Z index of 2 to display it behind the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dataLabels.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Options for the drag handles.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle
 */
export interface PlotAoDragDropDragHandleOptions {
    /**
     * (Highstock) The class name of the drag handles. Defaults to
     * `highcharts-drag-handle`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.className
     */
    className?: string;
    /**
     * (Highstock) The fill color of the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) The mouse cursor to use for the drag handles. By default this
     * is intelligently switching between `ew-resize` and `ns-resize` depending
     * on the direction the point is being dragged.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.cursor
     */
    cursor?: string;
    /**
     * (Highstock) The line color of the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) The line width for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Function to define the SVG path to use for the drag handles.
     * Takes the point as argument. Should return an SVG path in array format.
     * The SVG path is automatically positioned on the point.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.pathFormatter
     */
    pathFormatter?: () => void;
    /**
     * (Highstock) The z index for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Style options for the guide box default state.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default
 */
export interface PlotAoDragDropGuideBoxDefaultOptions {
    /**
     * (Highstock) CSS class name of the guide box in this state. Defaults to
     * `highcharts-drag-box-default`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default.className
     */
    className?: string;
    /**
     * (Highstock) Guide box fill color.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default.color
     */
    color?: (ColorString|GradientColorObject|PatternObject);
    /**
     * (Highstock) Guide box cursor.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default.cursor
     */
    cursor?: string;
    /**
     * (Highstock) Color of the border around the guide box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default.lineColor
     */
    lineColor?: ColorString;
    /**
     * (Highstock) Width of the line around the guide box.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default.lineWidth
     */
    lineWidth?: number;
    /**
     * (Highstock) Guide box zIndex.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default.zIndex
     */
    zIndex?: number;
}
/**
 * (Highstock) Style options for the guide box. The guide box has one state by
 * default, the `default` state.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox
 */
export interface PlotAoDragDropGuideBoxOptions {
    /**
     * (Highstock) Style options for the guide box default state.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox.default
     */
    default?: PlotAoDragDropGuideBoxDefaultOptions;
}
/**
 * (Highstock) The draggable-points module allows points to be moved around or
 * modified in the chart. In addition to the options mentioned under the
 * `dragDrop` API structure, the module fires three events, point.dragStart,
 * point.drag and point.drop.
 *
 * It requires the `modules/draggable-points.js` file to be loaded.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop
 */
export interface PlotAoDragDropOptions {
    /**
     * (Highstock) Enable dragging in the X dimension.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.draggableX
     */
    draggableX?: boolean;
    /**
     * (Highstock) Enable dragging in the Y dimension. Note that this is not
     * supported for TreeGrid axes (the default axis type in Gantt charts).
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.draggableY
     */
    draggableY?: boolean;
    /**
     * (Highstock) Options for the drag handles.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragHandle
     */
    dragHandle?: PlotAoDragDropDragHandleOptions;
    /**
     * (Highstock) Set the maximum X value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragMaxX
     */
    dragMaxX?: number;
    /**
     * (Highstock) Set the maximum Y value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragMaxY
     */
    dragMaxY?: number;
    /**
     * (Highstock) Set the minimum X value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragMinX
     */
    dragMinX?: number;
    /**
     * (Highstock) Set the minimum Y value the points can be moved to.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragMinY
     */
    dragMinY?: number;
    /**
     * (Highstock) The X precision value to drag to for this series. Set to 0 to
     * disable. By default this is disabled, except for category axes, where the
     * default is 1.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragPrecisionX
     */
    dragPrecisionX?: number;
    /**
     * (Highstock) The Y precision value to drag to for this series. Set to 0 to
     * disable. By default this is disabled, except for category axes, where the
     * default is 1.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragPrecisionY
     */
    dragPrecisionY?: number;
    /**
     * (Highstock) The amount of pixels to drag the pointer before it counts as
     * a drag operation. This prevents drag/drop to fire when just clicking or
     * selecting points.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.dragSensitivity
     */
    dragSensitivity?: number;
    /**
     * (Highstock) Group the points by a property. Points with the same property
     * value will be grouped together when moving.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.groupBy
     */
    groupBy?: string;
    /**
     * (Highstock) Style options for the guide box. The guide box has one state
     * by default, the `default` state.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.guideBox
     */
    guideBox?: (PlotAoDragDropGuideBoxOptions|Dictionary<PlotSeriesDragDropGuideBoxDefaultOptions>);
    /**
     * (Highstock) Update points as they are dragged. If false, a guide box is
     * drawn to illustrate the new point size.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.dragDrop.liveRedraw
     */
    liveRedraw?: boolean;
}
/**
 * (Highstock) General event handlers for the series items. These event hooks
 * can also be attached to the series at run time using the
 * `Highcharts.addEvent` function.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.events
 */
export interface PlotAoEventsOptions {
    /**
     * (Highcharts, Highstock, Gantt) Fires after the series has finished its
     * initial animation, or in case animation is disabled, immediately as the
     * series is displayed.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.events.afterAnimate
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.afterAnimate
     * @see https://api.highcharts.com/gantt/plotOptions.ao.events.afterAnimate
     */
    afterAnimate?: SeriesAfterAnimateCallbackFunction;
    /**
     * (Highstock) Fires when the checkbox next to the series' name in the
     * legend is clicked. One parameter, `event`, is passed to the function. The
     * state of the checkbox is found by `event.checked`. The checked item is
     * found by `event.item`. Return `false` to prevent the default action which
     * is to toggle the select state of the series.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.checkboxClick
     */
    checkboxClick?: SeriesCheckboxClickCallbackFunction;
    /**
     * (Highstock) Fires when the series is clicked. One parameter, `event`, is
     * passed to the function, containing common event information.
     * Additionally, `event.point` holds a pointer to the nearest point on the
     * graph.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.click
     */
    click?: SeriesClickCallbackFunction;
    /**
     * (Highstock) Fires when the series is hidden after chart generation time,
     * either by clicking the legend item or by calling `.hide()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.hide
     */
    hide?: SeriesHideCallbackFunction;
    /**
     * (Highstock) Fires when the legend item belonging to the series is
     * clicked. One parameter, `event`, is passed to the function. The default
     * action is to toggle the visibility of the series. This can be prevented
     * by returning `false` or calling `event.preventDefault()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.legendItemClick
     */
    legendItemClick?: SeriesLegendItemClickCallbackFunction;
    /**
     * (Highstock) Fires when the mouse leaves the graph. One parameter,
     * `event`, is passed to the function, containing common event information.
     * If the stickyTracking option is true, `mouseOut` doesn't happen before
     * the mouse enters another graph or leaves the plot area.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.mouseOut
     */
    mouseOut?: SeriesMouseOutCallbackFunction;
    /**
     * (Highstock) Fires when the mouse enters the graph. One parameter,
     * `event`, is passed to the function, containing common event information.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.mouseOver
     */
    mouseOver?: SeriesMouseOverCallbackFunction;
    /**
     * (Highstock) Fires when the series is shown after chart generation time,
     * either by clicking the legend item or by calling `.show()`.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.events.show
     */
    show?: SeriesShowCallbackFunction;
}
/**
 * (Highcharts, Highstock, Gantt) Series labels are placed as close to the
 * series as possible in a natural way, seeking to avoid other series. The goal
 * of this feature is to make the chart more easily readable, like if a human
 * designer placed the labels in the optimal position.
 *
 * The series labels currently work with series types having a `graph` or an
 * `area`.
 *
 * Requires the `series-label.js` module.
 *
 * @see https://api.highcharts.com/highcharts/plotOptions.ao.label
 * @see https://api.highcharts.com/highstock/plotOptions.ao.label
 * @see https://api.highcharts.com/gantt/plotOptions.ao.label
 */
export interface PlotAoLabelOptions {
    /**
     * (Highcharts, Highstock, Gantt) An array of boxes to avoid when laying out
     * the labels. Each item has a `left`, `right`, `top` and `bottom` property.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.boxesToAvoid
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.boxesToAvoid
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.boxesToAvoid
     */
    boxesToAvoid?: Array<LabelIntersectBoxObject>;
    /**
     * (Highcharts, Highstock, Gantt) Allow labels to be placed distant to the
     * graph if necessary, and draw a connector line to the graph. Setting this
     * option to true may decrease the performance significantly, since the
     * algorithm with systematically search for open spaces in the whole plot
     * area. Visually, it may also result in a more cluttered chart, though more
     * of the series will be labeled.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.connectorAllowed
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.connectorAllowed
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.connectorAllowed
     */
    connectorAllowed?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) If the label is closer than this to a
     * neighbour graph, draw a connector.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.connectorNeighbourDistance
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.connectorNeighbourDistance
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.connectorNeighbourDistance
     */
    connectorNeighbourDistance?: number;
    /**
     * (Highcharts, Highstock, Gantt) Enable the series label per series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.enabled
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.enabled
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.enabled
     */
    enabled?: boolean;
    /**
     * (Highcharts, Highstock, Gantt) For area-like series, allow the font size
     * to vary so that small areas get a smaller font size. The default applies
     * this effect to area-like series but not line-like series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.maxFontSize
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.maxFontSize
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.maxFontSize
     */
    maxFontSize?: (number|null);
    /**
     * (Highcharts, Highstock, Gantt) For area-like series, allow the font size
     * to vary so that small areas get a smaller font size. The default applies
     * this effect to area-like series but not line-like series.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.minFontSize
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.minFontSize
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.minFontSize
     */
    minFontSize?: (number|null);
    /**
     * (Highcharts, Highstock, Gantt) Draw the label on the area of an area
     * series. By default it is drawn on the area. Set it to `false` to draw it
     * next to the graph instead.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.onArea
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.onArea
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.onArea
     */
    onArea?: (boolean|null);
    /**
     * (Highcharts, Highstock, Gantt) Styles for the series label. The color
     * defaults to the series color, or a contrast color if `onArea`.
     *
     * @see https://api.highcharts.com/highcharts/plotOptions.ao.label.style
     * @see https://api.highcharts.com/highstock/plotOptions.ao.label.style
     * @see https://api.highcharts.com/gantt/plotOptions.ao.label.style
     */
    style?: CSSObject;
}
/**
 * (Highstock) The line marks the last price from all points.
 *
 * @see https://api.highcharts.com/highstock/plotOptions.ao.lastPrice
 */
export interface PlotAoLastPriceOptions {
    /**
     * (Highstock) The color of the line of last price.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.lastPrice.color
     */
    color?: string;
    /**
     * (Highstock) Enable or disable the indicator.
     *
     * @see https://api.highcharts.com/highstock/plotOptions.ao.lastPrice.enabled
     */
    enabled?: boolean;
}
export interface PlotAoLastVisiblePriceLabelOptions {
    /**
     * (Highstock) Enable or disable the label.
     *
     * @see https://api.highcharts.co