/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.utilities.test.net;

import java.io.IOException;
import java.net.Inet4Address;
import java.net.Inet6Address;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.charset.Charset;
import java.text.StringCharacterIterator;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.utilities.exec.Shell;
import org.terracotta.utilities.test.runtime.Os;

public class NetStat {
    private static final Logger LOGGER = LoggerFactory.getLogger(NetStat.class);
    private static final String[] POWERSHELL_NETSTAT_COMMAND = new String[]{"powershell.exe", "-NoLogo", "-NoProfile", "-NonInteractive", "-Command", "&{$ErrorActionPreference = 'Stop';  $processes = Get-WmiObject -Class \"Win32_Process\" -Namespace \"ROOT\\CIMV2\" | Select-Object -Property ProcessId, Caption, CommandLine;  Get-NetTCPConnection | Select-Object -Property OwningProcess, LocalAddress, LocalPort, RemoteAddress, RemotePort, State | Foreach-Object {  $connection = $_;  $process = $processes | Where-Object {$_.ProcessId -eq $connection.OwningProcess} | Select-Object -Unique;  if ($process -ne $null) {    $connection | Add-Member -NotePropertyMembers @{ProcessCaption = $process.Caption; ProcessCommandLine = $process.CommandLine}  };  $connection;}| ConvertTo-Csv -NoTypeInformation}"};
    private static final String[] SUDO_PREFIX = new String[]{"sudo", "--non-interactive", "--"};
    private static final String[] LSOF_COMMAND = new String[]{"lsof", "-nP", "-iTCP", "-F", "0pPRgLnTtc", "+c0", "-w"};
    private static final String[] NETTOP_COMMAND = new String[]{"nettop", "-L1", "-m", "tcp", "-n", "-J", "state"};
    private static final Pattern TCP4_ENDPOINT_PATTERN = Pattern.compile("([^:]+):(\\*|\\d+)");
    private static final Pattern TCP6_ENDPOINT_PATTERN = Pattern.compile("(?:([^%.]+(?:%[^.]+)?)|(?:\\*))\\.(\\*|\\d+)");

    private NetStat() {
    }

    public static List<BusyPort> info() {
        try {
            return Platform.getPlatform().netstat();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public static void main(String[] args) {
        for (BusyPort port : NetStat.info()) {
            System.out.println(port.toString("\t"));
        }
    }

    private static BusyPort parseWindowsCsv(String line) {
        Objects.requireNonNull(line, "line");
        if (line.isEmpty()) {
            throw new IllegalArgumentException("line cannot be empty");
        }
        ArrayList<String> fields = new ArrayList<String>();
        StringBuilder sb = new StringBuilder();
        int quoteCount = 0;
        StringCharacterIterator iterator = new StringCharacterIterator(line);
        char c = iterator.first();
        while (c != '\uffff') {
            switch (c) {
                case '\"': {
                    if (quoteCount++ == 0 || quoteCount % 2 == 0) break;
                    sb.append('\"');
                    break;
                }
                case ',': {
                    if (quoteCount % 2 == 0) {
                        fields.add(sb.toString());
                        sb.setLength(0);
                        quoteCount = 0;
                        break;
                    }
                    sb.append(',');
                    break;
                }
                default: {
                    sb.append(c);
                }
            }
            c = iterator.next();
        }
        if (quoteCount % 2 != 0) {
            throw new IllegalArgumentException("Line ends with unbalanced quote at index " + iterator.getIndex() + " - " + line);
        }
        if (sb.length() > 0) {
            fields.add(sb.toString());
        }
        BusyPort.Builder builder = BusyPort.builder();
        try {
            builder.processId((String)fields.get(0));
            builder.localEndpoint(BusyPort.IPVersion.IPV4, (String)fields.get(1), (String)fields.get(2));
            builder.remoteEndpoint(BusyPort.IPVersion.IPV4, (String)fields.get(3), (String)fields.get(4));
            builder.state(BusyPort.TcpState.fromMicrosoftString((String)fields.get(5)));
            builder.shortCommand((String)fields.get(6));
            if (fields.size() >= 8) {
                builder.commandLine((String)fields.get(7));
            }
        }
        catch (RuntimeException e) {
            throw new IllegalArgumentException("Error in line - " + line, e);
        }
        return builder.build();
    }

    private static List<BusyPort> parseLsof(Stream<String> lines) {
        ArrayList<BusyPort> busyPorts = new ArrayList<BusyPort>();
        Iterator iterator = lines.iterator();
        BusyPort.Builder builder = null;
        while (iterator.hasNext()) {
            String line = (String)iterator.next();
            String[] fields = line.split("\u0000");
            if (line.charAt(0) == 'p') {
                try {
                    builder = BusyPort.builder();
                    block14: for (String field : fields) {
                        char fieldId = field.charAt(0);
                        String fieldValue = field.substring(1);
                        switch (fieldId) {
                            case 'p': {
                                builder.processId(fieldValue);
                                continue block14;
                            }
                            case 'c': {
                                builder.shortCommand(fieldValue);
                                continue block14;
                            }
                        }
                    }
                    continue;
                }
                catch (Exception e) {
                    throw new IllegalStateException("Expecting process ('p') line; found '" + line + "'", e);
                }
            }
            if (builder == null) {
                throw new IllegalStateException("Expecting process ('p') line; found '" + line + "'");
            }
            if (line.charAt(0) == 'f') {
                try {
                    BusyPort.IPVersion ipVersion = null;
                    block15: for (String field : fields) {
                        char fieldId = field.charAt(0);
                        String fieldValue = field.substring(1);
                        switch (fieldId) {
                            case 't': {
                                ipVersion = BusyPort.IPVersion.fromId(fieldValue);
                                continue block15;
                            }
                            case 'n': {
                                InetEndpointPair endpointPair = NetStat.parseLsofEndpointPair(ipVersion, fieldValue);
                                builder.localEndpoint(endpointPair.getLocalEndpoint());
                                builder.remoteEndpoint(endpointPair.getRemoteEndpoint());
                                continue block15;
                            }
                            case 'T': {
                                String infoClass;
                                int p = fieldValue.indexOf(61);
                                if (p <= 0 || !"ST".equals(infoClass = fieldValue.substring(0, p))) continue block15;
                                BusyPort.TcpState state = BusyPort.TcpState.fromLsofString(fieldValue.substring(1 + p));
                                builder.state(state);
                                continue block15;
                            }
                        }
                    }
                    busyPorts.add(builder.build());
                    builder.resetTcp();
                    continue;
                }
                catch (Exception e) {
                    throw new IllegalStateException("Expecting fd ('f') line; found '" + line + "'", e);
                }
            }
            throw new IllegalStateException("Expecting fd ('f') line; found '" + line + "'");
        }
        return busyPorts;
    }

    private static InetEndpointPair parseLsofEndpointPair(BusyPort.IPVersion ipVersion, String endpointPair) {
        if (Objects.requireNonNull(endpointPair, "endpointPair").isEmpty()) {
            throw new IllegalArgumentException("endpointPair is empty");
        }
        String[] endpoints = endpointPair.split(Pattern.quote("->"));
        InetSocketAddress localEndpoint = NetStat.parseLsofEndpoint(ipVersion, endpoints[0]);
        if (endpoints.length == 1) {
            return new InetEndpointPair(localEndpoint, ipVersion.getInetSocketAddress("*", "*"));
        }
        return new InetEndpointPair(localEndpoint, NetStat.parseLsofEndpoint(ipVersion, endpoints[1]));
    }

    private static InetSocketAddress parseLsofEndpoint(BusyPort.IPVersion ipVersion, String endpoint) {
        int p = endpoint.lastIndexOf(58);
        return ipVersion.getInetSocketAddress(endpoint.substring(0, p), endpoint.substring(1 + p));
    }

    private static List<BusyPort> parseNetTop(Stream<String> lines) {
        ArrayList<BusyPort> busyPorts = new ArrayList<BusyPort>();
        Iterator iterator = lines.skip(1L).iterator();
        BusyPort.Builder builder = null;
        while (iterator.hasNext()) {
            String line = (String)iterator.next();
            if (line.startsWith("tcp4 ")) {
                if (builder == null) {
                    throw new IllegalStateException("Expecting process identifier line; found '" + line + "'");
                }
                NetStat.processNettopEndpoints(BusyPort.IPVersion.IPV4, builder, line, TCP4_ENDPOINT_PATTERN, Function.identity());
                busyPorts.add(builder.build());
                builder.resetTcp();
                continue;
            }
            if (line.startsWith("tcp6 ")) {
                if (builder == null) {
                    throw new IllegalStateException("Expecting process identifier line; found '" + line + "'");
                }
                NetStat.processNettopEndpoints(BusyPort.IPVersion.IPV6, builder, line, TCP6_ENDPOINT_PATTERN, host -> {
                    if (host.indexOf(58) != -1) {
                        host = '[' + host + ']';
                    }
                    return host;
                });
                busyPorts.add(builder.build());
                builder.resetTcp();
                continue;
            }
            if (builder == null) {
                builder = BusyPort.builder();
            }
            if (line.endsWith(",,")) {
                line = line.substring(0, line.length() - 2);
                int p = line.lastIndexOf(46);
                builder.processId(line.substring(1 + p));
                builder.shortCommand(line.substring(0, p));
                continue;
            }
            throw new IllegalStateException("Expecting process identifier line; found '" + line + "'");
        }
        return busyPorts;
    }

    private static void processNettopEndpoints(BusyPort.IPVersion ipVersion, BusyPort.Builder builder, String line, Pattern endpointPattern, Function<String, String> endpointHostMapper) {
        String[] fields = line.substring(5).split(",");
        if (fields.length != 2) {
            throw new IllegalStateException("Expecting tcp port line; found '" + line + "'");
        }
        String[] endpoints = fields[0].split(Pattern.quote("<->"));
        if (endpoints.length != 2) {
            throw new IllegalStateException("Expecting tcp port line; found '" + line + "'");
        }
        Matcher matcher = endpointPattern.matcher(endpoints[0]);
        if (!matcher.matches()) {
            throw new IllegalStateException("Expecting tcp port line; found '" + line + "'");
        }
        builder.localEndpoint(ipVersion, matcher.group(1), matcher.group(2));
        matcher = endpointPattern.matcher(endpoints[1]);
        if (!matcher.matches()) {
            throw new IllegalStateException("Expecting tcp port line; found '" + line + "'");
        }
        builder.remoteEndpoint(ipVersion, endpointHostMapper.apply(matcher.group(1)), matcher.group(2));
        builder.state(BusyPort.TcpState.fromNettopString(fields[1]));
    }

    public static final class HostExecutionException
    extends IOException {
        private static final long serialVersionUID = 3134255439257149986L;

        public HostExecutionException(String message) {
            super(message);
        }

        public HostExecutionException(String message, Throwable cause) {
            super(message, cause);
        }
    }

    public static class BusyPort {
        private final long processId;
        private final InetSocketAddress localEndpoint;
        private final InetSocketAddress remoteEndpoint;
        private final TcpState state;
        private final String shortCommand;
        private final String commandLine;

        private BusyPort(long processId, InetSocketAddress localEndpoint, InetSocketAddress remoteEndpoint, TcpState state, String shortCommand, String commandLine) {
            this.processId = processId;
            this.localEndpoint = localEndpoint;
            this.remoteEndpoint = remoteEndpoint;
            this.state = state;
            this.shortCommand = shortCommand;
            this.commandLine = commandLine;
        }

        public long processId() {
            return this.processId;
        }

        public InetSocketAddress localEndpoint() {
            return this.localEndpoint;
        }

        public InetSocketAddress remoteEndpoint() {
            return this.remoteEndpoint;
        }

        public TcpState state() {
            return this.state;
        }

        public String shortCommand() {
            return this.shortCommand;
        }

        public String commandLine() {
            return this.commandLine;
        }

        private static Builder builder() {
            return new Builder();
        }

        private static Builder builder(BusyPort busyPort) {
            return new Builder(busyPort);
        }

        public String toString(String fieldSeparator) {
            if (fieldSeparator == null || fieldSeparator.isEmpty()) {
                fieldSeparator = " ";
            }
            StringBuilder sb = new StringBuilder();
            sb.append(this.processId);
            sb.append(fieldSeparator).append(this.localEndpoint);
            sb.append(fieldSeparator);
            if (this.remoteEndpoint == null) {
                sb.append("*.*");
            } else {
                sb.append(this.remoteEndpoint);
            }
            sb.append(fieldSeparator).append((Object)this.state);
            sb.append(fieldSeparator);
            if (this.commandLine == null || this.commandLine.isEmpty()) {
                sb.append(this.shortCommand);
            } else {
                sb.append(this.commandLine);
            }
            return sb.toString();
        }

        public String toString() {
            return "BusyPort{processId=" + this.processId + ", localEndpoint=" + this.localEndpoint + ", remoteEndpoint=" + this.remoteEndpoint + ", state='" + (Object)((Object)this.state) + '\'' + ", shortCommand='" + this.shortCommand + '\'' + ", commandLine='" + this.commandLine + '\'' + '}';
        }

        public static enum TcpState {
            UNKNOWN("", 0, "", ""),
            CLOSED("Closed", 1, "Closed", "CLOSED"),
            LISTEN("Listen", 2, "Listen", "LISTEN"),
            SYN_SENT("SynSent", 3, "SynSent", "SYN_SENT"),
            SYN_RECEIVED("SynReceived", 4, "SynReceived", "SYN_RECV", "SYN_RCVD"),
            ESTABLISHED("Established", 5, "Established", "ESTABLISHED"),
            FIN_WAIT_1("FinWait1", 6, "FinWait1", "FIN_WAIT_1"),
            FIN_WAIT_2("FinWait2", 7, "FinWait2", "FIN_WAIT_2"),
            CLOSE_WAIT("CloseWait", 8, "CloseWait", "CLOSE_WAIT"),
            CLOSING("Closing", 9, "Closing", "CLOSING"),
            LAST_ACK("LastAck", 10, "LastAck", "LAST_ACK"),
            TIME_WAIT("TimeWait", 11, "TimeWait", "TIME_WAIT"),
            DELETE_TCB("DeleteTCB", 12, "", ""),
            BOUND("Bound", -1, "", "BOUND"),
            CLOSE("", -1, "", "CLOSE"),
            IDLE("", -1, "", "IDLE");

            private final String msStateString;
            private final int msStateNumber;
            private final String nettopStateString;
            private final List<String> lsofStateStrings;

            private TcpState(String msStateString, int msStateNumber, String nettopStateString, String ... lsofStateStrings) {
                this.msStateString = msStateString;
                this.msStateNumber = msStateNumber;
                this.nettopStateString = nettopStateString;
                this.lsofStateStrings = new ArrayList<String>(Arrays.asList(lsofStateStrings));
            }

            public static TcpState fromNettopString(String nettopStateString) {
                for (TcpState state : TcpState.values()) {
                    if (!state.nettopStateString.equals(nettopStateString)) continue;
                    return state;
                }
                throw new EnumConstantNotPresentException(TcpState.class, nettopStateString);
            }

            public static TcpState fromLsofString(String lsofStateString) {
                for (TcpState state : TcpState.values()) {
                    if (!state.lsofStateStrings.contains(lsofStateString)) continue;
                    return state;
                }
                throw new EnumConstantNotPresentException(TcpState.class, lsofStateString);
            }

            public static TcpState fromMicrosoftString(String msStateString) {
                for (TcpState state : TcpState.values()) {
                    if (!state.msStateString.equalsIgnoreCase(msStateString)) continue;
                    return state;
                }
                throw new EnumConstantNotPresentException(TcpState.class, msStateString);
            }

            public static TcpState fromMicrosoftNumber(int msStateNumber) {
                for (TcpState state : TcpState.values()) {
                    if (state.msStateNumber != msStateNumber) continue;
                    return state;
                }
                throw new EnumConstantNotPresentException(TcpState.class, Integer.toString(msStateNumber));
            }
        }

        public static enum IPVersion {
            IPV4((Class)Inet4Address.class, 4, new String[]{"IPv4", "tcp4"}){

                @Override
                public byte[] loopback() {
                    return new byte[]{127, 0, 0, 1};
                }
            }
            ,
            IPV6((Class)Inet6Address.class, 16, new String[]{"IPv6", "tcp6"}){

                @Override
                public byte[] loopback() {
                    byte[] loopback = new byte[16];
                    loopback[15] = 1;
                    return loopback;
                }
            };

            private final int addressBytes;
            private final List<String> versionIds;

            private IPVersion(Class<? extends InetAddress> addressClass, int addressBytes, String ... versionIds) {
                this.addressBytes = addressBytes;
                this.versionIds = Arrays.asList(versionIds);
            }

            public byte[] anyLocal() {
                return new byte[this.addressBytes];
            }

            public abstract byte[] loopback();

            public InetSocketAddress getInetSocketAddress(String address, String port) {
                Objects.requireNonNull(address, "address");
                if (address.isEmpty()) {
                    throw new IllegalArgumentException("address cannot be empty");
                }
                Objects.requireNonNull(port, "port");
                if (port.isEmpty()) {
                    throw new IllegalArgumentException("port cannot be empty");
                }
                int portNumber = "*".equals(port) ? 0 : Integer.parseInt(port);
                try {
                    InetAddress inetAddress = "*".equals(address) ? InetAddress.getByAddress(this.anyLocal()) : InetAddress.getByName(address);
                    return new InetSocketAddress(inetAddress, portNumber);
                }
                catch (UnknownHostException e) {
                    throw new IllegalArgumentException(e);
                }
            }

            public static IPVersion fromId(String versionId) {
                for (IPVersion ipVersion : IPVersion.values()) {
                    if (!ipVersion.versionIds.contains(versionId)) continue;
                    return ipVersion;
                }
                throw new EnumConstantNotPresentException(IPVersion.class, versionId);
            }
        }

        private static class Builder {
            private long processId;
            private InetSocketAddress localEndpoint;
            private InetSocketAddress remoteEndpoint;
            private TcpState state;
            private String shortCommand;
            private String commandLine;

            Builder() {
            }

            Builder(BusyPort busyPort) {
                this.processId = busyPort.processId;
                this.localEndpoint = busyPort.localEndpoint;
                this.remoteEndpoint = busyPort.remoteEndpoint;
                this.state = busyPort.state;
                this.shortCommand = busyPort.shortCommand;
                this.commandLine = busyPort.commandLine;
            }

            BusyPort build() {
                return new BusyPort(this.processId, this.localEndpoint, this.remoteEndpoint, this.state, this.shortCommand, this.commandLine);
            }

            Builder resetTcp() {
                this.localEndpoint = null;
                this.remoteEndpoint = null;
                this.state = null;
                return this;
            }

            Builder processId(String processId) {
                Objects.requireNonNull(processId, "processId");
                if (processId.isEmpty()) {
                    throw new IllegalArgumentException("processId cannot be empty");
                }
                if (processId.charAt(0) == '-' || processId.charAt(0) == '+') {
                    throw new IllegalArgumentException("processId '" + processId + "' is poorly formed");
                }
                try {
                    this.processId = Long.parseLong(processId);
                }
                catch (NumberFormatException e) {
                    throw new IllegalArgumentException("processId '" + processId + "' is poorly formed", e);
                }
                return this;
            }

            Builder shortCommand(String shortCommand) {
                this.shortCommand = shortCommand;
                return this;
            }

            Builder commandLine(String commandLine) {
                this.commandLine = commandLine;
                return this;
            }

            Builder state(TcpState state) {
                this.state = state;
                return this;
            }

            Builder localEndpoint(InetSocketAddress endpoint) {
                this.localEndpoint = endpoint;
                return this;
            }

            Builder localEndpoint(IPVersion ipVersion, String address, String port) {
                this.localEndpoint = this.processEndpoint(ipVersion, address, port);
                return this;
            }

            Builder remoteEndpoint(InetSocketAddress endpoint) {
                this.remoteEndpoint = endpoint;
                return this;
            }

            Builder remoteEndpoint(IPVersion ipVersion, String address, String port) {
                this.remoteEndpoint = this.processEndpoint(ipVersion, address, port);
                return this;
            }

            private InetSocketAddress processEndpoint(IPVersion ipVersion, String address, String port) {
                return Objects.requireNonNull(ipVersion, "ipVersion").getInetSocketAddress(address, port);
            }
        }
    }

    private static class InetEndpointPair {
        private final InetSocketAddress localEndpoint;
        private final InetSocketAddress remoteEndpoint;

        private InetEndpointPair(InetSocketAddress localEndpoint, InetSocketAddress remoteEndpoint) {
            this.localEndpoint = localEndpoint;
            this.remoteEndpoint = remoteEndpoint;
        }

        private InetSocketAddress getLocalEndpoint() {
            return this.localEndpoint;
        }

        private InetSocketAddress getRemoteEndpoint() {
            return this.remoteEndpoint;
        }
    }

    private static enum Platform {
        WINDOWS("win32"){

            @Override
            public List<BusyPort> netstat() throws HostExecutionException {
                Function<Stream, List> conversion = stream -> stream.skip(1L).map(x$0 -> NetStat.parseWindowsCsv(x$0)).collect(Collectors.toList());
                return Platform.runCommand(POWERSHELL_NETSTAT_COMMAND, conversion);
            }
        }
        ,
        MAC("mac"){

            @Override
            public List<BusyPort> netstat() throws HostExecutionException {
                List busyPorts = Platform.runCommand(NETTOP_COMMAND, x$0 -> NetStat.parseNetTop(x$0));
                return Platform.mergeCommands(busyPorts);
            }
        }
        ,
        LINUX("linux"){

            @Override
            public List<BusyPort> netstat() throws HostExecutionException {
                List busyPorts;
                String[] sudoLsof = Arrays.copyOf(SUDO_PREFIX, SUDO_PREFIX.length + LSOF_COMMAND.length);
                System.arraycopy(LSOF_COMMAND, 0, sudoLsof, SUDO_PREFIX.length, LSOF_COMMAND.length);
                try {
                    busyPorts = Platform.runCommand(sudoLsof, x$0 -> NetStat.parseLsof(x$0));
                }
                catch (HostExecutionException e) {
                    String message = "\n\n********************************************************************************\nObtaining a full set of in-use TCP ports requires use of 'sudo' to execute\n'lsof'; add sudoers permissions to allow this.  For example, add a line like the\nfollowing to the bottom of the '/etc/sudoers' file (using 'sudo visudo'):\n    %sudo   ALL=NOPASSWD: /usr/bin/lsof\nEnsure an appropriate group or username is used in place of '%sudo' and the\ncorrect path to 'lsof' is used.\n********************************************************************************\n";
                    LOGGER.warn(message, (Throwable)e);
                    try {
                        busyPorts = Platform.runCommand(LSOF_COMMAND, x$0 -> NetStat.parseLsof(x$0));
                    }
                    catch (Exception ex) {
                        HostExecutionException hostExecutionException = new HostExecutionException("Failed to obtain active TCP ports using 'lsof'" + message, ex);
                        hostExecutionException.addSuppressed(e);
                        throw hostExecutionException;
                    }
                }
                return Platform.mergeCommands(busyPorts);
            }
        };

        private final String osPlatform;
        private static final String[] PS_COMMAND;
        private static final Pattern PS_PATTERN;

        private Platform(String osPlatform) {
            this.osPlatform = osPlatform;
        }

        public abstract List<BusyPort> netstat() throws HostExecutionException;

        public static Platform getPlatform() throws EnumConstantNotPresentException {
            String platform = Os.platform();
            for (Platform value : Platform.values()) {
                if (!value.osPlatform.equals(platform)) continue;
                return value;
            }
            throw new EnumConstantNotPresentException(Platform.class, platform);
        }

        private static List<BusyPort> mergeCommands(List<BusyPort> busyPorts) throws HostExecutionException {
            return Platform.runCommand(PS_COMMAND, psStream -> {
                Map<Long, String> processMap = psStream.skip(1L).map(line -> {
                    Matcher matcher = PS_PATTERN.matcher((CharSequence)line);
                    if (!matcher.matches()) {
                        throw new IllegalStateException("Failed to process process line - '" + line + "'");
                    }
                    return matcher;
                }).collect(Collectors.toMap(m -> Long.parseLong(m.group(1)), m -> m.group(3)));
                return busyPorts.stream().map(p -> {
                    String command = (String)processMap.get(p.processId());
                    return command == null ? p : BusyPort.builder((BusyPort)p).commandLine(command).build();
                }).collect(Collectors.toList());
            });
        }

        private static <T> List<T> runCommand(String[] command, Function<Stream<String>, List<T>> conversion) throws HostExecutionException {
            Shell.Result result;
            try {
                result = Shell.execute((Charset)Shell.Encoding.CHARSET, (String[])command);
            }
            catch (IOException e) {
                throw new HostExecutionException("Failed to run command: " + Arrays.toString(command), e);
            }
            if (result.exitCode() == 0) {
                if (LOGGER.isTraceEnabled()) {
                    LOGGER.trace("Command complete {}; rc=0{}", (Object)Arrays.toString(command), (Object)("\n    " + String.join((CharSequence)"\n    ", result.lines())));
                }
                return conversion.apply(result.lines().stream());
            }
            String detail = "\n    " + String.join((CharSequence)"\n    ", result.lines());
            throw new HostExecutionException("Failed to run command: " + Arrays.toString(command) + "; rc=" + result.exitCode() + detail);
        }

        static {
            PS_COMMAND = new String[]{"ps", "-ax", "-opid,user,command"};
            PS_PATTERN = Pattern.compile("\\s*(\\S+)\\s(\\S+)\\s+(.+)");
        }
    }
}

