/*
 * Copyright Terracotta, Inc.
 * Copyright IBM Corp. 2024, 2025
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.terracotta.management.model.message;

import org.terracotta.management.model.context.Contextual;
import org.terracotta.management.sequence.Sequence;

import java.io.Serializable;
import java.util.Arrays;
import java.util.List;

/**
 * @author Mathieu Carbou
 */
public class DefaultMessage implements Message, Serializable {

  private static final long serialVersionUID = 1;

  private final Contextual[] data;
  private final Sequence sequence;
  private final String messageType;

  public DefaultMessage(Sequence sequence, String messageType, Contextual... data) {
    this.sequence = sequence;
    this.messageType = messageType;
    this.data = data;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <T extends Contextual> List<T> unwrap(Class<T> type) {
    return (List<T>) Arrays.asList(data);
  }

  @Override
  public Sequence getSequence() {
    return sequence;
  }

  @Override
  public long getTimestamp() {
    return sequence.getTimestamp();
  }

  @Override
  public String getType() {
    return messageType;
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() +
        "{" +
        "type=" + messageType +
        ", sequence=" + sequence +
        ", data=" + (data == null ? null : Arrays.deepToString(data)) +
        '}';
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    DefaultMessage that = (DefaultMessage) o;
    if (!sequence.equals(that.sequence)) return false;
    if (data.getClass() != that.data.getClass()) return false;
    if (!Arrays.equals(data, that.data)) return false;
    return messageType.equals(that.messageType);
  }

  @Override
  public int hashCode() {
    int result = Arrays.hashCode(data);
    result = 31 * result + sequence.hashCode();
    result = 31 * result + messageType.hashCode();
    return result;
  }

}
