/**
 * 
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.masterworker;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * Master interface for submitting work objects to registered routes.
 * 
 * @param <T> Type of the work object to submit. 
 */
public interface Master<T> {

    /**
     * Set up and start the master.<br>
     * Works can be submitted only after the master instance has been actually started.
     */
    public void start();

    /**
     * Register a route.<br>
     * This method does nothing if called after a {@link #shutdown()} request.
     * 
     * @param routingID The route identifier.
     */
    public void register(String routingID);
    
    /**
     * Unregister a route.<br>
     * This method does nothing if called after a {@link #shutdown()} request.
     * 
     * @param routingID The route identifier.
     */
    public void unregister(String routingID);

    /**
     * Submit a work object.<br>
     * This method does nothing if called after a {@link #shutdown()} request.
     * 
     * @param work The work to submit.
     * @return The submitted work, or null if called after a {@link #shutdown()} request.
     */
    public T submit(T work);

    /**
     * Shutdown the master.<br>
     * Please note that the master will try to wait for the completion of already submitted works.
     */
    public void shutdown();

    /**
     * Shutdown the master without waiting for the completion of already submitted works.
     * 
     * @return The list of works still waiting for execution when the shutdown was called.
     */
    public List<T> shutdownNow();
    
    /**
     * Check if the master is shutdown.
     * 
     * @return True if shutdown, false otherwise.
     */
    public boolean isShutdown();

    /**
     * Check if the master is shutdown and all submitted works have been terminated.
     * 
     * @return True if terminated, false otherwise.
     */
    public boolean isTerminated();
    
    /**
     * Wait for the master to terminate after a {@link #shutdown()} call.<br>
     * This method does nothing if not invoked after a {@link #shutdown()} call.<br>
     * Please note that if invoked after a {@link #shutdownNow()} call, this method
     * will never return true because the <code>shutdownNow</code> call abruptly
     * shutdowns the master without waiting for work completion.
     * 
     * @param timeout The amount of time to wait.
     * @param unit The time unit.
     * @return True if actually terminated on time, false otherwise.
     * @throws InterruptedException If interrupted while waiting.
     */
    public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException;
}
