/**
 *
 * All content copyright (c) 2003-2008 Terracotta, Inc.,
 * except as may otherwise be noted in a separate copyright notice.
 * All rights reserved.
 *
 */
package org.terracotta.executor;

import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.masterworker.AbstractWorker;
import org.terracotta.masterworker.WorkMessage;
import org.terracotta.message.topology.Topology;
import org.terracotta.message.topology.TopologyManager;

/**
 * Worker implementation for executing tasks submitted by a {@link DistributedExecutorService}.<br>
 * <code>DistributedExecutorService</code> and <code>DistributedWorkerService</code> instances must share the same
 * {@link org.terracotta.message.topology.Topology} in order to collaborate each other.
 */
public class DistributedWorkerService extends AbstractWorker<Runnable> {

    private transient static final Logger logger = LoggerFactory.getLogger(DistributedWorkerService.class);

    /**
     * Create a DistributedWorkerService which executes works sequentially in another worker thread.
     *
     * @param topologyName The name of the topology holding the pipe to listen to:
     * the actual {@link org.terracotta.message.topology.Topology} implementation is left to the worker.
     */
    public DistributedWorkerService(final String topologyName) {
        this(null, topologyName, null, Executors.newSingleThreadExecutor());
    }

    /**
     * Create a DistributedWorkerService which executes works sequentially in another worker thread.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to:
     * the actual {@link org.terracotta.message.topology.Topology} implementation is left to the worker.
     */
    public DistributedWorkerService(final String workerPrefix, final String topologyName) {
        this(workerPrefix, topologyName, null, Executors.newSingleThreadExecutor());
    }

    /**
     * Create a DistributedWorkerService which executes works sequentially in another worker thread.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to.
     * @param topologyFactory The factory to use for the topology implementation.
     */
    public DistributedWorkerService(final String workerPrefix, final String topologyName, final Topology.Factory topologyFactory) {
        this(workerPrefix, topologyName, topologyFactory, Executors.newSingleThreadExecutor());
    }

    /**
     * Create a DistributedWorkerService which executes works as specified by the passed {@link java.util.concurrent.ExecutorService}.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to: the actual {@link org.terracotta.message.topology.Topology}
     * implementation is left to the worker.
     * @param executorService The {@link java.util.concurrent.ExecutorService} used for actually executing works.
     */
    public DistributedWorkerService(final String workerPrefix, final String topologyName, final ExecutorService executorService) {
        this(workerPrefix, topologyName, null, executorService);
    }

    /**
     * Create a DistributedWorkerService which executes works as specified by the passed {@link java.util.concurrent.ExecutorService}.
     *
     * @param workerPrefix An optional string value to use as a prefix for this worker id; please note that
     * it is possible to assign the same prefix to more workers.
     * @param topologyName The name of the topology holding the pipe to listen to.
     * @param topologyFactory The factory to use for the topology implementation.
     * @param executorService The {@link java.util.concurrent.ExecutorService} used for actually executing works.
     */
    public DistributedWorkerService(final String workerPrefix, final String topologyName, final Topology.Factory topologyFactory, final ExecutorService executorService) {
        super(workerPrefix, TopologyManager.getInstance().<WorkMessage<Runnable>, String>getOrCreateTopology(topologyName, topologyFactory), executorService);
    }

    @Override
    protected void doExecute(final WorkMessage<Runnable> workMessage) {
        Runnable work = workMessage.getWorkObject();
        try {
            work.run();
        } catch (Exception ex) {
            logger.error("Exception while running work!", ex);
        } finally {
            doReply(workMessage);
        }
    }
}
