package org.terracotta.cache.value;

import org.terracotta.cache.TimestampedValue;
import org.terracotta.cache.CacheConfig;
import org.terracotta.cache.CacheConfigFactory;
import org.terracotta.cache.evictor.CapacityEvictionPolicyData;

import com.tc.object.bytecode.ManagerUtil;
import com.tc.object.bytecode.Manager;

/**
 * @author Alex Snaps
 */
public abstract class AbstractStatelessTimestampedValue<V> implements TimestampedValue<V> {
  /**
   * Indicates the TTI or TTL is UNUSED (0).
   */
  protected static final int UNUSED = 0;

  protected abstract CapacityEvictionPolicyData fastGetCapacityEvictionPolicyData();

  public abstract V getValue();

  /**
   * Mark this timestamp as being used and reset the idle timer (if in use). Updates <tt>lastAccessedTime</tt> of this
   * entry to <tt>usedAtTime</tt>
   * <p />
   * Special case when tti=1; this will update the <tt>lastAccessedTime</tt> at <tt>usedAtTime + 1</tt> instead. This is
   * because as time is not continuous, the entry will expire in the next second even when the entry is accessed in the
   * last second continuously. This is a result of time not being continuous. Effectively specifying tti=1 is equivalent
   * to tti=2
   *
   * @param usedAtTime Mark used at this time
   */
  public final void markUsed(int usedAtTime, final String lockId, final CacheConfig config) {
    boolean capacityEviction = (config.getTargetMaxInMemoryCount() > 0) || (config.getTargetMaxTotalCount() > 0);
    int configTTI = config.getMaxTTISeconds();
    if (configTTI == 1) usedAtTime += 1;
    if (shouldUpdateIdleTimer(usedAtTime, configTTI)) {
      setLastAccessedTime(usedAtTime, lockId);
    }
    if (capacityEviction) {
      if (!config.getCapacityEvictionPolicyDataFactory().isProductOfFactory(fastGetCapacityEvictionPolicyData())) {
        // reinitialize capacityEvictionPolicyData take care of CapacityEviction enabled/changed dynamically
        setCapacityEvictionPolicyData(config.getCapacityEvictionPolicyDataFactory().newCapacityEvictionPolicyData());
      }
      fastGetCapacityEvictionPolicyData().markUsed(usedAtTime);
    }
  }

  public abstract int getLastAccessedTime();

  protected abstract void setLastAccessedTimeInternal(int usedAtTime);

  private synchronized void setLastAccessedTime(final int usedAtTime, final String lockId) {
    if (CacheConfigFactory.DSO_ACTIVE) {
      ManagerUtil.beginLock(lockId, Manager.LOCK_TYPE_CONCURRENT);
      try {
        setLastAccessedTimeInternal(usedAtTime);
      } finally {
        ManagerUtil.commitLock(lockId, Manager.LOCK_TYPE_CONCURRENT);
      }
    } else {
      setLastAccessedTimeInternal(usedAtTime);
    }
  }

  /**
   * Only want to update the expire time if we are at least half way through the idle period.
   *
   * @param usedAtTime The time when the item is being used (==now) in seconds since the epoch
   * @return True if should update, false to skip it
   */
  private boolean shouldUpdateIdleTimer(final int usedAtTime, final int configTTI) {
    if (configTTI == UNUSED) { return false; }
    int timeSinceUsed = usedAtTime - getLastAccessedTime();

    // If so, only bother to update TTI if we're at least half way through
    // the TTI period
    final int halfTTI = configTTI / 2;
    if (timeSinceUsed >= halfTTI) {
      return true;
    } else {
      return false;
    }
  }

  /**
   * Determine whether this timestamp is expired at the specified time.
   *
   * @param atTime Usually the current time, in seconds since the epoch
   * @return True if this timestamp is expired at atTime
   */
  public final boolean isExpired(final int atTime, final CacheConfig config) {
    int expiresAt = expiresAt(config);
    return atTime >= expiresAt;
  }

  /**
   * Get the time at which this timestamp will become invalid. If the config specified ttl == tti == 0, making items
   * eternal, then this method returns {@link #NEVER_EXPIRE}.
   *
   * @return Timestamp of expiration, >0, may be NEVER_EXPIRE if eternal
   */
  public int expiresAt(final CacheConfig config) {
    int configTTI = config.getMaxTTISeconds();
    int configTTL = config.getMaxTTLSeconds();

    if (configTTI == UNUSED && configTTL == UNUSED) { return NEVER_EXPIRE; }

    int expiresAtTTL;
    if (configTTL == UNUSED || configTTL < 0) {
      expiresAtTTL = NEVER_EXPIRE;
    } else {
      expiresAtTTL = getCreateTime() + configTTL;
    }

    int expiresAtTTI;
    if (configTTI == UNUSED || configTTI < 0) {
      expiresAtTTI = NEVER_EXPIRE;
    } else {
      expiresAtTTI = getLastAccessedTime() + configTTI;
    }

    // expires at time which comes earliest between TTI and TTL
    return Math.min(expiresAtTTI, expiresAtTTL);
  }

  public abstract int getCreateTime();

  /**
   * Compare whether the value in this entry equals the value in another entry. Timestamps are ignored for equality
   * purposes. Two null values will compare equals.
   */
  @Override
  public boolean equals(final Object obj) {
    if (obj == this) {
      return true;
    } else if (obj instanceof AbstractTimestampedValue) {
      TimestampedValue<V> other = (TimestampedValue<V>) obj;

      V thisValue = getValue();

      if (other.getValue() == null) {
        if (thisValue == null) { return true; }
      } else if (thisValue != null) { return other.getValue().equals(thisValue); }
    }
    return false;
  }

  @Override
  public int hashCode() {
    V value = getValue();
    if (value == null) {
      return 0;
    } else {
      return value.hashCode();
    }
  }

  @Override
  public String toString() {
    return getClass().getSimpleName() + "<" + getValue() + ">";
  }
}
