/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.impl;

import org.terracotta.cache.CacheConfig;
import org.terracotta.cache.DistributedCache;
import org.terracotta.cache.TimestampedValue;
import org.terracotta.cache.evictor.Evictable;
import org.terracotta.cache.evictor.EvictionScheduler;
import org.terracotta.cache.evictor.EvictionStatistics;
import org.terracotta.cache.evictor.Evictor;
import org.terracotta.cache.evictor.TargetCapacityMapSizeListener;
import org.terracotta.cache.value.DefaultTimestampedValue;
import org.terracotta.collections.ConcurrentDistributedMap;
import org.terracotta.collections.FinegrainedLock;
import org.terracotta.collections.FinegrainedLockNoDso;

import com.tc.cluster.DsoCluster;

import java.util.Collection;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

public class LocalCache<K, V> implements DistributedCache<K, V>, Evictable<K> {

  // Provided configuration
  private final CacheConfig                                        config;

  // The actual data
  protected final ConcurrentDistributedMap<K, TimestampedValue<V>> data;

  // Boolean indicating whether statistics are enabled
  private final AtomicBoolean                                      statisticsEnabled     = new AtomicBoolean(false);

  // Eviction thread
  private transient EvictionScheduler                              evictionScheduler;

  // Eviction thread started flag
  private final AtomicBoolean                                      evictionThreadStarted = new AtomicBoolean(false);

  // A source for the current timestamp
  private transient TimeSource                                     timeSource;

  // JVM-local statistics instance
  private transient EvictionStatistics                             statistics            = new EvictionStatistics();

  /**
   * Construct with the map configuration
   *
   * @param config The configuration
   */
  public LocalCache(final CacheConfig config) {
    this.config = config;
    this.data = new ConcurrentDistributedMap<K, TimestampedValue<V>>();
    initializeOnLoad();
  }

  /**
   * Further initialize this object and launch the eviction thread.
   */
  public void initializeOnLoad() {
    this.timeSource = new SystemTimeSource();

    this.evictionScheduler = new EvictionScheduler(config, new Evictor<K>(this));

    this.statistics = new EvictionStatistics();

    this.data.registerMapSizeListener(new TargetCapacityMapSizeListener(data, getConfig()));
  }

  public boolean containsKey(final Object key) {
    checkExpired(key, false);
    return this.data.containsKey(key);
  }

  private V getValueSafe(final TimestampedValue<V> entry) {
    return entry == null ? null : entry.isExpired(getTime(), config) ? null : entry.getValue();
  }

  public V get(final Object key) {
    return getValueSafe(getTimestampedValue(key));
  }

  public TimestampedValue<V> getTimestampedValue(final Object key) {
    checkExpired(key, true);
    return this.data.get(key);
  }

  public TimestampedValue<V> getTimestampedValueQuiet(final Object key) {
    return getTimestampedValue(key);
  }

  public TimestampedValue<V> removeTimestampedValue(final K key) {
    return this.data.remove(key);
  }

  private void checkExpired(final Object key, final boolean markUsed) {
    TimestampedValue<V> entry = this.data.get(key);
    if (null == entry) return;
    if (isEvictionEnabled()) {
      int now = getTime();
      if (entry.isExpired(now, config)) {
        removeNoReturn(key);
      } else if (markUsed) {
        entry.markUsed(now, "", config);
      }
    }
  }

  public Set<K> keySet() {
    return this.data.keySet();
  }

  private TimestampedValue<V> newEntry(final V value) {
    if (value instanceof TimestampedValue) return (TimestampedValue<V>) value;
    return new DefaultTimestampedValue<V>(value, getTime());
  }

  public V put(final K key, final V value) {
    // ensure evictor is started once we put the first value
    startEvictionIfNecessary();

    return getValueSafe(this.data.put(key, newEntry(value)));
  }

  private void startEvictionIfNecessary() {
    if (this.evictionThreadStarted.compareAndSet(false, true)) {
      this.evictionScheduler.start();
    }
  }

  public V remove(final Object key) {
    return getValueSafe(this.data.remove(key));
  }

  public void clear() {
    this.data.clear();
  }

  public int size() {
    return this.data.size();
  }

  public int localSize() {
    return data.localSize();
  }

  public void shutdown() {
    this.evictionScheduler.stop();
  }

  public void evictExpiredLocalElements() {
    if (isEvictionEnabled()) {
      invalidateCacheEntries();
    }
  }

  public void evictOrphanElements(final DsoCluster cluster) {
    // no such thing as orphans in local map!
  }

  private void invalidateCacheEntries() {
    int totalCnt = 0;
    int evaled = 0;
    int notEvaled = 0;
    int errors = 0;

    Iterator<Entry<K, TimestampedValue<V>>> entryIter = data.entrySet().iterator();
    while (entryIter.hasNext()) {
      Map.Entry<K, TimestampedValue<V>> entry = entryIter.next();
      try {
        TimestampedValue tsEntry = entry.getValue();
        if (tsEntry == null) continue;

        totalCnt++;
        if (tsEntry.isExpired(getTime(), config)) {
          evaled++;
          entryIter.remove();
        } else {
          notEvaled++;
        }
      } catch (Throwable t) {
        errors++;
      }
    }

    if (isStatisticsEnabled()) {
      statistics.increment(totalCnt, evaled);
    }
  }

  /**
   * This is provided for testing purposes - it lets you override the source of System.currentTimeMillis() so that you
   * can control it yourself in the test. If it's not called, SystemTimeSource is used which just calls
   * System.currentTimeMillis().
   *
   * @param timeSource The alternate TimeSource implementation
   */
  public void setTimeSource(final TimeSource timeSource) {
    this.timeSource = timeSource;
  }

  public TimeSource getTimeSource() {
    return this.timeSource;
  }

  /**
   * This method should always be called instead of System.currentTimeMillis() so that time can be controlled by the
   * TimeSource.
   *
   * @return The current time according to the TimeSource
   */
  private int getTime() {
    return this.timeSource.now();
  }

  public Set<Entry<K, V>> entrySet() {
    return new EntrySet<K, V>(this, data.entrySet());
  }

  public void putNoReturn(final K key, final V value) {
    data.putNoReturn(key, newEntry(value));
  }

  public void removeNoReturn(final Object key) {
    data.removeNoReturn((K) key);
  }

  public V putIfAbsent(final K key, final V value) {
    return getValueSafe(data.putIfAbsent(key, newEntry(value)));
  }

  public V replace(final K key, final V value) {
    return getValueSafe(this.data.replace(key, newEntry(value)));
  }

  public boolean isStatisticsEnabled() {
    return statisticsEnabled.get();
  }

  public void setStatisticsEnabled(final boolean enabled) {
    synchronized (statistics) {
      if (enabled) {
        statistics.reset();
      } else {
        statistics.shutdown();
      }
      statisticsEnabled.set(enabled);
    }
  }

  public EvictionStatistics getStatistics() {
    return statistics;
  }

  public CacheConfig getConfig() {
    return config;
  }

  protected boolean isEvictionEnabled() {
    return config.getMaxTTISeconds() > 0 || config.getMaxTTLSeconds() > 0;
  }

  public boolean remove(final Object key, final Object value) {
    throw new UnsupportedOperationException();
  }

  public boolean replace(final K key, final V oldValue, final V newValue) {
    throw new UnsupportedOperationException();
  }

  public boolean containsValue(final Object value) {
    throw new UnsupportedOperationException();
  }

  public boolean isEmpty() {
    return size() == 0;
  }

  public void putAll(final Map<? extends K, ? extends V> t) {
    throw new UnsupportedOperationException();
  }

  public Collection<V> values() {
    throw new UnsupportedOperationException();
  }

  public FinegrainedLock createFinegrainedLock(final K key) {
    return new FinegrainedLockNoDso();
  }

  public void lockEntry(final K key) {
    //
  }

  public void unlockEntry(final K key) {
    //
  }

  public String getLockIdForKey(final K key) {
    return null;
  }
}
