/*
 * All content copyright (c) 2003-2009 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.cache.evictor;

import org.terracotta.cache.CacheConfigFactory;

import java.util.concurrent.locks.ReentrantReadWriteLock;

/**
 * Keeps track of eviction-related statistics.
 */
public class EvictionStatistics {
  private long                                   examined  = 0L;
  private long                                   evicted   = 0L;

  private final ReentrantReadWriteLock           lock      = new ReentrantReadWriteLock();
  private final ReentrantReadWriteLock.ReadLock  readLock  = lock.readLock();
  private final ReentrantReadWriteLock.WriteLock writeLock = lock.writeLock();

  private final EvictionStatisticsSampled sampled;

  public EvictionStatistics() {
    if (CacheConfigFactory.DSO_ACTIVE) {
      sampled = new EvictionStatisticsSampledDSO();
    } else {
      sampled = new EvictionStatisticsSampledNoDSO();
    }
  }

  /**
   * Startup new counters from zero.
   */
  public void reset() {
    writeLock.lock();
    try {
      examined = 0;
      evicted = 0;
      shutdown();
      sampled.reset();

    } finally {
      writeLock.unlock();
    }
  }

  /**
   * Stop sampling the counters.
   */
  public void shutdown() {
    writeLock.lock();
    try {
      sampled.shutdown();
    } finally {
      writeLock.unlock();
    }
  }

  /**
   * Increment the collected statistics data. This method is intended for internal usage and should be used directly by
   * the user.
   */
  public void increment(final long additionalExamined, final long additionalEvicted) {
    sampled.increment(additionalExamined, additionalEvicted);

    writeLock.lock();
    try {
      examined += additionalExamined;
      evicted += additionalEvicted;
    } finally {
      writeLock.unlock();
    }
  }

  /**
   * Retrieve the number of objects that were examined for eviction.
   */
  public long getExamined() {
    readLock.lock();
    try {
      return examined;
    } finally {
      readLock.unlock();
    }
  }

  /**
   * Get the most recent sample of the examined entries.
   * <p>
   * The number of objects that are returned are sampled per second and the latest value is returned.
   */
  public long getExaminedMostRecentSample() {
    return sampled.getExaminedMostRecentSample();
  }

  /**
   * Retrieve the number of objects that were evicted.
   */
  public long getEvicted() {
    readLock.lock();
    try {
      return evicted;
    } finally {
      readLock.unlock();
    }
  }

  /**
   * Get the most recent sample of the evicted entries.
   * <p>
   * The number of objects that are returned are sampled per second and the latest value is returned.
   */
  public long getEvictedMostRecentSample() {
    return sampled.getEvictedMostRecentSample();
  }
}