/**
 * Copyright 2003-2009 Terracotta, Inc. Licensed under the Apache License, Version 2.0 (the "License"); you may not use
 * this file except in compliance with the License. You may obtain a copy of the License at
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language governing permissions and limitations under the
 * License.
 */
package org.terracotta.collections.chm;

import com.tc.object.ObjectID;

import java.util.ArrayList;
import java.util.Map;
import java.util.Random;

/**
 * SelectableConcurrentHashMap subclasses a repackaged version of ConcurrentHashMap ito allow efficient random sampling
 * of the map values.
 * <p>
 * The random sampling technique involves randomly selecting a map Segment, and then selecting a number of random entry
 * chains from that segment.
 * 
 * @author Chris Dennis
 */
public class SelectableConcurrentHashMap<K, V> extends ConcurrentHashMap<K, V> {

  private final Random rndm = new Random();

  public SelectableConcurrentHashMap() {
    super();
  }

  public SelectableConcurrentHashMap(int initialCapacity, float loadFactor, int concurrency) {
    super(initialCapacity, loadFactor, concurrency);
  }

  public Map.Entry<K, V> getRandomEntry() {
    final int segmentIndex = rndm.nextInt(segments.length);

    int i = segmentIndex;
    do {
      final Segment<K, V> seg = segments[i];
      final HashEntry<K, V>[] table = seg.table;
      final int tableIndex = rndm.nextInt(table.length);

      int j = tableIndex;
      do {
        HashEntry<K, V> e = table[j];

        if (e != null && e.next == null) {
          if (e.value != null) { return new MapEntry<K, V>(e); }
        } else if (e != null) {
          ArrayList<HashEntry<K, V>> list = new ArrayList<HashEntry<K, V>>();
          for (; e != null; e = e.next) {
            if (e.value != null) {
              list.add(e);
            }
          }
          if (!list.isEmpty()) { return new MapEntry<K, V>(list.get(rndm.nextInt(list.size()))); }
        }

        // move onto next table slot
        j = (j + 1) & (table.length - 1);
      } while (j != tableIndex);

      // move onto next segment
      i = (i + 1) & segmentMask;
    } while (i != segmentIndex);

    return null;
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    final int segmentIndex = rndm.nextInt(segments.length);

    int i = segmentIndex;
    do {
      final Segment<K, V> seg = segments[i];
      final HashEntry<K, V>[] table = seg.table;
      final int tableIndex = rndm.nextInt(table.length);

      int j = tableIndex;
      do {
        HashEntry<K, V> e = table[j];

        if (e != null && e.next == null) {
          if (e.value != null && !(e.value instanceof ObjectID)) { return new MapEntry<K, V>(e); }
        } else if (e != null) {
          ArrayList<HashEntry<K, V>> list = new ArrayList<HashEntry<K, V>>();
          for (; e != null; e = e.next) {
            if (e.value != null && !(e.value instanceof ObjectID)) {
              list.add(e);
            }
          }
          if (!list.isEmpty()) { return new MapEntry<K, V>(list.get(rndm.nextInt(list.size()))); }
        }

        // move onto next table slot
        j = (j + 1) % table.length;
      } while (j != tableIndex);

      // move onto next segment
      i = (i + 1) & segmentMask;
    } while (i != segmentIndex);

    return null;
  }

  /**
   * Similar to {@link #replace(Object, Object, Object)}, only difference is that it uses reference equality instead of
   * object equality when comparing the <tt>oldValue</tt> and <tt>newValue</tt>
   * 
   * @throws NullPointerException if any of the arguments are null
   */
  public boolean replaceUsingReferenceEquality(K key, V oldValue, V newValue) {
    if (oldValue == null || newValue == null) throw new NullPointerException();
    int hash = hash(key.hashCode());
    return segmentFor(hash).replaceUsingReferenceEquality(key, hash, oldValue, newValue);
  }

  public HashEntry<K, V> removeReturnHashEntry(Object key) {
    int hash = hash(key.hashCode());
    return segmentFor(hash).remove(key, hash, null);
  }

  public HashEntry<K, V> removeReturnHashEntry(Object key, Object value) {
    int hash = hash(key.hashCode());
    return segmentFor(hash).remove(key, hash, value);
  }

  public HashEntry<K, V> replaceReturnHashEntry(K key, V oldValue, V newValue) {
    int hash = hash(key.hashCode());
    return segmentFor(hash).replace(key, hash, oldValue, newValue);
  }

  public HashEntry<K, V> putReturnHashEntry(K key, V newValue) {
    int hash = hash(key.hashCode());
    return segmentFor(hash).put(key, hash, newValue, false);
  }

  public HashEntry<K, V> replaceReturnHashEntry(K key, V newValue) {
    int hash = hash(key.hashCode());
    return segmentFor(hash).replace(key, hash, newValue);
  }

  private static class MapEntry<K, V> implements Map.Entry<K, V> {

    private final HashEntry<K, V> entry;

    public MapEntry(HashEntry<K, V> entry) {
      this.entry = entry;
    }

    public K getKey() {
      return entry.key;
    }

    public V getValue() {
      return entry.value;
    }

    public V setValue(V value) {
      throw new UnsupportedOperationException();
    }

    @Override
    public boolean equals(Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry e = (Map.Entry) o;

      K key = getKey();

      return eq(key, e.getKey()) && eq(getValue(), e.getValue());
    }

    @Override
    public int hashCode() {
      K key = getKey();
      V value = getValue();

      return (key == null ? 0 : key.hashCode()) ^ (value == null ? 0 : value.hashCode());
    }

    @Override
    public String toString() {
      return getKey() + "=" + getValue();
    }

    private static boolean eq(Object o1, Object o2) {
      return o1 == null ? o2 == null : o1.equals(o2);
    }

  }
}
