/*
 * All content copyright (c) 2003-2008 Terracotta, Inc., except as may otherwise be noted in a separate copyright
 * notice. All rights reserved.
 */
package org.terracotta.collections;

import java.util.AbstractCollection;
import java.util.AbstractMap;
import java.util.AbstractSet;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;

import org.terracotta.collections.chm.SelectableConcurrentHashMap;
import org.terracotta.collections.chm.ConcurrentHashMap.HashEntry;

import com.tc.exception.TCNonPortableObjectError;
import com.tc.exception.TCObjectNotFoundException;
import com.tc.logging.TCLogger;
import com.tc.object.ObjectID;
import com.tc.object.SerializationUtil;
import com.tc.object.TCObject;
import com.tc.object.bytecode.AAFairDistributionPolicyMarker;
import com.tc.object.bytecode.Clearable;
import com.tc.object.bytecode.Manageable;
import com.tc.object.bytecode.Manager;
import com.tc.object.bytecode.ManagerUtil;
import com.tc.object.bytecode.NullManager;
import com.tc.object.bytecode.TCMap;
import com.tc.util.Assert;

// NOTE: This type is public to allow it to be a reasonable alternative to HashMap/CHM in express products

public class ConcurrentDistributedMapDso<K, V> extends AbstractMap<K, V> implements
    ConcurrentDistributedMapBackend<K, V>, TCMap, Manageable, Clearable, AAFairDistributionPolicyMarker {

  private static final TCLogger              LOGGER              = ManagerUtil
                                                                     .getLogger(ConcurrentDistributedMapDso.class
                                                                         .getName());
  private static final Object[]              NO_ARGS             = new Object[0];

  private volatile transient TCObject        $__tc_MANAGED;
  private volatile boolean                   evictionEnabled     = true;

  private final int                          dsoLockType;

  private final SelectableConcurrentHashMap  store;
  private final LockStrategy<? super K>      lockStrategy;

  private volatile transient MapSizeListener listener;
  private transient AtomicInteger            localSize;
  private volatile transient String          instanceDsoLockName = null;

  public ConcurrentDistributedMapDso(final LockType lockType, final LockStrategy<? super K> lockingStrategy) {
    this(lockType.getDsoLockType(), lockingStrategy);
  }

  /*
   * This factory method is included to make for easier reflective invocation from the applicator. Otherwise the
   * applicator would have to worry about loading the LockType enum in the correct class loader and then go about
   * finding the matching enum etc...
   */
  public static <K, V> ConcurrentDistributedMapDso<K, V> newCDM(final int lockLevel, final Object strategy) {
    return new ConcurrentDistributedMapDso<K, V>(lockLevel, (LockStrategy<? super K>) strategy);
  }

  private ConcurrentDistributedMapDso(final int lockLevel, final LockStrategy<? super K> lockStrategy) {
    // ensure that DSO is active at construction time, ie. fail fast
    Assert.pre(!(ManagerUtil.getManager() instanceof NullManager));

    this.dsoLockType = lockLevel;
    this.store = new SelectableConcurrentHashMap();
    this.lockStrategy = lockStrategy;
    this.localSize = new AtomicInteger(0);
    Assert.post(store != null);
  }

  private String getInstanceDsoLockName() {
    if (instanceDsoLockName != null) { return instanceDsoLockName; }

    // The trailing colon (':') is relevant to avoid unintended lock collision when appending object ID to the key
    // Without the delimiter you'd get the same effective lock for oid 11, lockId 10 and oid 1 and lockId 110
    instanceDsoLockName = "@" + "CDM" + ((Manageable) this).__tc_managed().getObjectID().toLong() + ":";
    return instanceDsoLockName;
  }

  private String generateLockIdForKey(final K key) {
    if (null == key) { throw new NullPointerException(); }

    return lockStrategy.generateLockIdForKey(getInstanceDsoLockName(), key);
  }

  @Override
  public V put(final K key, final V value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);
      Object old;
      boolean sizeChanged = false;
      prefetch(key);
      beginLock(lockID, dsoLockType);
      try {
        pinLock(lockID);
        HashEntry<K, V> e = store.putReturnHashEntry(key, value);
        old = e.value;

        if (old != value) {

          doLogicalPut(e.key, value, old);

          if (old == null) {
            localSizeIncrement();
            sizeChanged = true;
          } else if (old instanceof ObjectID) {
            localSizeIncrement();
          }
        }

        old = lookup(old);
      } finally {
        commitLock(lockID, dsoLockType);
      }

      if (sizeChanged) sizeIncrement();

      return (V) old;
    } else {
      V old = (V) store.put(key, value);
      if (old == null) {
        localSizeIncrement();
        sizeIncrement();
      }
      return old;
    }
  }

  private void doLogicalPut(K key, V value, Object old) {
    try {
      ManagerUtil.logicalInvoke(this, SerializationUtil.PUT_SIGNATURE, new Object[] { key, value });
    } catch (TCNonPortableObjectError npoe) {
      if (old != null) {
        store.put(key, old);
      } else {
        store.remove(key);
      }
      throw npoe;
    }
  }

  private void doLogicalRemove(K key) {
    ManagerUtil.logicalInvoke(this, SerializationUtil.REMOVE_KEY_SIGNATURE, new Object[] { key });
  }

  private void doLogicalClear() {
    ManagerUtil.logicalInvoke(this, SerializationUtil.CLEAR_SIGNATURE, NO_ARGS);
  }

  public V unsafeGet(final K key) {
    Object obj = store.get(key);

    // Performing a lookup here without any locks would be a dirty read and could request a non-existent object
    // (read: Don't change this!)
    if (obj instanceof ObjectID) return null;

    return (V) obj;
  }

  @Override
  public V get(final Object key) {
    if (__tc_isManaged()) {

      final String lockID = generateLockIdForKey((K) key);

      beginLock(lockID, Manager.LOCK_TYPE_READ);
      try {
        V result = lookupAndFaultIn(key, store.get(key));
        if (result != null) {
          pinLock(lockID);
        }
        return result;
      } finally {
        commitLock(lockID, Manager.LOCK_TYPE_READ);
      }
    } else {
      return (V) store.get(key);
    }
  }

  public V unlockedGet(final Object key) {
    if (__tc_isManaged()) {
      return lookupAndFaultIn(key, store.get(key), false);
    } else {
      return (V) store.get(key);
    }
  }

  /**
   * Pre-fetches the value associated with the key, if it is not already faulted into the VM.
   * 
   * @param key
   * @return the current mapping, ObjectID if it is not faulted, Object if faulted, null if no mapping present.
   */
  private Object prefetch(final Object key) {
    Object obj = store.get(key);
    if (obj instanceof ObjectID) {
      // XXX::Note since we are reading outside the lock scope this might result in an ObjectNotFound Message sent from
      // the server, but we ignore it since it a result of pre-fetch request.
      ManagerUtil.preFetchObject((ObjectID) obj);
    }
    return obj;
  }

  @Override
  public V remove(final Object key) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey((K) key);

      Object old;
      boolean sizeChanged = false;

      prefetch(key);
      beginLock(lockID, dsoLockType);

      try {
        HashEntry<K, V> oldEntry = store.removeReturnHashEntry(key);
        old = oldEntry == null ? null : oldEntry.value;

        if (old != null) {
          doLogicalRemove(oldEntry.key);
          if (!(old instanceof ObjectID)) {
            localSizeDecrement();
          }
          sizeChanged = true;
        }
        old = lookup(old);
      } finally {
        commitLock(lockID, dsoLockType);
        unpinLock(lockID);
      }

      if (sizeChanged) sizeDecrement();

      return (V) old;
    } else {
      V old = (V) store.remove(key);
      if (old != null) {
        localSizeDecrement();
        sizeDecrement();
      }
      return old;
    }
  }

  public void removeNoReturn(final K key) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey(key);

      boolean sizeChanged = false;

      beginLock(lockID, dsoLockType);
      try {
        sizeChanged = doRemoveNoReturn(key);
      } finally {
        commitLock(lockID, dsoLockType);
        unpinLock(lockID);
      }

      if (sizeChanged) sizeDecrement();
    } else {
      if (store.remove(key) != null) {
        localSizeDecrement();
        sizeDecrement();
      }
    }
  }

  public void unlockedRemoveNoReturn(K key) {
    if (__tc_isManaged()) {
      boolean sizeChanged = doRemoveNoReturn(key);
      if (sizeChanged) sizeDecrement();
    } else {
      if (store.remove(key) != null) {
        localSizeDecrement();
        sizeDecrement();
      }
    }
  }

  private boolean doRemoveNoReturn(final K key) {
    boolean sizeChanged = false;
    HashEntry<K, V> oldEntry = store.removeReturnHashEntry(key);
    Object old = oldEntry == null ? null : oldEntry.value;
    if (old != null) {
      doLogicalRemove(oldEntry.key);
      if (!(old instanceof ObjectID)) {
        localSizeDecrement();
      }
      sizeChanged = true;
    }
    return sizeChanged;
  }

  public void putNoReturn(final K key, final V value) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey(key);

      boolean sizeChanged = false;
      beginLock(lockID, dsoLockType);
      try {
        pinLock(lockID);
        sizeChanged = doPutNoReturn(key, value);
      } finally {
        commitLock(lockID, dsoLockType);
      }

      if (sizeChanged) sizeIncrement();
    } else {
      if (store.put(key, value) == null) {
        localSizeIncrement();
        sizeIncrement();
      }
    }
  }

  public void unlockedPutNoReturn(final K key, final V value) {
    if (__tc_isManaged()) {
      boolean sizeChanged = doPutNoReturn(key, value);
      if (sizeChanged) sizeIncrement();
    } else {
      if (store.put(key, value) == null) {
        localSizeIncrement();
        sizeIncrement();
      }
    }
  }

  private boolean doPutNoReturn(final K key, final V value) {
    HashEntry<K, V> e = store.putReturnHashEntry(key, value);
    Object old = e.value;

    if (old != value) {
      doLogicalPut(e.key, value, old);
      if (old == null) {
        localSizeIncrement();
        return true;
      } else if (old instanceof ObjectID) {
        localSizeIncrement();
      }
    }
    return false;
  }

  @Override
  public boolean containsKey(final Object key) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey((K) key);

      beginLock(lockID, Manager.LOCK_TYPE_READ);
      try {
        return store.containsKey(key);
      } finally {
        commitLock(lockID, Manager.LOCK_TYPE_READ);
      }
    } else {
      return store.containsKey(key);
    }
  }

  @Override
  public boolean containsValue(final Object value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      for (Entry<K, V> entry : entrySet()) {
        final String lockID = generateLockIdForKey(entry.getKey());

        beginLock(lockID, Manager.LOCK_TYPE_READ);
        try {
          if (value.equals(lookupAndFaultIn(entry.getKey(), entry.getValue()))) { return true; }
        } finally {
          commitLock(lockID, Manager.LOCK_TYPE_READ);
        }
      }

      return false;
    } else {
      return store.containsValue(value);
    }
  }

  @Override
  public int size() {
    if (__tc_isManaged()) {
      beginLock(getInstanceDsoLockName(), Manager.LOCK_TYPE_READ);
      try {
        return store.size();
      } finally {
        commitLock(getInstanceDsoLockName(), Manager.LOCK_TYPE_READ);
      }
    } else {
      return store.size();
    }
  }

  public int localSize() {
    if (__tc_isManaged()) {
      return localSize.get();
    } else {
      return size();
    }
  }

  @Override
  public void clear() {
    if (__tc_isManaged()) {
      beginLock(getInstanceDsoLockName(), dsoLockType);

      try {
        unpinAllLocks();
        int size = store.size();
        store.clear();
        int local = localSize.getAndSet(0);
        if (listener != null) {
          listener.localSizeChanged(-local);
          listener.sizeChanged(-size);
        }
        doLogicalClear();
      } finally {
        commitLock(getInstanceDsoLockName(), dsoLockType);
      }
    } else {
      int local = localSize.getAndSet(0);
      store.clear();
      int size = store.size();
      if (listener != null) {
        listener.localSizeChanged(-local);
        listener.sizeChanged(-size);
      }
    }
  }

  @Override
  public Set<K> keySet() {
    return new KeySet(store.keySet());
  }

  @Override
  public Set<Entry<K, V>> entrySet() {
    return new EntrySet(store.entrySet());
  }

  /*
   * ConcurrentMap methods
   */

  public V putIfAbsent(final K key, final V value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      Object old;
      boolean sizeChanged = false;

      prefetch(key);

      beginLock(lockID, dsoLockType);
      try {
        pinLock(lockID);
        old = store.putIfAbsent(key, value);
        if (old == null) {
          doLogicalPut(key, value, old);
          localSizeIncrement();
          sizeChanged = true;
        }

        old = lookupAndFaultIn(key, old);
      } finally {
        commitLock(lockID, dsoLockType);
      }

      if (sizeChanged) sizeIncrement();

      return (V) old;
    } else {
      V old = (V) store.putIfAbsent(key, value);
      if (old == null) {
        localSizeIncrement();
        sizeIncrement();
      }
      return old;
    }
  }

  public boolean remove(final Object key, final Object value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey((K) key);

      boolean success = false;

      beginLock(lockID, dsoLockType);
      try {
        fault(key);
        HashEntry<K, V> oldEntry = store.removeReturnHashEntry(key, value);
        if (oldEntry != null && oldEntry.value != null) {
          doLogicalRemove(oldEntry.key);
          localSizeDecrement();
          success = true;
        } else {
          success = false;
        }
      } finally {
        commitLock(lockID, dsoLockType);
        if (success) unpinLock(lockID);
      }

      if (success) sizeDecrement();

      return success;
    } else {
      if (store.remove(key, value)) {
        localSizeDecrement();
        sizeDecrement();
        return true;
      } else {
        return false;
      }
    }
  }

  public boolean replace(final K key, final V oldValue, final V newValue) {
    if (__tc_isManaged()) {
      if (null == oldValue) { throw new NullPointerException(); }
      if (null == newValue) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      prefetch(key);
      beginLock(lockID, dsoLockType);
      try {
        pinLock(lockID);
        fault(key);
        HashEntry<K, V> e = store.replaceReturnHashEntry(key, oldValue, newValue);
        if (e != null) {
          doLogicalPut(e.key, newValue, e.value);
          return true;
        } else {
          return false;
        }
      } finally {
        commitLock(lockID, dsoLockType);
      }
    } else {
      return store.replace(key, oldValue, newValue);
    }
  }

  public V replace(final K key, final V value) {
    if (__tc_isManaged()) {
      if (null == value) { throw new NullPointerException(); }

      final String lockID = generateLockIdForKey(key);

      prefetch(key);
      beginLock(lockID, dsoLockType);
      try {
        HashEntry<K, V> entry = store.replaceReturnHashEntry(key, value);
        Object old = entry == null ? null : entry.value;
        if (old != null) {
          doLogicalPut(entry.key, value, old);
          if (old instanceof ObjectID) {
            localSizeIncrement();
          }
        }
        return lookup(old);
      } finally {
        commitLock(lockID, dsoLockType);
      }
    } else {
      return (V) store.replace(key, value);
    }
  }

  public Collection<Entry<K, V>> getAllEntriesSnapshot() {
    return Collections.unmodifiableCollection(store.entrySet());
  }

  public Collection<Entry<K, V>> getAllLocalEntriesSnapshot() {
    return new LocalEntriesCollection();
  }

  private class KeySet extends AbstractSet<K> {

    private final Set<K> delegate;

    private KeySet(final Set<K> delegate) {
      this.delegate = delegate;
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDso.this.size();
    }

    @Override
    public boolean contains(final Object key) {
      return ConcurrentDistributedMapDso.this.containsKey(key);
    }

    @Override
    public void clear() {
      ConcurrentDistributedMapDso.this.clear();
    }

    @Override
    public boolean remove(final Object key) {
      return ConcurrentDistributedMapDso.this.remove(key) != null;
    }

    @Override
    public Iterator<K> iterator() {
      return new KeyIterator(delegate.iterator());
    }

  }

  private class KeyIterator implements Iterator<K> {

    private final Iterator<K> delegate;
    private K                 lastKey;

    public KeyIterator(final Iterator<K> delegate) {
      this.delegate = delegate;
    }

    public boolean hasNext() {
      return delegate.hasNext();
    }

    public K next() {
      final K result = delegate.next();
      lastKey = result;
      return result;
    }

    public void remove() {
      if (null == lastKey) { throw new IllegalStateException("next needs to be called before calling remove"); }

      if (ManagerUtil.isManaged(ConcurrentDistributedMapDso.this)) {
        ConcurrentDistributedMapDso.this.remove(lastKey);
      } else {
        delegate.remove();
      }

      lastKey = null;
    }

  }

  private class EntrySet extends AbstractSet<Entry<K, V>> {

    private final Set<Entry<K, V>> delegate;

    private EntrySet(final Set<Entry<K, V>> delegate) {
      this.delegate = delegate;
    }

    @Override
    public Iterator<Entry<K, V>> iterator() {
      return new EntryIterator(delegate.iterator());
    }

    @Override
    public int size() {
      return ConcurrentDistributedMapDso.this.size();
    }

    @Override
    public boolean contains(final Object o) {
      if (!(o instanceof Entry)) return false;
      Entry e = (Entry) o;
      V value = ConcurrentDistributedMapDso.this.get(e.getKey());
      return value != null && value.equals(e.getValue());
    }

    @Override
    public boolean remove(final Object o) {
      if (!(o instanceof Entry)) return false;
      Entry e = (Entry) o;
      return ConcurrentDistributedMapDso.this.remove(e.getKey(), e.getValue());
    }

    @Override
    public void clear() {
      ConcurrentDistributedMapDso.this.clear();
    }
  }

  private class EntryIterator implements Iterator<Entry<K, V>> {

    private final Iterator<Entry<K, V>> delegate;
    private Entry<K, V>                 lastEntry;

    private EntryIterator(final Iterator<Entry<K, V>> delegate) {
      this.delegate = delegate;
    }

    public boolean hasNext() {
      return delegate.hasNext();
    }

    public synchronized Entry<K, V> next() {
      final Entry<K, V> result = delegate.next();
      if (null == result) {
        lastEntry = null;
        return null;
      }

      K key = result.getKey();

      // in theory we could filter out entries for which we observer a null value here (ie. removed)
      lastEntry = new TcmEntry(key, get(key));
      return lastEntry;
    }

    public synchronized void remove() {
      if (null == lastEntry) { throw new IllegalStateException("next needs to be called before calling remove"); }
      ConcurrentDistributedMapDso.this.remove(lastEntry.getKey(), lastEntry.getValue());
      lastEntry = null;
    }
  }

  private class TcmEntry implements Entry<K, V> {

    private final K key;
    private V       value;

    private TcmEntry(K key, V value) {
      this.key = key;
      this.value = value;
    }

    public K getKey() {
      // entry locking here doesn't make sense since
      // 1. Entry keys can't change
      // 2. you're have to get the key anyway to generate the lock ID
      return key;
    }

    public V getValue() {
      return value;
    }

    public V setValue(final V newValue) {
      V old = value;
      put(key, newValue);
      value = newValue;
      return old;
    }

    @Override
    public boolean equals(Object o) {
      if (!(o instanceof Map.Entry)) return false;
      Map.Entry e = (Map.Entry) o;
      return eq(key, e.getKey()) && eq(getValue(), e.getValue());
    }

    @Override
    public int hashCode() {
      return (key == null ? 0 : key.hashCode()) ^ (value == null ? 0 : value.hashCode());
    }

    @Override
    public String toString() {
      return key + "=" + getValue();
    }

    private boolean eq(Object o1, Object o2) {
      return o1 == null ? o2 == null : o1.equals(o2);
    }

  }

  /**
   * If the value associated with the supplied key is an ObjectID then the value is looked up and faulted into the map.
   * This operation is safe in the absence of all locking, but no guarantees can then be made about subsequent gets of
   * the key.
   */
  private void fault(final Object key) {
    Object obj = store.get(key);
    if (obj instanceof ObjectID) {
      try {
        V value = (V) ManagerUtil.lookupObject((ObjectID) obj);
        if (store.replace(key, obj, value)) {
          localSizeIncrement();
        }
      } catch (TCObjectNotFoundException e) {
        LOGGER.info("Missing object caused by concurrent map-level and key-level operations\n"
                    + "Removing local entry for this key [" + key + "] to restore L1-L2 correspondence.", e);
        store.remove(key, obj);
        unpinLock(generateLockIdForKey((K) key));
      }
    }
  }

  /**
   * If the passed object is an ObjectID looks it up and returns it, after attempting to replace the original ObjectID
   * value in the map. Safe in the absence of all locking.
   */
  private V lookupAndFaultIn(final Object key, final Object obj) {
    return lookupAndFaultIn(key, obj, true);
  }

  private V lookupAndFaultIn(final Object key, final Object obj, boolean coherent) {
    if (obj instanceof ObjectID) {
      try {
        V value = (V) ManagerUtil.lookupObject((ObjectID) obj);
        // We should really assert that the replace succeeds here
        if (store.replace(key, obj, value)) {
          localSizeIncrement();
        }
        return value;
      } catch (TCObjectNotFoundException e) {
        if (coherent) {
          LOGGER.info("Missing object caused by concurrent map-level and key-level operations\n"
                      + "Removing local entry for this key [" + key + "] to restore L1-L2 correspondence.", e);
          // Can't remove if it is not coherent.
          store.remove(key, obj);
          unpinLock(generateLockIdForKey((K) key));
        } else {
          String msg = "Retrieval attempt for Garbage collected object: key=" + key + ", ObjectID=" + obj.toString();
          if (LOGGER.isDebugEnabled()) {
            LOGGER.info(msg, e);
          } else {
            LOGGER.info(msg);
          }
        }
        return null;
      }
    } else {
      return (V) obj;
    }
  }

  /**
   * If the passed object is an ObjectID, looks it up and returns it.
   */
  private V lookup(final Object obj) {
    if (obj instanceof ObjectID) {
      try {
        return (V) ManagerUtil.lookupObject((ObjectID) obj);
      } catch (TCObjectNotFoundException e) {
        LOGGER.info("Missing object caused by concurrent map-level and key-level operations\n"
                    + "Returning null as the only sensible result.  Caller should have removed the local mapping.", e);
        return null;
      }
    } else {
      return (V) obj;
    }
  }

  public void __tc_applicator_clear() {
    store.clear();
  }

  public void __tc_applicator_put(final Object key, final Object value) {
    Object old = store.put(key, value);
    if (value instanceof ObjectID) {
      if (old != null && !(old instanceof ObjectID)) {
        localSizeDecrement();
      }
    } else {
      if (old == null || old instanceof ObjectID) {
        localSizeIncrement();
      }
    }
  }

  public void __tc_applicator_remove(final Object key) {
    unpinLock(generateLockIdForKey((K) key));

    Object old = store.remove(key);
    if (old != null && !(old instanceof ObjectID)) {
      localSizeDecrement();
    }
  }

  public Collection __tc_getAllEntriesSnapshot() {
    return getAllEntriesSnapshot();
  }

  public Collection __tc_getAllLocalEntriesSnapshot() {
    return getAllLocalEntriesSnapshot();
  }

  public void __tc_put_logical(final Object key, final Object value) {
    putNoReturn((K) key, (V) value);
  }

  public void __tc_remove_logical(final Object key) {
    removeNoReturn((K) key);
  }

  public boolean __tc_isManaged() {
    return $__tc_MANAGED != null;
  }

  public TCObject __tc_managed() {
    return $__tc_MANAGED;
  }

  public void __tc_managed(final TCObject tcObject) {
    $__tc_MANAGED = tcObject;
  }

  public int __tc_clearReferences(final int toClear) {
    if (!__tc_isManaged()) { throw new AssertionError("clearReferences() called on Unmanaged Map"); }

    int cleared = 0;
    for (Object key : store.keySet()) {
      Object v = store.get(key);
      if (v instanceof ObjectID) continue;

      TCObject tcObject = ManagerUtil.lookupExistingOrNull(v);
      if (tcObject != null && !tcObject.recentlyAccessed()) {
        ObjectID oid = tcObject.getObjectID();
        if (store.replaceUsingReferenceEquality(key, v, oid)) {
          localSizeDecrement();
          unpinLock(generateLockIdForKey((K) key));
          if (++cleared == toClear) break;
        }
      }
    }
    return cleared;
  }

  public boolean isEvictionEnabled() {
    return evictionEnabled;
  }

  public void setEvictionEnabled(final boolean flag) {
    evictionEnabled = flag;
  }

  /*
   * pinLock() should be called after begin so that the lock manager can first create the locks before pinning them.
   */
  private void pinLock(String lockId) {
    lockStrategy.pinLock(lockId);
  }

  private void commitLock(String lockID, int type) {
    lockStrategy.commitLock(lockID, type);
  }

  private void beginLock(String lockID, int type) {
    lockStrategy.beginLock(lockID, type);
  }

  private void unpinLock(final String lock) {
    lockStrategy.unpinLock(lock);
  }

  private void unpinAllLocks() {
    for (K key : (Set<K>) store.keySet()) {
      unpinLock(generateLockIdForKey(key));
    }
  }

  public FinegrainedLock createFinegrainedLock(final K key) {
    if (__tc_isManaged()) {
      String lockId = generateLockIdForKey(key);
      if (lockId == null) {
        //
        throw new UnsupportedOperationException("fine grained lock not supported with null lock for key [" + key + "]");
      }

      return new FinegrainedLockDso(lockId, dsoLockType);
    } else {
      return new FinegrainedLockNoDso();
    }
  }

  public void lockEntry(final K key) {
    if (__tc_isManaged()) {

      String lockId = generateLockIdForKey(key);

      /*
       * Note this allows locking on keys that are not present in the map. Hibernate uses this to lock a key prior to
       * creating a mapping for it.
       */
      beginLock(lockId, dsoLockType);
      pinLock(lockId);
    }
  }

  public void unlockEntry(final K key) {
    if (__tc_isManaged()) {
      String lockId = generateLockIdForKey(key);
      boolean validKey = containsKey(key); // should have write lock on this key here
      commitLock(lockId, dsoLockType);
      if (!validKey) {
        /*
         * If we were locking a key that is not in the map we must do our own cleanup. Cannot rely on the map to ever
         * evict otherwise. Would be a resource leak...
         */
        unpinLock(lockId);
      }
    }
  }

  public String getLockIdForKey(final K key) {
    if (__tc_isManaged()) {
      String lockId = generateLockIdForKey(key);
      if (lockId == null) { throw new UnsupportedOperationException("null lock for key [" + key + "]"); }
      return lockId;
    }

    return "";
  }

  public List<Map<K, ?>> getConstituentMaps() {
    return Collections.<Map<K, ?>> singletonList(this);
  }

  public Map.Entry<K, V> getRandomEntry() {
    if (isEmpty()) {
      return null;
    } else {
      return store.getRandomEntry();
    }
  }

  public Map.Entry<K, V> getRandomLocalEntry() {
    if (localSize() == 0) {
      return null;
    } else {
      return store.getRandomLocalEntry();
    }
  }

  public boolean flush(final Object key, final Object value) {
    if (__tc_isManaged()) {
      if (value instanceof ObjectID) { return false; }
      TCObject tcObject = ManagerUtil.lookupExistingOrNull(value);
      if (tcObject == null) { return false; }

      ObjectID oid = tcObject.getObjectID();
      boolean success = store.replaceUsingReferenceEquality(key, value, oid);
      if (success) {
        localSizeDecrement();
        unpinLock(generateLockIdForKey((K) key));
      }

      return success;
    } else {
      return remove(key, value);
    }
  }

  public boolean tryRemove(final Object key, final long time, final TimeUnit unit) {
    if (__tc_isManaged()) {
      final String lockID = generateLockIdForKey((K) key);

      try {
        if (tryBeginLock(lockID, dsoLockType, unit.toNanos(time))) {
          boolean removed = false;

          try {
            HashEntry<K, V> oldEntry = store.removeReturnHashEntry(key);
            Object old = oldEntry == null ? null : oldEntry.value;
            if (old != null) {
              doLogicalRemove(oldEntry.key);
              if (!(old instanceof ObjectID)) {
                localSizeDecrement();
              }
              removed = true;
            }
          } finally {
            commitLock(lockID, dsoLockType);
            unpinLock(lockID);
          }
          if (removed) {
            sizeDecrement();
          }

          return removed;
        } else {
          return false;
        }
      } catch (InterruptedException e) {
        return false;
      }
    } else {
      if (store.remove(key) != null) {
        localSizeDecrement();
        sizeDecrement();
        return true;
      } else {
        return false;
      }
    }
  }

  private boolean tryBeginLock(String lockID, int type, long nanos) throws InterruptedException {
    return lockStrategy.tryBeginLock(lockID, type, nanos);
  }

  private class LocalEntriesCollection extends AbstractCollection<Entry<K, V>> {

    private final Collection<Entry<K, V>> delegate = store.entrySet();

    @Override
    public Iterator<java.util.Map.Entry<K, V>> iterator() {
      return new LocalEntriesIterator<K, V>(delegate.iterator());
    }

    @Override
    public int size() {
      int size = 0;
      for (Object o : store.values()) {
        if (!(o instanceof ObjectID)) {
          size++;
        }
      }

      return size;
    }

    @Override
    public boolean contains(final Object o) {
      return delegate.contains(o);
    }

    @Override
    public boolean remove(final Object o) {
      throw new UnsupportedOperationException();
    }
  }

  private static class LocalEntriesIterator<K, V> implements Iterator<Entry<K, V>> {

    private final Iterator<Entry<K, V>> delegate;

    private Entry<K, V>                 next;

    public LocalEntriesIterator(final Iterator<Entry<K, V>> delegate) {
      this.delegate = delegate;
      findNext();
    }

    public boolean hasNext() {
      return next != null;
    }

    public java.util.Map.Entry<K, V> next() {
      Entry result = findNext();
      if (result == null) {
        throw new NoSuchElementException();
      } else {
        return result;
      }
    }

    public void remove() {
      throw new UnsupportedOperationException();
    }

    private Entry<K, V> findNext() {
      Entry<K, V> current = next;

      while (delegate.hasNext()) {
        Entry obj = delegate.next();
        if (!(obj.getValue() instanceof ObjectID)) {
          next = obj;
          return current;
        }
      }

      next = null;
      return current;
    }
  }

  private void localSizeIncrement() {
    localSize.incrementAndGet();
    MapSizeListener l = listener;
    if (l != null) {
      l.localSizeChanged(1);
    }
  }

  private void localSizeDecrement() {
    localSize.decrementAndGet();
    MapSizeListener l = listener;
    if (l != null) {
      l.localSizeChanged(-1);
    }
  }

  private void sizeIncrement() {
    MapSizeListener l = listener;
    if (l != null) {
      l.sizeChanged(1);
    }
  }

  private void sizeDecrement() {
    MapSizeListener l = listener;
    if (l != null) {
      l.sizeChanged(-1);
    }
  }

  public MapSizeListener registerMapSizeListener(final MapSizeListener newListener) {
    MapSizeListener old = listener;
    listener = newListener;
    return old;
  }
}