/*
 * Decompiled with CFR 0.152.
 */
package org.terracotta.persistence.sanskrit;

import java.io.BufferedInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Deque;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.terracotta.persistence.sanskrit.CopyUtils;
import org.terracotta.persistence.sanskrit.FileData;
import org.terracotta.persistence.sanskrit.FilesystemDirectory;
import org.terracotta.persistence.sanskrit.GroupingSpliterator;
import org.terracotta.persistence.sanskrit.HashChecker;
import org.terracotta.persistence.sanskrit.HashUtils;
import org.terracotta.persistence.sanskrit.JsonSanskritChangeVisitor;
import org.terracotta.persistence.sanskrit.JsonUtils;
import org.terracotta.persistence.sanskrit.MarkableLineParser;
import org.terracotta.persistence.sanskrit.MutableSanskritObject;
import org.terracotta.persistence.sanskrit.ObjectMapperSupplier;
import org.terracotta.persistence.sanskrit.Owner;
import org.terracotta.persistence.sanskrit.Sanskrit;
import org.terracotta.persistence.sanskrit.SanskritException;
import org.terracotta.persistence.sanskrit.SanskritObject;
import org.terracotta.persistence.sanskrit.SanskritObjectImpl;
import org.terracotta.persistence.sanskrit.UncheckedSanskritException;
import org.terracotta.persistence.sanskrit.change.SanskritChange;

public class SanskritImpl
implements Sanskrit {
    private static final Logger LOGGER = LoggerFactory.getLogger(SanskritImpl.class);
    private static final String APPEND_LOG_FILE = "append.log";
    private static final String HASH_0_FILE = "hash0";
    private static final String HASH_1_FILE = "hash1";
    private static final String FORMAT_VERSION = "format version: ";
    private final FilesystemDirectory filesystemDirectory;
    private final ObjectMapperSupplier objectMapperSupplier;
    private volatile MutableSanskritObject data;
    private volatile String lastHash;
    private volatile String nextHashFile;

    public SanskritImpl(FilesystemDirectory filesystemDirectory, ObjectMapperSupplier objectMapperSupplier) throws SanskritException {
        this.filesystemDirectory = filesystemDirectory;
        this.objectMapperSupplier = objectMapperSupplier;
        this.init();
    }

    private void init() throws SanskritException {
        this.lastHash = null;
        this.nextHashFile = null;
        this.data = this.newMutableSanskritObject();
        try {
            MutableSanskritObject result;
            HashChecker hashChecker;
            ArrayList<String> filesToDelete;
            block22: {
                filesToDelete = new ArrayList<String>();
                String hash0 = this.getHashFromFile(HASH_0_FILE, filesToDelete);
                String hash1 = this.getHashFromFile(HASH_1_FILE, filesToDelete);
                hashChecker = new HashChecker(hash0, hash1);
                result = this.newMutableSanskritObject();
                FileData appendLog = this.filesystemDirectory.getFileData(APPEND_LOG_FILE);
                Object object = null;
                try {
                    if (appendLog == null) break block22;
                    BufferedInputStream appendLogStream = new BufferedInputStream(Channels.newInputStream(appendLog));
                    MarkableLineParser parser = new MarkableLineParser(appendLogStream);
                    Stream<String> lines = parser.lines();
                    Stream<Deque<String>> records = this.groupByEmptyLines(lines);
                    AtomicReference error = new AtomicReference();
                    AtomicLong counter = new AtomicLong();
                    try {
                        records.forEach(record -> {
                            try {
                                String version;
                                String timestamp;
                                if (record.size() < 3) {
                                    throw new SanskritException("Invalid record");
                                }
                                long idx = counter.incrementAndGet();
                                String first = (String)record.removeFirst();
                                if (first.startsWith(FORMAT_VERSION)) {
                                    timestamp = (String)record.removeFirst();
                                    version = first.substring(16);
                                } else {
                                    timestamp = first;
                                    version = "";
                                }
                                String hash = (String)record.removeLast();
                                String json = String.join((CharSequence)"\n", record);
                                LOGGER.trace("init(): record {}: timestamp={}, version={}, hash={}, json={}", new Object[]{idx, timestamp, version, hash, json});
                                hash = this.checkHash(timestamp, json, hash);
                                String hashedHash = HashUtils.generateHash(hash);
                                boolean acceptRecord = hashChecker.check(hashedHash);
                                LOGGER.trace("init(): record {}: hash={}, hashedHash={}, acceptRecord={}", new Object[]{idx, hash, hashedHash, acceptRecord});
                                if (acceptRecord) {
                                    parser.mark();
                                    JsonUtils.parse(this.objectMapperSupplier, version, json, result);
                                    this.onNewRecord(timestamp, json);
                                    this.lastHash = hash;
                                }
                            }
                            catch (SanskritException e) {
                                error.set(e);
                                throw new UncheckedSanskritException(e);
                            }
                        });
                    }
                    catch (UncheckedSanskritException e) {
                        if (error.get() != null) {
                            throw (SanskritException)error.get();
                        }
                        throw e;
                    }
                    long mark = parser.getMark();
                    if (mark == 0L) {
                        filesToDelete.add(APPEND_LOG_FILE);
                        break block22;
                    }
                    try {
                        appendLog.truncate(mark);
                    }
                    catch (IOException e) {
                        throw new SanskritException(e);
                    }
                }
                catch (Throwable appendLogStream) {
                    object = appendLogStream;
                    throw appendLogStream;
                }
                finally {
                    if (appendLog != null) {
                        if (object != null) {
                            try {
                                appendLog.close();
                            }
                            catch (Throwable appendLogStream) {
                                ((Throwable)object).addSuppressed(appendLogStream);
                            }
                        } else {
                            appendLog.close();
                        }
                    }
                }
            }
            String hashToDelete = this.getHashToDelete(hashChecker);
            if (hashToDelete != null) {
                filesToDelete.add(hashToDelete);
            }
            LOGGER.trace("init(): filesToDelete={}", filesToDelete);
            for (String file : filesToDelete) {
                this.filesystemDirectory.delete(file);
            }
            this.nextHashFile = hashChecker.nextHashFile();
            this.data = result;
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
    }

    String getHashToDelete(HashChecker hashChecker) throws SanskritException {
        return hashChecker.done();
    }

    protected void onNewRecord(String timestamp, String json) throws SanskritException {
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String getHashFromFile(String hashFile, List<String> filesToDelete) throws SanskritException {
        LOGGER.trace("getHashFromFile({}, {})", (Object)hashFile, filesToDelete);
        ByteBuffer hashBuffer = ByteBuffer.allocate(40);
        try (FileData fileData = this.filesystemDirectory.getFileData(hashFile);){
            int read;
            if (fileData == null) {
                LOGGER.trace("getHashFromFile({}): <none>", (Object)hashFile);
                String string = null;
                return string;
            }
            if (fileData.size() > 40L) {
                throw new SanskritException("Hash file too long: " + hashFile);
            }
            if (fileData.size() < 40L) {
                filesToDelete.add(hashFile);
                LOGGER.trace("getHashFromFile({}): <none>, {}", (Object)hashFile, filesToDelete);
                String string = null;
                return string;
            }
            while (hashBuffer.hasRemaining() && (read = fileData.read(hashBuffer)) != -1) {
            }
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
        hashBuffer.flip();
        String hash = StandardCharsets.UTF_8.decode(hashBuffer).toString();
        LOGGER.trace("getHashFromFile({}): {}", (Object)hashFile, (Object)hash);
        return hash;
    }

    String checkHash(String timestamp, String json, String hash) throws SanskritException {
        String expectedHash = this.calculateHash(timestamp, json);
        if (!hash.equals(expectedHash)) {
            throw new SanskritException("Hash mismatch. Got: " + hash + ". Computed: " + expectedHash);
        }
        return hash;
    }

    String calculateHash(String timestamp, String json) {
        LOGGER.trace("calculateHash({}, {})", (Object)timestamp, (Object)json);
        if (this.lastHash == null) {
            return HashUtils.generateHash(timestamp, "\n", json);
        }
        return HashUtils.generateHash(this.lastHash, "\n", "\n", timestamp, "\n", json);
    }

    private Stream<Deque<String>> groupByEmptyLines(Stream<String> lines) {
        return StreamSupport.stream(new GroupingSpliterator(lines), false);
    }

    @Override
    public void close() {
    }

    @Override
    public String getString(String key) {
        return this.data.getString(key);
    }

    @Override
    public Long getLong(String key) {
        return this.data.getLong(key);
    }

    @Override
    public SanskritObject getObject(String key) {
        return CopyUtils.makeCopy(this.objectMapperSupplier, this.data.getObject(key));
    }

    @Override
    public void applyChange(SanskritChange change) throws SanskritException {
        change.accept(this.data);
        this.appendChange(change);
    }

    @Override
    public MutableSanskritObject newMutableSanskritObject() {
        return new SanskritObjectImpl(this.objectMapperSupplier);
    }

    @Override
    public void reset() throws SanskritException {
        try {
            this.filesystemDirectory.delete(HASH_0_FILE);
            this.filesystemDirectory.delete(HASH_1_FILE);
            this.filesystemDirectory.backup(APPEND_LOG_FILE);
            this.init();
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
    }

    private void appendChange(SanskritChange change) throws SanskritException {
        String json = this.changeAsJson(change);
        LOGGER.trace("appendChange(): {}", (Object)json);
        this.appendChange(json);
    }

    private String changeAsJson(SanskritChange change) throws SanskritException {
        JsonSanskritChangeVisitor visitor = new JsonSanskritChangeVisitor(this.objectMapperSupplier);
        change.accept(visitor);
        return visitor.getJson(null);
    }

    private void appendChange(String json) throws SanskritException {
        String timestamp = this.getTimestamp();
        this.appendRecord(timestamp, json);
    }

    void appendRecord(String timestamp, String json) throws SanskritException {
        LOGGER.trace("appendRecord({}, {})", (Object)timestamp, (Object)json);
        String hash = this.calculateHash(timestamp, json);
        this.appendEntry(FORMAT_VERSION + this.objectMapperSupplier.getCurrentVersion() + "\n" + timestamp + "\n" + json + "\n" + hash + "\n" + "\n", hash);
    }

    private String getTimestamp() {
        return Instant.now().toString();
    }

    private void appendEntry(String logEntry, String entryHash) throws SanskritException {
        LOGGER.trace("appendEntry({}, {})", (Object)logEntry, (Object)entryHash);
        String finalHash = HashUtils.generateHash(entryHash);
        LOGGER.trace("appendEntry({}): finalHash: {}", (Object)entryHash, (Object)finalHash);
        try (FileData appendLog = this.getAppendLogForAppend();
             FileData hashFile = this.createNewHashFile();){
            this.write(appendLog, logEntry);
            this.write(hashFile, finalHash);
            this.nextHashFile = this.flipHashFile();
            this.filesystemDirectory.delete(this.nextHashFile);
            this.lastHash = entryHash;
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
    }

    private String flipHashFile() {
        if (Objects.equals(this.nextHashFile, HASH_0_FILE)) {
            return HASH_1_FILE;
        }
        return HASH_0_FILE;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private FileData getAppendLogForAppend() throws SanskritException {
        try (Owner<FileData, IOException> appendLogOwner = Owner.own(this.filesystemDirectory.create(APPEND_LOG_FILE, true), IOException.class);){
            FileData appendLog = appendLogOwner.borrow();
            appendLog.position(appendLog.size());
            FileData fileData = appendLogOwner.release();
            return fileData;
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
    }

    private FileData createNewHashFile() throws SanskritException {
        try {
            return this.filesystemDirectory.create(this.nextHashFile, false);
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
    }

    private void write(FileData fileData, String text) throws SanskritException {
        try {
            ByteBuffer bytes = StandardCharsets.UTF_8.encode(text);
            while (bytes.hasRemaining()) {
                fileData.write(bytes);
            }
            fileData.force(false);
        }
        catch (IOException e) {
            throw new SanskritException(e);
        }
    }
}

