/*
 * Copyright 2017-present the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.springframework.kafka.support.serializer;

import java.util.Map;

import org.apache.kafka.common.serialization.Serde;
import org.jspecify.annotations.Nullable;
import tools.jackson.core.type.TypeReference;
import tools.jackson.databind.JavaType;
import tools.jackson.databind.json.JsonMapper;

import org.springframework.core.ResolvableType;
import org.springframework.kafka.support.JacksonMapperUtils;
import org.springframework.kafka.support.mapping.JacksonJavaTypeMapper;
import org.springframework.util.Assert;

/**
 * A {@link org.apache.kafka.common.serialization.Serde} that provides serialization and
 * deserialization in JSON format. Based on Jackson 3.
 * <p>
 * The implementation delegates to underlying {@link JacksonJsonSerializer} and
 * {@link JacksonJsonDeserializer} implementations.
 *
 * @param <T> target class for serialization/deserialization
 *
 * @author Marius Bogoevici
 * @author Elliot Kennedy
 * @author Gary Russell
 * @author Ivan Ponomarev
 * @author Soby Chacko
 *
 * @since 4.0
 */
public class JacksonJsonSerde<T> implements Serde<T> {

	private final JacksonJsonSerializer<T> jsonSerializer;

	private final JacksonJsonDeserializer<T> jsonDeserializer;

	public JacksonJsonSerde() {
		this((JavaType) null, JacksonMapperUtils.enhancedJsonMapper());
	}

	public JacksonJsonSerde(@Nullable Class<? super T> targetType) {
		this(targetType, JacksonMapperUtils.enhancedJsonMapper());
	}

	public JacksonJsonSerde(@Nullable TypeReference<? super T> targetType) {
		this(targetType, JacksonMapperUtils.enhancedJsonMapper());
	}

	public JacksonJsonSerde(@Nullable JavaType targetType) {
		this(targetType, JacksonMapperUtils.enhancedJsonMapper());
	}

	public JacksonJsonSerde(JsonMapper jsonMapper) {
		this((JavaType) null, jsonMapper);
	}

	public JacksonJsonSerde(@Nullable TypeReference<? super T> targetType, JsonMapper jsonMapper) {
		this(targetType == null ? null : jsonMapper.constructType(targetType.getType()), jsonMapper);
	}

	public JacksonJsonSerde(@Nullable Class<? super T> targetType, JsonMapper jsonMapper) {
		this(targetType == null ? null : jsonMapper.constructType(targetType), jsonMapper);
	}

	public JacksonJsonSerde(@Nullable JavaType targetTypeArg, @Nullable JsonMapper jsonMapperArg) {
		JsonMapper jsonMapper = jsonMapperArg == null ? JacksonMapperUtils.enhancedJsonMapper() : jsonMapperArg;
		JavaType actualJavaType;
		if (targetTypeArg != null) {
			actualJavaType = targetTypeArg;
		}
		else {
			Class<?> resolvedGeneric = ResolvableType.forClass(getClass()).getSuperType().resolveGeneric(0);
			actualJavaType = resolvedGeneric != null ? jsonMapper.constructType(resolvedGeneric) : null;
		}
		this.jsonSerializer = new JacksonJsonSerializer<>(actualJavaType, jsonMapper);
		this.jsonDeserializer = new JacksonJsonDeserializer<>(actualJavaType, jsonMapper);
	}

	public JacksonJsonSerde(JacksonJsonSerializer<T> jsonSerializer, JacksonJsonDeserializer<T> jsonDeserializer) {
		Assert.notNull(jsonSerializer, "'jsonSerializer' must not be null.");
		Assert.notNull(jsonDeserializer, "'jsonDeserializer' must not be null.");
		this.jsonSerializer = jsonSerializer;
		this.jsonDeserializer = jsonDeserializer;
	}

	@Override
	public void configure(Map<String, ?> configs, boolean isKey) {
		this.jsonSerializer.configure(configs, isKey);
		this.jsonDeserializer.configure(configs, isKey);
	}

	@Override
	public void close() {
		this.jsonSerializer.close();
		this.jsonDeserializer.close();
	}

	@Override
	public JacksonJsonSerializer<T> serializer() {
		return this.jsonSerializer;
	}

	@Override
	public JacksonJsonDeserializer<T> deserializer() {
		return this.jsonDeserializer;
	}

	/**
	 * Copies this serde with same configuration, except new target type is used.
	 * @param newTargetType type reference forced for serialization, and used as default for deserialization, not null
	 * @param <X> new deserialization result type and serialization source type
	 * @return new instance of serde with type changes
	 */
	public <X> JacksonJsonSerde<X> copyWithType(Class<? super X> newTargetType) {
		return new JacksonJsonSerde<>(this.jsonSerializer.copyWithType(newTargetType),
				this.jsonDeserializer.copyWithType(newTargetType));
	}

	/**
	 * Copies this serde with same configuration, except new target type reference is used.
	 * @param newTargetType type reference forced for serialization, and used as default for deserialization, not null
	 * @param <X> new deserialization result type and serialization source type
	 * @return new instance of serde with type changes
	 */
	public <X> JacksonJsonSerde<X> copyWithType(TypeReference<? super X> newTargetType) {
		return new JacksonJsonSerde<>(this.jsonSerializer.copyWithType(newTargetType),
				this.jsonDeserializer.copyWithType(newTargetType));
	}

	/**
	 * Copies this serde with same configuration, except new target java type is used.
	 * @param newTargetType java type forced for serialization, and used as default for deserialization, not null
	 * @param <X> new deserialization result type and serialization source type
	 * @return new instance of serde with type changes
	 */
	public <X> JacksonJsonSerde<X> copyWithType(JavaType newTargetType) {
		return new JacksonJsonSerde<>(this.jsonSerializer.copyWithType(newTargetType),
				this.jsonDeserializer.copyWithType(newTargetType));
	}

	// Fluent API

	/**
	 * Designate this Serde for serializing/deserializing keys (default is values).
	 * @return the serde.
	 */
	public JacksonJsonSerde<T> forKeys() {
		this.jsonSerializer.forKeys();
		this.jsonDeserializer.forKeys();
		return this;
	}

	/**
	 * Configure the serializer to not add type information.
	 * @return the serde.
	 */
	public JacksonJsonSerde<T> noTypeInfo() {
		this.jsonSerializer.noTypeInfo();
		return this;
	}

	/**
	 * Don't remove type information headers after deserialization.
	 * @return the serde.
	 */
	public JacksonJsonSerde<T> dontRemoveTypeHeaders() {
		this.jsonDeserializer.dontRemoveTypeHeaders();
		return this;
	}

	/**
	 * Ignore type information headers and use the configured target class.
	 * @return the serde.
	 */
	public JacksonJsonSerde<T> ignoreTypeHeaders() {
		this.jsonDeserializer.ignoreTypeHeaders();
		return this;
	}

	/**
	 * Use the supplied {@link JacksonJavaTypeMapper}.
	 * @param mapper the mapper.
	 * @return the serde.
	 */
	public JacksonJsonSerde<T> typeMapper(JacksonJavaTypeMapper mapper) {
		this.jsonSerializer.setTypeMapper(mapper);
		this.jsonDeserializer.setTypeMapper(mapper);
		return this;
	}

}
