/*
 * Decompiled with CFR 0.152.
 */
package org.springframework.data.redis.cache;

import java.util.Arrays;
import java.util.Set;
import org.springframework.cache.Cache;
import org.springframework.cache.support.SimpleValueWrapper;
import org.springframework.dao.DataAccessException;
import org.springframework.data.redis.cache.RedisCacheElement;
import org.springframework.data.redis.cache.RedisCacheKey;
import org.springframework.data.redis.connection.RedisConnection;
import org.springframework.data.redis.connection.ReturnType;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

public class RedisCache
implements Cache {
    private final RedisTemplate template;
    private final RedisCacheMetadata cacheMetadata;
    private final CacheValueAccessor cacheValueAccessor;

    public RedisCache(String name, byte[] prefix, RedisTemplate<? extends Object, ? extends Object> template, long expiration) {
        Assert.hasText((String)name, (String)"non-empty cache name is required");
        this.cacheMetadata = new RedisCacheMetadata(name, prefix);
        this.cacheMetadata.setDefaultExpiration(expiration);
        this.template = template;
        this.cacheValueAccessor = new CacheValueAccessor(template.getValueSerializer());
    }

    public <T> T get(Object key, Class<T> type) {
        Cache.ValueWrapper wrapper = this.get(key);
        return (T)(wrapper == null ? null : wrapper.get());
    }

    public Cache.ValueWrapper get(Object key) {
        return this.get(new RedisCacheKey(key).usePrefix(this.cacheMetadata.getKeyPrefix()).withKeySerializer(this.template.getKeySerializer()));
    }

    public RedisCacheElement get(RedisCacheKey cacheKey) {
        Assert.notNull((Object)cacheKey, (String)"CacheKey must not be null!");
        return this.template.execute(new AbstractRedisCacheCallback<RedisCacheElement>(new RedisCacheElement(cacheKey, null), this.cacheMetadata){

            @Override
            public RedisCacheElement doInRedis(RedisCacheElement element, RedisConnection connection) throws DataAccessException {
                byte[] bs = connection.get(element.getKeyBytes());
                byte[] value = RedisCache.this.template.getValueSerializer() != null ? (Object)RedisCache.this.template.getValueSerializer().deserialize(bs) : bs;
                return bs == null ? null : new RedisCacheElement(element.getKey(), value);
            }
        }, true);
    }

    public void put(Object key, Object value) {
        this.put(new RedisCacheElement(new RedisCacheKey(key).usePrefix(this.cacheMetadata.getKeyPrefix()).withKeySerializer(this.template.getKeySerializer()), value).expireAfter(this.cacheMetadata.getDefaultExpiration()));
    }

    public void put(RedisCacheElement element) {
        Assert.notNull((Object)((Object)element), (String)"Element must not be null!");
        this.template.execute(new RedisCachePutCallback(element, this.cacheValueAccessor, this.cacheMetadata), true);
    }

    public Cache.ValueWrapper putIfAbsent(Object key, Object value) {
        return this.putIfAbsent(new RedisCacheElement(new RedisCacheKey(key).usePrefix(this.cacheMetadata.getKeyPrefix()).withKeySerializer(this.template.getKeySerializer()), value).expireAfter(this.cacheMetadata.getDefaultExpiration()));
    }

    public Cache.ValueWrapper putIfAbsent(RedisCacheElement element) {
        Assert.notNull((Object)((Object)element), (String)"Element must not be null!");
        return this.toWrapper(this.template.execute(new RedisCachePutIfAbsentCallback(element, this.cacheValueAccessor, this.cacheMetadata), true));
    }

    public void evict(Object key) {
        this.evict(new RedisCacheElement(new RedisCacheKey(key).usePrefix(this.cacheMetadata.getKeyPrefix()).withKeySerializer(this.template.getKeySerializer()), null));
    }

    public void evict(RedisCacheElement element) {
        Assert.notNull((Object)((Object)element), (String)"Element must not be null!");
        this.template.execute(new RedisCacheEvictCallback(element, this.cacheMetadata));
    }

    public void clear() {
        this.template.execute(this.cacheMetadata.usesKeyPrefix() ? new RedisCacheCleanByPrefixCallback(this.cacheMetadata) : new RedisCacheCleanByKeysCallback(this.cacheMetadata), true);
    }

    public String getName() {
        return this.cacheMetadata.getCacheName();
    }

    public Object getNativeCache() {
        return this.template;
    }

    private Cache.ValueWrapper toWrapper(Object value) {
        return value != null ? new SimpleValueWrapper(value) : null;
    }

    static class RedisCachePutIfAbsentCallback
    extends AbstractRedisCacheCallback<Object> {
        private final CacheValueAccessor valueAccessor;

        public RedisCachePutIfAbsentCallback(RedisCacheElement element, CacheValueAccessor valueAccessor, RedisCacheMetadata metadata) {
            super(element, metadata);
            this.valueAccessor = valueAccessor;
        }

        @Override
        public Object doInRedis(RedisCacheElement element, RedisConnection connection) throws DataAccessException {
            this.waitForLock(connection);
            Object resultValue = this.put(element, connection);
            if (ObjectUtils.nullSafeEquals((Object)element.get(), (Object)resultValue)) {
                this.processKeyExpiration(element, connection);
                this.maintainKnownKeys(element, connection);
            }
            return resultValue;
        }

        private Object put(RedisCacheElement element, RedisConnection connection) {
            boolean valueWasSet = connection.setNX(element.getKeyBytes(), this.valueAccessor.convertToBytesIfNecessary(element.get()));
            return valueWasSet ? element.get() : this.valueAccessor.deserializeIfNecessary(connection.get(element.getKeyBytes()));
        }
    }

    static class RedisCachePutCallback
    extends AbstractRedisCacheCallback<Void> {
        private final CacheValueAccessor valueAccessor;

        public RedisCachePutCallback(RedisCacheElement element, CacheValueAccessor valueAccessor, RedisCacheMetadata metadata) {
            super(element, metadata);
            this.valueAccessor = valueAccessor;
        }

        @Override
        public Void doInRedis(RedisCacheElement element, RedisConnection connection) throws DataAccessException {
            connection.multi();
            connection.set(element.getKeyBytes(), this.valueAccessor.convertToBytesIfNecessary(element.get()));
            this.processKeyExpiration(element, connection);
            this.maintainKnownKeys(element, connection);
            connection.exec();
            return null;
        }
    }

    static class RedisCacheEvictCallback
    extends AbstractRedisCacheCallback<Void> {
        public RedisCacheEvictCallback(RedisCacheElement element, RedisCacheMetadata metadata) {
            super(element, metadata);
        }

        @Override
        public Void doInRedis(RedisCacheElement element, RedisConnection connection) throws DataAccessException {
            connection.del(new byte[][]{element.getKeyBytes()});
            this.cleanKnownKeys(element, connection);
            return null;
        }
    }

    static class RedisCacheCleanByPrefixCallback
    extends LockingRedisCacheCallback<Void> {
        private static final byte[] REMOVE_KEYS_BY_PATTERN_LUA = new StringRedisSerializer().serialize("local keys = redis.call('KEYS', ARGV[1]); local keysCount = table.getn(keys); if(keysCount > 0) then for _, key in ipairs(keys) do redis.call('del', key); end; end; return keysCount;");
        private static final byte[] WILD_CARD = new StringRedisSerializer().serialize("*");
        private final RedisCacheMetadata metadata;

        public RedisCacheCleanByPrefixCallback(RedisCacheMetadata metadata) {
            super(metadata);
            this.metadata = metadata;
        }

        @Override
        public Void doInLock(RedisConnection connection) throws DataAccessException {
            byte[] prefixToUse = Arrays.copyOf(this.metadata.getKeyPrefix(), this.metadata.getKeyPrefix().length + WILD_CARD.length);
            System.arraycopy(WILD_CARD, 0, prefixToUse, this.metadata.getKeyPrefix().length, WILD_CARD.length);
            connection.eval(REMOVE_KEYS_BY_PATTERN_LUA, ReturnType.INTEGER, 0, new byte[][]{prefixToUse});
            return null;
        }
    }

    static class RedisCacheCleanByKeysCallback
    extends LockingRedisCacheCallback<Void> {
        private static final int PAGE_SIZE = 128;
        private final RedisCacheMetadata metadata;

        RedisCacheCleanByKeysCallback(RedisCacheMetadata metadata) {
            super(metadata);
            this.metadata = metadata;
        }

        @Override
        public Void doInLock(RedisConnection connection) {
            int offset = 0;
            boolean finished = false;
            do {
                Set<byte[]> keys;
                finished = (keys = connection.zRange(this.metadata.getSetOfKnownKeysKey(), offset * 128, (offset + 1) * 128 - 1)).size() < 128;
                ++offset;
                if (keys.isEmpty()) continue;
                connection.del((byte[][])keys.toArray((T[])new byte[keys.size()][]));
            } while (!finished);
            connection.del(new byte[][]{this.metadata.getSetOfKnownKeysKey()});
            return null;
        }
    }

    static abstract class LockingRedisCacheCallback<T>
    implements RedisCallback<T> {
        private final RedisCacheMetadata metadata;

        public LockingRedisCacheCallback(RedisCacheMetadata metadata) {
            this.metadata = metadata;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public T doInRedis(RedisConnection connection) throws DataAccessException {
            T t;
            if (connection.exists(this.metadata.getCacheLockKey()).booleanValue()) {
                return null;
            }
            try {
                connection.set(this.metadata.getCacheLockKey(), this.metadata.getCacheLockKey());
                t = this.doInLock(connection);
            }
            catch (Throwable throwable) {
                connection.del(new byte[][]{this.metadata.getCacheLockKey()});
                throw throwable;
            }
            connection.del(new byte[][]{this.metadata.getCacheLockKey()});
            return t;
        }

        public abstract T doInLock(RedisConnection var1);
    }

    static abstract class AbstractRedisCacheCallback<T>
    implements RedisCallback<T> {
        private long WAIT_FOR_LOCK_TIMEOUT = 300L;
        private final RedisCacheElement element;
        private final RedisCacheMetadata cacheMetadata;

        public AbstractRedisCacheCallback(RedisCacheElement element, RedisCacheMetadata metadata) {
            this.element = element;
            this.cacheMetadata = metadata;
        }

        @Override
        public T doInRedis(RedisConnection connection) throws DataAccessException {
            this.waitForLock(connection);
            return this.doInRedis(this.element, connection);
        }

        public abstract T doInRedis(RedisCacheElement var1, RedisConnection var2) throws DataAccessException;

        protected void processKeyExpiration(RedisCacheElement element, RedisConnection connection) {
            if (!element.isEternal()) {
                connection.expire(element.getKeyBytes(), element.getTimeToLive());
            }
        }

        protected void maintainKnownKeys(RedisCacheElement element, RedisConnection connection) {
            if (!element.hasKeyPrefix()) {
                connection.zAdd(this.cacheMetadata.getSetOfKnownKeysKey(), 0.0, element.getKeyBytes());
                if (!element.isEternal()) {
                    connection.expire(this.cacheMetadata.getSetOfKnownKeysKey(), element.getTimeToLive());
                }
            }
        }

        protected void cleanKnownKeys(RedisCacheElement element, RedisConnection connection) {
            if (!element.hasKeyPrefix()) {
                connection.zRem(this.cacheMetadata.getSetOfKnownKeysKey(), new byte[][]{element.getKeyBytes()});
            }
        }

        protected boolean waitForLock(RedisConnection connection) {
            boolean retry;
            boolean foundLock = false;
            do {
                retry = false;
                if (!connection.exists(this.cacheMetadata.getCacheLockKey()).booleanValue()) continue;
                foundLock = true;
                try {
                    Thread.sleep(this.WAIT_FOR_LOCK_TIMEOUT);
                }
                catch (InterruptedException ex) {
                    Thread.currentThread().interrupt();
                }
                retry = true;
            } while (retry);
            return foundLock;
        }
    }

    static class CacheValueAccessor {
        private final RedisSerializer valueSerializer;

        CacheValueAccessor(RedisSerializer valueRedisSerializer) {
            this.valueSerializer = valueRedisSerializer;
        }

        byte[] convertToBytesIfNecessary(Object value) {
            if (this.valueSerializer == null && value instanceof byte[]) {
                return (byte[])value;
            }
            return this.valueSerializer.serialize(value);
        }

        Object deserializeIfNecessary(byte[] value) {
            if (this.valueSerializer != null) {
                return this.valueSerializer.deserialize(value);
            }
            return value;
        }
    }

    static class RedisCacheMetadata {
        private final String cacheName;
        private final byte[] keyPrefix;
        private final byte[] setOfKnownKeys;
        private final byte[] cacheLockName;
        private long defaultExpiration = 0L;

        public RedisCacheMetadata(String cacheName, byte[] keyPrefix) {
            Assert.hasText((String)cacheName, (String)"CacheName must not be null or empty!");
            this.cacheName = cacheName;
            this.keyPrefix = keyPrefix;
            StringRedisSerializer stringSerializer = new StringRedisSerializer();
            this.setOfKnownKeys = this.usesKeyPrefix() ? new byte[]{} : stringSerializer.serialize(cacheName + "~keys");
            this.cacheLockName = stringSerializer.serialize(cacheName + "~lock");
        }

        public boolean usesKeyPrefix() {
            return this.keyPrefix != null && this.keyPrefix.length > 0;
        }

        public byte[] getKeyPrefix() {
            return this.keyPrefix;
        }

        public byte[] getSetOfKnownKeysKey() {
            return this.setOfKnownKeys;
        }

        public byte[] getCacheLockKey() {
            return this.cacheLockName;
        }

        public String getCacheName() {
            return this.cacheName;
        }

        public void setDefaultExpiration(long seconds) {
            this.defaultExpiration = seconds;
        }

        public long getDefaultExpiration() {
            return this.defaultExpiration;
        }
    }
}

