/*
 * Decompiled with CFR 0.152.
 */
package org.sonarsource.scanner.api.internal.cache;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.attribute.FileAttribute;
import javax.annotation.CheckForNull;
import org.sonarsource.scanner.api.internal.cache.FileHashes;
import org.sonarsource.scanner.api.internal.cache.Logger;

public class FileCache {
    private static final int TEMP_DIR_ATTEMPTS = 10000;
    private final File dir;
    private final File tmpDir;
    private final FileHashes hashes;
    private final Logger logger;

    FileCache(File dir, FileHashes fileHashes, Logger logger) {
        this.hashes = fileHashes;
        this.logger = logger;
        this.dir = this.createDir(dir, "user cache");
        logger.info(String.format("User cache: %s", dir.getAbsolutePath()));
        this.tmpDir = this.createDir(new File(dir, "_tmp"), "temp dir");
    }

    static FileCache create(File dir, Logger logger) {
        return new FileCache(dir, new FileHashes(), logger);
    }

    public File getDir() {
        return this.dir;
    }

    @CheckForNull
    public File get(String filename, String hash) {
        File cachedFile = new File(new File(this.dir, hash), filename);
        if (cachedFile.exists()) {
            return cachedFile;
        }
        this.logger.debug(String.format("No file found in the cache with name %s and hash %s", filename, hash));
        return null;
    }

    public File get(String filename, String hash, Downloader downloader) {
        File hashDir = this.hashDir(hash);
        File targetFile = new File(hashDir, filename);
        if (!targetFile.exists()) {
            File tempFile = this.newTempFile();
            this.download(downloader, filename, tempFile);
            String downloadedHash = this.hashes.of(tempFile);
            if (!hash.equals(downloadedHash)) {
                throw new IllegalStateException("INVALID HASH: File " + tempFile.getAbsolutePath() + " was expected to have hash " + hash + " but was downloaded with hash " + downloadedHash);
            }
            FileCache.mkdirQuietly(hashDir);
            this.renameQuietly(tempFile, targetFile);
        }
        return targetFile;
    }

    private void download(Downloader downloader, String filename, File tempFile) {
        try {
            downloader.download(filename, tempFile);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to download " + filename + " to " + tempFile, e);
        }
    }

    private void renameQuietly(File sourceFile, File targetFile) {
        boolean rename = sourceFile.renameTo(targetFile);
        if (!rename && !targetFile.exists()) {
            this.logger.warn(String.format("Unable to rename %s to %s", sourceFile.getAbsolutePath(), targetFile.getAbsolutePath()));
            this.logger.warn("A copy/delete will be tempted but with no guarantee of atomicity");
            try {
                Files.move(sourceFile.toPath(), targetFile.toPath(), new CopyOption[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Fail to move " + sourceFile.getAbsolutePath() + " to " + targetFile, e);
            }
        }
    }

    private File hashDir(String hash) {
        return new File(this.dir, hash);
    }

    private static void mkdirQuietly(File hashDir) {
        try {
            Files.createDirectories(hashDir.toPath(), new FileAttribute[0]);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create cache directory: " + hashDir, e);
        }
    }

    private File newTempFile() {
        try {
            return File.createTempFile("fileCache", null, this.tmpDir);
        }
        catch (IOException e) {
            throw new IllegalStateException("Fail to create temp file in " + this.tmpDir, e);
        }
    }

    public File createTempDir() {
        String baseName = System.currentTimeMillis() + "-";
        for (int counter = 0; counter < 10000; ++counter) {
            File tempDir = new File(this.tmpDir, baseName + counter);
            if (!tempDir.mkdir()) continue;
            return tempDir;
        }
        throw new IllegalStateException("Failed to create directory in " + this.tmpDir);
    }

    private File createDir(File dir, String debugTitle) {
        if (!dir.isDirectory() || !dir.exists()) {
            this.logger.debug("Create : " + dir.getAbsolutePath());
            try {
                Files.createDirectories(dir.toPath(), new FileAttribute[0]);
            }
            catch (IOException e) {
                throw new IllegalStateException("Unable to create " + debugTitle + dir.getAbsolutePath(), e);
            }
        }
        return dir;
    }

    public static interface Downloader {
        public void download(String var1, File var2) throws IOException;
    }
}

