/*
 * Copyright © 2009 Benny Bottema (benny@bennybottema.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.simplejavamail.api.email;

import jakarta.activation.DataSource;
import lombok.extern.slf4j.Slf4j;
import lombok.val;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Objects;

import static org.simplejavamail.internal.util.MiscUtil.normalizeNewlines;

/**
 * Util class to get rid of some boilerplate code in the core classes. The equals code was needed to analyze junit test errors.
 * <p>
 * Initial equals code generated by IntelliJ, expanded to manually compare objects that don't override {@link Object#equals(Object)} (Recipient and DataSource implementations).
 */
@SuppressWarnings("SimplifiableIfStatement")
@Slf4j
public final class EqualsHelper {

    @SuppressWarnings("WeakerAccess")
    public static boolean equalsEmail(final Email email1, final Email email2) {
        if (!fieldIsEqual(email1.getFromRecipient(), email2.getFromRecipient(), "fromRecipient")) {
            return false;
        }
        if (!fieldIsEqual(email1.getId(), email2.getId(), "id")) {
            return false;
        }
        if (!fieldIsEqual(email1.getSentDate(), email2.getSentDate(), "sendDate")) {
            return false;
        }
        if (!isEqualRecipientList("ReplyTo", email1.getReplyToRecipients(), email2.getReplyToRecipients())) {
            return false;
        }
        if (!fieldIsEqual(email1.getBounceToRecipient(), email2.getBounceToRecipient(), "bounceToRecipient")) {
            return false;
        }
        if (!fieldIsEqual(email1.getPlainText(), email2.getPlainText(), "plainText")) {
            return false;
        }
        if (!fieldIsEqual(email1.getCalendarText(), email2.getCalendarText(), "calendarText")) {
            return false;
        }
        if (!fieldIsEqual(email1.getCalendarMethod(), email2.getCalendarMethod(), "calendarMethod")) {
            return false;
        }
        //noinspection SimplifiableConditionalExpression
        if (email1.getEmailToForward() != null ? email2.getEmailToForward() == null : email2.getEmailToForward() != null) {
            log.debug("Email unqual for emailToForward: {} vs {}", email1.getEmailToForward(), email2.getEmailToForward());
            return false;
        }
        if (email1.getHTMLText() != null ? !normalizeNewlines(email1.getHTMLText()).equals(normalizeNewlines(email2.getHTMLText())) : email2.getHTMLText() != null) {
            log.debug("Email unqual for HTML text: {} vs {}", email1.getHTMLText(), email2.getHTMLText());
            return false;
        }
        if (!fieldIsEqual(email1.getSubject(), email2.getSubject(), "subject")) {
            return false;
        }
        if (!isEqualRecipientList("TO/CC/BCC", email1.getRecipients(), email2.getRecipients())) {
            return false;
        }
        if (!isEqualRecipientList("Override receivers", email1.getOverrideReceivers(), email2.getOverrideReceivers())) {
            return false;
        }
        if (!email1.getEmbeddedImages().containsAll(email2.getEmbeddedImages()) || !email2.getEmbeddedImages().containsAll(email1.getEmbeddedImages())) {
            log.debug("Email unqual for embedded images: {} vs {}", email1.getEmbeddedImages(), email2.getEmbeddedImages());
            return false;
        }
        if (!email1.getAttachments().containsAll(email2.getAttachments()) || !email2.getAttachments().containsAll(email1.getAttachments())) {
            log.debug("Email unqual for attachments: {} vs {}", email1.getAttachments(), email2.getAttachments());
            return false;
        }
        if (!email1.getHeaders().equals(email2.getHeaders())) {
            log.debug("Email unqual for headers: {} vs {}", email1.getHeaders(), email2.getHeaders());
            return false;
        }
        if (!Objects.equals(email1.getUseDispositionNotificationTo(), email2.getUseDispositionNotificationTo())) {
            log.debug("Email unqual for useDispositionNotificationTo: {} vs {}", email1.getUseDispositionNotificationTo(), email2.getUseDispositionNotificationTo());
            return false;
        }
        if (!Objects.equals(email1.getUseReturnReceiptTo(), email2.getUseReturnReceiptTo())) {
            log.debug("Email unqual for useReturnReceiptTo: {} vs {}", email1.getUseReturnReceiptTo(), email2.getUseReturnReceiptTo());
            return false;
        }
        if (!fieldIsEqual(email1.getDispositionNotificationTo(), email2.getDispositionNotificationTo(), "dispositionNotificationTo")) {
            return false;
        }
        if (!fieldIsEqual(email1.getOriginalSmimeDetails(), email2.getOriginalSmimeDetails(), "originalSmimeDetails")) {
            return false;
        }
        if (!fieldIsEqual(email1.getSmimeSigningConfig(), email2.getSmimeSigningConfig(), "smimeSigningConfig")) {
            return false;
        }
        if (!fieldIsEqual(email1.getSmimeEncryptionConfig(), email2.getSmimeEncryptionConfig(), "smimeEncryptionConfig")) {
            return false;
        }
        return fieldIsEqual(email1.getReturnReceiptTo(), email2.getReturnReceiptTo(), "returnReceiptTo");
    }

    private static boolean isEqualRecipientList(final String recipientType, final List<Recipient> recipients, final List<Recipient> otherRecipients) {
        if (recipients.size() != otherRecipients.size()) {
            log.debug("Email unqual for [{}] recipients: {} vs {}", recipientType, recipients, otherRecipients);
            return false;
        }
        for (final Recipient otherRecipient : otherRecipients) {
            if (!containsRecipient(recipients, otherRecipient)) {
                log.debug("Email unqual for recipients: {} vs {}", recipients, otherRecipients);
                return false;
            }
        }
        return true;
    }

    private static boolean containsRecipient(final List<Recipient> recipients, @Nullable final Recipient otherRecipient) {
        for (final Recipient recipient : recipients) {
            if (Objects.equals(recipient, otherRecipient)) {
                return true;
            }
        }
        return false;
    }

    static boolean isEqualDataSource(@Nullable final DataSource a, @Nullable final DataSource b) {
        return (a == b) || (a != null && b != null &&
                fieldIsEqual(a.getName(), b.getName(), "name") &&
                fieldIsEqual(a.getContentType(), b.getContentType(), "contentType"));
    }

    private static boolean fieldIsEqual(final Object obj1, final Object obj2, final String name) {
        val isEqual = Objects.equals(obj1, obj2);
        if (!isEqual) log.debug("Email unqual for {}: {} vs {}", name, obj1, obj2);
        return isEqual;
    }
}