/*
 * Copyright 2017 University of Rostock
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package sessl.ml3

import org.jamesii.ml3.experiment.init.{ExpressionStateBuilder, IInitialStateBuilder}
import org.jamesii.ml3.experiment.{Job, Experiment => ML3Experiment}
import org.jamesii.ml3.model.maps.IValueMap
import org.jamesii.ml3.model.values._
import org.jamesii.ml3.simulator.simulators.ISimulator
import sessl.AbstractExperiment

import scala.collection.JavaConversions._
import scala.collection.mutable


/**
  * @author Tom Warnke
  */
class Experiment extends AbstractExperiment with SupportStopConditions with SupportReplicationConditions {

  /** default: use one core if not set otherwise.
    * Can be overwritten by the ParallelExecution trait */
  private[ml3] var parallelThreadsToUse = 1

  /** maps id of each run to the corresponding assignment id */
  private[this] val runToAssignment = scala.collection.mutable.Map[Int, Int]()

  /** maps assignment id to the number of issued runs */
  private[this] val assignmentToNumIssuedRuns = scala.collection.mutable.Map[Int, Int]()

  /** maps assignment id to the number of finished runs */
  private[this] val assignmentToNumFinishedRuns = scala.collection.mutable.Map[Int, Int]()

  private[this] val finishedAssignments = scala.collection.mutable.MutableList[Int]()

  private[this] var experiment: ML3Experiment = _

  private[this] var initialStateBuilder: Option[IInitialStateBuilder] = None

  private[this] var theStartTime: Option[Double] = None

  private[this] var errorOccured = false

  /** functions to be called when creating a job, mainly for observation */
  protected val instrumentations = scala.collection.mutable.Set.empty[(Int, Job, ISimulator) => Unit]

  /** map-type parameters */
  protected[this] var paramMaps: mutable.Map[String, IValueMap] =
    scala.collection.mutable.Map[String, IValueMap]()

  /**
    * Abstract method to create the basic setup (as configure is already used for checking whether the traits
    * properly called super.configure()).
    * In this function, the experiment should be initialized to conform to all elements provided in this class.
    */
  override protected def basicConfiguration(): Unit = {}

  /** All variables assignments specified */
  lazy val variableAssignments: scala.List[Predef.Map[String, IValue]] = {
    // create assignments from variables to scan and variables to set
    val assignments = createVariableSetups().map(assignment => assignment ++ fixedVariables)
    // convert assignment values to AnyRef (equivalent to Object)
    assignments.map(assignment => assignment.mapValues(v => parameterToML3Value(v)))
  }

  def parameterToML3Value(v: Any): IValue = v match {
    case i: Int => new IntValue(i)
    case r: Double => new RealValue(r)
    case s: String => new StringValue(s)
    case b: Boolean => new BoolValue(b)
  }

  /** Called to execute the experiment. */
  override protected[sessl] def executeExperiment(): Unit = {

    require(initialStateBuilder.isDefined, "No specification of the initial state given")

    if(theStartTime.isEmpty)
      theStartTime = Some(0)

    /* use standard simulator as default */
    if (simulators.isEmpty)
      simulator = FirstReactionMethod()

    experiment = new ML3Experiment(
      model,
      parallelThreadsToUse,
      simulatorStopCondition,
      initialStateBuilder.get,
      simulator.asInstanceOf[ML3Simulator].get,
      startTime)

    for (assignmentId <- variableAssignments.indices) {
      issueRuns(assignmentId, minReplicationNumber(assignmentId))
    }

    // wait until all runs are finished
    while (!this.isDone) {
      require(!errorOccured, "An error in the ML3 simulation package has occurred. Exiting...")
      Thread sleep 1000
    }

    logger.info("All jobs finished.")
  }

  /** Sends of a number of runs of an assignment to the experiment */
  def issueRuns(assignmentId: Int, number: Int) {

    for (i <- Range(0, number)) {

      // create unique run id and register it everywhere
      val runId = RunIdGenerator()
      addAssignmentForRun(runId, assignmentId, variableAssignments(assignmentId).toList)
      runToAssignment(runId) = assignmentId
      assignmentToNumIssuedRuns(assignmentId) = assignmentToNumIssuedRuns.getOrElse(assignmentId, 0) + 1

      // create new job with given assignment and give implementations for abstract methods
      val assignment = variableAssignments(assignmentId)
      val job: Job = new Job(experiment, assignment, paramMaps) {

        override def onFailure(t: Throwable): Unit = {
          errorOccured = true
          throw new RuntimeException(
            "Simulation run with configuration " + variableAssignments(assignmentId) + " threw an exception:", t)
        }

        override def onSuccess(): Unit = onFinishedRun(runId)

        override def instrument(simulator: ISimulator): Unit = {
          for (instrumentation <- instrumentations)
            instrumentation(runId, this, simulator)
        }
      }

      // send the job to the simulation package
      val success = experiment.addJob(job)

      require(success, "Failed to add job! Quitting...")

      logger.info("Submitted job nr " + runId + " with configuration " + variableAssignments(assignmentId))
    }
  }

  def getNumberOfFinishedRunsForAssignment(assignmentId: Int): Int = {
    val runIds = runToAssignment.toList.filter(_._2 == assignmentId).map(_._1)
    val runResults = runIds.map(this.results.forRun)
    runResults.size
  }

  def initializeWith(initialState: String): Unit = {
    initialStateBuilder = Some(new ExpressionStateBuilder(initialState))
  }

  def initializeWith(builder: IInitialStateBuilder): Unit = {
    initialStateBuilder = Some(builder)
  }

  def startTime_=(startTime: Double): Unit = {
    theStartTime = Some(startTime)
  }

  def startTime:Double = theStartTime.get

  /** Synchronized method to be called by returning simulation job threads. */
  def onFinishedRun(runId: Int): Unit = {

    this.synchronized {
      runDone(runId)
      logger.info("Run " + runId + " is finished.")

      val assignmentId = runToAssignment(runId)

      assignmentToNumFinishedRuns(assignmentId) = assignmentToNumFinishedRuns.getOrElse(assignmentId, 0) + 1

      // check if all issued replications of this assignment are done
      if (assignmentToNumFinishedRuns(assignmentId) == assignmentToNumIssuedRuns(assignmentId))
        onFinishedAssignment(assignmentId)

    }
  }

  def onFinishedAssignment(assignmentId: Int): Unit = {

    this.synchronized {
      if (enoughReplications(assignmentId)) {
        finishAssignment(assignmentId)
      } else {
        issueRuns(assignmentId, minReplicationNumber(assignmentId))
      }
    }
  }

  def finishAssignment(assignmentId: Int): Unit = {
    replicationsDone(assignmentId)

    finishedAssignments += assignmentId
    // check if all assignments are done
    if (finishedAssignments.size == variableAssignments.indices.size) {
      experiment.finish()
      experimentDone()
    }
  }

  /** creates unique run ids. Thread-safe */
  object RunIdGenerator {
    private[this] var runId = 0

    def apply(): Int = {
      this.synchronized {
        runId += 1
        runId
      }
    }
  }


}
