/*
 * Decompiled with CFR 0.152.
 */
package com.google.common.reflect;

import com.google.common.annotations.Beta;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Maps;
import com.google.common.collect.Ordering;
import com.google.common.reflect.Reflection;
import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.net.URLClassLoader;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.logging.Logger;
import javax.annotation.Nullable;

@Beta
public final class ClassPath {
    private static final Logger logger = Logger.getLogger(ClassPath.class.getName());
    private static final Splitter CLASS_PATH_ATTRIBUTE_SEPARATOR = Splitter.on(" ").omitEmptyStrings();
    private static final String CLASS_FILE_NAME_EXTENSION = ".class";
    private final ImmutableSet<ClassInfo> classes;

    private ClassPath(ImmutableSet<ClassInfo> classes) {
        this.classes = classes;
    }

    public static ClassPath from(ClassLoader classloader) throws IOException {
        ImmutableSortedSet.Builder<Object> builder = new ImmutableSortedSet.Builder<Object>(Ordering.usingToString());
        for (Map.Entry entry : ClassPath.getClassPathEntries(classloader).entrySet()) {
            builder.addAll(ClassPath.readClassesFrom((URI)entry.getKey(), (ClassLoader)entry.getValue()));
        }
        return new ClassPath(builder.build());
    }

    public ImmutableSet<ClassInfo> getTopLevelClasses() {
        return this.classes;
    }

    public ImmutableSet<ClassInfo> getTopLevelClasses(String packageName) {
        Preconditions.checkNotNull(packageName);
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (ClassInfo classInfo : this.classes) {
            if (!classInfo.getPackageName().equals(packageName)) continue;
            builder.add(classInfo);
        }
        return builder.build();
    }

    public ImmutableSet<ClassInfo> getTopLevelClassesRecursive(String packageName) {
        Preconditions.checkNotNull(packageName);
        String packagePrefix = packageName + '.';
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (ClassInfo classInfo : this.classes) {
            if (!classInfo.getName().startsWith(packagePrefix)) continue;
            builder.add(classInfo);
        }
        return builder.build();
    }

    @VisibleForTesting
    static ImmutableMap<URI, ClassLoader> getClassPathEntries(ClassLoader classloader) {
        LinkedHashMap<URI, ClassLoader> entries = Maps.newLinkedHashMap();
        ClassLoader parent = classloader.getParent();
        if (parent != null) {
            entries.putAll(ClassPath.getClassPathEntries(parent));
        }
        if (classloader instanceof URLClassLoader) {
            URLClassLoader urlClassLoader = (URLClassLoader)classloader;
            for (URL entry : urlClassLoader.getURLs()) {
                URI uri;
                try {
                    uri = entry.toURI();
                }
                catch (URISyntaxException e) {
                    throw new IllegalArgumentException(e);
                }
                if (entries.containsKey(uri)) continue;
                entries.put(uri, classloader);
            }
        }
        return ImmutableMap.copyOf(entries);
    }

    private static ImmutableSet<ClassInfo> readClassesFrom(URI uri, ClassLoader classloader) throws IOException {
        if (uri.getScheme().equals("file")) {
            return ClassPath.readClassesFrom(new File(uri), classloader);
        }
        return ImmutableSet.of();
    }

    @VisibleForTesting
    static ImmutableSet<ClassInfo> readClassesFrom(File file, ClassLoader classloader) throws IOException {
        if (!file.exists()) {
            return ImmutableSet.of();
        }
        if (file.isDirectory()) {
            return ClassPath.readClassesFromDirectory(file, classloader);
        }
        return ClassPath.readClassesFromJar(file, classloader);
    }

    private static ImmutableSet<ClassInfo> readClassesFromDirectory(File directory, ClassLoader classloader) {
        ImmutableSet.Builder<ClassInfo> builder = ImmutableSet.builder();
        ClassPath.readClassesFromDirectory(directory, classloader, "", builder);
        return builder.build();
    }

    private static void readClassesFromDirectory(File directory, ClassLoader classloader, String packagePrefix, ImmutableSet.Builder<ClassInfo> builder) {
        for (File f : directory.listFiles()) {
            String name = f.getName();
            if (f.isDirectory()) {
                ClassPath.readClassesFromDirectory(f, classloader, packagePrefix + name + ".", builder);
                continue;
            }
            if (!ClassPath.isTopLevelClassFile(name)) continue;
            String className = packagePrefix + ClassPath.getClassName(name);
            builder.add((Object)new ClassInfo(className, classloader));
        }
    }

    private static ImmutableSet<ClassInfo> readClassesFromJar(File file, ClassLoader classloader) throws IOException {
        ImmutableSet.Builder builder = ImmutableSet.builder();
        JarFile jarFile = new JarFile(file);
        for (URI uri : ClassPath.getClassPathFromManifest(file, jarFile.getManifest())) {
            builder.addAll(ClassPath.readClassesFrom(uri, classloader));
        }
        Enumeration<JarEntry> entries = jarFile.entries();
        while (entries.hasMoreElements()) {
            JarEntry entry = entries.nextElement();
            if (!ClassPath.isTopLevelClassFile(entry.getName())) continue;
            String className = ClassPath.getClassName(entry.getName().replace('/', '.'));
            builder.add(new ClassInfo(className, classloader));
        }
        return builder.build();
    }

    @VisibleForTesting
    static ImmutableSet<URI> getClassPathFromManifest(File jarFile, @Nullable Manifest manifest) {
        if (manifest == null) {
            return ImmutableSet.of();
        }
        ImmutableSet.Builder builder = ImmutableSet.builder();
        String classpathAttribute = manifest.getMainAttributes().getValue("Class-Path");
        if (classpathAttribute != null) {
            for (String path : CLASS_PATH_ATTRIBUTE_SEPARATOR.split(classpathAttribute)) {
                URI uri;
                try {
                    uri = ClassPath.getClassPathEntry(jarFile, path);
                }
                catch (URISyntaxException e) {
                    logger.warning("Invalid Class-Path entry: " + path);
                    continue;
                }
                builder.add(uri);
            }
        }
        return builder.build();
    }

    @VisibleForTesting
    static URI getClassPathEntry(File jarFile, String path) throws URISyntaxException {
        URI uri = new URI(path);
        if (uri.isAbsolute()) {
            return uri;
        }
        return new File(jarFile.getParentFile(), path.replace('/', File.separatorChar)).toURI();
    }

    @VisibleForTesting
    static boolean isTopLevelClassFile(String filename) {
        return filename.endsWith(CLASS_FILE_NAME_EXTENSION) && filename.indexOf(36) < 0;
    }

    @VisibleForTesting
    static String getClassName(String filename) {
        int classNameEnd = filename.length() - CLASS_FILE_NAME_EXTENSION.length();
        return filename.substring(0, classNameEnd);
    }

    public static final class ClassInfo {
        private final String className;
        private final ClassLoader loader;

        @VisibleForTesting
        ClassInfo(String className, ClassLoader loader) {
            this.className = Preconditions.checkNotNull(className);
            this.loader = Preconditions.checkNotNull(loader);
        }

        public String getPackageName() {
            return Reflection.getPackageName(this.className);
        }

        public String getSimpleName() {
            String packageName = this.getPackageName();
            if (packageName.isEmpty()) {
                return this.className;
            }
            return this.className.substring(packageName.length() + 1);
        }

        public String getName() {
            return this.className;
        }

        public Class<?> load() {
            try {
                return this.loader.loadClass(this.className);
            }
            catch (ClassNotFoundException e) {
                throw new IllegalStateException(e);
            }
        }

        public int hashCode() {
            return this.className.hashCode();
        }

        public boolean equals(Object obj) {
            if (obj instanceof ClassInfo) {
                ClassInfo that = (ClassInfo)obj;
                return this.className.equals(that.className) && this.loader == that.loader;
            }
            return false;
        }

        public String toString() {
            return this.className;
        }
    }
}

