package org.robolectric.shadows;

import android.content.Context;
import android.net.wifi.p2p.WifiP2pGroup;
import android.net.wifi.p2p.WifiP2pManager;
import android.net.wifi.p2p.WifiP2pManager.ActionListener;
import android.net.wifi.p2p.WifiP2pManager.Channel;
import android.os.Handler;
import android.os.Looper;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import org.robolectric.annotation.Implementation;
import org.robolectric.annotation.Implements;
import org.robolectric.annotation.Resetter;
import org.robolectric.util.ReflectionHelpers;

@Implements(WifiP2pManager.class)
public class ShadowWifiP2pManager {

  private static final int NO_FAILURE = -1;

  private static int listeningChannel;
  private static int operatingChannel;
  private static WifiP2pManager.GroupInfoListener groupInfoListener;
  private static Handler handler;
  private static int nextActionFailure = NO_FAILURE;
  private static final Map<Channel, WifiP2pGroup> p2pGroupmap = new HashMap<>();

  public int getListeningChannel() {
    return listeningChannel;
  }

  public int getOperatingChannel() {
    return operatingChannel;
  }

  public WifiP2pManager.GroupInfoListener getGroupInfoListener() {
    return groupInfoListener;
  }

  @Implementation
  protected void setWifiP2pChannels(
      Channel c, int listeningChannel, int operatingChannel, ActionListener al) {
    Objects.requireNonNull(c);
    Objects.requireNonNull(al);
    ShadowWifiP2pManager.listeningChannel = listeningChannel;
    ShadowWifiP2pManager.operatingChannel = operatingChannel;
  }

  @Implementation
  protected Channel initialize(
      Context context, Looper looper, WifiP2pManager.ChannelListener listener) {
    handler = new Handler(looper);
    return ReflectionHelpers.newInstance(Channel.class);
  }

  @Implementation
  protected void createGroup(Channel c, ActionListener al) {
    postActionListener(al);
  }

  private void postActionListener(final ActionListener al) {
    if (al == null) {
      return;
    }

    handler.post(
        () -> {
          if (nextActionFailure == -1) {
            al.onSuccess();
          } else {
            al.onFailure(nextActionFailure);
          }
          nextActionFailure = NO_FAILURE;
        });
  }

  @Implementation
  protected void requestGroupInfo(final Channel c, final WifiP2pManager.GroupInfoListener gl) {
    if (gl == null) {
      return;
    }

    handler.post(() -> gl.onGroupInfoAvailable(p2pGroupmap.get(c)));
  }

  @Implementation
  protected void removeGroup(Channel c, ActionListener al) {
    postActionListener(al);
  }

  public void setNextActionFailure(int nextActionFailure) {
    ShadowWifiP2pManager.nextActionFailure = nextActionFailure;
  }

  public void setGroupInfo(Channel channel, WifiP2pGroup wifiP2pGroup) {
    p2pGroupmap.put(channel, wifiP2pGroup);
  }

  @Resetter
  public static void reset() {
    listeningChannel = 0;
    operatingChannel = 0;
    groupInfoListener = null;
    handler = null;
    nextActionFailure = NO_FAILURE;
    p2pGroupmap.clear();
  }
}
