package org.refcodes.checkerboard.alt.javafx;

import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import org.refcodes.checkerboard.ChangePositionEvent;
import org.refcodes.checkerboard.Checkerboard;
import org.refcodes.checkerboard.CheckerboardEvent;
import org.refcodes.checkerboard.CheckerboardObserver;
import org.refcodes.checkerboard.CheckerboardViewer;
import org.refcodes.checkerboard.DraggabilityChangedEvent;
import org.refcodes.checkerboard.GraphicalCheckerboardViewer;
import org.refcodes.checkerboard.GridDimensionChangedEvent;
import org.refcodes.checkerboard.GridModeChangedEvent;
import org.refcodes.checkerboard.Player;
import org.refcodes.checkerboard.PlayerAddedEvent;
import org.refcodes.checkerboard.PlayerEvent;
import org.refcodes.checkerboard.PlayerRemovedEvent;
import org.refcodes.checkerboard.PositionChangedEvent;
import org.refcodes.checkerboard.StateChangedEvent;
import org.refcodes.checkerboard.ViewportDimensionChangedEvent;
import org.refcodes.checkerboard.ViewportOffsetChangedEvent;
import org.refcodes.checkerboard.VisibilityChangedEvent;
import org.refcodes.component.InitializeException;
import org.refcodes.exception.ExceptionUtility;
import org.refcodes.exception.VetoException;
import org.refcodes.exception.VetoException.VetoRuntimeException;
import org.refcodes.graphical.Dimension;
import org.refcodes.graphical.FieldDimension;
import org.refcodes.graphical.GridDimension;
import org.refcodes.graphical.GridMode;
import org.refcodes.graphical.MoveMode;
import org.refcodes.graphical.Offset;
import org.refcodes.graphical.Position;
import org.refcodes.graphical.ScaleMode;
import org.refcodes.graphical.ViewportDimension;
import org.refcodes.graphical.ViewportDimensionPropertyBuilder;
import org.refcodes.graphical.ViewportOffset;
import org.refcodes.graphical.ext.javafx.AbstractFxGridViewportPane;
import org.refcodes.graphical.ext.javafx.FxGridViewportPane;
import org.refcodes.mixin.Disposable;
import org.refcodes.observer.SubscribeEvent;
import org.refcodes.observer.UnsubscribeEvent;

import javafx.animation.FadeTransition;
import javafx.animation.TranslateTransition;
import javafx.application.Platform;
import javafx.beans.value.ChangeListener;
import javafx.beans.value.ObservableValue;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.geometry.Point2D;
import javafx.geometry.Pos;
import javafx.scene.Group;
import javafx.scene.Node;
import javafx.scene.input.MouseEvent;
import javafx.scene.layout.StackPane;
import javafx.util.Duration;

/**
 * The class {@link FxCheckerboardViewer} uses the {@link FxGridViewportPane} to
 * implement a {@link CheckerboardViewer}.
 * 
 * For scaling, this might be an idea:
 * "http://gillius.org/blog/2013/02/javafx-window-scaling-on-resize.html" The
 * Class FxCheckerboardViewerImpl.
 *
 * @param <P> The type representing a {@link Player}
 * @param <S> The type which's instances represent a {@link Player} state.
 */
public class FxCheckerboardViewer<P extends Player<P, S>, S> extends AbstractFxGridViewportPane<FxCheckerboardViewer<P, S>> implements GraphicalCheckerboardViewer<P, S, Node, FxSpriteFactory<S>, FxBackgroundFactory<P, S>, FxCheckerboardViewer<P, S>>, CheckerboardObserver<P, S> {

	// /////////////////////////////////////////////////////////////////////////
	// STATICS:
	// /////////////////////////////////////////////////////////////////////////

	private static Logger LOGGER = Logger.getLogger( FxCheckerboardViewer.class.getName() );

	// /////////////////////////////////////////////////////////////////////////
	// CONSTANTS:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// VARIABLES:
	// /////////////////////////////////////////////////////////////////////////

	private int _addPlayerDurationInMillis = 1000;
	private int _adjustPlayerDurationInMillis = 50;
	private Node _backgroundNode = null;
	private double _bordersH = Double.NaN;
	private double _bordersV = Double.NaN;
	private int _changePlayerStateInMillis = 300;
	private Checkerboard<P, S> _checkerboard;
	private Group _checkers = new Group();
	private int _initGridInMillis = 1500;
	private int _movePlayerDurationInMillis = 150;
	private Map<P, ClickPlayerEventHandler> _playerToClickEventHandler = new HashMap<>();
	private Map<P, DragPlayerEventHandler> _playerToDragEventHandler = new HashMap<>();
	private Map<P, Node> _playerToSprite = new HashMap<>();
	private int _playerVisibilityDurationInMillis = 100;
	private int _removePlayerDurationInMillis = 1000;
	private int _resizeGridInMillis = 500;
	private double _windowDecorationH = Double.NaN;
	private double _windowDecorationV = Double.NaN;
	private ViewportDimensionPropertyBuilder _minViewportDimension = new ViewportDimensionPropertyBuilder( -1, -1 );
	private ScaleMode _scaleMode = ScaleMode.GRID;
	private FxSpriteFactory<S> _spriteFactory = null;
	private FxBackgroundFactory<P, S> _backgroundFactory = null;

	// -------------------------------------------------------------------------
	// WINDOW HEIGHT CHANGE HANDLER:
	// -------------------------------------------------------------------------

	private ChangeListener<Number> _onWindowHeightChangedEventHandler = new ChangeListener<Number>() {

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void changed( ObservableValue<? extends Number> observable, Number aOldValue, Number aNewValue ) {
			// _lastResizeTimeInMs = System.currentTimeMillis();
			// if ( Math.abs( aNewValue.doubleValue() - aOldValue.doubleValue() ) > 10 ) return;

			// -----------------------------------------------------------------
			// Calculate the sum of the H-borders:
			// -----------------------------------------------------------------
			if ( Double.isNaN( _bordersV ) && !Double.isNaN( aOldValue.doubleValue() ) ) {
				initBordersV( aOldValue.doubleValue() );
				initMinStageHeight();
			}
			// -----------------------------------------------------------------

			switch ( getScaleMode() ) {
			case GRID:
				int theNewViewportHeight = toScaledViewportDimension( aNewValue.doubleValue(), getViewportHeight(), getFieldHeight(), getFieldGap(), _bordersV + _windowDecorationV );
				if ( theNewViewportHeight != -1 ) {
					LOGGER.log( Level.FINE, "Viewport height changed to := " + theNewViewportHeight );
					setViewportHeight( theNewViewportHeight );
				}
				break;
			case FIELDS:
				int theNewFieldHeight = toScaledFieldDimension( aNewValue.doubleValue(), getViewportHeight(), getFieldHeight(), getFieldGap(), _bordersV + _windowDecorationV );
				if ( theNewFieldHeight != -1 ) {
					LOGGER.log( Level.FINE, "Field height changed to := " + theNewFieldHeight );
					setFieldHeight( theNewFieldHeight );
				}
				break;
			case NONE:
				break;
			}
		}
	};

	// -------------------------------------------------------------------------
	// WINDOW WIDTH CHANGE HANDLER:
	// -------------------------------------------------------------------------

	private ChangeListener<Number> _onWindowWidthChangedEventHandler = new ChangeListener<Number>() {

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void changed( ObservableValue<? extends Number> observable, Number aOldValue, Number aNewValue ) {

			// _lastResizeTimeInMs = System.currentTimeMillis();
			// if ( Math.abs( aNewValue.doubleValue() - aOldValue.doubleValue() ) > 10 ) return;

			// -----------------------------------------------------------------
			// Calculate the sum of the H-borders:
			// -----------------------------------------------------------------
			if ( Double.isNaN( _bordersH ) && !Double.isNaN( aOldValue.doubleValue() ) ) {
				initBordersH( aOldValue.doubleValue() );
				initMinStageWidth();
			}
			// -----------------------------------------------------------------

			switch ( getScaleMode() ) {
			case GRID:
				int theNewViewportWidth = toScaledViewportDimension( aNewValue.doubleValue(), getViewportWidth(), getFieldWidth(), getFieldGap(), _bordersH + _windowDecorationH );
				if ( theNewViewportWidth != -1 ) {
					LOGGER.log( Level.FINE, "Viewport width changed to := " + theNewViewportWidth );
					setViewportWidth( theNewViewportWidth );
				}
				break;
			case FIELDS:
				int theNewFieldWidth = toScaledFieldDimension( aNewValue.doubleValue(), getViewportWidth(), getFieldWidth(), getFieldGap(), _bordersH + _windowDecorationH );
				if ( theNewFieldWidth != -1 ) {
					LOGGER.log( Level.FINE, "Field width changed to := " + theNewFieldWidth );
					setFieldWidth( theNewFieldWidth );
				}
				break;
			case NONE:
				break;
			}
		}
	};

	// /////////////////////////////////////////////////////////////////////////
	// CONSTRUCTORS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Instantiates a new {@link FxCheckerboardViewer} instance. ATTENTION: As
	 * no {@link FxSpriteFactory} is provided to this constructor, no sprites
	 * can be fabricated when players are added until the
	 * {@link #setSpriteFactory(org.refcodes.checkerboard.SpriteFactory)} has
	 * been set!
	 *
	 * @param aCheckerboard the {@link Checkerboard} to be viewed.
	 */
	public FxCheckerboardViewer( Checkerboard<P, S> aCheckerboard ) {
		_checkerboard = aCheckerboard;
		aCheckerboard.subscribeObserver( this );
		widthProperty().addListener( _onWindowWidthChangedEventHandler );
		heightProperty().addListener( _onWindowHeightChangedEventHandler );
	}

	// /////////////////////////////////////////////////////////////////////////
	// INJECTION:
	// /////////////////////////////////////////////////////////////////////////

	// /////////////////////////////////////////////////////////////////////////
	// METHODS:
	// /////////////////////////////////////////////////////////////////////////

	public void destroy() {
		_checkerboard.destroy();
	}

	/**
	 * Gets the adds the player duration in milliseconds.
	 *
	 * @return the adds the player duration in milliseconds
	 */
	public int getAddPlayerDurationInMillis() {
		return _addPlayerDurationInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxBackgroundFactory<P, S> getBackgroundFactory() {
		return _backgroundFactory;
	}

	/**
	 * Gets the change player state in millis.
	 *
	 * @return the change player state in millis
	 */
	public int getChangePlayerStateInMillis() {
		return _changePlayerStateInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getContainerHeight() {
		return getFieldHeight() * getViewportHeight() + (getFieldGap() * (getViewportHeight() - 1)) + ((_checkerboard.getGridMode() == GridMode.CLOSED) ? (getFieldGap() * 2) : 0);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getContainerWidth() {
		return getFieldWidth() * getViewportWidth() + (getFieldGap() * (getViewportWidth() - 1)) + ((_checkerboard.getGridMode() == GridMode.CLOSED) ? (getFieldGap() * 2) : 0);
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getGridHeight() {
		return _checkerboard.getGridHeight();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public GridMode getGridMode() {
		return _checkerboard.getGridMode();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int getGridWidth() {
		return _checkerboard.getGridWidth();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ViewportDimension getMinViewportDimension() {
		return _minViewportDimension;
	}

	/**
	 * Gets the move player duration in milliseconds.
	 *
	 * @return the move player duration in milliseconds
	 */
	public int getMovePlayerDurationInMillis() {
		return _movePlayerDurationInMillis;
	}

	/**
	 * Gets the remove the player duration in milliseconds.
	 *
	 * @return the player remove duration in milliseconds
	 */
	public int getRemovePlayerDurationInMillis() {
		return _removePlayerDurationInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ScaleMode getScaleMode() {
		return _scaleMode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxSpriteFactory<S> getSpriteFactory() {
		return _spriteFactory;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public synchronized void initialize() throws InitializeException {

		// Add players |-->
		for ( P ePlayer : _checkerboard.getPlayers() ) {
			if ( !_playerToSprite.containsKey( ePlayer ) ) {
				fxAddPlayer( ePlayer, getAddPlayerDurationInMillis() );
			}
		}
		// Add players <--|

		if ( Platform.isFxApplicationThread() ) {
			fxInitialize();
		}
		else {
			Platform.runLater( new Runnable() {
				@Override
				public void run() {
					try {
						fxInitialize();
					}
					catch ( InitializeException e ) {
						LOGGER.log( Level.SEVERE, "Exception during initialization: " + ExceptionUtility.toMessage( e ), e );
					}
				}

			} );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onChangePositionEvent( ChangePositionEvent<P> aEvent, Checkerboard<P, S> aCheckerboard ) throws VetoException {
		LOGGER.log( Level.FINE, aEvent.toString() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onCheckerboardEvent( CheckerboardEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onDraggabilityChangedEvent( DraggabilityChangedEvent<P> aEvent, Checkerboard<P, S> aCheckerboard ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxPlayerDraggability( aEvent.getSource() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onGridDimensionChangedEvent( GridDimensionChangedEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxResizeGrid( aEvent, aEvent.getPrecedingGridDimension(), _resizeGridInMillis );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onGridModeChangedEvent( GridModeChangedEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		setGridMode( aEvent.getGridMode() );
		initMinStageWidth();
		initMinStageHeight();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onPlayerAddedEvent( PlayerAddedEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxAddPlayer( aEvent.getPlayer(), _addPlayerDurationInMillis );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onPlayerEvent( PlayerEvent<P> aEvent, Checkerboard<P, S> aCheckerboard ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onPlayerRemovedEvent( PlayerRemovedEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxRemovePlayer( aEvent.getPlayer(), _removePlayerDurationInMillis );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onPositionChangedEvent( PositionChangedEvent<P> aEvent, Checkerboard<P, S> aCheckerboard ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxMovePlayer( aEvent.getSource(), aEvent.getPrecedingPosition(), _movePlayerDurationInMillis );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onStateChangedEvent( StateChangedEvent<P, S> aEvent, Checkerboard<P, S> aCheckerboard ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		Node theSprite = _spriteFactory.createInstance( aEvent.getSource().getStatus(), this );
		Node prevSprite = _playerToSprite.get( aEvent.getSource() );
		if ( theSprite != prevSprite ) {
			fxRemovePlayer( aEvent.getSource(), _changePlayerStateInMillis / 2 );
			fxAddPlayer( aEvent.getSource(), _changePlayerStateInMillis / 2 );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onSubscribe( SubscribeEvent<Checkerboard<P, S>> aSubscribeEvent ) {
		_checkerboard = aSubscribeEvent.getSource();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onUnsubscribe( UnsubscribeEvent<Checkerboard<P, S>> aUnsubscribeEvent ) {
		if ( _checkerboard == aUnsubscribeEvent.getSource() ) _checkerboard = null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onViewportDimensionChangedEvent( ViewportDimensionChangedEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxResizeViewport( aEvent, aEvent.getPrecedingViewportDimension(), aEvent.getViewportOffset(), _resizeGridInMillis );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onViewportOffsetChangedEvent( ViewportOffsetChangedEvent<P, S> aEvent ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		setViewportOffset( this );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void onVisibilityChangedEvent( VisibilityChangedEvent<P> aEvent, Checkerboard<P, S> aCheckerboard ) {
		LOGGER.log( Level.FINE, aEvent.toString() );
		fxPlayerVisibility( aEvent.getSource(), _playerVisibilityDurationInMillis );
	}

	/**
	 * Sets the adds the player duration in milliseconds.
	 *
	 * @param aAddPlayerDurationInMillis the new adds the player duration in
	 *        milliseconds
	 */
	public void setAddPlayerDurationInMillis( int aAddPlayerDurationInMillis ) {
		_addPlayerDurationInMillis = aAddPlayerDurationInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setBackgroundFactory( FxBackgroundFactory<P, S> aBackgroundFactory ) {
		_backgroundFactory = aBackgroundFactory;
	}

	/**
	 * Sets the change player state in millis.
	 *
	 * @param aChangePlayerStateInMillis the new change player state in millis
	 */
	public void setChangePlayerStateInMillis( int aChangePlayerStateInMillis ) {
		_changePlayerStateInMillis = aChangePlayerStateInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFieldDimension( Dimension aDimension ) {
		setFieldDimension( aDimension.getWidth(), aDimension.getHeight() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setFieldDimension( FieldDimension aField ) {
		setFieldDimension( aField.getFieldWidth(), aField.getFieldHeight() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMinViewportDimension( Dimension aDimension ) {
		_minViewportDimension.setViewportDimension( aDimension );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMinViewportDimension( int aWidth, int aHeight ) {
		_minViewportDimension.setViewportDimension( aWidth, aHeight );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setMinViewportDimension( ViewportDimension aDimension ) {
		_minViewportDimension.setViewportDimension( aDimension );
	}

	/**
	 * Sets the move player duration in milliseconds.
	 *
	 * @param aMovePlayerDurationInMillis the new move player duration in
	 *        milliseconds
	 */
	public void setMovePlayerDurationInMillis( int aMovePlayerDurationInMillis ) {
		_movePlayerDurationInMillis = aMovePlayerDurationInMillis;
	}

	/**
	 * Sets the removes the player duration in millis.
	 *
	 * @param aRemovePlayerDurationInMillis the new removes the player duration
	 *        in millis
	 */
	public void setRemovePlayerDurationInMillis( int aRemovePlayerDurationInMillis ) {
		_removePlayerDurationInMillis = aRemovePlayerDurationInMillis;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setScaleMode( ScaleMode aMode ) {
		_scaleMode = aMode;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setSpriteFactory( FxSpriteFactory<S> aSpriteFactory ) {
		_spriteFactory = aSpriteFactory;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setViewportDimension( Dimension aDimension ) {
		setViewportDimension( aDimension.getWidth(), aDimension.getHeight() );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void setViewportOffset( int aPosX, int aPosY ) {
		if ( aPosX != getViewportOffsetX() || aPosY != getViewportOffsetY() ) {
			ViewportOffsetChangedEvent<P, S> theEvent = new ViewportOffsetChangedEvent<P, S>( aPosX, aPosY, getViewportOffsetX(), getViewportOffsetY(), this );
			super.setViewportOffset( aPosX, aPosY );
			onViewportOffsetChangedEvent( theEvent );
		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void show() {
		setVisible( true );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void hide() {
		setVisible( false );
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int toTotalHeight() {
		int theTotalHeight = getFieldHeight() * getViewportHeight() + (getFieldGap() * (getViewportHeight() - 1));
		if ( getGridMode() == GridMode.CLOSED ) {
			theTotalHeight += (getFieldGap() * 2);
		}
		return theTotalHeight;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public int toTotalWidth() {
		int theTotalWidth = getFieldWidth() * getViewportWidth() + (getFieldGap() * (getViewportWidth() - 1));
		if ( getGridMode() == GridMode.CLOSED ) {
			theTotalWidth += (getFieldGap() * 2);
		}
		return theTotalWidth;
	}

	/**
	 * With add player duration in millis.
	 *
	 * @param aAddPlayerDurationInMillis the add player duration in millis
	 * 
	 * @return the fx checkerboard viewer
	 */
	public FxCheckerboardViewer<P, S> withAddPlayerDurationInMillis( int aAddPlayerDurationInMillis ) {
		setAddPlayerDurationInMillis( aAddPlayerDurationInMillis );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withBackgroundFactory( FxBackgroundFactory<P, S> aBackgroundFactory ) {
		setBackgroundFactory( aBackgroundFactory );
		return this;
	}

	/**
	 * With change player state in millis.
	 *
	 * @param aChangePlayerStateInMillis the change player state in millis
	 * 
	 * @return the fx checkerboard viewer
	 */
	public FxCheckerboardViewer<P, S> withChangePlayerStateInMillis( int aChangePlayerStateInMillis ) {
		setChangePlayerStateInMillis( aChangePlayerStateInMillis );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withContent( Node aContent ) {
		setContent( aContent );
		return this;

	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withDragOpacity( double aOpacity ) {
		setDragOpacity( aOpacity );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldDimension( Dimension aDimension ) {
		setFieldDimension( aDimension );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldDimension( FieldDimension aField ) {
		setFieldDimension( aField );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldDimension( int aFieldWidth, int aFieldHeight ) {
		setFieldDimension( aFieldWidth, aFieldHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldDimension( int aFieldWidth, int aFieldHeight, int aGap ) {
		setFieldDimension( aFieldWidth, aFieldHeight, aGap );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldGap( int aFieldGap ) {
		setFieldGap( aFieldGap );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldHeight( int aHeight ) {
		setFieldHeight( aHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withFieldWidth( int aWidth ) {
		setFieldWidth( aWidth );
		return this;
	}

	@Override
	public FxCheckerboardViewer<P, S> withGridMode( GridMode aGridMode ) {
		setGridMode( aGridMode );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withHide() {
		hide();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withInitialize() throws InitializeException {
		initialize();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withMinViewportDimension( Dimension aDimension ) {
		setMinViewportDimension( aDimension );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withMinViewportDimension( int aWidth, int aHeight ) {
		setMinViewportDimension( aWidth, aHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withMinViewportDimension( ViewportDimension aDimension ) {
		setMinViewportDimension( aDimension );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withMoveMode( MoveMode aMode ) {
		setMoveMode( aMode );
		return this;
	}

	/**
	 * With move player duration in milliseconds.
	 *
	 * @param aMovePlayerDurationInMillis the move player duration in
	 *        milliseconds
	 * 
	 * @return the fx checkerboard viewer
	 */
	public FxCheckerboardViewer<P, S> withMovePlayerDurationInMillis( int aMovePlayerDurationInMillis ) {
		setMovePlayerDurationInMillis( aMovePlayerDurationInMillis );
		return this;
	}

	/**
	 * With remove player duration in millis.
	 *
	 * @param aRemovePlayerDurationInMillis the remove player duration in millis
	 * 
	 * @return the fx checkerboard viewer
	 */
	public FxCheckerboardViewer<P, S> withRemovePlayerDurationInMillis( int aRemovePlayerDurationInMillis ) {
		setRemovePlayerDurationInMillis( aRemovePlayerDurationInMillis );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withScaleMode( ScaleMode aMode ) {
		setScaleMode( aMode );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withShow() {
		show();
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withSpriteFactory( FxSpriteFactory<S> aSpriteFactory ) {
		setSpriteFactory( aSpriteFactory );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportDimension( Dimension aDimension ) {
		setViewportDimension( aDimension );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportDimension( int aWidth, int aHeight ) {
		setViewportDimension( aWidth, aHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportDimension( ViewportDimension aGridDimension ) {
		setViewportDimension( aGridDimension );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportHeight( int aGridHeight ) {
		setViewportHeight( aGridHeight );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportOffset( int aPosX, int aPosY ) {
		setViewportOffset( aPosX, aPosY );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportOffset( Offset aOffset ) {
		setViewportOffset( aOffset );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportOffset( Position aOffset ) {
		setViewportOffset( aOffset );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportOffset( ViewportOffset aOffset ) {
		setViewportOffset( aOffset );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportOffsetX( int aPosX ) {
		setViewportOffsetX( aPosX );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportOffsetY( int aPosY ) {
		setViewportOffsetY( aPosY );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withViewportWidth( int aGridWidth ) {
		setViewportWidth( aGridWidth );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public FxCheckerboardViewer<P, S> withVisible( boolean isVisible ) {
		setVisible( isVisible );
		return this;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public String toString() {
		return _checkerboard.toString();
	}

	// /////////////////////////////////////////////////////////////////////////
	// HOOKS:
	// /////////////////////////////////////////////////////////////////////////

	/**
	 * Hide players.
	 *
	 * @param aDurationInMillis the duration in milliseconds
	 */
	protected void hidePlayers( int aDurationInMillis ) {
		for ( P ePlayer : _checkerboard.getPlayers() ) {
			fxHidePlayer( ePlayer, aDurationInMillis );
		}
	}

	/**
	 * Reset players.
	 *
	 * @param aDurationInMillis the duration in milliseconds
	 */
	protected void resetPlayers( int aDurationInMillis ) {
		for ( P ePlayer : _checkerboard.getPlayers() ) {
			fxResetPlayer( ePlayer, aDurationInMillis );
		}
	}

	/**
	 * Scale players.
	 *
	 * @param aFieldDimension the field dimension
	 * @param aPrecedingFieldDimension the preceding field dimension
	 */
	protected void scalePlayers( FieldDimension aFieldDimension, FieldDimension aPrecedingFieldDimension ) {
		for ( P ePlayer : _checkerboard.getPlayers() ) {
			fxScalePlayer( ePlayer, aFieldDimension, aPrecedingFieldDimension );
		}
	}

	// /////////////////////////////////////////////////////////////////////////
	// HELPER:
	// /////////////////////////////////////////////////////////////////////////

	private void fxAddPlayer( P aPlayer, int aDurationInMillis ) {
		Node theSprite = _spriteFactory.createInstance( aPlayer.getStatus(), this );
		// State-Change on same Instance |-->
		//	Node prevSprite = _playerToSprite.get( aPlayer );
		//	if ( theSprite == prevSprite ) {
		//		return;
		//	}
		// State-Change on same Instance <--|
		fxAddPlayer( aPlayer, aDurationInMillis, theSprite );
	}

	private void fxAddPlayer( P aPlayer, int aDurationInMillis, Node aSprite ) {
		synchronized ( _playerToSprite ) {
			_playerToSprite.put( aPlayer, aSprite );
		}
		// Mouse events |-->
		if ( aPlayer.isDraggable() ) {
			DragPlayerEventHandler theDragEventHandler = new DragPlayerEventHandler( aPlayer, aSprite );
			_playerToDragEventHandler.put( aPlayer, theDragEventHandler );
		}
		ClickPlayerEventHandler theClickEventHandler = new ClickPlayerEventHandler( aPlayer, aSprite );
		_playerToClickEventHandler.put( aPlayer, theClickEventHandler );
		// Mouse events <--|

		Point2D thePoint = aSprite.localToParent( 0, 0 );
		aSprite.setLayoutX( -(thePoint.getX()) );
		aSprite.setLayoutY( -(thePoint.getY()) );
		aSprite.setTranslateX( toPixelPositionX( aPlayer ) );
		aSprite.setTranslateY( toPixelPositionY( aPlayer ) );

		Runnable theRunner = new Runnable() {
			@Override
			public void run() {
				if ( isViewport( aPlayer ) ) {
					_checkers.getChildren().add( aSprite );
					if ( aPlayer.isVisible() ) {
						if ( aDurationInMillis > 0 ) {
							aSprite.setOpacity( 0 );
							FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ), aSprite );
							theTransition.setFromValue( 0 );
							theTransition.setToValue( 1 );
							theTransition.setCycleCount( 1 );
							theTransition.setAutoReverse( false );
							theTransition.play();
						}
						else {
							aSprite.setOpacity( 1 );
							aSprite.setVisible( true );
						}
					}
				}
				else {
					aSprite.setVisible( true );
					_checkers.getChildren().add( aSprite );
				}
			}
		};
		if ( Platform.isFxApplicationThread() ) {
			theRunner.run();
		}
		else {
			Platform.runLater( theRunner );
		}
	}

	private void fxHidePlayer( P aPlayer, int aDurationInMillis ) {
		Node theSprite;
		synchronized ( _playerToSprite ) {
			theSprite = _playerToSprite.get( aPlayer );
		}
		if ( theSprite != null ) {
			if ( aDurationInMillis == 0 ) {
				theSprite.setOpacity( 0 );
			}
			else if ( theSprite.getOpacity() != 0 ) {
				Runnable theRunner = new Runnable() {
					@Override
					public void run() {
						FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ), theSprite );
						theTransition.setFromValue( theSprite.getOpacity() );
						theTransition.setToValue( 0 );
						theTransition.setCycleCount( 1 );
						theTransition.setAutoReverse( false );
						theTransition.play();
					}
				};
				if ( Platform.isFxApplicationThread() ) {
					theRunner.run();
				}
				else {
					Platform.runLater( theRunner );
				}
			}
		}
	}

	private void fxInitialize() throws InitializeException {
		try {
			LOGGER.log( Level.FINE, "Initializing ..." );
			if ( getViewportWidth() == -1 ) {
				setViewportWidth( getGridWidth() );
			}
			if ( getViewportHeight() == -1 ) {
				setViewportHeight( getGridHeight() );
			}
			if ( getViewportWidth() == -1 || getViewportHeight() == -1 ) {
				throw new IllegalStateException( "The viewport dimension for the checkerboard must be set!" );
			}
			fxResizeViewport( this, null, this, _initGridInMillis );
			fxResizeGrid( this, null, _initGridInMillis );
			switch ( getScaleMode() ) {
			case NONE:
				StackPane thePane = new StackPane();
				thePane.getChildren().add( _checkers );
				StackPane.setAlignment( _checkers, Pos.CENTER );
				setContent( thePane );
				break;
			case GRID:
			case FIELDS: {
				setContent( _checkers );
				setViewportOffset( this );
				setFieldDimension( this );
				break;
			}
			}
		}
		catch ( Exception e ) {
			throw new InitializeException( "Exception during initialization: " + ExceptionUtility.toMessage( e ), e );
		}
	}

	private void fxMovePlayer( P aPlayer, Position aPrecedingPosition, int aDurationInMillis ) {
		Node theSprite;
		synchronized ( this ) {
			theSprite = _playerToSprite.get( aPlayer );
		}
		Runnable theRunner = new Runnable() {
			@Override
			public void run() {
				if ( _checkers.getChildren().remove( theSprite ) ) {
					_checkers.getChildren().add( theSprite );
					if ( getMoveMode() == MoveMode.SMOOTH ) {
						TranslateTransition theTransition = new TranslateTransition( Duration.millis( aDurationInMillis ), theSprite );
						theTransition.setByX( toPixelPositionX( aPlayer ) - theSprite.getTranslateX() );
						theTransition.setByY( toPixelPositionY( aPlayer ) - theSprite.getTranslateY() );
						theTransition.setCycleCount( 1 );
						theTransition.setAutoReverse( false );
						theTransition.play();
					}
					else {
						theSprite.setTranslateX( toPixelPositionX( aPlayer ) );
						theSprite.setTranslateY( toPixelPositionY( aPlayer ) );
					}
				}
			}
		};
		if ( Platform.isFxApplicationThread() ) {
			theRunner.run();
		}
		else {
			Platform.runLater( theRunner );
		}
	}

	private void fxPlayerDraggability( P aPlayer ) {
		Node theSprite;
		synchronized ( _playerToSprite ) {
			theSprite = _playerToSprite.get( aPlayer );
		}
		if ( theSprite != null ) {
			if ( aPlayer.isDraggable() && !_playerToDragEventHandler.containsKey( aPlayer ) ) {
				DragPlayerEventHandler theDragEventHandler = new DragPlayerEventHandler( aPlayer, theSprite );
				_playerToDragEventHandler.put( aPlayer, theDragEventHandler );
			}
			else {
				DragPlayerEventHandler theDragEventHandler = _playerToDragEventHandler.remove( aPlayer );
				if ( theDragEventHandler != null ) {
					theDragEventHandler.dispose();
				}
			}
		}
		else {
			throw new IllegalStateException( "The player <" + aPlayer + "> is unknwon by this checkerboard." );
		}
	}

	private void fxPlayerVisibility( P aPlayer, int aDurationInMillis ) {
		Node theSprite;
		synchronized ( _playerToSprite ) {
			theSprite = _playerToSprite.get( aPlayer );
		}
		if ( theSprite != null ) {
			FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ), theSprite );
			theTransition.setFromValue( aPlayer.isVisible() ? 0 : 1 );
			theTransition.setToValue( aPlayer.isVisible() ? 1 : 0 );
			theTransition.setCycleCount( 1 );
			theTransition.setAutoReverse( false );
			theTransition.play();
		}
	}

	private void fxRemovePlayer( P aPlayer, int aDurationInMillis ) {
		Node theSprite;
		synchronized ( _playerToSprite ) {
			theSprite = _playerToSprite.remove( aPlayer );
		}
		fxRemovePlayer( aPlayer, theSprite, aDurationInMillis );
	}

	private void fxRemovePlayer( P aPlayer, Node aSprite, int aDurationInMillis ) {
		// Mouse events |-->
		DragPlayerEventHandler theDragEventHandler = _playerToDragEventHandler.remove( aPlayer );
		if ( theDragEventHandler != null ) {
			theDragEventHandler.dispose();
		}
		ClickPlayerEventHandler theClickEventHandler = _playerToClickEventHandler.remove( aPlayer );
		if ( theClickEventHandler != null ) {
			theClickEventHandler.dispose();
		}
		// Mouse events <--|

		if ( aSprite != null ) {
			FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ), aSprite );
			theTransition.setFromValue( 1 );
			theTransition.setToValue( 0 );
			theTransition.setCycleCount( 1 );
			theTransition.setAutoReverse( false );
			theTransition.setOnFinished( new EventHandler<ActionEvent>() {
				@Override
				public void handle( ActionEvent event ) {
					_checkers.getChildren().remove( aSprite );

				}
			} );
			theTransition.play();
		}
	}

	private void fxResetPlayer( P aPlayer, int aDurationInMillis ) {
		Node theSprite;
		Node thePrevSprite;
		synchronized ( _playerToSprite ) {
			theSprite = _spriteFactory.createInstance( aPlayer.getStatus(), this );
			thePrevSprite = _playerToSprite.put( aPlayer, theSprite );
		}
		if ( aPlayer.isDraggable() ) {
			DragPlayerEventHandler theDragEventHandler = new DragPlayerEventHandler( aPlayer, theSprite );
			_playerToDragEventHandler.put( aPlayer, theDragEventHandler );
		}
		ClickPlayerEventHandler theClickEventHandler = new ClickPlayerEventHandler( aPlayer, theSprite );
		_playerToClickEventHandler.put( aPlayer, theClickEventHandler );

		theSprite.setOpacity( 0 );
		Point2D thePoint = theSprite.localToParent( 0, 0 );
		theSprite.setLayoutX( -(thePoint.getX()) );
		theSprite.setLayoutY( -(thePoint.getY()) );
		theSprite.setTranslateX( toPixelPositionX( aPlayer ) );
		theSprite.setTranslateY( toPixelPositionY( aPlayer ) );
		_checkers.getChildren().add( theSprite );
		if ( thePrevSprite != null ) _checkers.getChildren().remove( thePrevSprite );
		Runnable theRunner = new Runnable() {
			@Override
			public void run() {
				if ( aPlayer.isVisible() && theSprite.getOpacity() == 0 ) {
					FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ), theSprite );
					theTransition.setFromValue( 0 );
					theTransition.setToValue( 1 );
					theTransition.setCycleCount( 1 );
					theTransition.setAutoReverse( false );
					theTransition.play();
				}
			}
		};
		if ( Platform.isFxApplicationThread() ) {
			theRunner.run();
		}
		else {
			Platform.runLater( theRunner );
		}
	}

	private synchronized void fxResizeGrid( GridDimension aDimension, GridDimension aPrecedingDimension, int aDurationInMillis ) {
		Runnable theRunner = new Runnable() {
			@Override
			public void run() {
				if ( _backgroundFactory != null ) {
					fxUpdateBackground( aDimension, aPrecedingDimension, aDurationInMillis );
				}
			}
		};

		if ( Platform.isFxApplicationThread() ) {
			theRunner.run();
		}
		else {
			Platform.runLater( theRunner );
		}
	}

	private synchronized void fxResizeViewport( ViewportDimension aDimension, ViewportDimension aPrecedingDimension, ViewportOffset aOffset, int aDurationInMillis ) {
		Runnable theRunner = new Runnable() {
			@Override
			public void run() {
				setFieldDimension( FxCheckerboardViewer.this );
				setViewportDimension( aDimension );
			}
		};

		if ( Platform.isFxApplicationThread() ) {
			theRunner.run();
		}
		else {
			Platform.runLater( theRunner );
		}
	}

	//	private void fxResizeStage() {
	//		Runnable theRunner = new Runnable() {
	//			@Override
	//			public void run() {
	//				if ( _stage != null ) {
	//					switch ( getScaleMode() ) {
	//					case GRID:
	//					case FIELDS: {
	//						if ( _stage != null ) {
	//							_stage.setScene( _scene );
	//						}
	//					}
	//						break;
	//					case NONE: {
	//						if ( _stage != null ) {
	//							_stage.setScene( _scene );
	//							_stage.sizeToScene();
	//							_stage.setResizable( false );
	//						}
	//						break;
	//					}
	//					}
	//				}
	//			}
	//		};
	//		if ( Platform.isFxApplicationThread() ) {
	//			theRunner.run();
	//		}
	//		else {
	//			Platform.runLater( theRunner );
	//		}
	//	}

	//	private synchronized void fxResizeFields( FieldDimensionChangedEvent<S, FxNodeFactory, FxCheckerboardViewer<S>> aEvent, FieldDimension precedingFieldDimension ) {
	//		Runnable theRunner = new Runnable() {
	//			@Override
	//			public void run() {
	//				if ( getBackground() != null ) {
	//					int index = 0;
	//					Node theOldBackgroundNode = _backgroundNode;
	//					_backgroundNode = getBackground().createInstance( FxCheckerboardViewerImpl.this );
	//					if ( theOldBackgroundNode != null ) {
	//						index = _checkers.getChildren().indexOf( theOldBackgroundNode );
	//						_checkers.getChildren().remove( _checkers.getChildren().remove( theOldBackgroundNode ) );
	//					}
	//					_checkers.getChildren().add( index, _backgroundNode );
	//				}
	//				fxResizeStage();
	//			}
	//		};
	//		if ( Platform.isFxApplicationThread() ) {
	//			theRunner.run();
	//		}
	//		else {
	//			Platform.runLater( theRunner );
	//		}
	//	}

	private void fxScalePlayer( P aPlayer, FieldDimension aFieldDimension, FieldDimension aPrecedingFieldDimension ) {
		Node theSprite;
		synchronized ( _playerToSprite ) {
			theSprite = _playerToSprite.get( aPlayer );
		}
		theSprite.setScaleX( aFieldDimension.getFieldWidth() / theSprite.getBoundsInLocal().getWidth() );
		theSprite.setScaleY( aFieldDimension.getFieldHeight() / theSprite.getBoundsInLocal().getHeight() );
		double theLayoutX = (aFieldDimension.getFieldWidth() - theSprite.getBoundsInLocal().getWidth()) / 2;
		double theLayoutY = (aFieldDimension.getFieldHeight() - theSprite.getBoundsInLocal().getHeight()) / 2;
		theSprite.setLayoutX( theLayoutX );
		theSprite.setLayoutY( theLayoutY );
		theSprite.setTranslateX( toPixelPositionX( aPlayer ) );
		theSprite.setTranslateY( toPixelPositionY( aPlayer ) );
	}

	private void fxUpdateBackground( GridDimension aDimension, GridDimension aPrecedingDimension, int aDurationInMillis ) {
		int index = 0;
		Node theOldBackgroundNode = _backgroundNode;
		Node theNewBackgroundNode = _backgroundFactory.createInstance( this );
		FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ) );
		if ( theOldBackgroundNode != null ) {
			index = _checkers.getChildren().indexOf( theOldBackgroundNode );
			theTransition.setOnFinished( new EventHandler<ActionEvent>() {
				@Override
				public void handle( ActionEvent event ) {
					_checkers.getChildren().remove( theOldBackgroundNode );
				}
			} );
		}
		Node theTransitionNode = null;
		double theFromValue = 0;
		double theToValue = 1;
		if ( aPrecedingDimension != null && aDimension.getGridWidth() >= aPrecedingDimension.getGridWidth() && aDimension.getGridHeight() >= aPrecedingDimension.getGridHeight() ) {
			theTransitionNode = theNewBackgroundNode;
			theNewBackgroundNode.setOpacity( 0 );
			theFromValue = 0;
			theToValue = 1;
			index++;
		}
		else if ( aPrecedingDimension != null && aDimension.getGridWidth() <= aPrecedingDimension.getGridWidth() && aDimension.getGridHeight() <= aPrecedingDimension.getGridHeight() ) {
			theTransitionNode = theOldBackgroundNode;
			theFromValue = 1;
			theToValue = 0;
		}

		if ( aPrecedingDimension == null ) {
			theTransitionNode = theNewBackgroundNode;
			theNewBackgroundNode.setOpacity( 0 );
		}

		_backgroundNode = theNewBackgroundNode;
		_checkers.getChildren().add( index, _backgroundNode );
		theTransition.setNode( theTransitionNode );
		theTransition.setFromValue( theFromValue );
		theTransition.setToValue( theToValue );
		theTransition.setCycleCount( 1 );
		theTransition.setAutoReverse( false );
		theTransition.play();
	}

	//	private void fxPlayerVisibility( P aPlayer, boolean isVisible, int aDurationInMillis ) {
	//		Node theSprite;
	//		synchronized ( _playerToSprite ) {
	//			theSprite = _playerToSprite.get( aPlayer );
	//		}
	//		if ( theSprite != null ) {
	//			FadeTransition theTransition = new FadeTransition( Duration.millis( aDurationInMillis ), theSprite );
	//			theTransition.setFromValue( isVisible ? 0 : 1 );
	//			theTransition.setToValue( isVisible ? 1 : 0 );
	//			theTransition.setCycleCount( 1 );
	//			theTransition.setAutoReverse( false );
	//			if ( !isVisible ) {
	//				theTransition.setOnFinished( new EventHandler<ActionEvent>() {
	//					@Override
	//					public void handle( ActionEvent event ) {
	//						theSprite.setVisible( false );
	//					}
	//				} );
	//			}
	//			if ( isVisible ) {
	//				if ( !theSprite.visibleProperty().getValue() ) theSprite.setVisible( true );
	//			}
	//			theTransition.play();
	//		}
	//	}

	private void initBordersH( double aWindowWidth ) {
		if ( Double.isNaN( _bordersH ) ) {
			double theBoardersH = aWindowWidth - ((getViewportWidth() - 1) * getFieldGap());
			if ( _checkerboard.getGridMode() == GridMode.CLOSED ) theBoardersH -= getFieldGap() * 2;
			theBoardersH = theBoardersH - (getViewportWidth() * getFieldWidth());
			// -----------------------------------------------------------------
			// Adjust one (phantom) gap for accurate calculation:
			// -----------------------------------------------------------------
			if ( _checkerboard.getGridMode() == GridMode.CLOSED ) {
				theBoardersH += getFieldGap();
			}
			else {
				theBoardersH -= getFieldGap();
			}
			// -----------------------------------------------------------------
			LOGGER.log( Level.FINE, "Horizontal (left and right) [phantom] borders from width <" + aWindowWidth + "> := " + theBoardersH );
			// ???: _windowDecorationH = _stage.getWidth() - _scene.getWidth();
			// ???: _bordersH = 0;
			_bordersH = theBoardersH;
		}
	}

	private void initBordersV( double aWindowHeight ) {
		if ( Double.isNaN( _bordersV ) ) {
			double theBoardersV = aWindowHeight - ((getViewportHeight() - 1) * getFieldGap());
			if ( _checkerboard.getGridMode() == GridMode.CLOSED ) theBoardersV -= getFieldGap() * 2;
			theBoardersV = theBoardersV - (getViewportHeight() * getFieldHeight());
			// -----------------------------------------------------------------
			// Adjust one (phantom) gap for accurate calculation:
			// -----------------------------------------------------------------
			if ( _checkerboard.getGridMode() == GridMode.CLOSED ) {
				theBoardersV += getFieldGap();
			}
			else {
				theBoardersV -= getFieldGap();
			}
			// -----------------------------------------------------------------
			LOGGER.log( Level.FINE, "Vertical (top and bottom) [phantom] borders from height <" + aWindowHeight + "> := " + theBoardersV );
			// ???: _windowDecorationV = _stage.getHeight() - _scene.getHeight();
			// ???: _bordersV = 0;
			_bordersV = theBoardersV;
		}
	}

	private void initMinStageHeight() {
		if ( Double.isNaN( _bordersV ) ) {
			initBordersV( getHeight() );
		}
		else {
			int theHeight = (int) _bordersV;
			theHeight += (getViewportHeight() * getFieldHeight() + getViewportHeight() * getFieldGap());
			setMinHeight( theHeight + _windowDecorationV );
		}
	}

	private void initMinStageWidth() {
		if ( Double.isNaN( _bordersH ) ) {
			initBordersH( getWidth() );
		}
		else {
			int theWidth = (int) _bordersH;
			theWidth += (getViewportWidth() * getFieldWidth() + getViewportWidth() * getFieldGap());
			setMinWidth( theWidth + _windowDecorationH );
		}
	}

	private boolean isViewport( Position aPosition ) {
		return isViewportPosX( aPosition.getPositionX() ) && isViewportPosY( aPosition.getPositionY() );
	}

	private boolean isViewportPosX( int aPositionX ) {
		return (aPositionX >= getViewportOffsetX() && aPositionX < getViewportWidth() + getViewportOffsetX());
	}

	private boolean isViewportPosY( int aPositionY ) {
		return (aPositionY >= getViewportOffsetY() && aPositionY < getViewportHeight() + getViewportOffsetY());
	}

	private int toPixelPositionX( Position aPosition ) {
		return (getFieldWidth() + getFieldGap()) * aPosition.getPositionX() + (getGridMode() == GridMode.CLOSED ? getFieldGap() : 0);
	}

	private int toPixelPositionY( Position aPosition ) {
		return (getFieldHeight() + getFieldGap()) * aPosition.getPositionY() + (getGridMode() == GridMode.CLOSED ? getFieldGap() : 0);
	}

	private int toScaledFieldDimension( double aNewWindowDim, int aViewportDim, int aFieldDim, int aFieldGap, double aBordersDim ) {
		if ( !Double.isNaN( aBordersDim ) && aNewWindowDim != 1.0 ) {
			double theCheckerboardDim = aNewWindowDim - aBordersDim;
			// Just consider top (left) gap as we drag:
			theCheckerboardDim -= aFieldGap;
			int theFieldDim = (int) Math.round( (theCheckerboardDim - ((aViewportDim - 1) * aFieldGap)) / aViewportDim );
			if ( theFieldDim != aFieldDim ) {
				try {
					return theFieldDim;
				}
				catch ( VetoRuntimeException veto ) { /* Do nothing */ }
			}
		}
		return -1;
	}

	private int toScaledViewportDimension( double aNewWindowDim, int aViewportDim, int aFieldDim, int aFieldGap, double aBordersDim ) {
		if ( !Double.isNaN( aBordersDim ) && aNewWindowDim != 1.0 ) {
			double theCheckerboardDim = aNewWindowDim - aBordersDim;
			// Just consider top (left) gap as we drag |-->
			theCheckerboardDim -= aFieldGap;
			// Just consider top (left) gap as we drag <--|
			int theViewportDim = (int) Math.round( theCheckerboardDim / (aFieldDim + aFieldGap) );
			if ( theViewportDim != aViewportDim ) {
				try {
					return theViewportDim;
				}
				catch ( VetoRuntimeException veto ) { /* Do nothing */ }
			}
		}
		return -1;
	}

	// /////////////////////////////////////////////////////////////////////////
	// INNER CLASSES:
	// /////////////////////////////////////////////////////////////////////////

	private class ClickPlayerEventHandler implements Disposable {

		private EventHandler<MouseEvent> _onMouseClickedEventHandler = new EventHandler<MouseEvent>() {
			@Override
			public void handle( MouseEvent aEvent ) {
				if ( _player != null && _player.isVisible() ) {
					LOGGER.log( Level.FINE, "Player mouse press X := " + aEvent.getSceneX() );
					LOGGER.log( Level.FINE, "Player mouse press Y := " + aEvent.getSceneY() );
					_player.click();
					aEvent.consume();
				}
			}
		};
		private P _player;

		private Node _sprite;

		/**
		 * Instantiates a new drag player event handler.
		 *
		 * @param aPlayer the player
		 * @param aSprite the sprite
		 */
		public ClickPlayerEventHandler( P aPlayer, Node aSprite ) {
			aSprite.setOnMouseClicked( _onMouseClickedEventHandler );
			_player = aPlayer;
			_sprite = aSprite;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void dispose() {
			_sprite.removeEventHandler( MouseEvent.MOUSE_CLICKED, _onMouseClickedEventHandler );
			_sprite = null;
			_player = null;
		}
	}

	private class DragPlayerEventHandler implements Disposable {

		private P _player;
		private double _sceneX;
		private double _sceneY;
		private Node _sprite;
		private double _translateX;
		private double _translateY;

		private EventHandler<MouseEvent> _onMouseDraggedEventHandler = new EventHandler<MouseEvent>() {

			@Override
			public void handle( MouseEvent aEvent ) {
				if ( _player.isVisible() ) {
					Node theSprite = (Node) (aEvent.getSource());
					double theSceneOffsetX = aEvent.getSceneX() - _sceneX;
					double theSceneOffsetY = aEvent.getSceneY() - _sceneY;
					double theTranslateX = _translateX + theSceneOffsetX;
					double theTranslateY = _translateY + theSceneOffsetY;
					// if ( theTranslateX < -(_bordersH / 2) ) return;
					// if ( theTranslateY < -(_bordersV / 2) ) return;
					int theBorderWidth = (int) _bordersH / 2;
					int theBorderHeight = (int) _bordersV / 2;

					// @formatter:off
					/*
						 LOGGER.log( Level.FINE, "Translate X := " + theTranslateX );
						 LOGGER.log( Level.FINE, "Translate Y := " + theTranslateY );
						 LOGGER.log( Level.FINE, "Field width = " + getFieldWidth() );
						 LOGGER.log( Level.FINE, "Field height = " + getFieldHeight() );
						 LOGGER.log( Level.FINE, "Field gap = " + getFieldGap() );
						 LOGGER.log( Level.FINE, "Viewport offset X = " + getViewportOffsetX() );
						 LOGGER.log( Level.FINE, "Viewport offset Y = " + getViewportOffsetY() );
						 LOGGER.log( Level.FINE, "Grid width = " + getGridWidth() );
						 LOGGER.log( Level.FINE, "Grid height = " + getGridHeight() );
						 LOGGER.log( Level.FINE, "Border width := " + theBorderWidth );
						 LOGGER.log( Level.FINE, "Border height := " + theBorderHeight );
						 LOGGER.log( Level.FINE, "Absolute Field width := " + (getFieldWidth() + getFieldGap()) );
						 LOGGER.log( Level.FINE, "Absolute Field height := " + (getFieldHeight() + getFieldGap()) );
						 LOGGER.log( Level.FINE, "Offset X + grid with - 1 := " + (getViewportOffsetX() + getGridWidth() - 1) );
						 LOGGER.log( Level.FINE, "Offset Y + grid height - 1 := " + (getViewportOffsetY() + getGridHeight() - 1) );
						 LOGGER.log( Level.FINE, "... for X := " + ( getFieldWidth() + getFieldGap()) * (getViewportOffsetX() + getGridWidth() - 1) );
						 LOGGER.log( Level.FINE, "... for Y := " + ( getFieldHeight() + getFieldGap()) * (getViewportOffsetY() + getGridHeight() - 1) );
					*/
					// @formatter:on

					if ( theTranslateX >= -theBorderWidth && theTranslateX <= (getFieldWidth() + getFieldGap()) * (getGridWidth() - 1) + theBorderWidth ) theSprite.setTranslateX( theTranslateX );
					if ( theTranslateY >= -theBorderHeight && theTranslateY <= (getFieldHeight() + getFieldGap()) * (getGridHeight() - 1) + theBorderHeight ) theSprite.setTranslateY( theTranslateY );
					aEvent.consume();
				}
			}
		};

		private EventHandler<MouseEvent> _onMousePressedEventHandler = new EventHandler<MouseEvent>() {
			@Override
			public void handle( MouseEvent aEvent ) {
				if ( _player.isVisible() ) {
					Node theSprite = (Node) (aEvent.getSource());
					// _checkerboard.getChildren().remove( theSprite );
					// _checkerboard.getChildren().add( theSprite );
					theSprite.setOpacity( 0.5 );
					_sceneX = aEvent.getSceneX();
					_sceneY = aEvent.getSceneY();
					_translateX = theSprite.getTranslateX();
					_translateY = theSprite.getTranslateY();
					LOGGER.log( Level.FINE, "Player mouse press X := " + aEvent.getSceneX() );
					LOGGER.log( Level.FINE, "Player mouse press Y := " + aEvent.getSceneY() );
					aEvent.consume();
				}
			}
		};

		private EventHandler<MouseEvent> _onMouseReleasedEventHandler = new EventHandler<MouseEvent>() {
			@Override
			public void handle( MouseEvent aEvent ) {
				Node theSprite = (Node) (aEvent.getSource());
				if ( _player.isVisible() ) {
					double theSceneOffsetX = aEvent.getSceneX() - _sceneX;
					double theSceneOffsetY = aEvent.getSceneY() - _sceneY;
					double theStepX = Math.round( theSceneOffsetX / (getFieldWidth() + getFieldGap()) );
					double theStepY = Math.round( theSceneOffsetY / (getFieldHeight() + getFieldGap()) );
					int thePosX = (int) (_player.getPositionX() + theStepX);
					int thePosY = (int) (_player.getPositionY() + theStepY);
					if ( thePosX >= getGridWidth() || thePosX < 0 || thePosY >= getGridHeight() || thePosY < 0 ) {
						theStepY = 0;
						theStepX = 0;
						thePosY = (int) (_player.getPositionY() + theStepY);
						thePosX = (int) (_player.getPositionX() + theStepX);
					}
					try {
						_player.setPosition( thePosX, thePosY );
					}
					catch ( VetoRuntimeException e ) {
						LOGGER.log( Level.WARNING, "Change player <" + _player + "> position to (" + thePosX + ", " + thePosY + ") has been vetoed: " + ExceptionUtility.toMessage( e ), e );
						theStepX = 0;
						theStepY = 0;
					}
					if ( theStepX == 0 && theStepY == 0 ) fxMovePlayer( _player, _player, _adjustPlayerDurationInMillis );
					theSprite.setOpacity( 1 );
					// @formatter:off
					/*
						LOGGER.log( Level.FINE, "Player mouse release X := " + aEvent.getSceneX() );
						LOGGER.log( Level.FINE, "Player mouse release Y := " + aEvent.getSceneY() );
						LOGGER.log( Level.FINE, "Player mouse release offset X := " + theSceneOffsetX );
						LOGGER.log( Level.FINE, "Player mouse release offset Y := " + theSceneOffsetY );
						LOGGER.log( Level.FINE, "Player grid step X := " + theStepX );
						LOGGER.log( Level.FINE, "Player grid step Y := " + theStepY );
						LOGGER.log( Level.FINE, "Player new pos X := " + thePosX );
						LOGGER.log( Level.FINE, "Player new pos Y := " + thePosY );
					*/
					// @formatter:on
					aEvent.consume();
				}
				// else {
				// theSprite.setTranslateX( toPositionX( _player ) );
				// theSprite.setTranslateY( toPositionY( _player ) );
				// }
			}
		};

		/**
		 * Instantiates a new drag player event handler.
		 *
		 * @param aPlayer the player
		 * @param aSprite the sprite
		 */
		public DragPlayerEventHandler( P aPlayer, Node aSprite ) {
			aSprite.setOnMousePressed( _onMousePressedEventHandler );
			aSprite.setOnMouseDragged( _onMouseDraggedEventHandler );
			aSprite.setOnMouseReleased( _onMouseReleasedEventHandler );
			_player = aPlayer;
			_sprite = aSprite;
		}

		/**
		 * {@inheritDoc}
		 */
		@Override
		public void dispose() {
			_sprite.removeEventHandler( MouseEvent.MOUSE_PRESSED, _onMousePressedEventHandler );
			_sprite.removeEventHandler( MouseEvent.MOUSE_DRAGGED, _onMouseDraggedEventHandler );
			_sprite.removeEventHandler( MouseEvent.MOUSE_RELEASED, _onMouseReleasedEventHandler );
			_sprite = null;
			_player = null;
		}
	}
}