/*
 * Decompiled with CFR 0.152.
 */
package org.redisson.connection;

import io.netty.resolver.AddressResolver;
import io.netty.resolver.dns.DnsAddressResolverGroup;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.FutureListener;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.ScheduledFuture;
import java.net.InetSocketAddress;
import java.net.URI;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.redisson.api.RFuture;
import org.redisson.client.RedisClient;
import org.redisson.connection.ClientConnectionsEntry;
import org.redisson.connection.ConnectionManager;
import org.redisson.connection.MasterSlaveEntry;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DNSMonitor {
    private static final Logger log = LoggerFactory.getLogger(DNSMonitor.class);
    private final AddressResolver<InetSocketAddress> resolver;
    private final ConnectionManager connectionManager;
    private final Map<URI, InetSocketAddress> masters = new HashMap<URI, InetSocketAddress>();
    private final Map<URI, InetSocketAddress> slaves = new HashMap<URI, InetSocketAddress>();
    private ScheduledFuture<?> dnsMonitorFuture;
    private long dnsMonitoringInterval;

    public DNSMonitor(ConnectionManager connectionManager, RedisClient masterHost, Collection<URI> slaveHosts, long dnsMonitoringInterval, DnsAddressResolverGroup resolverGroup) {
        this.resolver = resolverGroup.getResolver(connectionManager.getGroup().next());
        masterHost.resolveAddr().syncUninterruptibly();
        this.masters.put(masterHost.getConfig().getAddress(), masterHost.getAddr());
        for (URI host : slaveHosts) {
            Future<InetSocketAddress> resolveFuture = this.resolver.resolve(InetSocketAddress.createUnresolved(host.getHost(), host.getPort()));
            resolveFuture.syncUninterruptibly();
            this.slaves.put(host, resolveFuture.getNow());
        }
        this.connectionManager = connectionManager;
        this.dnsMonitoringInterval = dnsMonitoringInterval;
    }

    public void start() {
        this.monitorDnsChange();
        log.debug("DNS monitoring enabled; Current masters: {}, slaves: {}", (Object)this.masters, (Object)this.slaves);
    }

    public void stop() {
        if (this.dnsMonitorFuture != null) {
            this.dnsMonitorFuture.cancel(true);
        }
    }

    private void monitorDnsChange() {
        this.dnsMonitorFuture = this.connectionManager.getGroup().schedule(new Runnable(){

            @Override
            public void run() {
                if (DNSMonitor.this.connectionManager.isShuttingDown()) {
                    return;
                }
                AtomicInteger counter = new AtomicInteger(DNSMonitor.this.masters.size() + DNSMonitor.this.slaves.size());
                DNSMonitor.this.monitorMasters(counter);
                DNSMonitor.this.monitorSlaves(counter);
            }
        }, this.dnsMonitoringInterval, TimeUnit.MILLISECONDS);
    }

    private void monitorMasters(final AtomicInteger counter) {
        for (final Map.Entry<URI, InetSocketAddress> entry : this.masters.entrySet()) {
            Future<InetSocketAddress> resolveFuture = this.resolver.resolve(InetSocketAddress.createUnresolved(entry.getKey().getHost(), entry.getKey().getPort()));
            resolveFuture.addListener((GenericFutureListener<Future<InetSocketAddress>>)new FutureListener<InetSocketAddress>(){

                @Override
                public void operationComplete(Future<InetSocketAddress> future) throws Exception {
                    if (counter.decrementAndGet() == 0) {
                        DNSMonitor.this.monitorDnsChange();
                    }
                    if (!future.isSuccess()) {
                        log.error("Unable to resolve " + ((URI)entry.getKey()).getHost(), future.cause());
                        return;
                    }
                    InetSocketAddress currentMasterAddr = (InetSocketAddress)entry.getValue();
                    InetSocketAddress newMasterAddr = future.getNow();
                    if (!newMasterAddr.getAddress().equals(currentMasterAddr.getAddress())) {
                        log.info("Detected DNS change. Master {} has changed ip from {} to {}", entry.getKey(), currentMasterAddr.getAddress().getHostAddress(), newMasterAddr.getAddress().getHostAddress());
                        MasterSlaveEntry masterSlaveEntry = DNSMonitor.this.connectionManager.getEntry(currentMasterAddr);
                        if (masterSlaveEntry == null) {
                            log.error("Unable to find master entry for {}", (Object)currentMasterAddr);
                            return;
                        }
                        masterSlaveEntry.changeMaster(newMasterAddr, (URI)entry.getKey());
                        DNSMonitor.this.masters.put((URI)entry.getKey(), newMasterAddr);
                    }
                }
            });
        }
    }

    private void monitorSlaves(final AtomicInteger counter) {
        for (final Map.Entry<URI, InetSocketAddress> entry : this.slaves.entrySet()) {
            Future<InetSocketAddress> resolveFuture = this.resolver.resolve(InetSocketAddress.createUnresolved(entry.getKey().getHost(), entry.getKey().getPort()));
            resolveFuture.addListener((GenericFutureListener<Future<InetSocketAddress>>)new FutureListener<InetSocketAddress>(){

                @Override
                public void operationComplete(Future<InetSocketAddress> future) throws Exception {
                    if (counter.decrementAndGet() == 0) {
                        DNSMonitor.this.monitorDnsChange();
                    }
                    if (!future.isSuccess()) {
                        log.error("Unable to resolve " + ((URI)entry.getKey()).getHost(), future.cause());
                        return;
                    }
                    final InetSocketAddress currentSlaveAddr = (InetSocketAddress)entry.getValue();
                    final InetSocketAddress newSlaveAddr = future.getNow();
                    if (!newSlaveAddr.getAddress().equals(currentSlaveAddr.getAddress())) {
                        log.info("Detected DNS change. Slave {} has changed ip from {} to {}", ((URI)entry.getKey()).getHost(), currentSlaveAddr.getAddress().getHostAddress(), newSlaveAddr.getAddress().getHostAddress());
                        for (final MasterSlaveEntry masterSlaveEntry : DNSMonitor.this.connectionManager.getEntrySet()) {
                            if (!masterSlaveEntry.hasSlave(currentSlaveAddr)) continue;
                            if (masterSlaveEntry.hasSlave(newSlaveAddr)) {
                                masterSlaveEntry.slaveUp(newSlaveAddr, ClientConnectionsEntry.FreezeReason.MANAGER);
                                masterSlaveEntry.slaveDown(currentSlaveAddr, ClientConnectionsEntry.FreezeReason.MANAGER);
                                break;
                            }
                            RFuture<Void> addFuture = masterSlaveEntry.addSlave(newSlaveAddr, (URI)entry.getKey());
                            addFuture.addListener(new FutureListener<Void>(){

                                @Override
                                public void operationComplete(Future<Void> future) throws Exception {
                                    if (!future.isSuccess()) {
                                        log.error("Can't add slave: " + newSlaveAddr, future.cause());
                                        return;
                                    }
                                    masterSlaveEntry.slaveDown(currentSlaveAddr, ClientConnectionsEntry.FreezeReason.MANAGER);
                                }
                            });
                            break;
                        }
                        DNSMonitor.this.slaves.put((URI)entry.getKey(), newSlaveAddr);
                    }
                }
            });
        }
    }
}

