/**
 * Copyright (c) 2013-2024 Nikita Koksharov
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.redisson.api;

/**
 * Distributed implementation to the AtomicDouble
 *
 * @author Nikita Koksharov
 *
 */
public interface RAtomicDoubleAsync extends RExpirableAsync {

    /**
     * Atomically sets the value to the given updated value
     * only if the current value {@code ==} the expected value.
     *
     * @param expect the expected value
     * @param update the new value
     * @return true if successful; or false if the actual value
     *         was not equal to the expected value.
     */
    RFuture<Boolean> compareAndSetAsync(double expect, double update);

    /**
     * Atomically adds the given value to the current value.
     *
     * @param delta the value to add
     * @return the updated value
     */
    RFuture<Double> addAndGetAsync(double delta);

    /**
     * Atomically decrements the current value by one.
     *
     * @return the updated value
     */
    RFuture<Double> decrementAndGetAsync();

    /**
     * Returns current value.
     *
     * @return current value
     */
    RFuture<Double> getAsync();

    /**
     * Returns and deletes object
     * 
     * @return the current value
     */
    RFuture<Double> getAndDeleteAsync();
    
    /**
     * Atomically adds the given value to the current value.
     *
     * @param delta the value to add
     * @return the updated value
     */
    RFuture<Double> getAndAddAsync(double delta);

    /**
     * Atomically sets the given value and returns the old value.
     *
     * @param newValue the new value
     * @return the old value
     */
    RFuture<Double> getAndSetAsync(double newValue);

    /**
     * Atomically increments the current value by one.
     *
     * @return the updated value
     */
    RFuture<Double> incrementAndGetAsync();

    /**
     * Atomically increments the current value by one.
     *
     * @return the old value
     */
    RFuture<Double> getAndIncrementAsync();

    /**
     * Atomically decrements by one the current value.
     *
     * @return the previous value
     */
    RFuture<Double> getAndDecrementAsync();

    /**
     * Atomically sets the given value.
     *
     * @param newValue the new value
     * @return void
     */
    RFuture<Void> setAsync(double newValue);
    
    /**
     * Atomically sets the given value if current value is less than
     * the special value
     *
     * @param less  compare value
     * @param value newValue
     * @return true when the value update is successful
     */
    RFuture<Boolean> setIfLessAsync(double less, double value);
    
    /**
     * Atomically sets the given value if current value is greater than
     * the special value
     *
     * @param greater  compare value
     * @param value newValue
     * @return true when the value update is successful
     */
    RFuture<Boolean> setIfGreaterAsync(double greater, double value);

    /**
     * Adds object event listener
     *
     * @see org.redisson.api.listener.IncrByListener
     * @see org.redisson.api.ExpiredObjectListener
     * @see org.redisson.api.DeletedObjectListener
     *
     * @param listener - object event listener
     * @return listener id
     */
    RFuture<Integer> addListenerAsync(ObjectListener listener);

}
